/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 */

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>


#include "im-ja.h"
#include "im-ja-impl.h"
#include "error.h"
#include "common.h"
#include "romakana.h"
#include "hiragana-convtable.h"
#include "zenkaku-convtable.h"
#include "halfkata-convtable.h"

extern IMJAConfig cfg;

/* * * * * * * * * * * * * * * * */

gboolean im_ja_kana_convert_tailing_n(IMJAContext *cn) {
	if (strlen(cn->preedit_buf) == 0) return FALSE;
	if (cn->preedit_buf[strlen(cn->preedit_buf) - 1] == 'n') {
		cn->preedit_buf[strlen(cn->preedit_buf) - 1] = 0;
		switch (cn->input_method) {
		case IM_JA_HIRAGANA_INPUT:
			g_strlcat(cn->preedit_buf, "\xE3\x82\x93", BUFFERSIZE);
			break;
		case IM_JA_KATAKANA_INPUT:
			g_strlcat(cn->preedit_buf, "\xE3\x83\xB3", BUFFERSIZE);
			break;
		case IM_JA_HALFKATA_INPUT:
			g_strlcat(cn->preedit_buf, "\xEF\xBE\x9D", BUFFERSIZE);
			break;
		}			
		return TRUE;
	}
	return FALSE;
}


void im_ja_kana_commit_converted(IMJAContext *cn, gchar *kanastr) {
	gchar *lastkana;

	/* Find any converted char */
	if (strlen(kanastr) == 0) return;
	lastkana = kanastr + strlen(kanastr);

	while (isJPChar(g_utf8_get_char(g_utf8_prev_char(lastkana))) == FALSE) {
		lastkana = g_utf8_prev_char(lastkana);
		if (lastkana == kanastr) break;
	}
	
	/* IM_JA_DEBUG("lastkana: %s, kanastr: %s\n", lastkana, kanastr); */

	if (lastkana == kanastr + strlen(kanastr)) {
		g_strlcpy(cn->preedit_buf, kanastr, BUFFERSIZE);
		im_ja_commit(cn);
		im_ja_context_reset(cn);
	}
	else {   
		if (lastkana != kanastr) {
			g_strlcpy(cn->preedit_buf, kanastr, BUFFERSIZE);
			cn->preedit_buf[lastkana - kanastr] = 0;
			im_ja_commit(cn);
		}
		/* copy new string back to preedit buffer */
		g_strlcpy(cn->preedit_buf, lastkana, BUFFERSIZE);
		cn->cursor_char_pos = g_utf8_strlen(cn->preedit_buf, -1);
		*lastkana = 0;
	}
}


gboolean im_ja_kana_filter_keypress(IMJAContext *cn, GdkEventKey *key) {
  gchar *tmpstr;

  if (key->type == GDK_KEY_RELEASE) {
    return FALSE;
  }
  if (key->keyval == GDK_space) {
    /* g_strlcat(cn->preedit_buf, " ", BUFFERSIZE); */
		gtk_im_context_reset(GTK_IM_CONTEXT(cn));
		g_strlcat(cn->preedit_buf, " ", BUFFERSIZE);
		im_ja_commit(cn); /* Insert a space */
    /* FIXME: maybe this should be zenkaku space //ascii & unicode zenkaku diff: 65248 */
    return TRUE;
  }      
  
  if (key->keyval == GDK_BackSpace) {
    if (strlen(cn->preedit_buf) > 0) {
      buffer_delchar(cn->preedit_buf);
			im_ja_preedit_changed(cn);
      return TRUE;
    }
    return FALSE;
  }
  
  if ((key->keyval == GDK_Return) || (ishotkey(key, COMMIT_PREEDIT, &cfg) == TRUE)) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		im_ja_kana_convert_tailing_n(cn);
    if (g_utf8_validate(cn->preedit_buf, -1, NULL) != TRUE) {
      printf("Cannot commit. utf8_validate failed: %s\n", cn->preedit_buf);
    }
    else im_ja_commit(cn);
		im_ja_on_reset(cn);
    return TRUE;
  } 
  
  if (im_ja_is_printable_key(key) == TRUE) {
    gchar utf8strg[7];
		gchar *tmpeuc;
		int utf8strg_end;

		utf8strg_end = g_unichar_to_utf8(gdk_keyval_to_unicode(key->keyval), utf8strg);
		utf8strg[utf8strg_end] = 0;
		tmpeuc = utf82euc(utf8strg);
		if ((unsigned int)tmpeuc[0] > 128) {
			g_free(tmpeuc);
			IM_JA_DEBUG("*ERROR* non-ascii input\n");
			im_ja_input_utf8(cn, utf8strg);
			return TRUE;
		}
		g_free(tmpeuc);

		g_strlcat(cn->preedit_buf, utf8strg, BUFFERSIZE);

		tmpstr = roma2kana(cn->preedit_buf, cn->input_method);
		IM_JA_DEBUG("converted \"%s\" to \"%s\"\n", cn->preedit_buf, tmpstr);
		im_ja_kana_commit_converted(cn, tmpstr);
		g_free(tmpstr);

		if (g_utf8_validate(cn->preedit_buf, -1, NULL) != TRUE) {
			printf("utf8_validate failed: %s\n", cn->preedit_buf);
		}
		im_ja_preedit_changed(cn);
		return TRUE;
  }

  return FALSE; 
}

int g_strrncmp(gchar* s1, size_t s1len, gchar* s2, size_t s2len) {

  if( s1len > s2len )	s1 += s1len-s2len;
  return strncmp(s1, s2, s2len);
}


/* Convert romaji to kana. Returns the kana string */
gchar *roma2kana(gchar *romastr, gint input_method) {
	gchar *result = NULL;
	gchar *tmpstr = NULL;
	romapair *romatable = NULL;
	gint i = 0, s1len, s2len;

  if (romastr == NULL) return NULL;
  if (*romastr == '\0') return NULL;

  s1len = strlen(romastr);

	switch (input_method) {
		case IM_JA_HIRAGANA_INPUT:
		case IM_JA_KATAKANA_INPUT:
			romatable = hiraganatable;
			break;
		case IM_JA_HALFKATA_INPUT:
			romatable = halfkatatable;
			break;
		case IM_JA_ZENKAKU_INPUT:
			romatable = zenkakutable;
			break;
	}

	for (i = 0; ; i++) {
    romapair rp = romatable[i];
    if (rp.kana[0] == '0' && rp.roma[0] == '0') break;
		s2len = strlen(rp.roma);
    /* Compare end of input and string in kana table */
    if (g_strrncmp(romastr, s1len, rp.roma, s2len) == 0) {
      result = g_new0(gchar, BUFFERSIZE);
      strncat(result, romastr, s1len - s2len);
      g_strlcat(result, rp.kana, BUFFERSIZE);

			if (input_method == IM_JA_KATAKANA_INPUT) {
				tmpstr = result;
				result = hira2kata(tmpstr);
				g_free(tmpstr);
			}
			return result;
    }
  }
	return g_strdup(romastr); /* no change */
}

gchar *roma2kana_i(gchar *romastr, gint *index, gint input_method)
{
	gchar *result = NULL;
	gchar *tmpstr = NULL;
	gint i = 0,s1len,s2len;
	romapair *romatable = NULL;

	if (romastr == NULL)
		return NULL;

	result = g_new0(gchar, BUFFERSIZE);

	if (*romastr == '\0')
		return result;

	s1len = *index;

	switch (input_method) {
		case IM_JA_HIRAGANA_INPUT:
		case IM_JA_KATAKANA_INPUT:
			romatable = hiraganatable;
			break;
		case IM_JA_HALFKATA_INPUT:
			romatable = halfkatatable;
			break;
		case IM_JA_ZENKAKU_INPUT:
			romatable = zenkakutable;
			break;
	}

	for (i = 0; ; i++)
	{
		romapair rp = romatable[i];
		if (rp.kana[0] == '0' && rp.roma[0] == '0')
			break;

		s2len = strlen(rp.roma);

		/* Compare offset of input and string in kana table */
		if(g_strrncmp(romastr, s1len, rp.roma,s2len) == 0 )
		{
			*index += strlen(rp.kana) - s2len;
			strncat(result, romastr, s1len - s2len );
			g_strlcat(result, rp.kana, BUFFERSIZE);
			g_strlcat(result, romastr+s1len, BUFFERSIZE);
			if (input_method == IM_JA_KATAKANA_INPUT) {
				tmpstr = result;
				result = hira2kata(tmpstr);
				g_free(tmpstr);
			}
			return result;
		}
	}

	/* no change */
	g_strlcat(result, romastr, BUFFERSIZE);
	return result;
}


gchar *roma2kana_i_lastpass(gchar *romastr, gint *index, gint input_method)
/* IMHO the cursor position change should be done in canna_rk.c
	 depending on the return value of this function. See e.g
	 im_ja_wnn_convert_tailing_n() or im_ja_kana_convert_tailing_n()
*/
{
	gchar *result = NULL;
	gchar *subst_str = NULL;
	gchar *pattern="n";
	gint s1len;

	if (romastr == NULL)
		return NULL;

	result = g_new0(gchar, BUFFERSIZE);

	if (*romastr == '\0')
		return result;

	s1len = *index;

	switch (input_method) {
	case IM_JA_HIRAGANA_INPUT:
		subst_str = "\xE3\x82\x93";
		break;
	case IM_JA_KATAKANA_INPUT:
		subst_str = "\xE3\x83\xB3";
		break;
	case IM_JA_HALFKATA_INPUT:
		subst_str = "\xEF\xBE\x9D";
		break;
	}

	/* Compare offset of input and string in kana table */
	if(g_strrncmp(romastr, s1len, pattern, 1) == 0 )
	{
		*index += 3 - 1;
		strncat(result, romastr, s1len - 1 );
		g_strlcat(result, subst_str, BUFFERSIZE);
		g_strlcat(result, romastr+s1len, BUFFERSIZE);
		return result;
	}

	/* no change */
	g_strlcat(result, romastr, BUFFERSIZE);
	return result;
}

