/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* vi: set ts=2 sw=2: */
/* Adopted from gjiten's kanjidic.c
  
   Copyright (C) 1999-2006 Botond Botyanszki <boti@rocketmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published  by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <gdk/gdkkeysyms.h>

#include "nls.h"
#include "im-ja.h"
#include "radical.h"
#include "error.h"
#include "radical-convtable.h"

static void kanji_selected(GtkWidget *button, IMJAContext *cn);


GList *klinklist = NULL, *tmpklinklist = NULL;
static RadTable *radtable = NULL;

/* compares two lists and combines the matching kanji into one */
static void klists_merge(void) {
  GList *ptr1, *ptr2, *nextptr;
  int found;

  ptr1 = klinklist;
  while (ptr1 != NULL) {
    nextptr = g_list_next(ptr1);
    found = FALSE;
    ptr2 = tmpklinklist;
    while (ptr2 != NULL) {
      if ((gunichar) ptr1->data == (gunichar) ptr2->data) {
				found = TRUE;
				break;
      }
      ptr2 = g_list_next(ptr2);
    }
    if (found == FALSE) {
			klinklist = g_list_remove(klinklist, ptr1->data);
		}
    ptr1 = nextptr;
  }
  g_list_free(tmpklinklist);
	tmpklinklist = NULL;
}


static void findk_by_radical(gchar *radstrg) { 
  gint i, radnum;
	RadInfo *rad_info;
	GList *kanji_info_list;
	gchar *tmprad;
  gchar *radstr_ptr;

  if (g_utf8_strlen(radstrg, -1) == 0) return;

	radstr_ptr = radstrg;
	rad_info = g_hash_table_lookup(radtable->rad_info_hash, (gpointer) g_utf8_get_char(radstr_ptr));
  if (rad_info == NULL) {
    im_ja_print_error(_("Invalid radical!\n"));
    return;
  }

  for (kanji_info_list = rad_info->kanji_info_list;
			 kanji_info_list != NULL;
			 kanji_info_list = g_list_next(kanji_info_list)) {
    klinklist = g_list_prepend(klinklist, (gpointer) ((KanjiInfo *) kanji_info_list->data)->kanji);
  }

  // if we have more than 1 radical 
  radnum = g_utf8_strlen(radstrg, -1); 
  if (radnum > 1) {
    for (i = 1; i <= radnum; i++) {
      rad_info = g_hash_table_lookup(radtable->rad_info_hash, (gpointer) g_utf8_get_char(radstr_ptr));
      if (rad_info == NULL) {
				tmprad = g_strndup(radstr_ptr, sizeof(gunichar));
				im_ja_print_error(_("I don't seem to recognize this radical: '%s' !!!\n"), tmprad);
				g_free(tmprad);
				return;
      }
			for (kanji_info_list = rad_info->kanji_info_list;
					 kanji_info_list != NULL;
					 kanji_info_list = g_list_next(kanji_info_list)) {
				tmpklinklist = g_list_prepend(tmpklinklist, (gpointer) ((KanjiInfo *) kanji_info_list->data)->kanji);
      }
      klists_merge();
      radstr_ptr = g_utf8_next_char(radstr_ptr);
    }
  }
}

static void set_radical_button_sensitive(gpointer radical, RadInfo *rad_info, gpointer user_data) {
	GtkWidget *rad_button = g_hash_table_lookup(radtable->rad_button_hash, radical);
	if (GTK_IS_WIDGET(rad_button)) gtk_widget_set_sensitive(rad_button, TRUE);
}

static void set_radical_button_unsensitive(gunichar radical, GtkWidget *rad_button, gboolean sensitive) {
	if (GTK_IS_WIDGET(rad_button)) gtk_widget_set_sensitive(rad_button, sensitive);
}


static void do_search(IMJAContext *cn) {
  static gchar *radentry;
  int i;
  GList *node_ptr;
  GtkWidget *kanji_result_button;
  gchar kanji_result_str[6];
  gchar kanji_result_labelstr[100];
  GtkWidget *kanji_result_label;
	GtkTextChildAnchor *kanji_results_anchor;
	gint result_num = 0;
	GList *rad_info_list = NULL;
	GList *kanji_list = NULL;
	GList *kanji_list_ptr = NULL;
	GHashTable *rad_info_hash = NULL;
	KanjiInfo *kanji_info;

  gtk_text_buffer_set_text(GTK_TEXT_BUFFER(radtable->kanji_results_buffer), "", 0);
  gtk_text_buffer_get_start_iter(radtable->kanji_results_buffer, &radtable->kanji_results_iter);

	g_hash_table_foreach(radtable->rad_button_hash, (GHFunc) set_radical_button_unsensitive, (gpointer) TRUE);

  radentry = g_strdup(gtk_entry_get_text(GTK_ENTRY(radtable->entry_radical)));
  if ((radentry == NULL) || (strlen(radentry) < 1)) {
		return;
  }

  if (klinklist != NULL) g_list_free(klinklist);
  tmpklinklist = NULL;
  klinklist = NULL;

  //FIND BY RADICAL
  if (g_utf8_strlen(radentry, -1) > 0) {
    findk_by_radical(radentry);
    if (klinklist == NULL) {
      //gnome_appbar_set_status(GNOME_APPBAR(radtable->appbar_kanji),_("No such kanji with this radical combination."));
      return;
    }
  }

	result_num = g_list_length(klinklist);
  //snprintf(kappbarmsg, 100, _("Kanji found: %d"), result_num);
  //gnome_appbar_set_status(GNOME_APPBAR(radtable->appbar_kanji), kappbarmsg);

  // PRINT OUT KANJI FOUND
  node_ptr = klinklist;
  i = 0;
  while (node_ptr != NULL) { 
		kanji_list = g_list_prepend(kanji_list, node_ptr->data);
		memset(kanji_result_str, 0, sizeof(kanji_result_str));
    g_unichar_to_utf8((gunichar) node_ptr->data, kanji_result_str);
    //printf("%s\n", kanji_result_str);
    g_snprintf(kanji_result_labelstr, 100, "<span size=\"xx-large\">%s</span>", kanji_result_str); 
    kanji_results_anchor = gtk_text_buffer_create_child_anchor(radtable->kanji_results_buffer, &radtable->kanji_results_iter);
    
		kanji_result_label = gtk_label_new(kanji_result_str);
		gtk_widget_show(kanji_result_label);
		kanji_result_button = gtk_button_new();
		gtk_container_add(GTK_CONTAINER(kanji_result_button), kanji_result_label);
    
    gtk_widget_show(kanji_result_button);
		g_object_set_data(G_OBJECT(kanji_result_button), "im-ja-kanji-value", (gpointer) node_ptr->data);
    g_signal_connect(G_OBJECT(kanji_result_button), "clicked", G_CALLBACK(kanji_selected), cn);
    gtk_text_view_add_child_at_anchor(GTK_TEXT_VIEW(radtable->kanji_results_view), kanji_result_button, kanji_results_anchor);
    node_ptr = g_list_next(node_ptr);
  }

	// find all different radicals in all the kanji found
	if (GTK_IS_WIDGET(radtable->window)) {
		if (kanji_list != NULL) {
			rad_info_hash = g_hash_table_new(NULL, NULL);
			for (kanji_list_ptr = kanji_list;
					 kanji_list_ptr != NULL;
					 kanji_list_ptr = g_list_next(kanji_list_ptr)) {
				kanji_info = g_hash_table_lookup(radtable->kanji_info_hash, kanji_list_ptr->data);
				for (rad_info_list = kanji_info->rad_info_list;
						 rad_info_list != NULL;
						 rad_info_list = g_list_next(rad_info_list)) {
					g_hash_table_insert(rad_info_hash, (gpointer) ((RadInfo *) rad_info_list->data)->radical, rad_info_list->data);
				}
			}
			g_hash_table_foreach(radtable->rad_button_hash, (GHFunc) set_radical_button_unsensitive, (gpointer) FALSE);
			
			g_hash_table_foreach(rad_info_hash, (GHFunc) set_radical_button_sensitive, NULL);
			g_hash_table_destroy(rad_info_hash);
			g_list_free(kanji_list);
		}
		else { // none found, so set everything sesnisitve
			g_hash_table_foreach(radtable->rad_button_hash, (GHFunc) set_radical_button_unsensitive, (gpointer) TRUE);
		}
	}
}


static int radical_selected(GtkWidget *button, IMJAContext *cn) {
  int i, j;
  gchar radical_selected[6];
  gchar tmpchar[6];
  gchar *radline_ptr, *newradline;
  int removed;
  int radline_length = 0;
	gunichar radical;

	radical = (gunichar) g_object_get_data(G_OBJECT(button), "im-ja-radical-value");

  memset(radical_selected, 0, sizeof(radical_selected)); 
  g_unichar_to_utf8(radical, radical_selected);

  radline_ptr = (gchar*) gtk_entry_get_text(GTK_ENTRY(radtable->entry_radical));
  newradline = g_strndup(radline_ptr, strlen(radline_ptr) + 6); //Enough space for one more character
  radline_length = g_utf8_strlen(newradline, -1);

  for (i = 0; i < (int) (strlen(newradline) + 6); i++) newradline[i] = 0; //clear newradline

  removed = FALSE;
  for (i = 0; i < radline_length; i++) {  //Check if we already have the radical in the line
    if (g_utf8_get_char(radline_ptr) != radical) {
      for (j = 0; j < 6; j++) tmpchar[j] = 0; 
      g_unichar_to_utf8(g_utf8_get_char(radline_ptr), tmpchar);
      strncat(newradline, tmpchar, 5);
    }
    else removed = TRUE;  //if it's there then remove it
    radline_ptr = g_utf8_next_char(radline_ptr);
  }
  
	if (removed == FALSE) strncat(newradline, radical_selected, 5); //Add the radical to the line
  gtk_entry_set_text(GTK_ENTRY(radtable->entry_radical), newradline);

  g_free(newradline);
  do_search(cn);
  
  return 0;
}

static void radical_window_close() {
	if (GTK_IS_WIDGET(radtable->window) == TRUE) {
		gtk_widget_destroy(radtable->window);
		radtable->window = NULL;
	}
}

static void kanji_selected(GtkWidget *button, IMJAContext *cn) {
	gchar kanjiutf8[6]; 
	gunichar kanji;

	kanji = (gunichar) g_object_get_data(G_OBJECT(button), "im-ja-kanji-value");

	memset(kanjiutf8, 0, sizeof(kanjiutf8));
	g_unichar_to_utf8(kanji, kanjiutf8);

	im_ja_input_utf8(cn, kanjiutf8);
	radical_window_close();
}

static gunichar jis_radical_to_unicode(gchar *radical) {
	gint i;

	gchar jisutf8[6]; 

	memset(jisutf8, 0, sizeof(jisutf8));
	g_unichar_to_utf8(g_utf8_get_char(radical), jisutf8);
	
	for (i = 0; i < sizeof(radicaltable) / sizeof(radpair); i++) {
		if (strcmp(radicaltable[i].jis, jisutf8) == 0) {
			return g_utf8_get_char(radicaltable[i].uni);
		}
	}
	return g_utf8_get_char(radical);
}

static gchar *get_eof_line(gchar *ptr, gchar *end_ptr) {
  gchar *tmpptr = ptr;

  while (*tmpptr != '\n') {
    if (end_ptr == tmpptr) return NULL;
    tmpptr++;
  }
  tmpptr++;
  return tmpptr;
}

// Load the radical data from the file
static gboolean load_radkfile() {
  int error = FALSE;
  struct stat radk_stat;
  gint rad_cnt = 0;
  gchar *radkfile_ptr;
  gchar *radkfile_end;
	int radkfile_size;
  int fd = 0;
  RadInfo *rad_info = NULL;
	KanjiInfo *kanji_info;
	gunichar kanji;

  if (radtable->radkfile != NULL) {
    //printf("radkfile already initialized.\n");
    return TRUE;
  }
  if (stat(RADKFILE_NAME, &radk_stat) != 0) {
    im_ja_print_error_cmdline(RADKFILE_NAME": stat() failed.\n");
    error = TRUE;
  }
  radkfile_size = radk_stat.st_size;
  fd = open(RADKFILE_NAME, O_RDONLY);
  if (fd == -1) {
    im_ja_print_error_cmdline(RADKFILE_NAME": open() failed.\n");
    error = TRUE;
  }
  radtable->radkfile = (gchar *) mmap(NULL, radkfile_size, PROT_READ, MAP_SHARED, fd, 0);
  if (radtable->radkfile == NULL) {
		im_ja_print_error_cmdline("mmap() failed for radkfile\n");
		return FALSE;
	}

  if (error == TRUE) {
		im_ja_print_error(_("Error opening %s.\n"),	RADKFILE_NAME);
		return FALSE;
  }

  radkfile_end = radtable->radkfile + strlen(radtable->radkfile); //FIXME: lseek
  radkfile_ptr = radtable->radkfile;
  
	if (radtable->kanji_info_hash == NULL) {
		radtable->kanji_info_hash = g_hash_table_new(NULL, NULL);
	}
	if (radtable->rad_info_hash == NULL) {
		radtable->rad_info_hash = g_hash_table_new(NULL, NULL);
	}

  while((radkfile_ptr < radkfile_end) && (radkfile_ptr != NULL)) {
    if (*radkfile_ptr == '#') {  //find $ as first char on the line
      radkfile_ptr = get_eof_line(radkfile_ptr, radkfile_end); //Goto next line
      continue;
    }
    if (*radkfile_ptr == '$') {  //Radical info line
      rad_cnt++;          //Increase number of radicals found
      radkfile_ptr = g_utf8_next_char(radkfile_ptr);
      while (g_unichar_iswide(g_utf8_get_char(radkfile_ptr)) == FALSE) { //Find radical
				radkfile_ptr = g_utf8_next_char(radkfile_ptr);
			}

			rad_info = g_new0(RadInfo, 1);
			rad_info->radical = jis_radical_to_unicode(radkfile_ptr); //store radical

      while (g_ascii_isdigit(*radkfile_ptr) == FALSE) { //Find stroke number
				radkfile_ptr = g_utf8_next_char(radkfile_ptr);
			}

      rad_info->strokes = atoi(radkfile_ptr);  //Store the stroke number
			radtable->rad_info_list = g_list_append(radtable->rad_info_list, rad_info);
			g_hash_table_insert(radtable->rad_info_hash, (gpointer) rad_info->radical, rad_info);
			radkfile_ptr = get_eof_line(radkfile_ptr, radkfile_end); //Goto next line
    }
    else {   //Kanji
      while ((*radkfile_ptr != '$') && (radkfile_ptr < radkfile_end)) {
				if (*radkfile_ptr == '\n') {
					radkfile_ptr++;
					continue;
				}
				kanji = g_utf8_get_char(radkfile_ptr);
				kanji_info = g_hash_table_lookup(radtable->kanji_info_hash, (gconstpointer) kanji);
				if (kanji_info == NULL) {
					kanji_info = g_new0(KanjiInfo, 1);
					kanji_info->kanji = kanji;
					g_hash_table_insert(radtable->kanji_info_hash, (gpointer) kanji, (gpointer) kanji_info);
				}
				kanji_info->rad_info_list = g_list_prepend(kanji_info->rad_info_list, rad_info);
				rad_info->kanji_info_list = g_list_prepend(rad_info->kanji_info_list, kanji_info);
				radkfile_ptr = g_utf8_next_char(radkfile_ptr);
      }
    }
  }
	return TRUE;
}

static gboolean key_press_cb(GtkWidget *window, GdkEventKey *event, gpointer data) {
	if (event->keyval == GDK_Escape) {
		gtk_widget_destroy(window);
		return TRUE;
	}
	return FALSE;
}

void im_ja_radtable_show(IMJAContext *cn) {
  GtkWidget *vbox_main;
  GtkWidget *vbox_radicals;
  GtkWidget *frame_kresults;
  GtkWidget *frame_radicals;
  GtkWidget *scrolledwin_kresults;
  GtkWidget *hbox;
  int i = 0, j = 0;
  int curr_strokecount = 0;
  GtkWidget *radical_table; 
  GtkWidget *tmpwidget = NULL;
  GtkWidget *radical_label;
  gchar *strokenum_label;
  gchar radical[6];
	RadInfo *rad_info = NULL;
	GList *rad_info_list;
	
  if ((radtable != NULL) && (GTK_IS_WIDGET(radtable->window) == TRUE)) {
		gtk_window_present(GTK_WINDOW(radtable->window));
		return;
  }

	if (radtable == NULL) {
		radtable = g_new0(RadTable, 1);
	}

  if (load_radkfile() == FALSE)
	{
		g_free(radtable);
		radtable = NULL;
		return;
	}

  radtable->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(radtable->window), _("Radicals"));
  g_signal_connect(GTK_OBJECT(radtable->window), "destroy", GTK_SIGNAL_FUNC(radical_window_close), NULL);
//  gtk_window_set_default_size(GTK_WINDOW(radtable->window), 500, 500);
	gtk_window_set_modal(GTK_WINDOW(radtable->window), TRUE);

	/* FIXME */
#ifdef IMJA_TARGET_GTK
  if (GTK_IS_WINDOW(cn->toplevel_gtk)) {
		gtk_window_set_transient_for(GTK_WINDOW(radtable->window), GTK_WINDOW(cn->toplevel_gtk));
	}
#endif

	g_signal_connect(GTK_OBJECT(radtable->window), "key_press_event", G_CALLBACK(key_press_cb), NULL);

  vbox_main = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox_main);
  gtk_container_add(GTK_CONTAINER(radtable->window), vbox_main);

	if (radtable->rad_button_hash != NULL)	{
		g_hash_table_destroy(radtable->rad_button_hash);
		radtable->rad_button_hash = NULL;
	}
	
	radtable->rad_button_hash = g_hash_table_new_full(NULL, NULL, NULL, NULL);

  frame_radicals = gtk_frame_new(NULL);
  gtk_widget_show(frame_radicals);
  gtk_container_set_border_width(GTK_CONTAINER(frame_radicals), 0);
  gtk_box_pack_start(GTK_BOX(vbox_main), frame_radicals, FALSE, FALSE, 0);

  vbox_radicals = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox_radicals);
  gtk_container_add(GTK_CONTAINER(frame_radicals), vbox_radicals);
	
  radical_table = gtk_table_new(11, RADLISTLEN, TRUE); 
  gtk_box_pack_start(GTK_BOX(vbox_radicals), radical_table, FALSE, FALSE, 0);
  gtk_widget_show(radical_table);

	for (rad_info_list = radtable->rad_info_list; rad_info_list != NULL; rad_info_list = g_list_next(rad_info_list)) {
    if (i == RADLISTLEN) {
      i = 0;
      j++;
    }
		rad_info = (RadInfo *) rad_info_list->data;
    if (curr_strokecount != rad_info->strokes) {
			if (i == RADLISTLEN - 1) {
				i = 0;
				j++;
			}
      curr_strokecount = rad_info->strokes;
      strokenum_label = g_strdup_printf("<b>%d</b>", curr_strokecount); //Make a label with the strokenumber
      tmpwidget = gtk_label_new(""); //radical stroke number label
      gtk_label_set_markup(GTK_LABEL(tmpwidget), strokenum_label);
			g_free(strokenum_label);

      gtk_table_attach(GTK_TABLE(radical_table), tmpwidget, i, i+1, j, j+1,
											 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
											 (GtkAttachOptions)(0), 0, 0);
      gtk_widget_show(tmpwidget);
      i++;
    }
    memset(radical, 0, sizeof(radical));
    g_unichar_to_utf8(rad_info->radical, radical);
    radical_label = gtk_label_new(radical);
    gtk_widget_show(radical_label);
    tmpwidget = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(tmpwidget), radical_label);
		g_object_set_data(G_OBJECT(tmpwidget), "im-ja-radical-value", (gpointer) rad_info->radical);
    g_signal_connect(GTK_OBJECT(tmpwidget), "clicked", GTK_SIGNAL_FUNC(radical_selected), cn);
    
    gtk_table_attach(GTK_TABLE(radical_table), tmpwidget , i, i+1, j, j+1,
		     (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
		     (GtkAttachOptions)(GTK_EXPAND | GTK_FILL), 0, 0);
    //      gtk_widget_set_usize(tmpwidget,20,20);
    gtk_widget_show(tmpwidget);
		g_hash_table_insert(radtable->rad_button_hash, (gpointer) rad_info->radical, tmpwidget);
    i++;
  }  


  hbox = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox_main), hbox, FALSE, FALSE, 7);

	radtable->entry_radical = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(radtable->entry_radical), FALSE);
	gtk_entry_set_width_chars(GTK_ENTRY(radtable->entry_radical), 15);
  gtk_widget_show(radtable->entry_radical);
	tmpwidget = gtk_label_new(NULL);
	gtk_widget_show(tmpwidget);
  gtk_box_pack_start(GTK_BOX(hbox), tmpwidget, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), radtable->entry_radical, FALSE, FALSE, 0);
	tmpwidget = gtk_label_new(NULL);
	gtk_widget_show(tmpwidget);
  gtk_box_pack_start(GTK_BOX(hbox), tmpwidget, TRUE, TRUE, 0);

  frame_kresults = gtk_frame_new(_("Search Results :"));
  gtk_widget_show(frame_kresults);
  gtk_container_set_border_width(GTK_CONTAINER(frame_kresults), 2);

  scrolledwin_kresults = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwin_kresults),
				 GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwin_kresults), GTK_SHADOW_IN);
  gtk_widget_show(scrolledwin_kresults);
  gtk_container_add(GTK_CONTAINER(frame_kresults), scrolledwin_kresults);

  radtable->kanji_results_view = gtk_text_view_new();
  radtable->kanji_results_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(radtable->kanji_results_view));
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(radtable->kanji_results_view), GTK_WRAP_CHAR);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(radtable->kanji_results_view), FALSE);
	gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(radtable->kanji_results_view), FALSE);
  gtk_widget_show(radtable->kanji_results_view);
  gtk_container_add(GTK_CONTAINER(scrolledwin_kresults), radtable->kanji_results_view);
  gtk_widget_set_size_request(radtable->kanji_results_view, -1, 66);

  gtk_box_pack_start(GTK_BOX(vbox_main), frame_kresults, TRUE, TRUE, 0);

	gtk_widget_show(radtable->window);
}
