/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 * Copyright (C) 2003 Botond Botyanszki <boti@rocketmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Based on gtklabel.c
 */

#include <config.h>
#include <gtk/gtk.h>
#include <math.h>

#include "preeditarea.h"
#include "error.h"

static void preedit_area_class_init(PreeditAreaClass *klass);
static void preedit_area_init(PreeditArea *area);
static void preedit_area_destroy(GtkObject *object);
static void preedit_area_finalize(GObject *object);
static void preedit_area_size_request(GtkWidget *widget,
																			GtkRequisition *requisition);
static void preedit_area_size_allocate(GtkWidget *widget,
																			 GtkAllocation *allocation);
static void preedit_area_state_changed(GtkWidget *widget,
																			 GtkStateType state);
static void preedit_area_style_set(GtkWidget *widget,
																	 GtkStyle *previous_style);
static void preedit_area_direction_changed(GtkWidget *widget,
																					 GtkTextDirection previous_dir);
static gint preedit_area_expose(GtkWidget *widget,
																GdkEventExpose *event);

static void preedit_area_realize(GtkWidget *widget);
static void preedit_area_unrealize(GtkWidget *widget);
static void preedit_area_map(GtkWidget *widget);
static void preedit_area_unmap(GtkWidget *widget);


static void preedit_area_set_attributes_internal(PreeditArea *area,
																								 PangoAttrList *attrs);
static void preedit_area_recalculate(PreeditArea *area);
static void preedit_area_hierarchy_changed(GtkWidget *widget,
																					 GtkWidget *old_toplevel);
static void preedit_area_screen_changed(GtkWidget *widget,
																				GdkScreen *old_screen);

static void preedit_area_create_window(PreeditArea *area);
static void preedit_area_destroy_window(PreeditArea *area);
static void preedit_area_clear_layout(PreeditArea *area);
static void preedit_area_ensure_layout(PreeditArea *area);
static gboolean preedit_area_focus(GtkWidget *widget,
																	 GtkDirectionType direction);


static GtkMiscClass *parent_class = NULL;


GType preedit_area_get_type(void) {
  static GType type = 0;
  
  if (!type) {
      static const GTypeInfo info = {
				sizeof (PreeditAreaClass),
				NULL,           /* base_init */
				NULL,           /* base_finalize */
				(GClassInitFunc) preedit_area_class_init,
				NULL,           /* class_finalize */
				NULL,           /* class_data */
				sizeof (PreeditArea),
				32,             /* n_preallocs */
				(GInstanceInitFunc) preedit_area_init,
				NULL            /* value_table */
      };

      type = g_type_register_static(GTK_TYPE_MISC, "PreeditArea",
				     &info, 0);
    }

  return type;
}

static void preedit_area_class_init(PreeditAreaClass *class) {
	GObjectClass *gobject_class = G_OBJECT_CLASS(class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS(class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(class);

	/* IM_JA_DEBUG("preedit_area_class_init()\n"); */

  parent_class = g_type_class_peek_parent(class);
  
	gobject_class->finalize = preedit_area_finalize;

  object_class->destroy = preedit_area_destroy;
  
  widget_class->size_request = preedit_area_size_request;
  widget_class->size_allocate = preedit_area_size_allocate;
  widget_class->state_changed = preedit_area_state_changed;
  widget_class->style_set = preedit_area_style_set;
  widget_class->direction_changed = preedit_area_direction_changed;
  widget_class->expose_event = preedit_area_expose;
  widget_class->realize = preedit_area_realize;
  widget_class->unrealize = preedit_area_unrealize;
  widget_class->map = preedit_area_map;
  widget_class->unmap = preedit_area_unmap;
  widget_class->hierarchy_changed = preedit_area_hierarchy_changed;
  widget_class->screen_changed = preedit_area_screen_changed;
  widget_class->focus = preedit_area_focus;

}

static void preedit_area_init(PreeditArea *area) {
  GTK_WIDGET_SET_FLAGS(area, GTK_NO_WINDOW);

	/* IM_JA_DEBUG("preedit_area_init\n"); */
  
	area->layout = NULL;
  area->text = NULL;
  area->attrs = NULL;
  area->cursor_pos = 0;

  preedit_area_set_text(area, "");
}

GtkWidget *preedit_area_new(const gchar *str) {
  PreeditArea *area;

	/* IM_JA_DEBUG("preedit_area_new\n"); */
  
  area = g_object_new(TYPE_PREEDIT_AREA, NULL);
  if (str && *str) preedit_area_set_text(area, str);
 
  return GTK_WIDGET(area);
}

static void preedit_area_hierarchy_changed(GtkWidget *widget, GtkWidget *old_toplevel) {
	/* IM_JA_DEBUG("preedit_area_hierarchy_changed\n"); */
}

static void preedit_area_screen_changed(GtkWidget *widget, GdkScreen *old_screen) {
  preedit_area_clear_layout(PREEDIT_AREA(widget));
	/* IM_JA_DEBUG("preedit_area_screen_changed\n"); */
}


static void preedit_area_set_text_internal(PreeditArea *area, gchar *str) {
  g_free(area->text);

	/* IM_JA_DEBUG("preedit_area_set_text_internal\n"); */
  
  area->text = str;

}

static void preedit_area_set_attributes_internal(PreeditArea *area, PangoAttrList *attrs) {
	/* IM_JA_DEBUG("preedit_area_set_attributes_internal\n"); */

  if (attrs) pango_attr_list_ref(attrs);
  
  if (area->attrs) pango_attr_list_unref(area->attrs);

	pango_attr_list_ref(attrs);
	if (area->effective_attrs) pango_attr_list_unref(area->effective_attrs);
	area->effective_attrs = attrs;
	area->attrs = attrs;
	/*  g_object_notify(G_OBJECT(area), "attributes"); */
}

static void preedit_area_recalculate(PreeditArea *area) {
	/* IM_JA_DEBUG("preedit_area_recalculate\n"); */

  if (area->attrs)  pango_attr_list_ref(area->attrs);
  if (area->effective_attrs) pango_attr_list_unref(area->effective_attrs);
  area->effective_attrs = area->attrs;
 
  preedit_area_clear_layout(area);  
  gtk_widget_queue_resize(GTK_WIDGET(area));
}

void preedit_area_set_text(PreeditArea *area, const gchar *str) {

	/* IM_JA_DEBUG("preedit_area_set_text\n"); */

  g_return_if_fail(IS_PREEDIT_AREA(area));
  
  g_object_freeze_notify(G_OBJECT(area));

  preedit_area_set_text_internal(area, g_strdup(str ? str : ""));
  
  preedit_area_recalculate(area);

  g_object_thaw_notify(G_OBJECT(area));
}


G_CONST_RETURN gchar *preedit_area_get_text(PreeditArea *area) {
	/* IM_JA_DEBUG("preedit_area_get_text\n"); */

  g_return_val_if_fail(IS_PREEDIT_AREA(area), NULL);

  return area->text;
}

void preedit_area_set_attributes(PreeditArea *area, PangoAttrList *attrs) {

	/* IM_JA_DEBUG("preedit_area_set_attributes\n"); */

  g_return_if_fail(IS_PREEDIT_AREA(area));

  preedit_area_set_attributes_internal(area, attrs);
  
  preedit_area_clear_layout(area);  
  gtk_widget_queue_resize(GTK_WIDGET(area));
}

PangoAttrList *preedit_area_get_attributes(PreeditArea *area) {

	/* IM_JA_DEBUG("preedit_area_get_attributes\n"); */

  g_return_val_if_fail(IS_PREEDIT_AREA(area), NULL);

  return area->attrs;
}


static void preedit_area_destroy(GtkObject *object) {
  /* PreeditArea *area = PREEDIT_AREA(object); */

	/* IM_JA_DEBUG("preedit_area_destroy\n"); */

  GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static void preedit_area_finalize(GObject *object) {
  PreeditArea *area;
  
	/* IM_JA_DEBUG("preedit_area_finalize\n"); */

  g_return_if_fail(IS_PREEDIT_AREA(object));
  
  area = PREEDIT_AREA(object);
  
  g_free(area->text);

  if (area->layout) g_object_unref(area->layout);

  if (area->attrs) pango_attr_list_unref(area->attrs);

  if (area->effective_attrs) pango_attr_list_unref(area->effective_attrs);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void preedit_area_clear_layout(PreeditArea *area) {
	/* IM_JA_DEBUG("preedit_area_clear_layout\n"); */

  if (area->layout) {
		g_object_unref(area->layout);
		area->layout = NULL;
	}
}

static void preedit_area_ensure_layout(PreeditArea *area) {
  GtkWidget *widget;
  gint rwidth, rheight;
	/* IM_JA_DEBUG("preedit_area_ensure_layout\n"); */

  widget = GTK_WIDGET(area);
	
	rwidth = area->misc.xpad * 2;
  rheight = area->misc.ypad * 2;

  if (!area->layout) {
		PangoAlignment align = PANGO_ALIGN_LEFT; 
		area->layout = gtk_widget_create_pango_layout(widget, area->text);
		
		if (area->effective_attrs) pango_layout_set_attributes(area->layout, area->effective_attrs);
      
		pango_layout_set_alignment(area->layout, align);
		pango_layout_set_width(area->layout, -1);
	}
}

static void preedit_area_size_request(GtkWidget *widget, GtkRequisition *requisition) {
  PreeditArea *area;
  gint width, height;
  PangoRectangle logical_rect;

	/* IM_JA_DEBUG("preedit_area_size_request\n"); */
  
  g_return_if_fail(IS_PREEDIT_AREA(widget));
  g_return_if_fail(requisition != NULL);
  
  area = PREEDIT_AREA(widget);

  preedit_area_ensure_layout(area);

  width = area->misc.xpad * 2;
  height = area->misc.ypad * 2;

  pango_layout_get_extents(area->layout, NULL, &logical_rect);
  
	width += PANGO_PIXELS(logical_rect.width);
  
  height += PANGO_PIXELS(logical_rect.height);

  requisition->width = width;
  requisition->height = height;
}

static void preedit_area_size_allocate(GtkWidget *widget, GtkAllocation *allocation) {
  PreeditArea *area;
	/* IM_JA_DEBUG("preedit_area_size_allocate\n"); */

  area = PREEDIT_AREA(widget);

	(*GTK_WIDGET_CLASS(parent_class)->size_allocate)(widget, allocation);

}

static void preedit_area_state_changed(GtkWidget *widget, GtkStateType prev_state) {
  PreeditArea *area;
	/* IM_JA_DEBUG("preedit_area_state_changed\n"); */
  
  area = PREEDIT_AREA(widget);

if (GTK_WIDGET_CLASS(parent_class)->state_changed) GTK_WIDGET_CLASS(parent_class)->state_changed(widget, prev_state);
}

static void preedit_area_style_set(GtkWidget *widget, GtkStyle  *previous_style) {
  PreeditArea *area;
	/* IM_JA_DEBUG("preedit_area_style_set\n"); */
  
  g_return_if_fail(IS_PREEDIT_AREA(widget));
  
  area = PREEDIT_AREA(widget);

  /* We have to clear the layout, fonts etc. may have changed */
  preedit_area_clear_layout(area);
}

static void preedit_area_direction_changed(GtkWidget *widget, GtkTextDirection previous_dir) {
  PreeditArea *area = PREEDIT_AREA(widget);
	/* IM_JA_DEBUG("preedit_area_direction_changed\n"); */

  if (area->layout) pango_layout_context_changed(area->layout);

  GTK_WIDGET_CLASS(parent_class)->direction_changed(widget, previous_dir);
}

static void get_layout_location(PreeditArea *area, gint *xp, gint *yp) {
  GtkMisc *misc;
  GtkWidget *widget;
  gfloat xalign;
  gint x, y;

	/* IM_JA_DEBUG("get_layout_location\n"); */
  
  misc = GTK_MISC(area);
  widget = GTK_WIDGET(area);
  
  if (gtk_widget_get_direction(widget) == GTK_TEXT_DIR_LTR) xalign = misc->xalign;
  else xalign = 1.0 - misc->xalign;

  x = floor(widget->allocation.x +(gint)misc->xpad +
						xalign *(widget->allocation.width - widget->requisition.width)
						+ 0.5);

  if (gtk_widget_get_direction(widget) == GTK_TEXT_DIR_LTR)  x = MAX(x, widget->allocation.x + misc->xpad);
  else x = MIN(x, widget->allocation.x + widget->allocation.width -
							 widget->requisition.width - misc->xpad);

  y = floor(widget->allocation.y +(gint)misc->ypad 
						+ MAX(((widget->allocation.height - widget->requisition.height) * misc->yalign)	+ 0.5, 0));

  if (xp) *xp = x;
  if (yp) *yp = y;
}


void preedit_area_set_cursor_pos(PreeditArea *area, gint pos) {
	area->cursor_pos = pos;
}

static void preedit_area_draw_cursor(PreeditArea *area, gint xoffset, gint yoffset) {
	/* IM_JA_DEBUG("preedit_area_draw_cursor\n"); */

  if (GTK_WIDGET_DRAWABLE(area)) {
      GtkWidget *widget = GTK_WIDGET(area);

			PangoRectangle cursor_pos;
      GdkRectangle cursor_location;
#ifdef GTK_2_2
      GdkGC *gc;
#endif
			PangoLayout *layout = preedit_area_get_layout(PREEDIT_AREA(area));

      preedit_area_ensure_layout(area);
      
			pango_layout_get_cursor_pos(layout, area->cursor_pos, &cursor_pos, NULL);

      cursor_location.x = xoffset + PANGO_PIXELS(cursor_pos.x);
      cursor_location.y = yoffset + PANGO_PIXELS(cursor_pos.y);
      cursor_location.width = 0;
      cursor_location.height = PANGO_PIXELS(cursor_pos.height);

#ifdef GTK_2_2
      gc = _gtk_get_insertion_cursor_gc(widget, TRUE);
			_gtk_draw_insertion_cursor(widget, widget->window, gc, &cursor_location,
																 GTK_TEXT_DIR_LTR, FALSE);
			g_object_unref(gc);
#else
			gtk_draw_insertion_cursor(widget, widget->window, NULL, &cursor_location, 
																TRUE, GTK_TEXT_DIR_LTR, FALSE);
#endif
      
 	}

	
}


static gint preedit_area_expose(GtkWidget *widget, GdkEventExpose *event) {
  PreeditArea *area;
  gint x, y;

	/* IM_JA_DEBUG("preedit_area_expose\n"); */
  
  g_return_val_if_fail(IS_PREEDIT_AREA(widget), FALSE);
  g_return_val_if_fail(event != NULL, FALSE);
  
  area = PREEDIT_AREA(widget);

  preedit_area_ensure_layout(area);
  
  if (GTK_WIDGET_VISIBLE(widget) && GTK_WIDGET_MAPPED(widget) && area->text &&(*area->text != '\0')) {
      get_layout_location(area, &x, &y);
      
      gtk_paint_layout(widget->style,
											 widget->window,
											 GTK_WIDGET_STATE(widget),
											 FALSE,
											 &event->area,
											 widget,
											 "area",
											 x, y,
											 area->layout);
      
			preedit_area_draw_cursor(area, x, y);
			
	}

  return FALSE;
}


static void preedit_area_realize(GtkWidget *widget) {
  PreeditArea *area;

	/* IM_JA_DEBUG("preedit_area_realize\n"); */

  area = PREEDIT_AREA(widget);
  
	(*GTK_WIDGET_CLASS(parent_class)->realize)(widget);

  preedit_area_create_window(area);
}

static void preedit_area_unrealize(GtkWidget *widget) {
  PreeditArea *area;

	/* IM_JA_DEBUG("preedit_area_unrealize\n"); */

  area = PREEDIT_AREA(widget);

	preedit_area_destroy_window(area);
  
	(*GTK_WIDGET_CLASS(parent_class)->unrealize)(widget);
}

static void preedit_area_map(GtkWidget *widget) {
  PreeditArea *area;

	/* IM_JA_DEBUG("preedit_area_map\n"); */

  area = PREEDIT_AREA(widget);
  
	(*GTK_WIDGET_CLASS(parent_class)->map)(widget);
}

static void preedit_area_unmap(GtkWidget *widget) {
  PreeditArea *area;

	/* IM_JA_DEBUG("preedit_area_unmap\n"); */

  area = PREEDIT_AREA(widget);

	(*GTK_WIDGET_CLASS(parent_class)->unmap)(widget);
}

static void preedit_area_create_window(PreeditArea *area) {
  GtkWidget *widget;
  GdkWindowAttr attributes;
  gint attributes_mask;

	/* IM_JA_DEBUG("preedit_area_create_window\n"); */
  
  g_assert(GTK_WIDGET_REALIZED(area));
  
  widget = GTK_WIDGET(area);

  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.window_type = GDK_WINDOW_TEMP;
  attributes.wclass = GDK_INPUT_ONLY;
  attributes.override_redirect = TRUE;
  attributes.cursor = gdk_cursor_new_for_display(gtk_widget_get_display(widget), GDK_XTERM);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_NOREDIR | GDK_WA_CURSOR;

  gdk_cursor_unref(attributes.cursor);
}

static void preedit_area_destroy_window(PreeditArea *area) {
	/* IM_JA_DEBUG("preedit_area_destroy_window\n"); */

  
}

PangoLayout *preedit_area_get_layout(PreeditArea *area) {
  g_return_val_if_fail(IS_PREEDIT_AREA(area), NULL);

	/* IM_JA_DEBUG("preedit_area_get_layout\n"); */

  preedit_area_ensure_layout(area);

  return area->layout;
}

static gboolean preedit_area_focus(GtkWidget *widget, GtkDirectionType direction) {
	/* IM_JA_DEBUG("preedit_area_focus\n"); */
  return FALSE;
}

