/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* KanjiPad - Japanese handwriting recognition front end
 * Copyright (C) 1997 Owen Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>

#include <gtk/gtk.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "padarea.h"
#include "kanjipad.h"
#include "../im-ja.h"
#include "../im-ja-impl.h"
#include "../error.h"
#include "../common.h"
#include "../candwin.h"

#define WCHAR_EQ(a,b) (a == b)

extern IMJAConfig cfg; 
extern gboolean kanjipad_focus_out_disabled;

/* user interface elements */
GtkWidget *clear_button;
GtkWidget *lookup_button;

#define MAX_GUESSES 10
unsigned long kanjiguess[MAX_GUESSES];
int num_guesses = 0;
unsigned long kselected;

PadArea *pad_area;

/* globals for engine communication */
static int engine_pid;
static GIOChannel *from_engine;
static GIOChannel *to_engine;

static char *data_file = NULL;

static void clear_callback ();
static void look_up_callback ();

void put_guesses();

static gchar *
utf8_for_char (unsigned long ch)
{
    gchar *string_utf=g_malloc(30);

    if (!string_utf)
    {
        g_printerr ("Cannot alloc\n");
        exit (1);
    }
	string_utf[g_unichar_to_utf8(ch,string_utf)]=0;

    return string_utf;
}



void put_guesses(IMJAContext *context) {
  int i;
  gchar *tmpptr;

  im_ja_free_candidate_list(context);

  for (i = 0; i < num_guesses; i++) {
    tmpptr = utf8_for_char(kanjiguess[i]);
    context->candidate_list = g_list_append(context->candidate_list, tmpptr);
    /* printf("added: %s\n", utf82euc(tmpptr)); */
  }
  candidate_window_show(context, 0);

}

static void
look_up_callback (GtkWidget *w)
{
    /*	     kill 'HUP',$engine_pid; */
    GList *tmp_list;
		gsize written;
    GString *message = g_string_new (NULL);

    tmp_list = pad_area->strokes;
    while (tmp_list)
    {
        GList *stroke_list = tmp_list->data;
        while (stroke_list)
        {
            gint16 x = ((GdkPoint *)stroke_list->data)->x;
            gint16 y = ((GdkPoint *)stroke_list->data)->y;
            g_string_append_printf (message, "%d %d ", x, y);
            stroke_list = stroke_list->next;
        }
        g_string_append (message, "\n");
        tmp_list = tmp_list->next;
    }
    g_string_append (message, "\n");
/*fprintf(stderr,"%s",message->str);
 * apparently "g_io_channel_write_chars" is borken
    if (g_io_channel_write_chars (to_engine,
                                  message->str, message->len,
                                  &written, &err) != G_IO_STATUS_NORMAL)
    {
        g_printerr ("Cannot write message to engine: %s\n",
                    err->message);
        exit (1);
    }
*/
written = write( g_io_channel_unix_get_fd(to_engine),message->str, message->len);
/*fprintf(stderr,"Message LEN:%d, written:%d\n",message->len,written);
    if (g_io_channel_flush (to_engine, &err) != G_IO_STATUS_NORMAL)
    {
        g_printerr ("Error flushing message to engine: %s\n",
                    err->message);
        exit (1);
    }
*/
    g_string_free (message, FALSE);
}

static void 
clear_callback (GtkWidget *w)
{
  pad_area_clear (pad_area);
}

/*
static void 
save_callback (GtkWidget *w)
{
  static int unknownID = 0;
  static FILE *samples = NULL;

  int found = FALSE;
  int i;
  GList *tmp_list;
  
  if (!samples)
    {
      if (!(samples = fopen("samples.dat", "a")))
	g_error ("Can't open 'samples.dat': %s", g_strerror(errno));
    }
  
  if (kselected.d[0] || kselected.d[1])
    {
      for (i=0; i<num_guesses; i++)
	{
	  if (WCHAR_EQ (kselected, kanjiguess[i]))
	    found = TRUE;
	}
    }
  
  if (found)
    fprintf(samples,"%2x%2x %c%c\n", kselected.d[0], kselected.d[1],
	   0x80 | kselected.d[0], 0x80 | kselected.d[1]);
  else
    {
      fprintf (samples, "0000 ??%d\n", unknownID);
      fprintf (stderr, "Unknown character saved, ID: %d\n", unknownID);
      unknownID++;
    }

  tmp_list = pad_area->strokes;
  while (tmp_list)
    {
     GList *stroke_list = tmp_list->data;
     while (stroke_list)
       {
	 gint16 x = ((GdkPoint *)stroke_list->data)->x;
	 gint16 y = ((GdkPoint *)stroke_list->data)->y;
	 fprintf(samples, "%d %d ", x, y);
	 stroke_list = stroke_list->next;
       }
     fprintf(samples, "\n");
     tmp_list = tmp_list->next;
    }
  fprintf(samples, "\n");
  fflush(samples);
}
*/



#define BUFLEN 256

static gboolean
engine_input_handler (GIOChannel *source, GIOCondition condition, IMJAContext *context)
{
    static gchar *p;
    static gchar *line;
    GError *err = NULL;
    GIOStatus status;
    int i;

    status = g_io_channel_read_line (from_engine, &line, NULL, NULL, &err);
    switch (status)
    {
    case G_IO_STATUS_ERROR:
        im_ja_print_error("Error reading from kpengine: %s\n", err->message);
        /* exit(1); */
        return TRUE;
    case G_IO_STATUS_NORMAL:
        break;
    case G_IO_STATUS_EOF:
        im_ja_print_error("kpengine no longer exists");
        /* exit (1); */
        return TRUE;
    case G_IO_STATUS_AGAIN:
        g_assert_not_reached ();
        break;
    }

    if (line[0] == 'K') {

			unsigned long t1;
			p = line + 1;

			for (i = 0; i < MAX_GUESSES; i++) {

				gchar *next_p;
				while (*p && isspace((unsigned char) *p)) p++;

				t1 = strtol(p, &next_p, 0);
				if (next_p == p) {
					i--;
					break;
				}
				p = next_p;

			kanjiguess[i] = t1;

			while (*p && !isspace((unsigned char) *p)) p++;
			}
			num_guesses = i+1;
			put_guesses(context);
    }

    g_free(line);

    return TRUE;
}

/* Open the connection to the engine */
static gboolean
init_engine(IMJAContext *context)
{
  gchar *argv[] = { IM_JA_LIBPROGDIR G_DIR_SEPARATOR_S "kpengine", "--data-file", NULL, NULL };
  GError *err = NULL;
  gchar *uninstalled;
  int stdin_fd, stdout_fd;

  uninstalled = g_build_filename (".", "kpengine", NULL);
  if (g_file_test (uninstalled, G_FILE_TEST_EXISTS))
    argv[0] = uninstalled;

  if (data_file)
    argv[2] = data_file;
  else
    argv[1] = NULL;

  if (!g_spawn_async_with_pipes (NULL, /* working directory */
				 argv, NULL,	/* argv, envp */
				 0,
				 NULL, NULL,	/* child_setup */
				 &engine_pid,   /* child pid */
				 &stdin_fd, &stdout_fd, NULL,
				 &err))
    {
      /*GtkWidget *dialog;

      dialog = gtk_message_dialog_new (NULL, 0,
				       GTK_MESSAGE_ERROR,
				       GTK_BUTTONS_OK,
				       "Could not start engine '%s': %s",
				       argv[0], err->message);
      gtk_dialog_run (GTK_DIALOG (dialog));
	  */
      im_ja_print_error("Could not start engine '%s': %s", argv[0], err->message);
      g_error_free (err);
      return FALSE;
    }

  g_free (uninstalled);
  
  if (!(to_engine = g_io_channel_unix_new (stdin_fd)))
  {
    im_ja_print_error("Couldn't create pipe to child process: %s", g_strerror(errno));
    return FALSE;
  }
  if (!(from_engine = g_io_channel_unix_new (stdout_fd)))
  {
    im_ja_print_error("Couldn't create pipe from child process: %s", g_strerror(errno));
    return FALSE;
  }

  g_io_add_watch (from_engine, G_IO_IN, (GIOFunc) engine_input_handler, context);
  return TRUE;
}

  
/* Create Interface */
GtkWidget *kanjipad_window_create(IMJAContext *context, gint width, gint height) {
  GtkWidget *window;
  GtkWidget *main_hbox;
  GtkWidget *vseparator;
  GtkWidget *button;
  GtkWidget *main_vbox;
  GtkWidget *border;
  GtkWidget *vbox;
  
	window = gtk_window_new(GTK_WINDOW_POPUP);
  context->kanjipad = window;
  gtk_window_set_default_size(GTK_WINDOW(window), width, height);

  border = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(border), GTK_SHADOW_OUT);
  gtk_container_add(GTK_CONTAINER(window), border);
  gtk_widget_show(border);

  main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (border), main_vbox);
  gtk_widget_show (main_vbox);

  main_hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX(main_vbox), main_hbox, TRUE, TRUE, 0);
  gtk_widget_show (main_hbox);

  /* Area for user to draw characters in */

  pad_area = pad_area_create ();

  gtk_box_pack_start (GTK_BOX (main_hbox), pad_area->widget, TRUE, TRUE, 0);
  gtk_widget_show (pad_area->widget);

  vseparator = gtk_vseparator_new();
  gtk_box_pack_start (GTK_BOX (main_hbox), vseparator, FALSE, FALSE, 0);
  gtk_widget_show (vseparator);
  
  /* Buttons */
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (main_hbox), vbox, FALSE, FALSE, 0);
  gtk_widget_show (vbox);

  lookup_button = button = gtk_button_new_with_label("\xe5\xbc\x95");
  g_signal_connect (button, "clicked", G_CALLBACK (look_up_callback), NULL);

  gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);

  clear_button = button = gtk_button_new_with_label("\xe6\xb6\x88");
  g_signal_connect (button, "clicked", G_CALLBACK (clear_callback), NULL);

  gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);


  if(!init_engine(context) )
  {
	  gtk_widget_destroy(window);
	  context->kanjipad = 0;
	  window=0;
  }

  return window;
}

void kanjipad_disable_focus_out(GtkWidget *widget, GdkEvent *event, IMJAContext* cn) {
	IM_JA_DEBUG("kanjipad_disable_focus_out()\n");
	kanjipad_focus_out_disabled = TRUE;
}

void kanjipad_show(IMJAContext *cn) {

	IM_JA_DEBUG("kanjipad_show()\n");
	if (candidate_window_is_shown(cn) == TRUE) return; /* don't show if the candidate window is shown */
	if (cn->show_first == FALSE) return; /* don't show until we get focus once */
	im_ja_context_reset(cn);

	if (cn->kanjipad == NULL) {
		int w = 80, h = 70;
		if (cfg.kanjipad_custom_size == TRUE) {
			w = cfg.kanjipad_size[0];
			h = cfg.kanjipad_size[1];
		}
		if (kanjipad_window_create(cn, w, h) != NULL) {
			im_ja_join_modal_window(cn, cn->kanjipad);
			/* This hack is to disable flashy behaviour when the window gets the focus*/
			g_signal_connect(GTK_OBJECT(cn->kanjipad), "enter_notify_event", 
											 GTK_SIGNAL_FUNC(kanjipad_disable_focus_out), cn);

			kanjipad_set_location(cn);
		}
		else {
			im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		}
	}
	
	if (cn->kanjipad != NULL) {
		gtk_widget_show_all(cn->kanjipad);		
	}
}

void kanjipad_hide(IMJAContext *cn) {
	if (cn->kanjipad != NULL) {
		IM_JA_DEBUG("hide kanjipad\n");
		gtk_widget_hide_all(cn->kanjipad);
	}
}


void kanjipad_set_location(IMJAContext *cn) {
	gint kanjipad_w, kanjipad_h;
	GdkRectangle client_rect;
	gint target_x = 0, target_y = 0;
	GdkRectangle toplevel_rect;
			
	if (cn->kanjipad !=  NULL) {
		if ((cn->cursor_pos_offset_x == 0) && (cn->cursor_pos_offset_y == 0)) {
			im_ja_center_on_client_win(cn, GTK_WINDOW(cn->kanjipad));
		}
		else {
			im_ja_get_client_window_geometry(cn, &client_rect);
			target_x = client_rect.x + cn->cursor_pos_offset_x;
			target_y = client_rect.y + cn->cursor_pos_offset_y;

			/* check if target_ is within the toplevel window */
			im_ja_get_toplevel_window_geometry(cn, &toplevel_rect);
			im_ja_move_within_rect(cn, &target_x, &target_y, &client_rect);

			gtk_window_get_size(GTK_WINDOW(cn->kanjipad), &kanjipad_w, &kanjipad_h);
			gtk_window_move((GTK_WINDOW(cn->kanjipad)), target_x - kanjipad_w/3, target_y);

		}

	}
}

