/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 * Based on code by:
 *          Owen Taylor <otaylor@redhat.com>
 *          Yusuke Tabata <tee@kuis.kyoto-u.ac.jp>
 *          Yukihiro Nakai <ynakai@redhat.com>
 *          
 */
#include <config.h>

#include <string.h>
#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#include "im-ja.h"
#include "im-ja-impl.h"
#include "common.h"
#include "error.h"
#include "conf.h"
#include "nls.h"
#include "romakana.h"
#include "statuswin.h"
#include "preeditwin.h"
#include "kanjipad/kanjipad.h"
#include "candwin.h"
#include "symbols.h"
#include "helper/helper-client.h"
#include "radical.h"

#ifndef DISABLE_CANNA
# include "canna/canna_rk.h"
#endif
#ifndef DISABLE_WNN
# include "wnn/wnn.h"
#endif
#ifndef DISABLE_ANTHY
# include "anthy/anthy.h"
#endif
#ifndef DISABLE_SKK
# include "skk/skk.h"
#endif

extern GConfClient *gconf_client;
GType type_im_ja_context = 0;

enum {
	STRING_COLUMN,
	N_COLUMNS
};

extern IMJAConfig cfg; 
gboolean kanjipad_focus_out_disabled;
gint requested_input_method;
gboolean im_changed_by_applet;
ClientIO *helper_client = NULL;
GList *context_list = NULL;

/* * * * * * * * * * * */

static void im_ja_shutdown_conversion_engine(IMJAContext *cn);
gboolean im_ja_helper_input_handler(GIOChannel *source,
																		GIOCondition condition,
																		ClientIO *client);
/* * * * * * * * * * * */

void im_ja_simple_context_commit(GtkIMContext *imcontext, gchar *string, IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_simple_context_commit:%s\n", string);
	g_snprintf(cn->preedit_buf, BUFFERSIZE, "%s", string);
	im_ja_commit(cn);
}

IMJAContext *im_ja_context_new(void) {
  IMJAContext *context;
	IM_JA_DEBUG("im_ja_context_new()\n");
  context = IM_JA_CONTEXT(g_object_new(TYPE_IM_JA_CONTEXT, NULL));
	IM_JA_DEBUG("im_ja_context_new() [%d]\n", (int) context);
  return context;
}


/* Called for every input widget instance */
void im_ja_context_init(IMJAContext *cn) {

	IM_JA_DEBUG("im_ja_context_init()\n");
	cn->finalized = FALSE;
	cn->kanjipad = NULL;
	cn->preedit_buf = g_new0(gchar, BUFFERSIZE);
	cn->preedit_insert = g_new0(gchar, BUFFERSIZE);
	cn->status_win = NULL;
	cn->preedit_win = NULL;
	cn->preedit_reverse_start = 0;
	cn->preedit_reverse_end = 0;
	cn->cursor_ndx = -1;
	cn->show_first = FALSE;
	cn->candwin_pos_offset_x = 0;
  cn->candwin_pos_offset_y = 0;
  cn->preedit_win_pos_offset_x = 0;
  cn->preedit_win_pos_offset_y = 0;
	cn->cursor_pos_offset_x = 0;
	cn->cursor_pos_offset_y = 0;
  cn->update_candwin_pos = FALSE;
  cn->update_preedit_pos = FALSE;

	cn->preedit_win_on = cfg.preedit_window_on;
	cn->candwin_style = cfg.candwin_style;

	cn->simple_context = gtk_im_context_simple_new();
	g_signal_connect(G_OBJECT(cn->simple_context), "commit",
                   G_CALLBACK(im_ja_simple_context_commit), cn);


	cn->conv_engine_initialized = FALSE;
	cn->conv_engine = cfg.default_conv_engine;
	IM_JA_DEBUG("CONV ENGINE: %d\n", cn->conv_engine);

	context_list = g_list_append(context_list, cn);

	if (helper_client == NULL) {
		helper_client = helper_client_io_new_connection(TRUE);
		if (helper_client != NULL) {
			helper_client->watch_id = g_io_add_watch(helper_client->io,
																							 G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
																							 (GIOFunc) im_ja_helper_input_handler, helper_client);
		}
	}


	im_ja_set_input_method(cn, cfg.startup_input_method);
}

void im_ja_context_class_finalize(GObject *obj) {
	IMJAContext *cn = IM_JA_CONTEXT(obj);

	/*
#ifdef IMJA_TARGET_GTK
	GtkIMContextClass *im_context_class = GTK_IM_CONTEXT_GET_CLASS(obj);
	
	im_context_class->set_client_window = NULL;
	im_context_class->filter_keypress = NULL;
	im_context_class->get_preedit_string = NULL;
	im_context_class->set_cursor_location = NULL;
	im_context_class->focus_in = NULL;
	im_context_class->focus_out = NULL;
	im_context_class->reset = NULL;
	im_context_class->set_use_preedit = NULL;

#endif
*/
	IM_JA_DEBUG("im_ja_context_class_finalize() [%d]\n", (int)obj);

	im_ja_context_destroy(cn);
	
}


#ifndef IMJA_TARGET_GTK
/* This is called only once for the application */
static void im_ja_context_class_init(IMJAContextClass *class) {
	GObjectClass *parent_class = G_OBJECT_CLASS(class);

	IM_JA_DEBUG("im_ja_class_init\n");

	parent_class->finalize = im_ja_context_class_finalize;

}

static void im_ja_context_finalize(IMJAContext *context) {
	IM_JA_DEBUG("im_ja_context_finalize()  [%d]\n", (int)context);
	status_window_destroy_all();
	preedit_window_destroy_all();
}

GType im_ja_context_get_type(void) {

	IM_JA_DEBUG("im_ja_context_get_type\n");

  if (type_im_ja_context == 0) {
		static const GTypeInfo object_info = {
			sizeof (IMJAContextClass),
			(GBaseInitFunc) NULL,                      /* base_init */
			(GBaseFinalizeFunc) im_ja_context_finalize,/* base_finalize */
			(GClassInitFunc) im_ja_context_class_init, /* class init */
			NULL,                                      /* class_finalize */
			NULL,						                           /* class_data */
			sizeof (IMJAContext),
			0,                                         /* n_preallocs */
			(GInstanceInitFunc) im_ja_context_init,
		};
		IM_JA_DEBUG(" registering new g_type\n");

		type_im_ja_context = g_type_register_static(G_TYPE_OBJECT, "IMJAContext", &object_info, 0);
	}

  return type_im_ja_context;
}
#endif

void im_ja_destroy_helper_client(ClientIO *client) {
	IM_JA_DEBUG("destroy_client: %d\n", (int) client);
	if (client != NULL) {
		helper_client_io_close(client);
		client = NULL;
	}
}

void im_ja_process_helper_message(ClientIO *client, gchar *msg) {
	gchar *ptr;
	gint input_method = -1;

	if (msg == NULL) return;
	if (client == NULL) return;

	msg[strlen(msg) - 1] = 0; /* remove trailing newline */
	IM_JA_DEBUG("processing [%s]\n", msg);

	if (g_str_has_prefix(msg, HELPER_MSG_PING) == TRUE) {
		helper_client_io_send(client, HELPER_MSG_ACK);
	}
	else if (g_str_has_prefix(msg, HELPER_MSG_SET_INPUT_METHOD) == TRUE) {
		GList *cn_list = NULL;
		IMJAContext *cn = NULL;
		ptr = msg + strlen(HELPER_MSG_SET_INPUT_METHOD);
		input_method = atoi(ptr);
		IM_JA_DEBUG("Input method change request to: %d\n", input_method);
		requested_input_method = input_method;
		im_changed_by_applet = TRUE;
		cn_list = context_list;
		while (cn_list != NULL) {
			cn = (IMJAContext *) cn_list->data;
			if (cn->has_focus == TRUE) {
				IM_JA_DEBUG("setting input method on focused context.\n");
				im_ja_set_input_method(cn, input_method);
			}
			cn_list = cn_list->next;
		}
		helper_client_send_command(client, HELPER_MSG_CHANGE_STATUS, input_method);
	}
	else {
		IM_JA_DEBUG("Unprocessed helper message: %s\n", msg);
	}
}



gboolean im_ja_helper_input_handler(GIOChannel *source,
																		GIOCondition condition,
																		ClientIO *client) {
	GIOStatus status;
	gchar *line = NULL;
	GError *err = NULL;

	IM_JA_DEBUG("im_ja_helper_input_handler()\n");

	if (condition & G_IO_ERR) {
		IM_JA_DEBUG("IO Error\n");
		im_ja_destroy_helper_client(client);
		return FALSE;
	}

	if (condition & G_IO_IN) {
		IM_JA_DEBUG("data input:\n");

		status = g_io_channel_read_line(source, &line, NULL, NULL, &err);

		if (status & G_IO_STATUS_ERROR) {
			g_error("Error reading from client: %s\n", err->message);
			im_ja_destroy_helper_client(client);
			return FALSE;
		}

		else if (status & G_IO_STATUS_EOF) {
			IM_JA_DEBUG("EOF\n");
			im_ja_destroy_helper_client(client);
			return FALSE;
		}
		else if (status & G_IO_STATUS_NORMAL) {
			IM_JA_DEBUG("G_IO_STATUS_NORMAL\n");
			if ((line == NULL) || (strlen(line) == 0)) {
				IM_JA_DEBUG("connection closed\n");
				im_ja_destroy_helper_client(client);
				return FALSE;
			}
			else { /* OK */
				im_ja_process_helper_message(client, line);
				g_free(line);
				line = NULL;
				return TRUE;
			}
		}
		else if (status & G_IO_STATUS_AGAIN) {
			IM_JA_DEBUG("G_IO_STATUS_AGAIN\n");
			return TRUE;
		}
		return TRUE;
	}

	else if (condition & G_IO_ERR) {
		IM_JA_DEBUG("io error\n");
		im_ja_destroy_helper_client(client);
		return FALSE;
	}
	else if (condition & G_IO_HUP) {
		IM_JA_DEBUG("disconnection\n");
		im_ja_destroy_helper_client(client);
		return FALSE;
	}
	else if (condition & G_IO_NVAL) {
		IM_JA_DEBUG("invalid request, file descriptor is closed.\n");
		im_ja_destroy_helper_client(client);
		return FALSE;
	}
	return FALSE;
}



void im_ja_run_configurator() {
	static gchar *command[] = {IM_JA_BINDIR"/im-ja-conf"};
	GError *g_error = NULL;
	if (g_spawn_async(NULL, command, NULL, G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_STDERR_TO_DEV_NULL,
										NULL, NULL, NULL, &g_error) == FALSE) {
		im_ja_print_error(g_error->message); 
	}
}

void im_ja_set_use_preedit(IMJAContext *cn, gboolean use_preedit) {
	IM_JA_DEBUG("im_ja_set_use_preedit: %d\n", use_preedit);
	if (cfg.preedit_window_on != TRUE) {
		cn->preedit_win_on = !use_preedit;
	}
}


gboolean im_ja_filter_keypress(IMJAContext *cn, GdkEventKey *key) {
	gboolean retval = FALSE;
	gchar utf8strg[7];
	gint utf8strg_end;
	gchar *tmpstrg;

	if (key->type == GDK_KEY_RELEASE) return FALSE;

	IM_JA_DEBUG(" Gkeystate: %d\n", key->state);
	IM_JA_DEBUG(" Gkeyval: %d\n", key->keyval);
	IM_JA_DEBUG(" Gkeycode: %d\n", key->hardware_keycode);


	/* Discard NoSymbol keyevents */
	if ((key->keyval == 0) && (key->state == 0)) return FALSE;
		
	IM_JA_DEBUG("GOT KEYEVENT. keyname: %s, keyval : %d,	state: %d\n", 
							gdk_keyval_name(key->keyval), key->keyval, key->state);
	

	if (ishotkey(key, START_CONFIGURATOR, &cfg) == TRUE) {
		im_ja_run_configurator();
		return TRUE;
	}
	if (ishotkey(key, NEXT_INPUT_MODE, &cfg) == TRUE) {
		im_ja_next_input_method(cn);
		return TRUE;
	}
	if (ishotkey(key, PREV_INPUT_MODE, &cfg) == TRUE) {
		im_ja_prev_input_method(cn);
		return TRUE;
	}
	if (ishotkey(key, DIRECT_INPUT_MODE, &cfg) == TRUE) {
		im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		return TRUE;
	}
	if (ishotkey(key, HIRAGANA_INPUT_MODE, &cfg) == TRUE) {
		/* if it's already hiragana, then set to katakana */
		if (cn->input_method == IM_JA_HIRAGANA_INPUT) im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		else im_ja_set_input_method(cn, IM_JA_HIRAGANA_INPUT);
		return TRUE;
	}
	if (ishotkey(key, KATAKANA_INPUT_MODE, &cfg) == TRUE) {
		/* if it's already katakana, then set to half-width katakana */
		if (cn->input_method == IM_JA_KATAKANA_INPUT) im_ja_set_input_method(cn, IM_JA_HALFKATA_INPUT);
		else im_ja_set_input_method(cn, IM_JA_KATAKANA_INPUT);
		return TRUE;
	}
	if (ishotkey(key, HALFKATA_INPUT_MODE, &cfg) == TRUE) {
		/* if it's already half-katakana, then set to full katakana */
		if (cn->input_method == IM_JA_HALFKATA_INPUT) im_ja_set_input_method(cn, IM_JA_KATAKANA_INPUT);
		else im_ja_set_input_method(cn, IM_JA_HALFKATA_INPUT);
		return TRUE;
	}
	if (ishotkey(key, ZENKAKU_INPUT_MODE, &cfg) == TRUE) {
		/* if it's already zenkaku, then set to DIRECT */
		if (cn->input_method == IM_JA_ZENKAKU_INPUT) im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		else im_ja_set_input_method(cn, IM_JA_ZENKAKU_INPUT);
		return TRUE;
	}
	/*
	if (ishotkey(key, CANNA_INPUT_MODE, &cfg) == TRUE) {
		im_ja_context_reset(cn);
		im_ja_shutdown_conversion_engine(cn);
		cfg.default_conv_engine = CONV_ENGINE_CANNA;
		cn->conv_engine = cfg.default_conv_engine;
		return TRUE;
	}
	if (ishotkey(key, WNN_INPUT_MODE, &cfg) == TRUE) {
		im_ja_context_reset(cn);
		im_ja_shutdown_conversion_engine(cn);
		cfg.default_conv_engine = CONV_ENGINE_WNN;
		cn->conv_engine = cfg.default_conv_engine;
		return TRUE;
	}
	*/
	if (ishotkey(key, KANJIPAD_INPUT_MODE, &cfg) == TRUE) {
		/* if it's already KANJIPAD, then set to DIRECT */
		if (cn->input_method == IM_JA_KANJIPAD_INPUT) im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		else im_ja_set_input_method(cn, IM_JA_KANJIPAD_INPUT);
		return TRUE;
	}
	if (ishotkey(key, TOGGLE_PREEDIT_WINDOW, &cfg) == TRUE) {
		cn->preedit_win_on = !cn->preedit_win_on;
		if (cn->preedit_win_on == TRUE) {
			status_window_force_hide(cn);
			preedit_window_show(cn);
		}
		else {
			preedit_window_force_hide(cn);
			status_window_show(cn);
		}
		return TRUE;
	}
	if (ishotkey(key, SYMBOL_INPUT, &cfg) == TRUE) {
		im_ja_symbol_table_show(cn);
	}
	if (ishotkey(key, UNICODE_INPUT, &cfg) == TRUE) {
		im_ja_unicode_entry_show(cn);
	}
	if (ishotkey(key, JIS_CODE_INPUT, &cfg) == TRUE) {
		im_ja_jiscode_entry_show(cn);
	}
	if (ishotkey(key, RADICAL_INPUT, &cfg) == TRUE) {
		im_ja_radtable_show(cn);
	}

	switch (cn->input_method) {
	case IM_JA_DIRECT_INPUT: 
	case IM_JA_KANJIPAD_INPUT:
		if (strlen(cn->preedit_buf) > 0) {
			im_ja_commit(cn);
		}

		if (gtk_im_context_filter_keypress(cn->simple_context, key) == TRUE) return TRUE;

#ifndef IMJA_TARGET_GTK /* don't do any emulation for xim/iiimf in DIRECT mode */
		return FALSE;
#endif

		/* if (key->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK | GDK_MOD3_MASK | GDK_MOD4_MASK)) return FALSE; */
		if (key->state & (GDK_CONTROL_MASK)) return FALSE; 
		if (key->keyval == GDK_space) {
			tmpstrg = g_strdup_printf(" %s", cn->preedit_buf);
			g_strlcpy(cn->preedit_buf, tmpstrg, BUFFERSIZE);
			im_ja_commit(cn);
			return TRUE;
		}

		if ((key->keyval == GDK_Escape) && (cn->input_method == IM_JA_KANJIPAD_INPUT)) {
			im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
			return TRUE;
		}

		if (im_ja_is_printable_key(key) == TRUE) { 
			utf8strg_end = g_unichar_to_utf8(gdk_keyval_to_unicode(key->keyval), utf8strg);
			utf8strg[utf8strg_end] = 0;
			if (strlen(utf8strg) > 0) {
				if (((guchar)utf8strg[0] >= 0x20) && (utf8strg[0] != 0x7f)) {
					g_snprintf(cn->preedit_buf, BUFFERSIZE, "%s", utf8strg);
					im_ja_commit(cn);
				}
				return TRUE;
			}
		}
		else {
			IM_JA_DEBUG("Not a printable key.\n");
		}
		return FALSE;

	case IM_JA_HIRAGANA_INPUT:
	case IM_JA_KATAKANA_INPUT:
	case IM_JA_HALFKATA_INPUT:
	case IM_JA_ZENKAKU_INPUT:
		if (cn->conv_engine_initialized == FALSE) {
			if (im_ja_init_conversion_engine(cn) == FALSE) return FALSE; /* Don't handle the key if we cannot init the engine. */
		}

		if (cn->im_ja_conv_engine_filter_keypress != NULL) {
			retval = cn->im_ja_conv_engine_filter_keypress(cn, key);
		}
		else { /* Unknown conv engine */
			retval = im_ja_kana_filter_keypress(cn, key);
		}
	}

	/* ESC pressed: drop current pre-edit buffer if this keypress wasn't handled */
	if ((key->keyval == GDK_Escape) && (retval == FALSE)) {
		IM_JA_DEBUG("ESC: cancel preedit\n");
		IM_JA_DEBUG("preedit_insert: %d\n", strlen(cn->preedit_insert));
		IM_JA_DEBUG("preedit_buf: %d\n", strlen(cn->preedit_buf));
		if ((strlen(cn->preedit_buf) > 0) || (strlen(cn->preedit_insert) > 0)) { /* Cancel preedit buffer */
			im_ja_context_reset(cn);
			if (cn->im_ja_conv_engine_reset_context != NULL) cn->im_ja_conv_engine_reset_context(cn);
		}
		else {
			im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		}
		return TRUE;
	}

	return retval;
}

void im_ja_set_preedit_string(IMJAContext *cn, gchar *str, gint reverse_start, gint reverse_end) {
	if (str == NULL) memset(cn->preedit_buf, 0, BUFFERSIZE);
	else g_strlcpy(cn->preedit_buf, str, BUFFERSIZE);
	cn->preedit_reverse_start = reverse_start;
	cn->preedit_reverse_end = reverse_end;
}

gint im_ja_get_cursor_pos_chars(IMJAContext *cn) {
		
	if (cn->conv_engine == CONV_ENGINE_CANNA) {
		if (cn->cursor_ndx == -1) {
			return strlen(cn->preedit_buf);
		}
		else {
			/* convert byte index into character index */
			return g_utf8_strlen(cn->preedit_buf, cn->cursor_ndx);
		}
	}
	else {
		return cn->cursor_char_pos;
	}

}

gint im_ja_get_cursor_pos_bytes(IMJAContext *cn) {
		
	if (cn->conv_engine == CONV_ENGINE_CANNA) {
		if (cn->cursor_ndx == -1) {
			return strlen(cn->preedit_buf);
		}
		else {
			return cn->cursor_ndx;
		}
	}
	else {
		gchar *str;

		str = g_utf8_offset_to_pointer(cn->preedit_buf, cn->cursor_char_pos);
		return str - cn->preedit_buf;
	}

}


/* this doesn't free the context memory itself! */
void im_ja_context_destroy(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_context_destroy(()\n");

	im_ja_context_impl_destroy(cn);
 	
	cn->finalized = TRUE;

	preedit_window_destroy(cn);
	cn->preedit_win = NULL;
	status_window_destroy(cn);
	cn->status_win = NULL;
	if (cn->kanjipad != NULL) gtk_widget_destroy(cn->kanjipad);
	if ((cn->candidate_win != NULL) && GTK_IS_WIDGET(cn->candidate_win->window)) {
		g_signal_handler_disconnect(cn->candidate_win->window, cn->candidate_win->destroy_handler);
	}
	candidate_window_destroy(cn);

	im_ja_shutdown_conversion_engine(cn);
	if (cn->preedit_buf != NULL) {
		g_free(cn->preedit_buf);
		cn->preedit_buf = NULL;
	}
	if (cn->preedit_insert != NULL) {
		g_free(cn->preedit_insert);
		cn->preedit_insert = NULL;
	}

	g_object_unref(cn->simple_context);
	cn->simple_context = NULL;

	context_list = g_list_remove(context_list, cn);

}

void im_ja_free_candidate_list(IMJAContext *cn) {
	GList *tmplist;
	
	tmplist = cn->candidate_list;
	if (tmplist == NULL) return;

	while (tmplist != NULL) {
		g_free(tmplist->data);
		tmplist = g_list_next(tmplist);
	}
	g_list_free(cn->candidate_list);
	cn->candidate_list = NULL;
}

void im_ja_context_reset(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_context_reset()\n");

	*cn->preedit_insert = 0;

	if (cn->preedit_buf == NULL) cn->preedit_buf = g_new0(gchar, BUFFERSIZE);
	if (cn->preedit_insert == NULL) cn->preedit_insert = g_new0(gchar, BUFFERSIZE);

	memset(cn->preedit_buf, 0, BUFFERSIZE); 

	cn->preedit_reverse_start = 0;
	cn->preedit_reverse_end = 0;
	cn->cand_stat = 0;
	cn->cursor_ndx = -1;
	cn->cursor_char_pos = 0;
	im_ja_preedit_changed(cn);
	cn->candwin_pos_offset_x = cn->cursor_pos_offset_x; /* Cursor position does not always change */
	cn->candwin_pos_offset_y = cn->cursor_pos_offset_y; /* so we need to update manually */
	gtk_im_context_reset(cn->simple_context);
}


void im_ja_cursor_location_changed(IMJAContext *cn, gint x, gint y) {

	if (((strlen(cn->preedit_buf) == 0) || (cn->update_preedit_pos == TRUE)) 
			&& ((cn->preedit_win_pos_offset_x != x) || (cn->preedit_win_pos_offset_y != y))) {
		cn->preedit_win_pos_offset_x = x;
		cn->preedit_win_pos_offset_y = y;
		cn->update_preedit_pos = FALSE;
		if (cn->preedit_win_on == TRUE) preedit_window_update_location(cn);
	}			
	
	/* IM_JA_DEBUG("im_ja_cursor_location_changed: %d, %d\n", x, y); */
	if ((cn->cursor_pos_offset_x != x) || (cn->cursor_pos_offset_y != y)) {
		if ((strlen(cn->preedit_buf) == 0) || (cn->update_candwin_pos == TRUE)) {
			cn->candwin_pos_offset_x = x;
			cn->candwin_pos_offset_y = y;
			cn->update_candwin_pos = FALSE;
			/* IM_JA_DEBUG("candwin cursor_location_changed(%d, %d)\n", x, y); */
		}
		cn->cursor_pos_offset_x = x;
		cn->cursor_pos_offset_y = y;
		/* IM_JA_DEBUG("cursor_location_changed(%d, %d)\n", x, y); */
		kanjipad_set_location(cn);
		if (cn->preedit_win_on == TRUE) preedit_window_update_location(cn);
		else status_window_update_location(cn);
	}
}

void im_ja_lost_focus(IMJAContext *cn) {
	IM_JA_DEBUG("im-ja lost focus\n");

	preedit_window_hide(cn);
	status_window_hide(cn);

	if (kanjipad_focus_out_disabled != TRUE) {
		kanjipad_hide(cn);
	}
	cn->has_focus = FALSE;
}

void im_ja_got_focus(IMJAContext *cn) {
	IM_JA_DEBUG("im-ja got focus\n");
	
	cn->has_focus = TRUE;

	if (im_changed_by_applet == TRUE) {
		im_changed_by_applet = FALSE;
		im_ja_set_input_method(cn, requested_input_method);
	}

	/* Change status on the helper */
	helper_client_send_command(helper_client, HELPER_MSG_CHANGE_STATUS, cn->input_method);

	/* if (cn->status_window != NULL) gtk_grab_remove(cn->status_window); */
	if (cn->status_win != NULL) cn->status_win->can_hide = TRUE;
	if (cn->preedit_win != NULL) cn->preedit_win->can_hide = TRUE;
	if (cn->show_first == FALSE) {
		cn->show_first = TRUE;
	}

	if (cn->preedit_win_on == TRUE) {
		preedit_window_show(cn);
	}
	else {
		status_window_show(cn);
	}
	kanjipad_set_location(cn);
	kanjipad_focus_out_disabled = FALSE;
	if (cn->input_method == IM_JA_KANJIPAD_INPUT) {
		kanjipad_show(cn);
	}
}

void im_ja_on_reset(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_on_reset()\n");

	if (cfg.commit_on_reset == FALSE) return;
 
	cn->preedit_win_pos_offset_x = cn->cursor_pos_offset_x;
	cn->preedit_win_pos_offset_y = cn->cursor_pos_offset_y;

	if (cn->preedit_buf == NULL) return;
	if (strlen(cn->preedit_buf) > 0) {
		im_ja_commit(cn);		
		if (cn->im_ja_conv_engine_reset_context != NULL) cn->im_ja_conv_engine_reset_context(cn);
	}
}

static void im_ja_shutdown_conversion_engine(IMJAContext *cn) {
	IM_JA_DEBUG("shutdown_conversion_engine()\n");
	if (cn->conv_engine_initialized == TRUE) {
		cn->conv_engine_initialized = FALSE;
		if (cn->im_ja_conv_engine_shutdown != NULL) {
			cn->im_ja_conv_engine_shutdown(cn);
		}
	}
	cn->im_ja_conv_engine_reset_context = NULL;
	cn->im_ja_conv_engine_filter_keypress = NULL;
	cn->im_ja_conv_engine_shutdown = NULL;
	cn->im_ja_conv_engine_select_candidate = NULL;
	cn->im_ja_conv_engine_update_preedit = NULL;
	cn->im_ja_conv_engine_commit = NULL;

}

static void im_ja_conv_engine_init_failed(IMJAContext *cn) {
	/*im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);*/

	/* Set kana input (no conversion) */
	cn->im_ja_conv_engine_filter_keypress = im_ja_kana_filter_keypress;
	cn->conv_engine_initialized = TRUE;
	cn->conv_engine = CONV_ENGINE_NONE;
}

gboolean im_ja_init_conversion_engine(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_init_conversion_engine: %d\n", cn->conv_engine);

	if (cn->conv_engine_initialized == TRUE) return TRUE;
	if (cn->conv_engine == CONV_ENGINE_WNN) {
#ifndef DISABLE_WNN
		if (im_ja_wnn_init(cn) != TRUE) {
			im_ja_conv_engine_init_failed(cn);
			return FALSE;
		} 
		else cn->conv_engine_initialized = TRUE;
#else
		im_ja_print_error(_("Wnn support is disabled. If you want wnn support, recompile im-ja."));
		im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		return FALSE;
#endif
	}
	else if (cn->conv_engine == CONV_ENGINE_CANNA) {
#ifndef DISABLE_CANNA
		if (canna_rk_init(cn) != TRUE) {
			im_ja_print_error(_("canna init failed.\nPlease check your settings and make sure the canna server is running!")); 
			im_ja_conv_engine_init_failed(cn);
			return FALSE;
		}
		else cn->conv_engine_initialized = TRUE;
#else
		im_ja_print_error(_("Canna support is disabled. If you want canna support, recompile im-ja."));
		im_ja_conv_engine_init_failed(cn);
		return FALSE;
#endif
	}
	if (cn->conv_engine == CONV_ENGINE_ANTHY) {
#ifndef DISABLE_ANTHY
		if (im_ja_anthy_init(cn) != TRUE) {
			im_ja_print_error(_("anthy init failed.")); 
			im_ja_conv_engine_init_failed(cn);
			return FALSE;
		} 
		else cn->conv_engine_initialized = TRUE;
#else
		im_ja_print_error(_("Anthy support is disabled. If you want anthy support, recompile im-ja."));
		im_ja_conv_engine_init_failed(cn);
		return FALSE;
#endif
	}
	else if (cn->conv_engine == CONV_ENGINE_SKK) {
#ifndef DISABLE_SKK
		if (im_ja_skk_init(cn) != TRUE) {
			im_ja_print_error(_("SKK init failed.\nPlease check your settings and make sure \'skkserv\' is running at localhost:1178!")); 
			im_ja_conv_engine_init_failed(cn);
			return FALSE;
		} 
		else cn->conv_engine_initialized = TRUE;
#else
		im_ja_print_error(_("SKK support is disabled. If you want SKK support, recompile im-ja."));
		im_ja_conv_engine_init_failed(cn);
		return FALSE;
#endif
	}

	return TRUE;
}

void im_ja_move_within_rect(IMJAContext *cn, gint *target_x, gint *target_y, GdkRectangle *rect) {
	/*
	//IM_JA_DEBUG("im_ja_move_within_rect\n");

	//IM_JA_DEBUG("	RECT: x: %d, y: %d, width: %d, height: %d\n", rect->x, rect->y, rect->width, rect->height);
	//IM_JA_DEBUG("	TARGET: x: %d, y: %d\n", *target_x, *target_y);
	*/

	if ((rect->width != 0) || (rect->height != 0)) {
		if (*target_y > rect->y + rect->height) *target_y = rect->y + rect->height;
		if (*target_x > rect->x + rect->width) *target_x = rect->x + rect->width;
		if (*target_y < rect->y) *target_y = rect->y;
		if (*target_x < rect->x) *target_x = rect->x;

		/* IM_JA_DEBUG(" move to: x: %d, y: %d\n", *target_x, *target_y); */
	}
}

gboolean im_ja_execute_action(IMJAContext *cn, gint action_id, gboolean set_input_method) {
	IM_JA_DEBUG("im_ja_execute_action[immodule]\n");
	if (set_input_method == TRUE) {
		im_ja_set_input_method(cn, action_id);
		return TRUE;
	}
	switch (action_id) {
	case START_CONFIGURATOR:
		im_ja_run_configurator();
		return TRUE;
	case TOGGLE_PREEDIT_WINDOW:
		cn->preedit_win_on = !cn->preedit_win_on;
		if (cn->preedit_win_on == TRUE) {
			status_window_force_hide(cn);
			preedit_window_show(cn);
		}
		else {
			preedit_window_force_hide(cn);
			status_window_show(cn);
		}
		return TRUE;
	case SYMBOL_INPUT:
		im_ja_symbol_table_show(cn);
		return TRUE;
	case UNICODE_INPUT:
		im_ja_unicode_entry_show(cn);
		return TRUE;
	case JIS_CODE_INPUT:
		im_ja_jiscode_entry_show(cn);
		return TRUE;
	case RADICAL_INPUT:
		im_ja_radtable_show(cn);
		return TRUE;
	default:
		return FALSE;
	}
	return FALSE;
}

gboolean im_ja_set_input_method(IMJAContext *cn, gint input_method) {
	gint old_input_method;
	IM_JA_DEBUG("setting input method: %d\n", input_method);

	old_input_method = cn->input_method;

	cn->input_method = input_method;
	candidate_window_hide(cn);

	helper_client_send_command(helper_client, HELPER_MSG_CHANGE_STATUS, input_method);

	if (cn->input_method != IM_JA_KANJIPAD_INPUT)	kanjipad_hide(cn);

	switch (cn->input_method) {
	case IM_JA_KANJIPAD_INPUT:
		if (cfg.kanjipad_enabled == TRUE) {
			if (input_method != old_input_method) {
				im_ja_shutdown_conversion_engine(cn);
				if (strlen(cn->preedit_buf) > 0) im_ja_commit(cn);
			}
			/*cn->conv_engine = CONV_ENGINE_KANJIPAD;*/
			status_window_hide(cn);
			preedit_window_hide(cn);
			kanjipad_show(cn);
		}
		else {
			im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
			return FALSE;
		}
		break;
	case IM_JA_DIRECT_INPUT:
		if (input_method != old_input_method) {
			im_ja_on_reset(cn);
		}
		status_window_hide(cn);
		preedit_window_hide(cn);
		break;
	default: 
		cn->conv_engine = cfg.default_conv_engine;
		status_window_show(cn);
	}
	return TRUE;
}


void im_ja_next_input_method(IMJAContext *cn) {
	gint tmp;
	tmp = cn->input_method;
	tmp++;
	if ((tmp == IM_JA_KANJIPAD_INPUT) && (cfg.kanjipad_enabled == FALSE)) tmp++;
	if (tmp >= IM_JA_INPUT_METHODS_TOTAL) tmp = IM_JA_DIRECT_INPUT;
	im_ja_set_input_method(cn, tmp);
}

void im_ja_prev_input_method(IMJAContext *cn) {
	gint tmp;
	tmp = cn->input_method;
	tmp--;
	if ((tmp == IM_JA_KANJIPAD_INPUT) && (cfg.kanjipad_enabled == FALSE)) tmp--;
	if (tmp < IM_JA_DIRECT_INPUT) tmp = IM_JA_INPUT_METHODS_TOTAL - 1;
	im_ja_set_input_method(cn, tmp);
}
 
void im_ja_input_utf8(IMJAContext *cn, gchar *strg) {
	 
	if (strg == NULL) return;
	if ((cn->input_method != DIRECT_INPUT_MODE) || (cn->input_method != KANJIPAD_INPUT_MODE)) {
		im_ja_on_reset(cn);
	}
	IM_JA_DEBUG("im_ja_input_utf8(%s)\n", strg);
	g_snprintf(cn->preedit_buf, BUFFERSIZE, "%s", strg);
	im_ja_commit(cn);
	
}

void im_ja_center_on_client_win(IMJAContext *cn, GtkWindow *window) {
	GtkRequisition requisition;
	GdkRectangle rect;
	gint x, y;

	IM_JA_DEBUG("im_ja_center_on_client_win\n");

	im_ja_get_client_window_geometry(cn, &rect);
	gtk_widget_size_request(GTK_WIDGET(window), &requisition);
	x = rect.x + (rect.width - requisition.width) / 2;
	y = rect.y + (rect.height - requisition.height) / 2;

	gtk_window_move(GTK_WINDOW(window), x, y);
}

gboolean im_ja_is_cursor_over_window(GtkWidget *window) {
	gint x = 0, y = 0;
	gint size_x = 0, size_y = 0;

	IM_JA_DEBUG("im_ja_is_cursor_over_window()\n");
	if (GTK_IS_WINDOW(window) == FALSE) return FALSE;

	gtk_widget_get_pointer(window, &x, &y);
	IM_JA_DEBUG("POINTER: X[%d] Y[%d]\n", x, y);

	gtk_window_get_size(GTK_WINDOW(window), &size_x, &size_y);
	IM_JA_DEBUG("SIZE: X[%d] Y[%d]\n", size_x, size_y);
	if ((x < 0) || (x > size_x)) return FALSE;
	if ((y < 0) || (y > size_y)) return FALSE;
	IM_JA_DEBUG(" TRUE\n");
	return TRUE;
}

void im_ja_join_modal_window(IMJAContext *cn, GtkWidget *window) {
#ifdef IMJA_TARGET_GTK
	if (GTK_IS_WINDOW(window) == FALSE) return;
	if (cn->modal_grp == NULL) cn->modal_grp = gtk_window_group_new();
	gtk_window_group_add_window(cn->modal_grp, GTK_WINDOW(window));
#endif
}

