/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 */

#include <config.h>

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glade/glade.h>

#include "../nls.h"
#include "../im-ja.h"
#include "../conf.h"
#include "../error.h"
#include "about-box.h"


#define GETWIDGET(s) glade_xml_get_widget(gladexml, s)
#define ENUMSIZE(s) sizeof(s)/sizeof(gchar *)

static gboolean show_help();
static void checkbutton_simple_cb(GtkWidget *button, gboolean *cfgval);

static gchar *hotkey_labels[] = {
	N_("Start the im-ja configurator"),
	N_("Switch to next input mode"),
	N_("Switch to previous input mode"),
	N_("Set direct input mode"),
	N_("Toggle hiragana/direct mode"),
	N_("Toggle full/half-width katakana mode"),
	N_("Toggle half/full-width katakana mode"),
	N_("Toggle zenkaku/direct mode"),
	N_("Toggle canna/direct mode"),
	N_("Toggle wnn/direct input mode"),
	N_("Toggle kanjipad/direct input mode"),
	N_("Unconvert current bunsetsu"),
	N_("Convert current bunsetsu"),
	N_("Get previous candidate"),
	N_("Get next candidate"),
	N_("Show all candidates"),
	N_("Select previous bunsetsu"),
	N_("Select next bunsetsu"),
	N_("Expand current bunsetsu"),
	N_("Shrink current bunsetsu"),
	N_("Toggle preedit window"),
	N_("Commit preedit string"),
	N_("Symbol input"),
	N_("UniCode input"),
	N_("JIS Code input"),
	N_("Radical input"),
};


static GtkWidget *window;
static GtkWidget *popup_window = NULL;
static IMJAConfig cfg;
static GtkTreeModel *model;
static GtkTreeIter iter;
static GladeXML *gladexml;
static GtkWidget *aboutbox = NULL;
static gboolean settings_saved = FALSE;
extern gchar *hotkey_gconf_keys[];
extern GConfEnumStringPair status_win_labels[];

enum {
	HOTKEY_DESC,
	HOTKEY_KEYVAL,
	HOTKEY_IDX,
	NUM_COLS
};

enum {
	COLOR_NORMAL_FG,
	COLOR_NORMAL_BG,
	COLOR_HIGHL_FG,
	COLOR_HIGHL_BG
};


static gboolean check_duplicate_keys(gint keyval, gint state) {
	gint i;
	for (i = 0; i < (gint) (ENUMSIZE(hotkey_labels)); i++) {
		if (((gint)cfg.hotkey_values[i] == keyval) &&
				((gint)cfg.hotkey_states[i] == state))
			return FALSE;
	}
	return TRUE;
}


static void page_switch_cb(GtkNotebook *notebook,
													 GtkNotebookPage *page,
													 guint page_num,
													 gpointer user_data) {
	const gchar *label;
	label = gtk_notebook_get_tab_label_text(GTK_NOTEBOOK(notebook), GTK_WIDGET(gtk_notebook_get_nth_page(notebook, (gint)page_num)));
	IM_JA_DEBUG("Switched page to \"%s\"\n", label);
	if (strcmp(label, _("About")) == 0) {
		GtkWidget *parent = GETWIDGET("about_hbox");
		aboutbox = about_box_new();
		gtk_box_pack_start(GTK_BOX(parent), aboutbox, TRUE, TRUE, 10);
		gtk_box_reorder_child(GTK_BOX(parent), aboutbox, 1);
		gtk_widget_show(aboutbox);
	}
	else if (GTK_IS_WIDGET(aboutbox) == TRUE) {
		gtk_widget_destroy(aboutbox);
		aboutbox = NULL;
	}
		
}

static gboolean get_key(GtkWidget *widget, GdkEventKey *event, gpointer data) {
  guint state;
  guint keyval;
  gchar *keystr;
  gint idx;
  
  if (event != NULL) {
    if ((event)->type == GDK_KEY_RELEASE) {
      gdk_keyboard_ungrab(GDK_CURRENT_TIME);
      /* We don't want Num lock, caps lock and scroll lock */
      keyval = event->keyval;
      state = event->state;
      state &= ~GDK_LOCK_MASK; 
      state &= ~GDK_MOD2_MASK;
      state &= ~GDK_MOD5_MASK;
      gtk_tree_model_get(model, &iter, HOTKEY_IDX, &idx, -1);
      IM_JA_DEBUG("selected: %d\n", idx);
      if ((keyval == GDK_Return) || (keyval == GDK_Escape)) return FALSE;
			if (check_duplicate_keys(keyval, state) == FALSE) {
				if (!((cfg.hotkey_values[idx] == keyval) && (cfg.hotkey_states[idx] == state))) {
					keystr = im_ja_get_keyname(state, keyval);
					im_ja_print_error(_("%s is already bound to a different action"), keystr);
					g_free(keystr);
					gtk_widget_hide(popup_window);
					gtk_widget_destroy(popup_window);
					return FALSE;
				}
			}
      cfg.hotkey_values[idx] = keyval;
      cfg.hotkey_states[idx] = state;
      keystr = im_ja_get_keyname(state, keyval);
      IM_JA_DEBUG("GOT KEY: keyval = %d, keyval_name = %s, state = %d str = \"%s\" \n", keyval, gdk_keyval_name(keyval), state, keystr);
			gtk_list_store_set(GTK_LIST_STORE(model), &iter, HOTKEY_KEYVAL, keystr, -1);
      g_free(keystr);
    }
  }
  gtk_widget_hide(popup_window);
  gtk_widget_destroy(popup_window);
  return TRUE;
}

static void keygrab_dialog_response_cb(GtkDialog *dialog, gint response_id, gpointer user_data) {
	if (response_id >= 0) {
		gchar *keystr;
		cfg.hotkey_values[response_id] = 16777215; /* Undefined */
		cfg.hotkey_states[response_id] = 0;
		keystr = im_ja_get_keyname(0, 0);
		gtk_list_store_set(GTK_LIST_STORE(model), &iter, HOTKEY_KEYVAL, keystr, -1);
		g_free(keystr);
	}
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void show_keygrab_dialog(gint row) {
  GtkWidget *button_cancel;
  GtkWidget *button_undefine;

  popup_window = gtk_message_dialog_new(GTK_WINDOW(window), GTK_DIALOG_DESTROY_WITH_PARENT, 
																				GTK_MESSAGE_INFO, GTK_BUTTONS_NONE, _("waiting for keypress..."));
  gtk_window_set_modal(GTK_WINDOW(popup_window), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(popup_window), GTK_WINDOW(window));

  button_undefine = gtk_button_new_with_label(_("Undefine"));
  gtk_dialog_add_action_widget(GTK_DIALOG(popup_window), button_undefine, row);

  button_cancel = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  gtk_dialog_add_action_widget(GTK_DIALOG(popup_window), button_cancel, GTK_RESPONSE_CANCEL);

  g_signal_connect(G_OBJECT(popup_window), "key-release-event", G_CALLBACK(get_key), NULL);
  g_signal_connect(GTK_OBJECT(popup_window), "response",
									 G_CALLBACK(keygrab_dialog_response_cb),
									 NULL);
  GTK_WIDGET_UNSET_FLAGS(button_cancel, GTK_CAN_FOCUS);

  gtk_widget_show(button_cancel);
  gtk_widget_show(button_undefine);
  gtk_widget_show(popup_window);
}

static void row_activated_cb(GtkTreeView *treeview, GtkTreePath *treepath, GtkTreeViewColumn *column, gpointer data) {
	gint idx;
  gtk_tree_model_get_iter(model, &iter, treepath);
	gtk_tree_model_get(model, &iter, HOTKEY_IDX, &idx, -1);
  show_keygrab_dialog(idx);
}

/*
	static void click_cb(GtkTreeView *treeview, gpointer data) {
  GtkTreeSelection *tree_selection;

  tree_selection = gtk_tree_view_get_selection(treeview);
  if (gtk_tree_selection_get_selected (tree_selection, &model, &iter) == FALSE) return;
  do_keygrab();
	}

	static gboolean select_cb(GtkTreeSelection *tree_selection, gpointer data) {
  IM_JA_DEBUG("selection in list.\n");
  if (gtk_tree_selection_get_selected(tree_selection, &model, &iter) == FALSE) return FALSE;
  if (gtk_tree_selection_iter_is_selected(tree_selection, &iter) == FALSE) return FALSE;
  IM_JA_DEBUG("rows selected: %d\n", gtk_tree_selection_count_selected_rows(tree_selection));
  do_keygrab();
  return TRUE;
	}
*/

static void configurator_exit() {
	GtkWidget *dialog;
	GtkWidget *checkbox;

	IM_JA_DEBUG("configurator_exit\n");
	if ((settings_saved == TRUE) && (cfg.dont_show_save_warning == FALSE)) {
	  dialog = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_ERROR,  GTK_BUTTONS_OK, 
																		_("You will need to restart already running applications for the new settings to take effect."));

		checkbox = gtk_check_button_new_with_label(_("don't show this message again"));
		g_signal_connect(GTK_OBJECT(checkbox), "toggled",
										 G_CALLBACK(checkbutton_simple_cb), &cfg.dont_show_save_warning);

		gtk_widget_show(checkbox);
		gtk_container_add (GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), checkbox);
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		/* save again */
		if (cfg.dont_show_save_warning == TRUE) {
			im_ja_save_conf(&cfg);
		}
		settings_saved = FALSE;
	}
	im_ja_finalize_conf_handler(&cfg);
	gtk_main_quit();
}

static void dialog_response_cb(GtkDialog *dialog, gint response_id) {
	int i;
	gchar *tmpstr;

	switch (response_id) {
	case GTK_RESPONSE_APPLY:
		cfg.status_win_coords[0] = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_status_xpos")));
		cfg.status_win_coords[1] = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_status_ypos")));

		cfg.kanjipad_size[0] = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_kanjipad_width")));
		cfg.kanjipad_size[1] = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_kanjipad_height")));

		for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
			tmpstr = g_strdup_printf("entry_status_label_%d", i);
			cfg.status_win_labels[i] = g_strdup(gtk_entry_get_text(GTK_ENTRY(GETWIDGET(tmpstr))));
			g_free(tmpstr);
		}

		cfg.wnn_address = g_strdup(gtk_entry_get_text(GTK_ENTRY(GETWIDGET("entry_wnn_address"))));
		/*		cfg.wnn_port = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_wnn_port"))); */
		cfg.wnnenvrc = g_strdup(gtk_entry_get_text(GTK_ENTRY(GETWIDGET("entry_wnnenvrc"))));

		/*
		//		cfg.canna_address = g_strdup(gtk_entry_get_text(GTK_ENTRY(GETWIDGET("entry_canna_address"))));
		//		cfg.canna_port = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_canna_port")));
		*/

    IM_JA_DEBUG("save config.\n");
    im_ja_save_conf(&cfg);
		settings_saved = TRUE;
		break;
	case GTK_RESPONSE_HELP:
		show_help();
		break;
  default:
		configurator_exit();
		break;
	}
}

static void set_preview_style() {
	gtk_widget_modify_text(GETWIDGET("preview_n_entry"), GTK_STATE_NORMAL, &cfg.preedit_colors[0]);
	gtk_widget_modify_base(GETWIDGET("preview_n_entry"), GTK_STATE_NORMAL, &cfg.preedit_colors[1]);
	gtk_widget_modify_text(GETWIDGET("preview_hl_entry"), GTK_STATE_NORMAL, &cfg.preedit_colors[2]);
	gtk_widget_modify_base(GETWIDGET("preview_hl_entry"), GTK_STATE_NORMAL, &cfg.preedit_colors[3]);
}

static void change_color_cb(GtkWidget *button, int which) {
  GtkWidget *dialog;
  GtkColorSelection *colorsel;
  gint response;
  GdkColor *cfgcolor = &cfg.preedit_colors[which];

	dialog = gtk_color_selection_dialog_new(_("Color selection"));

  gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(window));
  
  colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dialog)->colorsel);

	gtk_color_selection_set_previous_color(colorsel, cfgcolor);
	gtk_color_selection_set_current_color(colorsel, cfgcolor);
  
  gtk_color_selection_set_has_palette(colorsel, TRUE);
  
  response = gtk_dialog_run(GTK_DIALOG(dialog));

  if (response == GTK_RESPONSE_OK) {
    IM_JA_DEBUG("Color selection for %d\n", which);
    gtk_color_selection_get_current_color(colorsel, cfgcolor);
		set_preview_style();
  }
  gtk_widget_destroy(dialog);
}

static void radiobutton_n_cb(GtkWidget *button) {
	gboolean iscustom  = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_n_default")));
	
	gtk_widget_set_sensitive(GETWIDGET("preview_n_entry"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("preview_n_label"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_n_b_button"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_n_f_button"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_n_b_label"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_n_f_label"), iscustom);

	cfg.custom_preedit_n = iscustom;
}

static void radiobutton_hl_cb(GtkWidget *button) {
	gboolean iscustom  = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_hl_default")));
	
	gtk_widget_set_sensitive(GETWIDGET("preview_hl_entry"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("preview_hl_label"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_hl_b_button"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_hl_f_button"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_hl_b_label"), iscustom);
	gtk_widget_set_sensitive(GETWIDGET("colsel_hl_f_label"), iscustom);

	cfg.custom_preedit_hl = iscustom;
}

static void radiobutton_status_win_pos_cb(GtkWidget *button) {
	gboolean fixed = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_fixed_coord")));
	
	gtk_widget_set_sensitive(GETWIDGET("label_status_xpos"), fixed);
	gtk_widget_set_sensitive(GETWIDGET("label_status_ypos"), fixed);
	gtk_widget_set_sensitive(GETWIDGET("spinbutton_status_xpos"), fixed);
	gtk_widget_set_sensitive(GETWIDGET("spinbutton_status_ypos"), fixed);
													 
	
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_below_cursor"))) == TRUE) {
		cfg.status_win_position = STATUS_BELOW_CURSOR;
		return;
	}
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_window_bottom_l"))) == TRUE) {
		cfg.status_win_position = STATUS_WINDOW_BOTTOM_LEFT;	
		return;
	}
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_fixed_coord"))) == TRUE) {
		cfg.status_win_position = STATUS_FIXED_COORD;
		return;
	}
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_status_disabled"))) == TRUE) {
		cfg.status_win_position = STATUS_DISABLED;
	}
}

static void radiobutton_candwin_style_cb(GtkWidget *button) {
	gboolean table_style = FALSE, menu_style = FALSE;


	menu_style = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("candwin_style_menu")));
	table_style = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("candwin_style_table")));
	if (table_style == TRUE) cfg.candwin_style = CANDWIN_STYLE_TABLE;
	if (menu_style == TRUE) cfg.candwin_style = CANDWIN_STYLE_MENU;

}


static void radiobutton_conv_engine_cb(GtkWidget *button) {
	gboolean wnn_default = FALSE;
	gboolean canna_default = FALSE;
	gboolean anthy_default = FALSE;
	gboolean skk_default = FALSE;
	gboolean none_default = FALSE;


	wnn_default = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_wnn_default")));
	canna_default = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_canna_default")));
	anthy_default = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_anthy_default")));
	skk_default = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_skk_default")));
	none_default = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_none_default")));

#ifdef DISABLE_WNN
	wnn_default = FALSE;
#endif
#ifdef DISABLE_CANNA
	canna_default = FALSE;
#endif
#ifdef DISABLE_ANTHY
	anthy_default = FALSE;
#endif
#ifdef DISABLE_SKK
	skk_default = FALSE;
#endif

	gtk_widget_set_sensitive(GETWIDGET("label_wnn_address"), wnn_default);
	gtk_widget_set_sensitive(GETWIDGET("label_wnnenvrc"), wnn_default);
	/*	gtk_widget_set_sensitive(GETWIDGET("label_wnn_port"), wnn_default); */
	gtk_widget_set_sensitive(GETWIDGET("entry_wnn_address"), wnn_default);
	/*	gtk_widget_set_sensitive(GETWIDGET("spinbutton_wnn_port"), wnn_default); */
	gtk_widget_set_sensitive(GETWIDGET("entry_wnnenvrc"), wnn_default);
	gtk_widget_set_sensitive(GETWIDGET("label_wnn_text"), wnn_default);
	/*
	gtk_widget_set_sensitive(GETWIDGET("label_canna_address"), canna_default);
	gtk_widget_set_sensitive(GETWIDGET("label_canna_port"), canna_default);
	gtk_widget_set_sensitive(GETWIDGET("entry_canna_address"), canna_default);
	gtk_widget_set_sensitive(GETWIDGET("spinbutton_canna_port"), canna_default);
	gtk_widget_set_sensitive(GETWIDGET("label_canna_text"), canna_default);
	*/
	if (wnn_default == TRUE) cfg.default_conv_engine = CONV_ENGINE_WNN;
	if (canna_default == TRUE) cfg.default_conv_engine = CONV_ENGINE_CANNA;
	if (anthy_default == TRUE) cfg.default_conv_engine = CONV_ENGINE_ANTHY;
	if (skk_default == TRUE) cfg.default_conv_engine = CONV_ENGINE_SKK;
	if (none_default == TRUE) cfg.default_conv_engine = CONV_ENGINE_NONE;

}

static void checkbutton_enable_kanjipad_cb(GtkWidget *button) {
	gboolean enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_enable_kanjipad")));

	if (((cfg.kanjipad_custom_size == TRUE) && (enabled == TRUE)) || (enabled == FALSE)) {
		gtk_widget_set_sensitive(GETWIDGET("spinbutton_kanjipad_width"), enabled);
		gtk_widget_set_sensitive(GETWIDGET("spinbutton_kanjipad_height"), enabled);
		gtk_widget_set_sensitive(GETWIDGET("label_kanjipad_width"), enabled);
		gtk_widget_set_sensitive(GETWIDGET("label_kanjipad_height"), enabled);
	}
	gtk_widget_set_sensitive(GETWIDGET("checkbutton_kanjipad_size"), enabled);
	
	cfg.kanjipad_enabled = enabled;

}

static void checkbutton_kanjipad_size_cb(GtkWidget *button) {
	gboolean enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_kanjipad_size")));

	gtk_widget_set_sensitive(GETWIDGET("spinbutton_kanjipad_width"), enabled);
	gtk_widget_set_sensitive(GETWIDGET("spinbutton_kanjipad_height"), enabled);
	gtk_widget_set_sensitive(GETWIDGET("label_kanjipad_width"), enabled);
	gtk_widget_set_sensitive(GETWIDGET("label_kanjipad_height"), enabled);

	cfg.kanjipad_custom_size = enabled;
}

static void checkbutton_candwin_font_cb(GtkWidget *button) {
	gboolean enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_candwin_font")));
	gtk_widget_set_sensitive(GETWIDGET("candwin_fontbutton"), enabled);
	cfg.custom_candwin_font = enabled;
}

static void checkbutton_simple_cb(GtkWidget *button, gboolean *cfgval) {
	*cfgval = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
}

static void startup_menu_cb(GtkWidget *menu) {
	gint input_method = 0;

	input_method = gtk_combo_box_get_active(GTK_COMBO_BOX(menu));
	IM_JA_DEBUG("startup_menu_cb() selected: %d.\n", input_method);
	if (input_method == -1) input_method = 0;
	cfg.startup_input_method = input_method;
}

static void candwin_font_selected(GtkWidget *font_button, gpointer data) {
	IM_JA_DEBUG("candwin_font_selected()\n");
	cfg.candwin_font = (gchar *) gtk_font_button_get_font_name(GTK_FONT_BUTTON(font_button));
}

static GtkWidget *create_dialog() {
  GtkWidget *treeview;
	GtkWidget *menu, *hbox_startup;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
	GtkWidget *font_button;
	gint i;
  gchar *tmpstr;

  gladexml = glade_xml_new(IM_JA_DATADIR"/im-ja-conf.glade", "notebook", NULL);
  
  /* Set up the table of hotkey assignments */
	treeview = GETWIDGET("treeview");

  model = GTK_TREE_MODEL(gtk_list_store_new(NUM_COLS, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT));
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
	gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), model);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Action"), renderer, "text", HOTKEY_DESC, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Key"), renderer, "text", HOTKEY_KEYVAL, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);
  g_signal_connect(G_OBJECT(treeview), "row_activated", G_CALLBACK(row_activated_cb), NULL);

  for (i = 0; i < (gint) (ENUMSIZE(hotkey_labels)); i++) {
		/* Don't show canna and wnn keys in configurator */
#ifndef DEBUG_IM_JA
		if ((i != 8) && (i != 9)) { /* FIXME: use GConfEnumStringPair constant */
#endif
			gtk_list_store_append(GTK_LIST_STORE(model), &iter);
			tmpstr = im_ja_get_keyname(cfg.hotkey_states[i], cfg.hotkey_values[i]);
			gtk_list_store_set(GTK_LIST_STORE(model), &iter,
												 HOTKEY_DESC, _(hotkey_labels[i]),
												 HOTKEY_KEYVAL, tmpstr,
												 HOTKEY_IDX, i,
												 -1);
			IM_JA_DEBUG("adding %d: %s\n", i, hotkey_labels[i]);
			g_free(tmpstr);
#ifndef DEBUG_IM_JA
		}
#endif
  }

  /* Set up color pickers */
  g_signal_connect(GTK_OBJECT(GETWIDGET("colsel_n_f_button")), "clicked",
									 G_CALLBACK(change_color_cb),
									 (gpointer) COLOR_NORMAL_FG);
  g_signal_connect(GTK_OBJECT(GETWIDGET("colsel_n_b_button")), "clicked",
									 G_CALLBACK(change_color_cb),
									 (gpointer) COLOR_NORMAL_BG);
  g_signal_connect(GTK_OBJECT(GETWIDGET("colsel_hl_f_button")), "clicked",
									 G_CALLBACK(change_color_cb),
									 (gpointer) COLOR_HIGHL_FG);
  g_signal_connect(GTK_OBJECT(GETWIDGET("colsel_hl_b_button")), "clicked",
									 G_CALLBACK(change_color_cb),
									 (gpointer) COLOR_HIGHL_BG);

	/* Set up custom color radio buttons */
	if (cfg.custom_preedit_n == TRUE) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_n_custom")), TRUE);	
	}
	else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_n_default")), TRUE);	
	
	if (cfg.custom_preedit_hl == TRUE) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_hl_custom")), TRUE);	
	}
	else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_hl_default")), TRUE);	
	
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_hl_default")), "toggled",
									 G_CALLBACK(radiobutton_hl_cb), NULL);
  g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_n_default")), "toggled",
									 G_CALLBACK(radiobutton_n_cb), NULL);
	
	radiobutton_n_cb(NULL);
	radiobutton_hl_cb(NULL);
	
	/* Set up status window position radio buttons */
	switch (cfg.status_win_position) {
	case STATUS_BELOW_CURSOR:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_below_cursor")), TRUE);	
		break;
	case STATUS_WINDOW_BOTTOM_LEFT:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_window_bottom_l")), TRUE);	
		break;
	case STATUS_FIXED_COORD:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_fixed_coord")), TRUE);	
		break;
	case STATUS_DISABLED:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_status_disabled")), TRUE);	
		break;
	}

	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_below_cursor")), "toggled",
									 G_CALLBACK(radiobutton_status_win_pos_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_window_bottom_l")), "toggled",
									 G_CALLBACK(radiobutton_status_win_pos_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_fixed_coord")), "toggled",
									 G_CALLBACK(radiobutton_status_win_pos_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_status_disabled")), "toggled",
									 G_CALLBACK(radiobutton_status_win_pos_cb), NULL);
	radiobutton_status_win_pos_cb(NULL);	

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_status_xpos")), cfg.status_win_coords[0]);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_status_ypos")), cfg.status_win_coords[1]);

	for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
		tmpstr = g_strdup_printf("entry_status_label_%d", i);
		if (cfg.status_win_labels[i] != NULL) gtk_entry_set_text(GTK_ENTRY(GETWIDGET(tmpstr)), cfg.status_win_labels[i]);
		g_free(tmpstr);
	}

	switch (cfg.default_conv_engine) {
	case CONV_ENGINE_WNN:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_wnn_default")), TRUE);	
		break;
	case CONV_ENGINE_CANNA:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_canna_default")), TRUE);	
		break;
	case CONV_ENGINE_ANTHY:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_anthy_default")), TRUE);	
		break;
	case CONV_ENGINE_SKK:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_skk_default")), TRUE);	
		break;
	case CONV_ENGINE_NONE:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("radiobutton_none_default")), TRUE);	
		break;
	}
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_wnn_default")), "toggled",
									 G_CALLBACK(radiobutton_conv_engine_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_canna_default")), "toggled",
									 G_CALLBACK(radiobutton_conv_engine_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_anthy_default")), "toggled",
									 G_CALLBACK(radiobutton_conv_engine_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_skk_default")), "toggled",
									 G_CALLBACK(radiobutton_conv_engine_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("radiobutton_none_default")), "toggled",
									 G_CALLBACK(radiobutton_conv_engine_cb), NULL);
	radiobutton_conv_engine_cb(NULL);
#ifdef DISABLE_WNN
	gtk_widget_set_sensitive(GETWIDGET("radiobutton_wnn_default"), FALSE);
#endif
#ifdef DISABLE_CANNA
	gtk_widget_set_sensitive(GETWIDGET("radiobutton_canna_default"), FALSE);
#endif
#ifdef DISABLE_ANTHY
	gtk_widget_set_sensitive(GETWIDGET("radiobutton_anthy_default"), FALSE);
#endif
#ifdef DISABLE_SKK
	gtk_widget_set_sensitive(GETWIDGET("radiobutton_skk_default"), FALSE);
#endif

	/*
	//	gtk_spin_button_set_value(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_wnn_port")), cfg.wnn_port);
	//	gtk_spin_button_set_value(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_canna_port")), cfg.canna_port);
	*/
	if (cfg.wnn_address != NULL) gtk_entry_set_text(GTK_ENTRY(GETWIDGET("entry_wnn_address")), cfg.wnn_address);
	if (cfg.wnnenvrc != NULL) gtk_entry_set_text(GTK_ENTRY(GETWIDGET("entry_wnnenvrc")), cfg.wnnenvrc);
	/*	if (cfg.canna_address != NULL) gtk_entry_set_text(GTK_ENTRY(GETWIDGET("entry_canna_address")), cfg.canna_address); */

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_enable_kanjipad")), cfg.kanjipad_enabled);	
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_kanjipad_size")), cfg.kanjipad_custom_size);	
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_kanjipad_width")), cfg.kanjipad_size[0]);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(GETWIDGET("spinbutton_kanjipad_height")), cfg.kanjipad_size[1]);
	checkbutton_enable_kanjipad_cb(NULL);
	checkbutton_kanjipad_size_cb(NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("checkbutton_enable_kanjipad")), "toggled",
									 G_CALLBACK(checkbutton_enable_kanjipad_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("checkbutton_kanjipad_size")), "toggled",
									 G_CALLBACK(checkbutton_kanjipad_size_cb), NULL);

	
	hbox_startup = GETWIDGET("hbox_startup_mode");
	menu = gtk_combo_box_new_text();
	gtk_box_pack_start(GTK_BOX(hbox_startup), menu, FALSE, FALSE, 0);

	for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
		gtk_combo_box_append_text(GTK_COMBO_BOX(menu), _(gconf_enum_to_string(status_win_labels, i)));
	}
	gtk_widget_show(menu);
	g_signal_connect_swapped(GTK_OBJECT(menu), "changed", 
													 GTK_SIGNAL_FUNC(startup_menu_cb), (gpointer) menu);

	gtk_combo_box_set_active(GTK_COMBO_BOX(menu), cfg.startup_input_method);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_preedit_win")), cfg.preedit_window_on);	
	g_signal_connect(GTK_OBJECT(GETWIDGET("checkbutton_preedit_win")), "toggled",
									 G_CALLBACK(checkbutton_simple_cb), &cfg.preedit_window_on);


	g_signal_connect(GTK_OBJECT(GETWIDGET("candwin_style_menu")), "toggled",
									 G_CALLBACK(radiobutton_candwin_style_cb), NULL);
	g_signal_connect(GTK_OBJECT(GETWIDGET("candwin_style_table")), "toggled",
									 G_CALLBACK(radiobutton_candwin_style_cb), NULL);

	switch (cfg.candwin_style) {
	case CANDWIN_STYLE_TABLE:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("candwin_style_table")), TRUE);	
		break;
	case CANDWIN_STYLE_MENU:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("candwin_style_menu")), TRUE);	
		break;
	}

	font_button = GETWIDGET("candwin_fontbutton");
	if (cfg.candwin_font != NULL) {
		gtk_font_button_set_font_name(GTK_FONT_BUTTON(font_button), cfg.candwin_font);
	}
	g_signal_connect(GTK_OBJECT(font_button), "font-set", G_CALLBACK(candwin_font_selected), NULL);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_candwin_font")), cfg.custom_candwin_font);	
	checkbutton_candwin_font_cb(NULL);

	g_signal_connect(GTK_OBJECT(GETWIDGET("checkbutton_candwin_font")), "toggled",
									 G_CALLBACK(checkbutton_candwin_font_cb), NULL);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_commit_on_reset")), cfg.commit_on_reset);	
	g_signal_connect(GTK_OBJECT(GETWIDGET("checkbutton_commit_on_reset")), "toggled",
									 G_CALLBACK(checkbutton_simple_cb), &cfg.commit_on_reset);


	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(GETWIDGET("checkbutton_use_systray")), cfg.use_systray);	
	g_signal_connect(GTK_OBJECT(GETWIDGET("checkbutton_use_systray")), "toggled",
									 G_CALLBACK(checkbutton_simple_cb), &cfg.use_systray);

	g_signal_connect(G_OBJECT(GETWIDGET("notebook")), "switch_page",
									 G_CALLBACK(page_switch_cb), NULL);

	return GETWIDGET("notebook");
}


static gboolean show_help() {
	static gchar *docpath = IM_JA_DATADIR"/im-ja-doc.html";
	static gchar *command[1];
	static gchar *command_argv[1];
	GError *g_error = NULL;
	gint i;

	command[0] = g_strdup_printf(cfg.browser, docpath);

	for (i = 0; i < (gint) (strlen(command[0])); i++) {
		if (command[0][i] == ' ') {
			command[0][i] = 0;
			command_argv[0] = command[0] + i + 1;
			break;
		}
	}

	/* trim the paranthesis */
	if (command_argv[0][0] == '"') command_argv[0]++;
	if (command_argv[0][strlen(command_argv[0]) - 1] == '"') command_argv[0][strlen(command_argv[0]) - 1] = 0;

	IM_JA_DEBUG("Running command: %s, args: %s\n", command[0], command_argv[0]);

	if (g_spawn_async(NULL, command, NULL, G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_STDERR_TO_DEV_NULL,
										NULL, NULL, NULL, &g_error) == FALSE) {
		im_ja_print_error(_("Unable to start browser: %s\nPlease view the documentation manually: %s"), g_error->message, docpath); 
		return FALSE;
	}
	if (g_error != NULL) im_ja_print_error(g_error->message); 

	return TRUE;
}


int main(int argc, char *argv[]) {
  GtkWidget *dialog;
  GtkWidget *dialog_win;

#ifdef ENABLE_NLS
  bindtextdomain(GETTEXT_PACKAGE, IM_JA_LOCALE_DIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);
#endif

  gtk_init(&argc, &argv);
    
  if (im_ja_init_conf_handler() == FALSE) {
    im_ja_print_error(_("GConf initialization failed!"));
  }
	im_ja_get_gconf_client();
  if (im_ja_load_conf(&cfg) == FALSE) {
    im_ja_print_error(_("Couldn't load settings!"));
  }
	gtk_window_set_default_icon_from_file(PIXMAPDIR"/im-ja-capplet.png", NULL);
	dialog = create_dialog();
	set_preview_style();
  
  dialog_win = gtk_dialog_new_with_buttons(_("im-ja preferences"), NULL, 0,
																					 GTK_STOCK_HELP, GTK_RESPONSE_HELP,
																					 GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
																					 GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
																					 NULL);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog_win),
																	GTK_RESPONSE_CLOSE);
  
  g_signal_connect(G_OBJECT(dialog_win), "response",
									 G_CALLBACK(dialog_response_cb), NULL);
  
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog_win)->vbox), 
										 dialog, TRUE, TRUE, 10);
	gtk_widget_show(dialog_win);

  gtk_main();
  
  return 0;
}
