/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/*
 * IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 */

#include <config.h>

#include <glib.h>

#include <netinet/in.h>
#include <string.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <pwd.h>
#include <errno.h>

gint fd_connect_unix(gchar *path) {
	gint sock;
	struct sockaddr_un addr;

	sock = socket(PF_UNIX, SOCK_STREAM, 0);
	if (sock < 0) {
		g_error("sock_connect_unix(): socket");
		return -1;
	}

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = PF_UNIX;
	strncpy(addr.sun_path, path, sizeof(addr.sun_path) - 1);

	if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
		close(sock);
		return -1;
	}
	return sock;

}

gint fd_open_unix(gchar *path) {
	gint sock;
	struct sockaddr_un addr;

	sock = socket(PF_UNIX, SOCK_STREAM, 0);

	if (sock < 0) {
		g_error("socket");
		return -1;
	}

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = PF_UNIX;
	strncpy(addr.sun_path, path, sizeof(addr.sun_path) - 1);

	if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
		g_error("bind");
		close(sock);
		return -1;
	}
	if (listen(sock, 100) < 0) {
		g_error("listen");
		close(sock);
		return -1;
	}

	return sock;
}

gint fd_accept(gint sock) {
	struct sockaddr_un saddr;
	socklen_t saddr_len;
	struct timeval tv;
	gint sockfd = -1;
	fd_set fdset;

	tv.tv_sec = 0;
	tv.tv_usec = 100000;
	saddr_len = sizeof(saddr);

 try_again:
  FD_ZERO(&fdset);
  FD_SET(sock, &fdset);
  
  if (select(sock + 1, &fdset, NULL, NULL, &tv) == -1) {
    if (errno == EINTR) goto try_again;
    return -1;
  }
  if ((sockfd = accept(sock, (struct sockaddr *) &saddr, &saddr_len)) == -1) {
    if (errno == EWOULDBLOCK ||
        errno == ECONNABORTED ||
        errno == EINTR)
      goto try_again;
    return -1;
  }

	return sockfd;

}


gint set_nonblocking_mode(gint fd, gboolean nonblock) {
	gint flags;
	
	flags = fcntl(fd, F_GETFL, 0);
	if (flags < 0) {
		g_error("fcntl");
		return -1;
	}

	if (nonblock)	flags |= O_NONBLOCK;
	else flags &= ~O_NONBLOCK;
	return fcntl(fd, F_SETFL, flags);
}


gchar *get_sock_path() {
	gchar *path = NULL;
	gchar *login = NULL;
	struct passwd *pw = NULL;

	if ((getuid() == geteuid()) && (getgid() == getegid())) {
		gchar *env = getenv("LOGNAME");
		if (env != NULL) login = g_strdup(env);
	}

	if (login == NULL) {
		pw = getpwuid(getuid());
		if (pw != NULL) {
			login = g_strdup(pw->pw_name);
		}
		else return NULL;
	}	

	/* g_get_tmp_dir(), g_get_user_name(), */
	path = g_strdup_printf("/tmp/im-ja-%s", login);

	g_free(login);
	
	return path;
}
