/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/*
 * IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 */

#include <gtk/gtk.h>
#include <string.h>
#include <unistd.h>

#include <config.h>

#include "helper-socket.h"
#include "helper-client.h"
#include "../error.h"

static gboolean spawn_helper(gboolean usetray);

ClientIO *helper_client_io_new_from_socket(gint socket) {
	ClientIO *client = g_new0(ClientIO, 1);
	
	client->sock = socket;
	client->io = g_io_channel_unix_new(socket);
	return client;

}

ClientIO *helper_client_io_new_connection(gboolean usetray) {
	gchar *sock_path = NULL;
	gint serv_sock;
	ClientIO *client = NULL;

	sock_path = get_sock_path();
	if (sock_path == NULL) {
		im_ja_print_error_cmdline("couldn't get socket path\n");
		return NULL;
	}
	IM_JA_DEBUG("%s\n", sock_path);

	serv_sock = fd_connect_unix(sock_path);
	if (serv_sock < 0) { /* try to start im-ja-helper */
		IM_JA_DEBUG("couldn't connect to socket %s\n", sock_path);
		if (spawn_helper(usetray) == FALSE) return NULL;
		serv_sock = fd_connect_unix(sock_path);
		if (serv_sock < 0) {
			im_ja_print_error_cmdline("couldn't connect to socket after spawning helper %s\n", sock_path);	 
			return NULL;
		}
	}
	client = helper_client_io_new_from_socket(serv_sock);
	
	return client;
}

static gboolean spawn_helper(gboolean usetray) {
	gint exitstat = -1;
	gchar *argv[4];
		
	argv[0] = IM_JA_LIBPROGDIR"/im-ja-helper";
	argv[1] = "-d";
	argv[2] = NULL;
	argv[3] = NULL;

	if (usetray == FALSE) {
		argv[2] = "-t";
	}
	if (g_spawn_sync(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL, &exitstat, NULL) == FALSE) {
		im_ja_print_error_cmdline("Failed to spawn helper process: %s\n", argv[0]);
		return FALSE;
	}
	else {
		IM_JA_DEBUG("Spawned process. helper daemon returned with %d\n", exitstat);
		return TRUE;
	}
}

gboolean helper_client_io_send(ClientIO *client, gchar *string) {
	gsize written = 0;
	GError *err = NULL;
	gchar *line;

	if (string == NULL) {
		im_ja_print_error_cmdline("NULL string in helper_client_io_send()\n");
		return FALSE;
	}
	else {
		line = g_strdup_printf("%s\n", string);
	}
	if (client != NULL) {
		if (g_io_channel_write_chars(client->io, line, -1, &written, &err) != G_IO_STATUS_NORMAL) {
			im_ja_print_error_cmdline("couldn't send message\n");
			g_free(line);
			return FALSE;
		}
		if (g_io_channel_flush(client->io, &err) != G_IO_STATUS_NORMAL) {
			im_ja_print_error_cmdline("couldn't flush message\n");
			g_free(line);
			return FALSE;
		}
	}
	g_free(line);
	/*	g_print("sent %d bytes\n", written); */
	if (written == strlen(string) + 1) return TRUE;
	else return FALSE;
}

void helper_client_io_close(ClientIO *client) {
	GError *err = NULL;

	if (client != NULL) {
	 	if (client->watch_id != 0) g_source_remove(client->watch_id);
		/* g_io_channel_unref(client->io); */
		g_io_channel_shutdown(client->io, FALSE, &err);
		close(client->sock);
	}
}

gboolean helper_client_send_command(ClientIO *client, gchar *command, gint param) {
	gchar *msg;

	if (client == NULL) {
		IM_JA_DEBUG("Failed to send message to helper: helper client doesn't exist\n");
		return FALSE;
	}

	msg = g_strdup_printf("%s %d", command, param);
	if (helper_client_io_send(client, msg) == FALSE) {
		IM_JA_DEBUG("Failed to send %s to im-ja-helper.\n", command);
		g_free(msg);
		return FALSE;
	}
	g_free(msg);
	return TRUE;
}
		
