/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/*
 * IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 */

#include "config.h"

#include <gtk/gtkimmodule.h>
#include <gtk/gtk.h>
#include <gmodule.h>
#include <gconf/gconf-client.h>

#include <string.h>

#include "../im-ja.h"
#include "../conf.h"
#include "../common.h"
#include "../error.h"
#include "../nls.h"
#include "../actionmenu.h"
#include "../helper/helper-client.h"

IMJAConfig cfg;
extern ClientIO *helper_client;
extern GConfClient *gconf_client;

GConfEnumStringPair label_keys[] = {
	{IM_JA_DIRECT_INPUT,    N_("direct")},
	{IM_JA_HIRAGANA_INPUT,  N_("hiragana")},
	{IM_JA_KATAKANA_INPUT,  N_("katakana")},
	{IM_JA_HALFKATA_INPUT,  N_("halfkata")},
	{IM_JA_ZENKAKU_INPUT,   N_("zenkaku")},
	{IM_JA_KANJIPAD_INPUT,  N_("kanjipad")},
	{-1, NULL}
};

GConfEnumStringPair input_method_labels[] = {
	{IM_JA_DIRECT_INPUT,    N_("[direct input]")},
	{IM_JA_HIRAGANA_INPUT,  N_("[hiragana input]")},
	{IM_JA_KATAKANA_INPUT,  N_("[katakana input]")},
	{IM_JA_HALFKATA_INPUT,  N_("[narrow katakana input]")},
	{IM_JA_ZENKAKU_INPUT,   N_("[zenkaku input]")},
	{IM_JA_KANJIPAD_INPUT,  N_("[kanjipad input]")},
	{-1, NULL}
};

typedef struct _GtkIMModule {
	GTypeModule parent_instance;
	GModule *library;
} GtkIMModule;

static const GtkIMContextInfo im_ja_info = { 
	"im-ja",           /* ID */
	N_("Japanese"),    /* Human readable name */
	GETTEXT_PACKAGE,   /* Translation domain */
	IM_JA_LOCALE_DIR,  /* Dir for bindtextdomain (not strictly needed for "gtk+") */
	"ja"               /* Languages for which this module is the default */
};

static const GtkIMContextInfo *info_list[] = {
	&im_ja_info
};

static gboolean im_ja_gtk_filter_keypress(GtkIMContext *context, GdkEventKey *key) {
	IMJAContext *cn = IM_JA_CONTEXT(context);
	
	return im_ja_filter_keypress(cn, key);
}

static void im_ja_gtk_cursor_location_changed(GtkIMContext *context, GdkRectangle *area) {
	IMJAContext *cn = IM_JA_CONTEXT(context);
	gint x, y, area_h;

	if (cn->finalized == TRUE) {
		IM_JA_DEBUG("*ERROR* IMContext is already finalized.\n");
		return;
	}

	/*Bug in GTK?:	width & height is sometimes (on first set_location) a huge_number */
	area_h = area->height;
	if ((area->width > 1000) || (area->width < 0) ||
			(area->height > 1000) || (area->height < 0)) {
		area_h = 21;
	}
	x = area->x;
	y = area->y + area_h + 1;

	im_ja_cursor_location_changed(cn, x, y);
}

static void im_ja_populate_popup(GtkTextView *textview, GtkMenu *menu, IMJAContext *cn) {
  GtkWidget *menuitem;
	gchar *labels[IM_JA_INPUT_METHODS_TOTAL];
	gchar *gconf_key;
	gint i;

  IM_JA_DEBUG("im_ja_populate_popup()\n");
	
	for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
		gconf_key = g_strdup_printf(GCONF_NAMESPACE"status_window/label_%s", gconf_enum_to_string(label_keys, i));
		labels[i] = gconf_client_get_string(gconf_client, gconf_key, NULL);
		g_free(gconf_key);
	}

	menuitem = gtk_separator_menu_item_new();
	gtk_widget_show(menuitem);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	im_ja_actionmenu_populate(cn, menu, IM_JA_GTKIM_MENU);

} 

static void im_ja_gtk_set_client_window(GtkIMContext *context, GdkWindow *win) {
	gboolean populate = TRUE;

	IMJAContext *cn = IM_JA_CONTEXT(context);

	IM_JA_DEBUG("im_ja_gtk_set_client_window [context:%d, window:%d]\n", (int) cn, (int) win);

	if (cn->finalized == TRUE) {
		IM_JA_DEBUG("*ERROR* IMContext is already finalized.\n");
		return;
	}
	cn->client_gdk = win;
	if (win == NULL) {
		if ((cn->client_gtk != NULL) && (cn->popup_signal_h != 0)) {
			g_signal_handler_disconnect(cn->client_gtk, cn->popup_signal_h);
			cn->popup_signal_h = 0;
		}
		cn->client_gdk = NULL;
		cn->client_gtk = NULL;
		cn->toplevel_gtk = NULL;
		cn->toplevel_gdk = NULL;
		return; 
	}
	if (GDK_IS_WINDOW(cn->client_gdk) == TRUE) cn->toplevel_gdk = gdk_window_get_toplevel(cn->client_gdk);
	gdk_window_get_user_data(cn->toplevel_gdk, (gpointer *)&cn->toplevel_gtk);
	gdk_window_get_user_data(cn->client_gdk, (gpointer *)&cn->client_gtk);

	IM_JA_DEBUG(" TOPLEVEL GDK: %d [%s]\n", (int) cn->toplevel_gdk, G_OBJECT_TYPE_NAME(cn->toplevel_gdk));
	IM_JA_DEBUG(" TOPLEVEL GTK: %d [%s]\n", (int) cn->toplevel_gtk, G_OBJECT_TYPE_NAME(cn->toplevel_gtk));
	IM_JA_DEBUG(" CLIENT GDK: %d [%s]\n", (int) cn->client_gdk, G_OBJECT_TYPE_NAME(cn->client_gdk));
	IM_JA_DEBUG(" CLIENT GTK: %d [%s]\n", (int) cn->client_gtk, G_OBJECT_TYPE_NAME(cn->client_gtk));

	/*
	IM_JA_DEBUG("TYPE NAME: %s, TYPE: %d\n", G_OBJECT_TYPE_NAME(cn->client_gtk), G_OBJECT_TYPE(cn->client_gtk));
	IM_JA_DEBUG("SIGNAL SUPPORTED: %d\n", g_signal_lookup("populate-popup", G_OBJECT_TYPE(cn->client_gtk)));
	if ((GTK_IS_ENTRY(cn->client_gtk) == TRUE)
			|| (GTK_IS_TEXT_VIEW(cn->client_gtk) == TRUE)) {
	*/

	/* FIXME: gtk bug */

	if (GTK_IS_ENTRY(cn->client_gtk) == TRUE) {
		if (gtk_editable_get_editable(GTK_EDITABLE(cn->client_gtk)) == FALSE) {
			IM_JA_DEBUG("non-editable GtkEntry.\n");
			populate = FALSE;
			im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		}
	}

	if (GTK_IS_TEXT_VIEW(cn->client_gtk) == TRUE) {
		if (gtk_text_view_get_editable(GTK_TEXT_VIEW(cn->client_gtk)) == FALSE) {
			IM_JA_DEBUG("non-editable GtkTextView.\n");
			populate = FALSE;
			im_ja_set_input_method(cn, IM_JA_DIRECT_INPUT);
		}
	}

	if ((populate == TRUE) && (g_signal_lookup("populate-popup", G_OBJECT_TYPE(cn->client_gtk)) != 0)) {
		cn->popup_signal_h = g_signal_connect(G_OBJECT(cn->client_gtk), "populate-popup",
																					G_CALLBACK(im_ja_populate_popup), cn);
	}

	/*
	if (cn->client_gtk == NULL) {
		cn->client_gtk = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_widget_realize(cn->client_gtk);
		cn->client_gtk->window = cn->client_gdk;
		g_object_set_data(G_OBJECT(cn->client_gtk), "created-by-im-ja", (gpointer) TRUE);
	}
	if (cn->toplevel_gtk == NULL) {
		if (cn->toplevel_gdk == cn->client_gdk) {
			cn->toplevel_gtk = cn->client_gtk;
		}
		else {
			cn->toplevel_gtk = gtk_window_new(GTK_WINDOW_TOPLEVEL);
			gtk_widget_realize(cn->toplevel_gtk);
			cn->toplevel_gtk->window = cn->toplevel_gdk;
			g_object_set_data(G_OBJECT(cn->toplevel_gtk), "created-by-im-ja", (gpointer) TRUE);
		}
	}
	*/

	if (cn->client_gtk != NULL) {
		/* Maybe this is better ?
			 cn->original_colors[1] = cn->client_gtk->style->base[GTK_STATE_SELECTED];
			 cn->original_colors[0] = cn->client_gtk->style->text[GTK_STATE_SELECTED];
		*/
		cn->original_colors[0] = cn->client_gtk->style->base[GTK_STATE_NORMAL];
		cn->original_colors[1] = cn->client_gtk->style->text[GTK_STATE_NORMAL];
	}


}

static void im_ja_gtk_get_preedit_string(GtkIMContext *context, gchar **str, PangoAttrList **attrs, gint *cursor_pos) {
	IMJAContext *cn = IM_JA_CONTEXT(context);

	IM_JA_DEBUG("im_ja_gtk_get_preedit_string()\n");

	preedit_window_update(cn);

	if (attrs != NULL) {
		*attrs = pango_attr_list_new(); /* Create new attrib list */
	}

	if (g_utf8_validate(cn->preedit_buf, -1, NULL) != TRUE) {
		im_ja_print_error_cmdline("utf8_validate failed for preedit string: %s\n", cn->preedit_buf);
	}
	else *str = g_strdup(cn->preedit_buf);
	
	if (*str == NULL || strlen(*str) == 0) {
		if (cursor_pos != NULL) *cursor_pos = 0;
		/* else IM_JA_DEBUG("cursor_pos == NULL !!!\n"); */
		return;
	}

	if (attrs != NULL) {
		/* Normal (and underlined) preedit string */
		PangoAttribute *backgr;
		PangoAttribute *foregr;
		PangoAttribute *underline;

		underline = pango_attr_underline_new(PANGO_UNDERLINE_SINGLE);
		underline->start_index = 0;
		underline->end_index = strlen(*str);
		pango_attr_list_insert(*attrs, underline);

		if (cfg.custom_preedit_n == TRUE) {
			foregr = pango_attr_foreground_new(cfg.preedit_colors[0].red, cfg.preedit_colors[0].green, cfg.preedit_colors[0].blue);
			backgr = pango_attr_background_new(cfg.preedit_colors[1].red, cfg.preedit_colors[1].green, cfg.preedit_colors[1].blue);
			backgr->start_index = 0;
			foregr->start_index = 0;
			backgr->end_index = strlen(*str);
			foregr->end_index = strlen(*str);
			pango_attr_list_insert(*attrs, backgr);
			pango_attr_list_insert(*attrs, foregr);
		}

		if (cn->preedit_reverse_start != cn->preedit_reverse_end) {
			/* Highlighted (==reverse) field */
			if (cfg.custom_preedit_hl == TRUE) {
				foregr = pango_attr_foreground_new(cfg.preedit_colors[2].red, cfg.preedit_colors[2].green, cfg.preedit_colors[2].blue);
				backgr = pango_attr_background_new(cfg.preedit_colors[3].red, cfg.preedit_colors[3].green, cfg.preedit_colors[3].blue);
			}
			else { /* Use the default */
				foregr = pango_attr_foreground_new(cn->original_colors[0].red, cn->original_colors[0].green, cn->original_colors[0].blue);
				backgr = pango_attr_background_new(cn->original_colors[1].red, cn->original_colors[1].green, cn->original_colors[1].blue);
			}
			backgr->start_index = cn->preedit_reverse_start;
			foregr->start_index = cn->preedit_reverse_start;
			backgr->end_index = cn->preedit_reverse_end;
			foregr->end_index = cn->preedit_reverse_end;
			IM_JA_DEBUG("reverse start: %d, end: %d\n", cn->preedit_reverse_start, cn->preedit_reverse_end);

			pango_attr_list_insert(*attrs, backgr);
			pango_attr_list_insert(*attrs, foregr);
		}
	}

	if (cursor_pos != NULL) {
		*cursor_pos = im_ja_get_cursor_pos_chars(cn);
		IM_JA_DEBUG("im_ja_get_preedit_string: cursor char pos: %d\n", *cursor_pos);
	}
}

static void im_ja_gtk_set_use_preedit(GtkIMContext *context, gboolean use_preedit) {
	IMJAContext *cn = IM_JA_CONTEXT(context);
	im_ja_set_use_preedit(cn, use_preedit);
}

static void im_ja_gtk_lost_focus(GtkIMContext *context) {
	IMJAContext *cn = IM_JA_CONTEXT(context);
	im_ja_lost_focus(cn);
}

static void im_ja_gtk_got_focus(GtkIMContext *context) {
	IMJAContext *cn = IM_JA_CONTEXT(context);
	im_ja_got_focus(cn);
}

static void im_ja_gtk_reset(GtkIMContext *context) {
	IMJAContext *cn = IM_JA_CONTEXT(context);
	im_ja_on_reset(cn);
}

/* This is called only once for the application */
static void im_ja_gtk_class_init(GtkIMContextClass *class) {
	GtkIMContextClass *im_context_class = GTK_IM_CONTEXT_CLASS(class);
	GObjectClass *object_class = G_OBJECT_CLASS(class);
	IM_JA_DEBUG("im_ja_gtk_class_init\n");
	im_context_class->set_client_window = im_ja_gtk_set_client_window;
	im_context_class->filter_keypress = im_ja_gtk_filter_keypress;
	im_context_class->get_preedit_string = im_ja_gtk_get_preedit_string;
	im_context_class->set_cursor_location = im_ja_gtk_cursor_location_changed;
	im_context_class->focus_in = im_ja_gtk_got_focus;
	im_context_class->focus_out = im_ja_gtk_lost_focus;
	im_context_class->reset = im_ja_gtk_reset;
	im_context_class->set_use_preedit = im_ja_gtk_set_use_preedit;
	object_class->finalize = im_ja_context_class_finalize;

	gconf_client = NULL;

	if (im_ja_init_conf_handler() == FALSE) {
		im_ja_print_error(_("GConf initialization failed!"));
	}
	im_ja_get_gconf_client();
	cfg.im_ja_version = NULL;
	if (im_ja_load_conf(&cfg) == FALSE) {
		im_ja_print_error(_("Couldn't load settings!"));
	}

	if (cfg.im_ja_version == NULL) {
		if (im_ja_print_question(_("You have not yet configured im-ja.\nWould yo like to do so now?")) == TRUE) {
			gconf_client_set_string(gconf_client, GCONF_NAMESPACE"other/im_ja_version", VERSION, NULL);
			im_ja_run_configurator();
		}
		else gconf_client_set_string(gconf_client, GCONF_NAMESPACE"other/im_ja_version", VERSION, NULL);
	}
	else {
		if (strcmp(cfg.im_ja_version, VERSION) != 0) {
			if (im_ja_print_question(_("Would you like to update your im-ja settings?")) == TRUE) {
				gconf_client_set_string(gconf_client, GCONF_NAMESPACE"other/im_ja_version", VERSION, NULL);
				im_ja_run_configurator();
			}
			else gconf_client_set_string(gconf_client, GCONF_NAMESPACE"other/im_ja_version", VERSION, NULL);
		}
	}
}


static void im_ja_gtk_finalize(GtkIMContext *context) {
	IM_JA_DEBUG("im_ja_[gtk]_finalize()  [%d]\n", (int)context);
	helper_client_io_close(helper_client);
	helper_client = NULL;

	im_ja_finalize_conf_handler(&cfg);
	status_window_destroy_all();
	preedit_window_destroy_all();
}

static void im_ja_gtk_init(GtkIMContext *context) {
	IMJAContext *cn = IM_JA_CONTEXT(context);
	cn->client_gdk = NULL;
	cn->client_gtk = NULL;
	cn->toplevel_gtk = NULL;
	cn->toplevel_gdk = NULL;
	cn->popup_signal_h = 0;
	im_ja_context_init(cn);
}

void im_ja_gtk_register_type(GTypeModule *module) {
	static const GTypeInfo object_info = {
		sizeof (IMJAContextClass),
		(GBaseInitFunc) NULL,
		(GBaseFinalizeFunc) im_ja_gtk_finalize,
		(GClassInitFunc) im_ja_gtk_class_init,
		NULL, 
		NULL,
		sizeof (IMJAContext),
		0,
		(GInstanceInitFunc) im_ja_gtk_init,
	};
	
	IM_JA_DEBUG("im_ja_register_type\n");
	
	type_im_ja_context = g_type_module_register_type(module, GTK_TYPE_IM_CONTEXT, "IMJAContext", &object_info, 0);
}

void im_module_init(GTypeModule *module) {
	GtkIMModule *im_module;
	IM_JA_DEBUG("im_module_init()\n");

	im_module = (GtkIMModule *) module;
	g_module_make_resident(im_module->library); /*FIXME: This is only a workaround for the GConf bug */

	im_ja_gtk_register_type(module);
}

void im_module_exit(void) {
	IM_JA_DEBUG("im_module_exit()\n");
}


void im_module_list(const GtkIMContextInfo ***contexts, int *n_contexts) {
	*contexts = info_list;
	*n_contexts = G_N_ELEMENTS(info_list);
}

GtkIMContext *im_ja_gtk_context_new(void) {
	IM_JA_DEBUG("im_ja_gtk_context_new()\n");
  return GTK_IM_CONTEXT(im_ja_context_new());
}

GtkIMContext *im_module_create(const gchar *context_id) {
	IM_JA_DEBUG("im_module_create()\n");
	if (strcmp(context_id, "im-ja") == 0) return im_ja_gtk_context_new(); 
	else return NULL;
}

