/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 *  Copyright (C) 2003 Botond Botyanszki <boti@rocketmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <config.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gnome.h>
#include <libgnomeui/gnome-about.h>
#include <gconf/gconf-client.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <string.h>
#include <stdarg.h>

#include "../im-ja.h"
#include "../error.h"
#include "../actionmenu.h"
#include "../helper/helper-client.h"


GConfEnumStringPair label_keys[] = {
	{IM_JA_DIRECT_INPUT, "direct"},
	{IM_JA_HIRAGANA_INPUT, "hiragana"},
	{IM_JA_KATAKANA_INPUT, "katakana"},
	{IM_JA_HALFKATA_INPUT, "halfkata"},
	{IM_JA_ZENKAKU_INPUT, "zenkaku"},
	{IM_JA_KANJIPAD_INPUT, "kanjipad"},
	{-1, NULL}
};


GConfEnumStringPair input_method_labels[] = {
	{IM_JA_DIRECT_INPUT, N_("[direct input]")},
	{IM_JA_HIRAGANA_INPUT, N_("[hiragana input]")},
	{IM_JA_KATAKANA_INPUT, N_("[katakana input]")},
	{IM_JA_HALFKATA_INPUT, N_("[narrow katakana input]")},
	{IM_JA_ZENKAKU_INPUT, N_("[zenkaku input]")},
	{IM_JA_KANJIPAD_INPUT, N_("[kanjipad input]")},
	{-1, NULL}
};


typedef struct _ImJaApplet ImJaApplet;
struct _ImJaApplet {
	PanelApplet *applet;
	GtkWidget *vbox;
	GtkDialog *about;
	GConfClient *gconf_client;
	gchar *labels[IM_JA_INPUT_METHODS_TOTAL];
	guint notify_id;
	GtkWidget *status_widget;
	GtkWidget *border;
	GtkWidget *menu;
	gint input_method;
	ClientIO *helper_client;
};

static void applet_change_size(PanelApplet *applet_widget, int size, ImJaApplet *applet);
static void applet_change_orient(PanelApplet *applet_widget, int orient, ImJaApplet *applet);
static void applet_destroy(PanelApplet *applet_widget, ImJaApplet *applet);
static void about_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname);
static void help_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname);
static void settings_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname);
static void update_status_label(ImJaApplet *applet);

gboolean im_ja_applet_factory(PanelApplet *applet_widget, const gchar *iid, gpointer data);

static const BonoboUIVerb im_ja_applet_menu_verbs[] =  {
	BONOBO_UI_VERB("Preferences", settings_cb),
	BONOBO_UI_VERB("Help", help_cb),
	BONOBO_UI_VERB("About", about_cb),
	BONOBO_UI_VERB_END
};


static int print_error(const char *fmt, ... ) {
	GtkWidget *dialog;
	int ret = -1;
	va_list args;
	gchar *pstr;

	va_start(args, fmt);
	pstr = g_strdup_vprintf(fmt, args);
	va_end(args);
 
	if (pstr != NULL) {
	  dialog = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_ERROR,  GTK_BUTTONS_OK, "%s", pstr );

		g_signal_connect_swapped(GTK_OBJECT(dialog), "response", 
														 G_CALLBACK(gtk_widget_destroy),
														 GTK_OBJECT(dialog));
		
		gtk_widget_show_all(dialog);
	  g_free(pstr);
	}
	return ret;
}

static void applet_change_size(PanelApplet *applet_widget, int size, ImJaApplet *applet) {

}

static void applet_change_orient(PanelApplet *applet_widget, int orient, ImJaApplet *applet) {

}

static void applet_destroy(PanelApplet *applet_widget, ImJaApplet *applet) {
	g_object_unref(G_OBJECT(applet->gconf_client));
	applet->gconf_client = NULL;
	g_assert(applet);
	g_free(applet);
	return;
}

static void about_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname) {
	GdkPixbuf *icon = NULL;
	const char *authors[] = {"Botond Botyanszki <boti@rocketmail.com>", NULL};
	ImJaApplet *applet = (ImJaApplet*)data;
	char *translators = _("TRANSLATORS");

	if (applet->about) {
		gtk_window_present(GTK_WINDOW(applet->about));
		return;
	}
	
	/* FIXME: make a decent logo */
	icon = gdk_pixbuf_new_from_file(PIXMAPDIR"/im-ja-capplet.png", NULL);
	
	applet->about = GTK_DIALOG(gnome_about_new (_("IM-JA Applet"), VERSION, "Copyright 2003 Botond Botyanszki",
				_("A Japanese input module for GTK2"),
				(const char **) authors, NULL, strcmp("TRANSLATORS", translators) ? translators : NULL, 
				icon));
 
	if (icon != NULL) gdk_pixbuf_unref(icon);
	g_object_add_weak_pointer(G_OBJECT(applet->about), (gpointer*)&applet->about);
	gtk_widget_show(GTK_WIDGET(applet->about));
}

static void help_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname) {
	GError *error = NULL;
	gboolean retval = FALSE;

	/*
	gnome_help_display(PACKAGE, NULL, &error);
	*/
	/*im_ja_print_error("file:/"IM_JA_DATADIR"/im-ja-doc.html");*/

	retval = gnome_url_show("ghelp:"IM_JA_DATADIR"/im-ja-doc.html", &error);
	if (retval == FALSE) {
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new(NULL, 
																		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
																		GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
																		error->message);
		gtk_dialog_run(GTK_DIALOG (dialog));
		gtk_widget_destroy(dialog);
		g_error_free(error);
		error = NULL;
	}
}

static void settings_cb(BonoboUIComponent *uic, gpointer data, const gchar *verbname) {
	gchar *command[] = {IM_JA_BINDIR"/im-ja-conf"};
	GError *g_error = NULL;
	if (g_spawn_async(NULL, command, NULL, G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_STDERR_TO_DEV_NULL,
										NULL, NULL, NULL, &g_error) == FALSE) {
		print_error(g_error->message); 
	}
	
}

static void set_input_method_cb(GtkWidget *menuitem, ImJaApplet *applet) {
	gint input_method;
	gchar *msg;

	input_method = (gint) g_object_get_data(G_OBJECT(menuitem), "input-method-number");
	msg = g_strdup_printf("%s %d", HELPER_MSG_SET_INPUT_METHOD, input_method);

	if (helper_client_io_send(applet->helper_client, msg) == FALSE) {
		IM_JA_DEBUG("Failed to send HELPER_MSG_SET_INPUT_METHOD to im-ja-helper.\n");
	}
	g_free(msg);

	/*
	if (helper_client_io_send(applet->helper_client, HELPER_MSG_QUERY_INPUT_METHOD) == FALSE) {
		IM_JA_DEBUG("Failed to send HELPER_MSG_QUERY_INPUT_METHOD to im-ja-helper.\n");
	}
	*/
	return;
}


/* FIXME: unused action menu callback */
gboolean im_ja_execute_action(ImJaApplet *applet, gint action_id, gboolean set_input_method) {
	IM_JA_DEBUG("im_ja_execute_action[applet]\n");
	/*
	if (set_input_method == TRUE) {
		set_input_method(applet, action_id);
		return TRUE;
	}
	switch (action_id) {
	case START_CONFIGURATOR:
		settings_cb(NULL, NULL, NULL);
		return TRUE;
	}
	*/
	return FALSE;
}

static void set_menu_position(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, ImJaApplet *applet) {
	GtkRequisition requisition;
	GdkScreen *screen;
	GtkWidget *widget = GTK_WIDGET(applet->applet);
	gint menu_x = 0;
	gint menu_y = 0;
	PanelAppletOrient orient;

	g_return_if_fail(PANEL_IS_APPLET(applet->applet));
	screen = gtk_widget_get_screen(widget);

	gtk_widget_size_request(GTK_WIDGET(menu), &requisition);

	gdk_window_get_origin(widget->window, &menu_x, &menu_y);

	menu_x += widget->allocation.x;
	menu_y += widget->allocation.y;

	orient = panel_applet_get_orient(applet->applet);
	if (orient == PANEL_APPLET_ORIENT_UP || orient == PANEL_APPLET_ORIENT_DOWN) {
		if (menu_y > gdk_screen_get_height (screen) / 2)menu_y -= requisition.height;
		else menu_y += widget->allocation.height;
	} 
	else {
		if (menu_x > gdk_screen_get_width (screen) / 2) menu_x -= requisition.width;
		else menu_x += widget->allocation.width;
	}
	*x = menu_x;
	*y = menu_y;
	*push_in = TRUE;
}

static void button_press_cb(GtkWidget *button, GdkEventButton *event, ImJaApplet *applet) {
  gint i;
	GtkWidget *menuitem;
	gchar *tmpstrg;
 
	if (event != NULL) {
		if (event->button == 3 || event->button == 2) {
			gtk_propagate_event(GTK_WIDGET(applet->applet), (GdkEvent *) event);
			return;
		}
	}

	applet->menu = gtk_menu_new();
	g_signal_connect_swapped(GTK_OBJECT(applet->menu), "destroy", G_CALLBACK(gtk_widget_destroy), NULL);
	g_signal_connect_swapped(GTK_OBJECT(applet->menu), "cancel", G_CALLBACK(gtk_widget_destroy), NULL);

	for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
		tmpstrg = g_strdup_printf("%s %s", applet->labels[i], gconf_enum_to_string(input_method_labels, i));
		menuitem = gtk_menu_item_new_with_label(tmpstrg);
		g_free(tmpstrg);
		g_object_set_data(G_OBJECT(menuitem), "input-method-number", (gpointer) i);
		gtk_widget_show(menuitem);
		gtk_menu_shell_append(GTK_MENU_SHELL(applet->menu), menuitem);
		g_signal_connect(GTK_OBJECT(menuitem), "activate", 
										 G_CALLBACK(set_input_method_cb), (gpointer) applet);
	}

	/*im_ja_actionmenu_populate(applet, GTK_MENU(applet->menu), IM_JA_PANEL_MENU);*/

	gtk_widget_show_all(applet->menu);
	gtk_menu_popup(GTK_MENU(applet->menu), NULL, NULL, (GtkMenuPositionFunc) set_menu_position, applet, 0, 
								 gtk_get_current_event_time());
}

static void button_clicked_cb(GtkWidget *button, ImJaApplet *applet) {
	button_press_cb(button, NULL, applet);
}

static void update_status_label(ImJaApplet *applet) {
	gtk_button_set_label(GTK_BUTTON(applet->status_widget), applet->labels[applet->input_method]);
}

static void update_status_label_strings(ImJaApplet *applet) {
	gchar *gconf_key;
	gint i;
	
	for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
		gconf_key = g_strdup_printf(GCONF_NAMESPACE"status_window/label_%s", gconf_enum_to_string(label_keys, i));
		applet->labels[i] = gconf_client_get_string(applet->gconf_client, gconf_key, NULL);
		g_free(gconf_key);
	}
	update_status_label(applet);
}

static void label_strings_changed_cb(GConfClient *client, guint cnxn_id, GConfEntry *entry, ImJaApplet *applet) {
	update_status_label_strings(applet);
}

static void got_focus_cb(GtkWidget *widget, GdkEventCrossing *event, ImJaApplet *applet) {
	gtk_button_set_relief(GTK_BUTTON(applet->status_widget), GTK_RELIEF_NORMAL);
	gtk_frame_set_shadow_type(GTK_FRAME(applet->border), GTK_SHADOW_NONE);
}

static void lost_focus_cb(GtkWidget *widget, GdkEventCrossing *event, ImJaApplet *applet) {
	gtk_button_set_relief(GTK_BUTTON(applet->status_widget), GTK_RELIEF_NONE);
	gtk_frame_set_shadow_type(GTK_FRAME(applet->border), GTK_SHADOW_ETCHED_IN);
}


void destroy_helper_client(ClientIO *client) {
	IM_JA_DEBUG("destroy_client: %d\n", (int) client);
	if (client != NULL) {
		helper_client_io_close(client);
		client = NULL;
	}
}

void process_helper_message(ImJaApplet *applet, gchar *msg) {
	gchar *ptr;
	gint input_method = -1;
	if (msg == NULL) return;

	msg[strlen(msg) - 1] = 0; /* remove trailing newline */
	IM_JA_DEBUG("processing [%s]\n", msg);

	if (g_str_has_prefix(msg, HELPER_MSG_PING) == TRUE) {
		helper_client_io_send(applet->helper_client, HELPER_MSG_ACK);
	}
	else if (g_str_has_prefix(msg, HELPER_MSG_CHANGE_STATUS) == TRUE) {
		ptr = msg + strlen(HELPER_MSG_CHANGE_STATUS);
		input_method = atoi(ptr);
		IM_JA_DEBUG("Change to: %d\n", input_method);
		applet->input_method = input_method;
		update_status_label(applet);
	}
	/*
	else if (g_str_has_prefix(msg, HELPER_MSG_SET_INPUT_METHOD) == TRUE) {
		ptr = msg + strlen(HELPER_MSG_SET_INPUT_METHOD);
		input_method = atoi(ptr);
		IM_JA_DEBUG("Change to: %d\n", input_method);
		applet->input_method = input_method;
		update_status_label(applet);
	}
	*/
	else {
		IM_JA_DEBUG("Unprocessed helper message: %s\n", msg);
	}
}



gboolean helper_input_handler(GIOChannel *source,
															GIOCondition condition,
															ImJaApplet *applet) {

	GIOStatus status;
	gchar *line = NULL;
	GError *err = NULL;

	IM_JA_DEBUG("im_ja_helper_input_handler()\n");

	if (condition & G_IO_ERR) {
		IM_JA_DEBUG("IO Error\n");
		destroy_helper_client(applet->helper_client);
		return FALSE;
	}

	if (condition & G_IO_IN) {
		IM_JA_DEBUG("data input:\n");

		status = g_io_channel_read_line(source, &line, NULL, NULL, &err);

		if (status & G_IO_STATUS_ERROR) {
			g_error("Error reading from client: %s\n", err->message);
			destroy_helper_client(applet->helper_client);
			return FALSE;
		}

		else if (status & G_IO_STATUS_EOF) {
			IM_JA_DEBUG("EOF\n");
			destroy_helper_client(applet->helper_client);
			return FALSE;
		}
		else if (status & G_IO_STATUS_NORMAL) {
			IM_JA_DEBUG("G_IO_STATUS_NORMAL\n");
			if ((line == NULL) || (strlen(line) == 0)) {
				IM_JA_DEBUG("connection closed\n");
				destroy_helper_client(applet->helper_client);
				return FALSE;
			}
			else { /* OK */
				process_helper_message(applet, line);
				g_free(line);
				line = NULL;
				return TRUE;
			}
		}
		else if (status & G_IO_STATUS_AGAIN) {
			IM_JA_DEBUG("G_IO_STATUS_AGAIN\n");
			return TRUE;
		}
		return TRUE;
	}

	else if (condition & G_IO_ERR) {
		IM_JA_DEBUG("io error\n");
		destroy_helper_client(applet->helper_client);
		return FALSE;
	}
	else if (condition & G_IO_HUP) {
		IM_JA_DEBUG("disconnection\n");
		destroy_helper_client(applet->helper_client);
		return FALSE;
	}
	else if (condition & G_IO_NVAL) {
		IM_JA_DEBUG("invalid request, file descriptor is closed.\n");
		destroy_helper_client(applet->helper_client);
		return FALSE;
	}
	return FALSE;
}



gboolean im_ja_applet_factory(PanelApplet *applet_widget, const gchar *iid, gpointer data) {
	ImJaApplet *applet;
  GError *error = NULL;

	if (strcmp(iid, "OAFIID:GNOME_ImJaApplet") != 0) return FALSE;

	applet = g_new0(ImJaApplet, 1);
	applet->applet = applet_widget;

	applet->helper_client = helper_client_io_new_connection(FALSE);
	if (applet->helper_client == NULL) {
		im_ja_print_error(_("Couldn't start im-ja-helper daemon."));
		return FALSE;
	}
	applet->helper_client->watch_id = g_io_add_watch(applet->helper_client->io,
																									 G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
																									 (GIOFunc) helper_input_handler, applet);


  if (gconf_init(0, NULL, &error) == FALSE) print_error("GConf init failed: %s\n", error->message);
  if (applet->gconf_client == NULL) applet->gconf_client = gconf_client_get_default();
  if (applet->gconf_client == NULL) print_error("Could not get gconf_client.\n");
  
	gconf_client_add_dir(applet->gconf_client, GCONF_NAMESPACE"status_window", GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_notify_add(applet->gconf_client,
													GCONF_NAMESPACE"status_window",
													(GConfClientNotifyFunc) label_strings_changed_cb,
													applet, NULL, &error);
								
	if (error != NULL) print_error("GConf init failed: %s\n", error->message);

	applet->vbox = gtk_vbox_new(FALSE, 0);

	applet->border = gtk_frame_new(NULL);
  gtk_box_pack_start(GTK_BOX(applet->vbox), applet->border, TRUE, TRUE, 0);

	applet->status_widget = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(applet->status_widget), GTK_RELIEF_NONE);
	
	gtk_container_add(GTK_CONTAINER(applet->border), applet->status_widget);

  g_signal_connect(G_OBJECT(applet->status_widget), "leave_notify_event",
									 G_CALLBACK(lost_focus_cb), applet);
	g_signal_connect(G_OBJECT(applet->status_widget), "focus_out_event",
									 G_CALLBACK(lost_focus_cb), applet);
  g_signal_connect(G_OBJECT(applet->status_widget), "enter_notify_event",
									 G_CALLBACK(got_focus_cb), applet);
  g_signal_connect(G_OBJECT(applet->status_widget), "focus_in_event",
									 G_CALLBACK(got_focus_cb), applet);
  g_signal_connect(G_OBJECT(applet->status_widget), "button_press_event",
									 G_CALLBACK(button_press_cb), applet);
  g_signal_connect(G_OBJECT(applet->status_widget), "clicked",
									 G_CALLBACK(button_clicked_cb), applet);

	applet->input_method = gconf_client_get_int(applet->gconf_client, GCONF_NAMESPACE"other/last_input_method", NULL);
	update_status_label_strings(applet);							

	gtk_container_add(GTK_CONTAINER(applet_widget), applet->vbox);
	gtk_widget_show_all(GTK_WIDGET(applet_widget));

	panel_applet_setup_menu_from_file(applet_widget,
																		NULL,
																		"GNOME_ImJaApplet.xml",
																		NULL,
																		im_ja_applet_menu_verbs,
																		applet);

	g_signal_connect(GTK_OBJECT(applet_widget), "change_size",
                           GTK_SIGNAL_FUNC(applet_change_size),
                           (gpointer)applet);
	g_signal_connect(GTK_OBJECT(applet_widget), "change_orient",
                           GTK_SIGNAL_FUNC(applet_change_orient),
                           (gpointer)applet);
	g_signal_connect(GTK_OBJECT(applet_widget), "destroy",
                           GTK_SIGNAL_FUNC(applet_destroy),
                           (gpointer)applet);

	return TRUE;
}	

PANEL_APPLET_BONOBO_FACTORY("OAFIID:GNOME_ImJaApplet_Factory", 
														PANEL_TYPE_APPLET,
														"im-ja-applet", 
														VERSION, 
														(PanelAppletFactoryCallback) im_ja_applet_factory, 
														NULL)
