/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *          
 */

#include <config.h>

#include <string.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkkeysyms.h>
#include <stdlib.h>


#include "conf.h"
#include "error.h"
#include "nls.h"

#define ENUMSIZE(s) sizeof(s)/sizeof(gchar *) 

GConfClient *gconf_client = NULL;

static gchar *key_states[] = { "SHFT",
															 "LOCK", 
															 "CTRL",
															 "MOD1",
															 "MOD2", 
															 "MOD3", 
															 "MOD4", 
															 "MOD5", 
															 "BUT1",
															 "BUT2",
															 "BUT3",
															 "BUT4",
															 "BUT5",
															 "BUT6"
};

GConfEnumStringPair hotkey_gconf_keys[] = {
  { START_CONFIGURATOR, "START_CONFIGURATOR" },
  { NEXT_INPUT_MODE, "NEXT_INPUT_MODE" },
  { PREV_INPUT_MODE, "PREV_INPUT_MODE" },
  { DIRECT_INPUT_MODE, "DIRECT_INPUT_MODE" },
  { HIRAGANA_INPUT_MODE, "HIRAGANA_INPUT_MODE" },
  { KATAKANA_INPUT_MODE, "KATAKANA_INPUT_MODE" },
  { HALFKATA_INPUT_MODE, "HALFKATA_INPUT_MODE" },
  { ZENKAKU_INPUT_MODE, "ZENKAKU_INPUT_MODE" },
  { CANNA_INPUT_MODE, "CANNA_INPUT_MODE" },
  { WNN_INPUT_MODE, "WNN_INPUT_MODE" },
  { KANJIPAD_INPUT_MODE, "KANJIPAD_INPUT_MODE" },
  { UNCONVERT, "UNCONVERT" },
  { CONVERT, "CONVERT" },
  { PREV_CANDIDATE, "PREV_CANDIDATE" },
  { NEXT_CANDIDATE, "NEXT_CANDIDATE" },
  { LIST_CANDIDATES, "LIST_CANDIDATES" },
  { PREV_BUNSETSU, "PREV_BUNSETSU" },
  { NEXT_BUNSETSU, "NEXT_BUNSETSU" },
  { EXPAND_BUNSETSU, "EXPAND_BUNSETSU" },
  { SHRINK_BUNSETSU, "SHRINK_BUNSETSU" },
  { TOGGLE_PREEDIT_WINDOW, "TOGGLE_PREEDIT_WINDOW" },
  { COMMIT_PREEDIT, "COMMIT_PREEDIT" },
  { SYMBOL_INPUT, "SYMBOL_INPUT" },
  { UNICODE_INPUT, "UNICODE_INPUT" },
  { JIS_CODE_INPUT, "JIS_CODE_INPUT" },
  { RADICAL_INPUT, "RADICAL_INPUT" },
  { -1, NULL}

};

static const gchar *preedit_gconf_keys[] = {
	"normal_foreground",
	"normal_background",
	"highlighted_foreground",
	"highlighted_background"
};

GConfEnumStringPair status_win_labels[] = {
	{IM_JA_DIRECT_INPUT, "direct"},
	{IM_JA_HIRAGANA_INPUT, "hiragana"},
	{IM_JA_KATAKANA_INPUT, "katakana"},
	{IM_JA_HALFKATA_INPUT, "halfkata"},
	{IM_JA_ZENKAKU_INPUT, "zenkaku"},
	{IM_JA_KANJIPAD_INPUT, "kanjipad"},
	{-1, NULL}
};

gboolean im_ja_init_conf_handler() {
  GError *error;

	IM_JA_DEBUG("im_ja_init_conf_handler()\n");

  if (gconf_init(0, NULL, &error) == FALSE) {
    im_ja_print_error(_("GConf init failed: %s\n"), error->message);
    return FALSE;
  }
  return TRUE;
}

void im_ja_dispose_gconf_client() {
	if (gconf_client != NULL) {
		IM_JA_DEBUG("calling g_object_unref(G_OBJECT(gconf_client)) [%d]\n", (int) gconf_client);
		g_object_unref(G_OBJECT(gconf_client));
		gconf_client = NULL;
	}
}

void im_ja_finalize_conf_handler(IMJAConfig *cfg) {
	/*  im_ja_save_conf(cfg); */
	im_ja_dispose_gconf_client();
}

gboolean im_ja_get_gconf_client() {
	IM_JA_DEBUG("im_ja_get_gconf_client()\n");
  if (gconf_client == NULL) {
		IM_JA_DEBUG("gconf_client_get_default()\n");
		gconf_client = gconf_client_get_default();
		IM_JA_DEBUG("got gconf_client [%d]\n", (int)gconf_client);
	}

  if (gconf_client == NULL) {
    im_ja_print_error(_("Could not get gconf_client.\n"));
    return FALSE;
  }
	return TRUE;
}

gchar *im_ja_get_keyname(guint state, guint keyval) {
  int i;
  gchar *keyname = g_new0(gchar, 140);

  for (i = 0; i <= 13; i++) {
    if (state & (1 << i)) {
      g_strlcat(keyname, key_states[i], 140);
      g_strlcat(keyname, " ", 140);
    }
  }
	if ((keyval == 0) || (keyval == GDK_VoidSymbol)) {
		g_strlcat(keyname, _("Undefined"), 140);
	}
	else g_strlcat(keyname, gdk_keyval_name(keyval), 140);
  if (strlen(keyname) == 0) g_strlcat(keyname, "???", 140);
  return keyname;
}

gchar *get_string_from_color(GdkColor *color) {
	gchar tmp[8];
	gchar *colorstrg = g_new0(gchar, 8);
	
	colorstrg[0] = '#';
	g_snprintf(tmp, 8, "%02X", color->red);
	colorstrg[1] = tmp[0];
	colorstrg[2] = tmp[1];

	g_snprintf(tmp, 8, "%02X", color->green);
	colorstrg[3] = tmp[0];
	colorstrg[4] = tmp[1];

	g_snprintf(tmp, 8, "%02X", color->blue);
	colorstrg[5] = tmp[0];
	colorstrg[6] = tmp[1];
	colorstrg[7] = 0;
	
	return colorstrg; 
}

gboolean im_ja_load_conf(IMJAConfig *cfg) {
  int i;
  guint keyval, keystate;
  gchar *gconf_key;
  gchar *colorstr;
	gchar *tmpstrg;
	const gchar *gconfstr;
	gchar *help_conf[3] = {
		"/desktop/gnome/url-handlers/ghelp/command",
		"/desktop/gnome/url-handlers/http/command",
		"/desktop/gnome/url-handlers/unknown/command"
	};

	IM_JA_DEBUG("im_ja_load_conf()\n");

  for (i = 0; hotkey_gconf_keys[i].enum_value != -1; i++) {
		gconfstr = gconf_enum_to_string(hotkey_gconf_keys, i);
		if (gconfstr != NULL) {
			gconf_key = g_strdup_printf(GCONF_NAMESPACE"hotkeys/%s_keyval", gconfstr);
			keyval = gconf_client_get_int(gconf_client, gconf_key, NULL);
			IM_JA_DEBUG("%s: %d\n", gconf_key, keyval);
			g_free(gconf_key);
			if ((keyval != 0) && (keyval != GDK_VoidSymbol)) {
				cfg->hotkey_values[i] = keyval;
				gconf_key = g_strdup_printf(GCONF_NAMESPACE"hotkeys/%s_state", gconfstr);
				keystate = gconf_client_get_int(gconf_client, gconf_key, NULL);
				g_free(gconf_key);
				cfg->hotkey_states[i] = keystate;
			}
		}
		else {
			IM_JA_DEBUG(" NULL for %d\n", i);
		}
	}
	for (i = 0; i < ENUMSIZE(preedit_gconf_keys); i++) {
		gconf_key = g_strdup_printf(GCONF_NAMESPACE"preedit_style/%s", preedit_gconf_keys[i]);
		colorstr = gconf_client_get_string(gconf_client, gconf_key, NULL);
		if (colorstr != NULL) {
			gdk_color_parse((const gchar*)colorstr, &cfg->preedit_colors[i]);
			/* IM_JA_DEBUG("%s: %s\n", gconf_key, colorstr); */
		}
		g_free(colorstr);
		g_free(gconf_key);
	}

	cfg->custom_preedit_n =	gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"preedit_style/custom_normal", NULL);
	cfg->custom_preedit_hl = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"preedit_style/custom_highlight", NULL);

	cfg->status_win_position = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"status_window/position", NULL);
	cfg->status_win_coords[0] = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"status_window/fixed_xpos", NULL);
	cfg->status_win_coords[1] = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"status_window/fixed_ypos", NULL);

	for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
		gconf_key = g_strdup_printf(GCONF_NAMESPACE"status_window/label_%s", gconf_enum_to_string(status_win_labels, i));
		cfg->status_win_labels[i] = gconf_client_get_string(gconf_client, gconf_key, NULL);
		if (cfg->status_win_labels[i] == NULL) cfg->status_win_labels[i] = "";
		/*IM_JA_DEBUG("%s: %s\n", gconf_key, cfg->status_win_labels[i]); */
		g_free(gconf_key);
	}
	if (strcmp(cfg->status_win_labels[IM_JA_DIRECT_INPUT], "") == 0) cfg->status_win_labels[IM_JA_DIRECT_INPUT] = "a";

	cfg->default_conv_engine = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"conv_engine/default_conv_engine", NULL);

	IM_JA_DEBUG(" DEFAULT CONV ENGINE: %d\n", cfg->default_conv_engine);
	/*
	//	cfg->wnn_port =	gconf_client_get_int(gconf_client, GCONF_NAMESPACE"conv_engine/wnn_port", NULL);
	//	cfg->canna_port =	gconf_client_get_int(gconf_client, GCONF_NAMESPACE"conv_engine/canna_port", NULL);
	*/
	cfg->wnn_address = gconf_client_get_string(gconf_client, GCONF_NAMESPACE"conv_engine/wnn_address", NULL);
	cfg->wnnenvrc = gconf_client_get_string(gconf_client, GCONF_NAMESPACE"conv_engine/wnnenvrc", NULL);
	/*	cfg->canna_address = gconf_client_get_string(gconf_client, GCONF_NAMESPACE"conv_engine/canna_address", NULL); */

	cfg->kanjipad_enabled = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_enabled", NULL);
	cfg->kanjipad_custom_size = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_custom_size", NULL);
	cfg->kanjipad_size[0] = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_width", NULL);
	cfg->kanjipad_size[1] = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_height", NULL);

	cfg->preedit_window_on = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"other/preedit_window_on", NULL);
	cfg->startup_input_method = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"other/startup_input_method", NULL);
	cfg->im_ja_version = gconf_client_get_string(gconf_client, GCONF_NAMESPACE"other/im_ja_version", NULL);
	cfg->candwin_style = gconf_client_get_int(gconf_client, GCONF_NAMESPACE"other/candwin_style", NULL);
	cfg->candwin_font = gconf_client_get_string(gconf_client, GCONF_NAMESPACE"other/candwin_font", NULL);
	cfg->custom_candwin_font = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"other/custom_candwin_font", NULL);
	cfg->commit_on_reset = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"other/commit_on_reset", NULL);
	cfg->use_systray = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"other/use_systray", NULL);

	cfg->dont_show_save_warning = gconf_client_get_bool(gconf_client, GCONF_NAMESPACE"other/dont_show_save_warning", NULL);
	
	if (cfg->im_ja_version != NULL) {
		IM_JA_DEBUG(" CFG Version: %s\n", cfg->im_ja_version);
	}

	for (i = 0; i < ENUMSIZE(help_conf); i++) { 
		tmpstrg = gconf_client_get_string(gconf_client, help_conf[i], NULL);
		if (tmpstrg != NULL) break;
	}

	if ((tmpstrg == NULL) || (strlen(tmpstrg) == 0)) {
		cfg->browser = "/usr/bin/netscape %s";
	}
	else {
		cfg->browser = g_strdup_printf("/usr/bin/%s", tmpstrg);
		g_free(tmpstrg);
	}
	return TRUE;
}

gboolean im_ja_save_conf(IMJAConfig *cfg) {
  int i;
  gchar *gconf_key;
  gchar *colorstr;
	const gchar *gconfstr;

  for (i = 0; i < HOTKEYS_TOTAL; i++) {
		gconfstr = gconf_enum_to_string(hotkey_gconf_keys, i);
		if (gconfstr != NULL) {
			gconf_key = g_strdup_printf(GCONF_NAMESPACE"hotkeys/%s_keyval", gconfstr);
			if (cfg->hotkey_values[i] != 0) {
				gconf_client_set_int(gconf_client, gconf_key, cfg->hotkey_values[i], NULL);
				IM_JA_DEBUG("%s: %d\n", gconf_key, cfg->hotkey_values[i]);
			}
			g_free(gconf_key);
    
			gconf_key = g_strdup_printf(GCONF_NAMESPACE"hotkeys/%s_state", gconfstr);

			if (cfg->hotkey_values[i] != 0) {
				gconf_client_set_int(gconf_client, gconf_key, cfg->hotkey_states[i], NULL);
				IM_JA_DEBUG("%s: %d\n", gconf_key, cfg->hotkey_states[i]);
			}
			g_free(gconf_key);
		}
	}

  for (i = 0; i < ENUMSIZE(preedit_gconf_keys); i++) {
		gconf_key = g_strdup_printf(GCONF_NAMESPACE"preedit_style/%s", preedit_gconf_keys[i]);
		/* IM_JA_DEBUG("%s: %s\n", gconf_key, get_string_from_color(&cfg->preedit_colors[i])); */
		colorstr = get_string_from_color(&cfg->preedit_colors[i]);
		gconf_client_set_string(gconf_client, gconf_key, colorstr, NULL);
		g_free(colorstr);
		g_free(gconf_key);
	}

	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"preedit_style/custom_normal", cfg->custom_preedit_n, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"preedit_style/custom_highlight", cfg->custom_preedit_hl, NULL);

	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"status_window/position", cfg->status_win_position, NULL);
	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"status_window/fixed_xpos", cfg->status_win_coords[0], NULL);
	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"status_window/fixed_ypos", cfg->status_win_coords[1], NULL);

	
	for (i = 0; i < IM_JA_INPUT_METHODS_TOTAL; i++) {
		gconf_key = g_strdup_printf(GCONF_NAMESPACE"status_window/label_%s", gconf_enum_to_string(status_win_labels, i));
		if (cfg->status_win_labels[i] != NULL) gconf_client_set_string(gconf_client, gconf_key, cfg->status_win_labels[i], NULL);
		g_free(gconf_key);
	}
	
	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"conv_engine/default_conv_engine", cfg->default_conv_engine, NULL);
  /*
	//	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"conv_engine/wnn_port", cfg->wnn_port, NULL);
	//	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"conv_engine/canna_port", cfg->canna_port, NULL);
	*/
	gconf_client_set_string(gconf_client, GCONF_NAMESPACE"conv_engine/wnn_address", cfg->wnn_address, NULL);
	gconf_client_set_string(gconf_client, GCONF_NAMESPACE"conv_engine/wnnenvrc", cfg->wnnenvrc, NULL);
  /*	gconf_client_set_string(gconf_client, GCONF_NAMESPACE"conv_engine/canna_address", cfg->canna_address, NULL); */

	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_enabled", cfg->kanjipad_enabled, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_custom_size", cfg->kanjipad_custom_size, NULL);
	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_width", cfg->kanjipad_size[0], NULL);
	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"conv_engine/kanjipad_height", cfg->kanjipad_size[1], NULL);

	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"other/startup_input_method", cfg->startup_input_method, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"other/preedit_window_on", cfg->preedit_window_on, NULL);
	gconf_client_set_string(gconf_client, GCONF_NAMESPACE"other/im_ja_version", VERSION, NULL);
	gconf_client_set_int(gconf_client, GCONF_NAMESPACE"other/candwin_style", cfg->candwin_style, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"other/custom_candwin_font", cfg->custom_candwin_font, NULL);
	if (cfg->candwin_font != NULL) {
		gconf_client_set_string(gconf_client, GCONF_NAMESPACE"other/candwin_font", 	cfg->candwin_font, NULL);
	}
	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"other/commit_on_reset", cfg->commit_on_reset, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"other/dont_show_save_warning", cfg->dont_show_save_warning, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NAMESPACE"other/use_systray", cfg->use_systray, NULL);
	
	return TRUE;
}

