/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *          
 */

#include <config.h>

#include <string.h>
#include <math.h>

#include <gtk/gtk.h>
#include <gtk/gtkimmodule.h>
#include <gdk/gdkkeysyms.h>

#include "im-ja.h"
#include "im-ja-impl.h"
#include "error.h"
#include "conf.h"
#include "common.h"
#include "nls.h"
#include "candwin.h"
#include "statuswin.h"
#include "kanjipad/kanjipad.h"

extern IMJAConfig cfg; 

static void candidate_window_set_selection(GtkWidget *menu_item, IMJAContext *cn) {
	gchar *string;
	GtkWidget *label;
	gint selected;

	IM_JA_DEBUG("candidate_window_set_selection()\n");
	
	/* get the label of the menuitem */
	label = gtk_container_get_children(GTK_CONTAINER(menu_item))->data;
	string = (gchar *)gtk_label_get_text(GTK_LABEL(label));

	selected = (gint) g_object_get_data(G_OBJECT(menu_item), "candidate-number");
	if (cn->input_method == IM_JA_KANJIPAD_INPUT) {
		g_strlcpy(cn->preedit_buf, string, BUFFERSIZE);
		im_ja_commit(cn);
	}
	else {
		cn->im_ja_conv_engine_select_candidate(cn, selected);
		cn->im_ja_conv_engine_update_preedit(cn);
	}
	candidate_window_hide(cn);
	status_window_show(cn);
}

static void candidate_window_change_selection(GtkWidget *menu_item, IMJAContext *cn) {
	gint selected = 0;
	gchar *new_preedit;
	GtkWidget *label;
	gchar *string;

	/* get the label of the menuitem */
	label = gtk_container_get_children(GTK_CONTAINER(menu_item))->data;
	string = (gchar *)gtk_label_get_text(GTK_LABEL(label));

	selected = (gint) g_object_get_data(G_OBJECT(menu_item), "candidate-number");
	IM_JA_DEBUG("candidate_window_change_selection: %d\n", selected);

	if ((cn->conv_engine == CONV_ENGINE_WNN) &&
			(cn->input_method != IM_JA_KANJIPAD_INPUT)) {
		cn->im_ja_conv_engine_select_candidate(cn, selected);
		cn->im_ja_conv_engine_update_preedit(cn);
	}
	else { /* substitute `string' into highlighted part */
		new_preedit = g_new0(gchar, BUFFERSIZE);
		strncpy(new_preedit, cn->preedit_buf, cn->preedit_reverse_start);
		g_strlcat(new_preedit, string, BUFFERSIZE);
		g_strlcat(new_preedit, cn->preedit_buf + cn->preedit_reverse_end, BUFFERSIZE);
		cn->preedit_reverse_end = strlen(string);
		g_strlcpy(cn->preedit_buf, new_preedit, BUFFERSIZE);
		cn->preedit_reverse_end = cn->preedit_reverse_start + strlen(string);
		g_free(new_preedit);
		im_ja_preedit_changed(cn);
	}
}

static void candidate_window_set_position(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, IMJAContext *cn) {
	gint target_x = 0, target_y = 0;
	GdkRectangle toplevel_rect;
	GdkRectangle client_win;
		
	im_ja_get_client_window_geometry(cn, &client_win);
	im_ja_get_toplevel_window_geometry(cn, &toplevel_rect);

	if ((cn->candwin_pos_offset_x == 0) && (cn->candwin_pos_offset_y == 0)) {
		/* If we don't receive the position properly, center on client window */
		GtkRequisition requisition;
		gtk_widget_size_request(GTK_WIDGET(menu), &requisition);
		*x = toplevel_rect.x + toplevel_rect.width / 2 - requisition.width / 2;
		*y = toplevel_rect.y + toplevel_rect.height / 2 - requisition.height / 2;
	}
	else {
		target_x = client_win.x + cn->candwin_pos_offset_x;
		target_y = client_win.y + cn->candwin_pos_offset_y;
		/* check if target_ is within the toplevel window */
		if ((toplevel_rect.width != 0) || (toplevel_rect.height != 0)) {
			if (target_y > toplevel_rect.y + toplevel_rect.height) target_y = toplevel_rect.y + toplevel_rect.height;
			if (target_x > toplevel_rect.x + toplevel_rect.width) target_x = toplevel_rect.x + toplevel_rect.width;
		}
		*x = target_x;
		*y = target_y;
	}

	IM_JA_DEBUG("candwin pos x: %d, y: %d\n", *x, *y);
}

static gboolean key_press_cb(GtkWidget *widget, GdkEventKey *event, IMJAContext* cn) {
	gint ignorekeys[] = { GDK_Up, GDK_KP_Up, GDK_Down, GDK_KP_Down, GDK_KP_Left, 
												GDK_KP_Right, GDK_Home, GDK_KP_Home, GDK_End, GDK_KP_End, GDK_Page_Up,
												GDK_KP_Page_Up, GDK_Page_Down, GDK_KP_Page_Down, GDK_Return };
	gint i;
	gboolean found = FALSE;
	gint size = sizeof(ignorekeys)/sizeof(gint);
	IM_JA_DEBUG("candidate_win key_press_cb()\n");

	if (cn->candwin_style == CANDWIN_STYLE_TABLE) {
		if (event->keyval == GDK_Escape) {
			candidate_window_hide(cn);
			return TRUE;
		}
		return FALSE;
	}
	else { /* CANDWIN_STYLE_MENU */
		/* Select the next candidate if space was pressed */
		if (event->keyval == GDK_space) {
			GList *node = g_list_find(GTK_MENU_SHELL(cn->candidate_win->window)->children,
																GTK_MENU_SHELL(cn->candidate_win->window)->active_menu_item);
			node = node->next;
			if (node == NULL) node = GTK_MENU_SHELL(cn->candidate_win->window)->children;
			gtk_menu_shell_select_item(GTK_MENU_SHELL(cn->candidate_win->window), node->data);
			return TRUE;
		}
	
		/* Handle next/prev bunsetu selection */
		if ((ishotkey(event, NEXT_BUNSETSU, &cfg) == TRUE) ||
				(ishotkey(event, PREV_BUNSETSU, &cfg) == TRUE) ||
				(event->keyval == GDK_Left) || (event->keyval == GDK_Right)) {
			GList *node = g_list_find(GTK_MENU_SHELL(cn->candidate_win->window)->children,
																GTK_MENU_SHELL(cn->candidate_win->window)->active_menu_item);
			candidate_window_set_selection(node->data, cn);
			candidate_window_hide(cn);
			return im_ja_filter_keypress(cn, event);
		}

		/* Check if this key is handled by the menu or not */
		for (i = 0; i < size; i++) {
			if (ignorekeys[i] == (gint) (event->keyval)) {
				found = TRUE;
				break;
			}
		}

		/* Only process keys that the menu doesn't use for setting it's selections */
		if (found == FALSE) {
			IM_JA_DEBUG("key press in candidate menu.\n");
			candidate_window_hide(cn);
			return im_ja_filter_keypress(cn, event); /* Maybe we should commit ? */
		}
		else return FALSE;
	}
}

/*
static void candidate_window_cancel_cb(IMJAContext* cn) {
	IM_JA_DEBUG("candidate_window_cancel_cb()\n");

	if (selection_ok == TRUE) return;
	if (cn->candidate_win == NULL) return;
	if (GTK_IS_WIDGET(cn->candidate_win->window) == FALSE) return;

	if (cn->conv_engine == CONV_ENGINE_CANNA) {
#ifndef DISABLE_CANNA
		IM_JA_DEBUG("FIXME: candidate_window_cancel_cb for canna\n");
		//canna_rk_choose_candidate(cn, selected);
#endif
	}
	else if (cn->conv_engine == CONV_ENGINE_WNN) {
#ifndef DISABLE_WNN
		im_ja_wnn_select_candidate(cn, initial_selection);
		im_ja_wnn_update_preedit(cn);
#endif
	}
	candidate_window_hide(cn);
}
*/

void candidate_window_set_font_on_child(GtkWidget *parent) {
	GtkWidget *label;
	PangoFontDescription *desc;

	IM_JA_DEBUG("Setting custom font: %s\n", cfg.candwin_font);
	desc = pango_font_description_from_string(cfg.candwin_font);
	label = gtk_bin_get_child(GTK_BIN(parent));

	gtk_widget_modify_font(label, desc);
	pango_font_description_free(desc);
}


void candidate_window_show(IMJAContext *cn, gint selected) {
	GtkWidget *candidate_item;
	GList *tmplist = NULL;
	gchar *tmpptr;
	gint cnt = 0;
	gint total = 0;

	IM_JA_DEBUG("show_candwin() with selected: %d\n", selected);

	if (cn->candidate_win != NULL) {
		gtk_widget_destroy(cn->candidate_win->window);
	}
	cn->candidate_win = g_new(CandidateWin, 1);

	if ((cn->cursor_pos_offset_x == 0) && (cn->cursor_pos_offset_y == 0)) {
		/* no position is available */
		cn->candwin_style = CANDWIN_STYLE_TABLE;
	}

	if (cn->candwin_style == CANDWIN_STYLE_TABLE) {
		GtkWidget *candidate_table;
		gint rows = 0;
		gint cols = 0;
		gint col, row;
		gdouble root = 0;
		tmplist = cn->candidate_list;
		/* count the elements in the table */
		while (tmplist != NULL) {
			total++;
			tmplist = g_list_next(tmplist);
		}
		/* Calculate table size; */
		root = sqrt((double) total);
		rows = cols = (gint) root;
		if (((gdouble) rows) < root) {
			rows++;
			if (rows * cols < total) cols++;
		}
		/* IM_JA_DEBUG("Rows: %d, cols: %d\n", rows, cols); */

		/* Create the table */
		cn->candidate_win->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(cn->candidate_win->window), _("Candidates"));
		gtk_window_set_modal(GTK_WINDOW(cn->candidate_win->window), TRUE);

		/* FIXME */
#ifdef IMJA_TARGET_GTK 
		if (GTK_IS_WINDOW(cn->toplevel_gtk)) {
			gtk_window_set_transient_for(GTK_WINDOW(cn->candidate_win->window), GTK_WINDOW(cn->toplevel_gtk));
		}
#endif

		candidate_table = gtk_table_new(rows, cols, TRUE); 
		gtk_container_add(GTK_CONTAINER(cn->candidate_win->window), candidate_table);

		gtk_window_set_type_hint(GTK_WINDOW(cn->candidate_win->window), GDK_WINDOW_TYPE_HINT_DIALOG);

   	tmplist = cn->candidate_list;
		col = row = 0;
		while (tmplist != NULL) {
			while (col < cols) {
				tmpptr = (gchar *)tmplist->data;
				if (tmplist->data != NULL) {
					GtkWidget *tmpwidget = gtk_button_new_with_label(tmplist->data);
					if (cfg.custom_candwin_font == TRUE) candidate_window_set_font_on_child(tmpwidget);
					g_object_set_data(G_OBJECT(tmpwidget), "candidate-number", (gpointer) cnt);

					g_signal_connect(G_OBJECT(tmpwidget), "clicked", G_CALLBACK(candidate_window_set_selection), cn);

					gtk_table_attach(GTK_TABLE(candidate_table), tmpwidget, col, col + 1, row, row + 1,
													 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
													 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
					tmplist = g_list_next(tmplist);
					col++;
					if (tmplist == NULL) break;
				}
				else {
					IM_JA_DEBUG(" [candidates] null received in GList\n");
					tmplist = g_list_next(tmplist);
				}
				cnt++;
			}
			col = 0;
			row++;
		}
	}
	else { /* CANDWIN_STYLE_MENU */
		cn->candidate_win->window = gtk_menu_new();
		tmplist = cn->candidate_list;
		while (tmplist != NULL) {
			tmpptr = (gchar *)tmplist->data;
			if (tmplist->data != NULL) {
				candidate_item = gtk_menu_item_new_with_label(tmplist->data);
				if (cfg.custom_candwin_font == TRUE) candidate_window_set_font_on_child(candidate_item);
				gtk_widget_show(candidate_item);
				g_object_set_data(G_OBJECT(candidate_item), "candidate-number", (gpointer) cnt);
				gtk_menu_shell_append(GTK_MENU_SHELL(cn->candidate_win->window), candidate_item);
				if (cnt == selected) 	gtk_menu_shell_select_item(GTK_MENU_SHELL(cn->candidate_win->window), candidate_item);
				
				g_signal_connect(GTK_OBJECT(candidate_item), "select", 
												 G_CALLBACK(candidate_window_change_selection), cn);
				g_signal_connect(GTK_OBJECT(candidate_item), "activate", 
												 G_CALLBACK(candidate_window_set_selection), cn);
				tmplist = g_list_next(tmplist);
			}
			else {
				IM_JA_DEBUG(" [candidates] null received in GList\n");
				tmplist = g_list_next(tmplist);
			}
			cnt++;	
		}
		/*
		g_signal_connect_swapped(GTK_OBJECT(cn->candidate_win->window), "deactivate", G_CALLBACK(candidate_window_cancel_cb), cn);
		g_signal_connect_swapped(GTK_OBJECT(cn->candidate_win->window), "cancel", G_CALLBACK(candidate_window_cancel_cb), cn);
		g_signal_connect_swapped(GTK_OBJECT(cn->candidate_win->window), "selection-done", G_CALLBACK(candidate_window_selection_done), cn);
		*/
	}
	g_signal_connect(GTK_OBJECT(cn->candidate_win->window), "key_press_event", G_CALLBACK(key_press_cb), cn);
	cn->candidate_win->destroy_handler = g_signal_connect_swapped(GTK_OBJECT(cn->candidate_win->window),
																																"destroy", G_CALLBACK(candidate_window_hide), cn);

	gtk_widget_show_all(cn->candidate_win->window);

	/* FIXME: SET THE PARENT SOMEHOW so that it is centered on the app and doesn't flash because of the WM */
#ifdef IMJA_TARGET_XIM
	if (cn->candwin_style == CANDWIN_STYLE_TABLE) {
		im_ja_center_on_client_win(cn, GTK_WINDOW(cn->candidate_win->window));
		IM_JA_DEBUG("CANDWIN IS MODAL: %d\n", gtk_window_get_modal(GTK_WINDOW(cn->candidate_win->window)));
	}
#endif

	g_object_set_data(G_OBJECT(cn), "im-ja-candidate-window", cn->candidate_win);

	if (cn->candwin_style == CANDWIN_STYLE_MENU) {
		gtk_menu_popup(GTK_MENU(cn->candidate_win->window), NULL, NULL, 
									 (GtkMenuPositionFunc) candidate_window_set_position, cn, 0, 
									 gtk_get_current_event_time());
	}

	status_window_hide(cn);
	kanjipad_hide(cn);
}

gboolean candidate_window_is_shown(IMJAContext* cn) {
	/* IM_JA_DEBUG("candidate_window_is_shown: %d\n", (int) g_object_get_data(G_OBJECT(cn), "im-ja-candidate-window")); */

	if (g_object_get_data(G_OBJECT(cn), "im-ja-candidate-window") != NULL) return TRUE;
	return FALSE;
}

void candidate_window_destroy(IMJAContext* cn) {
	/* IM_JA_DEBUG("candidate_window_hide()\n"); */
	if (cn->candidate_win != NULL) {
		GtkWidget *tmpwidget = cn->candidate_win->window;
		cn->candidate_win->window = NULL;
		if (GTK_IS_WIDGET(tmpwidget) == TRUE) {
			gtk_widget_hide_all(tmpwidget);
			gtk_widget_destroy(tmpwidget);
		}
		g_free(cn->candidate_win);
		cn->candidate_win = NULL;
	}
}


void candidate_window_hide(IMJAContext* cn) {
	IM_JA_DEBUG("candidate_window_hide()\n");
	g_object_set_data(G_OBJECT(cn), "im-ja-candidate-window", NULL);
	candidate_window_destroy(cn);
	status_window_show(cn);
}

