/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 */
#include <config.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <anthy/anthy.h>
#include <anthy/input.h>

#include "../nls.h"
#include "../im-ja.h"
#include "../im-ja-impl.h"
#include "../error.h"
#include "../common.h"
#include "../conf.h"
#include "../romakana.h"
#include "../candwin.h"

#include "anthy.h"

/* * * * * * * * * * * */
extern IMJAConfig cfg;
/* * * * * * * * * * * */

void im_ja_anthy_shutdown(IMJAContext *cn) {
	IM_JA_DEBUG("FIXME: im_ja_anthy_shutdown()\n");
}

gboolean im_ja_anthy_init(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_anthy_init\n");
	if (anthy_input_init() == -1) {
		IM_JA_DEBUG(" anthy_input_init() failed!\n");
		return FALSE;
	}
	cn->anthy_context.cfg = anthy_input_create_config();
	cn->anthy_context.ictx = anthy_input_create_context(cn->anthy_context.cfg);

	cn->im_ja_conv_engine_reset_context = im_ja_anthy_reset;
	cn->im_ja_conv_engine_filter_keypress = im_ja_anthy_filter_keypress;
	cn->im_ja_conv_engine_shutdown = im_ja_anthy_shutdown;
	cn->im_ja_conv_engine_select_candidate = im_ja_anthy_select_candidate;
	cn->im_ja_conv_engine_update_preedit = im_ja_anthy_update_preedit;
	cn->im_ja_conv_engine_commit = im_ja_anthy_commit;

	return TRUE;
}

void im_ja_anthy_show_candidates(IMJAContext *cn) {
	gint cand, candnum;
	struct anthy_input_segment *cand_segment;

	IM_JA_DEBUG("im_ja_anthy_show_candidates()\n");
	im_ja_free_candidate_list(cn);
	if (cn->anthy_context.preedit == NULL) {
		IM_JA_DEBUG("**ERROR** no preedit!\n");
		return;
	}
	if (cn->anthy_context.preedit->cur_segment == NULL) {
		IM_JA_DEBUG("**ERROR** no selected clause!\n");
		return;
	}
	candnum = cn->anthy_context.preedit->cur_segment->nr_cand;
	for (cand = 0; cand < candnum; cand++) {
		cand_segment = anthy_input_get_candidate(cn->anthy_context.ictx, cand);
		cn->candidate_list = g_list_append(cn->candidate_list, euc2utf8(cand_segment->str));
	}
	candidate_window_show(cn, cn->anthy_context.preedit->cur_segment->cand_no);
}

/*
static void im_ja_anthy_convert(IMJAContext *cn) {

}
*/
/*
static void im_ja_anthy_get_new_preedit(IMJAContext *cn) {
	cn->anthy_context.preedit = anthy_input_get_preedit(cn->anthy_context.ictx);
}
*/

void im_ja_anthy_debug_context(IMJAContext *cn) {
	struct anthy_input_segment *anthy_segment;
	gchar *utf8str;

	anthy_segment = cn->anthy_context.preedit->segment;

	if ((cn->anthy_context.preedit) 
			&& (cn->anthy_context.preedit->cur_segment)
			&& (cn->anthy_context.preedit->cur_segment->str)) {
		IM_JA_DEBUG(" CURRENT:%s\n", euc2utf8(cn->anthy_context.preedit->cur_segment->str));
	}
	IM_JA_DEBUG(" ANTHY PREEDIT: ");
	while (anthy_segment != NULL) {
		if (anthy_segment->str != NULL) {
			utf8str = euc2utf8(anthy_segment->str);
			if (cn->anthy_context.preedit->cur_segment == anthy_segment) {
				IM_JA_DEBUG("[%s]", utf8str);
			}
			else IM_JA_DEBUG("%s ", utf8str);
		}
		if (anthy_segment->flag == ANTHY_INPUT_SF_CURSOR) {
			IM_JA_DEBUG("|");
		}
		anthy_segment = anthy_segment->next;
	}
	IM_JA_DEBUG("\n");
}

void im_ja_anthy_update_preedit(IMJAContext *cn) {
	struct anthy_input_segment *anthy_segment;
	gchar *utf8str;
	gint cursor_found = FALSE, reverse_start = 0;

	IM_JA_DEBUG("im_ja_anthy_update_preedit()\n");

	cn->anthy_context.preedit = anthy_input_get_preedit(cn->anthy_context.ictx);

	if (cn->anthy_context.preedit == NULL) return;
	anthy_segment = cn->anthy_context.preedit->segment;

	cn->cursor_char_pos = 0;
	cn->preedit_reverse_start = 0;
	cn->preedit_reverse_end = 0;
	*cn->preedit_buf = 0;

	while (anthy_segment != NULL) {
		if (anthy_segment->str != NULL) {
			utf8str = euc2utf8(anthy_segment->str);
			if (cn->anthy_context.preedit->cur_segment == anthy_segment) {
				cn->preedit_reverse_start = reverse_start;
				cn->preedit_reverse_end = reverse_start + strlen(utf8str);
				cn->update_candwin_pos = TRUE;
			}
			else if (cursor_found != TRUE) cn->cursor_char_pos += g_utf8_strlen(utf8str, -1);
			reverse_start += strlen(utf8str);
			g_strlcat(cn->preedit_buf, utf8str, BUFFERSIZE);
		}
		if (anthy_segment->flag == ANTHY_INPUT_SF_CURSOR) {
			cursor_found = TRUE;
		}
		/*
		if (anthy_segment->flag == ANTHY_INPUT_SF_CURSOR) {
			cursor_found = TRUE;
			//display cursor on the beginning of the converting clause
			if ((cn->anthy_context.preedit->cur_segment == anthy_segment) 
					|| (anthy_segment->str != NULL)) {
				cn->cursor_char_pos -= g_utf8_strlen(utf8str, -1);
			}
		}
		*/
		anthy_segment = anthy_segment->next;
	}
	/* im_ja_anthy_debug_context(cn); */
	im_ja_preedit_changed(cn);
}

void im_ja_anthy_reset(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_anthy_reset\n");
	anthy_input_quit(cn->anthy_context.ictx);
}


void im_ja_anthy_commit(IMJAContext *cn) {
	anthy_input_commit(cn->anthy_context.ictx);
}

gboolean im_ja_anthy_filter_keypress(IMJAContext *cn, GdkEventKey *key) {

	if (cn->input_method != IM_JA_HIRAGANA_INPUT) {
		/* FIXME use anthy map if it works! */
		return im_ja_kana_filter_keypress(cn, key);
	}
  
  if( key->type == GDK_KEY_RELEASE ) {
    return FALSE;
  };

	IM_JA_DEBUG("im_ja_anthy_filter_keypress()\n");

  /* ENTER */
  if ((key->keyval == GDK_Return) || (key->keyval == GDK_KP_Enter) || (ishotkey(key, COMMIT_PREEDIT, &cfg) == TRUE)) {
		if (strlen(cn->preedit_buf) > 0) { /* Commit preedit buffer */
			im_ja_commit(cn);
			return TRUE;
		}
		return FALSE;
  }

  /* BACKSPACE */
  if (key->keyval == GDK_BackSpace ) {
		if (strlen(cn->preedit_buf) > 0) { /* Delete preedit */
			anthy_input_erase_prev(cn->anthy_context.ictx);
			im_ja_anthy_update_preedit(cn);
			return TRUE;
		}
		return FALSE; /* Delete normal */
  }

  /* DELETE */
  if (key->keyval == GDK_Delete ) {
		if (strlen(cn->preedit_buf) > 0) { /* Delete preedit */
			anthy_input_erase_next(cn->anthy_context.ictx);
			im_ja_anthy_update_preedit(cn);
			return TRUE;
		}
    return FALSE; /* Delete normal */
  }

  /* SPACE */
  if (key->keyval == GDK_space) { 
    if (strlen(cn->preedit_buf) == 0) {
			g_strlcpy(cn->preedit_buf, " ", BUFFERSIZE);
			im_ja_commit(cn); /* Insert a space */
			return TRUE;         
    }
		if ((cn->anthy_context.preedit != NULL) 
				&& (cn->anthy_context.preedit->state == ANTHY_INPUT_ST_CONV)) {
			IM_JA_DEBUG("CONVERTED!\n");
			im_ja_anthy_show_candidates(cn);
			return TRUE;
		}
		anthy_input_space(cn->anthy_context.ictx);
		im_ja_anthy_update_preedit(cn);
		return TRUE; 
  }

  /* Convert current bunsetsu, if converted show the next candidate */
  if (ishotkey(key, CONVERT, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		if ((cn->anthy_context.preedit != NULL) 
				&& (cn->anthy_context.preedit->state == ANTHY_INPUT_ST_CONV)) {
			anthy_input_next_candidate(cn->anthy_context.ictx);
		}
		else anthy_input_space(cn->anthy_context.ictx);
		im_ja_anthy_update_preedit(cn);
    return TRUE;
  }

  /* Unconvert current bunsetsu */
  if (ishotkey(key, UNCONVERT, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		if ((cn->anthy_context.preedit != NULL) 
				&& (cn->anthy_context.preedit->state == ANTHY_INPUT_ST_CONV)) {
			anthy_input_quit(cn->anthy_context.ictx);
			im_ja_anthy_update_preedit(cn);
			return TRUE;
		}
		return FALSE;
  }

  /* Show candidates, if not converted then convert also */
  if (ishotkey(key, LIST_CANDIDATES, &cfg) == TRUE) {
		if ((cn->anthy_context.preedit != NULL) 
				&& (cn->anthy_context.preedit->state != ANTHY_INPUT_ST_CONV)) {
			anthy_input_space(cn->anthy_context.ictx);
			im_ja_anthy_update_preedit(cn);
		}
    im_ja_anthy_show_candidates(cn);
    return TRUE;
  }

  /* Expand current bunsetsu */
  if (ishotkey(key, EXPAND_BUNSETSU, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_resize(cn->anthy_context.ictx, 1);
		im_ja_anthy_update_preedit(cn);
    return TRUE; 
  }

  /* Shrink current bunsetsu */
  if (ishotkey(key, SHRINK_BUNSETSU, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_resize(cn->anthy_context.ictx, -1);
		im_ja_anthy_update_preedit(cn);
    return TRUE; 
  }

  /* Select next bunsetsu */
  if (ishotkey(key, NEXT_BUNSETSU, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_move(cn->anthy_context.ictx, 1);
		im_ja_anthy_update_preedit(cn);
    return TRUE; 
  } 
 
  /* Select prev bunsetsu */
  if (ishotkey(key, PREV_BUNSETSU, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_move(cn->anthy_context.ictx, -1);
		im_ja_anthy_update_preedit(cn);
    return TRUE; 
  }

  /* Select previous candidate */
  if (ishotkey(key, PREV_CANDIDATE, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_prev_candidate(cn->anthy_context.ictx);
		im_ja_anthy_update_preedit(cn);
    return TRUE; 
  }

  /* Select next candidate */
  if (ishotkey(key, NEXT_CANDIDATE, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_next_candidate(cn->anthy_context.ictx);
    im_ja_anthy_update_preedit(cn);
    return TRUE; 
  }

	/* Left arrow key: move cursor */
	if (key->keyval == GDK_Left || key->keyval == GDK_KP_Left) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_move(cn->anthy_context.ictx, -1);
		im_ja_anthy_update_preedit(cn);
		return TRUE;
	}

	/* Right arrow key: move cursor */
	if (key->keyval == GDK_Right || key->keyval == GDK_KP_Right) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		anthy_input_move(cn->anthy_context.ictx, 1);
		im_ja_anthy_update_preedit(cn);
		return TRUE;
	}

  /* NORMAL CHAR KEYS */
	if (im_ja_is_printable_key(key) == TRUE) { 
    gchar utf8strg[7];
		gchar *tmpeuc;
		gint anthy_map;

		utf8strg[g_unichar_to_utf8(gdk_keyval_to_unicode(key->keyval), utf8strg)] = 0;
		tmpeuc = utf82euc(utf8strg);
		if ((unsigned int)tmpeuc[0] > 128) {
			g_free(tmpeuc);
			IM_JA_DEBUG("*ERROR* non-ascii input\n");
			im_ja_input_utf8(cn, utf8strg);
			return TRUE;
		}

		if ((cn->anthy_context.preedit != NULL) 
				&& (cn->anthy_context.preedit->state == ANTHY_INPUT_ST_CONV)) {
			im_ja_commit(cn);
		}

		/* DOESN'T WORK: ANTHY BUG??*/
		anthy_map = anthy_input_get_selected_map(cn->anthy_context.ictx);
		switch (anthy_map) {
		case ANTHY_INPUT_MAP_HIRAGANA:
			if (cn->input_method != IM_JA_HIRAGANA_INPUT) {
				anthy_input_map_select(cn->anthy_context.ictx, ANTHY_INPUT_MAP_HIRAGANA);
			}
			break;
		case ANTHY_INPUT_MAP_KATAKANA:
			if (cn->input_method != IM_JA_KATAKANA_INPUT) {
				anthy_input_map_select(cn->anthy_context.ictx, ANTHY_INPUT_MAP_KATAKANA);
			}			
			break;
		case ANTHY_INPUT_MAP_ALPHABET:
			IM_JA_DEBUG("How did we get here? THIS SHOULD BE DIRECT MODE!\n");
			break;
		case ANTHY_INPUT_MAP_WALPHABET:
			if (cn->input_method != IM_JA_ZENKAKU_INPUT) {
				anthy_input_map_select(cn->anthy_context.ictx, ANTHY_INPUT_MAP_WALPHABET);
			}
			break;
		}
		anthy_input_str(cn->anthy_context.ictx, tmpeuc);
		g_free(tmpeuc);

    im_ja_anthy_update_preedit(cn);
		return TRUE;
  } 

  /* Unconvert current bunsetsu on ESC */
  if (key->keyval == GDK_Escape) {
		if ((cn->anthy_context.preedit != NULL) 
				&& (cn->anthy_context.preedit->state == ANTHY_INPUT_ST_CONV)) {
			anthy_input_quit(cn->anthy_context.ictx);
		}
		else return FALSE; /* This will kill the preedit if not converted */
    im_ja_anthy_update_preedit(cn);
    return TRUE;
  }
  
  return FALSE; /* We didn't handle the keypress */
}

void im_ja_anthy_select_candidate(IMJAContext *cn, int selected) {
	anthy_input_select_candidate(cn->anthy_context.ictx, selected);
}

