/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method 
 *
 * Copyright (C) 2004 Botond Botyanszki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

#include <config.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkkeysyms.h>

#include "im-ja.h"
#include "error.h"
#include "nls.h"
#include "conf.h"
#include "actionmenu.h"


extern GConfClient *gconf_client;
extern GConfEnumStringPair hotkey_gconf_keys[];
extern gboolean im_ja_execute_action(gpointer data, gint action_id, gboolean set_input_method);

static ActionMenuProperty actionMenuProperties[] = {
  { DIRECT_INPUT_MODE, N_("direct input"), N_("Set direct input mode"), 
    NULL, NULL, IM_JA_PANEL_MENU | IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, IM_JA_DIRECT_INPUT },
  { HIRAGANA_INPUT_MODE, N_("hiragana input"), N_("Set hiragana input mode"), 
    NULL, NULL, IM_JA_PANEL_MENU | IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, IM_JA_HIRAGANA_INPUT },
  { KATAKANA_INPUT_MODE, N_("katakana input"), N_("Set katakana input mode"), 
    NULL, NULL, IM_JA_PANEL_MENU | IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, IM_JA_KATAKANA_INPUT },
  { HALFKATA_INPUT_MODE, N_("narrow katakana input"), N_("Set narrow katakana input mode"), 
    NULL, NULL, IM_JA_PANEL_MENU | IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, IM_JA_HALFKATA_INPUT },
  { ZENKAKU_INPUT_MODE, N_("zenkaku input"), N_("Set zenkaku (double-width romaji) input mode"), 
    NULL, NULL, IM_JA_PANEL_MENU | IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, IM_JA_ZENKAKU_INPUT },
  { KANJIPAD_INPUT_MODE, N_("kanjipad input"), N_("Set kanjipad (character recognition) input mode"), 
    NULL, NULL, IM_JA_PANEL_MENU | IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, IM_JA_KANJIPAD_INPUT },

  { SYMBOL_INPUT, N_("symbol input"), N_("Input special symbols"), 
    NULL, NULL, IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, -1 },
  { UNICODE_INPUT, N_("unicode input"), N_("Input unicode characters using their four digit hex numbers"), 
    NULL, NULL, IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, -1 },
  { JIS_CODE_INPUT, N_("jiscode input"), N_("Input kanji characters using their jis code"), 
    NULL, NULL, IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, -1 },
  { RADICAL_INPUT, N_("radical input"), N_("Input kanji characters through radical lookup"), 
    NULL, NULL, IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, -1 },
  { START_CONFIGURATOR, N_("configure im-ja"), N_("Run the im-ja configurator"), 
    NULL, NULL, IM_JA_PANEL_MENU | IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, -1 },
  { TOGGLE_PREEDIT_WINDOW, N_("toggle preedit window"), N_("Toggle over-the-spot preedit window"), 
    NULL, NULL, IM_JA_STATUSWIN_MENU | IM_JA_GTKIM_MENU, -1 },
  { -1, NULL, NULL, 
    NULL, NULL, 0, -1 }
};


GConfEnumStringPair gconf_label_keys[] = {
	{DIRECT_INPUT_MODE, "direct"},
	{HIRAGANA_INPUT_MODE, "hiragana"},
	{KATAKANA_INPUT_MODE, "katakana"},
	{HALFKATA_INPUT_MODE, "halfkata"},
	{ZENKAKU_INPUT_MODE, "zenkaku"},
	{KANJIPAD_INPUT_MODE, "kanjipad"},
	{-1, NULL}
};



void im_ja_actionmenu_set_properties() {
  gint i = 0, j = 0, id;
  gchar *gconf_key;
	guint keyval, keystate;
  static gboolean properties_initialized = FALSE;
  
  if (properties_initialized == TRUE) return;

	for (j = 0; actionMenuProperties[j].id != -1; j++) { 
		/* get labels */
		if (actionMenuProperties[j].input_method != -1) {
		  for (i = 0; gconf_label_keys[i].enum_value != -1; i++) {
				gconf_key = g_strdup_printf(GCONF_NAMESPACE"status_window/label_%s", 
																		gconf_enum_to_string(gconf_label_keys, gconf_label_keys[i].enum_value));
				if (actionMenuProperties[j].id == gconf_label_keys[i].enum_value) {
					actionMenuProperties[j].label = gconf_client_get_string(gconf_client, gconf_key, NULL);
					if (actionMenuProperties[j].label == NULL) actionMenuProperties[j].label = "  ";
				}
				g_free(gconf_key);
			}
		}
		else {
			actionMenuProperties[j].label = "";
		}
		/* get hotkeys */
		id = actionMenuProperties[j].id;
		if (hotkey_gconf_keys[id].str != NULL) {
			gconf_key = g_strdup_printf(GCONF_NAMESPACE"hotkeys/%s_keyval", hotkey_gconf_keys[id].str);
			keyval = gconf_client_get_int(gconf_client, gconf_key, NULL);
			g_free(gconf_key);
			if ((keyval != 0) && (keyval != GDK_VoidSymbol)) {
				gconf_key = g_strdup_printf(GCONF_NAMESPACE"hotkeys/%s_state", hotkey_gconf_keys[id].str);
				keystate = gconf_client_get_int(gconf_client, gconf_key, NULL);
				g_free(gconf_key);
				actionMenuProperties[j].hotkey_name = im_ja_get_keyname(keystate, keyval);
			}
			else {
				actionMenuProperties[j].hotkey_name = "";
			}
			/*
			IM_JA_DEBUG("HOTKEY: %s %s %s\n", actionMenuProperties[j].label, 
									actionMenuProperties[j].name, actionMenuProperties[j].hotkey_name);
			*/
		}
	}

  properties_initialized = TRUE;
}

static gchar *im_ja_actionmenu_name_for_id(gint action_id) {
	gint j;
	gchar *name = NULL;
	for (j = 0; actionMenuProperties[j].id != -1; j++) { 
		if (actionMenuProperties[j].id == action_id) {
			name = actionMenuProperties[j].name;
			break;
		}
	}
	return name;
}

gint im_ja_action_to_input_method(gint action_id) {
	switch (action_id) {
	case DIRECT_INPUT_MODE: 
		return IM_JA_DIRECT_INPUT;
	case HIRAGANA_INPUT_MODE:
		return IM_JA_HIRAGANA_INPUT;
	case KATAKANA_INPUT_MODE:
		return IM_JA_KATAKANA_INPUT;
	case HALFKATA_INPUT_MODE:
		return IM_JA_HALFKATA_INPUT;
	case ZENKAKU_INPUT_MODE:
		return IM_JA_ZENKAKU_INPUT;
	case KANJIPAD_INPUT_MODE:
		return IM_JA_KANJIPAD_INPUT;
	}

	return -1; /* input method not changed by action */
}

static void im_ja_actionmenu_selected(GtkWidget *menuitem, gpointer data) {
	gboolean set_input_method = FALSE;
	gint input_method = -1;
	gint action_id = (gint) g_object_get_data(G_OBJECT(menuitem), "im-ja-action-number");

	IM_JA_DEBUG("im_ja_actionmenu_selected: %d [%s]\n", action_id, 
							im_ja_actionmenu_name_for_id(action_id));
	input_method = im_ja_action_to_input_method(action_id);
	if (input_method != -1) {
		action_id = input_method;
		set_input_method = TRUE;
	}
	if (im_ja_execute_action(data, action_id, set_input_method) == FALSE) {
		im_ja_print_error(_("FIXME: Failed to execute action."));
	}
}


void im_ja_actionmenu_populate(gpointer data, GtkMenu *menu, ImJaActionMenuType menu_type) {
  GtkWidget *menuitem;
	gint j;
	gchar *tmpstrg;

	im_ja_actionmenu_set_properties();
	
	for (j = 0; actionMenuProperties[j].id != -1; j++) { 
		if (actionMenuProperties[j].menu_type & menu_type) {
			tmpstrg = g_strdup_printf("%s [%s] %s", actionMenuProperties[j].label,
																_(actionMenuProperties[j].name), actionMenuProperties[j].hotkey_name);
			IM_JA_DEBUG(" populate: %s\n", tmpstrg);
			menuitem = gtk_menu_item_new_with_label(tmpstrg);
			g_free(tmpstrg);
			g_object_set_data(G_OBJECT(menuitem), "im-ja-action-number", (gpointer) actionMenuProperties[j].id);
			gtk_widget_show(menuitem);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);
			g_signal_connect(GTK_OBJECT(menuitem), "activate", 
											 G_CALLBACK(im_ja_actionmenu_selected), data);
		}
	}	
}

static void set_menu_position(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, GtkWidget *button) {
	GtkRequisition requisition;
	GdkScreen *screen;
	gint menu_x = 0;
	gint menu_y = 0;

	screen = gtk_widget_get_screen(button);

	gtk_widget_size_request(GTK_WIDGET(menu), &requisition);

	gdk_window_get_origin(button->window, &menu_x, &menu_y);

	menu_x += button->allocation.x;
	menu_y += button->allocation.y;

	if (menu_y > gdk_screen_get_height(screen) / 2) menu_y -= requisition.height;
	else menu_y += button->allocation.height;
	*x = menu_x;
	*y = menu_y;
	*push_in = TRUE;
}


void im_ja_actionmenu_button_press_cb(GtkWidget *widget, GdkEventButton *event, GtkWidget **menu, 
																			ImJaActionMenuType type, gpointer data) { 

	IM_JA_DEBUG("im_ja_actionmenu_button_press_cb()\n");

	if (event != NULL) {
		if (event->button == 3 || event->button == 2) {
			gtk_propagate_event(GTK_WIDGET(widget), (GdkEvent *) event);
			return;
		}
	}

	if (*menu == NULL) { 
		*menu = gtk_menu_new();
		/*g_signal_connect_swapped(GTK_OBJECT(*menu), "destroy", G_CALLBACK(gtk_widget_destroy), NULL);
		g_signal_connect_swapped(GTK_OBJECT(*menu), "cancel", G_CALLBACK(gtk_widget_destroy), NULL);
		*/
		im_ja_actionmenu_populate(data, GTK_MENU(*menu), type);
	}
	
	gtk_widget_show_all(*menu);

	if (type == IM_JA_STATUSWIN_MENU) {
		gtk_menu_popup(GTK_MENU(*menu), NULL, NULL, NULL, widget, 0, 
									 gtk_get_current_event_time());
	}
	else {
		gtk_menu_popup(GTK_MENU(*menu), NULL, NULL, (GtkMenuPositionFunc) set_menu_position, widget, 0, 
									 gtk_get_current_event_time());
	}
}

