functor LrValsFn(structure Token : TOKEN) = 
struct
structure ParserData=
struct
structure Header = 
struct
(*
 * (c) Andreas Rossberg 1999-2013
 *
 * Standard ML syntactical analysis
 *
 * Definition, Sections 2, 3, and 8, Appendix A and B
 *
 * Notes:
 * - Two phrases named Fmatch and Fmrule have been added to factorize Fvalbind.
 * - A phrase named SynDesc has been added to factorize type synonym
 *   specifications. Similarly, a phrase named TyReaDesc has been added to
 *   factorize type realisation signature expressions.
 * - Infix expressions [Definition, Section 2.6] are resolved externally in
 *   structure Infix. The parser just maintains the infix environment J by
 *   side effect. To achieve correct treatment of scoped fixity directives,
 *   a stack of environments is used. To handle `local' we even need a
 *   second environment J' (together with a a second stack).
 * - Syntactic restrictions [Definition, Sections 2.9 and 3.5] are checked
 *   in a separate pass (structures SyntacticRestrictions* ).
 * - Although not completely clear from the wording we assume that the
 *   Definition rules out the use of `=' as a tycon. Otherwise we would have
 *   a massive amount of grammar conflicts.
 * - The Definition is also vague about what consists a non-infixed occurrence
 *   of an infix identifier: we assume any occurrences in expressions
 *   or patterns. This implies that uses of the keyword `op' in constructor
 *   and exception bindings are completely redundant.
 * - Datatype replication requires rules for datatype to be duplicated to
 *   avoid conflicts on empty tyvarseqs.
 * - Layered patterns require some grammar transformation hack, see pat.
 * - The messy `sigexp where type ... and type ...' syntax requires some
 *   really ugly transformations (in absence of a lookahead of 2), watch out
 *   for non-terminals of the form xxx__AND_yyybind_opt.
 *
 * Bugs:
 * - We do NOT support declarations like
 *     fun f p1 = case e1 of p2 => e2
 *       | f p3 = e3
 *   (without parentheses around the case) because the transformations
 *   required to support this would be even a magnitude uglier than those
 *   above. In fact, no known compiler supports this.
 *)


  (* Import *)

  structure A = Annotation

  open SyntaxCore
  open SyntaxModule
  open SyntaxProgram
  open DerivedFormsCore
  open DerivedFormsModule
  open DerivedFormsProgram


  (* Building annotations *)

  fun x@@loc = A.@@(x, A.fromLoc loc)


  (* Handling infix environments *)

  val J  = ref Infix.empty	(* context *)
  val J' = ref Infix.empty	(* local environment (+ enclosing one) *)

  val stackJ  = ref [] : Infix.InfEnv list ref
  val stackJ' = ref [] : Infix.InfEnv list ref

  fun initJandJ'(J0) = ( J := J0; J' := J0; stackJ := []; stackJ' := [] )
  fun pushJ() = stackJ := !J :: !stackJ
  fun popJ() = ( J := List.hd(!stackJ); stackJ := List.tl(!stackJ) )
  fun pushJ'shiftJ() = ( stackJ' := !J' :: !stackJ'; J' := List.hd(!stackJ) )
  fun popJandJ'() =
      ( J  := !J';
	J' := List.hd(!stackJ');
	stackJ  := List.tl(!stackJ);
	stackJ' := List.tl(!stackJ')
      )

  fun assignInfix(infstatus, vids) =
      let
	val vids' = List.map A.syntax vids
      in
	J  := Infix.assign(!J, vids', infstatus);
	J' := Infix.assign(!J', vids', infstatus)
      end

  fun cancelInfix(vids) =
      let
	val vids' = List.map A.syntax vids
      in
	J  := Infix.cancel(!J, vids');
	J' := Infix.cancel(!J', vids')
      end


  (* Helper for long identifiers *)

  fun toLongId toId (strids, id) = (List.map StrId.fromString strids, toId id)


  (* Helper to handle typed patterns (needed for layered patterns) *)

  fun typedPat(pat, []) = pat
    | typedPat(pat, ty::tys) =
      let open A in typedPat(COLONPat(pat, ty)@@over(pat, ty), tys) end


end
structure LrTable = Token.LrTable
structure Token = Token
local open LrTable in 
val table=let val actionRows =
"\
\\001\000\001\000\000\000\000\000\
\\001\000\001\000\202\003\002\000\051\000\006\000\050\000\008\000\049\000\
\\011\000\048\000\012\000\047\000\013\000\046\000\015\000\045\000\
\\017\000\044\000\018\000\043\000\019\000\042\000\020\000\041\000\
\\021\000\040\000\023\000\039\000\024\000\038\000\026\000\037\000\
\\029\000\036\000\030\000\035\000\033\000\034\000\034\000\033\000\
\\036\000\032\000\038\000\031\000\042\000\192\003\046\000\165\002\
\\049\000\030\000\051\000\029\000\055\000\028\000\057\000\027\000\
\\060\000\026\000\061\000\025\000\062\000\024\000\063\000\023\000\
\\064\000\022\000\065\000\021\000\066\000\020\000\067\000\019\000\
\\068\000\018\000\069\000\017\000\070\000\165\002\071\000\165\002\
\\072\000\165\002\075\000\165\002\000\000\
\\001\000\002\000\153\002\003\000\153\002\008\000\153\002\010\000\153\002\
\\011\000\153\002\013\000\153\002\016\000\153\002\017\000\153\002\
\\018\000\153\002\020\000\153\002\021\000\153\002\024\000\153\002\
\\029\000\153\002\030\000\153\002\034\000\155\002\035\000\153\002\
\\041\000\153\002\042\000\153\002\051\000\153\002\055\000\153\002\
\\057\000\153\002\059\000\153\002\000\000\
\\001\000\002\000\192\002\003\000\192\002\004\000\192\002\007\000\192\002\
\\008\000\192\002\009\000\192\002\010\000\192\002\011\000\192\002\
\\013\000\192\002\014\000\192\002\016\000\192\002\017\000\192\002\
\\018\000\192\002\019\000\042\000\020\000\192\002\021\000\192\002\
\\022\000\192\002\023\000\039\000\024\000\192\002\025\000\192\002\
\\028\000\192\002\029\000\192\002\030\000\192\002\034\000\033\000\
\\035\000\192\002\036\000\032\000\037\000\192\002\038\000\031\000\
\\039\000\192\002\040\000\192\002\041\000\192\002\042\000\192\002\
\\045\000\192\002\046\000\165\002\049\000\030\000\051\000\192\002\
\\055\000\192\002\057\000\192\002\060\000\026\000\061\000\025\000\
\\062\000\024\000\063\000\023\000\064\000\022\000\065\000\021\000\
\\066\000\020\000\067\000\019\000\068\000\018\000\069\000\017\000\
\\070\000\165\002\071\000\165\002\072\000\165\002\075\000\165\002\000\000\
\\001\000\002\000\048\003\003\000\048\003\004\000\048\003\005\000\048\003\
\\007\000\048\003\008\000\048\003\009\000\048\003\010\000\048\003\
\\011\000\048\003\013\000\048\003\014\000\048\003\016\000\048\003\
\\017\000\048\003\018\000\048\003\020\000\048\003\021\000\048\003\
\\022\000\048\003\024\000\048\003\025\000\048\003\028\000\048\003\
\\029\000\048\003\030\000\048\003\031\000\048\003\032\000\048\003\
\\035\000\048\003\037\000\048\003\039\000\048\003\040\000\048\003\
\\041\000\048\003\042\000\048\003\045\000\048\003\046\000\048\003\
\\047\000\048\003\048\000\048\003\050\000\048\003\051\000\048\003\
\\052\000\048\003\053\000\048\003\055\000\048\003\057\000\048\003\
\\058\000\048\003\059\000\048\003\070\000\059\003\071\000\059\003\
\\072\000\250\000\075\000\059\003\000\000\
\\001\000\002\000\090\003\003\000\087\001\008\000\090\003\010\000\090\003\
\\011\000\090\003\013\000\090\003\016\000\090\003\017\000\090\003\
\\018\000\090\003\020\000\090\003\021\000\090\003\024\000\090\003\
\\029\000\090\003\030\000\090\003\035\000\090\003\041\000\070\003\
\\042\000\090\003\051\000\090\003\055\000\090\003\057\000\090\003\
\\059\000\070\003\000\000\
\\001\000\002\000\090\003\003\000\087\001\008\000\090\003\010\000\090\003\
\\011\000\090\003\013\000\090\003\016\000\090\003\017\000\090\003\
\\018\000\090\003\020\000\090\003\021\000\090\003\024\000\090\003\
\\029\000\090\003\030\000\090\003\035\000\090\003\041\000\099\003\
\\042\000\090\003\051\000\090\003\055\000\090\003\057\000\090\003\
\\058\000\099\003\059\000\099\003\000\000\
\\001\000\002\000\090\003\003\000\107\002\008\000\090\003\010\000\090\003\
\\011\000\090\003\013\000\090\003\016\000\090\003\017\000\090\003\
\\018\000\090\003\020\000\090\003\021\000\090\003\024\000\090\003\
\\029\000\090\003\030\000\090\003\035\000\090\003\041\000\114\003\
\\042\000\090\003\051\000\090\003\055\000\090\003\057\000\090\003\
\\058\000\114\003\059\000\114\003\000\000\
\\001\000\002\000\106\003\003\000\095\001\008\000\106\003\011\000\106\003\
\\013\000\106\003\017\000\106\003\018\000\106\003\020\000\106\003\
\\021\000\106\003\024\000\106\003\029\000\106\003\030\000\106\003\
\\042\000\106\003\051\000\106\003\055\000\106\003\057\000\106\003\
\\058\000\099\003\000\000\
\\001\000\002\000\106\003\003\000\091\002\008\000\106\003\011\000\106\003\
\\013\000\106\003\017\000\106\003\018\000\106\003\020\000\106\003\
\\021\000\106\003\024\000\106\003\029\000\106\003\030\000\106\003\
\\042\000\106\003\051\000\106\003\055\000\106\003\057\000\106\003\
\\058\000\114\003\000\000\
\\001\000\002\000\182\003\003\000\016\002\008\000\182\003\011\000\182\003\
\\013\000\182\003\017\000\182\003\018\000\182\003\020\000\182\003\
\\021\000\182\003\024\000\182\003\029\000\182\003\030\000\182\003\
\\041\000\070\003\042\000\182\003\051\000\182\003\055\000\182\003\
\\057\000\182\003\059\000\070\003\000\000\
\\001\000\002\000\182\003\003\000\016\002\008\000\182\003\011\000\182\003\
\\013\000\182\003\017\000\182\003\018\000\182\003\020\000\182\003\
\\021\000\182\003\024\000\182\003\029\000\182\003\030\000\182\003\
\\041\000\099\003\042\000\182\003\051\000\182\003\055\000\182\003\
\\057\000\182\003\058\000\099\003\059\000\099\003\000\000\
\\001\000\002\000\182\003\003\000\120\002\008\000\182\003\011\000\182\003\
\\013\000\182\003\017\000\182\003\018\000\182\003\020\000\182\003\
\\021\000\182\003\024\000\182\003\029\000\182\003\030\000\182\003\
\\041\000\114\003\042\000\182\003\051\000\182\003\055\000\182\003\
\\057\000\182\003\058\000\114\003\059\000\114\003\000\000\
\\001\000\002\000\051\000\006\000\050\000\008\000\049\000\011\000\048\000\
\\012\000\047\000\013\000\046\000\015\000\045\000\017\000\044\000\
\\018\000\043\000\019\000\042\000\020\000\041\000\021\000\040\000\
\\023\000\039\000\024\000\038\000\026\000\037\000\029\000\036\000\
\\030\000\035\000\033\000\034\000\034\000\033\000\036\000\032\000\
\\038\000\031\000\042\000\192\003\046\000\165\002\049\000\030\000\
\\051\000\029\000\055\000\028\000\057\000\027\000\060\000\026\000\
\\061\000\025\000\062\000\024\000\063\000\023\000\064\000\022\000\
\\065\000\021\000\066\000\020\000\067\000\019\000\068\000\018\000\
\\069\000\017\000\070\000\165\002\071\000\165\002\072\000\165\002\
\\075\000\165\002\000\000\
\\001\000\003\000\024\002\008\000\168\003\010\000\168\003\011\000\168\003\
\\029\000\168\003\030\000\168\003\035\000\168\003\042\000\168\003\
\\050\000\168\003\052\000\168\003\053\000\168\003\057\000\168\003\
\\058\000\099\003\000\000\
\\001\000\003\000\115\002\008\000\168\003\010\000\168\003\011\000\168\003\
\\029\000\168\003\030\000\168\003\035\000\168\003\042\000\168\003\
\\050\000\168\003\052\000\168\003\053\000\168\003\057\000\168\003\
\\058\000\114\003\000\000\
\\001\000\004\000\061\000\007\000\185\000\014\000\060\000\025\000\059\000\
\\041\000\058\000\000\000\
\\001\000\004\000\061\000\009\000\142\001\014\000\060\000\025\000\059\000\
\\041\000\058\000\000\000\
\\001\000\004\000\061\000\014\000\060\000\022\000\244\000\025\000\059\000\
\\041\000\058\000\000\000\
\\001\000\004\000\061\000\014\000\060\000\025\000\059\000\028\000\226\000\
\\041\000\058\000\000\000\
\\001\000\004\000\061\000\014\000\060\000\025\000\059\000\035\000\182\000\
\\040\000\181\000\041\000\058\000\042\000\180\000\000\000\
\\001\000\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\
\\042\000\057\000\000\000\
\\001\000\005\000\147\002\039\000\147\002\040\000\147\002\041\000\147\002\
\\046\000\140\002\000\000\
\\001\000\005\000\148\002\039\000\148\002\040\000\148\002\041\000\148\002\
\\046\000\141\002\000\000\
\\001\000\005\000\149\002\039\000\149\002\040\000\149\002\041\000\149\002\
\\046\000\142\002\000\000\
\\001\000\005\000\026\003\035\000\043\003\037\000\043\003\039\000\043\003\
\\040\000\043\003\041\000\188\000\046\000\043\003\047\000\043\003\000\000\
\\001\000\005\000\027\003\023\000\158\002\034\000\158\002\035\000\158\002\
\\036\000\158\002\037\000\158\002\038\000\158\002\039\000\158\002\
\\040\000\158\002\041\000\040\001\044\000\158\002\046\000\158\002\
\\047\000\158\002\060\000\158\002\061\000\158\002\062\000\158\002\
\\063\000\158\002\064\000\158\002\065\000\158\002\066\000\158\002\
\\067\000\158\002\068\000\158\002\069\000\158\002\070\000\158\002\
\\071\000\158\002\072\000\158\002\075\000\158\002\000\000\
\\001\000\005\000\129\001\000\000\
\\001\000\006\000\050\000\012\000\047\000\015\000\045\000\019\000\042\000\
\\023\000\039\000\026\000\037\000\033\000\034\000\034\000\033\000\
\\036\000\032\000\037\000\177\002\038\000\031\000\046\000\165\002\
\\049\000\030\000\060\000\026\000\061\000\025\000\062\000\024\000\
\\063\000\023\000\064\000\022\000\065\000\021\000\066\000\020\000\
\\067\000\019\000\068\000\018\000\069\000\017\000\070\000\165\002\
\\071\000\165\002\072\000\165\002\075\000\165\002\000\000\
\\001\000\008\000\251\001\070\000\069\000\071\000\068\000\072\000\067\000\000\000\
\\001\000\010\000\161\001\000\000\
\\001\000\010\000\216\001\000\000\
\\001\000\010\000\000\002\000\000\
\\001\000\010\000\011\002\000\000\
\\001\000\010\000\039\002\000\000\
\\001\000\010\000\073\002\000\000\
\\001\000\010\000\085\002\000\000\
\\001\000\016\000\054\001\000\000\
\\001\000\016\000\056\001\000\000\
\\001\000\016\000\208\001\000\000\
\\001\000\016\000\227\001\000\000\
\\001\000\019\000\012\001\056\000\011\001\070\000\010\001\075\000\120\000\000\000\
\\001\000\023\000\039\000\034\000\144\000\035\000\032\003\036\000\109\000\
\\037\000\032\003\038\000\108\000\039\000\032\003\040\000\032\003\
\\044\000\107\000\046\000\032\003\047\000\032\003\060\000\026\000\
\\061\000\025\000\062\000\024\000\063\000\023\000\064\000\022\000\
\\065\000\021\000\066\000\020\000\067\000\019\000\068\000\018\000\
\\069\000\017\000\070\000\165\002\071\000\165\002\072\000\165\002\
\\075\000\165\002\000\000\
\\001\000\023\000\039\000\034\000\144\000\035\000\040\003\036\000\109\000\
\\037\000\040\003\038\000\108\000\039\000\040\003\040\000\040\003\
\\041\000\040\003\044\000\107\000\046\000\040\003\047\000\040\003\
\\060\000\026\000\061\000\025\000\062\000\024\000\063\000\023\000\
\\064\000\022\000\065\000\021\000\066\000\020\000\067\000\019\000\
\\068\000\018\000\069\000\017\000\070\000\165\002\071\000\165\002\
\\072\000\165\002\075\000\165\002\000\000\
\\001\000\023\000\039\000\034\000\144\000\036\000\109\000\037\000\017\003\
\\038\000\108\000\044\000\107\000\060\000\026\000\061\000\025\000\
\\062\000\024\000\063\000\023\000\064\000\022\000\065\000\021\000\
\\066\000\020\000\067\000\019\000\068\000\018\000\069\000\017\000\
\\070\000\165\002\071\000\165\002\072\000\165\002\075\000\165\002\000\000\
\\001\000\029\000\107\001\070\000\073\000\075\000\120\000\000\000\
\\001\000\029\000\159\001\000\000\
\\001\000\029\000\176\001\000\000\
\\001\000\029\000\176\001\070\000\076\000\000\000\
\\001\000\029\000\179\001\070\000\073\000\075\000\120\000\000\000\
\\001\000\029\000\008\002\000\000\
\\001\000\029\000\008\002\070\000\073\000\000\000\
\\001\000\029\000\059\002\000\000\
\\001\000\029\000\059\002\070\000\073\000\000\000\
\\001\000\029\000\093\002\000\000\
\\001\000\029\000\093\002\070\000\079\000\000\000\
\\001\000\031\000\075\001\000\000\
\\001\000\034\000\115\000\070\000\152\000\071\000\151\000\073\000\106\000\000\000\
\\001\000\034\000\175\000\000\000\
\\001\000\034\000\090\001\000\000\
\\001\000\035\000\183\000\000\000\
\\001\000\035\000\184\000\000\000\
\\001\000\035\000\047\001\000\000\
\\001\000\035\000\049\001\040\000\048\001\000\000\
\\001\000\035\000\050\001\000\000\
\\001\000\035\000\080\001\000\000\
\\001\000\035\000\082\001\040\000\081\001\000\000\
\\001\000\035\000\101\001\000\000\
\\001\000\035\000\222\001\000\000\
\\001\000\035\000\225\001\041\000\224\001\059\000\223\001\000\000\
\\001\000\035\000\235\001\058\000\083\001\000\000\
\\001\000\037\000\179\000\000\000\
\\001\000\037\000\046\001\000\000\
\\001\000\039\000\176\000\000\000\
\\001\000\039\000\042\001\000\000\
\\001\000\039\000\078\001\000\000\
\\001\000\041\000\173\000\046\000\172\000\059\000\171\000\000\000\
\\001\000\041\000\079\001\000\000\
\\001\000\041\000\089\001\059\000\088\001\000\000\
\\001\000\041\000\102\001\000\000\
\\001\000\041\000\182\001\046\000\181\001\059\000\180\001\000\000\
\\001\000\041\000\184\001\000\000\
\\001\000\041\000\191\001\000\000\
\\001\000\041\000\018\002\059\000\017\002\000\000\
\\001\000\041\000\022\002\046\000\021\002\059\000\020\002\000\000\
\\001\000\042\000\052\000\000\000\
\\001\000\043\000\202\000\061\000\085\000\062\000\084\000\070\000\201\000\
\\071\000\200\000\072\000\199\000\000\000\
\\001\000\046\000\070\000\070\000\069\000\071\000\068\000\072\000\067\000\
\\075\000\066\000\000\000\
\\001\000\046\000\124\000\070\000\069\000\071\000\068\000\072\000\067\000\000\000\
\\001\000\046\000\174\000\000\000\
\\001\000\046\000\177\000\000\000\
\\001\000\046\000\189\000\000\000\
\\001\000\046\000\243\000\000\000\
\\001\000\046\000\044\001\000\000\
\\001\000\046\000\052\001\000\000\
\\001\000\046\000\059\001\000\000\
\\001\000\046\000\068\001\000\000\
\\001\000\046\000\074\001\000\000\
\\001\000\046\000\084\001\058\000\083\001\000\000\
\\001\000\046\000\093\001\058\000\083\001\000\000\
\\001\000\046\000\185\001\000\000\
\\001\000\046\000\195\001\000\000\
\\001\000\046\000\241\001\000\000\
\\001\000\046\000\245\001\000\000\
\\001\000\046\000\248\001\000\000\
\\001\000\046\000\014\002\058\000\083\001\000\000\
\\001\000\046\000\019\002\058\000\083\001\000\000\
\\001\000\046\000\042\002\000\000\
\\001\000\046\000\045\002\000\000\
\\001\000\046\000\077\002\058\000\083\001\000\000\
\\001\000\046\000\078\002\058\000\083\001\000\000\
\\001\000\046\000\088\002\000\000\
\\001\000\046\000\098\002\000\000\
\\001\000\046\000\103\002\000\000\
\\001\000\046\000\112\002\000\000\
\\001\000\047\000\235\000\000\000\
\\001\000\054\000\004\001\070\000\076\000\000\000\
\\001\000\058\000\096\001\000\000\
\\001\000\058\000\025\002\000\000\
\\001\000\061\000\085\000\062\000\084\000\070\000\083\000\071\000\082\000\
\\072\000\081\000\000\000\
\\001\000\070\000\069\000\071\000\068\000\072\000\067\000\000\000\
\\001\000\070\000\069\000\071\000\068\000\072\000\067\000\075\000\066\000\000\000\
\\001\000\070\000\069\000\071\000\068\000\072\000\067\000\075\000\194\000\000\000\
\\001\000\070\000\073\000\000\000\
\\001\000\070\000\073\000\075\000\120\000\000\000\
\\001\000\070\000\076\000\000\000\
\\001\000\070\000\079\000\000\000\
\\001\000\070\000\152\000\071\000\151\000\000\000\
\\001\000\070\000\152\000\071\000\151\000\075\000\249\000\000\000\
\\001\000\073\000\106\000\000\000\
\\123\002\000\000\
\\124\002\000\000\
\\125\002\000\000\
\\125\002\041\000\224\001\059\000\223\001\000\000\
\\126\002\000\000\
\\127\002\000\000\
\\128\002\000\000\
\\129\002\000\000\
\\130\002\000\000\
\\131\002\000\000\
\\132\002\000\000\
\\133\002\000\000\
\\134\002\000\000\
\\135\002\000\000\
\\136\002\000\000\
\\137\002\000\000\
\\138\002\000\000\
\\139\002\000\000\
\\140\002\000\000\
\\141\002\000\000\
\\142\002\000\000\
\\143\002\000\000\
\\144\002\000\000\
\\145\002\000\000\
\\146\002\000\000\
\\147\002\000\000\
\\148\002\000\000\
\\149\002\000\000\
\\150\002\000\000\
\\151\002\000\000\
\\152\002\000\000\
\\153\002\000\000\
\\154\002\000\000\
\\155\002\000\000\
\\156\002\000\000\
\\157\002\000\000\
\\158\002\000\000\
\\159\002\000\000\
\\159\002\041\000\188\000\000\000\
\\160\002\000\000\
\\161\002\000\000\
\\162\002\000\000\
\\163\002\000\000\
\\164\002\000\000\
\\165\002\006\000\050\000\012\000\047\000\015\000\045\000\019\000\042\000\
\\023\000\039\000\026\000\037\000\033\000\034\000\034\000\033\000\
\\035\000\095\000\036\000\032\000\038\000\031\000\049\000\030\000\
\\060\000\026\000\061\000\025\000\062\000\024\000\063\000\023\000\
\\064\000\022\000\065\000\021\000\066\000\020\000\067\000\019\000\
\\068\000\018\000\069\000\017\000\000\000\
\\165\002\006\000\050\000\012\000\047\000\015\000\045\000\019\000\042\000\
\\023\000\039\000\026\000\037\000\033\000\034\000\034\000\033\000\
\\036\000\032\000\038\000\031\000\049\000\030\000\060\000\026\000\
\\061\000\025\000\062\000\024\000\063\000\023\000\064\000\022\000\
\\065\000\021\000\066\000\020\000\067\000\019\000\068\000\018\000\
\\069\000\017\000\000\000\
\\165\002\008\000\072\001\023\000\039\000\000\000\
\\165\002\023\000\039\000\000\000\
\\165\002\023\000\039\000\027\000\111\000\034\000\110\000\036\000\109\000\
\\038\000\108\000\044\000\107\000\060\000\026\000\061\000\025\000\
\\062\000\024\000\063\000\023\000\064\000\022\000\065\000\021\000\
\\066\000\020\000\067\000\019\000\068\000\018\000\069\000\017\000\
\\073\000\106\000\000\000\
\\165\002\023\000\039\000\027\000\111\000\034\000\144\000\036\000\109\000\
\\038\000\108\000\044\000\107\000\060\000\026\000\061\000\025\000\
\\062\000\024\000\063\000\023\000\064\000\022\000\065\000\021\000\
\\066\000\020\000\067\000\019\000\068\000\018\000\069\000\017\000\000\000\
\\165\002\023\000\039\000\034\000\110\000\036\000\109\000\038\000\108\000\
\\044\000\107\000\060\000\026\000\061\000\025\000\062\000\024\000\
\\063\000\023\000\064\000\022\000\065\000\021\000\066\000\020\000\
\\067\000\019\000\068\000\018\000\069\000\017\000\073\000\106\000\000\000\
\\165\002\023\000\039\000\034\000\144\000\035\000\210\000\036\000\109\000\
\\038\000\108\000\044\000\107\000\060\000\026\000\061\000\025\000\
\\062\000\024\000\063\000\023\000\064\000\022\000\065\000\021\000\
\\066\000\020\000\067\000\019\000\068\000\018\000\069\000\017\000\000\000\
\\165\002\023\000\039\000\034\000\144\000\035\000\210\000\036\000\109\000\
\\038\000\108\000\044\000\107\000\060\000\026\000\061\000\025\000\
\\062\000\024\000\063\000\023\000\064\000\022\000\065\000\021\000\
\\066\000\020\000\067\000\019\000\068\000\018\000\069\000\017\000\
\\073\000\106\000\000\000\
\\165\002\023\000\039\000\034\000\144\000\036\000\109\000\038\000\108\000\
\\044\000\107\000\060\000\026\000\061\000\025\000\062\000\024\000\
\\063\000\023\000\064\000\022\000\065\000\021\000\066\000\020\000\
\\067\000\019\000\068\000\018\000\069\000\017\000\000\000\
\\166\002\000\000\
\\167\002\000\000\
\\168\002\000\000\
\\169\002\000\000\
\\170\002\000\000\
\\171\002\000\000\
\\172\002\000\000\
\\173\002\000\000\
\\174\002\000\000\
\\175\002\000\000\
\\176\002\000\000\
\\178\002\000\000\
\\179\002\004\000\061\000\014\000\060\000\025\000\059\000\040\000\178\000\
\\041\000\058\000\000\000\
\\180\002\000\000\
\\181\002\000\000\
\\182\002\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\
\\042\000\206\001\000\000\
\\183\002\000\000\
\\184\002\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\
\\042\000\180\000\000\000\
\\185\002\000\000\
\\186\002\000\000\
\\187\002\004\000\061\000\014\000\060\000\025\000\059\000\040\000\125\001\
\\041\000\058\000\000\000\
\\188\002\000\000\
\\189\002\061\000\085\000\062\000\084\000\070\000\083\000\071\000\082\000\
\\072\000\081\000\000\000\
\\190\002\000\000\
\\191\002\000\000\
\\193\002\000\000\
\\194\002\000\000\
\\195\002\004\000\061\000\041\000\058\000\000\000\
\\196\002\004\000\061\000\025\000\059\000\041\000\058\000\000\000\
\\197\002\000\000\
\\198\002\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\000\000\
\\199\002\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\000\000\
\\200\002\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\000\000\
\\201\002\000\000\
\\202\002\000\000\
\\203\002\000\000\
\\204\002\000\000\
\\205\002\045\000\237\000\000\000\
\\206\002\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\000\000\
\\207\002\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\020\000\223\000\021\000\040\000\
\\024\000\038\000\029\000\036\000\030\000\035\000\042\000\222\000\000\000\
\\208\002\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\020\000\223\000\021\000\040\000\
\\024\000\038\000\029\000\036\000\030\000\035\000\042\000\222\000\000\000\
\\209\002\000\000\
\\210\002\000\000\
\\211\002\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\020\000\223\000\021\000\040\000\
\\024\000\038\000\029\000\036\000\030\000\035\000\042\000\222\000\000\000\
\\212\002\000\000\
\\213\002\000\000\
\\214\002\000\000\
\\215\002\000\000\
\\216\002\000\000\
\\217\002\000\000\
\\218\002\000\000\
\\219\002\000\000\
\\220\002\000\000\
\\221\002\000\000\
\\222\002\000\000\
\\223\002\000\000\
\\224\002\000\000\
\\225\002\000\000\
\\226\002\000\000\
\\227\002\000\000\
\\228\002\032\000\241\000\000\000\
\\229\002\000\000\
\\230\002\046\000\124\000\070\000\069\000\071\000\068\000\072\000\067\000\000\000\
\\231\002\000\000\
\\232\002\070\000\073\000\075\000\120\000\000\000\
\\233\002\000\000\
\\234\002\060\000\130\000\061\000\129\000\000\000\
\\235\002\000\000\
\\236\002\000\000\
\\237\002\000\000\
\\238\002\003\000\128\001\004\000\061\000\014\000\060\000\025\000\059\000\
\\041\000\058\000\000\000\
\\239\002\000\000\
\\240\002\000\000\
\\241\002\003\000\234\000\000\000\
\\242\002\000\000\
\\243\002\000\000\
\\244\002\045\000\232\000\000\000\
\\245\002\004\000\061\000\014\000\060\000\025\000\059\000\041\000\058\000\000\000\
\\246\002\000\000\
\\247\002\000\000\
\\248\002\003\000\204\001\000\000\
\\249\002\000\000\
\\250\002\000\000\
\\251\002\000\000\
\\252\002\000\000\
\\253\002\003\000\150\001\000\000\
\\254\002\000\000\
\\255\002\000\000\
\\000\003\045\000\004\002\000\000\
\\001\003\000\000\
\\002\003\022\000\067\001\000\000\
\\002\003\022\000\067\001\046\000\066\001\000\000\
\\003\003\000\000\
\\004\003\000\000\
\\005\003\000\000\
\\006\003\003\000\145\001\000\000\
\\007\003\000\000\
\\007\003\041\000\188\000\000\000\
\\008\003\000\000\
\\009\003\000\000\
\\010\003\000\000\
\\011\003\000\000\
\\012\003\000\000\
\\013\003\000\000\
\\014\003\000\000\
\\015\003\000\000\
\\016\003\000\000\
\\018\003\000\000\
\\019\003\040\000\045\001\000\000\
\\020\003\000\000\
\\021\003\000\000\
\\022\003\000\000\
\\023\003\000\000\
\\024\003\000\000\
\\025\003\040\000\200\001\000\000\
\\026\003\000\000\
\\027\003\041\000\229\000\000\000\
\\028\003\000\000\
\\029\003\005\000\132\001\000\000\
\\030\003\000\000\
\\031\003\043\000\202\000\061\000\085\000\062\000\084\000\070\000\201\000\
\\071\000\200\000\072\000\199\000\000\000\
\\033\003\041\000\188\000\000\000\
\\034\003\000\000\
\\035\003\000\000\
\\036\003\000\000\
\\037\003\000\000\
\\038\003\000\000\
\\039\003\000\000\
\\041\003\000\000\
\\042\003\000\000\
\\043\003\041\000\188\000\000\000\
\\044\003\048\000\251\000\000\000\
\\045\003\000\000\
\\046\003\000\000\
\\047\003\000\000\
\\049\003\000\000\
\\050\003\000\000\
\\051\003\000\000\
\\052\003\000\000\
\\053\003\000\000\
\\054\003\000\000\
\\055\003\000\000\
\\056\003\040\000\218\001\000\000\
\\057\003\000\000\
\\058\003\061\000\085\000\062\000\084\000\070\000\083\000\071\000\082\000\
\\072\000\081\000\000\000\
\\060\003\034\000\167\000\038\000\166\000\073\000\106\000\000\000\
\\061\003\000\000\
\\062\003\000\000\
\\063\003\040\000\081\001\000\000\
\\064\003\000\000\
\\065\003\034\000\115\000\073\000\106\000\000\000\
\\066\003\000\000\
\\067\003\000\000\
\\068\003\000\000\
\\069\003\040\000\051\001\000\000\
\\070\003\000\000\
\\071\003\058\000\083\001\000\000\
\\071\003\058\000\221\001\000\000\
\\071\003\058\000\076\002\000\000\
\\072\003\058\000\083\001\000\000\
\\072\003\058\000\221\001\000\000\
\\072\003\058\000\076\002\000\000\
\\073\003\000\000\
\\074\003\000\000\
\\075\003\000\000\
\\076\003\000\000\
\\077\003\000\000\
\\078\003\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\020\000\041\000\021\000\040\000\
\\024\000\038\000\029\000\036\000\030\000\035\000\042\000\218\000\
\\057\000\027\000\000\000\
\\079\003\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\019\000\012\001\020\000\041\000\
\\021\000\040\000\024\000\038\000\029\000\036\000\030\000\035\000\
\\042\000\218\000\056\000\011\001\057\000\027\000\070\000\010\001\
\\075\000\120\000\000\000\
\\079\003\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\020\000\041\000\021\000\040\000\
\\024\000\038\000\029\000\036\000\030\000\035\000\042\000\218\000\
\\057\000\027\000\000\000\
\\080\003\000\000\
\\081\003\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\020\000\041\000\021\000\040\000\
\\024\000\038\000\029\000\036\000\030\000\035\000\042\000\218\000\
\\057\000\027\000\000\000\
\\082\003\000\000\
\\083\003\000\000\
\\084\003\000\000\
\\085\003\000\000\
\\086\003\000\000\
\\087\003\000\000\
\\088\003\000\000\
\\089\003\000\000\
\\091\003\000\000\
\\092\003\000\000\
\\093\003\000\000\
\\094\003\000\000\
\\095\003\000\000\
\\096\003\000\000\
\\097\003\000\000\
\\098\003\000\000\
\\099\003\000\000\
\\100\003\000\000\
\\101\003\000\000\
\\102\003\000\000\
\\102\003\070\000\076\000\000\000\
\\103\003\000\000\
\\104\003\000\000\
\\105\003\000\000\
\\107\003\000\000\
\\108\003\000\000\
\\109\003\000\000\
\\110\003\000\000\
\\111\003\000\000\
\\112\003\000\000\
\\113\003\000\000\
\\114\003\003\000\087\002\000\000\
\\115\003\008\000\029\001\011\000\028\001\029\000\027\001\030\000\026\001\
\\042\000\100\001\050\000\024\001\052\000\023\001\053\000\099\001\
\\057\000\021\001\000\000\
\\116\003\008\000\029\001\011\000\028\001\029\000\027\001\030\000\026\001\
\\042\000\025\001\050\000\024\001\052\000\023\001\053\000\022\001\
\\057\000\021\001\000\000\
\\116\003\008\000\029\001\011\000\028\001\029\000\027\001\030\000\026\001\
\\042\000\025\001\050\000\024\001\052\000\023\001\053\000\022\001\
\\057\000\021\001\070\000\073\000\000\000\
\\117\003\000\000\
\\118\003\008\000\029\001\011\000\028\001\029\000\027\001\030\000\026\001\
\\042\000\100\001\050\000\024\001\052\000\023\001\057\000\021\001\000\000\
\\119\003\000\000\
\\120\003\000\000\
\\121\003\000\000\
\\122\003\000\000\
\\123\003\000\000\
\\124\003\000\000\
\\125\003\008\000\029\001\011\000\028\001\029\000\027\001\030\000\026\001\
\\042\000\100\001\050\000\024\001\052\000\023\001\057\000\021\001\000\000\
\\126\003\000\000\
\\127\003\000\000\
\\128\003\000\000\
\\129\003\000\000\
\\130\003\000\000\
\\131\003\000\000\
\\132\003\000\000\
\\133\003\000\000\
\\134\003\000\000\
\\135\003\000\000\
\\136\003\058\000\083\001\000\000\
\\137\003\000\000\
\\138\003\000\000\
\\139\003\070\000\076\000\000\000\
\\140\003\000\000\
\\141\003\046\000\061\002\000\000\
\\142\003\000\000\
\\143\003\000\000\
\\144\003\046\000\026\002\000\000\
\\145\003\000\000\
\\146\003\000\000\
\\147\003\000\000\
\\148\003\003\000\031\002\000\000\
\\149\003\000\000\
\\150\003\000\000\
\\151\003\003\000\243\001\000\000\
\\151\003\003\000\243\001\046\000\245\001\000\000\
\\152\003\000\000\
\\153\003\000\000\
\\154\003\003\000\064\002\000\000\
\\155\003\000\000\
\\156\003\000\000\
\\157\003\000\000\
\\158\003\000\000\
\\159\003\003\000\036\002\000\000\
\\160\003\000\000\
\\161\003\000\000\
\\162\003\045\000\069\002\000\000\
\\163\003\000\000\
\\164\003\000\000\
\\165\003\003\000\247\001\000\000\
\\166\003\000\000\
\\167\003\000\000\
\\169\003\000\000\
\\170\003\000\000\
\\171\003\000\000\
\\172\003\000\000\
\\173\003\000\000\
\\174\003\000\000\
\\175\003\000\000\
\\176\003\000\000\
\\177\003\000\000\
\\178\003\000\000\
\\179\003\000\000\
\\180\003\000\000\
\\181\003\000\000\
\\183\003\000\000\
\\184\003\000\000\
\\185\003\000\000\
\\186\003\000\000\
\\187\003\000\000\
\\188\003\000\000\
\\189\003\000\000\
\\190\003\000\000\
\\191\003\000\000\
\\193\003\000\000\
\\194\003\000\000\
\\195\003\000\000\
\\196\003\000\000\
\\197\003\002\000\051\000\008\000\049\000\011\000\048\000\013\000\046\000\
\\017\000\044\000\018\000\043\000\020\000\041\000\021\000\040\000\
\\024\000\038\000\029\000\036\000\030\000\035\000\051\000\029\000\
\\055\000\028\000\057\000\027\000\000\000\
\\198\003\000\000\
\\199\003\000\000\
\\200\003\000\000\
\\201\003\000\000\
\"
val actionRowNumbers =
"\130\000\013\000\214\001\208\001\
\\085\000\213\001\213\001\213\001\
\\101\001\021\000\209\000\003\000\
\\207\000\087\000\184\000\145\000\
\\144\000\143\000\142\000\141\000\
\\140\000\139\000\138\000\137\000\
\\136\000\123\000\125\000\126\000\
\\119\000\206\000\028\000\174\000\
\\175\000\178\000\078\001\175\000\
\\124\000\173\000\088\000\131\000\
\\131\000\250\000\250\000\175\000\
\\180\000\183\000\177\000\057\000\
\\175\000\078\001\001\000\211\001\
\\212\001\210\001\209\001\001\000\
\\073\001\175\000\183\000\175\000\
\\208\000\164\000\185\000\166\000\
\\167\000\157\000\156\000\155\000\
\\165\000\102\001\076\000\161\000\
\\121\001\089\000\162\000\192\001\
\\058\000\163\000\187\000\150\000\
\\149\000\148\000\152\000\151\000\
\\073\000\205\000\090\000\196\000\
\\194\000\071\000\020\000\060\000\
\\061\000\188\000\016\000\179\000\
\\049\001\091\000\025\001\042\000\
\\229\000\122\000\079\001\028\001\
\\160\000\027\001\048\001\044\000\
\\182\000\179\000\077\001\127\000\
\\233\000\129\000\214\000\239\000\
\\248\000\171\000\172\000\242\000\
\\153\000\246\000\154\000\097\001\
\\224\000\088\000\249\000\147\000\
\\146\000\088\000\019\000\183\000\
\\044\001\024\001\043\000\004\001\
\\001\001\231\000\121\000\115\000\
\\221\000\218\000\181\000\238\000\
\\120\000\127\000\244\000\244\000\
\\092\000\159\000\158\000\018\000\
\\127\000\244\000\215\001\217\001\
\\216\001\128\000\063\001\004\000\
\\061\001\059\001\210\000\065\001\
\\072\001\073\001\212\000\213\000\
\\211\000\116\000\041\000\116\000\
\\116\000\134\001\186\000\175\000\
\\175\000\190\000\175\000\175\000\
\\193\000\191\000\189\000\175\000\
\\230\000\051\001\073\001\175\000\
\\050\001\056\001\026\001\026\000\
\\168\000\074\000\047\001\044\001\
\\093\000\024\000\023\000\022\000\
\\038\001\036\001\034\001\072\000\
\\062\000\063\000\064\000\082\001\
\\030\001\252\000\094\000\247\000\
\\245\000\098\001\095\001\037\000\
\\100\001\225\000\223\000\038\000\
\\228\000\131\000\241\000\240\000\
\\175\000\232\000\095\000\073\001\
\\055\001\002\001\183\000\255\000\
\\183\000\175\000\219\000\183\000\
\\019\001\096\000\235\000\078\001\
\\234\000\176\000\183\000\097\000\
\\056\000\064\001\169\000\170\000\
\\073\001\073\001\075\000\071\001\
\\077\000\065\000\066\000\116\001\
\\098\000\119\001\133\001\104\001\
\\005\000\078\000\091\001\059\000\
\\002\000\131\000\131\000\099\000\
\\122\001\008\000\117\000\135\001\
\\132\001\067\000\079\000\123\000\
\\045\000\116\000\078\001\137\001\
\\120\000\078\001\120\000\057\000\
\\204\000\195\000\201\000\200\000\
\\197\000\216\000\058\001\254\000\
\\052\001\027\000\073\001\053\001\
\\029\001\046\001\183\000\183\000\
\\032\001\080\001\183\000\033\001\
\\031\001\129\000\073\001\099\001\
\\134\000\227\000\175\000\224\000\
\\017\000\175\000\043\001\003\001\
\\000\001\222\000\220\000\023\001\
\\177\000\073\001\177\000\243\000\
\\013\001\120\000\128\000\217\000\
\\177\000\224\000\062\001\060\001\
\\066\001\073\001\074\001\073\001\
\\067\001\046\000\041\000\030\000\
\\108\001\123\000\116\000\116\000\
\\096\001\097\001\097\001\041\000\
\\124\001\125\000\047\000\142\001\
\\136\001\049\000\144\001\080\000\
\\116\000\153\001\081\000\140\001\
\\100\000\128\000\155\001\154\001\
\\120\001\147\001\127\000\145\001\
\\082\000\148\001\146\001\127\000\
\\152\001\018\001\150\001\149\001\
\\127\000\101\000\202\000\119\000\
\\057\001\251\000\179\000\183\000\
\\025\000\042\001\183\000\042\001\
\\035\001\037\001\081\001\008\001\
\\097\001\199\000\132\000\039\000\
\\175\000\005\001\020\001\177\000\
\\087\000\017\001\013\001\010\001\
\\078\001\018\001\236\000\013\001\
\\031\000\070\001\075\001\076\001\
\\117\001\078\001\106\001\118\001\
\\107\001\006\000\088\001\110\001\
\\085\001\109\001\068\000\083\001\
\\069\000\132\000\040\000\105\001\
\\123\001\125\001\078\001\143\001\
\\141\001\128\000\116\000\041\000\
\\116\000\070\000\116\000\124\000\
\\138\001\102\000\156\001\157\001\
\\169\001\073\001\170\001\184\001\
\\104\000\029\000\203\000\253\000\
\\054\001\040\001\086\000\045\001\
\\039\001\006\001\078\001\135\000\
\\175\000\032\000\134\000\215\000\
\\022\001\023\001\011\001\012\001\
\\016\001\009\001\237\000\068\001\
\\119\000\128\000\111\001\050\000\
\\093\001\116\000\116\000\092\001\
\\033\000\041\000\128\000\139\001\
\\105\000\196\001\010\000\083\000\
\\106\000\084\000\185\001\014\000\
\\118\000\163\001\162\001\128\000\
\\167\001\078\001\166\001\073\001\
\\182\001\120\000\120\000\178\001\
\\018\001\128\000\041\001\007\001\
\\034\000\198\000\192\000\224\000\
\\021\001\014\001\177\000\069\001\
\\107\000\112\001\078\001\087\001\
\\084\001\090\001\133\000\108\000\
\\041\000\200\001\126\000\116\000\
\\116\000\041\000\116\000\041\000\
\\116\000\187\001\123\000\052\000\
\\124\000\160\001\159\001\168\001\
\\164\001\120\000\173\001\183\001\
\\178\001\175\001\078\001\181\001\
\\151\001\103\001\135\000\015\001\
\\073\001\128\000\035\000\073\001\
\\198\001\199\001\202\001\011\000\
\\089\001\201\001\086\001\197\001\
\\109\000\193\001\110\000\186\001\
\\188\001\078\001\161\001\128\000\
\\165\001\171\001\078\001\176\001\
\\177\001\127\000\179\001\120\000\
\\036\000\131\001\111\000\094\001\
\\009\000\203\001\054\000\041\000\
\\041\000\128\000\158\001\172\001\
\\127\000\112\000\180\001\226\000\
\\129\001\046\000\073\001\126\001\
\\127\001\048\000\204\001\078\001\
\\195\001\194\001\113\000\103\000\
\\120\000\130\001\007\000\128\001\
\\128\000\073\001\178\001\113\001\
\\114\001\051\000\114\000\015\000\
\\174\001\115\001\073\001\189\001\
\\190\001\053\000\012\000\191\001\
\\205\001\206\001\055\000\207\001\
\\000\000"
val gotoT =
"\
\\142\000\120\002\145\000\001\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\009\000\033\000\008\000\087\000\007\000\
\\096\000\006\000\132\000\005\000\139\000\004\000\140\000\003\000\
\\143\000\002\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\033\000\008\000\087\000\007\000\096\000\006\000\132\000\005\000\
\\140\000\052\000\141\000\051\000\000\000\
\\033\000\008\000\087\000\007\000\096\000\006\000\132\000\005\000\
\\140\000\052\000\141\000\053\000\000\000\
\\033\000\008\000\087\000\007\000\096\000\006\000\132\000\005\000\
\\140\000\052\000\141\000\054\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\060\000\000\000\
\\000\000\
\\005\000\063\000\011\000\062\000\012\000\061\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\008\000\070\000\088\000\069\000\000\000\
\\009\000\073\000\097\000\072\000\000\000\
\\010\000\076\000\133\000\075\000\000\000\
\\003\000\078\000\000\000\
\\003\000\086\000\022\000\085\000\023\000\084\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\017\000\089\000\
\\018\000\088\000\025\000\011\000\026\000\010\000\027\000\087\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\019\000\092\000\
\\021\000\091\000\025\000\011\000\026\000\010\000\027\000\090\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\094\000\000\000\
\\001\000\103\000\007\000\102\000\015\000\101\000\038\000\100\000\
\\056\000\099\000\057\000\098\000\066\000\097\000\068\000\096\000\
\\081\000\095\000\000\000\
\\007\000\102\000\045\000\112\000\080\000\111\000\081\000\110\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\114\000\000\000\
\\008\000\117\000\014\000\116\000\036\000\115\000\000\000\
\\000\000\
\\004\000\121\000\005\000\120\000\035\000\119\000\000\000\
\\146\000\123\000\000\000\
\\146\000\124\000\000\000\
\\002\000\126\000\037\000\125\000\000\000\
\\002\000\126\000\037\000\129\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\130\000\000\000\
\\001\000\103\000\007\000\102\000\015\000\138\000\040\000\137\000\
\\042\000\136\000\044\000\135\000\056\000\134\000\057\000\133\000\
\\067\000\132\000\081\000\131\000\000\000\
\\001\000\103\000\015\000\101\000\028\000\141\000\030\000\140\000\
\\056\000\099\000\057\000\098\000\066\000\139\000\068\000\096\000\000\000\
\\015\000\144\000\054\000\143\000\000\000\
\\006\000\148\000\007\000\102\000\048\000\147\000\049\000\146\000\
\\081\000\145\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\151\000\000\000\
\\007\000\102\000\047\000\153\000\080\000\152\000\081\000\110\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\009\000\033\000\008\000\087\000\007\000\
\\096\000\006\000\132\000\005\000\139\000\004\000\140\000\003\000\
\\143\000\155\000\144\000\154\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\009\000\033\000\008\000\087\000\007\000\
\\096\000\006\000\132\000\005\000\139\000\004\000\140\000\003\000\
\\143\000\155\000\144\000\156\000\000\000\
\\007\000\163\000\070\000\162\000\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\166\000\000\000\
\\001\000\103\000\015\000\101\000\028\000\167\000\030\000\140\000\
\\056\000\099\000\057\000\098\000\066\000\139\000\068\000\096\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\168\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\001\000\103\000\015\000\101\000\038\000\184\000\056\000\099\000\
\\057\000\098\000\066\000\097\000\068\000\096\000\000\000\
\\069\000\185\000\000\000\
\\000\000\
\\069\000\188\000\000\000\
\\001\000\103\000\015\000\138\000\056\000\134\000\057\000\133\000\
\\067\000\189\000\000\000\
\\000\000\
\\005\000\191\000\012\000\190\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\003\000\196\000\005\000\195\000\061\000\194\000\062\000\193\000\000\000\
\\001\000\103\000\015\000\101\000\056\000\099\000\057\000\098\000\
\\058\000\203\000\059\000\202\000\066\000\201\000\068\000\096\000\000\000\
\\001\000\103\000\007\000\207\000\015\000\101\000\056\000\099\000\
\\057\000\098\000\060\000\206\000\066\000\205\000\068\000\096\000\
\\082\000\204\000\000\000\
\\001\000\103\000\015\000\101\000\038\000\209\000\056\000\099\000\
\\057\000\098\000\066\000\097\000\068\000\096\000\000\000\
\\000\000\
\\006\000\210\000\000\000\
\\000\000\
\\007\000\207\000\082\000\204\000\000\000\
\\000\000\
\\000\000\
\\008\000\117\000\014\000\116\000\036\000\211\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\004\000\121\000\005\000\120\000\035\000\212\000\000\000\
\\000\000\
\\033\000\008\000\085\000\215\000\086\000\214\000\087\000\213\000\000\000\
\\031\000\219\000\032\000\218\000\033\000\217\000\000\000\
\\004\000\121\000\005\000\120\000\035\000\222\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\004\000\121\000\005\000\120\000\035\000\223\000\000\000\
\\000\000\
\\001\000\103\000\015\000\138\000\040\000\225\000\042\000\136\000\
\\044\000\135\000\056\000\134\000\057\000\133\000\067\000\132\000\000\000\
\\064\000\226\000\000\000\
\\000\000\
\\001\000\103\000\015\000\138\000\056\000\134\000\057\000\133\000\
\\067\000\228\000\000\000\
\\043\000\229\000\000\000\
\\041\000\231\000\000\000\
\\000\000\
\\005\000\063\000\012\000\190\000\000\000\
\\000\000\
\\029\000\234\000\000\000\
\\000\000\
\\001\000\103\000\015\000\101\000\056\000\099\000\057\000\098\000\
\\060\000\206\000\066\000\205\000\068\000\096\000\000\000\
\\000\000\
\\005\000\236\000\000\000\
\\006\000\237\000\000\000\
\\034\000\238\000\000\000\
\\034\000\240\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\006\000\243\000\000\000\
\\034\000\244\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\006\000\246\000\013\000\245\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\003\000\252\000\075\000\251\000\076\000\250\000\000\000\
\\007\000\163\000\070\000\254\000\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\079\000\253\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\009\000\001\001\094\000\000\001\095\000\255\000\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\005\001\
\\084\000\004\001\090\000\003\001\000\000\
\\009\000\001\001\094\000\011\001\095\000\255\000\000\000\
\\009\000\001\001\094\000\014\001\095\000\013\001\099\000\012\001\000\000\
\\008\000\018\001\104\000\017\001\105\000\016\001\107\000\015\001\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\028\001\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\018\000\029\001\
\\025\000\011\000\026\000\010\000\027\000\087\000\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\021\000\031\001\
\\025\000\011\000\026\000\010\000\027\000\030\001\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\018\000\032\001\
\\025\000\011\000\026\000\010\000\027\000\087\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\033\001\000\000\
\\000\000\
\\000\000\
\\007\000\163\000\070\000\034\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\035\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\064\000\037\001\069\000\036\001\000\000\
\\069\000\039\001\000\000\
\\000\000\
\\000\000\
\\064\000\041\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\033\000\008\000\086\000\051\001\087\000\213\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\032\000\053\001\033\000\217\000\000\000\
\\000\000\
\\000\000\
\\146\000\055\001\000\000\
\\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\056\001\000\000\
\\000\000\
\\000\000\
\\007\000\163\000\070\000\058\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\000\000\
\\001\000\103\000\015\000\138\000\042\000\059\001\044\000\135\000\
\\056\000\134\000\057\000\133\000\067\000\132\000\000\000\
\\000\000\
\\001\000\103\000\015\000\138\000\040\000\060\001\042\000\136\000\
\\044\000\135\000\056\000\134\000\057\000\133\000\067\000\132\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\061\001\000\000\
\\000\000\
\\001\000\103\000\015\000\101\000\028\000\062\001\030\000\140\000\
\\056\000\099\000\057\000\098\000\066\000\139\000\068\000\096\000\000\000\
\\053\000\063\001\000\000\
\\000\000\
\\000\000\
\\007\000\102\000\045\000\067\001\080\000\111\000\081\000\110\000\000\000\
\\000\000\
\\015\000\069\001\051\000\068\001\000\000\
\\001\000\103\000\015\000\101\000\028\000\071\001\030\000\140\000\
\\056\000\099\000\057\000\098\000\066\000\139\000\068\000\096\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\007\000\163\000\072\000\074\001\073\000\159\000\074\000\158\000\
\\078\000\157\000\000\000\
\\007\000\163\000\070\000\075\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\104\000\083\001\105\000\016\001\107\000\015\001\000\000\
\\000\000\
\\089\000\084\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\146\000\089\001\000\000\
\\146\000\090\001\000\000\
\\000\000\
\\000\000\
\\098\000\092\001\000\000\
\\000\000\
\\000\000\
\\106\000\096\001\107\000\095\001\000\000\
\\000\000\
\\000\000\
\\008\000\102\001\127\000\101\001\000\000\
\\008\000\117\000\014\000\104\001\112\000\103\001\000\000\
\\009\000\108\001\094\000\107\001\095\000\255\000\108\000\106\001\000\000\
\\007\000\102\000\080\000\110\001\081\000\110\000\115\000\109\001\000\000\
\\000\000\
\\005\000\112\001\113\000\111\001\000\000\
\\007\000\102\000\080\000\115\001\081\000\110\000\115\000\114\001\
\\117\000\113\001\000\000\
\\005\000\117\001\125\000\116\001\000\000\
\\006\000\121\001\007\000\102\000\081\000\120\001\120\000\119\001\
\\121\000\118\001\000\000\
\\024\000\122\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\069\000\124\001\000\000\
\\039\000\125\001\000\000\
\\000\000\
\\000\000\
\\007\000\163\000\070\000\128\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\000\000\
\\065\000\129\001\000\000\
\\001\000\103\000\015\000\101\000\056\000\099\000\057\000\098\000\
\\066\000\131\001\068\000\096\000\000\000\
\\001\000\103\000\015\000\101\000\056\000\099\000\057\000\098\000\
\\059\000\132\001\066\000\201\000\068\000\096\000\000\000\
\\000\000\
\\000\000\
\\001\000\103\000\015\000\101\000\056\000\099\000\057\000\098\000\
\\059\000\133\001\066\000\201\000\068\000\096\000\000\000\
\\000\000\
\\000\000\
\\007\000\207\000\082\000\134\001\000\000\
\\007\000\163\000\070\000\135\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\033\000\008\000\086\000\051\001\087\000\213\000\000\000\
\\148\000\136\001\000\000\
\\032\000\053\001\033\000\217\000\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\020\000\138\001\
\\025\000\011\000\026\000\010\000\027\000\137\001\000\000\
\\031\000\139\001\032\000\218\000\033\000\217\000\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\141\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\055\000\142\001\000\000\
\\015\000\144\001\000\000\
\\007\000\163\000\070\000\145\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\015\000\069\001\051\000\146\001\000\000\
\\000\000\
\\050\000\147\001\000\000\
\\005\000\149\001\000\000\
\\006\000\246\000\013\000\150\001\000\000\
\\000\000\
\\015\000\069\001\051\000\151\001\000\000\
\\031\000\152\001\032\000\218\000\033\000\217\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\007\000\163\000\070\000\153\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\007\000\163\000\070\000\155\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\079\000\154\001\000\000\
\\000\000\
\\102\000\156\001\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\005\001\
\\084\000\004\001\090\000\158\001\000\000\
\\000\000\
\\000\000\
\\008\000\070\000\088\000\160\001\000\000\
\\009\000\001\001\091\000\163\001\094\000\162\001\095\000\161\001\000\000\
\\009\000\001\001\091\000\165\001\094\000\164\001\095\000\161\001\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\033\000\008\000\
\\083\000\168\001\084\000\167\001\085\000\166\001\086\000\214\000\
\\087\000\213\000\000\000\
\\033\000\008\000\085\000\169\001\086\000\214\000\087\000\213\000\000\000\
\\033\000\008\000\085\000\170\001\086\000\214\000\087\000\213\000\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\005\001\
\\084\000\004\001\090\000\171\001\000\000\
\\000\000\
\\009\000\073\000\097\000\172\001\000\000\
\\100\000\173\001\102\000\156\001\000\000\
\\000\000\
\\106\000\175\001\107\000\095\001\000\000\
\\008\000\117\000\014\000\104\001\112\000\176\001\000\000\
\\000\000\
\\000\000\
\\009\000\001\001\094\000\181\001\095\000\255\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\006\000\246\000\013\000\185\001\110\000\184\001\000\000\
\\000\000\
\\000\000\
\\009\000\187\001\108\000\186\001\000\000\
\\000\000\
\\006\000\188\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\006\000\190\001\000\000\
\\000\000\
\\053\000\191\001\000\000\
\\000\000\
\\000\000\
\\006\000\192\001\000\000\
\\000\000\
\\000\000\
\\003\000\086\000\022\000\194\001\000\000\
\\000\000\
\\000\000\
\\001\000\103\000\015\000\101\000\038\000\195\001\056\000\099\000\
\\057\000\098\000\066\000\097\000\068\000\096\000\000\000\
\\001\000\103\000\015\000\101\000\056\000\099\000\057\000\098\000\
\\066\000\196\001\068\000\096\000\000\000\
\\069\000\124\001\000\000\
\\063\000\197\001\000\000\
\\001\000\103\000\015\000\101\000\056\000\099\000\057\000\098\000\
\\066\000\199\001\068\000\096\000\000\000\
\\063\000\200\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\046\000\201\001\000\000\
\\033\000\008\000\085\000\203\001\086\000\214\000\087\000\213\000\000\000\
\\000\000\
\\147\000\205\001\000\000\
\\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\025\000\011\000\
\\026\000\010\000\027\000\207\001\000\000\
\\000\000\
\\000\000\
\\015\000\144\000\054\000\208\001\000\000\
\\005\000\063\000\011\000\209\001\012\000\061\000\000\000\
\\000\000\
\\050\000\210\001\000\000\
\\000\000\
\\007\000\102\000\047\000\211\001\080\000\152\000\081\000\110\000\000\000\
\\053\000\212\001\000\000\
\\000\000\
\\050\000\213\001\000\000\
\\000\000\
\\077\000\215\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\007\000\102\000\080\000\217\001\081\000\110\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\089\000\218\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\147\000\224\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\007\000\102\000\080\000\226\001\081\000\110\000\000\000\
\\106\000\175\001\107\000\095\001\000\000\
\\000\000\
\\006\000\246\000\013\000\185\001\110\000\227\001\000\000\
\\009\000\001\001\094\000\228\001\095\000\255\000\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\231\001\
\\084\000\230\001\135\000\229\001\000\000\
\\009\000\001\001\094\000\232\001\095\000\255\000\000\000\
\\000\000\
\\009\000\001\001\094\000\236\001\095\000\235\001\129\000\234\001\000\000\
\\008\000\117\000\014\000\238\001\111\000\237\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\009\000\187\001\108\000\186\001\000\000\
\\116\000\240\001\000\000\
\\007\000\163\000\070\000\242\001\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\116\000\240\001\000\000\
\\126\000\244\001\000\000\
\\000\000\
\\005\000\248\001\123\000\247\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\003\000\196\000\005\000\195\000\061\000\250\001\000\000\
\\000\000\
\\000\000\
\\000\000\
\\007\000\102\000\045\000\251\001\080\000\111\000\081\000\110\000\000\000\
\\149\000\252\001\000\000\
\\001\000\014\000\015\000\013\000\016\000\012\000\020\000\253\001\
\\025\000\011\000\026\000\010\000\027\000\137\001\000\000\
\\000\000\
\\148\000\255\001\000\000\
\\000\000\
\\000\000\
\\055\000\000\002\000\000\
\\000\000\
\\000\000\
\\052\000\001\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\003\000\252\000\075\000\003\002\000\000\
\\006\000\246\000\013\000\004\002\000\000\
\\000\000\
\\092\000\005\002\102\000\156\001\000\000\
\\000\000\
\\009\000\001\001\094\000\007\002\095\000\255\000\000\000\
\\009\000\001\001\094\000\008\002\095\000\255\000\000\000\
\\000\000\
\\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\010\002\
\\084\000\167\001\000\000\
\\006\000\246\000\013\000\011\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\134\000\013\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\128\000\021\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\006\000\246\000\013\000\026\002\109\000\025\002\000\000\
\\000\000\
\\007\000\102\000\080\000\110\001\081\000\110\000\115\000\027\002\000\000\
\\114\000\028\002\000\000\
\\007\000\163\000\070\000\030\002\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\005\000\117\001\125\000\031\002\000\000\
\\005\000\248\001\123\000\032\002\000\000\
\\122\000\033\002\000\000\
\\053\000\035\002\000\000\
\\006\000\246\000\013\000\036\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\031\000\038\002\032\000\218\000\033\000\217\000\000\000\
\\000\000\
\\000\000\
\\015\000\069\001\051\000\039\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\007\000\102\000\080\000\041\002\081\000\110\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\147\000\042\002\000\000\
\\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\231\001\
\\084\000\230\001\135\000\044\002\000\000\
\\000\000\
\\010\000\076\000\133\000\045\002\000\000\
\\009\000\001\001\094\000\048\002\095\000\047\002\136\000\046\002\000\000\
\\009\000\001\001\094\000\050\002\095\000\047\002\136\000\049\002\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\231\001\
\\084\000\230\001\135\000\051\002\000\000\
\\009\000\001\001\094\000\052\002\095\000\255\000\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\231\001\
\\084\000\230\001\135\000\053\002\000\000\
\\009\000\001\001\094\000\054\002\095\000\255\000\000\000\
\\000\000\
\\008\000\102\001\127\000\055\002\000\000\
\\102\000\156\001\130\000\056\002\000\000\
\\008\000\117\000\014\000\238\001\111\000\058\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\005\000\112\001\113\000\060\002\000\000\
\\118\000\061\002\000\000\
\\000\000\
\\122\000\063\002\000\000\
\\000\000\
\\007\000\102\000\080\000\065\002\081\000\110\000\119\000\064\002\000\000\
\\124\000\066\002\000\000\
\\000\000\
\\000\000\
\\149\000\068\002\000\000\
\\000\000\
\\007\000\163\000\070\000\069\002\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\006\000\246\000\013\000\070\002\000\000\
\\000\000\
\\007\000\163\000\070\000\072\002\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\134\000\073\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\007\000\102\000\080\000\077\002\081\000\110\000\000\000\
\\000\000\
\\006\000\246\000\013\000\026\002\109\000\078\002\000\000\
\\000\000\
\\000\000\
\\007\000\102\000\080\000\080\002\081\000\110\000\117\000\079\002\000\000\
\\000\000\
\\000\000\
\\006\000\081\002\000\000\
\\000\000\
\\005\000\248\001\123\000\082\002\000\000\
\\000\000\
\\103\000\084\002\000\000\
\\000\000\
\\000\000\
\\098\000\088\002\101\000\087\002\103\000\084\002\000\000\
\\000\000\
\\102\000\156\001\137\000\090\002\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\231\001\
\\084\000\230\001\135\000\092\002\000\000\
\\008\000\117\000\010\000\007\001\014\000\006\001\083\000\231\001\
\\084\000\230\001\135\000\093\002\000\000\
\\006\000\246\000\013\000\094\002\000\000\
\\000\000\
\\000\000\
\\006\000\095\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\102\000\097\002\000\000\
\\007\000\163\000\070\000\098\002\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\000\000\
\\009\000\073\000\097\000\172\001\100\000\099\002\102\000\097\002\000\000\
\\000\000\
\\007\000\102\000\080\000\100\002\081\000\110\000\000\000\
\\000\000\
\\000\000\
\\000\000\
\\000\000\
\\005\000\248\001\123\000\102\002\000\000\
\\000\000\
\\089\000\104\002\093\000\103\002\103\000\084\002\000\000\
\\000\000\
\\006\000\246\000\013\000\106\002\000\000\
\\007\000\163\000\070\000\107\002\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\122\000\108\002\000\000\
\\000\000\
\\000\000\
\\008\000\070\000\088\000\160\001\092\000\109\002\102\000\097\002\000\000\
\\000\000\
\\103\000\084\002\128\000\112\002\131\000\111\002\000\000\
\\000\000\
\\000\000\
\\007\000\163\000\070\000\114\002\071\000\161\000\072\000\160\000\
\\073\000\159\000\074\000\158\000\078\000\157\000\000\000\
\\000\000\
\\000\000\
\\008\000\102\001\102\000\097\002\127\000\055\002\130\000\115\002\000\000\
\\103\000\084\002\134\000\117\002\138\000\116\002\000\000\
\\000\000\
\\000\000\
\\000\000\
\\010\000\076\000\102\000\097\002\133\000\045\002\137\000\119\002\000\000\
\\000\000\
\\000\000\
\"
val numstates = 633
val numrules = 336
val s = ref "" and index = ref 0
val string_to_int = fn () => 
let val i = !index
in index := i+2; Char.ord(String.sub(!s,i)) + Char.ord(String.sub(!s,i+1)) * 256
end
val string_to_list = fn s' =>
    let val len = String.size s'
        fun f () =
           if !index < len then string_to_int() :: f()
           else nil
   in index := 0; s := s'; f ()
   end
val string_to_pairlist = fn (conv_key,conv_entry) =>
     let fun f () =
         case string_to_int()
         of 0 => EMPTY
          | n => PAIR(conv_key (n-1),conv_entry (string_to_int()),f())
     in f
     end
val string_to_pairlist_default = fn (conv_key,conv_entry) =>
    let val conv_row = string_to_pairlist(conv_key,conv_entry)
    in fn () =>
       let val default = conv_entry(string_to_int())
           val row = conv_row()
       in (row,default)
       end
   end
val string_to_table = fn (convert_row,s') =>
    let val len = String.size s'
        fun f ()=
           if !index < len then convert_row() :: f()
           else nil
     in (s := s'; index := 0; f ())
     end
local
  val memo = Array.array(numstates+numrules,ERROR)
  val _ =let fun g i=(Array.update(memo,i,REDUCE(i-numstates)); g(i+1))
       fun f i =
            if i=numstates then g i
            else (Array.update(memo,i,SHIFT (STATE i)); f (i+1))
          in f 0 handle General.Subscript => ()
          end
in
val entry_to_action = fn 0 => ACCEPT | 1 => ERROR | j => Array.sub(memo,(j-2))
end
val gotoT=Array.fromList(string_to_table(string_to_pairlist(NT,STATE),gotoT))
val actionRows=string_to_table(string_to_pairlist_default(T,entry_to_action),actionRows)
val actionRowNumbers = string_to_list actionRowNumbers
val actionT = let val actionRowLookUp=
let val a=Array.fromList(actionRows) in fn i=>Array.sub(a,i) end
in Array.fromList(List.map actionRowLookUp actionRowNumbers)
end
in LrTable.mkLrTable {actions=actionT,gotos=gotoT,numRules=numrules,
numStates=numstates,initialState=STATE 0}
end
end
local open Header in
type pos = Source.pos
type arg =  ( Source.region -> Source.loc ) *Infix.InfEnv
structure MlyValue = 
struct
datatype svalue = VOID | ntVOID of unit ->  unit
 | LONGID of unit ->  (string list*string)
 | ETYVAR of unit ->  (string) | TYVAR of unit ->  (string)
 | SYMBOL of unit ->  (string) | ALPHA of unit ->  (string)
 | CHAR of unit ->  (string) | STRING of unit ->  (string)
 | REAL of unit ->  (string) | HEXWORD of unit ->  (string)
 | WORD of unit ->  (string) | HEXINT of unit ->  (string)
 | INT of unit ->  (string) | NUMERIC of unit ->  (string)
 | DIGIT of unit ->  (string) | popLocalInfix of unit ->  (unit)
 | pushLocalInfix of unit ->  (unit) | popInfix of unit ->  (unit)
 | pushInfix of unit ->  (unit) | initInfix of unit ->  (unit)
 | program_opt of unit ->  (Program option)
 | program' of unit ->  (Program)
 | program of unit ->  (Program*Infix.InfEnv)
 | topdec_opt of unit ->  (TopDec option)
 | topdec1 of unit ->  (TopDec) | topdec of unit ->  (TopDec)
 | AND_tyreadesc_opt__AND_funbind_opt of unit ->  (TyReaDesc option*FunBind option)
 | tyreadesc__AND_funbind_opt of unit ->  (TyReaDesc*FunBind option)
 | sigexp__AND_funbind_opt of unit ->  (SigExp*FunBind option)
 | strexp__AND_funbind_opt of unit ->  (StrExp*FunBind option)
 | AND_funbind_opt of unit ->  (FunBind option)
 | funbind of unit ->  (FunBind) | fundec of unit ->  (FunDec)
 | AND_tyreadesc_opt__AND_strdesc_opt of unit ->  (TyReaDesc option*StrDesc option)
 | tyreadesc__AND_strdesc_opt of unit ->  (TyReaDesc*StrDesc option)
 | sigexp__AND_strdesc_opt of unit ->  (SigExp*StrDesc option)
 | AND_strdesc_opt of unit ->  (StrDesc option)
 | strdesc of unit ->  (StrDesc)
 | AND_exdesc_opt of unit ->  (ExDesc option)
 | exdesc of unit ->  (ExDesc)
 | BAR_condesc_opt of unit ->  (ConDesc option)
 | condesc of unit ->  (ConDesc)
 | AND_datdesc_opt of unit ->  (DatDesc option)
 | datdesc1 of unit ->  (DatDesc) | datdesc0 of unit ->  (DatDesc)
 | datdesc of unit ->  (DatDesc)
 | AND_syndesc_opt of unit ->  (SynDesc option)
 | syndesc of unit ->  (SynDesc)
 | AND_typdesc_opt of unit ->  (TypDesc option)
 | typdesc of unit ->  (TypDesc)
 | AND_valdesc_opt of unit ->  (ValDesc option)
 | valdesc of unit ->  (ValDesc)
 | longstrid_EQUALS_list2 of unit ->  (longStrId list)
 | longstrid_EQUALS_list1 of unit ->  (longStrId list)
 | longtycon_EQUALS_list2 of unit ->  (longTyCon list)
 | longtycon_EQUALS_list1 of unit ->  (longTyCon list)
 | sigid_list2 of unit ->  (SigId list) | spec1'' of unit ->  (Spec)
 | spec1' of unit ->  (Spec) | spec1 of unit ->  (Spec)
 | spec of unit ->  (Spec)
 | AND_tyreadesc_opt of unit ->  (TyReaDesc option)
 | tyreadesc of unit ->  (TyReaDesc)
 | AND_tyreadesc_opt__AND_sigbind_opt of unit ->  (TyReaDesc option*SigBind option)
 | tyreadesc__AND_sigbind_opt of unit ->  (TyReaDesc*SigBind option)
 | sigexp__AND_sigbind_opt of unit ->  (SigExp*SigBind option)
 | AND_sigbind_opt of unit ->  (SigBind option)
 | sigbind of unit ->  (SigBind) | sigdec of unit ->  (SigDec)
 | sigexp' of unit ->  (SigExp) | sigexp of unit ->  (SigExp)
 | AND_tyreadesc_opt__AND_strbind_opt of unit ->  (TyReaDesc option*StrBind option)
 | tyreadesc__AND_strbind_opt of unit ->  (TyReaDesc*StrBind option)
 | sigexp__AND_strbind_opt of unit ->  (SigExp*StrBind option)
 | strexp__AND_strbind_opt of unit ->  (StrExp*StrBind option)
 | AND_strbind_opt of unit ->  (StrBind option)
 | strbind of unit ->  (StrBind) | strdec1' of unit ->  (StrDec)
 | strdec1 of unit ->  (StrDec) | strdec of unit ->  (StrDec)
 | strexp' of unit ->  (StrExp) | strexp of unit ->  (StrExp)
 | tyvar_COMMA_list1 of unit ->  (TyVar list)
 | tyvarseq1 of unit ->  (TyVar seq)
 | tyvarseq of unit ->  (TyVar seq)
 | ty_COMMA_list2 of unit ->  (Ty list) | tyseq of unit ->  (Ty seq)
 | COMMA_tyrow_opt of unit ->  (TyRow option)
 | tyrow_opt of unit ->  (TyRow option) | tyrow of unit ->  (TyRow)
 | atty of unit ->  (Ty) | consty of unit ->  (Ty)
 | ty_STAR_list of unit ->  (Ty list) | tupty of unit ->  (Ty)
 | ty of unit ->  (Ty) | COLON_ty_list1 of unit ->  (Ty list)
 | atpat_list2 of unit ->  (AtPat list)
 | atpat_list1 of unit ->  (AtPat list) | pat of unit ->  (Pat)
 | AS_pat_opt of unit ->  (Pat option)
 | COLON_ty_opt of unit ->  (Ty option)
 | COMMA_patrow_opt of unit ->  (PatRow option)
 | patrow_opt of unit ->  (PatRow option)
 | patrow of unit ->  (PatRow)
 | pat_COMMA_list2 of unit ->  (Pat list)
 | pat_COMMA_list1 of unit ->  (Pat list)
 | pat_COMMA_list0 of unit ->  (Pat list) | atpat' of unit ->  (AtPat)
 | atpat of unit ->  (AtPat)
 | AND_exbind_opt of unit ->  (ExBind option)
 | exbind of unit ->  (ExBind) | OF_ty_opt of unit ->  (Ty option)
 | BAR_conbind_opt of unit ->  (ConBind option)
 | conbind of unit ->  (ConBind)
 | AND_datbind_opt of unit ->  (DatBind option)
 | datbind1 of unit ->  (DatBind) | datbind0 of unit ->  (DatBind)
 | datbind of unit ->  (DatBind)
 | AND_typbind_opt of unit ->  (TypBind option)
 | typbind of unit ->  (TypBind) | fmrule of unit ->  (Fmrule)
 | BAR_fmatch_opt of unit ->  (Fmatch option)
 | fmatch of unit ->  (Fmatch)
 | AND_fvalbind_opt of unit ->  (FvalBind option)
 | fvalbind of unit ->  (FvalBind)
 | AND_valbind_opt of unit ->  (ValBind option)
 | valbind of unit ->  (ValBind) | d_opt of unit ->  (int)
 | longstrid_list1 of unit ->  (longStrId list)
 | vid_list1 of unit ->  (VId list)
 | WITHTYPE_typbind_opt of unit ->  (TypBind option)
 | dec1' of unit ->  (Dec) | dec1 of unit ->  (Dec)
 | dec of unit ->  (Dec) | mrule of unit ->  (Mrule)
 | BAR_match_opt of unit ->  (Match option)
 | match of unit ->  (Match) | exp of unit ->  (Exp)
 | infexp of unit ->  (InfExp) | appexp of unit ->  (AppExp)
 | COMMA_exprow_opt of unit ->  (ExpRow option)
 | exprow_opt of unit ->  (ExpRow option)
 | exprow of unit ->  (ExpRow)
 | exp_SEMICOLON_list2 of unit ->  (Exp list)
 | exp_SEMICOLON_list1 of unit ->  (Exp list)
 | exp_COMMA_list2 of unit ->  (Exp list)
 | exp_COMMA_list1 of unit ->  (Exp list)
 | exp_COMMA_list0 of unit ->  (Exp list) | atexp of unit ->  (AtExp)
 | OP_opt of unit ->  (Op option) | longstrid of unit ->  (longStrId)
 | longtycon of unit ->  (longTyCon) | longvid' of unit ->  (longVId)
 | longvid of unit ->  (longVId) | funid of unit ->  (FunId)
 | sigid of unit ->  (SigId) | strid of unit ->  (StrId)
 | tyvar of unit ->  (TyVar) | tycon of unit ->  (TyCon)
 | vid' of unit ->  (VId) | vid of unit ->  (VId)
 | lab of unit ->  (Lab) | d of unit ->  (int)
 | scon of unit ->  (SCon)
end
type svalue = MlyValue.svalue
type result = Program*Infix.InfEnv
end
structure EC=
struct
open LrTable
infix 5 $$
fun x $$ y = y::x
val is_keyword =
fn (T 1) => true | (T 2) => true | (T 3) => true | (T 4) => true | (T 
5) => true | (T 6) => true | (T 7) => true | (T 8) => true | (T 9)
 => true | (T 10) => true | (T 11) => true | (T 12) => true | (T 13)
 => true | (T 14) => true | (T 15) => true | (T 16) => true | (T 17)
 => true | (T 18) => true | (T 19) => true | (T 20) => true | (T 21)
 => true | (T 22) => true | (T 23) => true | (T 24) => true | (T 25)
 => true | (T 26) => true | (T 27) => true | (T 28) => true | (T 29)
 => true | (T 30) => true | (T 31) => true | (T 32) => true | (T 49)
 => true | (T 50) => true | (T 51) => true | (T 52) => true | (T 53)
 => true | (T 54) => true | (T 55) => true | (T 56) => true | (T 57)
 => true | _ => false
val preferred_change : (term list * term list) list = 
nil
val noShift = 
fn (T 0) => true | _ => false
val showTerminal =
fn (T 0) => "EOF"
  | (T 1) => "ABSTYPE"
  | (T 2) => "AND"
  | (T 3) => "ANDALSO"
  | (T 4) => "AS"
  | (T 5) => "CASE"
  | (T 6) => "DO"
  | (T 7) => "DATATYPE"
  | (T 8) => "ELSE"
  | (T 9) => "END"
  | (T 10) => "EXCEPTION"
  | (T 11) => "FN"
  | (T 12) => "FUN"
  | (T 13) => "HANDLE"
  | (T 14) => "IF"
  | (T 15) => "IN"
  | (T 16) => "INFIX"
  | (T 17) => "INFIXR"
  | (T 18) => "LET"
  | (T 19) => "LOCAL"
  | (T 20) => "NONFIX"
  | (T 21) => "OF"
  | (T 22) => "OP"
  | (T 23) => "OPEN"
  | (T 24) => "ORELSE"
  | (T 25) => "RAISE"
  | (T 26) => "REC"
  | (T 27) => "THEN"
  | (T 28) => "TYPE"
  | (T 29) => "VAL"
  | (T 30) => "WITH"
  | (T 31) => "WITHTYPE"
  | (T 32) => "WHILE"
  | (T 33) => "LPAR"
  | (T 34) => "RPAR"
  | (T 35) => "LBRACK"
  | (T 36) => "RBRACK"
  | (T 37) => "LBRACE"
  | (T 38) => "RBRACE"
  | (T 39) => "COMMA"
  | (T 40) => "COLON"
  | (T 41) => "SEMICOLON"
  | (T 42) => "DOTS"
  | (T 43) => "UNDERBAR"
  | (T 44) => "BAR"
  | (T 45) => "EQUALS"
  | (T 46) => "DARROW"
  | (T 47) => "ARROW"
  | (T 48) => "HASH"
  | (T 49) => "EQTYPE"
  | (T 50) => "FUNCTOR"
  | (T 51) => "INCLUDE"
  | (T 52) => "SHARING"
  | (T 53) => "SIG"
  | (T 54) => "SIGNATURE"
  | (T 55) => "STRUCT"
  | (T 56) => "STRUCTURE"
  | (T 57) => "WHERE"
  | (T 58) => "SEAL"
  | (T 59) => "ZERO"
  | (T 60) => "DIGIT"
  | (T 61) => "NUMERIC"
  | (T 62) => "INT"
  | (T 63) => "HEXINT"
  | (T 64) => "WORD"
  | (T 65) => "HEXWORD"
  | (T 66) => "REAL"
  | (T 67) => "STRING"
  | (T 68) => "CHAR"
  | (T 69) => "ALPHA"
  | (T 70) => "SYMBOL"
  | (T 71) => "STAR"
  | (T 72) => "TYVAR"
  | (T 73) => "ETYVAR"
  | (T 74) => "LONGID"
  | _ => "bogus-term"
local open Header in
val errtermvalue=
fn _ => MlyValue.VOID
end
val terms : term list = nil
 $$ (T 71) $$ (T 59) $$ (T 58) $$ (T 57) $$ (T 56) $$ (T 55) $$ (T 54)
 $$ (T 53) $$ (T 52) $$ (T 51) $$ (T 50) $$ (T 49) $$ (T 48) $$ (T 47)
 $$ (T 46) $$ (T 45) $$ (T 44) $$ (T 43) $$ (T 42) $$ (T 41) $$ (T 40)
 $$ (T 39) $$ (T 38) $$ (T 37) $$ (T 36) $$ (T 35) $$ (T 34) $$ (T 33)
 $$ (T 32) $$ (T 31) $$ (T 30) $$ (T 29) $$ (T 28) $$ (T 27) $$ (T 26)
 $$ (T 25) $$ (T 24) $$ (T 23) $$ (T 22) $$ (T 21) $$ (T 20) $$ (T 19)
 $$ (T 18) $$ (T 17) $$ (T 16) $$ (T 15) $$ (T 14) $$ (T 13) $$ (T 12)
 $$ (T 11) $$ (T 10) $$ (T 9) $$ (T 8) $$ (T 7) $$ (T 6) $$ (T 5) $$ 
(T 4) $$ (T 3) $$ (T 2) $$ (T 1) $$ (T 0)end
structure Actions =
struct 
exception mlyAction of int
local open Header in
val actions = 
fn (i392,defaultPos,stack,
    (A, J0):arg) =>
case (i392,stack)
of  ( 0, ( rest671)) => let val  result = MlyValue.initInfix (fn _ =>
 ( initJandJ'(J0) ))
 in ( LrTable.NT 144, ( result, defaultPos, defaultPos), rest671)
end
|  ( 1, ( rest671)) => let val  result = MlyValue.pushInfix (fn _ => (
 pushJ() ))
 in ( LrTable.NT 145, ( result, defaultPos, defaultPos), rest671)
end
|  ( 2, ( rest671)) => let val  result = MlyValue.popInfix (fn _ => (
 popJ() ))
 in ( LrTable.NT 146, ( result, defaultPos, defaultPos), rest671)
end
|  ( 3, ( rest671)) => let val  result = MlyValue.pushLocalInfix (fn _
 => ( pushJ'shiftJ() ))
 in ( LrTable.NT 147, ( result, defaultPos, defaultPos), rest671)
end
|  ( 4, ( rest671)) => let val  result = MlyValue.popLocalInfix (fn _
 => ( popJandJ'() ))
 in ( LrTable.NT 148, ( result, defaultPos, defaultPos), rest671)
end
|  ( 5, ( ( _, ( _, (ZEROleft as ZERO1left), (ZEROright as ZERO1right)
)) :: rest671)) => let val  result = MlyValue.scon (fn _ => (
 SCon.INT(SCon.DEC, "0")@@A(ZEROleft, ZEROright) ))
 in ( LrTable.NT 0, ( result, ZERO1left, ZERO1right), rest671)
end
|  ( 6, ( ( _, ( MlyValue.DIGIT DIGIT1, (DIGITleft as DIGIT1left), (
DIGITright as DIGIT1right))) :: rest671)) => let val  result = 
MlyValue.scon (fn _ => let val  (DIGIT as DIGIT1) = DIGIT1 ()
 in ( SCon.INT(SCon.DEC, DIGIT)@@A(DIGITleft, DIGITright) )
end)
 in ( LrTable.NT 0, ( result, DIGIT1left, DIGIT1right), rest671)
end
|  ( 7, ( ( _, ( MlyValue.NUMERIC NUMERIC1, (NUMERICleft as 
NUMERIC1left), (NUMERICright as NUMERIC1right))) :: rest671)) => let
 val  result = MlyValue.scon (fn _ => let val  (NUMERIC as NUMERIC1) =
 NUMERIC1 ()
 in ( SCon.INT(SCon.DEC, NUMERIC)@@A(NUMERICleft, NUMERICright) )
end)
 in ( LrTable.NT 0, ( result, NUMERIC1left, NUMERIC1right), rest671)

end
|  ( 8, ( ( _, ( MlyValue.INT INT1, (INTleft as INT1left), (INTright
 as INT1right))) :: rest671)) => let val  result = MlyValue.scon (fn _
 => let val  (INT as INT1) = INT1 ()
 in ( SCon.INT(SCon.DEC, INT)@@A(INTleft, INTright) )
end)
 in ( LrTable.NT 0, ( result, INT1left, INT1right), rest671)
end
|  ( 9, ( ( _, ( MlyValue.HEXINT HEXINT1, (HEXINTleft as HEXINT1left),
 (HEXINTright as HEXINT1right))) :: rest671)) => let val  result = 
MlyValue.scon (fn _ => let val  (HEXINT as HEXINT1) = HEXINT1 ()
 in ( SCon.INT(SCon.HEX, HEXINT)@@A(HEXINTleft, HEXINTright) )
end)
 in ( LrTable.NT 0, ( result, HEXINT1left, HEXINT1right), rest671)
end
|  ( 10, ( ( _, ( MlyValue.WORD WORD1, (WORDleft as WORD1left), (
WORDright as WORD1right))) :: rest671)) => let val  result = 
MlyValue.scon (fn _ => let val  (WORD as WORD1) = WORD1 ()
 in ( SCon.WORD(SCon.DEC, WORD)@@A(WORDleft, WORDright) )
end)
 in ( LrTable.NT 0, ( result, WORD1left, WORD1right), rest671)
end
|  ( 11, ( ( _, ( MlyValue.HEXWORD HEXWORD1, (HEXWORDleft as 
HEXWORD1left), (HEXWORDright as HEXWORD1right))) :: rest671)) => let
 val  result = MlyValue.scon (fn _ => let val  (HEXWORD as HEXWORD1) =
 HEXWORD1 ()
 in ( SCon.WORD(SCon.HEX, HEXWORD)@@A(HEXWORDleft, HEXWORDright) )
end
)
 in ( LrTable.NT 0, ( result, HEXWORD1left, HEXWORD1right), rest671)

end
|  ( 12, ( ( _, ( MlyValue.REAL REAL1, (REALleft as REAL1left), (
REALright as REAL1right))) :: rest671)) => let val  result = 
MlyValue.scon (fn _ => let val  (REAL as REAL1) = REAL1 ()
 in ( SCon.REAL(REAL)@@A(REALleft, REALright) )
end)
 in ( LrTable.NT 0, ( result, REAL1left, REAL1right), rest671)
end
|  ( 13, ( ( _, ( MlyValue.STRING STRING1, (STRINGleft as STRING1left)
, (STRINGright as STRING1right))) :: rest671)) => let val  result = 
MlyValue.scon (fn _ => let val  (STRING as STRING1) = STRING1 ()
 in ( SCon.STRING(STRING)@@A(STRINGleft, STRINGright) )
end)
 in ( LrTable.NT 0, ( result, STRING1left, STRING1right), rest671)
end
|  ( 14, ( ( _, ( MlyValue.CHAR CHAR1, (CHARleft as CHAR1left), (
CHARright as CHAR1right))) :: rest671)) => let val  result = 
MlyValue.scon (fn _ => let val  (CHAR as CHAR1) = CHAR1 ()
 in ( SCon.CHAR(CHAR)@@A(CHARleft, CHARright) )
end)
 in ( LrTable.NT 0, ( result, CHAR1left, CHAR1right), rest671)
end
|  ( 15, ( ( _, ( _, ZERO1left, ZERO1right)) :: rest671)) => let val  
result = MlyValue.d (fn _ => ( 0 ))
 in ( LrTable.NT 1, ( result, ZERO1left, ZERO1right), rest671)
end
|  ( 16, ( ( _, ( MlyValue.DIGIT DIGIT1, DIGIT1left, DIGIT1right)) :: 
rest671)) => let val  result = MlyValue.d (fn _ => let val  (DIGIT as 
DIGIT1) = DIGIT1 ()
 in ( Option.valOf(Int.fromString DIGIT) )
end)
 in ( LrTable.NT 1, ( result, DIGIT1left, DIGIT1right), rest671)
end
|  ( 17, ( ( _, ( MlyValue.ALPHA ALPHA1, (ALPHAleft as ALPHA1left), (
ALPHAright as ALPHA1right))) :: rest671)) => let val  result = 
MlyValue.lab (fn _ => let val  (ALPHA as ALPHA1) = ALPHA1 ()
 in ( Lab.fromString(ALPHA)@@A(ALPHAleft, ALPHAright) )
end)
 in ( LrTable.NT 2, ( result, ALPHA1left, ALPHA1right), rest671)
end
|  ( 18, ( ( _, ( MlyValue.SYMBOL SYMBOL1, (SYMBOLleft as SYMBOL1left)
, (SYMBOLright as SYMBOL1right))) :: rest671)) => let val  result = 
MlyValue.lab (fn _ => let val  (SYMBOL as SYMBOL1) = SYMBOL1 ()
 in ( Lab.fromString(SYMBOL)@@A(SYMBOLleft, SYMBOLright) )
end)
 in ( LrTable.NT 2, ( result, SYMBOL1left, SYMBOL1right), rest671)
end
|  ( 19, ( ( _, ( _, (STARleft as STAR1left), (STARright as STAR1right
))) :: rest671)) => let val  result = MlyValue.lab (fn _ => (
 Lab.fromString("*")@@A(STARleft, STARright) ))
 in ( LrTable.NT 2, ( result, STAR1left, STAR1right), rest671)
end
|  ( 20, ( ( _, ( MlyValue.DIGIT DIGIT1, (DIGITleft as DIGIT1left), (
DIGITright as DIGIT1right))) :: rest671)) => let val  result = 
MlyValue.lab (fn _ => let val  (DIGIT as DIGIT1) = DIGIT1 ()
 in ( Lab.fromString(DIGIT)@@A(DIGITleft, DIGITright) )
end)
 in ( LrTable.NT 2, ( result, DIGIT1left, DIGIT1right), rest671)
end
|  ( 21, ( ( _, ( MlyValue.NUMERIC NUMERIC1, (NUMERICleft as 
NUMERIC1left), (NUMERICright as NUMERIC1right))) :: rest671)) => let
 val  result = MlyValue.lab (fn _ => let val  (NUMERIC as NUMERIC1) = 
NUMERIC1 ()
 in ( Lab.fromString(NUMERIC)@@A(NUMERICleft, NUMERICright) )
end)
 in ( LrTable.NT 2, ( result, NUMERIC1left, NUMERIC1right), rest671)

end
|  ( 22, ( ( _, ( MlyValue.vid' vid'1, vid'1left, vid'1right)) :: 
rest671)) => let val  result = MlyValue.vid (fn _ => let val  (vid'
 as vid'1) = vid'1 ()
 in ( vid' )
end)
 in ( LrTable.NT 3, ( result, vid'1left, vid'1right), rest671)
end
|  ( 23, ( ( _, ( _, (EQUALSleft as EQUALS1left), (EQUALSright as 
EQUALS1right))) :: rest671)) => let val  result = MlyValue.vid (fn _
 => ( VId.fromString("=")@@A(EQUALSleft, EQUALSright) ))
 in ( LrTable.NT 3, ( result, EQUALS1left, EQUALS1right), rest671)
end
|  ( 24, ( ( _, ( MlyValue.ALPHA ALPHA1, (ALPHAleft as ALPHA1left), (
ALPHAright as ALPHA1right))) :: rest671)) => let val  result = 
MlyValue.vid' (fn _ => let val  (ALPHA as ALPHA1) = ALPHA1 ()
 in ( VId.fromString(ALPHA)@@A(ALPHAleft, ALPHAright) )
end)
 in ( LrTable.NT 4, ( result, ALPHA1left, ALPHA1right), rest671)
end
|  ( 25, ( ( _, ( MlyValue.SYMBOL SYMBOL1, (SYMBOLleft as SYMBOL1left)
, (SYMBOLright as SYMBOL1right))) :: rest671)) => let val  result = 
MlyValue.vid' (fn _ => let val  (SYMBOL as SYMBOL1) = SYMBOL1 ()
 in ( VId.fromString(SYMBOL)@@A(SYMBOLleft, SYMBOLright) )
end)
 in ( LrTable.NT 4, ( result, SYMBOL1left, SYMBOL1right), rest671)
end
|  ( 26, ( ( _, ( _, (STARleft as STAR1left), (STARright as STAR1right
))) :: rest671)) => let val  result = MlyValue.vid' (fn _ => (
 VId.fromString("*")@@A(STARleft, STARright) ))
 in ( LrTable.NT 4, ( result, STAR1left, STAR1right), rest671)
end
|  ( 27, ( ( _, ( MlyValue.ALPHA ALPHA1, (ALPHAleft as ALPHA1left), (
ALPHAright as ALPHA1right))) :: rest671)) => let val  result = 
MlyValue.tycon (fn _ => let val  (ALPHA as ALPHA1) = ALPHA1 ()
 in ( TyCon.fromString(ALPHA)@@A(ALPHAleft, ALPHAright) )
end)
 in ( LrTable.NT 5, ( result, ALPHA1left, ALPHA1right), rest671)
end
|  ( 28, ( ( _, ( MlyValue.SYMBOL SYMBOL1, (SYMBOLleft as SYMBOL1left)
, (SYMBOLright as SYMBOL1right))) :: rest671)) => let val  result = 
MlyValue.tycon (fn _ => let val  (SYMBOL as SYMBOL1) = SYMBOL1 ()
 in ( TyCon.fromString(SYMBOL)@@A(SYMBOLleft, SYMBOLright) )
end)
 in ( LrTable.NT 5, ( result, SYMBOL1left, SYMBOL1right), rest671)
end
|  ( 29, ( ( _, ( MlyValue.TYVAR TYVAR1, (TYVARleft as TYVAR1left), (
TYVARright as TYVAR1right))) :: rest671)) => let val  result = 
MlyValue.tyvar (fn _ => let val  (TYVAR as TYVAR1) = TYVAR1 ()
 in ( TyVar.fromString(TYVAR)@@A(TYVARleft, TYVARright) )
end)
 in ( LrTable.NT 6, ( result, TYVAR1left, TYVAR1right), rest671)
end
|  ( 30, ( ( _, ( MlyValue.ALPHA ALPHA1, (ALPHAleft as ALPHA1left), (
ALPHAright as ALPHA1right))) :: rest671)) => let val  result = 
MlyValue.strid (fn _ => let val  (ALPHA as ALPHA1) = ALPHA1 ()
 in ( StrId.fromString(ALPHA)@@A(ALPHAleft, ALPHAright) )
end)
 in ( LrTable.NT 7, ( result, ALPHA1left, ALPHA1right), rest671)
end
|  ( 31, ( ( _, ( MlyValue.ALPHA ALPHA1, (ALPHAleft as ALPHA1left), (
ALPHAright as ALPHA1right))) :: rest671)) => let val  result = 
MlyValue.sigid (fn _ => let val  (ALPHA as ALPHA1) = ALPHA1 ()
 in ( SigId.fromString(ALPHA)@@A(ALPHAleft, ALPHAright) )
end)
 in ( LrTable.NT 8, ( result, ALPHA1left, ALPHA1right), rest671)
end
|  ( 32, ( ( _, ( MlyValue.ALPHA ALPHA1, (ALPHAleft as ALPHA1left), (
ALPHAright as ALPHA1right))) :: rest671)) => let val  result = 
MlyValue.funid (fn _ => let val  (ALPHA as ALPHA1) = ALPHA1 ()
 in ( FunId.fromString(ALPHA)@@A(ALPHAleft, ALPHAright) )
end)
 in ( LrTable.NT 9, ( result, ALPHA1left, ALPHA1right), rest671)
end
|  ( 33, ( ( _, ( MlyValue.longvid' longvid'1, longvid'1left, 
longvid'1right)) :: rest671)) => let val  result = MlyValue.longvid
 (fn _ => let val  (longvid' as longvid'1) = longvid'1 ()
 in ( longvid' )
end)
 in ( LrTable.NT 10, ( result, longvid'1left, longvid'1right), rest671
)
end
|  ( 34, ( ( _, ( _, (EQUALSleft as EQUALS1left), (EQUALSright as 
EQUALS1right))) :: rest671)) => let val  result = MlyValue.longvid (fn
 _ => (
 LongVId.fromId(VId.fromString "=")
                    @@A(EQUALSleft, EQUALSright) 
))
 in ( LrTable.NT 10, ( result, EQUALS1left, EQUALS1right), rest671)

end
|  ( 35, ( ( _, ( MlyValue.vid' vid'1, (vid'left as vid'1left), (
vid'right as vid'1right))) :: rest671)) => let val  result = 
MlyValue.longvid' (fn _ => let val  (vid' as vid'1) = vid'1 ()
 in ( LongVId.fromId(A.syntax vid')@@A(vid'left, vid'right) )
end)
 in ( LrTable.NT 11, ( result, vid'1left, vid'1right), rest671)
end
|  ( 36, ( ( _, ( MlyValue.LONGID LONGID1, (LONGIDleft as LONGID1left)
, (LONGIDright as LONGID1right))) :: rest671)) => let val  result = 
MlyValue.longvid' (fn _ => let val  (LONGID as LONGID1) = LONGID1 ()
 in (
 LongVId.implode(toLongId VId.fromString LONGID)
	            @@A(LONGIDleft, LONGIDright) 
)
end)
 in ( LrTable.NT 11, ( result, LONGID1left, LONGID1right), rest671)

end
|  ( 37, ( ( _, ( MlyValue.tycon tycon1, (tyconleft as tycon1left), (
tyconright as tycon1right))) :: rest671)) => let val  result = 
MlyValue.longtycon (fn _ => let val  (tycon as tycon1) = tycon1 ()
 in ( LongTyCon.fromId(A.syntax tycon)@@A(tyconleft, tyconright) )
end
)
 in ( LrTable.NT 12, ( result, tycon1left, tycon1right), rest671)
end
|  ( 38, ( ( _, ( MlyValue.LONGID LONGID1, (LONGIDleft as LONGID1left)
, (LONGIDright as LONGID1right))) :: rest671)) => let val  result = 
MlyValue.longtycon (fn _ => let val  (LONGID as LONGID1) = LONGID1 ()
 in (
 LongTyCon.implode(toLongId TyCon.fromString LONGID)
	            @@A(LONGIDleft, LONGIDright) 
)
end)
 in ( LrTable.NT 12, ( result, LONGID1left, LONGID1right), rest671)

end
|  ( 39, ( ( _, ( MlyValue.strid strid1, (stridleft as strid1left), (
stridright as strid1right))) :: rest671)) => let val  result = 
MlyValue.longstrid (fn _ => let val  (strid as strid1) = strid1 ()
 in ( LongStrId.fromId(A.syntax strid)@@A(stridleft, stridright) )
end
)
 in ( LrTable.NT 13, ( result, strid1left, strid1right), rest671)
end
|  ( 40, ( ( _, ( MlyValue.LONGID LONGID1, (LONGIDleft as LONGID1left)
, (LONGIDright as LONGID1right))) :: rest671)) => let val  result = 
MlyValue.longstrid (fn _ => let val  (LONGID as LONGID1) = LONGID1 ()
 in (
 LongStrId.implode(toLongId StrId.fromString LONGID)
	            @@A(LONGIDleft, LONGIDright) 
)
end)
 in ( LrTable.NT 13, ( result, LONGID1left, LONGID1right), rest671)

end
|  ( 41, ( ( _, ( _, OP1left, OP1right)) :: rest671)) => let val  
result = MlyValue.OP_opt (fn _ => ( SOME Op ))
 in ( LrTable.NT 14, ( result, OP1left, OP1right), rest671)
end
|  ( 42, ( rest671)) => let val  result = MlyValue.OP_opt (fn _ => (
 NONE ))
 in ( LrTable.NT 14, ( result, defaultPos, defaultPos), rest671)
end
|  ( 43, ( ( _, ( MlyValue.scon scon1, (sconleft as scon1left), (
sconright as scon1right))) :: rest671)) => let val  result = 
MlyValue.atexp (fn _ => let val  (scon as scon1) = scon1 ()
 in ( SCONAtExp(scon)@@A(sconleft, sconright) )
end)
 in ( LrTable.NT 15, ( result, scon1left, scon1right), rest671)
end
|  ( 44, ( ( _, ( MlyValue.longvid longvid1, _, (longvidright as 
longvid1right))) :: ( _, ( MlyValue.OP_opt OP_opt1, (OP_optleft as 
OP_opt1left), _)) :: rest671)) => let val  result = MlyValue.atexp (fn
 _ => let val  (OP_opt as OP_opt1) = OP_opt1 ()
 val  (longvid as longvid1) = longvid1 ()
 in ( IDAtExp(OP_opt, longvid)@@A(OP_optleft, longvidright) )
end)
 in ( LrTable.NT 15, ( result, OP_opt1left, longvid1right), rest671)

end
|  ( 45, ( ( _, ( _, _, (RBRACEright as RBRACE1right))) :: ( _, ( 
MlyValue.exprow_opt exprow_opt1, _, _)) :: ( _, ( _, (LBRACEleft as 
LBRACE1left), _)) :: rest671)) => let val  result = MlyValue.atexp (fn
 _ => let val  (exprow_opt as exprow_opt1) = exprow_opt1 ()
 in ( RECORDAtExp(exprow_opt)@@A(LBRACEleft, RBRACEright) )
end)
 in ( LrTable.NT 15, ( result, LBRACE1left, RBRACE1right), rest671)

end
|  ( 46, ( ( _, ( MlyValue.lab lab1, _, (labright as lab1right))) :: (
 _, ( _, (HASHleft as HASH1left), _)) :: rest671)) => let val  result
 = MlyValue.atexp (fn _ => let val  (lab as lab1) = lab1 ()
 in ( HASHAtExp(lab)@@A(HASHleft, labright) )
end)
 in ( LrTable.NT 15, ( result, HASH1left, lab1right), rest671)
end
|  ( 47, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( _, (
LPARleft as LPAR1left), _)) :: rest671)) => let val  result = 
MlyValue.atexp (fn _ => ( UNITAtExp@@A(LPARleft, RPARright) ))
 in ( LrTable.NT 15, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 48, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.exp_COMMA_list2 exp_COMMA_list21, _, _)) :: ( _, ( _, (
LPARleft as LPAR1left), _)) :: rest671)) => let val  result = 
MlyValue.atexp (fn _ => let val  (exp_COMMA_list2 as exp_COMMA_list21)
 = exp_COMMA_list21 ()
 in ( TUPLEAtExp(exp_COMMA_list2)@@A(LPARleft, RPARright) )
end)
 in ( LrTable.NT 15, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 49, ( ( _, ( _, _, (RBRACKright as RBRACK1right))) :: ( _, ( 
MlyValue.exp_COMMA_list0 exp_COMMA_list01, _, _)) :: ( _, ( _, (
LBRACKleft as LBRACK1left), _)) :: rest671)) => let val  result = 
MlyValue.atexp (fn _ => let val  (exp_COMMA_list0 as exp_COMMA_list01)
 = exp_COMMA_list01 ()
 in ( LISTAtExp(exp_COMMA_list0)@@A(LBRACKleft, RBRACKright) )
end)
 in ( LrTable.NT 15, ( result, LBRACK1left, RBRACK1right), rest671)

end
|  ( 50, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.exp_SEMICOLON_list2 exp_SEMICOLON_list21, _, _)) :: ( _, ( _,
 (LPARleft as LPAR1left), _)) :: rest671)) => let val  result = 
MlyValue.atexp (fn _ => let val  (exp_SEMICOLON_list2 as 
exp_SEMICOLON_list21) = exp_SEMICOLON_list21 ()
 in ( SEQAtExp(exp_SEMICOLON_list2)@@A(LPARleft, RPARright) )
end)
 in ( LrTable.NT 15, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 51, ( ( _, ( _, _, (ENDright as END1right))) :: ( _, ( 
MlyValue.popInfix popInfix1, _, _)) :: ( _, ( 
MlyValue.exp_SEMICOLON_list1 exp_SEMICOLON_list11, _, _)) :: _ :: ( _,
 ( MlyValue.dec dec1, _, _)) :: ( _, ( MlyValue.pushInfix pushInfix1,
 _, _)) :: ( _, ( _, (LETleft as LET1left), _)) :: rest671)) => let
 val  result = MlyValue.atexp (fn _ => let val  pushInfix1 = 
pushInfix1 ()
 val  (dec as dec1) = dec1 ()
 val  (exp_SEMICOLON_list1 as exp_SEMICOLON_list11) = 
exp_SEMICOLON_list11 ()
 val  popInfix1 = popInfix1 ()
 in ( LETSEQAtExp(dec, exp_SEMICOLON_list1)@@A(LETleft, ENDright) )

end)
 in ( LrTable.NT 15, ( result, LET1left, END1right), rest671)
end
|  ( 52, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.exp exp1, _, _)) :: ( _, ( _, (LPARleft as LPAR1left), _)) ::
 rest671)) => let val  result = MlyValue.atexp (fn _ => let val  (exp
 as exp1) = exp1 ()
 in ( PARAtExp(exp)@@A(LPARleft, RPARright) )
end)
 in ( LrTable.NT 15, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 53, ( ( _, ( MlyValue.exp_COMMA_list1 exp_COMMA_list11, 
exp_COMMA_list11left, exp_COMMA_list11right)) :: rest671)) => let val 
 result = MlyValue.exp_COMMA_list0 (fn _ => let val  (exp_COMMA_list1
 as exp_COMMA_list11) = exp_COMMA_list11 ()
 in ( exp_COMMA_list1 )
end)
 in ( LrTable.NT 16, ( result, exp_COMMA_list11left, 
exp_COMMA_list11right), rest671)
end
|  ( 54, ( rest671)) => let val  result = MlyValue.exp_COMMA_list0 (fn
 _ => ( [] ))
 in ( LrTable.NT 16, ( result, defaultPos, defaultPos), rest671)
end
|  ( 55, ( ( _, ( MlyValue.exp_COMMA_list1 exp_COMMA_list11, _, 
exp_COMMA_list11right)) :: _ :: ( _, ( MlyValue.exp exp1, exp1left, _)
) :: rest671)) => let val  result = MlyValue.exp_COMMA_list1 (fn _ =>
 let val  (exp as exp1) = exp1 ()
 val  (exp_COMMA_list1 as exp_COMMA_list11) = exp_COMMA_list11 ()
 in ( exp::exp_COMMA_list1 )
end)
 in ( LrTable.NT 17, ( result, exp1left, exp_COMMA_list11right), 
rest671)
end
|  ( 56, ( ( _, ( MlyValue.exp exp1, exp1left, exp1right)) :: rest671)
) => let val  result = MlyValue.exp_COMMA_list1 (fn _ => let val  (exp
 as exp1) = exp1 ()
 in ( exp::[] )
end)
 in ( LrTable.NT 17, ( result, exp1left, exp1right), rest671)
end
|  ( 57, ( ( _, ( MlyValue.exp_COMMA_list1 exp_COMMA_list11, _, 
exp_COMMA_list11right)) :: _ :: ( _, ( MlyValue.exp exp1, exp1left, _)
) :: rest671)) => let val  result = MlyValue.exp_COMMA_list2 (fn _ =>
 let val  (exp as exp1) = exp1 ()
 val  (exp_COMMA_list1 as exp_COMMA_list11) = exp_COMMA_list11 ()
 in ( exp::exp_COMMA_list1 )
end)
 in ( LrTable.NT 18, ( result, exp1left, exp_COMMA_list11right), 
rest671)
end
|  ( 58, ( ( _, ( MlyValue.exp_SEMICOLON_list1 exp_SEMICOLON_list11, _
, exp_SEMICOLON_list11right)) :: _ :: ( _, ( MlyValue.exp exp1, 
exp1left, _)) :: rest671)) => let val  result = 
MlyValue.exp_SEMICOLON_list1 (fn _ => let val  (exp as exp1) = exp1 ()
 val  (exp_SEMICOLON_list1 as exp_SEMICOLON_list11) = 
exp_SEMICOLON_list11 ()
 in ( exp::exp_SEMICOLON_list1 )
end)
 in ( LrTable.NT 19, ( result, exp1left, exp_SEMICOLON_list11right), 
rest671)
end
|  ( 59, ( ( _, ( MlyValue.exp exp1, exp1left, exp1right)) :: rest671)
) => let val  result = MlyValue.exp_SEMICOLON_list1 (fn _ => let val 
 (exp as exp1) = exp1 ()
 in ( exp::[] )
end)
 in ( LrTable.NT 19, ( result, exp1left, exp1right), rest671)
end
|  ( 60, ( ( _, ( MlyValue.exp_SEMICOLON_list2 exp_SEMICOLON_list21, _
, exp_SEMICOLON_list21right)) :: _ :: ( _, ( MlyValue.exp exp1, 
exp1left, _)) :: rest671)) => let val  result = 
MlyValue.exp_SEMICOLON_list2 (fn _ => let val  (exp as exp1) = exp1 ()
 val  (exp_SEMICOLON_list2 as exp_SEMICOLON_list21) = 
exp_SEMICOLON_list21 ()
 in ( exp::exp_SEMICOLON_list2 )
end)
 in ( LrTable.NT 20, ( result, exp1left, exp_SEMICOLON_list21right), 
rest671)
end
|  ( 61, ( ( _, ( MlyValue.exp exp2, _, exp2right)) :: _ :: ( _, ( 
MlyValue.exp exp1, exp1left, _)) :: rest671)) => let val  result = 
MlyValue.exp_SEMICOLON_list2 (fn _ => let val  exp1 = exp1 ()
 val  exp2 = exp2 ()
 in ( [exp1, exp2] )
end)
 in ( LrTable.NT 20, ( result, exp1left, exp2right), rest671)
end
|  ( 62, ( ( _, ( MlyValue.COMMA_exprow_opt COMMA_exprow_opt1, _, (
COMMA_exprow_optright as COMMA_exprow_opt1right))) :: ( _, ( 
MlyValue.exp exp1, _, _)) :: _ :: ( _, ( MlyValue.lab lab1, (lableft
 as lab1left), _)) :: rest671)) => let val  result = MlyValue.exprow
 (fn _ => let val  (lab as lab1) = lab1 ()
 val  (exp as exp1) = exp1 ()
 val  (COMMA_exprow_opt as COMMA_exprow_opt1) = COMMA_exprow_opt1 ()
 in (
 ExpRow(lab, exp, COMMA_exprow_opt)@@A(lableft, COMMA_exprow_optright)
)
end)
 in ( LrTable.NT 21, ( result, lab1left, COMMA_exprow_opt1right), 
rest671)
end
|  ( 63, ( ( _, ( MlyValue.exprow exprow1, _, exprow1right)) :: ( _, (
 _, COMMA1left, _)) :: rest671)) => let val  result = 
MlyValue.COMMA_exprow_opt (fn _ => let val  (exprow as exprow1) = 
exprow1 ()
 in ( SOME exprow )
end)
 in ( LrTable.NT 23, ( result, COMMA1left, exprow1right), rest671)
end
|  ( 64, ( rest671)) => let val  result = MlyValue.COMMA_exprow_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 23, ( result, defaultPos, defaultPos), rest671)
end
|  ( 65, ( ( _, ( MlyValue.exprow exprow1, exprow1left, exprow1right))
 :: rest671)) => let val  result = MlyValue.exprow_opt (fn _ => let
 val  (exprow as exprow1) = exprow1 ()
 in ( SOME exprow )
end)
 in ( LrTable.NT 22, ( result, exprow1left, exprow1right), rest671)

end
|  ( 66, ( rest671)) => let val  result = MlyValue.exprow_opt (fn _ =>
 ( NONE ))
 in ( LrTable.NT 22, ( result, defaultPos, defaultPos), rest671)
end
|  ( 67, ( ( _, ( MlyValue.atexp atexp1, atexp1left, atexp1right)) :: 
rest671)) => let val  result = MlyValue.appexp (fn _ => let val  (
atexp as atexp1) = atexp1 ()
 in ( atexp::[] )
end)
 in ( LrTable.NT 24, ( result, atexp1left, atexp1right), rest671)
end
|  ( 68, ( ( _, ( MlyValue.atexp atexp1, _, atexp1right)) :: ( _, ( 
MlyValue.appexp appexp1, appexp1left, _)) :: rest671)) => let val  
result = MlyValue.appexp (fn _ => let val  (appexp as appexp1) = 
appexp1 ()
 val  (atexp as atexp1) = atexp1 ()
 in ( atexp::appexp )
end)
 in ( LrTable.NT 24, ( result, appexp1left, atexp1right), rest671)
end
|  ( 69, ( ( _, ( MlyValue.appexp appexp1, appexp1left, appexp1right))
 :: rest671)) => let val  result = MlyValue.infexp (fn _ => let val  (
appexp as appexp1) = appexp1 ()
 in ( Infix.parseExp(!J, List.rev appexp) )
end)
 in ( LrTable.NT 25, ( result, appexp1left, appexp1right), rest671)

end
|  ( 70, ( ( _, ( MlyValue.infexp infexp1, infexp1left, infexp1right))
 :: rest671)) => let val  result = MlyValue.exp (fn _ => let val  (
infexp as infexp1) = infexp1 ()
 in ( infexp )
end)
 in ( LrTable.NT 26, ( result, infexp1left, infexp1right), rest671)

end
|  ( 71, ( ( _, ( MlyValue.ty ty1, _, (tyright as ty1right))) :: _ :: 
( _, ( MlyValue.exp exp1, (expleft as exp1left), _)) :: rest671)) =>
 let val  result = MlyValue.exp (fn _ => let val  (exp as exp1) = exp1
 ()
 val  (ty as ty1) = ty1 ()
 in ( COLONExp(exp, ty)@@A(expleft, tyright) )
end)
 in ( LrTable.NT 26, ( result, exp1left, ty1right), rest671)
end
|  ( 72, ( ( _, ( MlyValue.exp exp2, _, exp2right)) :: _ :: ( _, ( 
MlyValue.exp exp1, exp1left, _)) :: rest671)) => let val  result = 
MlyValue.exp (fn _ => let val  exp1 = exp1 ()
 val  exp2 = exp2 ()
 in ( ANDALSOExp(exp1, exp2)@@A(exp1left, exp2right) )
end)
 in ( LrTable.NT 26, ( result, exp1left, exp2right), rest671)
end
|  ( 73, ( ( _, ( MlyValue.exp exp2, _, exp2right)) :: _ :: ( _, ( 
MlyValue.exp exp1, exp1left, _)) :: rest671)) => let val  result = 
MlyValue.exp (fn _ => let val  exp1 = exp1 ()
 val  exp2 = exp2 ()
 in ( ORELSEExp(exp1, exp2)@@A(exp1left, exp2right) )
end)
 in ( LrTable.NT 26, ( result, exp1left, exp2right), rest671)
end
|  ( 74, ( ( _, ( MlyValue.match match1, _, (matchright as match1right
))) :: _ :: ( _, ( MlyValue.exp exp1, (expleft as exp1left), _)) :: 
rest671)) => let val  result = MlyValue.exp (fn _ => let val  (exp as 
exp1) = exp1 ()
 val  (match as match1) = match1 ()
 in ( HANDLEExp(exp, match)@@A(expleft, matchright) )
end)
 in ( LrTable.NT 26, ( result, exp1left, match1right), rest671)
end
|  ( 75, ( ( _, ( MlyValue.exp exp1, _, (expright as exp1right))) :: (
 _, ( _, (RAISEleft as RAISE1left), _)) :: rest671)) => let val  
result = MlyValue.exp (fn _ => let val  (exp as exp1) = exp1 ()
 in ( RAISEExp(exp)@@A(RAISEleft, expright) )
end)
 in ( LrTable.NT 26, ( result, RAISE1left, exp1right), rest671)
end
|  ( 76, ( ( _, ( MlyValue.exp exp3, _, exp3right)) :: _ :: ( _, ( 
MlyValue.exp exp2, _, _)) :: _ :: ( _, ( MlyValue.exp exp1, _, _)) :: 
( _, ( _, (IFleft as IF1left), _)) :: rest671)) => let val  result = 
MlyValue.exp (fn _ => let val  exp1 = exp1 ()
 val  exp2 = exp2 ()
 val  exp3 = exp3 ()
 in ( IFExp(exp1, exp2, exp3)@@A(IFleft, exp3right) )
end)
 in ( LrTable.NT 26, ( result, IF1left, exp3right), rest671)
end
|  ( 77, ( ( _, ( MlyValue.exp exp2, _, exp2right)) :: _ :: ( _, ( 
MlyValue.exp exp1, _, _)) :: ( _, ( _, (WHILEleft as WHILE1left), _))
 :: rest671)) => let val  result = MlyValue.exp (fn _ => let val  exp1
 = exp1 ()
 val  exp2 = exp2 ()
 in ( WHILEExp(exp1, exp2)@@A(WHILEleft, exp2right) )
end)
 in ( LrTable.NT 26, ( result, WHILE1left, exp2right), rest671)
end
|  ( 78, ( ( _, ( MlyValue.match match1, _, (matchright as match1right
))) :: _ :: ( _, ( MlyValue.exp exp1, _, _)) :: ( _, ( _, (CASEleft
 as CASE1left), _)) :: rest671)) => let val  result = MlyValue.exp (fn
 _ => let val  (exp as exp1) = exp1 ()
 val  (match as match1) = match1 ()
 in ( CASEExp(exp, match)@@A(CASEleft, matchright) )
end)
 in ( LrTable.NT 26, ( result, CASE1left, match1right), rest671)
end
|  ( 79, ( ( _, ( MlyValue.match match1, _, (matchright as match1right
))) :: ( _, ( _, (FNleft as FN1left), _)) :: rest671)) => let val  
result = MlyValue.exp (fn _ => let val  (match as match1) = match1 ()
 in ( FNExp(match)@@A(FNleft, matchright) )
end)
 in ( LrTable.NT 26, ( result, FN1left, match1right), rest671)
end
|  ( 80, ( ( _, ( MlyValue.BAR_match_opt BAR_match_opt1, _, (
BAR_match_optright as BAR_match_opt1right))) :: ( _, ( MlyValue.mrule 
mrule1, (mruleleft as mrule1left), _)) :: rest671)) => let val  result
 = MlyValue.match (fn _ => let val  (mrule as mrule1) = mrule1 ()
 val  (BAR_match_opt as BAR_match_opt1) = BAR_match_opt1 ()
 in ( Match(mrule, BAR_match_opt)@@A(mruleleft, BAR_match_optright) )

end)
 in ( LrTable.NT 27, ( result, mrule1left, BAR_match_opt1right), 
rest671)
end
|  ( 81, ( ( _, ( MlyValue.match match1, _, match1right)) :: ( _, ( _,
 BAR1left, _)) :: rest671)) => let val  result = 
MlyValue.BAR_match_opt (fn _ => let val  (match as match1) = match1 ()
 in ( SOME match )
end)
 in ( LrTable.NT 28, ( result, BAR1left, match1right), rest671)
end
|  ( 82, ( rest671)) => let val  result = MlyValue.BAR_match_opt (fn _
 => ( NONE ))
 in ( LrTable.NT 28, ( result, defaultPos, defaultPos), rest671)
end
|  ( 83, ( ( _, ( MlyValue.exp exp1, _, (expright as exp1right))) :: _
 :: ( _, ( MlyValue.pat pat1, (patleft as pat1left), _)) :: rest671))
 => let val  result = MlyValue.mrule (fn _ => let val  (pat as pat1) =
 pat1 ()
 val  (exp as exp1) = exp1 ()
 in ( Mrule(pat, exp)@@A(patleft, expright) )
end)
 in ( LrTable.NT 29, ( result, pat1left, exp1right), rest671)
end
|  ( 84, ( ( _, ( MlyValue.dec1 dec11, dec11left, dec11right)) :: 
rest671)) => let val  result = MlyValue.dec (fn _ => let val  (dec1
 as dec11) = dec11 ()
 in ( dec1 )
end)
 in ( LrTable.NT 30, ( result, dec11left, dec11right), rest671)
end
|  ( 85, ( rest671)) => let val  result = MlyValue.dec (fn _ => (
 EMPTYDec@@A(defaultPos, defaultPos) ))
 in ( LrTable.NT 30, ( result, defaultPos, defaultPos), rest671)
end
|  ( 86, ( ( _, ( MlyValue.dec1' dec1'1, dec1'1left, dec1'1right)) :: 
rest671)) => let val  result = MlyValue.dec1 (fn _ => let val  (dec1'
 as dec1'1) = dec1'1 ()
 in ( dec1' )
end)
 in ( LrTable.NT 31, ( result, dec1'1left, dec1'1right), rest671)
end
|  ( 87, ( ( _, ( _, _, (ENDright as END1right))) :: ( _, ( 
MlyValue.popLocalInfix popLocalInfix1, _, _)) :: ( _, ( MlyValue.dec 
dec2, _, _)) :: ( _, ( MlyValue.pushLocalInfix pushLocalInfix1, _, _))
 :: _ :: ( _, ( MlyValue.dec dec1, _, _)) :: ( _, ( MlyValue.pushInfix
 pushInfix1, _, _)) :: ( _, ( _, (LOCALleft as LOCAL1left), _)) :: 
rest671)) => let val  result = MlyValue.dec1 (fn _ => let val  
pushInfix1 = pushInfix1 ()
 val  dec1 = dec1 ()
 val  pushLocalInfix1 = pushLocalInfix1 ()
 val  dec2 = dec2 ()
 val  popLocalInfix1 = popLocalInfix1 ()
 in ( LOCALDec(dec1, dec2)@@A(LOCALleft, ENDright) )
end)
 in ( LrTable.NT 31, ( result, LOCAL1left, END1right), rest671)
end
|  ( 88, ( ( _, ( MlyValue.dec1 dec12, _, dec12right)) :: ( _, ( 
MlyValue.dec1 dec11, dec11left, _)) :: rest671)) => let val  result = 
MlyValue.dec1 (fn _ => let val  dec11 = dec11 ()
 val  dec12 = dec12 ()
 in ( SEQDec(dec11, dec12)@@A(dec11left, dec12right) )
end)
 in ( LrTable.NT 31, ( result, dec11left, dec12right), rest671)
end
|  ( 89, ( ( _, ( _, (SEMICOLONleft as SEMICOLON1left), (
SEMICOLONright as SEMICOLON1right))) :: rest671)) => let val  result =
 MlyValue.dec1 (fn _ => (
 SEQDec(
	    EMPTYDec@@A(SEMICOLONleft, SEMICOLONleft),
	    EMPTYDec@@A(SEMICOLONright, SEMICOLONright)
	  )@@A(SEMICOLONleft, SEMICOLONright) 
))
 in ( LrTable.NT 31, ( result, SEMICOLON1left, SEMICOLON1right), 
rest671)
end
|  ( 90, ( ( _, ( MlyValue.valbind valbind1, _, (valbindright as 
valbind1right))) :: ( _, ( _, (VALleft as VAL1left), _)) :: rest671))
 => let val  result = MlyValue.dec1' (fn _ => let val  (valbind as 
valbind1) = valbind1 ()
 in (
 VALDec(Seq[]@@A(defaultPos, defaultPos), valbind)
	    @@A(VALleft, valbindright) 
)
end)
 in ( LrTable.NT 32, ( result, VAL1left, valbind1right), rest671)
end
|  ( 91, ( ( _, ( MlyValue.valbind valbind1, _, (valbindright as 
valbind1right))) :: ( _, ( MlyValue.tyvarseq1 tyvarseq11, _, _)) :: (
 _, ( _, (VALleft as VAL1left), _)) :: rest671)) => let val  result = 
MlyValue.dec1' (fn _ => let val  (tyvarseq1 as tyvarseq11) = 
tyvarseq11 ()
 val  (valbind as valbind1) = valbind1 ()
 in ( VALDec(tyvarseq1, valbind)@@A(VALleft, valbindright) )
end)
 in ( LrTable.NT 32, ( result, VAL1left, valbind1right), rest671)
end
|  ( 92, ( ( _, ( MlyValue.fvalbind fvalbind1, _, (fvalbindright as 
fvalbind1right))) :: ( _, ( _, (FUNleft as FUN1left), _)) :: rest671))
 => let val  result = MlyValue.dec1' (fn _ => let val  (fvalbind as 
fvalbind1) = fvalbind1 ()
 in (
 FUNDec(Seq[]@@A(defaultPos, defaultPos), fvalbind)
	    @@A(FUNleft, fvalbindright) 
)
end)
 in ( LrTable.NT 32, ( result, FUN1left, fvalbind1right), rest671)
end
|  ( 93, ( ( _, ( MlyValue.fvalbind fvalbind1, _, (fvalbindright as 
fvalbind1right))) :: ( _, ( MlyValue.tyvarseq1 tyvarseq11, _, _)) :: (
 _, ( _, (FUNleft as FUN1left), _)) :: rest671)) => let val  result = 
MlyValue.dec1' (fn _ => let val  (tyvarseq1 as tyvarseq11) = 
tyvarseq11 ()
 val  (fvalbind as fvalbind1) = fvalbind1 ()
 in ( FUNDec(tyvarseq1, fvalbind)@@A(FUNleft, fvalbindright) )
end)
 in ( LrTable.NT 32, ( result, FUN1left, fvalbind1right), rest671)
end
|  ( 94, ( ( _, ( MlyValue.typbind typbind1, _, (typbindright as 
typbind1right))) :: ( _, ( _, (TYPEleft as TYPE1left), _)) :: rest671)
) => let val  result = MlyValue.dec1' (fn _ => let val  (typbind as 
typbind1) = typbind1 ()
 in ( TYPEDec(typbind)@@A(TYPEleft, typbindright) )
end)
 in ( LrTable.NT 32, ( result, TYPE1left, typbind1right), rest671)
end
|  ( 95, ( ( _, ( MlyValue.WITHTYPE_typbind_opt WITHTYPE_typbind_opt1,
 _, (WITHTYPE_typbind_optright as WITHTYPE_typbind_opt1right))) :: ( _
, ( MlyValue.datbind0 datbind01, _, _)) :: ( _, ( _, (DATATYPEleft as 
DATATYPE1left), _)) :: rest671)) => let val  result = MlyValue.dec1'
 (fn _ => let val  (datbind0 as datbind01) = datbind01 ()
 val  (WITHTYPE_typbind_opt as WITHTYPE_typbind_opt1) = 
WITHTYPE_typbind_opt1 ()
 in (
 DATATYPEWITHTYPEDec(datbind0, WITHTYPE_typbind_opt)
	    @@A(DATATYPEleft, WITHTYPE_typbind_optright) 
)
end)
 in ( LrTable.NT 32, ( result, DATATYPE1left, 
WITHTYPE_typbind_opt1right), rest671)
end
|  ( 96, ( ( _, ( MlyValue.WITHTYPE_typbind_opt WITHTYPE_typbind_opt1,
 _, (WITHTYPE_typbind_optright as WITHTYPE_typbind_opt1right))) :: ( _
, ( MlyValue.datbind1 datbind11, _, _)) :: ( _, ( _, (DATATYPEleft as 
DATATYPE1left), _)) :: rest671)) => let val  result = MlyValue.dec1'
 (fn _ => let val  (datbind1 as datbind11) = datbind11 ()
 val  (WITHTYPE_typbind_opt as WITHTYPE_typbind_opt1) = 
WITHTYPE_typbind_opt1 ()
 in (
 DATATYPEWITHTYPEDec(datbind1, WITHTYPE_typbind_opt)
	    @@A(DATATYPEleft, WITHTYPE_typbind_optright) 
)
end)
 in ( LrTable.NT 32, ( result, DATATYPE1left, 
WITHTYPE_typbind_opt1right), rest671)
end
|  ( 97, ( ( _, ( MlyValue.longtycon longtycon1, _, (longtyconright
 as longtycon1right))) :: _ :: _ :: ( _, ( MlyValue.tycon tycon1, _, _
)) :: ( _, ( _, (DATATYPEleft as DATATYPE1left), _)) :: rest671)) =>
 let val  result = MlyValue.dec1' (fn _ => let val  (tycon as tycon1)
 = tycon1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 in ( DATATYPE2Dec(tycon, longtycon)@@A(DATATYPEleft, longtyconright) 
)
end)
 in ( LrTable.NT 32, ( result, DATATYPE1left, longtycon1right), 
rest671)
end
|  ( 98, ( ( _, ( _, _, (ENDright as END1right))) :: ( _, ( 
MlyValue.dec dec1, _, _)) :: _ :: ( _, ( MlyValue.WITHTYPE_typbind_opt
 WITHTYPE_typbind_opt1, _, _)) :: ( _, ( MlyValue.datbind datbind1, _,
 _)) :: ( _, ( _, (ABSTYPEleft as ABSTYPE1left), _)) :: rest671)) =>
 let val  result = MlyValue.dec1' (fn _ => let val  (datbind as 
datbind1) = datbind1 ()
 val  (WITHTYPE_typbind_opt as WITHTYPE_typbind_opt1) = 
WITHTYPE_typbind_opt1 ()
 val  (dec as dec1) = dec1 ()
 in (
 ABSTYPEWITHTYPEDec(datbind, WITHTYPE_typbind_opt, dec)
	    @@A(ABSTYPEleft, ENDright) 
)
end)
 in ( LrTable.NT 32, ( result, ABSTYPE1left, END1right), rest671)
end
|  ( 99, ( ( _, ( MlyValue.exbind exbind1, _, (exbindright as 
exbind1right))) :: ( _, ( _, (EXCEPTIONleft as EXCEPTION1left), _)) ::
 rest671)) => let val  result = MlyValue.dec1' (fn _ => let val  (
exbind as exbind1) = exbind1 ()
 in ( EXCEPTIONDec(exbind)@@A(EXCEPTIONleft, exbindright) )
end)
 in ( LrTable.NT 32, ( result, EXCEPTION1left, exbind1right), rest671)

end
|  ( 100, ( ( _, ( MlyValue.longstrid_list1 longstrid_list11, _, (
longstrid_list1right as longstrid_list11right))) :: ( _, ( _, (
OPENleft as OPEN1left), _)) :: rest671)) => let val  result = 
MlyValue.dec1' (fn _ => let val  (longstrid_list1 as longstrid_list11)
 = longstrid_list11 ()
 in ( OPENDec(longstrid_list1)@@A(OPENleft, longstrid_list1right) )

end)
 in ( LrTable.NT 32, ( result, OPEN1left, longstrid_list11right), 
rest671)
end
|  ( 101, ( ( _, ( MlyValue.vid_list1 vid_list11, _, (vid_list1right
 as vid_list11right))) :: ( _, ( MlyValue.d_opt d_opt1, _, _)) :: ( _,
 ( _, (INFIXleft as INFIX1left), _)) :: rest671)) => let val  result =
 MlyValue.dec1' (fn _ => let val  (d_opt as d_opt1) = d_opt1 ()
 val  (vid_list1 as vid_list11) = vid_list11 ()
 in (
 assignInfix((Infix.LEFT, d_opt), vid_list1);
	  EMPTYDec@@A(INFIXleft, vid_list1right) 
)
end)
 in ( LrTable.NT 32, ( result, INFIX1left, vid_list11right), rest671)

end
|  ( 102, ( ( _, ( MlyValue.vid_list1 vid_list11, _, (vid_list1right
 as vid_list11right))) :: ( _, ( MlyValue.d_opt d_opt1, _, _)) :: ( _,
 ( _, (INFIXRleft as INFIXR1left), _)) :: rest671)) => let val  result
 = MlyValue.dec1' (fn _ => let val  (d_opt as d_opt1) = d_opt1 ()
 val  (vid_list1 as vid_list11) = vid_list11 ()
 in (
 assignInfix((Infix.RIGHT, d_opt), vid_list1);
	  EMPTYDec@@A(INFIXRleft, vid_list1right) 
)
end)
 in ( LrTable.NT 32, ( result, INFIXR1left, vid_list11right), rest671)

end
|  ( 103, ( ( _, ( MlyValue.vid_list1 vid_list11, _, (vid_list1right
 as vid_list11right))) :: ( _, ( _, (NONFIXleft as NONFIX1left), _))
 :: rest671)) => let val  result = MlyValue.dec1' (fn _ => let val  (
vid_list1 as vid_list11) = vid_list11 ()
 in (
 cancelInfix(vid_list1);
	  EMPTYDec@@A(NONFIXleft, vid_list1right) )

end)
 in ( LrTable.NT 32, ( result, NONFIX1left, vid_list11right), rest671)

end
|  ( 104, ( ( _, ( MlyValue.typbind typbind1, _, typbind1right)) :: (
 _, ( _, WITHTYPE1left, _)) :: rest671)) => let val  result = 
MlyValue.WITHTYPE_typbind_opt (fn _ => let val  (typbind as typbind1)
 = typbind1 ()
 in ( SOME typbind )
end)
 in ( LrTable.NT 33, ( result, WITHTYPE1left, typbind1right), rest671)

end
|  ( 105, ( rest671)) => let val  result = 
MlyValue.WITHTYPE_typbind_opt (fn _ => ( NONE ))
 in ( LrTable.NT 33, ( result, defaultPos, defaultPos), rest671)
end
|  ( 106, ( ( _, ( MlyValue.vid_list1 vid_list11, _, vid_list11right))
 :: ( _, ( MlyValue.vid vid1, vid1left, _)) :: rest671)) => let val  
result = MlyValue.vid_list1 (fn _ => let val  (vid as vid1) = vid1 ()
 val  (vid_list1 as vid_list11) = vid_list11 ()
 in ( vid::vid_list1 )
end)
 in ( LrTable.NT 34, ( result, vid1left, vid_list11right), rest671)

end
|  ( 107, ( ( _, ( MlyValue.vid vid1, vid1left, vid1right)) :: rest671
)) => let val  result = MlyValue.vid_list1 (fn _ => let val  (vid as 
vid1) = vid1 ()
 in ( vid::[] )
end)
 in ( LrTable.NT 34, ( result, vid1left, vid1right), rest671)
end
|  ( 108, ( ( _, ( MlyValue.longstrid_list1 longstrid_list11, _, 
longstrid_list11right)) :: ( _, ( MlyValue.longstrid longstrid1, 
longstrid1left, _)) :: rest671)) => let val  result = 
MlyValue.longstrid_list1 (fn _ => let val  (longstrid as longstrid1) =
 longstrid1 ()
 val  (longstrid_list1 as longstrid_list11) = longstrid_list11 ()
 in ( longstrid::longstrid_list1 )
end)
 in ( LrTable.NT 35, ( result, longstrid1left, longstrid_list11right),
 rest671)
end
|  ( 109, ( ( _, ( MlyValue.longstrid longstrid1, longstrid1left, 
longstrid1right)) :: rest671)) => let val  result = 
MlyValue.longstrid_list1 (fn _ => let val  (longstrid as longstrid1) =
 longstrid1 ()
 in ( longstrid::[] )
end)
 in ( LrTable.NT 35, ( result, longstrid1left, longstrid1right), 
rest671)
end
|  ( 110, ( ( _, ( MlyValue.d d1, d1left, d1right)) :: rest671)) =>
 let val  result = MlyValue.d_opt (fn _ => let val  (d as d1) = d1 ()
 in ( d )
end)
 in ( LrTable.NT 36, ( result, d1left, d1right), rest671)
end
|  ( 111, ( rest671)) => let val  result = MlyValue.d_opt (fn _ => (
 0 ))
 in ( LrTable.NT 36, ( result, defaultPos, defaultPos), rest671)
end
|  ( 112, ( ( _, ( MlyValue.AND_valbind_opt AND_valbind_opt1, _, (
AND_valbind_optright as AND_valbind_opt1right))) :: ( _, ( 
MlyValue.exp exp1, _, _)) :: _ :: ( _, ( MlyValue.pat pat1, (patleft
 as pat1left), _)) :: rest671)) => let val  result = MlyValue.valbind
 (fn _ => let val  (pat as pat1) = pat1 ()
 val  (exp as exp1) = exp1 ()
 val  (AND_valbind_opt as AND_valbind_opt1) = AND_valbind_opt1 ()
 in (
 PLAINValBind(pat, exp, AND_valbind_opt)
	    @@A(patleft, AND_valbind_optright) 
)
end)
 in ( LrTable.NT 37, ( result, pat1left, AND_valbind_opt1right), 
rest671)
end
|  ( 113, ( ( _, ( MlyValue.valbind valbind1, _, (valbindright as 
valbind1right))) :: ( _, ( _, (RECleft as REC1left), _)) :: rest671))
 => let val  result = MlyValue.valbind (fn _ => let val  (valbind as 
valbind1) = valbind1 ()
 in ( RECValBind(valbind)@@A(RECleft, valbindright) )
end)
 in ( LrTable.NT 37, ( result, REC1left, valbind1right), rest671)
end
|  ( 114, ( ( _, ( MlyValue.valbind valbind1, _, valbind1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_valbind_opt (fn _ => let val  (valbind as valbind1) = 
valbind1 ()
 in ( SOME valbind )
end)
 in ( LrTable.NT 38, ( result, AND1left, valbind1right), rest671)
end
|  ( 115, ( rest671)) => let val  result = MlyValue.AND_valbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 38, ( result, defaultPos, defaultPos), rest671)
end
|  ( 116, ( ( _, ( MlyValue.AND_fvalbind_opt AND_fvalbind_opt1, _, (
AND_fvalbind_optright as AND_fvalbind_opt1right))) :: ( _, ( 
MlyValue.fmatch fmatch1, (fmatchleft as fmatch1left), _)) :: rest671))
 => let val  result = MlyValue.fvalbind (fn _ => let val  (fmatch as 
fmatch1) = fmatch1 ()
 val  (AND_fvalbind_opt as AND_fvalbind_opt1) = AND_fvalbind_opt1 ()
 in (
 FvalBind(fmatch, AND_fvalbind_opt)
	    @@A(fmatchleft, AND_fvalbind_optright) 
)
end)
 in ( LrTable.NT 39, ( result, fmatch1left, AND_fvalbind_opt1right), 
rest671)
end
|  ( 117, ( ( _, ( MlyValue.fvalbind fvalbind1, _, fvalbind1right)) ::
 ( _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_fvalbind_opt (fn _ => let val  (fvalbind as fvalbind1) = 
fvalbind1 ()
 in ( SOME fvalbind )
end)
 in ( LrTable.NT 40, ( result, AND1left, fvalbind1right), rest671)
end
|  ( 118, ( rest671)) => let val  result = MlyValue.AND_fvalbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 40, ( result, defaultPos, defaultPos), rest671)
end
|  ( 119, ( ( _, ( MlyValue.BAR_fmatch_opt BAR_fmatch_opt1, _, (
BAR_fmatch_optright as BAR_fmatch_opt1right))) :: ( _, ( 
MlyValue.fmrule fmrule1, (fmruleleft as fmrule1left), _)) :: rest671))
 => let val  result = MlyValue.fmatch (fn _ => let val  (fmrule as 
fmrule1) = fmrule1 ()
 val  (BAR_fmatch_opt as BAR_fmatch_opt1) = BAR_fmatch_opt1 ()
 in (
 Fmatch(fmrule, BAR_fmatch_opt)@@A(fmruleleft, BAR_fmatch_optright) )

end)
 in ( LrTable.NT 41, ( result, fmrule1left, BAR_fmatch_opt1right), 
rest671)
end
|  ( 120, ( ( _, ( MlyValue.fmatch fmatch1, _, fmatch1right)) :: ( _, 
( _, BAR1left, _)) :: rest671)) => let val  result = 
MlyValue.BAR_fmatch_opt (fn _ => let val  (fmatch as fmatch1) = 
fmatch1 ()
 in ( SOME fmatch )
end)
 in ( LrTable.NT 42, ( result, BAR1left, fmatch1right), rest671)
end
|  ( 121, ( rest671)) => let val  result = MlyValue.BAR_fmatch_opt (fn
 _ => ( NONE ))
 in ( LrTable.NT 42, ( result, defaultPos, defaultPos), rest671)
end
|  ( 122, ( ( _, ( MlyValue.exp exp1, _, (expright as exp1right))) ::
 _ :: ( _, ( MlyValue.COLON_ty_opt COLON_ty_opt1, _, _)) :: ( _, ( 
MlyValue.atpat_list1 atpat_list11, (atpat_list1left as 
atpat_list11left), _)) :: rest671)) => let val  result = 
MlyValue.fmrule (fn _ => let val  (atpat_list1 as atpat_list11) = 
atpat_list11 ()
 val  (COLON_ty_opt as COLON_ty_opt1) = COLON_ty_opt1 ()
 val  (exp as exp1) = exp1 ()
 in (
 let
	    val (op_opt, vid, atpats) = Infix.parseFmrule(!J, atpat_list1)
	  in
	    Fmrule(op_opt, vid, atpats, COLON_ty_opt, exp)
	      @@A(atpat_list1left, expright)
	  end 
)
end)
 in ( LrTable.NT 43, ( result, atpat_list11left, exp1right), rest671)

end
|  ( 123, ( ( _, ( MlyValue.AND_typbind_opt AND_typbind_opt1, _, (
AND_typbind_optright as AND_typbind_opt1right))) :: ( _, ( MlyValue.ty
 ty1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1, _, _)) :: ( _, ( 
MlyValue.tyvarseq tyvarseq1, (tyvarseqleft as tyvarseq1left), _)) :: 
rest671)) => let val  result = MlyValue.typbind (fn _ => let val  (
tyvarseq as tyvarseq1) = tyvarseq1 ()
 val  (tycon as tycon1) = tycon1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_typbind_opt as AND_typbind_opt1) = AND_typbind_opt1 ()
 in (
 TypBind(tyvarseq, tycon, ty, AND_typbind_opt)
	    @@A(tyvarseqleft, AND_typbind_optright) 
)
end)
 in ( LrTable.NT 44, ( result, tyvarseq1left, AND_typbind_opt1right), 
rest671)
end
|  ( 124, ( ( _, ( MlyValue.typbind typbind1, _, typbind1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_typbind_opt (fn _ => let val  (typbind as typbind1) = 
typbind1 ()
 in ( SOME typbind )
end)
 in ( LrTable.NT 45, ( result, AND1left, typbind1right), rest671)
end
|  ( 125, ( rest671)) => let val  result = MlyValue.AND_typbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 45, ( result, defaultPos, defaultPos), rest671)
end
|  ( 126, ( ( _, ( MlyValue.AND_datbind_opt AND_datbind_opt1, _, (
AND_datbind_optright as AND_datbind_opt1right))) :: ( _, ( 
MlyValue.conbind conbind1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1
, _, _)) :: ( _, ( MlyValue.tyvarseq tyvarseq1, (tyvarseqleft as 
tyvarseq1left), _)) :: rest671)) => let val  result = MlyValue.datbind
 (fn _ => let val  (tyvarseq as tyvarseq1) = tyvarseq1 ()
 val  (tycon as tycon1) = tycon1 ()
 val  (conbind as conbind1) = conbind1 ()
 val  (AND_datbind_opt as AND_datbind_opt1) = AND_datbind_opt1 ()
 in (
 DatBind(tyvarseq, tycon, conbind, AND_datbind_opt)
	    @@A(tyvarseqleft, AND_datbind_optright) 
)
end)
 in ( LrTable.NT 46, ( result, tyvarseq1left, AND_datbind_opt1right), 
rest671)
end
|  ( 127, ( ( _, ( MlyValue.AND_datbind_opt AND_datbind_opt1, _, (
AND_datbind_optright as AND_datbind_opt1right))) :: ( _, ( 
MlyValue.conbind conbind1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1
, (tyconleft as tycon1left), _)) :: rest671)) => let val  result = 
MlyValue.datbind0 (fn _ => let val  (tycon as tycon1) = tycon1 ()
 val  (conbind as conbind1) = conbind1 ()
 val  (AND_datbind_opt as AND_datbind_opt1) = AND_datbind_opt1 ()
 in (
 DatBind(Seq[]@@A(tyconleft, tyconleft),
	    tycon, conbind, AND_datbind_opt)
              @@A(tyconleft, AND_datbind_optright) 
)
end)
 in ( LrTable.NT 47, ( result, tycon1left, AND_datbind_opt1right), 
rest671)
end
|  ( 128, ( ( _, ( MlyValue.AND_datbind_opt AND_datbind_opt1, _, (
AND_datbind_optright as AND_datbind_opt1right))) :: ( _, ( 
MlyValue.conbind conbind1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1
, _, _)) :: ( _, ( MlyValue.tyvarseq1 tyvarseq11, (tyvarseq1left as 
tyvarseq11left), _)) :: rest671)) => let val  result = 
MlyValue.datbind1 (fn _ => let val  (tyvarseq1 as tyvarseq11) = 
tyvarseq11 ()
 val  (tycon as tycon1) = tycon1 ()
 val  (conbind as conbind1) = conbind1 ()
 val  (AND_datbind_opt as AND_datbind_opt1) = AND_datbind_opt1 ()
 in (
 DatBind(tyvarseq1, tycon, conbind, AND_datbind_opt)
	    @@A(tyvarseq1left, AND_datbind_optright) 
)
end)
 in ( LrTable.NT 48, ( result, tyvarseq11left, AND_datbind_opt1right),
 rest671)
end
|  ( 129, ( ( _, ( MlyValue.datbind datbind1, _, datbind1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_datbind_opt (fn _ => let val  (datbind as datbind1) = 
datbind1 ()
 in ( SOME datbind )
end)
 in ( LrTable.NT 49, ( result, AND1left, datbind1right), rest671)
end
|  ( 130, ( rest671)) => let val  result = MlyValue.AND_datbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 49, ( result, defaultPos, defaultPos), rest671)
end
|  ( 131, ( ( _, ( MlyValue.BAR_conbind_opt BAR_conbind_opt1, _, (
BAR_conbind_optright as BAR_conbind_opt1right))) :: ( _, ( 
MlyValue.OF_ty_opt OF_ty_opt1, _, _)) :: ( _, ( MlyValue.vid' vid'1, _
, _)) :: ( _, ( MlyValue.OP_opt OP_opt1, (OP_optleft as OP_opt1left),
 _)) :: rest671)) => let val  result = MlyValue.conbind (fn _ => let
 val  (OP_opt as OP_opt1) = OP_opt1 ()
 val  (vid' as vid'1) = vid'1 ()
 val  (OF_ty_opt as OF_ty_opt1) = OF_ty_opt1 ()
 val  (BAR_conbind_opt as BAR_conbind_opt1) = BAR_conbind_opt1 ()
 in (
 ConBind(OP_opt, vid', OF_ty_opt, BAR_conbind_opt)
	    @@A(OP_optleft, BAR_conbind_optright) 
)
end)
 in ( LrTable.NT 50, ( result, OP_opt1left, BAR_conbind_opt1right), 
rest671)
end
|  ( 132, ( ( _, ( MlyValue.conbind conbind1, _, conbind1right)) :: (
 _, ( _, BAR1left, _)) :: rest671)) => let val  result = 
MlyValue.BAR_conbind_opt (fn _ => let val  (conbind as conbind1) = 
conbind1 ()
 in ( SOME conbind )
end)
 in ( LrTable.NT 51, ( result, BAR1left, conbind1right), rest671)
end
|  ( 133, ( rest671)) => let val  result = MlyValue.BAR_conbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 51, ( result, defaultPos, defaultPos), rest671)
end
|  ( 134, ( ( _, ( MlyValue.ty ty1, _, ty1right)) :: ( _, ( _, OF1left
, _)) :: rest671)) => let val  result = MlyValue.OF_ty_opt (fn _ =>
 let val  (ty as ty1) = ty1 ()
 in ( SOME ty )
end)
 in ( LrTable.NT 52, ( result, OF1left, ty1right), rest671)
end
|  ( 135, ( rest671)) => let val  result = MlyValue.OF_ty_opt (fn _ =>
 ( NONE ))
 in ( LrTable.NT 52, ( result, defaultPos, defaultPos), rest671)
end
|  ( 136, ( ( _, ( MlyValue.AND_exbind_opt AND_exbind_opt1, _, (
AND_exbind_optright as AND_exbind_opt1right))) :: ( _, ( 
MlyValue.OF_ty_opt OF_ty_opt1, _, _)) :: ( _, ( MlyValue.vid' vid'1, _
, _)) :: ( _, ( MlyValue.OP_opt OP_opt1, (OP_optleft as OP_opt1left),
 _)) :: rest671)) => let val  result = MlyValue.exbind (fn _ => let
 val  (OP_opt as OP_opt1) = OP_opt1 ()
 val  (vid' as vid'1) = vid'1 ()
 val  (OF_ty_opt as OF_ty_opt1) = OF_ty_opt1 ()
 val  (AND_exbind_opt as AND_exbind_opt1) = AND_exbind_opt1 ()
 in (
 NEWExBind(OP_opt, vid', OF_ty_opt, AND_exbind_opt)
	    @@A(OP_optleft, AND_exbind_optright) 
)
end)
 in ( LrTable.NT 53, ( result, OP_opt1left, AND_exbind_opt1right), 
rest671)
end
|  ( 137, ( ( _, ( MlyValue.AND_exbind_opt AND_exbind_opt1, _, (
AND_exbind_optright as AND_exbind_opt1right))) :: ( _, ( 
MlyValue.longvid longvid1, _, _)) :: ( _, ( MlyValue.OP_opt OP_opt2, _
, _)) :: _ :: ( _, ( MlyValue.vid' vid'1, _, _)) :: ( _, ( 
MlyValue.OP_opt OP_opt1, OP_opt1left, _)) :: rest671)) => let val  
result = MlyValue.exbind (fn _ => let val  OP_opt1 = OP_opt1 ()
 val  (vid' as vid'1) = vid'1 ()
 val  OP_opt2 = OP_opt2 ()
 val  (longvid as longvid1) = longvid1 ()
 val  (AND_exbind_opt as AND_exbind_opt1) = AND_exbind_opt1 ()
 in (
 EQUALExBind(OP_opt1, vid', OP_opt2, longvid, AND_exbind_opt)
	    @@A(OP_opt1left, AND_exbind_optright) 
)
end)
 in ( LrTable.NT 53, ( result, OP_opt1left, AND_exbind_opt1right), 
rest671)
end
|  ( 138, ( ( _, ( MlyValue.exbind exbind1, _, exbind1right)) :: ( _, 
( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_exbind_opt (fn _ => let val  (exbind as exbind1) = 
exbind1 ()
 in ( SOME exbind )
end)
 in ( LrTable.NT 54, ( result, AND1left, exbind1right), rest671)
end
|  ( 139, ( rest671)) => let val  result = MlyValue.AND_exbind_opt (fn
 _ => ( NONE ))
 in ( LrTable.NT 54, ( result, defaultPos, defaultPos), rest671)
end
|  ( 140, ( ( _, ( MlyValue.atpat' atpat'1, atpat'1left, atpat'1right)
) :: rest671)) => let val  result = MlyValue.atpat (fn _ => let val  (
atpat' as atpat'1) = atpat'1 ()
 in ( atpat' )
end)
 in ( LrTable.NT 55, ( result, atpat'1left, atpat'1right), rest671)

end
|  ( 141, ( ( _, ( MlyValue.longvid' longvid'1, _, (longvid'right as 
longvid'1right))) :: ( _, ( MlyValue.OP_opt OP_opt1, (OP_optleft as 
OP_opt1left), _)) :: rest671)) => let val  result = MlyValue.atpat (fn
 _ => let val  (OP_opt as OP_opt1) = OP_opt1 ()
 val  (longvid' as longvid'1) = longvid'1 ()
 in ( IDAtPat(OP_opt, longvid')@@A(OP_optleft, longvid'right) )
end)
 in ( LrTable.NT 55, ( result, OP_opt1left, longvid'1right), rest671)

end
|  ( 142, ( ( _, ( _, (UNDERBARleft as UNDERBAR1left), (UNDERBARright
 as UNDERBAR1right))) :: rest671)) => let val  result = 
MlyValue.atpat' (fn _ => (
 WILDCARDAtPat@@A(UNDERBARleft, UNDERBARright) ))
 in ( LrTable.NT 56, ( result, UNDERBAR1left, UNDERBAR1right), rest671
)
end
|  ( 143, ( ( _, ( MlyValue.scon scon1, (sconleft as scon1left), (
sconright as scon1right))) :: rest671)) => let val  result = 
MlyValue.atpat' (fn _ => let val  (scon as scon1) = scon1 ()
 in ( SCONAtPat(scon)@@A(sconleft, sconright) )
end)
 in ( LrTable.NT 56, ( result, scon1left, scon1right), rest671)
end
|  ( 144, ( ( _, ( _, _, (RBRACEright as RBRACE1right))) :: ( _, ( 
MlyValue.patrow_opt patrow_opt1, _, _)) :: ( _, ( _, (LBRACEleft as 
LBRACE1left), _)) :: rest671)) => let val  result = MlyValue.atpat'
 (fn _ => let val  (patrow_opt as patrow_opt1) = patrow_opt1 ()
 in ( RECORDAtPat(patrow_opt)@@A(LBRACEleft, RBRACEright) )
end)
 in ( LrTable.NT 56, ( result, LBRACE1left, RBRACE1right), rest671)

end
|  ( 145, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( _, (
LPARleft as LPAR1left), _)) :: rest671)) => let val  result = 
MlyValue.atpat' (fn _ => ( UNITAtPat@@A(LPARleft, RPARright) ))
 in ( LrTable.NT 56, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 146, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.pat_COMMA_list2 pat_COMMA_list21, _, _)) :: ( _, ( _, (
LPARleft as LPAR1left), _)) :: rest671)) => let val  result = 
MlyValue.atpat' (fn _ => let val  (pat_COMMA_list2 as pat_COMMA_list21
) = pat_COMMA_list21 ()
 in ( TUPLEAtPat(pat_COMMA_list2)@@A(LPARleft, RPARright) )
end)
 in ( LrTable.NT 56, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 147, ( ( _, ( _, _, (RBRACKright as RBRACK1right))) :: ( _, ( 
MlyValue.pat_COMMA_list0 pat_COMMA_list01, _, _)) :: ( _, ( _, (
LBRACKleft as LBRACK1left), _)) :: rest671)) => let val  result = 
MlyValue.atpat' (fn _ => let val  (pat_COMMA_list0 as pat_COMMA_list01
) = pat_COMMA_list01 ()
 in ( LISTAtPat(pat_COMMA_list0)@@A(LBRACKleft, RBRACKright))
end)
 in ( LrTable.NT 56, ( result, LBRACK1left, RBRACK1right), rest671)

end
|  ( 148, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.pat pat1, _, _)) :: ( _, ( _, (LPARleft as LPAR1left), _)) ::
 rest671)) => let val  result = MlyValue.atpat' (fn _ => let val  (pat
 as pat1) = pat1 ()
 in ( PARAtPat(pat)@@A(LPARleft, RPARright) )
end)
 in ( LrTable.NT 56, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 149, ( ( _, ( MlyValue.pat_COMMA_list1 pat_COMMA_list11, 
pat_COMMA_list11left, pat_COMMA_list11right)) :: rest671)) => let val 
 result = MlyValue.pat_COMMA_list0 (fn _ => let val  (pat_COMMA_list1
 as pat_COMMA_list11) = pat_COMMA_list11 ()
 in ( pat_COMMA_list1 )
end)
 in ( LrTable.NT 57, ( result, pat_COMMA_list11left, 
pat_COMMA_list11right), rest671)
end
|  ( 150, ( rest671)) => let val  result = MlyValue.pat_COMMA_list0
 (fn _ => ( [] ))
 in ( LrTable.NT 57, ( result, defaultPos, defaultPos), rest671)
end
|  ( 151, ( ( _, ( MlyValue.pat_COMMA_list1 pat_COMMA_list11, _, 
pat_COMMA_list11right)) :: _ :: ( _, ( MlyValue.pat pat1, pat1left, _)
) :: rest671)) => let val  result = MlyValue.pat_COMMA_list1 (fn _ =>
 let val  (pat as pat1) = pat1 ()
 val  (pat_COMMA_list1 as pat_COMMA_list11) = pat_COMMA_list11 ()
 in ( pat::pat_COMMA_list1 )
end)
 in ( LrTable.NT 58, ( result, pat1left, pat_COMMA_list11right), 
rest671)
end
|  ( 152, ( ( _, ( MlyValue.pat pat1, pat1left, pat1right)) :: rest671
)) => let val  result = MlyValue.pat_COMMA_list1 (fn _ => let val  (
pat as pat1) = pat1 ()
 in ( pat::[] )
end)
 in ( LrTable.NT 58, ( result, pat1left, pat1right), rest671)
end
|  ( 153, ( ( _, ( MlyValue.pat_COMMA_list1 pat_COMMA_list11, _, 
pat_COMMA_list11right)) :: _ :: ( _, ( MlyValue.pat pat1, pat1left, _)
) :: rest671)) => let val  result = MlyValue.pat_COMMA_list2 (fn _ =>
 let val  (pat as pat1) = pat1 ()
 val  (pat_COMMA_list1 as pat_COMMA_list11) = pat_COMMA_list11 ()
 in ( pat::pat_COMMA_list1 )
end)
 in ( LrTable.NT 59, ( result, pat1left, pat_COMMA_list11right), 
rest671)
end
|  ( 154, ( ( _, ( _, (DOTSleft as DOTS1left), (DOTSright as 
DOTS1right))) :: rest671)) => let val  result = MlyValue.patrow (fn _
 => ( DOTSPatRow@@A(DOTSleft, DOTSright) ))
 in ( LrTable.NT 60, ( result, DOTS1left, DOTS1right), rest671)
end
|  ( 155, ( ( _, ( MlyValue.COMMA_patrow_opt COMMA_patrow_opt1, _, (
COMMA_patrow_optright as COMMA_patrow_opt1right))) :: ( _, ( 
MlyValue.pat pat1, _, _)) :: _ :: ( _, ( MlyValue.lab lab1, (lableft
 as lab1left), _)) :: rest671)) => let val  result = MlyValue.patrow
 (fn _ => let val  (lab as lab1) = lab1 ()
 val  (pat as pat1) = pat1 ()
 val  (COMMA_patrow_opt as COMMA_patrow_opt1) = COMMA_patrow_opt1 ()
 in (
 FIELDPatRow(lab, pat, COMMA_patrow_opt)
	    @@A(lableft, COMMA_patrow_optright) 
)
end)
 in ( LrTable.NT 60, ( result, lab1left, COMMA_patrow_opt1right), 
rest671)
end
|  ( 156, ( ( _, ( MlyValue.COMMA_patrow_opt COMMA_patrow_opt1, _, (
COMMA_patrow_optright as COMMA_patrow_opt1right))) :: ( _, ( 
MlyValue.AS_pat_opt AS_pat_opt1, _, _)) :: ( _, ( 
MlyValue.COLON_ty_opt COLON_ty_opt1, _, _)) :: ( _, ( MlyValue.vid' 
vid'1, (vid'left as vid'1left), _)) :: rest671)) => let val  result = 
MlyValue.patrow (fn _ => let val  (vid' as vid'1) = vid'1 ()
 val  (COLON_ty_opt as COLON_ty_opt1) = COLON_ty_opt1 ()
 val  (AS_pat_opt as AS_pat_opt1) = AS_pat_opt1 ()
 val  (COMMA_patrow_opt as COMMA_patrow_opt1) = COMMA_patrow_opt1 ()
 in (
 IDPatRow(vid', COLON_ty_opt, AS_pat_opt, COMMA_patrow_opt)
	    @@A(vid'left, COMMA_patrow_optright)
)
end)
 in ( LrTable.NT 60, ( result, vid'1left, COMMA_patrow_opt1right), 
rest671)
end
|  ( 157, ( ( _, ( MlyValue.patrow patrow1, _, patrow1right)) :: ( _, 
( _, COMMA1left, _)) :: rest671)) => let val  result = 
MlyValue.COMMA_patrow_opt (fn _ => let val  (patrow as patrow1) = 
patrow1 ()
 in ( SOME patrow )
end)
 in ( LrTable.NT 62, ( result, COMMA1left, patrow1right), rest671)
end
|  ( 158, ( rest671)) => let val  result = MlyValue.COMMA_patrow_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 62, ( result, defaultPos, defaultPos), rest671)
end
|  ( 159, ( ( _, ( MlyValue.ty ty1, _, ty1right)) :: ( _, ( _, 
COLON1left, _)) :: rest671)) => let val  result = 
MlyValue.COLON_ty_opt (fn _ => let val  (ty as ty1) = ty1 ()
 in ( SOME ty )
end)
 in ( LrTable.NT 63, ( result, COLON1left, ty1right), rest671)
end
|  ( 160, ( rest671)) => let val  result = MlyValue.COLON_ty_opt (fn _
 => ( NONE ))
 in ( LrTable.NT 63, ( result, defaultPos, defaultPos), rest671)
end
|  ( 161, ( ( _, ( MlyValue.pat pat1, _, pat1right)) :: ( _, ( _, 
AS1left, _)) :: rest671)) => let val  result = MlyValue.AS_pat_opt (fn
 _ => let val  (pat as pat1) = pat1 ()
 in ( SOME pat )
end)
 in ( LrTable.NT 64, ( result, AS1left, pat1right), rest671)
end
|  ( 162, ( rest671)) => let val  result = MlyValue.AS_pat_opt (fn _
 => ( NONE ))
 in ( LrTable.NT 64, ( result, defaultPos, defaultPos), rest671)
end
|  ( 163, ( ( _, ( MlyValue.patrow patrow1, patrow1left, patrow1right)
) :: rest671)) => let val  result = MlyValue.patrow_opt (fn _ => let
 val  (patrow as patrow1) = patrow1 ()
 in ( SOME patrow )
end)
 in ( LrTable.NT 61, ( result, patrow1left, patrow1right), rest671)

end
|  ( 164, ( rest671)) => let val  result = MlyValue.patrow_opt (fn _
 => ( NONE ))
 in ( LrTable.NT 61, ( result, defaultPos, defaultPos), rest671)
end
|  ( 165, ( ( _, ( MlyValue.atpat atpat1, atpat1left, atpat1right)) ::
 rest671)) => let val  result = MlyValue.pat (fn _ => let val  (atpat
 as atpat1) = atpat1 ()
 in ( Infix.parsePat(!J, [atpat]) )
end)
 in ( LrTable.NT 65, ( result, atpat1left, atpat1right), rest671)
end
|  ( 166, ( ( _, ( MlyValue.atpat_list2 atpat_list21, atpat_list21left
, atpat_list21right)) :: rest671)) => let val  result = MlyValue.pat
 (fn _ => let val  (atpat_list2 as atpat_list21) = atpat_list21 ()
 in ( Infix.parsePat(!J, atpat_list2) )
end)
 in ( LrTable.NT 65, ( result, atpat_list21left, atpat_list21right), 
rest671)
end
|  ( 167, ( ( _, ( MlyValue.COLON_ty_list1 COLON_ty_list11, _, 
COLON_ty_list11right)) :: ( _, ( MlyValue.atpat' atpat'1, atpat'1left,
 _)) :: rest671)) => let val  result = MlyValue.pat (fn _ => let val 
 (atpat' as atpat'1) = atpat'1 ()
 val  (COLON_ty_list1 as COLON_ty_list11) = COLON_ty_list11 ()
 in (
 let val pat = Infix.parsePat(!J, [atpat'])
	  in typedPat(pat, COLON_ty_list1) end 
)
end)
 in ( LrTable.NT 65, ( result, atpat'1left, COLON_ty_list11right), 
rest671)
end
|  ( 168, ( ( _, ( MlyValue.COLON_ty_list1 COLON_ty_list11, _, 
COLON_ty_list11right)) :: ( _, ( MlyValue.atpat_list2 atpat_list21, 
atpat_list21left, _)) :: rest671)) => let val  result = MlyValue.pat
 (fn _ => let val  (atpat_list2 as atpat_list21) = atpat_list21 ()
 val  (COLON_ty_list1 as COLON_ty_list11) = COLON_ty_list11 ()
 in (
 let val pat = Infix.parsePat(!J, atpat_list2)
	  in typedPat(pat, COLON_ty_list1) end 
)
end)
 in ( LrTable.NT 65, ( result, atpat_list21left, COLON_ty_list11right)
, rest671)
end
|  ( 169, ( ( _, ( MlyValue.COLON_ty_list1 COLON_ty_list11, _, 
COLON_ty_list11right)) :: ( _, ( MlyValue.vid' vid'1, vid'left, 
vid'right)) :: ( _, ( MlyValue.OP_opt OP_opt1, (OP_optleft as 
OP_opt1left), _)) :: rest671)) => let val  result = MlyValue.pat (fn _
 => let val  (OP_opt as OP_opt1) = OP_opt1 ()
 val  (vid' as vid'1) = vid'1 ()
 val  (COLON_ty_list1 as COLON_ty_list11) = COLON_ty_list11 ()
 in (
 let
	    val longvid = LongVId.fromId(A.syntax vid')@@A(vid'left, vid'right)
	    val atpat   = IDAtPat(OP_opt, longvid)@@A(OP_optleft, vid'right)
	    val pat     = Infix.parsePat(!J, [atpat])
	  in typedPat(pat, COLON_ty_list1) end 
)
end)
 in ( LrTable.NT 65, ( result, OP_opt1left, COLON_ty_list11right), 
rest671)
end
|  ( 170, ( ( _, ( MlyValue.COLON_ty_list1 COLON_ty_list11, _, 
COLON_ty_list11right)) :: ( _, ( MlyValue.LONGID LONGID1, LONGIDleft, 
LONGIDright)) :: ( _, ( MlyValue.OP_opt OP_opt1, (OP_optleft as 
OP_opt1left), _)) :: rest671)) => let val  result = MlyValue.pat (fn _
 => let val  (OP_opt as OP_opt1) = OP_opt1 ()
 val  (LONGID as LONGID1) = LONGID1 ()
 val  (COLON_ty_list1 as COLON_ty_list11) = COLON_ty_list11 ()
 in (
 let
	    val longvid =
	        LongVId.implode(toLongId VId.fromString LONGID)
	          @@A(LONGIDleft, LONGIDright)
	    val atpat   = IDAtPat(OP_opt, longvid)@@A(OP_optleft, LONGIDright)
	    val pat     = Infix.parsePat(!J, [atpat])
	  in typedPat(pat, COLON_ty_list1) end 
)
end)
 in ( LrTable.NT 65, ( result, OP_opt1left, COLON_ty_list11right), 
rest671)
end
|  ( 171, ( ( _, ( MlyValue.pat pat1, _, (patright as pat1right))) ::
 _ :: ( _, ( MlyValue.COLON_ty_opt COLON_ty_opt1, _, _)) :: ( _, ( 
MlyValue.vid' vid'1, vid'left, vid'right)) :: ( _, ( MlyValue.OP_opt 
OP_opt1, (OP_optleft as OP_opt1left), _)) :: rest671)) => let val  
result = MlyValue.pat (fn _ => let val  (OP_opt as OP_opt1) = OP_opt1
 ()
 val  (vid' as vid'1) = vid'1 ()
 val  (COLON_ty_opt as COLON_ty_opt1) = COLON_ty_opt1 ()
 val  (pat as pat1) = pat1 ()
 in (
 let
	    val longvid = LongVId.fromId(A.syntax vid')@@A(vid'left, vid'right)
	    val atpat   = IDAtPat(OP_opt, longvid)@@A(OP_optleft, vid'right)
	  in
	    Infix.parsePat(!J, [atpat]);
	    ASPat(OP_opt, vid', COLON_ty_opt, pat)@@A(OP_optleft, patright)
	  end 
)
end)
 in ( LrTable.NT 65, ( result, OP_opt1left, pat1right), rest671)
end
|  ( 172, ( ( _, ( MlyValue.atpat_list1 atpat_list11, _, 
atpat_list11right)) :: ( _, ( MlyValue.atpat atpat1, atpat1left, _))
 :: rest671)) => let val  result = MlyValue.atpat_list1 (fn _ => let
 val  (atpat as atpat1) = atpat1 ()
 val  (atpat_list1 as atpat_list11) = atpat_list11 ()
 in ( atpat::atpat_list1 )
end)
 in ( LrTable.NT 66, ( result, atpat1left, atpat_list11right), rest671
)
end
|  ( 173, ( ( _, ( MlyValue.atpat atpat1, atpat1left, atpat1right)) ::
 rest671)) => let val  result = MlyValue.atpat_list1 (fn _ => let val 
 (atpat as atpat1) = atpat1 ()
 in ( atpat::[] )
end)
 in ( LrTable.NT 66, ( result, atpat1left, atpat1right), rest671)
end
|  ( 174, ( ( _, ( MlyValue.atpat_list1 atpat_list11, _, 
atpat_list11right)) :: ( _, ( MlyValue.atpat atpat1, atpat1left, _))
 :: rest671)) => let val  result = MlyValue.atpat_list2 (fn _ => let
 val  (atpat as atpat1) = atpat1 ()
 val  (atpat_list1 as atpat_list11) = atpat_list11 ()
 in ( atpat::atpat_list1 )
end)
 in ( LrTable.NT 67, ( result, atpat1left, atpat_list11right), rest671
)
end
|  ( 175, ( ( _, ( MlyValue.COLON_ty_list1 COLON_ty_list11, _, 
COLON_ty_list11right)) :: ( _, ( MlyValue.ty ty1, _, _)) :: ( _, ( _, 
COLON1left, _)) :: rest671)) => let val  result = 
MlyValue.COLON_ty_list1 (fn _ => let val  (ty as ty1) = ty1 ()
 val  (COLON_ty_list1 as COLON_ty_list11) = COLON_ty_list11 ()
 in ( ty::COLON_ty_list1 )
end)
 in ( LrTable.NT 68, ( result, COLON1left, COLON_ty_list11right), 
rest671)
end
|  ( 176, ( ( _, ( MlyValue.ty ty1, _, ty1right)) :: ( _, ( _, 
COLON1left, _)) :: rest671)) => let val  result = 
MlyValue.COLON_ty_list1 (fn _ => let val  (ty as ty1) = ty1 ()
 in ( ty::[] )
end)
 in ( LrTable.NT 68, ( result, COLON1left, ty1right), rest671)
end
|  ( 177, ( ( _, ( MlyValue.tupty tupty1, tupty1left, tupty1right)) ::
 rest671)) => let val  result = MlyValue.ty (fn _ => let val  (tupty
 as tupty1) = tupty1 ()
 in ( tupty )
end)
 in ( LrTable.NT 69, ( result, tupty1left, tupty1right), rest671)
end
|  ( 178, ( ( _, ( MlyValue.ty ty1, _, (tyright as ty1right))) :: _ ::
 ( _, ( MlyValue.tupty tupty1, (tuptyleft as tupty1left), _)) :: 
rest671)) => let val  result = MlyValue.ty (fn _ => let val  (tupty
 as tupty1) = tupty1 ()
 val  (ty as ty1) = ty1 ()
 in ( ARROWTy(tupty, ty)@@A(tuptyleft, tyright) )
end)
 in ( LrTable.NT 69, ( result, tupty1left, ty1right), rest671)
end
|  ( 179, ( ( _, ( MlyValue.ty_STAR_list ty_STAR_list1, (
ty_STAR_listleft as ty_STAR_list1left), (ty_STAR_listright as 
ty_STAR_list1right))) :: rest671)) => let val  result = MlyValue.tupty
 (fn _ => let val  (ty_STAR_list as ty_STAR_list1) = ty_STAR_list1 ()
 in ( TUPLETy(ty_STAR_list)@@A(ty_STAR_listleft, ty_STAR_listright) )

end)
 in ( LrTable.NT 70, ( result, ty_STAR_list1left, ty_STAR_list1right),
 rest671)
end
|  ( 180, ( ( _, ( MlyValue.ty_STAR_list ty_STAR_list1, _, 
ty_STAR_list1right)) :: _ :: ( _, ( MlyValue.consty consty1, 
consty1left, _)) :: rest671)) => let val  result = 
MlyValue.ty_STAR_list (fn _ => let val  (consty as consty1) = consty1
 ()
 val  (ty_STAR_list as ty_STAR_list1) = ty_STAR_list1 ()
 in ( consty::ty_STAR_list )
end)
 in ( LrTable.NT 71, ( result, consty1left, ty_STAR_list1right), 
rest671)
end
|  ( 181, ( ( _, ( MlyValue.consty consty1, consty1left, consty1right)
) :: rest671)) => let val  result = MlyValue.ty_STAR_list (fn _ => let
 val  (consty as consty1) = consty1 ()
 in ( consty::[] )
end)
 in ( LrTable.NT 71, ( result, consty1left, consty1right), rest671)

end
|  ( 182, ( ( _, ( MlyValue.atty atty1, atty1left, atty1right)) :: 
rest671)) => let val  result = MlyValue.consty (fn _ => let val  (atty
 as atty1) = atty1 ()
 in ( atty )
end)
 in ( LrTable.NT 72, ( result, atty1left, atty1right), rest671)
end
|  ( 183, ( ( _, ( MlyValue.longtycon longtycon1, _, (longtyconright
 as longtycon1right))) :: ( _, ( MlyValue.tyseq tyseq1, (tyseqleft as 
tyseq1left), _)) :: rest671)) => let val  result = MlyValue.consty (fn
 _ => let val  (tyseq as tyseq1) = tyseq1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 in ( CONTy(tyseq, longtycon)@@A(tyseqleft, longtyconright) )
end)
 in ( LrTable.NT 72, ( result, tyseq1left, longtycon1right), rest671)

end
|  ( 184, ( ( _, ( MlyValue.tyvar tyvar1, (tyvarleft as tyvar1left), (
tyvarright as tyvar1right))) :: rest671)) => let val  result = 
MlyValue.atty (fn _ => let val  (tyvar as tyvar1) = tyvar1 ()
 in ( VARTy(tyvar)@@A(tyvarleft, tyvarright) )
end)
 in ( LrTable.NT 73, ( result, tyvar1left, tyvar1right), rest671)
end
|  ( 185, ( ( _, ( _, _, (RBRACEright as RBRACE1right))) :: ( _, ( 
MlyValue.tyrow_opt tyrow_opt1, _, _)) :: ( _, ( _, (LBRACEleft as 
LBRACE1left), _)) :: rest671)) => let val  result = MlyValue.atty (fn
 _ => let val  (tyrow_opt as tyrow_opt1) = tyrow_opt1 ()
 in ( RECORDTy(tyrow_opt)@@A(LBRACEleft, RBRACEright) )
end)
 in ( LrTable.NT 73, ( result, LBRACE1left, RBRACE1right), rest671)

end
|  ( 186, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.ty ty1, _, _)) :: ( _, ( _, (LPARleft as LPAR1left), _)) :: 
rest671)) => let val  result = MlyValue.atty (fn _ => let val  (ty as 
ty1) = ty1 ()
 in ( PARTy(ty)@@A(LPARleft, RPARright) )
end)
 in ( LrTable.NT 73, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 187, ( ( _, ( MlyValue.COMMA_tyrow_opt COMMA_tyrow_opt1, _, (
COMMA_tyrow_optright as COMMA_tyrow_opt1right))) :: ( _, ( MlyValue.ty
 ty1, _, _)) :: _ :: ( _, ( MlyValue.lab lab1, (lableft as lab1left),
 _)) :: rest671)) => let val  result = MlyValue.tyrow (fn _ => let
 val  (lab as lab1) = lab1 ()
 val  (ty as ty1) = ty1 ()
 val  (COMMA_tyrow_opt as COMMA_tyrow_opt1) = COMMA_tyrow_opt1 ()
 in (
 TyRow(lab, ty, COMMA_tyrow_opt)@@A(lableft, COMMA_tyrow_optright) )

end)
 in ( LrTable.NT 74, ( result, lab1left, COMMA_tyrow_opt1right), 
rest671)
end
|  ( 188, ( ( _, ( MlyValue.tyrow tyrow1, _, tyrow1right)) :: ( _, ( _
, COMMA1left, _)) :: rest671)) => let val  result = 
MlyValue.COMMA_tyrow_opt (fn _ => let val  (tyrow as tyrow1) = tyrow1
 ()
 in ( SOME tyrow )
end)
 in ( LrTable.NT 76, ( result, COMMA1left, tyrow1right), rest671)
end
|  ( 189, ( rest671)) => let val  result = MlyValue.COMMA_tyrow_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 76, ( result, defaultPos, defaultPos), rest671)
end
|  ( 190, ( ( _, ( MlyValue.tyrow tyrow1, tyrow1left, tyrow1right)) ::
 rest671)) => let val  result = MlyValue.tyrow_opt (fn _ => let val  (
tyrow as tyrow1) = tyrow1 ()
 in ( SOME tyrow )
end)
 in ( LrTable.NT 75, ( result, tyrow1left, tyrow1right), rest671)
end
|  ( 191, ( rest671)) => let val  result = MlyValue.tyrow_opt (fn _ =>
 ( NONE ))
 in ( LrTable.NT 75, ( result, defaultPos, defaultPos), rest671)
end
|  ( 192, ( ( _, ( MlyValue.consty consty1, (constyleft as consty1left
), (constyright as consty1right))) :: rest671)) => let val  result = 
MlyValue.tyseq (fn _ => let val  (consty as consty1) = consty1 ()
 in ( Seq[consty]@@A(constyleft, constyright) )
end)
 in ( LrTable.NT 77, ( result, consty1left, consty1right), rest671)

end
|  ( 193, ( rest671)) => let val  result = MlyValue.tyseq (fn _ => (
 Seq[]@@A(defaultPos, defaultPos) ))
 in ( LrTable.NT 77, ( result, defaultPos, defaultPos), rest671)
end
|  ( 194, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.ty_COMMA_list2 ty_COMMA_list21, _, _)) :: ( _, ( _, (LPARleft
 as LPAR1left), _)) :: rest671)) => let val  result = MlyValue.tyseq
 (fn _ => let val  (ty_COMMA_list2 as ty_COMMA_list21) = 
ty_COMMA_list21 ()
 in ( Seq(ty_COMMA_list2)@@A(LPARleft, RPARright) )
end)
 in ( LrTable.NT 77, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 195, ( ( _, ( MlyValue.ty_COMMA_list2 ty_COMMA_list21, _, 
ty_COMMA_list21right)) :: _ :: ( _, ( MlyValue.ty ty1, ty1left, _)) ::
 rest671)) => let val  result = MlyValue.ty_COMMA_list2 (fn _ => let
 val  (ty as ty1) = ty1 ()
 val  (ty_COMMA_list2 as ty_COMMA_list21) = ty_COMMA_list21 ()
 in ( ty::ty_COMMA_list2 )
end)
 in ( LrTable.NT 78, ( result, ty1left, ty_COMMA_list21right), rest671
)
end
|  ( 196, ( ( _, ( MlyValue.ty ty2, _, ty2right)) :: _ :: ( _, ( 
MlyValue.ty ty1, ty1left, _)) :: rest671)) => let val  result = 
MlyValue.ty_COMMA_list2 (fn _ => let val  ty1 = ty1 ()
 val  ty2 = ty2 ()
 in ( [ty1, ty2] )
end)
 in ( LrTable.NT 78, ( result, ty1left, ty2right), rest671)
end
|  ( 197, ( ( _, ( MlyValue.tyvarseq1 tyvarseq11, tyvarseq11left, 
tyvarseq11right)) :: rest671)) => let val  result = MlyValue.tyvarseq
 (fn _ => let val  (tyvarseq1 as tyvarseq11) = tyvarseq11 ()
 in ( tyvarseq1 )
end)
 in ( LrTable.NT 79, ( result, tyvarseq11left, tyvarseq11right), 
rest671)
end
|  ( 198, ( rest671)) => let val  result = MlyValue.tyvarseq (fn _ =>
 ( Seq[]@@A(defaultPos, defaultPos) ))
 in ( LrTable.NT 79, ( result, defaultPos, defaultPos), rest671)
end
|  ( 199, ( ( _, ( MlyValue.tyvar tyvar1, (tyvarleft as tyvar1left), (
tyvarright as tyvar1right))) :: rest671)) => let val  result = 
MlyValue.tyvarseq1 (fn _ => let val  (tyvar as tyvar1) = tyvar1 ()
 in ( Seq[tyvar]@@A(tyvarleft, tyvarright) )
end)
 in ( LrTable.NT 80, ( result, tyvar1left, tyvar1right), rest671)
end
|  ( 200, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.tyvar_COMMA_list1 tyvar_COMMA_list11, _, _)) :: ( _, ( _, (
LPARleft as LPAR1left), _)) :: rest671)) => let val  result = 
MlyValue.tyvarseq1 (fn _ => let val  (tyvar_COMMA_list1 as 
tyvar_COMMA_list11) = tyvar_COMMA_list11 ()
 in ( Seq(tyvar_COMMA_list1)@@A(LPARleft, RPARright))
end)
 in ( LrTable.NT 80, ( result, LPAR1left, RPAR1right), rest671)
end
|  ( 201, ( ( _, ( MlyValue.tyvar_COMMA_list1 tyvar_COMMA_list11, _, 
tyvar_COMMA_list11right)) :: _ :: ( _, ( MlyValue.tyvar tyvar1, 
tyvar1left, _)) :: rest671)) => let val  result = 
MlyValue.tyvar_COMMA_list1 (fn _ => let val  (tyvar as tyvar1) = 
tyvar1 ()
 val  (tyvar_COMMA_list1 as tyvar_COMMA_list11) = tyvar_COMMA_list11
 ()
 in ( tyvar::tyvar_COMMA_list1 )
end)
 in ( LrTable.NT 81, ( result, tyvar1left, tyvar_COMMA_list11right), 
rest671)
end
|  ( 202, ( ( _, ( MlyValue.tyvar tyvar1, tyvar1left, tyvar1right)) ::
 rest671)) => let val  result = MlyValue.tyvar_COMMA_list1 (fn _ =>
 let val  (tyvar as tyvar1) = tyvar1 ()
 in ( tyvar::[] )
end)
 in ( LrTable.NT 81, ( result, tyvar1left, tyvar1right), rest671)
end
|  ( 203, ( ( _, ( MlyValue.strexp' strexp'1, strexp'1left, 
strexp'1right)) :: rest671)) => let val  result = MlyValue.strexp (fn
 _ => let val  (strexp' as strexp'1) = strexp'1 ()
 in ( strexp' )
end)
 in ( LrTable.NT 82, ( result, strexp'1left, strexp'1right), rest671)

end
|  ( 204, ( ( _, ( MlyValue.sigexp sigexp1, _, (sigexpright as 
sigexp1right))) :: _ :: ( _, ( MlyValue.strexp strexp1, (strexpleft
 as strexp1left), _)) :: rest671)) => let val  result = 
MlyValue.strexp (fn _ => let val  (strexp as strexp1) = strexp1 ()
 val  (sigexp as sigexp1) = sigexp1 ()
 in ( COLONStrExp(strexp, sigexp)@@A(strexpleft, sigexpright) )
end)
 in ( LrTable.NT 82, ( result, strexp1left, sigexp1right), rest671)

end
|  ( 205, ( ( _, ( MlyValue.sigexp sigexp1, _, (sigexpright as 
sigexp1right))) :: _ :: ( _, ( MlyValue.strexp strexp1, (strexpleft
 as strexp1left), _)) :: rest671)) => let val  result = 
MlyValue.strexp (fn _ => let val  (strexp as strexp1) = strexp1 ()
 val  (sigexp as sigexp1) = sigexp1 ()
 in ( SEALStrExp(strexp, sigexp)@@A(strexpleft, sigexpright))
end)
 in ( LrTable.NT 82, ( result, strexp1left, sigexp1right), rest671)

end
|  ( 206, ( ( _, ( _, _, (ENDright as END1right))) :: ( _, ( 
MlyValue.popInfix popInfix1, _, _)) :: ( _, ( MlyValue.strdec strdec1,
 _, _)) :: ( _, ( MlyValue.pushInfix pushInfix1, _, _)) :: ( _, ( _, (
STRUCTleft as STRUCT1left), _)) :: rest671)) => let val  result = 
MlyValue.strexp' (fn _ => let val  pushInfix1 = pushInfix1 ()
 val  (strdec as strdec1) = strdec1 ()
 val  popInfix1 = popInfix1 ()
 in ( STRUCTStrExp(strdec)@@A(STRUCTleft, ENDright) )
end)
 in ( LrTable.NT 83, ( result, STRUCT1left, END1right), rest671)
end
|  ( 207, ( ( _, ( MlyValue.longstrid longstrid1, (longstridleft as 
longstrid1left), (longstridright as longstrid1right))) :: rest671)) =>
 let val  result = MlyValue.strexp' (fn _ => let val  (longstrid as 
longstrid1) = longstrid1 ()
 in ( IDStrExp(longstrid)@@A(longstridleft, longstridright) )
end)
 in ( LrTable.NT 83, ( result, longstrid1left, longstrid1right), 
rest671)
end
|  ( 208, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.strexp strexp1, _, _)) :: _ :: ( _, ( MlyValue.funid funid1,
 (funidleft as funid1left), _)) :: rest671)) => let val  result = 
MlyValue.strexp' (fn _ => let val  (funid as funid1) = funid1 ()
 val  (strexp as strexp1) = strexp1 ()
 in ( APPStrExp(funid, strexp)@@A(funidleft, RPARright) )
end)
 in ( LrTable.NT 83, ( result, funid1left, RPAR1right), rest671)
end
|  ( 209, ( ( _, ( _, _, (RPARright as RPAR1right))) :: ( _, ( 
MlyValue.strdec strdec1, _, _)) :: _ :: ( _, ( MlyValue.funid funid1,
 (funidleft as funid1left), _)) :: rest671)) => let val  result = 
MlyValue.strexp' (fn _ => let val  (funid as funid1) = funid1 ()
 val  (strdec as strdec1) = strdec1 ()
 in ( APPDECStrExp(funid, strdec)@@A(funidleft, RPARright) )
end)
 in ( LrTable.NT 83, ( result, funid1left, RPAR1right), rest671)
end
|  ( 210, ( ( _, ( _, _, (ENDright as END1right))) :: ( _, ( 
MlyValue.popInfix popInfix1, _, _)) :: ( _, ( MlyValue.strexp strexp1,
 _, _)) :: _ :: ( _, ( MlyValue.strdec strdec1, _, _)) :: ( _, ( 
MlyValue.pushInfix pushInfix1, _, _)) :: ( _, ( _, (LETleft as 
LET1left), _)) :: rest671)) => let val  result = MlyValue.strexp' (fn
 _ => let val  pushInfix1 = pushInfix1 ()
 val  (strdec as strdec1) = strdec1 ()
 val  (strexp as strexp1) = strexp1 ()
 val  popInfix1 = popInfix1 ()
 in ( LETStrExp(strdec, strexp)@@A(LETleft, ENDright) )
end)
 in ( LrTable.NT 83, ( result, LET1left, END1right), rest671)
end
|  ( 211, ( ( _, ( MlyValue.strdec1 strdec11, strdec11left, 
strdec11right)) :: rest671)) => let val  result = MlyValue.strdec (fn
 _ => let val  (strdec1 as strdec11) = strdec11 ()
 in ( strdec1 )
end)
 in ( LrTable.NT 84, ( result, strdec11left, strdec11right), rest671)

end
|  ( 212, ( rest671)) => let val  result = MlyValue.strdec (fn _ => (
 EMPTYStrDec@@A(defaultPos, defaultPos) ))
 in ( LrTable.NT 84, ( result, defaultPos, defaultPos), rest671)
end
|  ( 213, ( ( _, ( MlyValue.strdec1' strdec1'1, strdec1'1left, 
strdec1'1right)) :: rest671)) => let val  result = MlyValue.strdec1
 (fn _ => let val  (strdec1' as strdec1'1) = strdec1'1 ()
 in ( strdec1' )
end)
 in ( LrTable.NT 85, ( result, strdec1'1left, strdec1'1right), rest671
)
end
|  ( 214, ( ( _, ( MlyValue.strdec1 strdec12, _, strdec12right)) :: (
 _, ( MlyValue.strdec1 strdec11, strdec11left, _)) :: rest671)) => let
 val  result = MlyValue.strdec1 (fn _ => let val  strdec11 = strdec11
 ()
 val  strdec12 = strdec12 ()
 in ( SEQStrDec(strdec11, strdec12)@@A(strdec11left, strdec12right) )

end)
 in ( LrTable.NT 85, ( result, strdec11left, strdec12right), rest671)

end
|  ( 215, ( ( _, ( _, (SEMICOLONleft as SEMICOLON1left), (
SEMICOLONright as SEMICOLON1right))) :: rest671)) => let val  result =
 MlyValue.strdec1 (fn _ => (
 SEQStrDec(
	    EMPTYStrDec@@A(SEMICOLONleft, SEMICOLONleft),
	    EMPTYStrDec@@A(SEMICOLONright, SEMICOLONright)
	  )@@A(SEMICOLONleft, SEMICOLONright) 
))
 in ( LrTable.NT 85, ( result, SEMICOLON1left, SEMICOLON1right), 
rest671)
end
|  ( 216, ( ( _, ( MlyValue.dec1' dec1'1, (dec1'left as dec1'1left), (
dec1'right as dec1'1right))) :: rest671)) => let val  result = 
MlyValue.strdec1' (fn _ => let val  (dec1' as dec1'1) = dec1'1 ()
 in ( DECStrDec(dec1')@@A(dec1'left, dec1'right) )
end)
 in ( LrTable.NT 86, ( result, dec1'1left, dec1'1right), rest671)
end
|  ( 217, ( ( _, ( MlyValue.strbind strbind1, _, (strbindright as 
strbind1right))) :: ( _, ( _, (STRUCTUREleft as STRUCTURE1left), _))
 :: rest671)) => let val  result = MlyValue.strdec1' (fn _ => let val 
 (strbind as strbind1) = strbind1 ()
 in ( STRUCTUREStrDec(strbind)@@A(STRUCTUREleft, strbindright) )
end)
 in ( LrTable.NT 86, ( result, STRUCTURE1left, strbind1right), rest671
)
end
|  ( 218, ( ( _, ( _, _, (ENDright as END1right))) :: ( _, ( 
MlyValue.popLocalInfix popLocalInfix1, _, _)) :: ( _, ( 
MlyValue.strdec strdec2, _, _)) :: ( _, ( MlyValue.pushLocalInfix 
pushLocalInfix1, _, _)) :: _ :: ( _, ( MlyValue.strdec strdec1, _, _))
 :: ( _, ( MlyValue.pushInfix pushInfix1, _, _)) :: ( _, ( _, (
LOCALleft as LOCAL1left), _)) :: rest671)) => let val  result = 
MlyValue.strdec1' (fn _ => let val  pushInfix1 = pushInfix1 ()
 val  strdec1 = strdec1 ()
 val  pushLocalInfix1 = pushLocalInfix1 ()
 val  strdec2 = strdec2 ()
 val  popLocalInfix1 = popLocalInfix1 ()
 in ( LOCALStrDec(strdec1, strdec2)@@A(LOCALleft, ENDright) )
end)
 in ( LrTable.NT 86, ( result, LOCAL1left, END1right), rest671)
end
|  ( 219, ( ( _, ( MlyValue.strexp__AND_strbind_opt 
strexp__AND_strbind_opt1, _, (strexp__AND_strbind_optright as 
strexp__AND_strbind_opt1right))) :: _ :: ( _, ( MlyValue.strid strid1,
 (stridleft as strid1left), _)) :: rest671)) => let val  result = 
MlyValue.strbind (fn _ => let val  (strid as strid1) = strid1 ()
 val  (strexp__AND_strbind_opt as strexp__AND_strbind_opt1) = 
strexp__AND_strbind_opt1 ()
 in (
 PLAINStrBind(strid,
	    #1 strexp__AND_strbind_opt, #2 strexp__AND_strbind_opt
	  )@@A(stridleft, strexp__AND_strbind_optright) 
)
end)
 in ( LrTable.NT 87, ( result, strid1left, 
strexp__AND_strbind_opt1right), rest671)
end
|  ( 220, ( ( _, ( MlyValue.strexp__AND_strbind_opt 
strexp__AND_strbind_opt1, _, (strexp__AND_strbind_optright as 
strexp__AND_strbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, _, _)) :: _ :: ( _, ( MlyValue.strid strid1, (stridleft as 
strid1left), _)) :: rest671)) => let val  result = MlyValue.strbind
 (fn _ => let val  (strid as strid1) = strid1 ()
 val  (sigexp as sigexp1) = sigexp1 ()
 val  (strexp__AND_strbind_opt as strexp__AND_strbind_opt1) = 
strexp__AND_strbind_opt1 ()
 in (
 COLONStrBind(strid, sigexp,
	    #1 strexp__AND_strbind_opt, #2 strexp__AND_strbind_opt
	  )@@A(stridleft, strexp__AND_strbind_optright) 
)
end)
 in ( LrTable.NT 87, ( result, strid1left, 
strexp__AND_strbind_opt1right), rest671)
end
|  ( 221, ( ( _, ( MlyValue.strexp__AND_strbind_opt 
strexp__AND_strbind_opt1, _, (strexp__AND_strbind_optright as 
strexp__AND_strbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, _, _)) :: _ :: ( _, ( MlyValue.strid strid1, (stridleft as 
strid1left), _)) :: rest671)) => let val  result = MlyValue.strbind
 (fn _ => let val  (strid as strid1) = strid1 ()
 val  (sigexp as sigexp1) = sigexp1 ()
 val  (strexp__AND_strbind_opt as strexp__AND_strbind_opt1) = 
strexp__AND_strbind_opt1 ()
 in (
 SEALStrBind(strid, sigexp,
	    #1 strexp__AND_strbind_opt, #2 strexp__AND_strbind_opt
	  )@@A(stridleft, strexp__AND_strbind_optright) 
)
end)
 in ( LrTable.NT 87, ( result, strid1left, 
strexp__AND_strbind_opt1right), rest671)
end
|  ( 222, ( ( _, ( MlyValue.strbind strbind1, _, strbind1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_strbind_opt (fn _ => let val  (strbind as strbind1) = 
strbind1 ()
 in ( SOME strbind )
end)
 in ( LrTable.NT 88, ( result, AND1left, strbind1right), rest671)
end
|  ( 223, ( rest671)) => let val  result = MlyValue.AND_strbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 88, ( result, defaultPos, defaultPos), rest671)
end
|  ( 224, ( ( _, ( MlyValue.AND_strbind_opt AND_strbind_opt1, _, 
AND_strbind_opt1right)) :: ( _, ( MlyValue.strexp' strexp'1, 
strexp'1left, _)) :: rest671)) => let val  result = 
MlyValue.strexp__AND_strbind_opt (fn _ => let val  (strexp' as 
strexp'1) = strexp'1 ()
 val  (AND_strbind_opt as AND_strbind_opt1) = AND_strbind_opt1 ()
 in ( (strexp', AND_strbind_opt) )
end)
 in ( LrTable.NT 89, ( result, strexp'1left, AND_strbind_opt1right), 
rest671)
end
|  ( 225, ( ( _, ( MlyValue.sigexp__AND_strbind_opt 
sigexp__AND_strbind_opt1, _, (sigexp__AND_strbind_optright as 
sigexp__AND_strbind_opt1right))) :: _ :: ( _, ( MlyValue.strexp 
strexp1, (strexpleft as strexp1left), _)) :: rest671)) => let val  
result = MlyValue.strexp__AND_strbind_opt (fn _ => let val  (strexp
 as strexp1) = strexp1 ()
 val  (sigexp__AND_strbind_opt as sigexp__AND_strbind_opt1) = 
sigexp__AND_strbind_opt1 ()
 in (
 ( COLONStrExp(strexp, #1 sigexp__AND_strbind_opt)
	      @@A(strexpleft, sigexp__AND_strbind_optright),
	    #2 sigexp__AND_strbind_opt ) 
)
end)
 in ( LrTable.NT 89, ( result, strexp1left, 
sigexp__AND_strbind_opt1right), rest671)
end
|  ( 226, ( ( _, ( MlyValue.sigexp__AND_strbind_opt 
sigexp__AND_strbind_opt1, _, (sigexp__AND_strbind_optright as 
sigexp__AND_strbind_opt1right))) :: _ :: ( _, ( MlyValue.strexp 
strexp1, (strexpleft as strexp1left), _)) :: rest671)) => let val  
result = MlyValue.strexp__AND_strbind_opt (fn _ => let val  (strexp
 as strexp1) = strexp1 ()
 val  (sigexp__AND_strbind_opt as sigexp__AND_strbind_opt1) = 
sigexp__AND_strbind_opt1 ()
 in (
 ( SEALStrExp(strexp, #1 sigexp__AND_strbind_opt)
	      @@A(strexpleft, sigexp__AND_strbind_optright),
	    #2 sigexp__AND_strbind_opt ) 
)
end)
 in ( LrTable.NT 89, ( result, strexp1left, 
sigexp__AND_strbind_opt1right), rest671)
end
|  ( 227, ( ( _, ( MlyValue.AND_strbind_opt AND_strbind_opt1, _, 
AND_strbind_opt1right)) :: ( _, ( MlyValue.sigexp' sigexp'1, 
sigexp'1left, _)) :: rest671)) => let val  result = 
MlyValue.sigexp__AND_strbind_opt (fn _ => let val  (sigexp' as 
sigexp'1) = sigexp'1 ()
 val  (AND_strbind_opt as AND_strbind_opt1) = AND_strbind_opt1 ()
 in ( (sigexp', AND_strbind_opt) )
end)
 in ( LrTable.NT 90, ( result, sigexp'1left, AND_strbind_opt1right), 
rest671)
end
|  ( 228, ( ( _, ( MlyValue.tyreadesc__AND_strbind_opt 
tyreadesc__AND_strbind_opt1, _, (tyreadesc__AND_strbind_optright as 
tyreadesc__AND_strbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, (sigexpleft as sigexp1left), _)) :: rest671)) => let val  
result = MlyValue.sigexp__AND_strbind_opt (fn _ => let val  (sigexp
 as sigexp1) = sigexp1 ()
 val  (tyreadesc__AND_strbind_opt as tyreadesc__AND_strbind_opt1) = 
tyreadesc__AND_strbind_opt1 ()
 in (
 ( WHERETYPEMULTISigExp(sigexp, #1 tyreadesc__AND_strbind_opt)
	      @@A(sigexpleft, tyreadesc__AND_strbind_optright),
	    #2 tyreadesc__AND_strbind_opt ) 
)
end)
 in ( LrTable.NT 90, ( result, sigexp1left, 
tyreadesc__AND_strbind_opt1right), rest671)
end
|  ( 229, ( ( _, ( MlyValue.AND_tyreadesc_opt__AND_strbind_opt 
AND_tyreadesc_opt__AND_strbind_opt1, _, (
AND_tyreadesc_opt__AND_strbind_optright as 
AND_tyreadesc_opt__AND_strbind_opt1right))) :: ( _, ( MlyValue.ty ty1,
 _, _)) :: _ :: ( _, ( MlyValue.longtycon longtycon1, _, _)) :: ( _, (
 MlyValue.tyvarseq tyvarseq1, _, _)) :: ( _, ( _, (TYPEleft as 
TYPE1left), _)) :: rest671)) => let val  result = 
MlyValue.tyreadesc__AND_strbind_opt (fn _ => let val  (tyvarseq as 
tyvarseq1) = tyvarseq1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_tyreadesc_opt__AND_strbind_opt as 
AND_tyreadesc_opt__AND_strbind_opt1) = 
AND_tyreadesc_opt__AND_strbind_opt1 ()
 in (
 ( TyReaDesc(tyvarseq, longtycon, ty,
	      #1 AND_tyreadesc_opt__AND_strbind_opt
	    )@@A(TYPEleft, AND_tyreadesc_opt__AND_strbind_optright),
	    #2 AND_tyreadesc_opt__AND_strbind_opt ) 
)
end)
 in ( LrTable.NT 91, ( result, TYPE1left, 
AND_tyreadesc_opt__AND_strbind_opt1right), rest671)
end
|  ( 230, ( ( _, ( MlyValue.AND_strbind_opt AND_strbind_opt1, 
AND_strbind_opt1left, AND_strbind_opt1right)) :: rest671)) => let val 
 result = MlyValue.AND_tyreadesc_opt__AND_strbind_opt (fn _ => let
 val  (AND_strbind_opt as AND_strbind_opt1) = AND_strbind_opt1 ()
 in ( (NONE, AND_strbind_opt) )
end)
 in ( LrTable.NT 92, ( result, AND_strbind_opt1left, 
AND_strbind_opt1right), rest671)
end
|  ( 231, ( ( _, ( MlyValue.tyreadesc__AND_strbind_opt 
tyreadesc__AND_strbind_opt1, _, tyreadesc__AND_strbind_opt1right)) :: 
( _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_tyreadesc_opt__AND_strbind_opt (fn _ => let val  (
tyreadesc__AND_strbind_opt as tyreadesc__AND_strbind_opt1) = 
tyreadesc__AND_strbind_opt1 ()
 in (
 (SOME(#1 tyreadesc__AND_strbind_opt), #2 tyreadesc__AND_strbind_opt) 
)
end)
 in ( LrTable.NT 92, ( result, AND1left, 
tyreadesc__AND_strbind_opt1right), rest671)
end
|  ( 232, ( ( _, ( MlyValue.sigexp' sigexp'1, sigexp'1left, 
sigexp'1right)) :: rest671)) => let val  result = MlyValue.sigexp (fn
 _ => let val  (sigexp' as sigexp'1) = sigexp'1 ()
 in ( sigexp' )
end)
 in ( LrTable.NT 93, ( result, sigexp'1left, sigexp'1right), rest671)

end
|  ( 233, ( ( _, ( MlyValue.tyreadesc tyreadesc1, _, (tyreadescright
 as tyreadesc1right))) :: _ :: ( _, ( MlyValue.sigexp sigexp1, (
sigexpleft as sigexp1left), _)) :: rest671)) => let val  result = 
MlyValue.sigexp (fn _ => let val  (sigexp as sigexp1) = sigexp1 ()
 val  (tyreadesc as tyreadesc1) = tyreadesc1 ()
 in (
 WHERETYPEMULTISigExp(sigexp, tyreadesc)
	    @@A(sigexpleft, tyreadescright) 
)
end)
 in ( LrTable.NT 93, ( result, sigexp1left, tyreadesc1right), rest671)

end
|  ( 234, ( ( _, ( _, _, (ENDright as END1right))) :: ( _, ( 
MlyValue.spec spec1, _, _)) :: ( _, ( _, (SIGleft as SIG1left), _)) ::
 rest671)) => let val  result = MlyValue.sigexp' (fn _ => let val  (
spec as spec1) = spec1 ()
 in ( SIGSigExp(spec)@@A(SIGleft, ENDright) )
end)
 in ( LrTable.NT 94, ( result, SIG1left, END1right), rest671)
end
|  ( 235, ( ( _, ( MlyValue.sigid sigid1, (sigidleft as sigid1left), (
sigidright as sigid1right))) :: rest671)) => let val  result = 
MlyValue.sigexp' (fn _ => let val  (sigid as sigid1) = sigid1 ()
 in ( IDSigExp(sigid)@@A(sigidleft, sigidright) )
end)
 in ( LrTable.NT 94, ( result, sigid1left, sigid1right), rest671)
end
|  ( 236, ( ( _, ( MlyValue.sigbind sigbind1, _, (sigbindright as 
sigbind1right))) :: ( _, ( _, (SIGNATUREleft as SIGNATURE1left), _))
 :: rest671)) => let val  result = MlyValue.sigdec (fn _ => let val  (
sigbind as sigbind1) = sigbind1 ()
 in ( SigDec(sigbind)@@A(SIGNATUREleft, sigbindright) )
end)
 in ( LrTable.NT 95, ( result, SIGNATURE1left, sigbind1right), rest671
)
end
|  ( 237, ( ( _, ( MlyValue.sigexp__AND_sigbind_opt 
sigexp__AND_sigbind_opt1, _, (sigexp__AND_sigbind_optright as 
sigexp__AND_sigbind_opt1right))) :: _ :: ( _, ( MlyValue.sigid sigid1,
 (sigidleft as sigid1left), _)) :: rest671)) => let val  result = 
MlyValue.sigbind (fn _ => let val  (sigid as sigid1) = sigid1 ()
 val  (sigexp__AND_sigbind_opt as sigexp__AND_sigbind_opt1) = 
sigexp__AND_sigbind_opt1 ()
 in (
 SigBind(sigid,
	    #1 sigexp__AND_sigbind_opt, #2 sigexp__AND_sigbind_opt
	  )@@A(sigidleft, sigexp__AND_sigbind_optright) 
)
end)
 in ( LrTable.NT 96, ( result, sigid1left, 
sigexp__AND_sigbind_opt1right), rest671)
end
|  ( 238, ( ( _, ( MlyValue.sigbind sigbind1, _, sigbind1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_sigbind_opt (fn _ => let val  (sigbind as sigbind1) = 
sigbind1 ()
 in ( SOME sigbind )
end)
 in ( LrTable.NT 97, ( result, AND1left, sigbind1right), rest671)
end
|  ( 239, ( rest671)) => let val  result = MlyValue.AND_sigbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 97, ( result, defaultPos, defaultPos), rest671)
end
|  ( 240, ( ( _, ( MlyValue.AND_sigbind_opt AND_sigbind_opt1, _, 
AND_sigbind_opt1right)) :: ( _, ( MlyValue.sigexp' sigexp'1, 
sigexp'1left, _)) :: rest671)) => let val  result = 
MlyValue.sigexp__AND_sigbind_opt (fn _ => let val  (sigexp' as 
sigexp'1) = sigexp'1 ()
 val  (AND_sigbind_opt as AND_sigbind_opt1) = AND_sigbind_opt1 ()
 in ( (sigexp', AND_sigbind_opt) )
end)
 in ( LrTable.NT 98, ( result, sigexp'1left, AND_sigbind_opt1right), 
rest671)
end
|  ( 241, ( ( _, ( MlyValue.tyreadesc__AND_sigbind_opt 
tyreadesc__AND_sigbind_opt1, _, (tyreadesc__AND_sigbind_optright as 
tyreadesc__AND_sigbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, (sigexpleft as sigexp1left), _)) :: rest671)) => let val  
result = MlyValue.sigexp__AND_sigbind_opt (fn _ => let val  (sigexp
 as sigexp1) = sigexp1 ()
 val  (tyreadesc__AND_sigbind_opt as tyreadesc__AND_sigbind_opt1) = 
tyreadesc__AND_sigbind_opt1 ()
 in (
 ( WHERETYPEMULTISigExp(sigexp, #1 tyreadesc__AND_sigbind_opt)
	      @@A(sigexpleft, tyreadesc__AND_sigbind_optright),
	    #2 tyreadesc__AND_sigbind_opt ) 
)
end)
 in ( LrTable.NT 98, ( result, sigexp1left, 
tyreadesc__AND_sigbind_opt1right), rest671)
end
|  ( 242, ( ( _, ( MlyValue.AND_tyreadesc_opt__AND_sigbind_opt 
AND_tyreadesc_opt__AND_sigbind_opt1, _, (
AND_tyreadesc_opt__AND_sigbind_optright as 
AND_tyreadesc_opt__AND_sigbind_opt1right))) :: ( _, ( MlyValue.ty ty1,
 _, _)) :: _ :: ( _, ( MlyValue.longtycon longtycon1, _, _)) :: ( _, (
 MlyValue.tyvarseq tyvarseq1, _, _)) :: ( _, ( _, (TYPEleft as 
TYPE1left), _)) :: rest671)) => let val  result = 
MlyValue.tyreadesc__AND_sigbind_opt (fn _ => let val  (tyvarseq as 
tyvarseq1) = tyvarseq1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_tyreadesc_opt__AND_sigbind_opt as 
AND_tyreadesc_opt__AND_sigbind_opt1) = 
AND_tyreadesc_opt__AND_sigbind_opt1 ()
 in (
 ( TyReaDesc(tyvarseq, longtycon, ty,
	      #1 AND_tyreadesc_opt__AND_sigbind_opt
	    )@@A(TYPEleft, AND_tyreadesc_opt__AND_sigbind_optright),
	    #2 AND_tyreadesc_opt__AND_sigbind_opt ) 
)
end)
 in ( LrTable.NT 99, ( result, TYPE1left, 
AND_tyreadesc_opt__AND_sigbind_opt1right), rest671)
end
|  ( 243, ( ( _, ( MlyValue.AND_sigbind_opt AND_sigbind_opt1, 
AND_sigbind_opt1left, AND_sigbind_opt1right)) :: rest671)) => let val 
 result = MlyValue.AND_tyreadesc_opt__AND_sigbind_opt (fn _ => let
 val  (AND_sigbind_opt as AND_sigbind_opt1) = AND_sigbind_opt1 ()
 in ( (NONE, AND_sigbind_opt) )
end)
 in ( LrTable.NT 100, ( result, AND_sigbind_opt1left, 
AND_sigbind_opt1right), rest671)
end
|  ( 244, ( ( _, ( MlyValue.tyreadesc__AND_sigbind_opt 
tyreadesc__AND_sigbind_opt1, _, tyreadesc__AND_sigbind_opt1right)) :: 
( _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_tyreadesc_opt__AND_sigbind_opt (fn _ => let val  (
tyreadesc__AND_sigbind_opt as tyreadesc__AND_sigbind_opt1) = 
tyreadesc__AND_sigbind_opt1 ()
 in (
 (SOME(#1 tyreadesc__AND_sigbind_opt), #2 tyreadesc__AND_sigbind_opt) 
)
end)
 in ( LrTable.NT 100, ( result, AND1left, 
tyreadesc__AND_sigbind_opt1right), rest671)
end
|  ( 245, ( ( _, ( MlyValue.AND_tyreadesc_opt AND_tyreadesc_opt1, _, (
AND_tyreadesc_optright as AND_tyreadesc_opt1right))) :: ( _, ( 
MlyValue.ty ty1, _, _)) :: _ :: ( _, ( MlyValue.longtycon longtycon1,
 _, _)) :: ( _, ( MlyValue.tyvarseq tyvarseq1, _, _)) :: ( _, ( _, (
TYPEleft as TYPE1left), _)) :: rest671)) => let val  result = 
MlyValue.tyreadesc (fn _ => let val  (tyvarseq as tyvarseq1) = 
tyvarseq1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_tyreadesc_opt as AND_tyreadesc_opt1) = AND_tyreadesc_opt1
 ()
 in (
 TyReaDesc(tyvarseq, longtycon, ty, AND_tyreadesc_opt)
	    @@A(TYPEleft, AND_tyreadesc_optright) 
)
end)
 in ( LrTable.NT 101, ( result, TYPE1left, AND_tyreadesc_opt1right), 
rest671)
end
|  ( 246, ( ( _, ( MlyValue.tyreadesc tyreadesc1, _, tyreadesc1right))
 :: ( _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_tyreadesc_opt (fn _ => let val  (tyreadesc as tyreadesc1)
 = tyreadesc1 ()
 in ( SOME tyreadesc )
end)
 in ( LrTable.NT 102, ( result, AND1left, tyreadesc1right), rest671)

end
|  ( 247, ( rest671)) => let val  result = MlyValue.AND_tyreadesc_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 102, ( result, defaultPos, defaultPos), rest671)
end
|  ( 248, ( ( _, ( MlyValue.spec1 spec11, spec11left, spec11right)) ::
 rest671)) => let val  result = MlyValue.spec (fn _ => let val  (spec1
 as spec11) = spec11 ()
 in ( spec1 )
end)
 in ( LrTable.NT 103, ( result, spec11left, spec11right), rest671)
end
|  ( 249, ( rest671)) => let val  result = MlyValue.spec (fn _ => (
 EMPTYSpec@@A(defaultPos, defaultPos) ))
 in ( LrTable.NT 103, ( result, defaultPos, defaultPos), rest671)
end
|  ( 250, ( ( _, ( MlyValue.spec1'' spec1''1, spec1''1left, 
spec1''1right)) :: rest671)) => let val  result = MlyValue.spec1 (fn _
 => let val  (spec1'' as spec1''1) = spec1''1 ()
 in ( spec1'' )
end)
 in ( LrTable.NT 104, ( result, spec1''1left, spec1''1right), rest671)

end
|  ( 251, ( ( _, ( MlyValue.spec1' spec1'1, _, (spec1'right as 
spec1'1right))) :: ( _, ( MlyValue.spec1 spec11, (spec1left as 
spec11left), _)) :: rest671)) => let val  result = MlyValue.spec1 (fn
 _ => let val  (spec1 as spec11) = spec11 ()
 val  (spec1' as spec1'1) = spec1'1 ()
 in ( SEQSpec(spec1, spec1')@@A(spec1left, spec1'right) )
end)
 in ( LrTable.NT 104, ( result, spec11left, spec1'1right), rest671)

end
|  ( 252, ( ( _, ( _, (SEMICOLONleft as SEMICOLON1left), (
SEMICOLONright as SEMICOLON1right))) :: rest671)) => let val  result =
 MlyValue.spec1 (fn _ => (
 SEQSpec(
	    EMPTYSpec@@A(SEMICOLONleft, SEMICOLONleft),
	    EMPTYSpec@@A(SEMICOLONright, SEMICOLONright)
	  )@@A(SEMICOLONleft, SEMICOLONright) 
))
 in ( LrTable.NT 104, ( result, SEMICOLON1left, SEMICOLON1right), 
rest671)
end
|  ( 253, ( ( _, ( MlyValue.longtycon_EQUALS_list2 
longtycon_EQUALS_list21, _, (longtycon_EQUALS_list2right as 
longtycon_EQUALS_list21right))) :: _ :: ( _, ( _, (SHARINGleft as 
SHARING1left), _)) :: rest671)) => let val  result = MlyValue.spec1
 (fn _ => let val  (longtycon_EQUALS_list2 as longtycon_EQUALS_list21)
 = longtycon_EQUALS_list21 ()
 in (
 SHARINGTYPESpec(
	    EMPTYSpec@@A(SHARINGleft, SHARINGleft), longtycon_EQUALS_list2
	  )@@A(SHARINGleft, longtycon_EQUALS_list2right) 
)
end)
 in ( LrTable.NT 104, ( result, SHARING1left, 
longtycon_EQUALS_list21right), rest671)
end
|  ( 254, ( ( _, ( MlyValue.longtycon_EQUALS_list2 
longtycon_EQUALS_list21, _, (longtycon_EQUALS_list2right as 
longtycon_EQUALS_list21right))) :: _ :: _ :: ( _, ( MlyValue.spec1 
spec11, (spec1left as spec11left), _)) :: rest671)) => let val  result
 = MlyValue.spec1 (fn _ => let val  (spec1 as spec11) = spec11 ()
 val  (longtycon_EQUALS_list2 as longtycon_EQUALS_list21) = 
longtycon_EQUALS_list21 ()
 in (
 SHARINGTYPESpec(spec1, longtycon_EQUALS_list2)
	    @@A(spec1left, longtycon_EQUALS_list2right) 
)
end)
 in ( LrTable.NT 104, ( result, spec11left, 
longtycon_EQUALS_list21right), rest671)
end
|  ( 255, ( ( _, ( MlyValue.longstrid_EQUALS_list2 
longstrid_EQUALS_list21, _, (longstrid_EQUALS_list2right as 
longstrid_EQUALS_list21right))) :: ( _, ( _, (SHARINGleft as 
SHARING1left), _)) :: rest671)) => let val  result = MlyValue.spec1
 (fn _ => let val  (longstrid_EQUALS_list2 as longstrid_EQUALS_list21)
 = longstrid_EQUALS_list21 ()
 in (
 SHARINGSpec(
	    EMPTYSpec@@A(SHARINGleft, SHARINGleft), longstrid_EQUALS_list2
	  )@@A(SHARINGleft, longstrid_EQUALS_list2right) 
)
end)
 in ( LrTable.NT 104, ( result, SHARING1left, 
longstrid_EQUALS_list21right), rest671)
end
|  ( 256, ( ( _, ( MlyValue.longstrid_EQUALS_list2 
longstrid_EQUALS_list21, _, (longstrid_EQUALS_list2right as 
longstrid_EQUALS_list21right))) :: _ :: ( _, ( MlyValue.spec1 spec11,
 (spec1left as spec11left), _)) :: rest671)) => let val  result = 
MlyValue.spec1 (fn _ => let val  (spec1 as spec11) = spec11 ()
 val  (longstrid_EQUALS_list2 as longstrid_EQUALS_list21) = 
longstrid_EQUALS_list21 ()
 in (
 SHARINGSpec(spec1, longstrid_EQUALS_list2)
	    @@A(spec1left, longstrid_EQUALS_list2right) 
)
end)
 in ( LrTable.NT 104, ( result, spec11left, 
longstrid_EQUALS_list21right), rest671)
end
|  ( 257, ( ( _, ( MlyValue.spec1'' spec1''1, spec1''1left, 
spec1''1right)) :: rest671)) => let val  result = MlyValue.spec1' (fn
 _ => let val  (spec1'' as spec1''1) = spec1''1 ()
 in ( spec1'' )
end)
 in ( LrTable.NT 105, ( result, spec1''1left, spec1''1right), rest671)

end
|  ( 258, ( ( _, ( MlyValue.spec1' spec1'2, _, spec1'2right)) :: ( _, 
( MlyValue.spec1' spec1'1, spec1'1left, _)) :: rest671)) => let val  
result = MlyValue.spec1' (fn _ => let val  spec1'1 = spec1'1 ()
 val  spec1'2 = spec1'2 ()
 in ( SEQSpec(spec1'1, spec1'2)@@A(spec1'1left, spec1'2right) )
end)
 in ( LrTable.NT 105, ( result, spec1'1left, spec1'2right), rest671)

end
|  ( 259, ( ( _, ( _, (SEMICOLONleft as SEMICOLON1left), (
SEMICOLONright as SEMICOLON1right))) :: rest671)) => let val  result =
 MlyValue.spec1' (fn _ => (
 SEQSpec(
	    EMPTYSpec@@A(SEMICOLONleft, SEMICOLONleft),
	    EMPTYSpec@@A(SEMICOLONright, SEMICOLONright)
	  )@@A(SEMICOLONleft, SEMICOLONright) 
))
 in ( LrTable.NT 105, ( result, SEMICOLON1left, SEMICOLON1right), 
rest671)
end
|  ( 260, ( ( _, ( MlyValue.valdesc valdesc1, _, (valdescright as 
valdesc1right))) :: ( _, ( _, (VALleft as VAL1left), _)) :: rest671))
 => let val  result = MlyValue.spec1'' (fn _ => let val  (valdesc as 
valdesc1) = valdesc1 ()
 in ( VALSpec(valdesc)@@A(VALleft, valdescright) )
end)
 in ( LrTable.NT 106, ( result, VAL1left, valdesc1right), rest671)
end
|  ( 261, ( ( _, ( MlyValue.typdesc typdesc1, _, (typdescright as 
typdesc1right))) :: ( _, ( _, (TYPEleft as TYPE1left), _)) :: rest671)
) => let val  result = MlyValue.spec1'' (fn _ => let val  (typdesc as 
typdesc1) = typdesc1 ()
 in ( TYPESpec(typdesc)@@A(TYPEleft, typdescright) )
end)
 in ( LrTable.NT 106, ( result, TYPE1left, typdesc1right), rest671)

end
|  ( 262, ( ( _, ( MlyValue.typdesc typdesc1, _, (typdescright as 
typdesc1right))) :: ( _, ( _, (EQTYPEleft as EQTYPE1left), _)) :: 
rest671)) => let val  result = MlyValue.spec1'' (fn _ => let val  (
typdesc as typdesc1) = typdesc1 ()
 in ( EQTYPESpec(typdesc)@@A(EQTYPEleft, typdescright) )
end)
 in ( LrTable.NT 106, ( result, EQTYPE1left, typdesc1right), rest671)

end
|  ( 263, ( ( _, ( MlyValue.syndesc syndesc1, _, (syndescright as 
syndesc1right))) :: ( _, ( _, (TYPEleft as TYPE1left), _)) :: rest671)
) => let val  result = MlyValue.spec1'' (fn _ => let val  (syndesc as 
syndesc1) = syndesc1 ()
 in ( SYNSpec(syndesc)@@A(TYPEleft, syndescright) )
end)
 in ( LrTable.NT 106, ( result, TYPE1left, syndesc1right), rest671)

end
|  ( 264, ( ( _, ( MlyValue.datdesc0 datdesc01, _, (datdesc0right as 
datdesc01right))) :: ( _, ( _, (DATATYPEleft as DATATYPE1left), _)) ::
 rest671)) => let val  result = MlyValue.spec1'' (fn _ => let val  (
datdesc0 as datdesc01) = datdesc01 ()
 in ( DATATYPESpec(datdesc0)@@A(DATATYPEleft, datdesc0right) )
end)
 in ( LrTable.NT 106, ( result, DATATYPE1left, datdesc01right), 
rest671)
end
|  ( 265, ( ( _, ( MlyValue.datdesc1 datdesc11, _, (datdesc1right as 
datdesc11right))) :: ( _, ( _, (DATATYPEleft as DATATYPE1left), _)) ::
 rest671)) => let val  result = MlyValue.spec1'' (fn _ => let val  (
datdesc1 as datdesc11) = datdesc11 ()
 in ( DATATYPESpec(datdesc1)@@A(DATATYPEleft, datdesc1right) )
end)
 in ( LrTable.NT 106, ( result, DATATYPE1left, datdesc11right), 
rest671)
end
|  ( 266, ( ( _, ( MlyValue.longtycon longtycon1, _, (longtyconright
 as longtycon1right))) :: _ :: _ :: ( _, ( MlyValue.tycon tycon1, _, _
)) :: ( _, ( _, (DATATYPEleft as DATATYPE1left), _)) :: rest671)) =>
 let val  result = MlyValue.spec1'' (fn _ => let val  (tycon as tycon1
) = tycon1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 in (
 DATATYPE2Spec(tycon, longtycon)@@A(DATATYPEleft, longtyconright) )

end)
 in ( LrTable.NT 106, ( result, DATATYPE1left, longtycon1right), 
rest671)
end
|  ( 267, ( ( _, ( MlyValue.exdesc exdesc1, _, (exdescright as 
exdesc1right))) :: ( _, ( _, (EXCEPTIONleft as EXCEPTION1left), _)) ::
 rest671)) => let val  result = MlyValue.spec1'' (fn _ => let val  (
exdesc as exdesc1) = exdesc1 ()
 in ( EXCEPTIONSpec(exdesc)@@A(EXCEPTIONleft, exdescright) )
end)
 in ( LrTable.NT 106, ( result, EXCEPTION1left, exdesc1right), rest671
)
end
|  ( 268, ( ( _, ( MlyValue.strdesc strdesc1, _, (strdescright as 
strdesc1right))) :: ( _, ( _, (STRUCTUREleft as STRUCTURE1left), _))
 :: rest671)) => let val  result = MlyValue.spec1'' (fn _ => let val 
 (strdesc as strdesc1) = strdesc1 ()
 in ( STRUCTURESpec(strdesc)@@A(STRUCTUREleft, strdescright))
end)
 in ( LrTable.NT 106, ( result, STRUCTURE1left, strdesc1right), 
rest671)
end
|  ( 269, ( ( _, ( MlyValue.sigexp sigexp1, _, (sigexpright as 
sigexp1right))) :: ( _, ( _, (INCLUDEleft as INCLUDE1left), _)) :: 
rest671)) => let val  result = MlyValue.spec1'' (fn _ => let val  (
sigexp as sigexp1) = sigexp1 ()
 in ( INCLUDESpec(sigexp)@@A(INCLUDEleft, sigexpright) )
end)
 in ( LrTable.NT 106, ( result, INCLUDE1left, sigexp1right), rest671)

end
|  ( 270, ( ( _, ( MlyValue.sigid_list2 sigid_list21, _, (
sigid_list2right as sigid_list21right))) :: ( _, ( _, (INCLUDEleft as 
INCLUDE1left), _)) :: rest671)) => let val  result = MlyValue.spec1''
 (fn _ => let val  (sigid_list2 as sigid_list21) = sigid_list21 ()
 in ( INCLUDEMULTISpec(sigid_list2)@@A(INCLUDEleft, sigid_list2right) 
)
end)
 in ( LrTable.NT 106, ( result, INCLUDE1left, sigid_list21right), 
rest671)
end
|  ( 271, ( ( _, ( MlyValue.sigid_list2 sigid_list21, _, 
sigid_list21right)) :: ( _, ( MlyValue.sigid sigid1, sigid1left, _))
 :: rest671)) => let val  result = MlyValue.sigid_list2 (fn _ => let
 val  (sigid as sigid1) = sigid1 ()
 val  (sigid_list2 as sigid_list21) = sigid_list21 ()
 in ( sigid::sigid_list2 )
end)
 in ( LrTable.NT 107, ( result, sigid1left, sigid_list21right), 
rest671)
end
|  ( 272, ( ( _, ( MlyValue.sigid sigid2, _, sigid2right)) :: ( _, ( 
MlyValue.sigid sigid1, sigid1left, _)) :: rest671)) => let val  result
 = MlyValue.sigid_list2 (fn _ => let val  sigid1 = sigid1 ()
 val  sigid2 = sigid2 ()
 in ( sigid1::sigid2::[] )
end)
 in ( LrTable.NT 107, ( result, sigid1left, sigid2right), rest671)
end
|  ( 273, ( ( _, ( MlyValue.longtycon_EQUALS_list1 
longtycon_EQUALS_list11, _, longtycon_EQUALS_list11right)) :: _ :: ( _
, ( MlyValue.longtycon longtycon1, longtycon1left, _)) :: rest671)) =>
 let val  result = MlyValue.longtycon_EQUALS_list1 (fn _ => let val  (
longtycon as longtycon1) = longtycon1 ()
 val  (longtycon_EQUALS_list1 as longtycon_EQUALS_list11) = 
longtycon_EQUALS_list11 ()
 in ( longtycon::longtycon_EQUALS_list1 )
end)
 in ( LrTable.NT 108, ( result, longtycon1left, 
longtycon_EQUALS_list11right), rest671)
end
|  ( 274, ( ( _, ( MlyValue.longtycon longtycon1, longtycon1left, 
longtycon1right)) :: rest671)) => let val  result = 
MlyValue.longtycon_EQUALS_list1 (fn _ => let val  (longtycon as 
longtycon1) = longtycon1 ()
 in ( longtycon::[] )
end)
 in ( LrTable.NT 108, ( result, longtycon1left, longtycon1right), 
rest671)
end
|  ( 275, ( ( _, ( MlyValue.longtycon_EQUALS_list1 
longtycon_EQUALS_list11, _, longtycon_EQUALS_list11right)) :: _ :: ( _
, ( MlyValue.longtycon longtycon1, longtycon1left, _)) :: rest671)) =>
 let val  result = MlyValue.longtycon_EQUALS_list2 (fn _ => let val  (
longtycon as longtycon1) = longtycon1 ()
 val  (longtycon_EQUALS_list1 as longtycon_EQUALS_list11) = 
longtycon_EQUALS_list11 ()
 in ( longtycon::longtycon_EQUALS_list1 )
end)
 in ( LrTable.NT 109, ( result, longtycon1left, 
longtycon_EQUALS_list11right), rest671)
end
|  ( 276, ( ( _, ( MlyValue.longstrid_EQUALS_list1 
longstrid_EQUALS_list11, _, longstrid_EQUALS_list11right)) :: _ :: ( _
, ( MlyValue.longstrid longstrid1, longstrid1left, _)) :: rest671)) =>
 let val  result = MlyValue.longstrid_EQUALS_list1 (fn _ => let val  (
longstrid as longstrid1) = longstrid1 ()
 val  (longstrid_EQUALS_list1 as longstrid_EQUALS_list11) = 
longstrid_EQUALS_list11 ()
 in ( longstrid::longstrid_EQUALS_list1 )
end)
 in ( LrTable.NT 110, ( result, longstrid1left, 
longstrid_EQUALS_list11right), rest671)
end
|  ( 277, ( ( _, ( MlyValue.longstrid longstrid1, longstrid1left, 
longstrid1right)) :: rest671)) => let val  result = 
MlyValue.longstrid_EQUALS_list1 (fn _ => let val  (longstrid as 
longstrid1) = longstrid1 ()
 in ( longstrid::[] )
end)
 in ( LrTable.NT 110, ( result, longstrid1left, longstrid1right), 
rest671)
end
|  ( 278, ( ( _, ( MlyValue.longstrid_EQUALS_list1 
longstrid_EQUALS_list11, _, longstrid_EQUALS_list11right)) :: _ :: ( _
, ( MlyValue.longstrid longstrid1, longstrid1left, _)) :: rest671)) =>
 let val  result = MlyValue.longstrid_EQUALS_list2 (fn _ => let val  (
longstrid as longstrid1) = longstrid1 ()
 val  (longstrid_EQUALS_list1 as longstrid_EQUALS_list11) = 
longstrid_EQUALS_list11 ()
 in ( longstrid::longstrid_EQUALS_list1 )
end)
 in ( LrTable.NT 111, ( result, longstrid1left, 
longstrid_EQUALS_list11right), rest671)
end
|  ( 279, ( ( _, ( MlyValue.AND_valdesc_opt AND_valdesc_opt1, _, (
AND_valdesc_optright as AND_valdesc_opt1right))) :: ( _, ( MlyValue.ty
 ty1, _, _)) :: _ :: ( _, ( MlyValue.vid' vid'1, (vid'left as 
vid'1left), _)) :: rest671)) => let val  result = MlyValue.valdesc (fn
 _ => let val  (vid' as vid'1) = vid'1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_valdesc_opt as AND_valdesc_opt1) = AND_valdesc_opt1 ()
 in (
 ValDesc(vid', ty, AND_valdesc_opt)
	    @@A(vid'left, AND_valdesc_optright) 
)
end)
 in ( LrTable.NT 112, ( result, vid'1left, AND_valdesc_opt1right), 
rest671)
end
|  ( 280, ( ( _, ( MlyValue.valdesc valdesc1, _, valdesc1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_valdesc_opt (fn _ => let val  (valdesc as valdesc1) = 
valdesc1 ()
 in ( SOME valdesc )
end)
 in ( LrTable.NT 113, ( result, AND1left, valdesc1right), rest671)
end
|  ( 281, ( rest671)) => let val  result = MlyValue.AND_valdesc_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 113, ( result, defaultPos, defaultPos), rest671)
end
|  ( 282, ( ( _, ( MlyValue.AND_typdesc_opt AND_typdesc_opt1, _, (
AND_typdesc_optright as AND_typdesc_opt1right))) :: ( _, ( 
MlyValue.tycon tycon1, _, _)) :: ( _, ( MlyValue.tyvarseq tyvarseq1, (
tyvarseqleft as tyvarseq1left), _)) :: rest671)) => let val  result = 
MlyValue.typdesc (fn _ => let val  (tyvarseq as tyvarseq1) = tyvarseq1
 ()
 val  (tycon as tycon1) = tycon1 ()
 val  (AND_typdesc_opt as AND_typdesc_opt1) = AND_typdesc_opt1 ()
 in (
 TypDesc(tyvarseq, tycon, AND_typdesc_opt)
	    @@A(tyvarseqleft, AND_typdesc_optright) 
)
end)
 in ( LrTable.NT 114, ( result, tyvarseq1left, AND_typdesc_opt1right),
 rest671)
end
|  ( 283, ( ( _, ( MlyValue.typdesc typdesc1, _, typdesc1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_typdesc_opt (fn _ => let val  (typdesc as typdesc1) = 
typdesc1 ()
 in ( SOME typdesc )
end)
 in ( LrTable.NT 115, ( result, AND1left, typdesc1right), rest671)
end
|  ( 284, ( rest671)) => let val  result = MlyValue.AND_typdesc_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 115, ( result, defaultPos, defaultPos), rest671)
end
|  ( 285, ( ( _, ( MlyValue.AND_syndesc_opt AND_syndesc_opt1, _, (
AND_syndesc_optright as AND_syndesc_opt1right))) :: ( _, ( MlyValue.ty
 ty1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1, _, _)) :: ( _, ( 
MlyValue.tyvarseq tyvarseq1, (tyvarseqleft as tyvarseq1left), _)) :: 
rest671)) => let val  result = MlyValue.syndesc (fn _ => let val  (
tyvarseq as tyvarseq1) = tyvarseq1 ()
 val  (tycon as tycon1) = tycon1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_syndesc_opt as AND_syndesc_opt1) = AND_syndesc_opt1 ()
 in (
 SynDesc(tyvarseq, tycon, ty, AND_syndesc_opt)
	    @@A(tyvarseqleft, AND_syndesc_optright) 
)
end)
 in ( LrTable.NT 116, ( result, tyvarseq1left, AND_syndesc_opt1right),
 rest671)
end
|  ( 286, ( ( _, ( MlyValue.syndesc syndesc1, _, syndesc1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_syndesc_opt (fn _ => let val  (syndesc as syndesc1) = 
syndesc1 ()
 in ( SOME syndesc )
end)
 in ( LrTable.NT 117, ( result, AND1left, syndesc1right), rest671)
end
|  ( 287, ( rest671)) => let val  result = MlyValue.AND_syndesc_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 117, ( result, defaultPos, defaultPos), rest671)
end
|  ( 288, ( ( _, ( MlyValue.AND_datdesc_opt AND_datdesc_opt1, _, (
AND_datdesc_optright as AND_datdesc_opt1right))) :: ( _, ( 
MlyValue.condesc condesc1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1
, _, _)) :: ( _, ( MlyValue.tyvarseq tyvarseq1, (tyvarseqleft as 
tyvarseq1left), _)) :: rest671)) => let val  result = MlyValue.datdesc
 (fn _ => let val  (tyvarseq as tyvarseq1) = tyvarseq1 ()
 val  (tycon as tycon1) = tycon1 ()
 val  (condesc as condesc1) = condesc1 ()
 val  (AND_datdesc_opt as AND_datdesc_opt1) = AND_datdesc_opt1 ()
 in (
 DatDesc(tyvarseq, tycon, condesc, AND_datdesc_opt)
	    @@A(tyvarseqleft, AND_datdesc_optright) 
)
end)
 in ( LrTable.NT 118, ( result, tyvarseq1left, AND_datdesc_opt1right),
 rest671)
end
|  ( 289, ( ( _, ( MlyValue.AND_datdesc_opt AND_datdesc_opt1, _, (
AND_datdesc_optright as AND_datdesc_opt1right))) :: ( _, ( 
MlyValue.condesc condesc1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1
, (tyconleft as tycon1left), _)) :: rest671)) => let val  result = 
MlyValue.datdesc0 (fn _ => let val  (tycon as tycon1) = tycon1 ()
 val  (condesc as condesc1) = condesc1 ()
 val  (AND_datdesc_opt as AND_datdesc_opt1) = AND_datdesc_opt1 ()
 in (
 DatDesc(Seq[]@@A(tyconleft, tyconleft),
	    tycon, condesc, AND_datdesc_opt
	  )@@A(tyconleft, AND_datdesc_optright) 
)
end)
 in ( LrTable.NT 119, ( result, tycon1left, AND_datdesc_opt1right), 
rest671)
end
|  ( 290, ( ( _, ( MlyValue.AND_datdesc_opt AND_datdesc_opt1, _, (
AND_datdesc_optright as AND_datdesc_opt1right))) :: ( _, ( 
MlyValue.condesc condesc1, _, _)) :: _ :: ( _, ( MlyValue.tycon tycon1
, _, _)) :: ( _, ( MlyValue.tyvarseq1 tyvarseq11, (tyvarseq1left as 
tyvarseq11left), _)) :: rest671)) => let val  result = 
MlyValue.datdesc1 (fn _ => let val  (tyvarseq1 as tyvarseq11) = 
tyvarseq11 ()
 val  (tycon as tycon1) = tycon1 ()
 val  (condesc as condesc1) = condesc1 ()
 val  (AND_datdesc_opt as AND_datdesc_opt1) = AND_datdesc_opt1 ()
 in (
 DatDesc(tyvarseq1, tycon, condesc, AND_datdesc_opt)
	    @@A(tyvarseq1left, AND_datdesc_optright) 
)
end)
 in ( LrTable.NT 120, ( result, tyvarseq11left, AND_datdesc_opt1right)
, rest671)
end
|  ( 291, ( ( _, ( MlyValue.datdesc datdesc1, _, datdesc1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_datdesc_opt (fn _ => let val  (datdesc as datdesc1) = 
datdesc1 ()
 in ( SOME datdesc )
end)
 in ( LrTable.NT 121, ( result, AND1left, datdesc1right), rest671)
end
|  ( 292, ( rest671)) => let val  result = MlyValue.AND_datdesc_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 121, ( result, defaultPos, defaultPos), rest671)
end
|  ( 293, ( ( _, ( MlyValue.BAR_condesc_opt BAR_condesc_opt1, _, (
BAR_condesc_optright as BAR_condesc_opt1right))) :: ( _, ( 
MlyValue.OF_ty_opt OF_ty_opt1, _, _)) :: ( _, ( MlyValue.vid' vid'1, (
vid'left as vid'1left), _)) :: rest671)) => let val  result = 
MlyValue.condesc (fn _ => let val  (vid' as vid'1) = vid'1 ()
 val  (OF_ty_opt as OF_ty_opt1) = OF_ty_opt1 ()
 val  (BAR_condesc_opt as BAR_condesc_opt1) = BAR_condesc_opt1 ()
 in (
 ConDesc(vid', OF_ty_opt, BAR_condesc_opt)
	    @@A(vid'left, BAR_condesc_optright) 
)
end)
 in ( LrTable.NT 122, ( result, vid'1left, BAR_condesc_opt1right), 
rest671)
end
|  ( 294, ( ( _, ( MlyValue.condesc condesc1, _, condesc1right)) :: (
 _, ( _, BAR1left, _)) :: rest671)) => let val  result = 
MlyValue.BAR_condesc_opt (fn _ => let val  (condesc as condesc1) = 
condesc1 ()
 in ( SOME condesc )
end)
 in ( LrTable.NT 123, ( result, BAR1left, condesc1right), rest671)
end
|  ( 295, ( rest671)) => let val  result = MlyValue.BAR_condesc_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 123, ( result, defaultPos, defaultPos), rest671)
end
|  ( 296, ( ( _, ( MlyValue.AND_exdesc_opt AND_exdesc_opt1, _, (
AND_exdesc_optright as AND_exdesc_opt1right))) :: ( _, ( 
MlyValue.OF_ty_opt OF_ty_opt1, _, _)) :: ( _, ( MlyValue.vid' vid'1, (
vid'left as vid'1left), _)) :: rest671)) => let val  result = 
MlyValue.exdesc (fn _ => let val  (vid' as vid'1) = vid'1 ()
 val  (OF_ty_opt as OF_ty_opt1) = OF_ty_opt1 ()
 val  (AND_exdesc_opt as AND_exdesc_opt1) = AND_exdesc_opt1 ()
 in (
 ExDesc(vid', OF_ty_opt, AND_exdesc_opt)
	    @@A(vid'left, AND_exdesc_optright)
)
end)
 in ( LrTable.NT 124, ( result, vid'1left, AND_exdesc_opt1right), 
rest671)
end
|  ( 297, ( ( _, ( MlyValue.exdesc exdesc1, _, exdesc1right)) :: ( _, 
( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_exdesc_opt (fn _ => let val  (exdesc as exdesc1) = 
exdesc1 ()
 in ( SOME exdesc )
end)
 in ( LrTable.NT 125, ( result, AND1left, exdesc1right), rest671)
end
|  ( 298, ( rest671)) => let val  result = MlyValue.AND_exdesc_opt (fn
 _ => ( NONE ))
 in ( LrTable.NT 125, ( result, defaultPos, defaultPos), rest671)
end
|  ( 299, ( ( _, ( MlyValue.sigexp__AND_strdesc_opt 
sigexp__AND_strdesc_opt1, _, (sigexp__AND_strdesc_optright as 
sigexp__AND_strdesc_opt1right))) :: _ :: ( _, ( MlyValue.strid strid1,
 (stridleft as strid1left), _)) :: rest671)) => let val  result = 
MlyValue.strdesc (fn _ => let val  (strid as strid1) = strid1 ()
 val  (sigexp__AND_strdesc_opt as sigexp__AND_strdesc_opt1) = 
sigexp__AND_strdesc_opt1 ()
 in (
 StrDesc(strid, #1 sigexp__AND_strdesc_opt, #2 sigexp__AND_strdesc_opt)
	    @@A(stridleft, sigexp__AND_strdesc_optright) 
)
end)
 in ( LrTable.NT 126, ( result, strid1left, 
sigexp__AND_strdesc_opt1right), rest671)
end
|  ( 300, ( ( _, ( MlyValue.strdesc strdesc1, _, strdesc1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_strdesc_opt (fn _ => let val  (strdesc as strdesc1) = 
strdesc1 ()
 in ( SOME strdesc )
end)
 in ( LrTable.NT 127, ( result, AND1left, strdesc1right), rest671)
end
|  ( 301, ( rest671)) => let val  result = MlyValue.AND_strdesc_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 127, ( result, defaultPos, defaultPos), rest671)
end
|  ( 302, ( ( _, ( MlyValue.AND_strdesc_opt AND_strdesc_opt1, _, 
AND_strdesc_opt1right)) :: ( _, ( MlyValue.sigexp' sigexp'1, 
sigexp'1left, _)) :: rest671)) => let val  result = 
MlyValue.sigexp__AND_strdesc_opt (fn _ => let val  (sigexp' as 
sigexp'1) = sigexp'1 ()
 val  (AND_strdesc_opt as AND_strdesc_opt1) = AND_strdesc_opt1 ()
 in ( (sigexp', AND_strdesc_opt) )
end)
 in ( LrTable.NT 128, ( result, sigexp'1left, AND_strdesc_opt1right), 
rest671)
end
|  ( 303, ( ( _, ( MlyValue.tyreadesc__AND_strdesc_opt 
tyreadesc__AND_strdesc_opt1, _, (tyreadesc__AND_strdesc_optright as 
tyreadesc__AND_strdesc_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, (sigexpleft as sigexp1left), _)) :: rest671)) => let val  
result = MlyValue.sigexp__AND_strdesc_opt (fn _ => let val  (sigexp
 as sigexp1) = sigexp1 ()
 val  (tyreadesc__AND_strdesc_opt as tyreadesc__AND_strdesc_opt1) = 
tyreadesc__AND_strdesc_opt1 ()
 in (
 ( WHERETYPEMULTISigExp(sigexp, #1 tyreadesc__AND_strdesc_opt)
	      @@A(sigexpleft, tyreadesc__AND_strdesc_optright),
	    #2 tyreadesc__AND_strdesc_opt ) 
)
end)
 in ( LrTable.NT 128, ( result, sigexp1left, 
tyreadesc__AND_strdesc_opt1right), rest671)
end
|  ( 304, ( ( _, ( MlyValue.AND_tyreadesc_opt__AND_strdesc_opt 
AND_tyreadesc_opt__AND_strdesc_opt1, _, (
AND_tyreadesc_opt__AND_strdesc_optright as 
AND_tyreadesc_opt__AND_strdesc_opt1right))) :: ( _, ( MlyValue.ty ty1,
 _, _)) :: _ :: ( _, ( MlyValue.longtycon longtycon1, _, _)) :: ( _, (
 MlyValue.tyvarseq tyvarseq1, _, _)) :: ( _, ( _, (TYPEleft as 
TYPE1left), _)) :: rest671)) => let val  result = 
MlyValue.tyreadesc__AND_strdesc_opt (fn _ => let val  (tyvarseq as 
tyvarseq1) = tyvarseq1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_tyreadesc_opt__AND_strdesc_opt as 
AND_tyreadesc_opt__AND_strdesc_opt1) = 
AND_tyreadesc_opt__AND_strdesc_opt1 ()
 in (
 ( TyReaDesc(tyvarseq, longtycon, ty,
	      #1 AND_tyreadesc_opt__AND_strdesc_opt
	    )@@A(TYPEleft, AND_tyreadesc_opt__AND_strdesc_optright),
	    #2 AND_tyreadesc_opt__AND_strdesc_opt ) 
)
end)
 in ( LrTable.NT 129, ( result, TYPE1left, 
AND_tyreadesc_opt__AND_strdesc_opt1right), rest671)
end
|  ( 305, ( ( _, ( MlyValue.AND_strdesc_opt AND_strdesc_opt1, 
AND_strdesc_opt1left, AND_strdesc_opt1right)) :: rest671)) => let val 
 result = MlyValue.AND_tyreadesc_opt__AND_strdesc_opt (fn _ => let
 val  (AND_strdesc_opt as AND_strdesc_opt1) = AND_strdesc_opt1 ()
 in ( (NONE, AND_strdesc_opt) )
end)
 in ( LrTable.NT 130, ( result, AND_strdesc_opt1left, 
AND_strdesc_opt1right), rest671)
end
|  ( 306, ( ( _, ( MlyValue.tyreadesc__AND_strdesc_opt 
tyreadesc__AND_strdesc_opt1, _, tyreadesc__AND_strdesc_opt1right)) :: 
( _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_tyreadesc_opt__AND_strdesc_opt (fn _ => let val  (
tyreadesc__AND_strdesc_opt as tyreadesc__AND_strdesc_opt1) = 
tyreadesc__AND_strdesc_opt1 ()
 in (
 (SOME(#1 tyreadesc__AND_strdesc_opt), #2 tyreadesc__AND_strdesc_opt) 
)
end)
 in ( LrTable.NT 130, ( result, AND1left, 
tyreadesc__AND_strdesc_opt1right), rest671)
end
|  ( 307, ( ( _, ( MlyValue.funbind funbind1, _, (funbindright as 
funbind1right))) :: ( _, ( _, (FUNCTORleft as FUNCTOR1left), _)) :: 
rest671)) => let val  result = MlyValue.fundec (fn _ => let val  (
funbind as funbind1) = funbind1 ()
 in ( FunDec(funbind)@@A(FUNCTORleft, funbindright) )
end)
 in ( LrTable.NT 131, ( result, FUNCTOR1left, funbind1right), rest671)

end
|  ( 308, ( ( _, ( MlyValue.strexp__AND_funbind_opt 
strexp__AND_funbind_opt1, _, (strexp__AND_funbind_optright as 
strexp__AND_funbind_opt1right))) :: _ :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, _, _)) :: _ :: ( _, ( MlyValue.strid strid1, _, _)) :: _ :: (
 _, ( MlyValue.funid funid1, (funidleft as funid1left), _)) :: rest671
)) => let val  result = MlyValue.funbind (fn _ => let val  (funid as 
funid1) = funid1 ()
 val  (strid as strid1) = strid1 ()
 val  (sigexp as sigexp1) = sigexp1 ()
 val  (strexp__AND_funbind_opt as strexp__AND_funbind_opt1) = 
strexp__AND_funbind_opt1 ()
 in (
 PLAINFunBind(funid, strid, sigexp,
	    #1 strexp__AND_funbind_opt, #2 strexp__AND_funbind_opt
	  )@@A(funidleft, strexp__AND_funbind_optright) 
)
end)
 in ( LrTable.NT 132, ( result, funid1left, 
strexp__AND_funbind_opt1right), rest671)
end
|  ( 309, ( ( _, ( MlyValue.strexp__AND_funbind_opt 
strexp__AND_funbind_opt1, _, (strexp__AND_funbind_optright as 
strexp__AND_funbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp2, _, _)) :: _ :: _ :: ( _, ( MlyValue.sigexp sigexp1, _, _)) ::
 _ :: ( _, ( MlyValue.strid strid1, _, _)) :: _ :: ( _, ( 
MlyValue.funid funid1, (funidleft as funid1left), _)) :: rest671)) =>
 let val  result = MlyValue.funbind (fn _ => let val  (funid as funid1
) = funid1 ()
 val  (strid as strid1) = strid1 ()
 val  sigexp1 = sigexp1 ()
 val  sigexp2 = sigexp2 ()
 val  (strexp__AND_funbind_opt as strexp__AND_funbind_opt1) = 
strexp__AND_funbind_opt1 ()
 in (
 COLONFunBind(funid, strid, sigexp1, sigexp2,
            #1 strexp__AND_funbind_opt, #2 strexp__AND_funbind_opt
	  )@@A(funidleft, strexp__AND_funbind_optright) 
)
end)
 in ( LrTable.NT 132, ( result, funid1left, 
strexp__AND_funbind_opt1right), rest671)
end
|  ( 310, ( ( _, ( MlyValue.strexp__AND_funbind_opt 
strexp__AND_funbind_opt1, _, (strexp__AND_funbind_optright as 
strexp__AND_funbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp2, _, _)) :: _ :: _ :: ( _, ( MlyValue.sigexp sigexp1, _, _)) ::
 _ :: ( _, ( MlyValue.strid strid1, _, _)) :: _ :: ( _, ( 
MlyValue.funid funid1, (funidleft as funid1left), _)) :: rest671)) =>
 let val  result = MlyValue.funbind (fn _ => let val  (funid as funid1
) = funid1 ()
 val  (strid as strid1) = strid1 ()
 val  sigexp1 = sigexp1 ()
 val  sigexp2 = sigexp2 ()
 val  (strexp__AND_funbind_opt as strexp__AND_funbind_opt1) = 
strexp__AND_funbind_opt1 ()
 in (
 SEALFunBind(funid, strid, sigexp1, sigexp2,
	    #1 strexp__AND_funbind_opt, #2 strexp__AND_funbind_opt
	  )@@A(funidleft, strexp__AND_funbind_optright) 
)
end)
 in ( LrTable.NT 132, ( result, funid1left, 
strexp__AND_funbind_opt1right), rest671)
end
|  ( 311, ( ( _, ( MlyValue.strexp__AND_funbind_opt 
strexp__AND_funbind_opt1, _, (strexp__AND_funbind_optright as 
strexp__AND_funbind_opt1right))) :: _ :: _ :: ( _, ( MlyValue.spec 
spec1, _, _)) :: _ :: ( _, ( MlyValue.funid funid1, (funidleft as 
funid1left), _)) :: rest671)) => let val  result = MlyValue.funbind
 (fn _ => let val  (funid as funid1) = funid1 ()
 val  (spec as spec1) = spec1 ()
 val  (strexp__AND_funbind_opt as strexp__AND_funbind_opt1) = 
strexp__AND_funbind_opt1 ()
 in (
 SPECFunBind(funid, spec,
	    #1 strexp__AND_funbind_opt, #2 strexp__AND_funbind_opt
	  )@@A(funidleft, strexp__AND_funbind_optright) 
)
end)
 in ( LrTable.NT 132, ( result, funid1left, 
strexp__AND_funbind_opt1right), rest671)
end
|  ( 312, ( ( _, ( MlyValue.strexp__AND_funbind_opt 
strexp__AND_funbind_opt1, _, (strexp__AND_funbind_optright as 
strexp__AND_funbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, _, _)) :: _ :: _ :: ( _, ( MlyValue.spec spec1, _, _)) :: _
 :: ( _, ( MlyValue.funid funid1, (funidleft as funid1left), _)) :: 
rest671)) => let val  result = MlyValue.funbind (fn _ => let val  (
funid as funid1) = funid1 ()
 val  (spec as spec1) = spec1 ()
 val  (sigexp as sigexp1) = sigexp1 ()
 val  (strexp__AND_funbind_opt as strexp__AND_funbind_opt1) = 
strexp__AND_funbind_opt1 ()
 in (
 COLONSPECFunBind(funid, spec, sigexp,
	    #1 strexp__AND_funbind_opt, #2 strexp__AND_funbind_opt
	  )@@A(funidleft, strexp__AND_funbind_optright) 
)
end)
 in ( LrTable.NT 132, ( result, funid1left, 
strexp__AND_funbind_opt1right), rest671)
end
|  ( 313, ( ( _, ( MlyValue.strexp__AND_funbind_opt 
strexp__AND_funbind_opt1, _, (strexp__AND_funbind_optright as 
strexp__AND_funbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, _, _)) :: _ :: _ :: ( _, ( MlyValue.spec spec1, _, _)) :: _
 :: ( _, ( MlyValue.funid funid1, (funidleft as funid1left), _)) :: 
rest671)) => let val  result = MlyValue.funbind (fn _ => let val  (
funid as funid1) = funid1 ()
 val  (spec as spec1) = spec1 ()
 val  (sigexp as sigexp1) = sigexp1 ()
 val  (strexp__AND_funbind_opt as strexp__AND_funbind_opt1) = 
strexp__AND_funbind_opt1 ()
 in (
 SEALSPECFunBind(funid, spec, sigexp,
	    #1 strexp__AND_funbind_opt, #2 strexp__AND_funbind_opt
	  )@@A(funidleft, strexp__AND_funbind_optright) 
)
end)
 in ( LrTable.NT 132, ( result, funid1left, 
strexp__AND_funbind_opt1right), rest671)
end
|  ( 314, ( ( _, ( MlyValue.funbind funbind1, _, funbind1right)) :: (
 _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_funbind_opt (fn _ => let val  (funbind as funbind1) = 
funbind1 ()
 in ( SOME funbind )
end)
 in ( LrTable.NT 133, ( result, AND1left, funbind1right), rest671)
end
|  ( 315, ( rest671)) => let val  result = MlyValue.AND_funbind_opt
 (fn _ => ( NONE ))
 in ( LrTable.NT 133, ( result, defaultPos, defaultPos), rest671)
end
|  ( 316, ( ( _, ( MlyValue.AND_funbind_opt AND_funbind_opt1, _, 
AND_funbind_opt1right)) :: ( _, ( MlyValue.strexp' strexp'1, 
strexp'1left, _)) :: rest671)) => let val  result = 
MlyValue.strexp__AND_funbind_opt (fn _ => let val  (strexp' as 
strexp'1) = strexp'1 ()
 val  (AND_funbind_opt as AND_funbind_opt1) = AND_funbind_opt1 ()
 in ( (strexp', AND_funbind_opt) )
end)
 in ( LrTable.NT 134, ( result, strexp'1left, AND_funbind_opt1right), 
rest671)
end
|  ( 317, ( ( _, ( MlyValue.sigexp__AND_funbind_opt 
sigexp__AND_funbind_opt1, _, (sigexp__AND_funbind_optright as 
sigexp__AND_funbind_opt1right))) :: _ :: ( _, ( MlyValue.strexp 
strexp1, (strexpleft as strexp1left), _)) :: rest671)) => let val  
result = MlyValue.strexp__AND_funbind_opt (fn _ => let val  (strexp
 as strexp1) = strexp1 ()
 val  (sigexp__AND_funbind_opt as sigexp__AND_funbind_opt1) = 
sigexp__AND_funbind_opt1 ()
 in (
 ( COLONStrExp(strexp, #1 sigexp__AND_funbind_opt)
	      @@A(strexpleft, sigexp__AND_funbind_optright),
	    #2 sigexp__AND_funbind_opt ) 
)
end)
 in ( LrTable.NT 134, ( result, strexp1left, 
sigexp__AND_funbind_opt1right), rest671)
end
|  ( 318, ( ( _, ( MlyValue.sigexp__AND_funbind_opt 
sigexp__AND_funbind_opt1, _, (sigexp__AND_funbind_optright as 
sigexp__AND_funbind_opt1right))) :: _ :: ( _, ( MlyValue.strexp 
strexp1, (strexpleft as strexp1left), _)) :: rest671)) => let val  
result = MlyValue.strexp__AND_funbind_opt (fn _ => let val  (strexp
 as strexp1) = strexp1 ()
 val  (sigexp__AND_funbind_opt as sigexp__AND_funbind_opt1) = 
sigexp__AND_funbind_opt1 ()
 in (
 ( SEALStrExp(strexp, #1 sigexp__AND_funbind_opt)
	      @@A(strexpleft, sigexp__AND_funbind_optright),
	    #2 sigexp__AND_funbind_opt ) 
)
end)
 in ( LrTable.NT 134, ( result, strexp1left, 
sigexp__AND_funbind_opt1right), rest671)
end
|  ( 319, ( ( _, ( MlyValue.AND_funbind_opt AND_funbind_opt1, _, 
AND_funbind_opt1right)) :: ( _, ( MlyValue.sigexp' sigexp'1, 
sigexp'1left, _)) :: rest671)) => let val  result = 
MlyValue.sigexp__AND_funbind_opt (fn _ => let val  (sigexp' as 
sigexp'1) = sigexp'1 ()
 val  (AND_funbind_opt as AND_funbind_opt1) = AND_funbind_opt1 ()
 in ( (sigexp', AND_funbind_opt) )
end)
 in ( LrTable.NT 135, ( result, sigexp'1left, AND_funbind_opt1right), 
rest671)
end
|  ( 320, ( ( _, ( MlyValue.tyreadesc__AND_funbind_opt 
tyreadesc__AND_funbind_opt1, _, (tyreadesc__AND_funbind_optright as 
tyreadesc__AND_funbind_opt1right))) :: _ :: ( _, ( MlyValue.sigexp 
sigexp1, (sigexpleft as sigexp1left), _)) :: rest671)) => let val  
result = MlyValue.sigexp__AND_funbind_opt (fn _ => let val  (sigexp
 as sigexp1) = sigexp1 ()
 val  (tyreadesc__AND_funbind_opt as tyreadesc__AND_funbind_opt1) = 
tyreadesc__AND_funbind_opt1 ()
 in (
 ( WHERETYPEMULTISigExp(sigexp, #1 tyreadesc__AND_funbind_opt)
	      @@A(sigexpleft, tyreadesc__AND_funbind_optright),
	    #2 tyreadesc__AND_funbind_opt ) 
)
end)
 in ( LrTable.NT 135, ( result, sigexp1left, 
tyreadesc__AND_funbind_opt1right), rest671)
end
|  ( 321, ( ( _, ( MlyValue.AND_tyreadesc_opt__AND_funbind_opt 
AND_tyreadesc_opt__AND_funbind_opt1, _, (
AND_tyreadesc_opt__AND_funbind_optright as 
AND_tyreadesc_opt__AND_funbind_opt1right))) :: ( _, ( MlyValue.ty ty1,
 _, _)) :: _ :: ( _, ( MlyValue.longtycon longtycon1, _, _)) :: ( _, (
 MlyValue.tyvarseq tyvarseq1, _, _)) :: ( _, ( _, (TYPEleft as 
TYPE1left), _)) :: rest671)) => let val  result = 
MlyValue.tyreadesc__AND_funbind_opt (fn _ => let val  (tyvarseq as 
tyvarseq1) = tyvarseq1 ()
 val  (longtycon as longtycon1) = longtycon1 ()
 val  (ty as ty1) = ty1 ()
 val  (AND_tyreadesc_opt__AND_funbind_opt as 
AND_tyreadesc_opt__AND_funbind_opt1) = 
AND_tyreadesc_opt__AND_funbind_opt1 ()
 in (
 ( TyReaDesc(tyvarseq, longtycon, ty,
	      #1 AND_tyreadesc_opt__AND_funbind_opt
	    )@@A(TYPEleft, AND_tyreadesc_opt__AND_funbind_optright),
	    #2 AND_tyreadesc_opt__AND_funbind_opt ) 
)
end)
 in ( LrTable.NT 136, ( result, TYPE1left, 
AND_tyreadesc_opt__AND_funbind_opt1right), rest671)
end
|  ( 322, ( ( _, ( MlyValue.AND_funbind_opt AND_funbind_opt1, 
AND_funbind_opt1left, AND_funbind_opt1right)) :: rest671)) => let val 
 result = MlyValue.AND_tyreadesc_opt__AND_funbind_opt (fn _ => let
 val  (AND_funbind_opt as AND_funbind_opt1) = AND_funbind_opt1 ()
 in ( (NONE, AND_funbind_opt) )
end)
 in ( LrTable.NT 137, ( result, AND_funbind_opt1left, 
AND_funbind_opt1right), rest671)
end
|  ( 323, ( ( _, ( MlyValue.tyreadesc__AND_funbind_opt 
tyreadesc__AND_funbind_opt1, _, tyreadesc__AND_funbind_opt1right)) :: 
( _, ( _, AND1left, _)) :: rest671)) => let val  result = 
MlyValue.AND_tyreadesc_opt__AND_funbind_opt (fn _ => let val  (
tyreadesc__AND_funbind_opt as tyreadesc__AND_funbind_opt1) = 
tyreadesc__AND_funbind_opt1 ()
 in (
 (SOME(#1 tyreadesc__AND_funbind_opt), #2 tyreadesc__AND_funbind_opt) 
)
end)
 in ( LrTable.NT 137, ( result, AND1left, 
tyreadesc__AND_funbind_opt1right), rest671)
end
|  ( 324, ( ( _, ( MlyValue.topdec1 topdec11, topdec11left, 
topdec11right)) :: rest671)) => let val  result = MlyValue.topdec (fn
 _ => let val  (topdec1 as topdec11) = topdec11 ()
 in ( topdec1 )
end)
 in ( LrTable.NT 138, ( result, topdec11left, topdec11right), rest671)

end
|  ( 325, ( rest671)) => let val  result = MlyValue.topdec (fn _ => (
 STRDECTopDec(EMPTYStrDec@@A(defaultPos, defaultPos), NONE)
	    @@A(defaultPos, defaultPos) 
))
 in ( LrTable.NT 138, ( result, defaultPos, defaultPos), rest671)
end
|  ( 326, ( ( _, ( MlyValue.topdec_opt topdec_opt1, _, (
topdec_optright as topdec_opt1right))) :: ( _, ( MlyValue.strdec1' 
strdec1'1, (strdec1'left as strdec1'1left), _)) :: rest671)) => let
 val  result = MlyValue.topdec1 (fn _ => let val  (strdec1' as 
strdec1'1) = strdec1'1 ()
 val  (topdec_opt as topdec_opt1) = topdec_opt1 ()
 in (
 STRDECTopDec(strdec1', topdec_opt)@@A(strdec1'left, topdec_optright) 
)
end)
 in ( LrTable.NT 139, ( result, strdec1'1left, topdec_opt1right), 
rest671)
end
|  ( 327, ( ( _, ( MlyValue.topdec_opt topdec_opt1, _, (
topdec_optright as topdec_opt1right))) :: ( _, ( MlyValue.sigdec 
sigdec1, (sigdecleft as sigdec1left), _)) :: rest671)) => let val  
result = MlyValue.topdec1 (fn _ => let val  (sigdec as sigdec1) = 
sigdec1 ()
 val  (topdec_opt as topdec_opt1) = topdec_opt1 ()
 in (
 SIGDECTopDec(sigdec, topdec_opt)@@A(sigdecleft, topdec_optright) )

end)
 in ( LrTable.NT 139, ( result, sigdec1left, topdec_opt1right), 
rest671)
end
|  ( 328, ( ( _, ( MlyValue.topdec_opt topdec_opt1, _, (
topdec_optright as topdec_opt1right))) :: ( _, ( MlyValue.fundec 
fundec1, (fundecleft as fundec1left), _)) :: rest671)) => let val  
result = MlyValue.topdec1 (fn _ => let val  (fundec as fundec1) = 
fundec1 ()
 val  (topdec_opt as topdec_opt1) = topdec_opt1 ()
 in (
 FUNDECTopDec(fundec, topdec_opt)@@A(fundecleft, topdec_optright) )

end)
 in ( LrTable.NT 139, ( result, fundec1left, topdec_opt1right), 
rest671)
end
|  ( 329, ( ( _, ( MlyValue.topdec1 topdec11, topdec11left, 
topdec11right)) :: rest671)) => let val  result = MlyValue.topdec_opt
 (fn _ => let val  (topdec1 as topdec11) = topdec11 ()
 in ( SOME topdec1 )
end)
 in ( LrTable.NT 140, ( result, topdec11left, topdec11right), rest671)

end
|  ( 330, ( rest671)) => let val  result = MlyValue.topdec_opt (fn _
 => ( NONE ))
 in ( LrTable.NT 140, ( result, defaultPos, defaultPos), rest671)
end
|  ( 331, ( ( _, ( MlyValue.program' program'1, _, program'1right)) ::
 ( _, ( MlyValue.initInfix initInfix1, initInfix1left, _)) :: rest671)
) => let val  result = MlyValue.program (fn _ => let val  initInfix1 =
 initInfix1 ()
 val  (program' as program'1) = program'1 ()
 in ( (program', !J) )
end)
 in ( LrTable.NT 141, ( result, initInfix1left, program'1right), 
rest671)
end
|  ( 332, ( ( _, ( MlyValue.program_opt program_opt1, _, 
program_opt1right)) :: ( _, ( _, _, SEMICOLONright)) :: ( _, ( 
MlyValue.topdec topdec1, (topdecleft as topdec1left), _)) :: rest671))
 => let val  result = MlyValue.program' (fn _ => let val  (topdec as 
topdec1) = topdec1 ()
 val  (program_opt as program_opt1) = program_opt1 ()
 in (
 TOPDECProgram(topdec, program_opt)@@A(topdecleft, SEMICOLONright))

end)
 in ( LrTable.NT 142, ( result, topdec1left, program_opt1right), 
rest671)
end
|  ( 333, ( ( _, ( MlyValue.program_opt program_opt1, _, 
program_opt1right)) :: ( _, ( _, _, SEMICOLONright)) :: ( _, ( 
MlyValue.exp exp1, (expleft as exp1left), _)) :: rest671)) => let val 
 result = MlyValue.program' (fn _ => let val  (exp as exp1) = exp1 ()
 val  (program_opt as program_opt1) = program_opt1 ()
 in ( EXPProgram(exp, program_opt)@@A(expleft, SEMICOLONright))
end)
 in ( LrTable.NT 142, ( result, exp1left, program_opt1right), rest671)

end
|  ( 334, ( ( _, ( MlyValue.program' program'1, program'1left, 
program'1right)) :: rest671)) => let val  result = 
MlyValue.program_opt (fn _ => let val  (program' as program'1) = 
program'1 ()
 in ( SOME program' )
end)
 in ( LrTable.NT 143, ( result, program'1left, program'1right), 
rest671)
end
|  ( 335, ( rest671)) => let val  result = MlyValue.program_opt (fn _
 => ( NONE ))
 in ( LrTable.NT 143, ( result, defaultPos, defaultPos), rest671)
end
| _ => raise (mlyAction i392)
end
val void = MlyValue.VOID
val extract = fn a => (fn MlyValue.program x => x
| _ => let exception ParseInternal
	in raise ParseInternal end) a ()
end
end
structure Tokens : Parser_TOKENS =
struct
type svalue = ParserData.svalue
type ('a,'b) token = ('a,'b) Token.token
fun EOF (p1,p2) = Token.TOKEN (ParserData.LrTable.T 0,(
ParserData.MlyValue.VOID,p1,p2))
fun ABSTYPE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 1,(
ParserData.MlyValue.VOID,p1,p2))
fun AND (p1,p2) = Token.TOKEN (ParserData.LrTable.T 2,(
ParserData.MlyValue.VOID,p1,p2))
fun ANDALSO (p1,p2) = Token.TOKEN (ParserData.LrTable.T 3,(
ParserData.MlyValue.VOID,p1,p2))
fun AS (p1,p2) = Token.TOKEN (ParserData.LrTable.T 4,(
ParserData.MlyValue.VOID,p1,p2))
fun CASE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 5,(
ParserData.MlyValue.VOID,p1,p2))
fun DO (p1,p2) = Token.TOKEN (ParserData.LrTable.T 6,(
ParserData.MlyValue.VOID,p1,p2))
fun DATATYPE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 7,(
ParserData.MlyValue.VOID,p1,p2))
fun ELSE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 8,(
ParserData.MlyValue.VOID,p1,p2))
fun END (p1,p2) = Token.TOKEN (ParserData.LrTable.T 9,(
ParserData.MlyValue.VOID,p1,p2))
fun EXCEPTION (p1,p2) = Token.TOKEN (ParserData.LrTable.T 10,(
ParserData.MlyValue.VOID,p1,p2))
fun FN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 11,(
ParserData.MlyValue.VOID,p1,p2))
fun FUN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 12,(
ParserData.MlyValue.VOID,p1,p2))
fun HANDLE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 13,(
ParserData.MlyValue.VOID,p1,p2))
fun IF (p1,p2) = Token.TOKEN (ParserData.LrTable.T 14,(
ParserData.MlyValue.VOID,p1,p2))
fun IN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 15,(
ParserData.MlyValue.VOID,p1,p2))
fun INFIX (p1,p2) = Token.TOKEN (ParserData.LrTable.T 16,(
ParserData.MlyValue.VOID,p1,p2))
fun INFIXR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 17,(
ParserData.MlyValue.VOID,p1,p2))
fun LET (p1,p2) = Token.TOKEN (ParserData.LrTable.T 18,(
ParserData.MlyValue.VOID,p1,p2))
fun LOCAL (p1,p2) = Token.TOKEN (ParserData.LrTable.T 19,(
ParserData.MlyValue.VOID,p1,p2))
fun NONFIX (p1,p2) = Token.TOKEN (ParserData.LrTable.T 20,(
ParserData.MlyValue.VOID,p1,p2))
fun OF (p1,p2) = Token.TOKEN (ParserData.LrTable.T 21,(
ParserData.MlyValue.VOID,p1,p2))
fun OP (p1,p2) = Token.TOKEN (ParserData.LrTable.T 22,(
ParserData.MlyValue.VOID,p1,p2))
fun OPEN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 23,(
ParserData.MlyValue.VOID,p1,p2))
fun ORELSE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 24,(
ParserData.MlyValue.VOID,p1,p2))
fun RAISE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 25,(
ParserData.MlyValue.VOID,p1,p2))
fun REC (p1,p2) = Token.TOKEN (ParserData.LrTable.T 26,(
ParserData.MlyValue.VOID,p1,p2))
fun THEN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 27,(
ParserData.MlyValue.VOID,p1,p2))
fun TYPE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 28,(
ParserData.MlyValue.VOID,p1,p2))
fun VAL (p1,p2) = Token.TOKEN (ParserData.LrTable.T 29,(
ParserData.MlyValue.VOID,p1,p2))
fun WITH (p1,p2) = Token.TOKEN (ParserData.LrTable.T 30,(
ParserData.MlyValue.VOID,p1,p2))
fun WITHTYPE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 31,(
ParserData.MlyValue.VOID,p1,p2))
fun WHILE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 32,(
ParserData.MlyValue.VOID,p1,p2))
fun LPAR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 33,(
ParserData.MlyValue.VOID,p1,p2))
fun RPAR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 34,(
ParserData.MlyValue.VOID,p1,p2))
fun LBRACK (p1,p2) = Token.TOKEN (ParserData.LrTable.T 35,(
ParserData.MlyValue.VOID,p1,p2))
fun RBRACK (p1,p2) = Token.TOKEN (ParserData.LrTable.T 36,(
ParserData.MlyValue.VOID,p1,p2))
fun LBRACE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 37,(
ParserData.MlyValue.VOID,p1,p2))
fun RBRACE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 38,(
ParserData.MlyValue.VOID,p1,p2))
fun COMMA (p1,p2) = Token.TOKEN (ParserData.LrTable.T 39,(
ParserData.MlyValue.VOID,p1,p2))
fun COLON (p1,p2) = Token.TOKEN (ParserData.LrTable.T 40,(
ParserData.MlyValue.VOID,p1,p2))
fun SEMICOLON (p1,p2) = Token.TOKEN (ParserData.LrTable.T 41,(
ParserData.MlyValue.VOID,p1,p2))
fun DOTS (p1,p2) = Token.TOKEN (ParserData.LrTable.T 42,(
ParserData.MlyValue.VOID,p1,p2))
fun UNDERBAR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 43,(
ParserData.MlyValue.VOID,p1,p2))
fun BAR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 44,(
ParserData.MlyValue.VOID,p1,p2))
fun EQUALS (p1,p2) = Token.TOKEN (ParserData.LrTable.T 45,(
ParserData.MlyValue.VOID,p1,p2))
fun DARROW (p1,p2) = Token.TOKEN (ParserData.LrTable.T 46,(
ParserData.MlyValue.VOID,p1,p2))
fun ARROW (p1,p2) = Token.TOKEN (ParserData.LrTable.T 47,(
ParserData.MlyValue.VOID,p1,p2))
fun HASH (p1,p2) = Token.TOKEN (ParserData.LrTable.T 48,(
ParserData.MlyValue.VOID,p1,p2))
fun EQTYPE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 49,(
ParserData.MlyValue.VOID,p1,p2))
fun FUNCTOR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 50,(
ParserData.MlyValue.VOID,p1,p2))
fun INCLUDE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 51,(
ParserData.MlyValue.VOID,p1,p2))
fun SHARING (p1,p2) = Token.TOKEN (ParserData.LrTable.T 52,(
ParserData.MlyValue.VOID,p1,p2))
fun SIG (p1,p2) = Token.TOKEN (ParserData.LrTable.T 53,(
ParserData.MlyValue.VOID,p1,p2))
fun SIGNATURE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 54,(
ParserData.MlyValue.VOID,p1,p2))
fun STRUCT (p1,p2) = Token.TOKEN (ParserData.LrTable.T 55,(
ParserData.MlyValue.VOID,p1,p2))
fun STRUCTURE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 56,(
ParserData.MlyValue.VOID,p1,p2))
fun WHERE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 57,(
ParserData.MlyValue.VOID,p1,p2))
fun SEAL (p1,p2) = Token.TOKEN (ParserData.LrTable.T 58,(
ParserData.MlyValue.VOID,p1,p2))
fun ZERO (p1,p2) = Token.TOKEN (ParserData.LrTable.T 59,(
ParserData.MlyValue.VOID,p1,p2))
fun DIGIT (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 60,(
ParserData.MlyValue.DIGIT (fn () => i),p1,p2))
fun NUMERIC (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 61,(
ParserData.MlyValue.NUMERIC (fn () => i),p1,p2))
fun INT (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 62,(
ParserData.MlyValue.INT (fn () => i),p1,p2))
fun HEXINT (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 63,(
ParserData.MlyValue.HEXINT (fn () => i),p1,p2))
fun WORD (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 64,(
ParserData.MlyValue.WORD (fn () => i),p1,p2))
fun HEXWORD (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 65,(
ParserData.MlyValue.HEXWORD (fn () => i),p1,p2))
fun REAL (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 66,(
ParserData.MlyValue.REAL (fn () => i),p1,p2))
fun STRING (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 67,(
ParserData.MlyValue.STRING (fn () => i),p1,p2))
fun CHAR (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 68,(
ParserData.MlyValue.CHAR (fn () => i),p1,p2))
fun ALPHA (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 69,(
ParserData.MlyValue.ALPHA (fn () => i),p1,p2))
fun SYMBOL (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 70,(
ParserData.MlyValue.SYMBOL (fn () => i),p1,p2))
fun STAR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 71,(
ParserData.MlyValue.VOID,p1,p2))
fun TYVAR (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 72,(
ParserData.MlyValue.TYVAR (fn () => i),p1,p2))
fun ETYVAR (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 73,(
ParserData.MlyValue.ETYVAR (fn () => i),p1,p2))
fun LONGID (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 74,(
ParserData.MlyValue.LONGID (fn () => i),p1,p2))
end
end
