/** 
 * Based on code and doc by samsvl@nlr.nl and  djjohn@essex.ac.uk.
 * 
 * Released under terms of the GNU Public Licence Version 2. See file
 * COPYING for more information.
 * 
 * This library is still in its early stages. Its is likely to change
 * dramatically up to the final release (1.0). It is also intended that
 * a similar Java libary will be maintained along with the C version. I'm 
 * going to try to keep the code as similar as possible in the two libraries.
 * 
 * Feedback on the API would be appreciated. Email: joe@wombat.ie
 */


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "libgps.h"


/**
 * Given an ephemeris record and a time instant calculate the exact
 * position of the SV (satellite) in ECEF XYZ co-ordinates.
 * 
 * @param eph (in) Ephemeris of SV
 * @param Ttr (in) Time instant (GPS Time Of Week)
 * @param Trel (out) Relativistic correction term
 * @param X   (out) Pointer to position that will be filled with the calculated
 *            position.
 */

void gps_satpos(eph_type eph, double Ttr, double *Trel, double *X)
{
  	double 	A, T, n0, n, M, E, Eold, snu, cnu, nu, phi, 
    du, dr, di, u, r, i,
	 	Xdash, Ydash, Wc;

  	A = eph.sqrta * eph.sqrta;
 
  	T = Ttr - eph.toe;
  	if (T > 302400L)
	{
    		T -= 604800L;
	}
  	if (T < -302400L)
	{
    		T += 604800L;
	}

  	n0 = sqrt(mu / (A * A * A));
  	n = n0 + eph.dn;

  	M = eph.m0 + n * T;	
  	E = M;   /*start value for E*/
  	do 
	{
    		Eold = E;
    		E = M + eph.e * sin(E);
  	} while (fabs(E - Eold) >= 1.0e-8);

  	snu = sqrt(1 - eph.e * eph.e) * sin(E) / (1 - eph.e * cos(E));
  	cnu = (cos(E) - eph.e) / (1 - eph.e * cos(E));

  	nu = atan2(snu, cnu);

  	phi = nu + eph.w;

  	du = eph.cuc * cos(2 * phi) + eph.cus * sin(2 * phi);
  	dr = eph.crc * cos(2 * phi) + eph.crs * sin(2 * phi);
  	di = eph.cic * cos(2 * phi) + eph.cis * sin(2 * phi);

  	u = phi + du;
  	r = A * (1 - eph.e * cos(E)) + dr;
  	i = eph.i0 + eph.idot * T + di;

  	Xdash = r * cos(u);
  	Ydash = r * sin(u);

  	Wc = eph.omg0 + (eph.odot - Wedot) * T - Wedot * eph.toe;

  	X[0] = Xdash * cos(Wc) - Ydash * cos(i) * sin(Wc);
  	X[1] = Xdash * sin(Wc) + Ydash * cos(i) * cos(Wc);
  	X[2] = Ydash * sin(i);

    /*relativistic correction term*/
    *Trel = F * eph.e * eph.sqrta * sin(E);   

}


/** 
 * Convert ECEF XYZ coordinates to WGS-84 Latitude, Longitude and Altitude
 * above ellipsoid
 * Latidude and longitude are in radians. Altitude in meters.
 * 
 * @param Xi0  Position in ECEF XYZ
 * @param Xo   Resulting LLA co-ordinates
 */

void gps_xyz2lla (vec3 Xi0, vec3 Xo)
{
	
  	double p, T, sT, cT, N;
	vec3  Xi;

	Xi[0] = Xi0[0] - OS_DX;
	Xi[1] = Xi0[1] - OS_DY;
	Xi[2] = Xi0[2] - OS_DZ;

  	p = sqrt(Xi[0] * Xi[0] + Xi[1] * Xi[1]);
  	T = atan(Xi[2] * a / (p * b));
  	sT = sin(T);
  	cT = cos(T);
  	Xo[0] = atan((Xi[2] + e2sqr * b * sT * sT * sT) / (p - e1sqr * a * cT * cT * cT));
  	if (Xi[0] == 0.0)
    		Xo[1] = pi / 2.0;
  	else
    		Xo[1] = atan(Xi[1] / Xi[0]);
  	N = a / sqrt(1.0 - e1sqr * sin(Xo[0]) * sin(Xo[0]));
  	Xo[2] = p / cos(Xo[0]) - N;

}

/** 
 * Convert WGS-84 Latitude, Longitude and Altitude to ECEF XYZ.
 * (Opposite to gps_xyz2lla).
 * 
 * @param Xi  Position in LLA 
 * @param Xo   Resulting ECEF XYZ (meters)
 */

void gps_lla2xyz (vec3 Xi, vec3 Xo)
{
  	double N;

  	N = a / sqrt(1.0 - e1sqr * sin(Xi[0]) * sin(Xi[0]));


  	Xo[0] = (N + Xi[2]) * cos(Xi[0]) * cos(Xi[1]);
  	Xo[1] = (N + Xi[2]) * cos(Xi[0]) * sin(Xi[1]);
  	Xo[2] = (N * (1.0 - e1sqr) + Xi[2]) * sin(Xi[0]);

	Xo[0] += OS_DX;
	Xo[1] += OS_DY;
	Xo[2] += OS_DZ;
	
}

/**
 * Utility function that returns the magniture of a vec3
 * 
 * @param vec  Vector
 * @return Magnitude of vector (FIXME: there is a proper term for this other?!)
 */
double vec3mag (vec3 vec)
{
    return sqrt (vec[0]*vec[0] + vec[1]*vec[1] + vec[2]*vec[2]);
}

/**
 * Utility function that subtracts two vec3 vectors and places result
 * in a third vector. ie vecresult = veca - vecb
 * 
 * @param vecresult Result of subtraction is returned here
 * @param veca (FIXME -- can't remember the proper math terms for this)
 * @param vecb
 */
void vec3sub (vec3 vecresult, vec3 veca, vec3 vecb)
{
    vecresult[0] = veca[0] - vecb[0];
    vecresult[1] = veca[1] - vecb[1];
    vecresult[2] = veca[2] - vecb[2];
}



/**
 * At many places in the following procedure solve the subdeterminant value of a
 * 4 x 4 array is required. For convenience the function sub is defined below
 * 
 * @param A  input 4 x 4 array
 * @param r  row number to be deleted
 * @param c_ column number to be deleted
 * 
 * @return value of 3 x 3 subdeterminant 
 */
static double sub(mat16 A, long r, long c_)
{
  	double B[3][3];
  	long i, i1, j, j1;

  	i1 = 0;
  	for (i = 0; i <= 3; i++) 
	{
    		if (i + 1 != r) 	
		{
      			i1++;
      			j1 = 0;
      			for (j = 1; j <= 4; j++) 
			{
				if (j != c_) 
				{
	  				j1++;
	  				B[i1 - 1][j1 - 1] = A[i][j - 1];
				}
      			}
    		}
  	}
  	return (B[0][0] * (B[1][1] * B[2][2] - B[1][2] * B[2][1]) + B[1]
	    	[0] * (B[2][1] * B[0][2] - B[0][1] * B[2][2]) + B[2]
	    	[0] * (B[0][1] * B[1][2] - B[0][2] * B[1][1]));
} 


/**
 * Given satellite positions and raw pseudoranges try to find position of
 * receiver (Xr).
 * 
 * @param Xs	(in) array with 3 columns and 32 rows for the coordinates of the sat's
 * @param SV	(in) valid prn's
 * @param P	(in) raw pseudoranges
 * @param Xr 	(in/out) input of initial final position
 * @param Cr	(out) receiver clock error
 * @param status (out)	true: calculation OK, false: no solution
 */

void gps_solve (double Xs[][3], 
	    boolean *SV, 
	    double *P, 
	    double *Xr, 
	    double *Cr, 
	    boolean *status)
{

    long prn, it, i, j, k;
    double R[32], L[32];
    double A[32][4];
    double AL[4];
    mat16 AA, AAi;
    long n;
    double det;
    double D[4];
    
    
    
    it = 0;   /*iteration counter*/
    
    do 
    {   /*iterations*/
	
	it++;   /*increase iteration counter*/
	
	for (prn = 0; prn <= 31; prn++) 
	{
	    if (SV[prn])
	    {
		R[prn] = sqrt((Xr[0] - Xs[prn][0]) * (Xr[0] - Xs[prn][0]) +
			      (Xr[1] - Xs[prn][1]) * (Xr[1] - Xs[prn][1]) +
			      (Xr[2] - Xs[prn][2]) * (Xr[2] - Xs[prn][2]));
		/*range from receiver to satellite*/
		L[prn] = P[prn] - R[prn];   /*range residual value*/
		for (k = 0; k <= 2; k++)
		{
		    A[prn][k] = (Xr[k] - Xs[prn][k]) / R[prn];
		}
		/* normalised user->satellite vectors */
		
		A[prn][3] = -1.0;   /*A is the geometry matrix or model matrix*/
	    }
	}
	
	for (k = 0; k <= 3; k++) 
	{   	/*calculate A.L*/
	    AL[k] = 0.0;
	    for (prn = 0; prn <= 31; prn++) 
	    {
		if (SV[prn])
		  AL[k] += A[prn][k] * L[prn];
	    }
	}
	
	for (k = 0; k <= 3; k++) 
	{
	    for (i = 0; i <= 3; i++) 
	    {   	/*calculate A.A*/
		AA[k][i] = 0.0;
		for (prn = 0; prn <= 31; prn++) 
		{
		    if (SV[prn])
		    {
			AA[k][i] += A[prn][k] * A[prn][i];
		    }
		}
	    }
	}
	
	/*invert A.A*/
	det = AA[0][0] * sub(AA, 1L, 1L) - 
	  AA[1][0] * sub(AA, 2L, 1L) + 
	  AA[2][0] * sub(AA, 3L, 1L) - 
	  AA[3][0] * sub(AA, 4L, 1L);
	
	if (det == 0.0)
	{
	    /*there is something wrong if more than 6 iterations are required*/
	    *status = false;
	}
	else 
	{
	    *status = true;
	    
	    for (k = 1; k <= 4; k++)
	    {
		for (i = 1; i <= 4; i++) 
		{
		    n = k + i;
		    if (n & 1)
		    {
			j = -1;
		    }
		    else
		    {
			j = 1;
		    }
		    AAi[k - 1][i - 1] = j * sub(AA, i, k) / det;
		}
	    }
	    
	    /*calculate (invA.A).(A.L)*/
	    for (k = 0; k <= 3; k++) 
	    {
		D[k] = 0.0;
		for (i = 0; i <= 3; i++)
		{
		    D[k] += AAi[k][i] * AL[i];
		}
	    }
	    
	    /*update position*/
	    for (k = 0; k <= 2; k++)
	    {
		Xr[k] += D[k];
		if (DEBUG)
		{
		    fprintf (stderr,"D[%d]=%15.2f  Xr[%d]=%15.2f\n",
			     (int)k,D[k],
			     (int)k,Xr[k]);
		}
	    }
	}
    }
    while 
      (
       it != 6 /* there is something wrong if more than 6 iterations are required*/
       && fabs(D[0]) + fabs(D[1]) + fabs(D[2]) >= 1.0e-2    /*iteration criterion*/
       && *status				       /*calculation not succeeded*/
       );
    
    
    *Cr = D[3];   /*receiver clock error in range (meters) */
    
    if (it == 6) 
    {
	/* fprintf(stderr,"solve it : too many interations\n"); */
	*status = false;
    }  /*iteration not succeeded*/
    
} 

/**
 * Given a initial guess as to current position, a instance in time, 
 * a list of pseudoranges and ephemeris records, ionospheric correction
 * coefficients calculate new position in ECEF XYZ co-ordinates.
 * 
 * 
 * @param   Xr       (in/out) Receiver position in ECEF XYZ 
 * @param   Trc      (in) Time of Week 
 * @param sv_valid  (in) Index i set to true if valid PR in Praw[i] and
 *                   valid ephemeris in eph[i]
 * @param      ion      (in) 8 ionospheric correction coefficients. If this is
 *                     NULL then no ionospheric corrections will be applied.
 * @param      Praw     (in) Raw pseudoranges
 * @param      *cclke_m  (out) Receiver clock error in meters
 * @return  True of sucessful. False if can't solve.
 */
boolean gps_sps_calc (
	       vec3 current_Xr,
	       double Trc, 
	       boolean sv_valid[32], 
	       eph_type eph[32], 
	       vec8 ion, 
	       double Praw[32],
	       double *cclke_m
)
{
    boolean status;
    vec3 Xr,Xlla,tmp3;
    double Ttr, tau, Trel, Az, El, Cr, alpha, dTclck, dTiono,
    dRtrop, Lat, Lon;
    
    vec32 Pcor;
    mat96 Xs;
    
    int i;
    int c = const_c;
    
    /* Work with a copy of current_Xr incase we can't solve */
    Xr[0]=current_Xr[0];
    Xr[1]=current_Xr[1];
    Xr[2]=current_Xr[2];
    
    gps_xyz2lla (Xr, Xlla);
    
    /*assuming the receiver clock error and initial position not sufficiently
     good known, I make two passes through the processing steps*/

    /*PASS 1*/
    
    for (i = 0; i < 32; i++) 
    {
	if (sv_valid[i]) 
	{  
	    /*do for each SV*/
	    /*set all transit times to nominal value and calculate time of transmission*/
	    tau = 0.075;
	    
	    Ttr = Trc - tau;
		
	    /*calculate SV position and correct for earth rotation*/
	    gps_satpos(eph[i], Ttr, &Trel, tmp3);
	    alpha = tau * We;
	    
	    Xs[i][0] = tmp3[0] * cos(alpha) + tmp3[1] * sin(alpha);
	    Xs[i][1] = tmp3[1] * cos(alpha) - tmp3[0] * sin(alpha);
	    Xs[i][2] = tmp3[2];
	    
	    if (DEBUG) 
	    {
		fprintf(stderr, "SV loc : %2d%15.3f%15.3f%15.3f %15.0f\n",
			i, 
			Xs[i][0], 
			Xs[i][1], 
			Xs[i][2],
			vec3mag (Xs[i])
			);
	    }
	    
	    /*calculate azimuth and elevation*/
	    gps_calcAzEl( Xs[i] , Xr, &Az, &El, &status);
	    if (!status) 
	    {
		printf("Error in calcAzEl - check input data\n");
		return false;
	    }
	    
	    if (DEBUG)
	    {
		
		fprintf(stderr, "Az, El : %2d%11.3f%10.3f\n",
			i, Az * 180.0 / pi, El * 180.0 / pi);
	    }
	    
	    /*calculate pseudorange corrections and apply to pseudoranges*/
	    
	    /*clock correction */
	    dTclck = eph[i].af0 +
	      eph[i].af1 * (Ttr - eph[i].toc) +
	      eph[i].af2 * (Ttr - eph[i].toc) * (Ttr - eph[i].toc) +
	      Trel - eph[i].gd;
	    
	    /*iono correction*/
	    if (ion == NULL) 
	    {
		dTiono = 0.0;
	    } 
	    else 
	    {
		Lat = Xlla[0];
		Lon = Xlla[1];
		ionocorr(ion, Lat, Lon, Az, El, Ttr, &dTiono);
	    }
	    
	    /*tropo correction using standard atmosphere values*/
	    dRtrop = 2.312 / sin(sqrt(El * El + 1.904e-3)) +
	      0.084 / sin(sqrt(El * El + 0.6854e-3));
	    
	    if (DEBUG) fprintf(stderr, "Corr   : %2d%11.3f%10.3f%10.3f\n",
			       i, dTclck * c, dTiono * c, dRtrop);
	    
	    /* correct pseudorange */
	    Pcor[i] = Praw[i] + dTclck * c - dTiono * c - dRtrop;
	    
	}  /*do for each SV*/
	
    }
    
    /*calculate receiver position*/
    gps_solve(Xs, sv_valid, Pcor, Xr, &Cr, &status);
    
    *cclke_m = -Cr;
    
    if (!status) 
    {
	int i;
	
	if (DEBUG) 
	{
	    fprintf (stderr,"gps_solve: usr_xyz=(%f, %f, %f)\n",
		     Xr[0],Xr[1],Xr[2]);
	}
	
	for (i=0; i<32; i++)
	{
	    /* printf("SV[%d] = %d\n",i,SV[i]); */
	    if ( sv_valid[i] )
	    {
		if (DEBUG)
		{
		    fprintf(stderr,"gps_solve: sat_xyz(%d) = (%.0f,%.0f,%.0f)  r = %.0f dclk = %f status=%d\n",
			    i+1,
			    Xs[i][0],Xs[i][1],Xs[i][2],
			    Pcor[i],
			    Cr,  
			    status
			    );
		}
	    }
	}
	if (DEBUG)
	{
	    fprintf(stderr,"Error in solve 1 - check input data\n");
	}
	return false;
    }
    if (DEBUG) 
    {
	fprintf(stderr, "Pos XYZ: %12.3f%12.3f%12.3f%12.3f\n", Xr[0], Xr[1], Xr[2], Cr);
    }
    /*convert back to Lat, Lon, Alt*/
    gps_xyz2lla(Xr, Xlla);
    
    /*PASS 2 - The receiver position and -clock error is now well enough known
     to calculate the final pseudorange corrections*/
    if (DEBUG) fprintf(stderr, "\nPASS 2\n");
    
    for (i = 0; i < 32; i++) 
    {
	if (sv_valid[i]) 
	{  	/*do for each SV*/
	    /*recalculate transit time and time of transmission*/
	    tau = (Pcor[i] + Cr) / c;
	    Ttr = Trc - tau;
	    
	    /*recalculate SV position and correct for earth rotation*/
	    
	    gps_satpos (eph[i], Ttr, &Trel, tmp3);
	    alpha = tau * We;
	    Xs[i][0] = tmp3[0] * cos(alpha) + tmp3[1] * sin(alpha);
	    Xs[i][1] = tmp3[1] * cos(alpha) - tmp3[0] * sin(alpha);
	    Xs[i][2] = tmp3[2];
	    if (DEBUG) fprintf(stderr, "SV     : %2d%15.3f%15.3f%15.3f\n",
			       i, Xs[i][0], Xs[i][1], Xs[i][2]);
	    
	    /*recalculate azimuth and elevation*/
	    gps_calcAzEl( Xs[i], Xr, &Az, &El, &status);
	    if (!status) 
	    {
		fprintf(stderr,"Error in calcAzEl - check input data\n");
		return false;
	    }
	    if (DEBUG) fprintf(stderr, "Az, El : %2d%11.3f%10.3f\n",
			       i, Az * 180.0 / pi, El * 180.0 / pi);
	    
	    /*recalculate pseudorange corrections and apply to pseudoranges*/
	    
	    /*clock correction*/
	    dTclck = eph[i].af0 +
	      eph[i].af1 * (Ttr - eph[i].toc) +
	      eph[i].af2 * (Ttr - eph[i].toc) * (Ttr - eph[i].toc) +
	      Trel - eph[i].gd;
	    
	    /*iono correction*/
	    if (ion == NULL) 
	    {
		dTiono = 0.0;
	    }
	    else 
	    {
		Lat = Xlla[0];
		Lon = Xlla[1];
		ionocorr(ion, Lat, Lon, Az, El, Ttr, &dTiono);
	    }
	    
	    /*tropo correction using standard atmosphere values*/
	    dRtrop = 2.312 / sin(sqrt(El * El + 1.904e-3)) +
	      0.084 / sin(sqrt(El * El + 0.6854e-3));
	    
	    if (DEBUG) fprintf(stderr, "Corr   : %2d%11.3f%10.3f%10.3f\n",
			       i, dTclck * c, dTiono * c, dRtrop);
	    
	    /*correct pseudorange*/
	    Pcor[i] = Praw[i] + dTclck * c - dTiono * c - dRtrop + Cr;
	    
	}  /*do for each SV*/
	
    }
    
    /*calculate receiver position*/
    gps_solve(Xs, sv_valid, Pcor, Xr, &Cr, &status);
    if (!status) 
    {
	printf("Error in solve 2 - check input data\n");
	return false;
    }

    /* Copy Xr back to current_Xr */
    current_Xr[0]=Xr[0];
    current_Xr[1]=Xr[1];
    current_Xr[2]=Xr[2];
    
    /*convert back to Lat, Lon, Alt*/
    gps_xyz2lla(Xr, Xlla);
    
    if ( DEBUG  && !isnan(Xlla[0]) ) {
	fprintf(stderr, "Pos LLA: %f %15.8f %15.8f %12.3f\n",
		Trc,
		Xlla[0] * RAD_TO_DEG, 
		Xlla[1] * RAD_TO_DEG, 
		Xlla[2]);
    }
    
    return true;
}

/**
 * Given satellite position (Xs) and receiver position (Xu) calculate
 * the Azimuth and Elevation of the satellite relative to the receiver
 * 
 * @param Xs Position of satellite in ECEF XYZ
 * @param Xu Position of user (receiver) in ECEF XYZ
 * @param Az (out) The azimuth of the satellite relative to the receiver (rads)
 * @param El (out) The elevation of the satellite relative to the receiver (rads)
 * @param stat (out) Status of calculation. true of successful
 */
void gps_calcAzEl(double *Xs, double *Xu, double *Az, double *El, boolean *stat)
{
  	double R, p, x, y, z, s;
 	double e[3][3];
  	long i, k;
  	vec3 d;

  	x = Xu[0];
  	y = Xu[1];
  	z = Xu[2];
  	p = sqrt(x * x + y * y);
  	if (p == 0.0)
    		*stat = false;
  	else
    		*stat = true;

  	if (!*stat)
    		return;

  	R = sqrt(x * x + y * y + z * z);
  	e[0][0] = -(y / p);
  	e[0][1] = x / p;
  	e[0][2] = 0.0;
  	e[1][0] = -(x * z / (p * R));
  	e[1][1] = -(y * z / (p * R));
  	e[1][2] = p / R;
  	e[2][0] = x / R;
  	e[2][1] = y / R;
  	e[2][2] = z / R;

	/* matrix multiply vector from user */
  	for (k = 0; k <= 2; k++) 
	{
    		d[k] = 0.0;
    		for (i = 0; i <= 2; i++)
		{
      			d[k] += (Xs[i] - Xu[i]) * e[k][i];
		}
  	}

  	s = d[2] / sqrt(d[0] * d[0] + d[1] * d[1] + d[2] * d[2]);
  	if (s == 1.0)
	{
    		*El = 0.5 * pi;
	}
  	else
	{
    		*El = atan(s / sqrt(1.0 - s * s));
	}

  	if (d[1] == 0.0 && d[0] > 0.0) 
	{
    		*Az = 0.5 * pi;
  	}
  	else if (d[1] == 0.0 && d[0] < 0.0) 
	{
    		*Az = 1.5 * pi;
  	}
	else
	{
  		*Az = atan(d[0] / d[1]);
  		if (d[1] < 0.0)
		{
    			*Az += pi;
		}
  		else if (d[1] > 0.0 && d[0] < 0.0)
		{
      			*Az += 2.0 * pi;
		}
 
	}
}  /*calcAzEl*/

void ionocorr
(
	double *ion, double Latu, double Lonu, double Az, double El, double Ttr, double *dTiono
)
{
  	/*ion 		iono correction coefficients from nav message*/
  	/*Latu		user's latitude [rad]*/
  	/*Lonu		user's longitude [rad]*/
  	/*Az		SV azimuth [rad]*/
  	/*El		SV elevation [rad]*/
  	/*Ttr		SV signal transmission time [sec]*/
  	/*dTiono	Ionospheric delay [sec]*/
  	double phi, Lati, Loni, Latm, T, F_, x, per, amp, a0, a1, a2, a3, b0, b1, b2, b3;


  	/*for clarity copy array*/
  	a0 = ion[0];
  	a1 = ion[1];
  	a2 = ion[2];
  	a3 = ion[3];
  	b0 = ion[4];
  	b1 = ion[5];
  	b2 = ion[6];
  	b3 = ion[7];

  	/*convert from radians to semicircles*/
  	Latu /= pi;
  	Lonu /= pi;
  	Az /= pi;
  	El /= pi;

  	/*calculation*/
  	phi = 0.0137 / (El + 0.11) - 0.022;
  	Lati = Latu + phi * cos(Az * pi);
  	if (Lati > 0.416)
	{
    		Lati = 0.416;
	}
  	else if (Lati < -0.416)
	{
    		Lati = -0.416;
	}
  	Loni = Lonu + phi * sin(Az * pi) / cos(Lati * pi);
  	Latm = Lati + 0.064 * cos((Loni - 1.617) * pi);
  	T = 4.32e+4 * Loni + Ttr;
  	if (T >= 86400L)
	{
    		T -= 86400L;
	}
  	else if (T < 0)
	{
    		T += 86400L;
	}
  	F_ = 1.0 + 16.0 * (0.53 - El) * (0.53 - El) * (0.53 - El);
  	per = b0 + b1 * Latm + b2 * Latm * Latm + b3 * Latm * Latm * Latm;
  	if (per < 72000.0)
	{
    		per = 72000.0;
	}
  	x = 2 * pi * (T - 50400.0) / per;
  	amp = a0 + a1 * Latm + a2 * Latm * Latm + a3 * Latm * Latm * Latm;
  	if (amp < 0.0)
	{
    		amp = 0.0;
	}
  	if (fabs(x) >= 1.57)
	{
    		*dTiono = F_ * 5.0e-9;
	}
  	else
	{
    		*dTiono = F_ * (5.0e-9 + amp * (1.0 - x * x / 2.0 + x * x * x * x / 24.0));
  	}
}
