/* ----------------------------------------------------------------------
 * select -- select sequences containing certain nucleotides
 * Copyright (C) 2000 January Weiner III
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 ---------------------------------------------------------------------- */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "genpak.h"
#include "gp_getopt.h"

#define VERSION "0.01"
#define PROGNAME "select"

char *progname ;

typedef struct {
	FILE *in ;
	FILE *out ; 
	char *nucl ;
	int search_degenerate ; } opt_s ;

/*
 *
 */

int select_check_sequence(sekw *s, opt_s o) ;

int main(int argc, char *argv[])
{
	extern int optind ;
	extern char *optarg ;
	int width = 70 ; /* width with which the sequence gets formatted */
	opt_s options ;
	sekw *inseq, *outseq ;

	int i, c;
	char message[100] ;

	progname = argv[0] ;
	textdomain("genpak") ;
	options.search_degenerate = FALSE ;
	options.nucl = NULL ;

	while ((c = gp_getopt(argc, argv, "n:DHqdvh")) != EOF)
		switch(c) {
		case 'n':
			options.nucl = strdup(optarg) ;
			for(i = 0 ; i < strlen(options.nucl) ; i++)
				options.nucl[i] = toupper(options.nucl[i]) ;
			gp_warn("Looking for sequences containing nucleotides matching any of %s", 
				options.nucl) ;
			break ;
		case 'D':
			gp_warn("Looking for sequences containing degenerate nucleotides") ;
			options.search_degenerate = TRUE ;
			break ;
		case 'H':
			html = TRUE ;
			break ;
		case 'q':
			quiet = TRUE ;
			break ;
		case 'v':
			fprintf(stderr,_("%s version %s\n"),progname,VERSION) ;
			exit(EXIT_SUCCESS) ;
			break ;
		case 'd':
			debug = TRUE ;
			gp_warn(_("Running in debug mode")) ;
			break ;
		case 'h':
			Help() ;
			break ;
		default:
			gp_error(_("Type '%s -h' for help"),progname) ;
			break;
		}

	if(!options.search_degenerate && !options.nucl) 
		gp_error("You have to specify one of the options -D and -n") ;


/* open the file pointer to read the sequences 
 * from: standard input or a file provided? */
	if(optind >= argc) options.in = stdin ;
	else options.in = gp_file_open(argv[optind],"r") ;

/* opening the file pointer to write the output: 
 * standard output or file provided? */
	optind++ ;

	if(optind >= argc) options.out = stdout ;
	else options.out = gp_file_open(argv[optind],"wb") ;

	for(i = 0 ; (inseq = gp_seq_read(options.in)) ; i++) {

		gp_warn("processing sequence %i (%s)", i + 1, inseq->name) ;
		if(select_check_sequence(inseq, options)) 
			gp_seq_print_fasta(options.out, inseq, 70) ;
	}

	gp_warn("%i matches found", i + 1) ;
	
	if(html) gp_warn_print_all(options.out) ;
	fclose(options.out) ;
	gzclose(options.in) ;
	return EXIT_SUCCESS ;
}


int select_check_sequence(sekw *s, opt_s o) {
	int i ;
	char *p, *needle, degen[20]  ;
	int res = FALSE ;

	strcpy(degen, "MKYRWSVHDBN") ;

	p = s->sequ ;

	if(o.search_degenerate) needle = degen ;
	if(o.nucl) needle = o.nucl ;

	for(i = 0 ; i < s->leng && !res ; i++, p++) {

		if(strchr(needle, toupper(*p)) ) res = TRUE ;

	}

	return res ;

}


/* Standard mesage */
void Help()
{
	fprintf(stdout,_(""
	"\n"
	"%s, v. %s- select sequences containing specified nucleotides"
	"\n"
	"  Usage:\n"
	"     %s [options] [ input file ] [ output file ]\n"
	"\n"
	"  Options:\n"
	"     -n string : select seq. containing any of nucleotides contained in \"string\"\n"
	"     -D        : select seq. containing degenerate nucleotides (that is, any of\n"
	"                 \"MKYRWSVHDBN\"\n"
	"     -v        : print version information & exit\n"
	"     -v        : print version information & exit\n"
	"     -h        : print this help screen & exit\n"
	"     -q        : quiet, suppress error messages\n\n"
	"You have to specify one of the options: -n, -D\n\n"),

	PROGNAME,VERSION,progname);
	exit(EXIT_SUCCESS);
}


			
