// Copyright 2023 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package chat provides access to the Google Chat API.
//
// For product documentation, see: https://developers.google.com/hangouts/chat
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/chat/v1"
//	...
//	ctx := context.Background()
//	chatService, err := chat.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	chatService, err := chat.NewService(ctx, option.WithScopes(chat.ChatSpacesReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	chatService, err := chat.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	chatService, err := chat.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package chat // import "google.golang.org/api/chat/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version

const apiId = "chat:v1"
const apiName = "chat"
const apiVersion = "v1"
const basePath = "https://chat.googleapis.com/"
const mtlsBasePath = "https://chat.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Private Service: https://www.googleapis.com/auth/chat.bot
	ChatBotScope = "https://www.googleapis.com/auth/chat.bot"

	// Delete conversations and spaces & remove access to associated files
	// in Google Chat
	ChatDeleteScope = "https://www.googleapis.com/auth/chat.delete"

	// Import spaces, messages, and memberships into Google Chat.
	ChatImportScope = "https://www.googleapis.com/auth/chat.import"

	// View, add, and remove members from conversations in Google Chat
	ChatMembershipsScope = "https://www.googleapis.com/auth/chat.memberships"

	// Add and remove itself from conversations in Google Chat
	ChatMembershipsAppScope = "https://www.googleapis.com/auth/chat.memberships.app"

	// View members in Google Chat conversations.
	ChatMembershipsReadonlyScope = "https://www.googleapis.com/auth/chat.memberships.readonly"

	// View, compose, send, update, and delete messages, and add, view, and
	// delete reactions to messages.
	ChatMessagesScope = "https://www.googleapis.com/auth/chat.messages"

	// Compose and send messages in Google Chat
	ChatMessagesCreateScope = "https://www.googleapis.com/auth/chat.messages.create"

	// View, add, and delete reactions to messages in Google Chat
	ChatMessagesReactionsScope = "https://www.googleapis.com/auth/chat.messages.reactions"

	// Add reactions to messages in Google Chat
	ChatMessagesReactionsCreateScope = "https://www.googleapis.com/auth/chat.messages.reactions.create"

	// View reactions to messages in Google Chat
	ChatMessagesReactionsReadonlyScope = "https://www.googleapis.com/auth/chat.messages.reactions.readonly"

	// View messages and reactions in Google Chat
	ChatMessagesReadonlyScope = "https://www.googleapis.com/auth/chat.messages.readonly"

	// Create conversations and spaces and view or update metadata
	// (including history settings) in Google Chat
	ChatSpacesScope = "https://www.googleapis.com/auth/chat.spaces"

	// Create new conversations in Google Chat
	ChatSpacesCreateScope = "https://www.googleapis.com/auth/chat.spaces.create"

	// View chat and spaces in Google Chat
	ChatSpacesReadonlyScope = "https://www.googleapis.com/auth/chat.spaces.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/chat.bot",
		"https://www.googleapis.com/auth/chat.delete",
		"https://www.googleapis.com/auth/chat.import",
		"https://www.googleapis.com/auth/chat.memberships",
		"https://www.googleapis.com/auth/chat.memberships.app",
		"https://www.googleapis.com/auth/chat.memberships.readonly",
		"https://www.googleapis.com/auth/chat.messages",
		"https://www.googleapis.com/auth/chat.messages.create",
		"https://www.googleapis.com/auth/chat.messages.reactions",
		"https://www.googleapis.com/auth/chat.messages.reactions.create",
		"https://www.googleapis.com/auth/chat.messages.reactions.readonly",
		"https://www.googleapis.com/auth/chat.messages.readonly",
		"https://www.googleapis.com/auth/chat.spaces",
		"https://www.googleapis.com/auth/chat.spaces.create",
		"https://www.googleapis.com/auth/chat.spaces.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Media = NewMediaService(s)
	s.Spaces = NewSpacesService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Media *MediaService

	Spaces *SpacesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewMediaService(s *Service) *MediaService {
	rs := &MediaService{s: s}
	return rs
}

type MediaService struct {
	s *Service
}

func NewSpacesService(s *Service) *SpacesService {
	rs := &SpacesService{s: s}
	rs.Members = NewSpacesMembersService(s)
	rs.Messages = NewSpacesMessagesService(s)
	return rs
}

type SpacesService struct {
	s *Service

	Members *SpacesMembersService

	Messages *SpacesMessagesService
}

func NewSpacesMembersService(s *Service) *SpacesMembersService {
	rs := &SpacesMembersService{s: s}
	return rs
}

type SpacesMembersService struct {
	s *Service
}

func NewSpacesMessagesService(s *Service) *SpacesMessagesService {
	rs := &SpacesMessagesService{s: s}
	rs.Attachments = NewSpacesMessagesAttachmentsService(s)
	rs.Reactions = NewSpacesMessagesReactionsService(s)
	return rs
}

type SpacesMessagesService struct {
	s *Service

	Attachments *SpacesMessagesAttachmentsService

	Reactions *SpacesMessagesReactionsService
}

func NewSpacesMessagesAttachmentsService(s *Service) *SpacesMessagesAttachmentsService {
	rs := &SpacesMessagesAttachmentsService{s: s}
	return rs
}

type SpacesMessagesAttachmentsService struct {
	s *Service
}

func NewSpacesMessagesReactionsService(s *Service) *SpacesMessagesReactionsService {
	rs := &SpacesMessagesReactionsService{s: s}
	return rs
}

type SpacesMessagesReactionsService struct {
	s *Service
}

// ActionParameter: List of string parameters to supply when the action
// method is invoked. For example, consider three snooze buttons: snooze
// now, snooze one day, snooze next week. You might use `action method =
// snooze()`, passing the snooze type and snooze time in the list of
// string parameters.
type ActionParameter struct {
	// Key: The name of the parameter for the action script.
	Key string `json:"key,omitempty"`

	// Value: The value of the parameter.
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Key") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Key") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ActionParameter) MarshalJSON() ([]byte, error) {
	type NoMethod ActionParameter
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ActionResponse: Parameters that a Chat app can use to configure how
// its response is posted.
type ActionResponse struct {
	// DialogAction: Input only. A response to an event related to a dialog
	// (https://developers.google.com/chat/how-tos/dialogs). Must be
	// accompanied by `ResponseType.Dialog`.
	DialogAction *DialogAction `json:"dialogAction,omitempty"`

	// Type: Input only. The type of Chat app response.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default type that's handled as `NEW_MESSAGE`.
	//   "NEW_MESSAGE" - Post as a new message in the topic.
	//   "UPDATE_MESSAGE" - Update the Chat app's message. This is only
	// permitted on a `CARD_CLICKED` event where the message sender type is
	// `BOT`.
	//   "UPDATE_USER_MESSAGE_CARDS" - Update the cards on a user's message.
	// This is only permitted as a response to a `MESSAGE` event with a
	// matched url, or a `CARD_CLICKED` event where the message sender type
	// is `HUMAN`. Text is ignored.
	//   "REQUEST_CONFIG" - Privately ask the user for additional
	// authentication or configuration.
	//   "DIALOG" - Presents a
	// [dialog](https://developers.google.com/chat/how-tos/dialogs).
	Type string `json:"type,omitempty"`

	// Url: Input only. URL for users to authenticate or configure. (Only
	// for `REQUEST_CONFIG` response types.)
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DialogAction") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DialogAction") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ActionResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ActionResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ActionStatus: Represents the status for a request to either invoke or
// submit a dialog (https://developers.google.com/chat/how-tos/dialogs).
type ActionStatus struct {
	// StatusCode: The status code.
	//
	// Possible values:
	//   "OK" - Not an error; returned on success. HTTP Mapping: 200 OK
	//   "CANCELLED" - The operation was cancelled, typically by the caller.
	// HTTP Mapping: 499 Client Closed Request
	//   "UNKNOWN" - Unknown error. For example, this error may be returned
	// when a `Status` value received from another address space belongs to
	// an error space that is not known in this address space. Also errors
	// raised by APIs that do not return enough error information may be
	// converted to this error. HTTP Mapping: 500 Internal Server Error
	//   "INVALID_ARGUMENT" - The client specified an invalid argument. Note
	// that this differs from `FAILED_PRECONDITION`. `INVALID_ARGUMENT`
	// indicates arguments that are problematic regardless of the state of
	// the system (e.g., a malformed file name). HTTP Mapping: 400 Bad
	// Request
	//   "DEADLINE_EXCEEDED" - The deadline expired before the operation
	// could complete. For operations that change the state of the system,
	// this error may be returned even if the operation has completed
	// successfully. For example, a successful response from a server could
	// have been delayed long enough for the deadline to expire. HTTP
	// Mapping: 504 Gateway Timeout
	//   "NOT_FOUND" - Some requested entity (e.g., file or directory) was
	// not found. Note to server developers: if a request is denied for an
	// entire class of users, such as gradual feature rollout or
	// undocumented allowlist, `NOT_FOUND` may be used. If a request is
	// denied for some users within a class of users, such as user-based
	// access control, `PERMISSION_DENIED` must be used. HTTP Mapping: 404
	// Not Found
	//   "ALREADY_EXISTS" - The entity that a client attempted to create
	// (e.g., file or directory) already exists. HTTP Mapping: 409 Conflict
	//   "PERMISSION_DENIED" - The caller does not have permission to
	// execute the specified operation. `PERMISSION_DENIED` must not be used
	// for rejections caused by exhausting some resource (use
	// `RESOURCE_EXHAUSTED` instead for those errors). `PERMISSION_DENIED`
	// must not be used if the caller can not be identified (use
	// `UNAUTHENTICATED` instead for those errors). This error code does not
	// imply the request is valid or the requested entity exists or
	// satisfies other pre-conditions. HTTP Mapping: 403 Forbidden
	//   "UNAUTHENTICATED" - The request does not have valid authentication
	// credentials for the operation. HTTP Mapping: 401 Unauthorized
	//   "RESOURCE_EXHAUSTED" - Some resource has been exhausted, perhaps a
	// per-user quota, or perhaps the entire file system is out of space.
	// HTTP Mapping: 429 Too Many Requests
	//   "FAILED_PRECONDITION" - The operation was rejected because the
	// system is not in a state required for the operation's execution. For
	// example, the directory to be deleted is non-empty, an rmdir operation
	// is applied to a non-directory, etc. Service implementors can use the
	// following guidelines to decide between `FAILED_PRECONDITION`,
	// `ABORTED`, and `UNAVAILABLE`: (a) Use `UNAVAILABLE` if the client can
	// retry just the failing call. (b) Use `ABORTED` if the client should
	// retry at a higher level. For example, when a client-specified
	// test-and-set fails, indicating the client should restart a
	// read-modify-write sequence. (c) Use `FAILED_PRECONDITION` if the
	// client should not retry until the system state has been explicitly
	// fixed. For example, if an "rmdir" fails because the directory is
	// non-empty, `FAILED_PRECONDITION` should be returned since the client
	// should not retry unless the files are deleted from the directory.
	// HTTP Mapping: 400 Bad Request
	//   "ABORTED" - The operation was aborted, typically due to a
	// concurrency issue such as a sequencer check failure or transaction
	// abort. See the guidelines above for deciding between
	// `FAILED_PRECONDITION`, `ABORTED`, and `UNAVAILABLE`. HTTP Mapping:
	// 409 Conflict
	//   "OUT_OF_RANGE" - The operation was attempted past the valid range.
	// E.g., seeking or reading past end-of-file. Unlike `INVALID_ARGUMENT`,
	// this error indicates a problem that may be fixed if the system state
	// changes. For example, a 32-bit file system will generate
	// `INVALID_ARGUMENT` if asked to read at an offset that is not in the
	// range [0,2^32-1], but it will generate `OUT_OF_RANGE` if asked to
	// read from an offset past the current file size. There is a fair bit
	// of overlap between `FAILED_PRECONDITION` and `OUT_OF_RANGE`. We
	// recommend using `OUT_OF_RANGE` (the more specific error) when it
	// applies so that callers who are iterating through a space can easily
	// look for an `OUT_OF_RANGE` error to detect when they are done. HTTP
	// Mapping: 400 Bad Request
	//   "UNIMPLEMENTED" - The operation is not implemented or is not
	// supported/enabled in this service. HTTP Mapping: 501 Not Implemented
	//   "INTERNAL" - Internal errors. This means that some invariants
	// expected by the underlying system have been broken. This error code
	// is reserved for serious errors. HTTP Mapping: 500 Internal Server
	// Error
	//   "UNAVAILABLE" - The service is currently unavailable. This is most
	// likely a transient condition, which can be corrected by retrying with
	// a backoff. Note that it is not always safe to retry non-idempotent
	// operations. See the guidelines above for deciding between
	// `FAILED_PRECONDITION`, `ABORTED`, and `UNAVAILABLE`. HTTP Mapping:
	// 503 Service Unavailable
	//   "DATA_LOSS" - Unrecoverable data loss or corruption. HTTP Mapping:
	// 500 Internal Server Error
	StatusCode string `json:"statusCode,omitempty"`

	// UserFacingMessage: The message to send users about the status of
	// their request. If unset, a generic message based on the `status_code`
	// is sent.
	UserFacingMessage string `json:"userFacingMessage,omitempty"`

	// ForceSendFields is a list of field names (e.g. "StatusCode") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "StatusCode") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ActionStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ActionStatus
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Annotation: Output only. Annotations associated with the plain-text
// body of the message. To add basic formatting to a text message, see
// Format text messages
// (https://developers.google.com/chat/format-messages). Example
// plain-text message body: ``` Hello @FooBot how are you!" ``` The
// corresponding annotations metadata: ``` "annotations":[{
// "type":"USER_MENTION", "startIndex":6, "length":7, "userMention": {
// "user": { "name":"users/{user}", "displayName":"FooBot",
// "avatarUrl":"https://goo.gl/aeDtrS", "type":"BOT" }, "type":"MENTION"
// } }] ```
type Annotation struct {
	// Length: Length of the substring in the plain-text message body this
	// annotation corresponds to.
	Length int64 `json:"length,omitempty"`

	// SlashCommand: The metadata for a slash command.
	SlashCommand *SlashCommandMetadata `json:"slashCommand,omitempty"`

	// StartIndex: Start index (0-based, inclusive) in the plain-text
	// message body this annotation corresponds to.
	StartIndex int64 `json:"startIndex,omitempty"`

	// Type: The type of this annotation.
	//
	// Possible values:
	//   "ANNOTATION_TYPE_UNSPECIFIED" - Default value for the enum. Don't
	// use.
	//   "USER_MENTION" - A user is mentioned.
	//   "SLASH_COMMAND" - A slash command is invoked.
	Type string `json:"type,omitempty"`

	// UserMention: The metadata of user mention.
	UserMention *UserMentionMetadata `json:"userMention,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Length") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Length") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Annotation) MarshalJSON() ([]byte, error) {
	type NoMethod Annotation
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// AttachedGif: A GIF image that's specified by a URL.
type AttachedGif struct {
	// Uri: Output only. The URL that hosts the GIF image.
	Uri string `json:"uri,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Uri") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Uri") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *AttachedGif) MarshalJSON() ([]byte, error) {
	type NoMethod AttachedGif
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Attachment: An attachment in Google Chat.
type Attachment struct {
	// AttachmentDataRef: A reference to the attachment data. This field is
	// used with the media API to download the attachment data.
	AttachmentDataRef *AttachmentDataRef `json:"attachmentDataRef,omitempty"`

	// ContentName: Output only. The original file name for the content, not
	// the full path.
	ContentName string `json:"contentName,omitempty"`

	// ContentType: Output only. The content type (MIME type) of the file.
	ContentType string `json:"contentType,omitempty"`

	// DownloadUri: Output only. The download URL which should be used to
	// allow a human user to download the attachment. Chat apps shouldn't
	// use this URL to download attachment content.
	DownloadUri string `json:"downloadUri,omitempty"`

	// DriveDataRef: Output only. A reference to the Google Drive
	// attachment. This field is used with the Google Drive API.
	DriveDataRef *DriveDataRef `json:"driveDataRef,omitempty"`

	// Name: Resource name of the attachment, in the form
	// `spaces/*/messages/*/attachments/*`.
	Name string `json:"name,omitempty"`

	// Source: Output only. The source of the attachment.
	//
	// Possible values:
	//   "SOURCE_UNSPECIFIED"
	//   "DRIVE_FILE"
	//   "UPLOADED_CONTENT"
	Source string `json:"source,omitempty"`

	// ThumbnailUri: Output only. The thumbnail URL which should be used to
	// preview the attachment to a human user. Chat apps shouldn't use this
	// URL to download attachment content.
	ThumbnailUri string `json:"thumbnailUri,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AttachmentDataRef")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AttachmentDataRef") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Attachment) MarshalJSON() ([]byte, error) {
	type NoMethod Attachment
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type AttachmentDataRef struct {
	// AttachmentUploadToken: Opaque token containing a reference to an
	// uploaded attachment. Treated by clients as an opaque string and used
	// to create or update Chat messages with attachments.
	AttachmentUploadToken string `json:"attachmentUploadToken,omitempty"`

	// ResourceName: The resource name of the attachment data. This field is
	// used with the media API to download the attachment data.
	ResourceName string `json:"resourceName,omitempty"`

	// ForceSendFields is a list of field names (e.g.
	// "AttachmentUploadToken") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. However, any non-pointer, non-interface field appearing in
	// ForceSendFields will be sent to the server regardless of whether the
	// field is empty or not. This may be used to include empty fields in
	// Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AttachmentUploadToken") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *AttachmentDataRef) MarshalJSON() ([]byte, error) {
	type NoMethod AttachmentDataRef
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Button: A button. Can be a text button or an image button.
type Button struct {
	// ImageButton: A button with image and `onclick` action.
	ImageButton *ImageButton `json:"imageButton,omitempty"`

	// TextButton: A button with text and `onclick` action.
	TextButton *TextButton `json:"textButton,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ImageButton") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ImageButton") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Button) MarshalJSON() ([]byte, error) {
	type NoMethod Button
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Card: A card is a UI element that can contain UI widgets such as text
// and images.
type Card struct {
	// CardActions: The actions of this card.
	CardActions []*CardAction `json:"cardActions,omitempty"`

	// Header: The header of the card. A header usually contains a title and
	// an image.
	Header *CardHeader `json:"header,omitempty"`

	// Name: Name of the card.
	Name string `json:"name,omitempty"`

	// Sections: Sections are separated by a line divider.
	Sections []*Section `json:"sections,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CardActions") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CardActions") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Card) MarshalJSON() ([]byte, error) {
	type NoMethod Card
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// CardAction: A card action is the action associated with the card. For
// an invoice card, a typical action would be: delete invoice, email
// invoice or open the invoice in browser. Not supported by Google Chat
// apps.
type CardAction struct {
	// ActionLabel: The label used to be displayed in the action menu item.
	ActionLabel string `json:"actionLabel,omitempty"`

	// OnClick: The onclick action for this action item.
	OnClick *OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CardAction) MarshalJSON() ([]byte, error) {
	type NoMethod CardAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type CardHeader struct {
	// ImageStyle: The image's type (for example, square border or circular
	// border).
	//
	// Possible values:
	//   "IMAGE_STYLE_UNSPECIFIED"
	//   "IMAGE" - Square border.
	//   "AVATAR" - Circular border.
	ImageStyle string `json:"imageStyle,omitempty"`

	// ImageUrl: The URL of the image in the card header.
	ImageUrl string `json:"imageUrl,omitempty"`

	// Subtitle: The subtitle of the card header.
	Subtitle string `json:"subtitle,omitempty"`

	// Title: The title must be specified. The header has a fixed height: if
	// both a title and subtitle is specified, each takes up one line. If
	// only the title is specified, it takes up both lines.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ImageStyle") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ImageStyle") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CardHeader) MarshalJSON() ([]byte, error) {
	type NoMethod CardHeader
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// CardWithId: A card
// (https://developers.google.com/chat/api/reference/rest/v1/cards) in a
// Google Chat message. Only Chat apps can create cards. If your Chat
// app authenticates as a user
// (https://developers.google.com/chat/api/guides/auth/users), the
// message can't contain cards.
type CardWithId struct {
	// Card: A card. Maximum size is 32 KB.
	Card *GoogleAppsCardV1Card `json:"card,omitempty"`

	// CardId: Required if the message contains multiple cards. A unique
	// identifier for a card in a message.
	CardId string `json:"cardId,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Card") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Card") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CardWithId) MarshalJSON() ([]byte, error) {
	type NoMethod CardWithId
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ChatAppLogEntry: JSON payload of error messages. If the Cloud Logging
// API is enabled, these error messages are logged to Google Cloud
// Logging (https://cloud.google.com/logging/docs).
type ChatAppLogEntry struct {
	// Deployment: The deployment that caused the error. For Chat apps built
	// in Apps Script, this is the deployment ID defined by Apps Script.
	Deployment string `json:"deployment,omitempty"`

	// DeploymentFunction: The unencrypted `callback_method` name that was
	// running when the error was encountered.
	DeploymentFunction string `json:"deploymentFunction,omitempty"`

	// Error: The error code and message.
	Error *Status `json:"error,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Deployment") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Deployment") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ChatAppLogEntry) MarshalJSON() ([]byte, error) {
	type NoMethod ChatAppLogEntry
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ChatClientDataSourceMarkup: Chat apps only. For a `SelectionInput`
// widget that uses a multi-select menu, a data source from Google Chat.
// For example, a list of Google Chat spaces of which the user is a
// member. Developer Preview
// (https://developers.google.com/workspace/preview).
type ChatClientDataSourceMarkup struct {
	// SpaceDataSource: A data source representing a Google Chat space.
	// Format: spaces/{space} Developer Preview
	// (https://developers.google.com/workspace/preview).
	SpaceDataSource *SpaceDataSource `json:"spaceDataSource,omitempty"`

	// ForceSendFields is a list of field names (e.g. "SpaceDataSource") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "SpaceDataSource") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *ChatClientDataSourceMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod ChatClientDataSourceMarkup
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Color: Represents a color in the RGBA color space. This
// representation is designed for simplicity of conversion to and from
// color representations in various languages over compactness. For
// example, the fields of this representation can be trivially provided
// to the constructor of `java.awt.Color` in Java; it can also be
// trivially provided to UIColor's `+colorWithRed:green:blue:alpha`
// method in iOS; and, with just a little work, it can be easily
// formatted into a CSS `rgba()` string in JavaScript. This reference
// page doesn't have information about the absolute color space that
// should be used to interpret the RGB value—for example, sRGB, Adobe
// RGB, DCI-P3, and BT.2020. By default, applications should assume the
// sRGB color space. When color equality needs to be decided,
// implementations, unless documented otherwise, treat two colors as
// equal if all their red, green, blue, and alpha values each differ by
// at most `1e-5`. Example (Java): import com.google.type.Color; // ...
// public static java.awt.Color fromProto(Color protocolor) { float
// alpha = protocolor.hasAlpha() ? protocolor.getAlpha().getValue() :
// 1.0; return new java.awt.Color( protocolor.getRed(),
// protocolor.getGreen(), protocolor.getBlue(), alpha); } public static
// Color toProto(java.awt.Color color) { float red = (float)
// color.getRed(); float green = (float) color.getGreen(); float blue =
// (float) color.getBlue(); float denominator = 255.0; Color.Builder
// resultBuilder = Color .newBuilder() .setRed(red / denominator)
// .setGreen(green / denominator) .setBlue(blue / denominator); int
// alpha = color.getAlpha(); if (alpha != 255) { result.setAlpha(
// FloatValue .newBuilder() .setValue(((float) alpha) / denominator)
// .build()); } return resultBuilder.build(); } // ... Example (iOS /
// Obj-C): // ... static UIColor* fromProto(Color* protocolor) { float
// red = [protocolor red]; float green = [protocolor green]; float blue
// = [protocolor blue]; FloatValue* alpha_wrapper = [protocolor alpha];
// float alpha = 1.0; if (alpha_wrapper != nil) { alpha = [alpha_wrapper
// value]; } return [UIColor colorWithRed:red green:green blue:blue
// alpha:alpha]; } static Color* toProto(UIColor* color) { CGFloat red,
// green, blue, alpha; if (![color getRed:&red green:&green blue:&blue
// alpha:&alpha]) { return nil; } Color* result = [[Color alloc] init];
// [result setRed:red]; [result setGreen:green]; [result setBlue:blue];
// if (alpha <= 0.9999) { [result
// setAlpha:floatWrapperWithValue(alpha)]; } [result autorelease];
// return result; } // ... Example (JavaScript): // ... var
// protoToCssColor = function(rgb_color) { var redFrac = rgb_color.red
// || 0.0; var greenFrac = rgb_color.green || 0.0; var blueFrac =
// rgb_color.blue || 0.0; var red = Math.floor(redFrac * 255); var green
// = Math.floor(greenFrac * 255); var blue = Math.floor(blueFrac * 255);
// if (!('alpha' in rgb_color)) { return rgbToCssColor(red, green,
// blue); } var alphaFrac = rgb_color.alpha.value || 0.0; var rgbParams
// = [red, green, blue].join(','); return ['rgba(', rgbParams, ',',
// alphaFrac, ')'].join(”); }; var rgbToCssColor = function(red, green,
// blue) { var rgbNumber = new Number((red << 16) | (green << 8) |
// blue); var hexString = rgbNumber.toString(16); var missingZeros = 6 -
// hexString.length; var resultBuilder = ['#']; for (var i = 0; i <
// missingZeros; i++) { resultBuilder.push('0'); }
// resultBuilder.push(hexString); return resultBuilder.join(”); }; //
// ...
type Color struct {
	// Alpha: The fraction of this color that should be applied to the
	// pixel. That is, the final pixel color is defined by the equation:
	// `pixel color = alpha * (this color) + (1.0 - alpha) * (background
	// color)` This means that a value of 1.0 corresponds to a solid color,
	// whereas a value of 0.0 corresponds to a completely transparent color.
	// This uses a wrapper message rather than a simple float scalar so that
	// it is possible to distinguish between a default value and the value
	// being unset. If omitted, this color object is rendered as a solid
	// color (as if the alpha value had been explicitly given a value of
	// 1.0).
	Alpha float64 `json:"alpha,omitempty"`

	// Blue: The amount of blue in the color as a value in the interval [0,
	// 1].
	Blue float64 `json:"blue,omitempty"`

	// Green: The amount of green in the color as a value in the interval
	// [0, 1].
	Green float64 `json:"green,omitempty"`

	// Red: The amount of red in the color as a value in the interval [0,
	// 1].
	Red float64 `json:"red,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Alpha") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Alpha") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Color) MarshalJSON() ([]byte, error) {
	type NoMethod Color
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *Color) UnmarshalJSON(data []byte) error {
	type NoMethod Color
	var s1 struct {
		Alpha gensupport.JSONFloat64 `json:"alpha"`
		Blue  gensupport.JSONFloat64 `json:"blue"`
		Green gensupport.JSONFloat64 `json:"green"`
		Red   gensupport.JSONFloat64 `json:"red"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Alpha = float64(s1.Alpha)
	s.Blue = float64(s1.Blue)
	s.Green = float64(s1.Green)
	s.Red = float64(s1.Red)
	return nil
}

// CommonEventObject: Represents information about the user's client,
// such as locale, host app, and platform. For Chat apps,
// `CommonEventObject` includes data submitted by users interacting with
// cards, like data entered in dialogs
// (https://developers.google.com/chat/how-tos/dialogs).
type CommonEventObject struct {
	// FormInputs: A map containing the current values of the widgets in a
	// card. The map keys are the string IDs assigned to each widget, and
	// the values represent inputs to the widget. Depending on the input
	// data type, a different object represents each input: For single-value
	// widgets, `StringInput`. For multi-value widgets, an array of
	// `StringInput` objects. For a date-time picker, a `DateTimeInput`. For
	// a date-only picker, a `DateInput`. For a time-only picker, a
	// `TimeInput`. Corresponds with the data entered by a user on a card in
	// a dialog (https://developers.google.com/chat/how-tos/dialogs).
	FormInputs map[string]Inputs `json:"formInputs,omitempty"`

	// HostApp: The hostApp enum which indicates the app the add-on is
	// invoked from. Always `CHAT` for Chat apps.
	//
	// Possible values:
	//   "UNSPECIFIED_HOST_APP" - Google can't identify a host app.
	//   "GMAIL" - The add-on launches from Gmail.
	//   "CALENDAR" - The add-on launches from Google Calendar.
	//   "DRIVE" - The add-on launches from Google Drive.
	//   "DEMO" - Not used.
	//   "DOCS" - The add-on launches from Google Docs.
	//   "MEET" - The add-on launches from Google Meet.
	//   "SHEETS" - The add-on launches from Google Sheets.
	//   "SLIDES" - The add-on launches from Google Slides.
	//   "DRAWINGS" - The add-on launches from Google Drawings.
	//   "CHAT" - A Google Chat app. Not used for Google Workspace Add-ons.
	HostApp string `json:"hostApp,omitempty"`

	// InvokedFunction: Name of the invoked function associated with the
	// widget. Only set for Chat apps.
	InvokedFunction string `json:"invokedFunction,omitempty"`

	// Parameters: Custom parameters
	// (/chat/api/reference/rest/v1/cards#ActionParameter) passed to the
	// invoked function. Both keys and values must be strings.
	Parameters map[string]string `json:"parameters,omitempty"`

	// Platform: The platform enum which indicates the platform where the
	// event originates (`WEB`, `IOS`, or `ANDROID`). Not supported by Chat
	// apps.
	//
	// Possible values:
	//   "UNKNOWN_PLATFORM"
	//   "WEB"
	//   "IOS"
	//   "ANDROID"
	Platform string `json:"platform,omitempty"`

	// TimeZone: The timezone ID and offset from Coordinated Universal Time
	// (UTC). Only supported for the event types `CARD_CLICKED`
	// (https://developers.google.com/chat/api/reference/rest/v1/EventType#ENUM_VALUES.CARD_CLICKED)
	// and `SUBMIT_DIALOG`
	// (https://developers.google.com/chat/api/reference/rest/v1/DialogEventType#ENUM_VALUES.SUBMIT_DIALOG).
	TimeZone *TimeZone `json:"timeZone,omitempty"`

	// UserLocale: The full `locale.displayName` in the format of [ISO 639
	// language code]-[ISO 3166 country/region code] such as "en-US".
	UserLocale string `json:"userLocale,omitempty"`

	// ForceSendFields is a list of field names (e.g. "FormInputs") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "FormInputs") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CommonEventObject) MarshalJSON() ([]byte, error) {
	type NoMethod CommonEventObject
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// CustomEmoji: Represents a custom emoji.
type CustomEmoji struct {
	// Uid: Unique key for the custom emoji resource.
	Uid string `json:"uid,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Uid") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Uid") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CustomEmoji) MarshalJSON() ([]byte, error) {
	type NoMethod CustomEmoji
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DateInput: Date input values.
type DateInput struct {
	// MsSinceEpoch: Time since epoch time, in milliseconds.
	MsSinceEpoch int64 `json:"msSinceEpoch,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "MsSinceEpoch") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "MsSinceEpoch") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DateInput) MarshalJSON() ([]byte, error) {
	type NoMethod DateInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DateTimeInput: Date and time input values.
type DateTimeInput struct {
	// HasDate: Whether the `datetime` input includes a calendar date.
	HasDate bool `json:"hasDate,omitempty"`

	// HasTime: Whether the `datetime` input includes a timestamp.
	HasTime bool `json:"hasTime,omitempty"`

	// MsSinceEpoch: Time since epoch time, in milliseconds.
	MsSinceEpoch int64 `json:"msSinceEpoch,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "HasDate") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "HasDate") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DateTimeInput) MarshalJSON() ([]byte, error) {
	type NoMethod DateTimeInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DeletionMetadata: Information about a deleted message. A message is
// deleted when `delete_time` is set.
type DeletionMetadata struct {
	// DeletionType: Indicates who deleted the message.
	//
	// Possible values:
	//   "DELETION_TYPE_UNSPECIFIED" - This value is unused.
	//   "CREATOR" - User deleted their own message.
	//   "SPACE_OWNER" - The space owner deleted the message.
	//   "ADMIN" - A Google Workspace admin deleted the message.
	//   "APP_MESSAGE_EXPIRY" - A Chat app deleted its own message when it
	// expired.
	//   "CREATOR_VIA_APP" - A Chat app deleted the message on behalf of the
	// user.
	//   "SPACE_OWNER_VIA_APP" - A Chat app deleted the message on behalf of
	// the space owner.
	DeletionType string `json:"deletionType,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DeletionType") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DeletionType") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DeletionMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod DeletionMetadata
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DeprecatedEvent: A Google Chat app interaction event. To learn about
// interaction events, see Receive and respond to interactions with your
// Google Chat app
// (https://developers.google.com/chat/api/guides/message-formats). To
// learn about event types and for example event payloads, see Types of
// Google Chat app interaction events
// (https://developers.google.com/chat/api/guides/message-formats/events).
type DeprecatedEvent struct {
	// Action: For `CARD_CLICKED` interaction events, the form action data
	// associated when a user clicks a card or dialog. To learn more, see
	// Read form data input by users on cards
	// (https://developers.google.com/chat/ui/read-form-data).
	Action *FormAction `json:"action,omitempty"`

	// Common: Represents information about the user's client, such as
	// locale, host app, and platform. For Chat apps, `CommonEventObject`
	// includes information submitted by users interacting with dialogs
	// (https://developers.google.com/chat/how-tos/dialogs), like data
	// entered on a card.
	Common *CommonEventObject `json:"common,omitempty"`

	// ConfigCompleteRedirectUrl: The URL the Chat app should redirect the
	// user to after they have completed an authorization or configuration
	// flow outside of Google Chat. For more information, see Connect a Chat
	// app with other services & tools
	// (https://developers.google.com/chat/how-tos/connect-web-services-tools).
	ConfigCompleteRedirectUrl string `json:"configCompleteRedirectUrl,omitempty"`

	// DialogEventType: The type of dialog
	// (https://developers.google.com/chat/how-tos/dialogs) interaction
	// event received.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value. Unspecified.
	//   "REQUEST_DIALOG" - A user opens a dialog.
	//   "SUBMIT_DIALOG" - A user clicks an interactive element of a dialog.
	// For example, a user fills out information in a dialog and clicks a
	// button to submit the information.
	//   "CANCEL_DIALOG" - A user closes a dialog without submitting
	// information, or the dialog is canceled.
	DialogEventType string `json:"dialogEventType,omitempty"`

	// EventTime: The timestamp indicating when the interaction event
	// occurred.
	EventTime string `json:"eventTime,omitempty"`

	// IsDialogEvent: For `CARD_CLICKED` interaction events, whether the
	// user interacted with a dialog
	// (https://developers.google.com/chat/how-tos/dialogs).
	IsDialogEvent bool `json:"isDialogEvent,omitempty"`

	// Message: The message that triggered the interaction event, if
	// applicable.
	Message *Message `json:"message,omitempty"`

	// Space: The space in which the interaction event occurred.
	Space *Space `json:"space,omitempty"`

	// ThreadKey: The Chat app-defined key for the thread related to the
	// interaction event. See `spaces.messages.thread.threadKey`
	// (/chat/api/reference/rest/v1/spaces.messages#Thread.FIELDS.thread_key)
	//  for more information.
	ThreadKey string `json:"threadKey,omitempty"`

	// Token: A secret value that legacy Chat apps can use to verify if a
	// request is from Google. Google randomly generates the token, and its
	// value remains static. You can obtain, revoke, or regenerate the token
	// from the Chat API configuration page
	// (https://console.cloud.google.com/apis/api/chat.googleapis.com/hangouts-chat)
	// in the Google Cloud Console. Modern Chat apps don't use this field.
	// It is absent from API responses and the Chat API configuration page
	// (https://console.cloud.google.com/apis/api/chat.googleapis.com/hangouts-chat).
	Token string `json:"token,omitempty"`

	// Type: The type of interaction event. For details, see Types of Google
	// Chat app interaction events
	// (https://developers.google.com/chat/api/guides/message-formats/events).
	//
	// Possible values:
	//   "UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "MESSAGE" - A user sends the Chat app a message, or invokes the
	// Chat app in a space.
	//   "ADDED_TO_SPACE" - A user adds the Chat app to a space, or a Google
	// Workspace administrator installs the Chat app in direct message
	// spaces for users in their organization.
	//   "REMOVED_FROM_SPACE" - A user removes the Chat app from a space.
	//   "CARD_CLICKED" - A user clicks an interactive element of a card or
	// dialog from a Chat app, such as a button. If a user interacts with a
	// dialog, the `CARD_CLICKED` interaction event's `isDialogEvent` field
	// is set to `true` and includes a
	// [`DialogEventType`](https://developers.google.com/chat/api/reference/r
	// est/v1/DialogEventType).
	Type string `json:"type,omitempty"`

	// User: The user that triggered the interaction event.
	User *User `json:"user,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Action") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DeprecatedEvent) MarshalJSON() ([]byte, error) {
	type NoMethod DeprecatedEvent
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Dialog: Wrapper around the card body of the dialog.
type Dialog struct {
	// Body: Input only. Body of the dialog, which is rendered in a modal.
	// Google Chat apps don't support the following card entities:
	// `DateTimePicker`, `OnChangeAction`.
	Body *GoogleAppsCardV1Card `json:"body,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Body") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Body") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Dialog) MarshalJSON() ([]byte, error) {
	type NoMethod Dialog
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DialogAction: Contains a dialog
// (https://developers.google.com/chat/how-tos/dialogs) and request
// status code.
type DialogAction struct {
	// ActionStatus: Input only. Status for a request to either invoke or
	// submit a dialog (https://developers.google.com/chat/how-tos/dialogs).
	// Displays a status and message to users, if necessary. For example, in
	// case of an error or success.
	ActionStatus *ActionStatus `json:"actionStatus,omitempty"`

	// Dialog: Input only. Dialog
	// (https://developers.google.com/chat/how-tos/dialogs) for the request.
	Dialog *Dialog `json:"dialog,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionStatus") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionStatus") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DialogAction) MarshalJSON() ([]byte, error) {
	type NoMethod DialogAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DriveDataRef: A reference to the data of a drive attachment.
type DriveDataRef struct {
	// DriveFileId: The ID for the drive file. Use with the Drive API.
	DriveFileId string `json:"driveFileId,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DriveFileId") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DriveFileId") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DriveDataRef) MarshalJSON() ([]byte, error) {
	type NoMethod DriveDataRef
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Emoji: An emoji that is used as a reaction to a message.
type Emoji struct {
	// CustomEmoji: Output only. A custom emoji.
	CustomEmoji *CustomEmoji `json:"customEmoji,omitempty"`

	// Unicode: A basic emoji represented by a unicode string.
	Unicode string `json:"unicode,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CustomEmoji") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CustomEmoji") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Emoji) MarshalJSON() ([]byte, error) {
	type NoMethod Emoji
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// EmojiReactionSummary: The number of people who reacted to a message
// with a specific emoji.
type EmojiReactionSummary struct {
	// Emoji: Emoji associated with the reactions.
	Emoji *Emoji `json:"emoji,omitempty"`

	// ReactionCount: The total number of reactions using the associated
	// emoji.
	ReactionCount int64 `json:"reactionCount,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Emoji") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Emoji") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *EmojiReactionSummary) MarshalJSON() ([]byte, error) {
	type NoMethod EmojiReactionSummary
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use
// it as the request or the response type of an API method. For
// instance: service Foo { rpc Bar(google.protobuf.Empty) returns
// (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`
}

// FormAction: A form action describes the behavior when the form is
// submitted. For example, you can invoke Apps Script to handle the
// form.
type FormAction struct {
	// ActionMethodName: The method name is used to identify which part of
	// the form triggered the form submission. This information is echoed
	// back to the Chat app as part of the card click event. You can use the
	// same method name for several elements that trigger a common behavior.
	ActionMethodName string `json:"actionMethodName,omitempty"`

	// Parameters: List of action parameters.
	Parameters []*ActionParameter `json:"parameters,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionMethodName") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionMethodName") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *FormAction) MarshalJSON() ([]byte, error) {
	type NoMethod FormAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Action: An action that describes the behavior when
// the form is submitted. For example, you can invoke an Apps Script
// script to handle the form. If the action is triggered, the form
// values are sent to the server.
type GoogleAppsCardV1Action struct {
	// Function: A custom function to invoke when the containing element is
	// clicked or othrwise activated. For example usage, see Create
	// interactive cards
	// (https://developers.google.com/chat/how-tos/cards-onclick).
	Function string `json:"function,omitempty"`

	// Interaction: Optional. Required when opening a dialog
	// (https://developers.google.com/chat/how-tos/dialogs). What to do in
	// response to an interaction with a user, such as a user clicking a
	// button in a card message. If unspecified, the app responds by
	// executing an `action`—like opening a link or running a
	// function—as normal. By specifying an `interaction`, the app can
	// respond in special interactive ways. For example, by setting
	// `interaction` to `OPEN_DIALOG`, the app can open a dialog
	// (https://developers.google.com/chat/how-tos/dialogs). When specified,
	// a loading indicator isn't shown. Supported by Chat apps, but not
	// Google Workspace Add-ons. If specified for an add-on, the entire card
	// is stripped and nothing is shown in the client.
	//
	// Possible values:
	//   "INTERACTION_UNSPECIFIED" - Default value. The `action` executes as
	// normal.
	//   "OPEN_DIALOG" - Opens a
	// [dialog](https://developers.google.com/chat/how-tos/dialogs), a
	// windowed, card-based interface that Chat apps use to interact with
	// users. Only supported by Chat apps in response to button-clicks on
	// card messages. Not supported by Google Workspace Add-ons. If
	// specified for an add-on, the entire card is stripped and nothing is
	// shown in the client.
	Interaction string `json:"interaction,omitempty"`

	// LoadIndicator: Specifies the loading indicator that the action
	// displays while making the call to the action.
	//
	// Possible values:
	//   "SPINNER" - Displays a spinner to indicate that content is loading.
	//   "NONE" - Nothing is displayed.
	LoadIndicator string `json:"loadIndicator,omitempty"`

	// Parameters: List of action parameters.
	Parameters []*GoogleAppsCardV1ActionParameter `json:"parameters,omitempty"`

	// PersistValues: Indicates whether form values persist after the
	// action. The default value is `false`. If `true`, form values remain
	// after the action is triggered. To let the user make changes while the
	// action is being processed, set `LoadIndicator`
	// (https://developers.google.com/workspace/add-ons/reference/rpc/google.apps.card.v1#loadindicator)
	// to `NONE`. For card messages
	// (https://developers.google.com/chat/api/guides/v1/messages/create#create)
	// in Chat apps, you must also set the action's `ResponseType`
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces.messages#responsetype)
	// to `UPDATE_MESSAGE` and use the same `card_id`
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces.messages#CardWithId)
	// from the card that contained the action. If `false`, the form values
	// are cleared when the action is triggered. To prevent the user from
	// making changes while the action is being processed, set
	// `LoadIndicator`
	// (https://developers.google.com/workspace/add-ons/reference/rpc/google.apps.card.v1#loadindicator)
	// to `SPINNER`.
	PersistValues bool `json:"persistValues,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Function") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Function") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Action) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Action
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ActionParameter: List of string parameters to supply
// when the action method is invoked. For example, consider three snooze
// buttons: snooze now, snooze one day, or snooze next week. You might
// use `action method = snooze()`, passing the snooze type and snooze
// time in the list of string parameters. To learn more, see
// `CommonEventObject`
// (https://developers.google.com/chat/api/reference/rest/v1/Event#commoneventobject).
type GoogleAppsCardV1ActionParameter struct {
	// Key: The name of the parameter for the action script.
	Key string `json:"key,omitempty"`

	// Value: The value of the parameter.
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Key") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Key") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ActionParameter) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ActionParameter
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1BorderStyle: The style options for the border of a
// card or widget, including the border type and color.
type GoogleAppsCardV1BorderStyle struct {
	// CornerRadius: The corner radius for the border.
	CornerRadius int64 `json:"cornerRadius,omitempty"`

	// StrokeColor: The colors to use when the type is `BORDER_TYPE_STROKE`.
	StrokeColor *Color `json:"strokeColor,omitempty"`

	// Type: The border type.
	//
	// Possible values:
	//   "BORDER_TYPE_UNSPECIFIED" - Don't use. Unspecified.
	//   "NO_BORDER" - Default value. No border.
	//   "STROKE" - Outline.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CornerRadius") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CornerRadius") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1BorderStyle) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1BorderStyle
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Button: A text, icon, or text and icon button that
// users can click. For an example in Google Chat apps, see Button list
// (https://developers.google.com/chat/ui/widgets/button-list). To make
// an image a clickable button, specify an `Image` (not an
// `ImageComponent`) and set an `onClick` action.
type GoogleAppsCardV1Button struct {
	// AltText: The alternative text that's used for accessibility. Set
	// descriptive text that lets users know what the button does. For
	// example, if a button opens a hyperlink, you might write: "Opens a new
	// browser tab and navigates to the Google Chat developer documentation
	// at https://developers.google.com/chat".
	AltText string `json:"altText,omitempty"`

	// Color: If set, the button is filled with a solid background color and
	// the font color changes to maintain contrast with the background
	// color. For example, setting a blue background likely results in white
	// text. If unset, the image background is white and the font color is
	// blue. For red, green, and blue, the value of each field is a `float`
	// number that you can express in either of two ways: as a number
	// between 0 and 255 divided by 255 (153/255), or as a value between 0
	// and 1 (0.6). 0 represents the absence of a color and 1 or 255/255
	// represent the full presence of that color on the RGB scale.
	// Optionally set `alpha`, which sets a level of transparency using this
	// equation: ``` pixel color = alpha * (this color) + (1.0 - alpha) *
	// (background color) ``` For `alpha`, a value of `1` corresponds with a
	// solid color, and a value of `0` corresponds with a completely
	// transparent color. For example, the following color represents a half
	// transparent red: ``` "color": { "red": 1, "green": 0, "blue": 0,
	// "alpha": 0.5 } ```
	Color *Color `json:"color,omitempty"`

	// Disabled: If `true`, the button is displayed in an inactive state and
	// doesn't respond to user actions.
	Disabled bool `json:"disabled,omitempty"`

	// Icon: The icon image. If both `icon` and `text` are set, then the
	// icon appears before the text.
	Icon *GoogleAppsCardV1Icon `json:"icon,omitempty"`

	// OnClick: Required. The action to perform when a user clicks the
	// button, such as opening a hyperlink or running a custom function.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// Text: The text displayed inside the button.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Button) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Button
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ButtonList: A list of buttons layed out horizontally.
// For an example in Google Chat apps, see Button list
// (https://developers.google.com/chat/ui/widgets/button-list).
type GoogleAppsCardV1ButtonList struct {
	// Buttons: An array of buttons.
	Buttons []*GoogleAppsCardV1Button `json:"buttons,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Buttons") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Buttons") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ButtonList) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ButtonList
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Card: A card interface displayed in a Google Chat
// message or Google Workspace Add-on. Cards support a defined layout,
// interactive UI elements like buttons, and rich media like images. Use
// cards to present detailed information, gather information from users,
// and guide users to take a next step. To learn how to build cards, see
// the following documentation: * For Google Chat apps, see Design
// dynamic, interactive, and consistent UIs with cards
// (https://developers.google.com/chat/ui). * For Google Workspace
// Add-ons, see Card-based interfaces
// (https://developers.google.com/apps-script/add-ons/concepts/cards).
// **Example: Card message for a Google Chat app** !Example contact card
// (https://developers.google.com/chat/images/card_api_reference.png) To
// create the sample card message in Google Chat, use the following
// JSON: ``` { "cardsV2": [ { "cardId": "unique-card-id", "card": {
// "header": { "title": "Sasha", "subtitle": "Software Engineer",
// "imageUrl":
// "https://developers.google.com/chat/images/quickstart-app-avatar.png",
//
//	"imageType": "CIRCLE", "imageAltText": "Avatar for Sasha", },
//
// "sections": [ { "header": "Contact Info", "collapsible": true,
// "uncollapsibleWidgetsCount": 1, "widgets": [ { "decoratedText": {
// "startIcon": { "knownIcon": "EMAIL", }, "text": "sasha@example.com",
// } }, { "decoratedText": { "startIcon": { "knownIcon": "PERSON", },
// "text": "Online", }, }, { "decoratedText": { "startIcon": {
// "knownIcon": "PHONE", }, "text": "+1 (555) 555-1234", } }, {
// "buttonList": { "buttons": [ { "text": "Share", "onClick": {
// "openLink": { "url": "https://example.com/share", } } }, { "text":
// "Edit", "onClick": { "action": { "function": "goToView",
// "parameters": [ { "key": "viewType", "value": "EDIT", } ], } } }, ],
// } }, ], }, ], }, } ], } ```
type GoogleAppsCardV1Card struct {
	// CardActions: The card's actions. Actions are added to the card's
	// toolbar menu. Because Chat app cards have no toolbar, `cardActions[]`
	// isn't supported by Chat apps. For example, the following JSON
	// constructs a card action menu with `Settings` and `Send Feedback`
	// options: ``` "card_actions": [ { "actionLabel": "Settings",
	// "onClick": { "action": { "functionName": "goToView", "parameters": [
	// { "key": "viewType", "value": "SETTING" } ], "loadIndicator":
	// "LoadIndicator.SPINNER" } } }, { "actionLabel": "Send Feedback",
	// "onClick": { "openLink": { "url": "https://example.com/feedback" } }
	// } ] ```
	CardActions []*GoogleAppsCardV1CardAction `json:"cardActions,omitempty"`

	// DisplayStyle: In Google Workspace add-ons, sets the display
	// properties of the `peekCardHeader`. Not supported by Chat apps.
	//
	// Possible values:
	//   "DISPLAY_STYLE_UNSPECIFIED" - Don't use. Unspecified.
	//   "PEEK" - The header of the card appears at the bottom of the
	// sidebar, partially covering the current top card of the stack.
	// Clicking the header pops the card into the card stack. If the card
	// has no header, a generated header is used instead.
	//   "REPLACE" - Default value. The card is shown by replacing the view
	// of the top card in the card stack.
	DisplayStyle string `json:"displayStyle,omitempty"`

	// FixedFooter: The fixed footer shown at the bottom of this card.
	// Setting `fixedFooter` without specifying a `primaryButton` or a
	// `secondaryButton` causes an error. Supported by Google Workspace
	// Add-ons and Chat apps. For Chat apps, you can use fixed footers in
	// dialogs (https://developers.google.com/chat/how-tos/dialogs), but not
	// card messages
	// (https://developers.google.com/chat/api/guides/v1/messages/create#create).
	FixedFooter *GoogleAppsCardV1CardFixedFooter `json:"fixedFooter,omitempty"`

	// Header: The header of the card. A header usually contains a leading
	// image and a title. Headers always appear at the top of a card.
	Header *GoogleAppsCardV1CardHeader `json:"header,omitempty"`

	// Name: Name of the card. Used as a card identifier in card navigation.
	// Because Chat apps don't support card navigation, they ignore this
	// field.
	Name string `json:"name,omitempty"`

	// PeekCardHeader: When displaying contextual content, the peek card
	// header acts as a placeholder so that the user can navigate forward
	// between the homepage cards and the contextual cards. Not supported by
	// Chat apps.
	PeekCardHeader *GoogleAppsCardV1CardHeader `json:"peekCardHeader,omitempty"`

	// SectionDividerStyle: The divider style between sections.
	//
	// Possible values:
	//   "DIVIDER_STYLE_UNSPECIFIED" - Don't use. Unspecified.
	//   "SOLID_DIVIDER" - Default option. Render a solid divider between
	// sections.
	//   "NO_DIVIDER" - If set, no divider is rendered between sections.
	SectionDividerStyle string `json:"sectionDividerStyle,omitempty"`

	// Sections: Contains a collection of widgets. Each section has its own,
	// optional header. Sections are visually separated by a line divider.
	// For an example in Google Chat apps, see Card section
	// (https://developers.google.com/chat/ui/widgets/card-section).
	Sections []*GoogleAppsCardV1Section `json:"sections,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CardActions") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CardActions") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Card) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Card
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardAction: A card action is the action associated
// with the card. For example, an invoice card might include actions
// such as delete invoice, email invoice, or open the invoice in a
// browser. Not supported by Chat apps.
type GoogleAppsCardV1CardAction struct {
	// ActionLabel: The label that displays as the action menu item.
	ActionLabel string `json:"actionLabel,omitempty"`

	// OnClick: The `onClick` action for this action item.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1CardAction) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardFixedFooter: A persistent (sticky) footer that
// that appears at the bottom of the card. For an example in Google Chat
// apps, see Card footer
// (https://developers.google.com/chat/ui/widgets/card-fixed-footer).
// Setting `fixedFooter` without specifying a `primaryButton` or a
// `secondaryButton` causes an error. Supported by Google Workspace
// Add-ons and Chat apps. For Chat apps, you can use fixed footers in
// dialogs (https://developers.google.com/chat/how-tos/dialogs), but not
// card messages
// (https://developers.google.com/chat/api/guides/v1/messages/create#create).
type GoogleAppsCardV1CardFixedFooter struct {
	// PrimaryButton: The primary button of the fixed footer. The button
	// must be a text button with text and color set.
	PrimaryButton *GoogleAppsCardV1Button `json:"primaryButton,omitempty"`

	// SecondaryButton: The secondary button of the fixed footer. The button
	// must be a text button with text and color set. If `secondaryButton`
	// is set, you must also set `primaryButton`.
	SecondaryButton *GoogleAppsCardV1Button `json:"secondaryButton,omitempty"`

	// ForceSendFields is a list of field names (e.g. "PrimaryButton") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "PrimaryButton") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1CardFixedFooter) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardFixedFooter
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardHeader: Represents a card header. For an example
// in Google Chat apps, see Card header
// (https://developers.google.com/chat/ui/widgets/card-header).
type GoogleAppsCardV1CardHeader struct {
	// ImageAltText: The alternative text of this image that's used for
	// accessibility.
	ImageAltText string `json:"imageAltText,omitempty"`

	// ImageType: The shape used to crop the image.
	//
	// Possible values:
	//   "SQUARE" - Default value. Applies a square mask to the image. For
	// example, a 4x3 image becomes 3x3.
	//   "CIRCLE" - Applies a circular mask to the image. For example, a 4x3
	// image becomes a circle with a diameter of 3.
	ImageType string `json:"imageType,omitempty"`

	// ImageUrl: The HTTPS URL of the image in the card header.
	ImageUrl string `json:"imageUrl,omitempty"`

	// Subtitle: The subtitle of the card header. If specified, appears on
	// its own line below the `title`.
	Subtitle string `json:"subtitle,omitempty"`

	// Title: Required. The title of the card header. The header has a fixed
	// height: if both a title and subtitle are specified, each takes up one
	// line. If only the title is specified, it takes up both lines.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ImageAltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ImageAltText") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1CardHeader) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardHeader
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Column: A column.
type GoogleAppsCardV1Column struct {
	// HorizontalAlignment: Specifies whether widgets align to the left,
	// right, or center of a column.
	//
	// Possible values:
	//   "HORIZONTAL_ALIGNMENT_UNSPECIFIED" - Don't use. Unspecified.
	//   "START" - Default value. Aligns widgets to the start position of
	// the column. For left-to-right layouts, aligns to the left. For
	// right-to-left layouts, aligns to the right.
	//   "CENTER" - Aligns widgets to the center of the column.
	//   "END" - Aligns widgets to the end position of the column. For
	// left-to-right layouts, aligns widgets to the right. For right-to-left
	// layouts, aligns widgets to the left.
	HorizontalAlignment string `json:"horizontalAlignment,omitempty"`

	// HorizontalSizeStyle: Specifies how a column fills the width of the
	// card.
	//
	// Possible values:
	//   "HORIZONTAL_SIZE_STYLE_UNSPECIFIED" - Don't use. Unspecified.
	//   "FILL_AVAILABLE_SPACE" - Default value. Column fills the available
	// space, up to 70% of the card's width. If both columns are set to
	// `FILL_AVAILABLE_SPACE`, each column fills 50% of the space.
	//   "FILL_MINIMUM_SPACE" - Column fills the least amount of space
	// possible and no more than 30% of the card's width.
	HorizontalSizeStyle string `json:"horizontalSizeStyle,omitempty"`

	// VerticalAlignment: Specifies whether widgets align to the top,
	// bottom, or center of a column.
	//
	// Possible values:
	//   "VERTICAL_ALIGNMENT_UNSPECIFIED" - Don't use. Unspecified.
	//   "CENTER" - Default value. Aligns widgets to the center of a column.
	//   "TOP" - Aligns widgets to the top of a column.
	//   "BOTTOM" - Aligns widgets to the bottom of a column.
	VerticalAlignment string `json:"verticalAlignment,omitempty"`

	// Widgets: An array of widgets included in a column. Widgets appear in
	// the order that they are specified.
	Widgets []*GoogleAppsCardV1Widgets `json:"widgets,omitempty"`

	// ForceSendFields is a list of field names (e.g. "HorizontalAlignment")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "HorizontalAlignment") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Column) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Column
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Columns: The `Columns` widget displays up to 2
// columns in a card message or dialog. You can add widgets to each
// column; the widgets appear in the order that they are specified. For
// an example in Google Chat apps, see Columns
// (https://developers.google.com/chat/ui/widgets/columns). The height
// of each column is determined by the taller column. For example, if
// the first column is taller than the second column, both columns have
// the height of the first column. Because each column can contain a
// different number of widgets, you can't define rows or align widgets
// between the columns. Columns are displayed side-by-side. You can
// customize the width of each column using the `HorizontalSizeStyle`
// field. If the user's screen width is too narrow, the second column
// wraps below the first: * On web, the second column wraps if the
// screen width is less than or equal to 480 pixels. * On iOS devices,
// the second column wraps if the screen width is less than or equal to
// 300 pt. * On Android devices, the second column wraps if the screen
// width is less than or equal to 320 dp. To include more than 2
// columns, or to use rows, use the `Grid` widget. Supported by Chat
// apps, but not Google Workspace Add-ons.
type GoogleAppsCardV1Columns struct {
	// ColumnItems: An array of columns. You can include up to 2 columns in
	// a card or dialog.
	ColumnItems []*GoogleAppsCardV1Column `json:"columnItems,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ColumnItems") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ColumnItems") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Columns) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Columns
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1DateTimePicker: Lets users input a date, a time, or
// both a date and a time. For an example in Google Chat apps, see Date
// time picker
// (https://developers.google.com/chat/ui/widgets/date-time-picker).
// Users can input text or use the picker to select dates and times. If
// users input an invalid date or time, the picker shows an error that
// prompts users to input the information correctly.
type GoogleAppsCardV1DateTimePicker struct {
	// Label: The text that prompts users to input a date, a time, or a date
	// and time. For example, if users are scheduling an appointment, use a
	// label such as `Appointment date` or `Appointment date and time`.
	Label string `json:"label,omitempty"`

	// Name: The name by which the `DateTimePicker` is identified in a form
	// input event. For details about working with form inputs, see Receive
	// form data (https://developers.google.com/chat/ui/read-form-data).
	Name string `json:"name,omitempty"`

	// OnChangeAction: Triggered when the user clicks **Save** or **Clear**
	// from the `DateTimePicker` interface.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// TimezoneOffsetDate: The number representing the time zone offset from
	// UTC, in minutes. If set, the `value_ms_epoch` is displayed in the
	// specified time zone. If unset, the value defaults to the user's time
	// zone setting.
	TimezoneOffsetDate int64 `json:"timezoneOffsetDate,omitempty"`

	// Type: Whether the widget supports inputting a date, a time, or the
	// date and time.
	//
	// Possible values:
	//   "DATE_AND_TIME" - Users input a date and time.
	//   "DATE_ONLY" - Users input a date.
	//   "TIME_ONLY" - Users input a time.
	Type string `json:"type,omitempty"`

	// ValueMsEpoch: The default value displayed in the widget, in
	// milliseconds since Unix epoch time
	// (https://en.wikipedia.org/wiki/Unix_time). Specify the value based on
	// the type of picker (`DateTimePickerType`): * `DATE_AND_TIME`: a
	// calendar date and time in UTC. For example, to represent January 1,
	// 2023 at 12:00 PM UTC, use `1672574400000`. * `DATE_ONLY`: a calendar
	// date at 00:00:00 UTC. For example, to represent January 1, 2023, use
	// `1672531200000`. * `TIME_ONLY`: a time in UTC. For example, to
	// represent 12:00 PM, use `43200000` (or `12 * 60 * 60 * 1000`).
	ValueMsEpoch int64 `json:"valueMsEpoch,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "Label") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Label") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1DateTimePicker) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1DateTimePicker
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1DecoratedText: A widget that displays text with
// optional decorations such as a label above or below the text, an icon
// in front of the text, a selection widget, or a button after the text.
// For an example in Google Chat apps, see Decorated text
// (https://developers.google.com/chat/ui/widgets/decorated-text).
type GoogleAppsCardV1DecoratedText struct {
	// BottomLabel: The text that appears below `text`. Always wraps.
	BottomLabel string `json:"bottomLabel,omitempty"`

	// Button: A button that a user can click to trigger an action.
	Button *GoogleAppsCardV1Button `json:"button,omitempty"`

	// EndIcon: An icon displayed after the text. Supports built-in
	// (https://developers.google.com/chat/format-messages#builtinicons) and
	// custom
	// (https://developers.google.com/chat/format-messages#customicons)
	// icons.
	EndIcon *GoogleAppsCardV1Icon `json:"endIcon,omitempty"`

	// Icon: Deprecated in favor of `startIcon`.
	Icon *GoogleAppsCardV1Icon `json:"icon,omitempty"`

	// OnClick: This action is triggered when users click `topLabel` or
	// `bottomLabel`.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// StartIcon: The icon displayed in front of the text.
	StartIcon *GoogleAppsCardV1Icon `json:"startIcon,omitempty"`

	// SwitchControl: A switch widget that a user can click to change its
	// state and trigger an action.
	SwitchControl *GoogleAppsCardV1SwitchControl `json:"switchControl,omitempty"`

	// Text: Required. The primary text. Supports simple formatting. For
	// more information about formatting text, see Formatting text in Google
	// Chat apps
	// (https://developers.google.com/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Text string `json:"text,omitempty"`

	// TopLabel: The text that appears above `text`. Always truncates.
	TopLabel string `json:"topLabel,omitempty"`

	// WrapText: The wrap text setting. If `true`, the text wraps and
	// displays on multiple lines. Otherwise, the text is truncated. Only
	// applies to `text`, not `topLabel` and `bottomLabel`.
	WrapText bool `json:"wrapText,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BottomLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BottomLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1DecoratedText) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1DecoratedText
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Divider: Displays a divider between widgets as a
// horizontal line. For an example in Google Chat apps, see Divider
// (https://developers.google.com/chat/ui/widgets/divider). For example,
// the following JSON creates a divider: ``` "divider": {} ```
type GoogleAppsCardV1Divider struct {
}

// GoogleAppsCardV1Grid: Displays a grid with a collection of items.
// Items can only include text or images. For responsive columns, or to
// include more than text or images, use `Columns`. For an example in
// Google Chat apps, see Grid
// (https://developers.google.com/chat/ui/widgets/grid). A grid supports
// any number of columns and items. The number of rows is determined by
// items divided by columns. A grid with 10 items and 2 columns has 5
// rows. A grid with 11 items and 2 columns has 6 rows. For example, the
// following JSON creates a 2 column grid with a single item: ```
// "grid": { "title": "A fine collection of items", "columnCount": 2,
// "borderStyle": { "type": "STROKE", "cornerRadius": 4 }, "items": [ {
// "image": { "imageUri": "https://www.example.com/image.png",
// "cropStyle": { "type": "SQUARE" }, "borderStyle": { "type": "STROKE"
// } }, "title": "An item", "textAlignment": "CENTER" } ], "onClick": {
// "openLink": { "url": "https://www.example.com" } } } ```
type GoogleAppsCardV1Grid struct {
	// BorderStyle: The border style to apply to each grid item.
	BorderStyle *GoogleAppsCardV1BorderStyle `json:"borderStyle,omitempty"`

	// ColumnCount: The number of columns to display in the grid. A default
	// value is used if this field isn't specified, and that default value
	// is different depending on where the grid is shown (dialog versus
	// companion).
	ColumnCount int64 `json:"columnCount,omitempty"`

	// Items: The items to display in the grid.
	Items []*GoogleAppsCardV1GridItem `json:"items,omitempty"`

	// OnClick: This callback is reused by each individual grid item, but
	// with the item's identifier and index in the items list added to the
	// callback's parameters.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// Title: The text that displays in the grid header.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BorderStyle") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BorderStyle") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Grid) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Grid
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1GridItem: Represents an item in a grid layout. Items
// can contain text, an image, or both text and an image.
type GoogleAppsCardV1GridItem struct {
	// Id: A user-specified identifier for this grid item. This identifier
	// is returned in the parent grid's `onClick` callback parameters.
	Id string `json:"id,omitempty"`

	// Image: The image that displays in the grid item.
	Image *GoogleAppsCardV1ImageComponent `json:"image,omitempty"`

	// Layout: The layout to use for the grid item.
	//
	// Possible values:
	//   "GRID_ITEM_LAYOUT_UNSPECIFIED" - Don't use. Unspecified.
	//   "TEXT_BELOW" - The title and subtitle are shown below the grid
	// item's image.
	//   "TEXT_ABOVE" - The title and subtitle are shown above the grid
	// item's image.
	Layout string `json:"layout,omitempty"`

	// Subtitle: The grid item's subtitle.
	Subtitle string `json:"subtitle,omitempty"`

	// Title: The grid item's title.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Id") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Id") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1GridItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1GridItem
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Icon: An icon displayed in a widget on a card. For an
// example in Google Chat apps, see Icon
// (https://developers.google.com/chat/ui/widgets/icon). Supports
// built-in
// (https://developers.google.com/chat/format-messages#builtinicons) and
// custom
// (https://developers.google.com/chat/format-messages#customicons)
// icons.
type GoogleAppsCardV1Icon struct {
	// AltText: Optional. A description of the icon used for accessibility.
	// If unspecified, the default value `Button` is provided. As a best
	// practice, you should set a helpful description for what the icon
	// displays, and if applicable, what it does. For example, `A user's
	// account portrait`, or `Opens a new browser tab and navigates to the
	// Google Chat developer documentation at
	// https://developers.google.com/chat`. If the icon is set in a
	// `Button`, the `altText` appears as helper text when the user hovers
	// over the button. However, if the button also sets `text`, the icon's
	// `altText` is ignored.
	AltText string `json:"altText,omitempty"`

	// IconUrl: Display a custom icon hosted at an HTTPS URL. For example:
	// ``` "iconUrl":
	// "https://developers.google.com/chat/images/quickstart-app-avatar.png"
	// ``` Supported file types include `.png` and `.jpg`.
	IconUrl string `json:"iconUrl,omitempty"`

	// ImageType: The crop style applied to the image. In some cases,
	// applying a `CIRCLE` crop causes the image to be drawn larger than a
	// built-in icon.
	//
	// Possible values:
	//   "SQUARE" - Default value. Applies a square mask to the image. For
	// example, a 4x3 image becomes 3x3.
	//   "CIRCLE" - Applies a circular mask to the image. For example, a 4x3
	// image becomes a circle with a diameter of 3.
	ImageType string `json:"imageType,omitempty"`

	// KnownIcon: Display one of the built-in icons provided by Google
	// Workspace. For example, to display an airplane icon, specify
	// `AIRPLANE`. For a bus, specify `BUS`. For a full list of supported
	// icons, see built-in icons
	// (https://developers.google.com/chat/format-messages#builtinicons).
	KnownIcon string `json:"knownIcon,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Icon) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Icon
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Image: An image that is specified by a URL and can
// have an `onClick` action. For an example, see Image
// (https://developers.google.com/chat/ui/widgets/image).
type GoogleAppsCardV1Image struct {
	// AltText: The alternative text of this image that's used for
	// accessibility.
	AltText string `json:"altText,omitempty"`

	// ImageUrl: The HTTPS URL that hosts the image. For example: ```
	// https://developers.google.com/chat/images/quickstart-app-avatar.png
	// ```
	ImageUrl string `json:"imageUrl,omitempty"`

	// OnClick: When a user clicks the image, the click triggers this
	// action.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Image) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Image
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ImageComponent: Represents an image.
type GoogleAppsCardV1ImageComponent struct {
	// AltText: The accessibility label for the image.
	AltText string `json:"altText,omitempty"`

	// BorderStyle: The border style to apply to the image.
	BorderStyle *GoogleAppsCardV1BorderStyle `json:"borderStyle,omitempty"`

	// CropStyle: The crop style to apply to the image.
	CropStyle *GoogleAppsCardV1ImageCropStyle `json:"cropStyle,omitempty"`

	// ImageUri: The image URL.
	ImageUri string `json:"imageUri,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ImageComponent) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ImageComponent
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ImageCropStyle: Represents the crop style applied to
// an image. For example, here's how to apply a 16:9 aspect ratio: ```
// cropStyle { "type": "RECTANGLE_CUSTOM", "aspectRatio": 16/9 } ```
type GoogleAppsCardV1ImageCropStyle struct {
	// AspectRatio: The aspect ratio to use if the crop type is
	// `RECTANGLE_CUSTOM`. For example, here's how to apply a 16:9 aspect
	// ratio: ``` cropStyle { "type": "RECTANGLE_CUSTOM", "aspectRatio":
	// 16/9 } ```
	AspectRatio float64 `json:"aspectRatio,omitempty"`

	// Type: The crop type.
	//
	// Possible values:
	//   "IMAGE_CROP_TYPE_UNSPECIFIED" - Don't use. Unspecified.
	//   "SQUARE" - Default value. Applies a square crop.
	//   "CIRCLE" - Applies a circular crop.
	//   "RECTANGLE_CUSTOM" - Applies a rectangular crop with a custom
	// aspect ratio. Set the custom aspect ratio with `aspectRatio`.
	//   "RECTANGLE_4_3" - Applies a rectangular crop with a 4:3 aspect
	// ratio.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AspectRatio") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AspectRatio") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ImageCropStyle) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ImageCropStyle
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *GoogleAppsCardV1ImageCropStyle) UnmarshalJSON(data []byte) error {
	type NoMethod GoogleAppsCardV1ImageCropStyle
	var s1 struct {
		AspectRatio gensupport.JSONFloat64 `json:"aspectRatio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AspectRatio = float64(s1.AspectRatio)
	return nil
}

// GoogleAppsCardV1OnClick: Represents how to respond when users click
// an interactive element on a card, such as a button.
type GoogleAppsCardV1OnClick struct {
	// Action: If specified, an action is triggered by this `onClick`.
	Action *GoogleAppsCardV1Action `json:"action,omitempty"`

	// Card: A new card is pushed to the card stack after clicking if
	// specified. Supported by Google Workspace Add-ons, but not Google Chat
	// apps.
	Card *GoogleAppsCardV1Card `json:"card,omitempty"`

	// OpenDynamicLinkAction: An add-on triggers this action when the action
	// needs to open a link. This differs from the `open_link` above in that
	// this needs to talk to server to get the link. Thus some preparation
	// work is required for web client to do before the open link action
	// response comes back. Supported by Google Workspace Add-ons, but not
	// Google Chat apps.
	OpenDynamicLinkAction *GoogleAppsCardV1Action `json:"openDynamicLinkAction,omitempty"`

	// OpenLink: If specified, this `onClick` triggers an open link action.
	OpenLink *GoogleAppsCardV1OpenLink `json:"openLink,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Action") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1OnClick) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OnClick
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1OpenLink: Represents an `onClick` event that opens a
// hyperlink.
type GoogleAppsCardV1OpenLink struct {
	// OnClose: Whether the client forgets about a link after opening it, or
	// observes it until the window closes. Not supported by Chat apps.
	//
	// Possible values:
	//   "NOTHING" - Default value. The card doesn't reload; nothing
	// happens.
	//   "RELOAD" - Reloads the card after the child window closes. If used
	// in conjunction with
	// [`OpenAs.OVERLAY`](https://developers.google.com/workspace/add-ons/ref
	// erence/rpc/google.apps.card.v1#openas), the child window acts as a
	// modal dialog and the parent card is blocked until the child window
	// closes.
	OnClose string `json:"onClose,omitempty"`

	// OpenAs: How to open a link. Not supported by Chat apps.
	//
	// Possible values:
	//   "FULL_SIZE" - The link opens as a full-size window (if that's the
	// frame used by the client).
	//   "OVERLAY" - The link opens as an overlay, such as a pop-up.
	OpenAs string `json:"openAs,omitempty"`

	// Url: The URL to open.
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "OnClose") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "OnClose") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1OpenLink) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OpenLink
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1PlatformDataSource: Chat apps only. For a
// `SelectionInput` widget that uses a multi-select menu, the data from
// a Google Workspace host application
// (https://developers.google.com/chat/api/reference/rest/v1/HostApp).
// Used to populate the items in the multi-select menu. Developer
// Preview (https://developers.google.com/workspace/preview).
type GoogleAppsCardV1PlatformDataSource struct {
	// CommonDataSource: For a `SelectionInput` widget that uses a
	// multi-select menu, a data source shared by all Google Workspace host
	// applications, such as users in a Google Workspace organization.
	// Developer Preview (https://developers.google.com/workspace/preview).
	//
	// Possible values:
	//   "UNKNOWN" - Default value. Don't use. [Developer
	// Preview](https://developers.google.com/workspace/preview).
	//   "USER" - A list of users provided by the Google Workspace host
	// application. For example, to source users from Google Chat, use the
	// resource name of the
	// [user](https://developers.google.com/chat/api/reference/rest/v1/User).
	//  Format: users/{user} [Developer
	// Preview](https://developers.google.com/workspace/preview).
	CommonDataSource string `json:"commonDataSource,omitempty"`

	// HostAppDataSource: A data source that's unique to a Google Workspace
	// host application, such as Gmail emails, Google Calendar events, or
	// Google Chat messages. Developer Preview
	// (https://developers.google.com/workspace/preview).
	HostAppDataSource *HostAppDataSourceMarkup `json:"hostAppDataSource,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CommonDataSource") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CommonDataSource") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1PlatformDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1PlatformDataSource
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Section: A section contains a collection of widgets
// that are rendered vertically in the order that they're specified.
type GoogleAppsCardV1Section struct {
	// Collapsible: Indicates whether this section is collapsible.
	// Collapsible sections hide some or all widgets, but users can expand
	// the section to reveal the hidden widgets by clicking **Show more**.
	// Users can hide the widgets again by clicking **Show less**. To
	// determine which widgets are hidden, specify
	// `uncollapsibleWidgetsCount`.
	Collapsible bool `json:"collapsible,omitempty"`

	// Header: Text that appears at the top of a section. Supports simple
	// HTML formatted text. For more information about formatting text, see
	// Formatting text in Google Chat apps
	// (https://developers.google.com/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Header string `json:"header,omitempty"`

	// UncollapsibleWidgetsCount: The number of uncollapsible widgets which
	// remain visible even when a section is collapsed. For example, when a
	// section contains five widgets and the `uncollapsibleWidgetsCount` is
	// set to `2`, the first two widgets are always shown and the last three
	// are collapsed by default. The `uncollapsibleWidgetsCount` is taken
	// into account only when `collapsible` is `true`.
	UncollapsibleWidgetsCount int64 `json:"uncollapsibleWidgetsCount,omitempty"`

	// Widgets: All the widgets in the section. Must contain at least one
	// widget.
	Widgets []*GoogleAppsCardV1Widget `json:"widgets,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Collapsible") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Collapsible") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Section) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Section
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SelectionInput: A widget that creates one or more UI
// items that users can select. For example, a dropdown menu or
// checkboxes. You can use this widget to collect data that can be
// predicted or enumerated. For an example in Google Chat apps, see
// Selection input
// (https://developers.google.com/chat/ui/widgets/selection-input). Chat
// apps can process the value of items that users select or input. For
// details about working with form inputs, see Receive form data
// (https://developers.google.com/chat/ui/read-form-data). To collect
// undefined or abstract data from users, use the TextInput widget.
type GoogleAppsCardV1SelectionInput struct {
	// ExternalDataSource: An external data source, such as a relational
	// data base. Developer Preview
	// (https://developers.google.com/workspace/preview).
	ExternalDataSource *GoogleAppsCardV1Action `json:"externalDataSource,omitempty"`

	// Items: An array of selectable items. For example, an array of radio
	// buttons or checkboxes. Supports up to 100 items.
	Items []*GoogleAppsCardV1SelectionItem `json:"items,omitempty"`

	// Label: The text that appears above the selection input field in the
	// user interface. Specify text that helps the user enter the
	// information your app needs. For example, if users are selecting the
	// urgency of a work ticket from a drop-down menu, the label might be
	// "Urgency" or "Select urgency".
	Label string `json:"label,omitempty"`

	// MultiSelectMaxSelectedItems: For multi-select menus, the maximum
	// number of items that a user can select. Minimum value is 1 item. If
	// unspecified, set to 3 items. Developer Preview
	// (https://developers.google.com/workspace/preview).
	MultiSelectMaxSelectedItems int64 `json:"multiSelectMaxSelectedItems,omitempty"`

	// MultiSelectMinQueryLength: For multi-select menus, the number of text
	// characters that a user inputs before the Chat app queries
	// autocomplete and displays suggested items on the card. If
	// unspecified, set to 0 characters for static data sources and 3
	// characters for external data sources. Developer Preview
	// (https://developers.google.com/workspace/preview).
	MultiSelectMinQueryLength int64 `json:"multiSelectMinQueryLength,omitempty"`

	// Name: The name that identifies the selection input in a form input
	// event. For details about working with form inputs, see Receive form
	// data (https://developers.google.com/chat/ui/read-form-data).
	Name string `json:"name,omitempty"`

	// OnChangeAction: If specified, the form is submitted when the
	// selection changes. If not specified, you must specify a separate
	// button that submits the form. For details about working with form
	// inputs, see Receive form data
	// (https://developers.google.com/chat/ui/read-form-data).
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// PlatformDataSource: A data source from a Google Workspace host
	// application
	// (https://developers.google.com/chat/api/reference/rest/v1/HostApp).
	// Developer Preview (https://developers.google.com/workspace/preview).
	PlatformDataSource *GoogleAppsCardV1PlatformDataSource `json:"platformDataSource,omitempty"`

	// Type: The type of items that are displayed to users in a
	// `SelectionInput` widget. Selection types support different types of
	// interactions. For example, users can select one or more checkboxes,
	// but they can only select one value from a dropdown menu.
	//
	// Possible values:
	//   "CHECK_BOX" - A set of checkboxes. Users can select one or more
	// checkboxes.
	//   "RADIO_BUTTON" - A set of radio buttons. Users can select one radio
	// button.
	//   "SWITCH" - A set of switches. Users can turn on one or more
	// switches.
	//   "DROPDOWN" - A dropdown menu. Users can select one item from the
	// menu.
	//   "MULTI_SELECT" - Supported by Chat apps, but not Google Workspace
	// Add-ons. A multi-select menu for static or dynamic data. From the
	// menu bar, users select one or more items. Users can also input values
	// to populate dynamic data. For example, users can start typing the
	// name of a Google Chat space and the widget autosuggests the space. To
	// populate items for a multi-select menu, you can use one of the
	// following types of data sources: * Static data: Items are specified
	// as `SelectionItem` objects in the widget. Up to 100 items. * Google
	// Workspace data: Items are populated using data from a Google
	// Workspace application, such as Google Chat users or spaces. *
	// External data: Items are populated from a dynamic external data
	// source. For examples of how to implement multi-select menus, see the
	// [`SelectionInput` widget
	// page](https://developers.google.com/chat/ui/widgets/selection-input).
	// [Developer Preview](https://developers.google.com/workspace/preview).
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ExternalDataSource")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ExternalDataSource") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SelectionInput) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SelectionInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SelectionItem: An item that users can select in a
// selection input, such as a checkbox or switch.
type GoogleAppsCardV1SelectionItem struct {
	// BottomText: For multi-select menus, a text description or label
	// that's displayed below the item's `text` field. Developer Preview
	// (https://developers.google.com/workspace/preview).
	BottomText string `json:"bottomText,omitempty"`

	// Selected: Whether the item is selected by default. If the selection
	// input only accepts one value (such as for radio buttons or a dropdown
	// menu), only set this field for one item.
	Selected bool `json:"selected,omitempty"`

	// StartIconUri: For multi-select menus, the URL for the icon displayed
	// next to the item's `text` field. Supports PNG and JPEG files. Must be
	// an `HTTPS` URL. For example,
	// `https://developers.google.com/chat/images/quickstart-app-avatar.png`.
	//  Developer Preview (https://developers.google.com/workspace/preview).
	StartIconUri string `json:"startIconUri,omitempty"`

	// Text: The text that identifies or describes the item to users.
	Text string `json:"text,omitempty"`

	// Value: The value associated with this item. The client should use
	// this as a form input value. For details about working with form
	// inputs, see Receive form data
	// (https://developers.google.com/chat/ui/read-form-data).
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BottomText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BottomText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SelectionItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SelectionItem
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SuggestionItem: One suggested value that users can
// enter in a text input field.
type GoogleAppsCardV1SuggestionItem struct {
	// Text: The value of a suggested input to a text input field. This is
	// equivalent to what users enter themselves.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Text") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Text") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SuggestionItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SuggestionItem
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Suggestions: Suggested values that users can enter.
// These values appear when users click inside the text input field. As
// users type, the suggested values dynamically filter to match what the
// users have typed. For example, a text input field for programming
// language might suggest Java, JavaScript, Python, and C++. When users
// start typing `Jav`, the list of suggestions filters to show `Java`
// and `JavaScript`. Suggested values help guide users to enter values
// that your app can make sense of. When referring to JavaScript, some
// users might enter `javascript` and others `java script`. Suggesting
// `JavaScript` can standardize how users interact with your app. When
// specified, `TextInput.type` is always `SINGLE_LINE`, even if it's set
// to `MULTIPLE_LINE`.
type GoogleAppsCardV1Suggestions struct {
	// Items: A list of suggestions used for autocomplete recommendations in
	// text input fields.
	Items []*GoogleAppsCardV1SuggestionItem `json:"items,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Items") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Suggestions) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Suggestions
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SwitchControl: Either a toggle-style switch or a
// checkbox inside a `decoratedText` widget. Only supported in the
// `decoratedText` widget.
type GoogleAppsCardV1SwitchControl struct {
	// ControlType: How the switch appears in the user interface.
	//
	// Possible values:
	//   "SWITCH" - A toggle-style switch.
	//   "CHECKBOX" - Deprecated in favor of `CHECK_BOX`.
	//   "CHECK_BOX" - A checkbox.
	ControlType string `json:"controlType,omitempty"`

	// Name: The name by which the switch widget is identified in a form
	// input event. For details about working with form inputs, see Receive
	// form data (https://developers.google.com/chat/ui/read-form-data).
	Name string `json:"name,omitempty"`

	// OnChangeAction: The action to perform when the switch state is
	// changed, such as what function to run.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// Selected: When `true`, the switch is selected.
	Selected bool `json:"selected,omitempty"`

	// Value: The value entered by a user, returned as part of a form input
	// event. For details about working with form inputs, see Receive form
	// data (https://developers.google.com/chat/ui/read-form-data).
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ControlType") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ControlType") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SwitchControl) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SwitchControl
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1TextInput: A field in which users can enter text.
// Supports suggestions and on-change actions. For an example in Google
// Chat apps, see Text input
// (https://developers.google.com/chat/ui/widgets/text-input). Chat apps
// receive and can process the value of entered text during form input
// events. For details about working with form inputs, see Receive form
// data (https://developers.google.com/chat/ui/read-form-data). When you
// need to collect undefined or abstract data from users, use a text
// input. To collect defined or enumerated data from users, use the
// SelectionInput widget.
type GoogleAppsCardV1TextInput struct {
	// AutoCompleteAction: Optional. Specify what action to take when the
	// text input field provides suggestions to users who interact with it.
	// If unspecified, the suggestions are set by `initialSuggestions` and
	// are processed by the client. If specified, the app takes the action
	// specified here, such as running a custom function. Supported by
	// Google Workspace Add-ons, but not Google Chat apps.
	AutoCompleteAction *GoogleAppsCardV1Action `json:"autoCompleteAction,omitempty"`

	// HintText: Text that appears below the text input field meant to
	// assist users by prompting them to enter a certain value. This text is
	// always visible. Required if `label` is unspecified. Otherwise,
	// optional.
	HintText string `json:"hintText,omitempty"`

	// InitialSuggestions: Suggested values that users can enter. These
	// values appear when users click inside the text input field. As users
	// type, the suggested values dynamically filter to match what the users
	// have typed. For example, a text input field for programming language
	// might suggest Java, JavaScript, Python, and C++. When users start
	// typing `Jav`, the list of suggestions filters to show just `Java` and
	// `JavaScript`. Suggested values help guide users to enter values that
	// your app can make sense of. When referring to JavaScript, some users
	// might enter `javascript` and others `java script`. Suggesting
	// `JavaScript` can standardize how users interact with your app. When
	// specified, `TextInput.type` is always `SINGLE_LINE`, even if it's set
	// to `MULTIPLE_LINE`.
	InitialSuggestions *GoogleAppsCardV1Suggestions `json:"initialSuggestions,omitempty"`

	// Label: The text that appears above the text input field in the user
	// interface. Specify text that helps the user enter the information
	// your app needs. For example, if you are asking someone's name, but
	// specifically need their surname, write `surname` instead of `name`.
	// Required if `hintText` is unspecified. Otherwise, optional.
	Label string `json:"label,omitempty"`

	// Name: The name by which the text input is identified in a form input
	// event. For details about working with form inputs, see Receive form
	// data (https://developers.google.com/chat/ui/read-form-data).
	Name string `json:"name,omitempty"`

	// OnChangeAction: What to do when a change occurs in the text input
	// field. For example, a user adding to the field or deleting text.
	// Examples of actions to take include running a custom function or
	// opening a dialog (https://developers.google.com/chat/how-tos/dialogs)
	// in Google Chat.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// PlaceholderText: Text that appears in the text input field when the
	// field is empty. Use this text to prompt users to enter a value. For
	// example, `Enter a number from 0 to 100`. Supported by Google Chat
	// apps, but not Google Workspace Add-ons.
	PlaceholderText string `json:"placeholderText,omitempty"`

	// Type: How a text input field appears in the user interface. For
	// example, whether the field is single or multi-line.
	//
	// Possible values:
	//   "SINGLE_LINE" - The text input field has a fixed height of one
	// line.
	//   "MULTIPLE_LINE" - The text input field has a fixed height of
	// multiple lines.
	Type string `json:"type,omitempty"`

	// Value: The value entered by a user, returned as part of a form input
	// event. For details about working with form inputs, see Receive form
	// data (https://developers.google.com/chat/ui/read-form-data).
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AutoCompleteAction")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AutoCompleteAction") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1TextInput) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1TextInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1TextParagraph: A paragraph of text that supports
// formatting. For an example in Google Chat apps, see Text paragraph
// (https://developers.google.com/chat/ui/widgets/text-paragraph). For
// more information about formatting text, see Formatting text in Google
// Chat apps
// (https://developers.google.com/chat/format-messages#card-formatting)
// and Formatting text in Google Workspace Add-ons
// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
type GoogleAppsCardV1TextParagraph struct {
	// Text: The text that's shown in the widget.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Text") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Text") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1TextParagraph) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1TextParagraph
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Widget: Each card is made up of widgets. A widget is
// a composite object that can represent one of text, images, buttons,
// and other object types.
type GoogleAppsCardV1Widget struct {
	// ButtonList: A list of buttons. For example, the following JSON
	// creates two buttons. The first is a blue text button and the second
	// is an image button that opens a link: ``` "buttonList": { "buttons":
	// [ { "text": "Edit", "color": { "red": 0, "green": 0, "blue": 1,
	// "alpha": 1 }, "disabled": true, }, { "icon": { "knownIcon": "INVITE",
	// "altText": "check calendar" }, "onClick": { "openLink": { "url":
	// "https://example.com/calendar" } } } ] } ```
	ButtonList *GoogleAppsCardV1ButtonList `json:"buttonList,omitempty"`

	// Columns: Displays up to 2 columns. To include more than 2 columns, or
	// to use rows, use the `Grid` widget. For example, the following JSON
	// creates 2 columns that each contain text paragraphs: ``` "columns": {
	// "columnItems": [ { "horizontalSizeStyle": "FILL_AVAILABLE_SPACE",
	// "horizontalAlignment": "CENTER", "verticalAlignment": "CENTER",
	// "widgets": [ { "textParagraph": { "text": "First column text
	// paragraph" } } ] }, { "horizontalSizeStyle": "FILL_AVAILABLE_SPACE",
	// "horizontalAlignment": "CENTER", "verticalAlignment": "CENTER",
	// "widgets": [ { "textParagraph": { "text": "Second column text
	// paragraph" } } ] } ] } ```
	Columns *GoogleAppsCardV1Columns `json:"columns,omitempty"`

	// DateTimePicker: Displays a widget that lets users input a date, time,
	// or date and time. For example, the following JSON creates a date time
	// picker to schedule an appointment: ``` "dateTimePicker": { "name":
	// "appointment_time", "label": "Book your appointment at:", "type":
	// "DATE_AND_TIME", "valueMsEpoch": "796435200000" } ```
	DateTimePicker *GoogleAppsCardV1DateTimePicker `json:"dateTimePicker,omitempty"`

	// DecoratedText: Displays a decorated text item. For example, the
	// following JSON creates a decorated text widget showing email address:
	// ``` "decoratedText": { "icon": { "knownIcon": "EMAIL" }, "topLabel":
	// "Email Address", "text": "sasha@example.com", "bottomLabel": "This is
	// a new Email address!", "switchControl": { "name":
	// "has_send_welcome_email_to_sasha", "selected": false, "controlType":
	// "CHECKBOX" } } ```
	DecoratedText *GoogleAppsCardV1DecoratedText `json:"decoratedText,omitempty"`

	// Divider: Displays a horizontal line divider between widgets. For
	// example, the following JSON creates a divider: ``` "divider": { } ```
	Divider *GoogleAppsCardV1Divider `json:"divider,omitempty"`

	// Grid: Displays a grid with a collection of items. A grid supports any
	// number of columns and items. The number of rows is determined by the
	// upper bounds of the number items divided by the number of columns. A
	// grid with 10 items and 2 columns has 5 rows. A grid with 11 items and
	// 2 columns has 6 rows. For example, the following JSON creates a 2
	// column grid with a single item: ``` "grid": { "title": "A fine
	// collection of items", "columnCount": 2, "borderStyle": { "type":
	// "STROKE", "cornerRadius": 4 }, "items": [ { "image": { "imageUri":
	// "https://www.example.com/image.png", "cropStyle": { "type": "SQUARE"
	// }, "borderStyle": { "type": "STROKE" } }, "title": "An item",
	// "textAlignment": "CENTER" } ], "onClick": { "openLink": { "url":
	// "https://www.example.com" } } } ```
	Grid *GoogleAppsCardV1Grid `json:"grid,omitempty"`

	// HorizontalAlignment: Specifies whether widgets align to the left,
	// right, or center of a column.
	//
	// Possible values:
	//   "HORIZONTAL_ALIGNMENT_UNSPECIFIED" - Don't use. Unspecified.
	//   "START" - Default value. Aligns widgets to the start position of
	// the column. For left-to-right layouts, aligns to the left. For
	// right-to-left layouts, aligns to the right.
	//   "CENTER" - Aligns widgets to the center of the column.
	//   "END" - Aligns widgets to the end position of the column. For
	// left-to-right layouts, aligns widgets to the right. For right-to-left
	// layouts, aligns widgets to the left.
	HorizontalAlignment string `json:"horizontalAlignment,omitempty"`

	// Image: Displays an image. For example, the following JSON creates an
	// image with alternative text: ``` "image": { "imageUrl":
	// "https://developers.google.com/chat/images/quickstart-app-avatar.png",
	//  "altText": "Chat app avatar" } ```
	Image *GoogleAppsCardV1Image `json:"image,omitempty"`

	// SelectionInput: Displays a selection control that lets users select
	// items. Selection controls can be checkboxes, radio buttons, switches,
	// or dropdown menus. For example, the following JSON creates a dropdown
	// menu that lets users choose a size: ``` "selectionInput": { "name":
	// "size", "label": "Size" "type": "DROPDOWN", "items": [ { "text": "S",
	// "value": "small", "selected": false }, { "text": "M", "value":
	// "medium", "selected": true }, { "text": "L", "value": "large",
	// "selected": false }, { "text": "XL", "value": "extra_large",
	// "selected": false } ] } ```
	SelectionInput *GoogleAppsCardV1SelectionInput `json:"selectionInput,omitempty"`

	// TextInput: Displays a text box that users can type into. For example,
	// the following JSON creates a text input for an email address: ```
	// "textInput": { "name": "mailing_address", "label": "Mailing Address"
	// } ``` As another example, the following JSON creates a text input for
	// a programming language with static suggestions: ``` "textInput": {
	// "name": "preferred_programing_language", "label": "Preferred
	// Language", "initialSuggestions": { "items": [ { "text": "C++" }, {
	// "text": "Java" }, { "text": "JavaScript" }, { "text": "Python" } ] }
	// } ```
	TextInput *GoogleAppsCardV1TextInput `json:"textInput,omitempty"`

	// TextParagraph: Displays a text paragraph. Supports simple HTML
	// formatted text. For more information about formatting text, see
	// Formatting text in Google Chat apps
	// (https://developers.google.com/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	// For example, the following JSON creates a bolded text: ```
	// "textParagraph": { "text": " *bold text*" } ```
	TextParagraph *GoogleAppsCardV1TextParagraph `json:"textParagraph,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ButtonList") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ButtonList") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Widget) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Widget
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Widgets: The supported widgets that you can include
// in a column.
type GoogleAppsCardV1Widgets struct {
	// ButtonList: ButtonList widget.
	ButtonList *GoogleAppsCardV1ButtonList `json:"buttonList,omitempty"`

	// DateTimePicker: DateTimePicker widget.
	DateTimePicker *GoogleAppsCardV1DateTimePicker `json:"dateTimePicker,omitempty"`

	// DecoratedText: DecoratedText widget.
	DecoratedText *GoogleAppsCardV1DecoratedText `json:"decoratedText,omitempty"`

	// Image: Image widget.
	Image *GoogleAppsCardV1Image `json:"image,omitempty"`

	// SelectionInput: SelectionInput widget.
	SelectionInput *GoogleAppsCardV1SelectionInput `json:"selectionInput,omitempty"`

	// TextInput: TextInput widget.
	TextInput *GoogleAppsCardV1TextInput `json:"textInput,omitempty"`

	// TextParagraph: TextParagraph widget.
	TextParagraph *GoogleAppsCardV1TextParagraph `json:"textParagraph,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ButtonList") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ButtonList") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Widgets) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Widgets
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// HostAppDataSourceMarkup: Chat apps only. For a `SelectionInput`
// widget that uses a multi-select menu, a data source from a Google
// Workspace host application. Developer Preview
// (https://developers.google.com/workspace/preview).
type HostAppDataSourceMarkup struct {
	// ChatDataSource: The data source is Google Chat. Developer Preview
	// (https://developers.google.com/workspace/preview).
	ChatDataSource *ChatClientDataSourceMarkup `json:"chatDataSource,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ChatDataSource") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ChatDataSource") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *HostAppDataSourceMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod HostAppDataSourceMarkup
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Image: An image that's specified by a URL and can have an `onclick`
// action.
type Image struct {
	// AspectRatio: The aspect ratio of this image (width and height). This
	// field lets you reserve the right height for the image while waiting
	// for it to load. It's not meant to override the built-in aspect ratio
	// of the image. If unset, the server fills it by prefetching the image.
	AspectRatio float64 `json:"aspectRatio,omitempty"`

	// ImageUrl: The URL of the image.
	ImageUrl string `json:"imageUrl,omitempty"`

	// OnClick: The `onclick` action.
	OnClick *OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AspectRatio") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AspectRatio") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Image) MarshalJSON() ([]byte, error) {
	type NoMethod Image
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *Image) UnmarshalJSON(data []byte) error {
	type NoMethod Image
	var s1 struct {
		AspectRatio gensupport.JSONFloat64 `json:"aspectRatio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AspectRatio = float64(s1.AspectRatio)
	return nil
}

// ImageButton: An image button with an `onclick` action.
type ImageButton struct {
	// Icon: The icon specified by an `enum` that indices to an icon
	// provided by Chat API.
	//
	// Possible values:
	//   "ICON_UNSPECIFIED"
	//   "AIRPLANE"
	//   "BOOKMARK"
	//   "BUS"
	//   "CAR"
	//   "CLOCK"
	//   "CONFIRMATION_NUMBER_ICON"
	//   "DOLLAR"
	//   "DESCRIPTION"
	//   "EMAIL"
	//   "EVENT_PERFORMER"
	//   "EVENT_SEAT"
	//   "FLIGHT_ARRIVAL"
	//   "FLIGHT_DEPARTURE"
	//   "HOTEL"
	//   "HOTEL_ROOM_TYPE"
	//   "INVITE"
	//   "MAP_PIN"
	//   "MEMBERSHIP"
	//   "MULTIPLE_PEOPLE"
	//   "OFFER"
	//   "PERSON"
	//   "PHONE"
	//   "RESTAURANT_ICON"
	//   "SHOPPING_CART"
	//   "STAR"
	//   "STORE"
	//   "TICKET"
	//   "TRAIN"
	//   "VIDEO_CAMERA"
	//   "VIDEO_PLAY"
	Icon string `json:"icon,omitempty"`

	// IconUrl: The icon specified by a URL.
	IconUrl string `json:"iconUrl,omitempty"`

	// Name: The name of this `image_button` that's used for accessibility.
	// Default value is provided if this name isn't specified.
	Name string `json:"name,omitempty"`

	// OnClick: The `onclick` action.
	OnClick *OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Icon") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Icon") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ImageButton) MarshalJSON() ([]byte, error) {
	type NoMethod ImageButton
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Inputs: Types of data that users can enter on cards or dialogs. To
// learn how to process information from users, see Read form data input
// by users on cards
// (https://developers.google.com/chat/ui/read-form-data).
type Inputs struct {
	// DateInput: Date input values.
	DateInput *DateInput `json:"dateInput,omitempty"`

	// DateTimeInput: Date and time input values.
	DateTimeInput *DateTimeInput `json:"dateTimeInput,omitempty"`

	// StringInputs: Input parameter for regular widgets. For single-valued
	// widgets, it is a single value list. For multi-valued widgets, such as
	// checkbox, all the values are presented.
	StringInputs *StringInputs `json:"stringInputs,omitempty"`

	// TimeInput: Time input values.
	TimeInput *TimeInput `json:"timeInput,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DateInput") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DateInput") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Inputs) MarshalJSON() ([]byte, error) {
	type NoMethod Inputs
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// KeyValue: A UI element contains a key (label) and a value (content).
// This element can also contain some actions such as `onclick` button.
type KeyValue struct {
	// BottomLabel: The text of the bottom label. Formatted text supported.
	// For more information about formatting text, see Formatting text in
	// Google Chat apps
	// (https://developers.google.com/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	BottomLabel string `json:"bottomLabel,omitempty"`

	// Button: A button that can be clicked to trigger an action.
	Button *Button `json:"button,omitempty"`

	// Content: The text of the content. Formatted text supported and always
	// required. For more information about formatting text, see Formatting
	// text in Google Chat apps
	// (https://developers.google.com/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Content string `json:"content,omitempty"`

	// ContentMultiline: If the content should be multiline.
	ContentMultiline bool `json:"contentMultiline,omitempty"`

	// Icon: An enum value that's replaced by the Chat API with the
	// corresponding icon image.
	//
	// Possible values:
	//   "ICON_UNSPECIFIED"
	//   "AIRPLANE"
	//   "BOOKMARK"
	//   "BUS"
	//   "CAR"
	//   "CLOCK"
	//   "CONFIRMATION_NUMBER_ICON"
	//   "DOLLAR"
	//   "DESCRIPTION"
	//   "EMAIL"
	//   "EVENT_PERFORMER"
	//   "EVENT_SEAT"
	//   "FLIGHT_ARRIVAL"
	//   "FLIGHT_DEPARTURE"
	//   "HOTEL"
	//   "HOTEL_ROOM_TYPE"
	//   "INVITE"
	//   "MAP_PIN"
	//   "MEMBERSHIP"
	//   "MULTIPLE_PEOPLE"
	//   "OFFER"
	//   "PERSON"
	//   "PHONE"
	//   "RESTAURANT_ICON"
	//   "SHOPPING_CART"
	//   "STAR"
	//   "STORE"
	//   "TICKET"
	//   "TRAIN"
	//   "VIDEO_CAMERA"
	//   "VIDEO_PLAY"
	Icon string `json:"icon,omitempty"`

	// IconUrl: The icon specified by a URL.
	IconUrl string `json:"iconUrl,omitempty"`

	// OnClick: The `onclick` action. Only the top label, bottom label, and
	// content region are clickable.
	OnClick *OnClick `json:"onClick,omitempty"`

	// TopLabel: The text of the top label. Formatted text supported. For
	// more information about formatting text, see Formatting text in Google
	// Chat apps
	// (https://developers.google.com/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	TopLabel string `json:"topLabel,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BottomLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BottomLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *KeyValue) MarshalJSON() ([]byte, error) {
	type NoMethod KeyValue
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type ListMembershipsResponse struct {
	// Memberships: List of memberships in the requested (or first) page.
	Memberships []*Membership `json:"memberships,omitempty"`

	// NextPageToken: A token that you can send as `pageToken` to retrieve
	// the next page of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Memberships") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ListMembershipsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListMembershipsResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type ListMessagesResponse struct {
	// Messages: List of messages.
	Messages []*Message `json:"messages,omitempty"`

	// NextPageToken: You can send a token as `pageToken` to retrieve the
	// next page of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Messages") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Messages") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ListMessagesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListMessagesResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type ListReactionsResponse struct {
	// NextPageToken: Continuation token to retrieve the next page of
	// results. It's empty for the last page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// Reactions: List of reactions in the requested (or first) page.
	Reactions []*Reaction `json:"reactions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "NextPageToken") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ListReactionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListReactionsResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type ListSpacesResponse struct {
	// NextPageToken: You can send a token as `pageToken` to retrieve the
	// next page of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// Spaces: List of spaces in the requested (or first) page.
	Spaces []*Space `json:"spaces,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "NextPageToken") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ListSpacesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSpacesResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// MatchedUrl: A matched URL in a Chat message. Chat apps can preview
// matched URLs. For more information, see Preview links
// (https://developers.google.com/chat/how-tos/preview-links).
type MatchedUrl struct {
	// Url: Output only. The URL that was matched.
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Url") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Url") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *MatchedUrl) MarshalJSON() ([]byte, error) {
	type NoMethod MatchedUrl
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Media: Media resource.
type Media struct {
	// ResourceName: Name of the media resource.
	ResourceName string `json:"resourceName,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "ResourceName") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ResourceName") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Media) MarshalJSON() ([]byte, error) {
	type NoMethod Media
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Membership: Represents a membership relation in Google Chat, such as
// whether a user or Chat app is invited to, part of, or absent from a
// space.
type Membership struct {
	// CreateTime: Output only. The creation time of the membership, such as
	// when a member joined or was invited to join a space.
	CreateTime string `json:"createTime,omitempty"`

	// Member: The Google Chat user or app the membership corresponds to. If
	// your Chat app authenticates as a user
	// (https://developers.google.com/chat/api/guides/auth/users), the
	// output populates the user
	// (https://developers.google.com/chat/api/reference/rest/v1/User)
	// `name` and `type`.
	Member *User `json:"member,omitempty"`

	// Name: Resource name of the membership, assigned by the server.
	// Format: `spaces/{space}/members/{member}`
	Name string `json:"name,omitempty"`

	// Role: Optional. User's role within a Chat space, which determines
	// their permitted actions in the space. Developer Preview
	// (https://developers.google.com/workspace/preview): This field can
	// only be used as input in `UpdateMembership`.
	//
	// Possible values:
	//   "MEMBERSHIP_ROLE_UNSPECIFIED" - Default value. For users: they
	// aren't a member of the space, but can be invited. For Google Groups:
	// they're always assigned this role (other enum values might be used in
	// the future).
	//   "ROLE_MEMBER" - A member of the space. The user has basic
	// permissions, like sending messages to the space. In 1:1 and unnamed
	// group conversations, everyone has this role.
	//   "ROLE_MANAGER" - A space manager. The user has all basic
	// permissions plus administrative permissions that let them manage the
	// space, like adding or removing members. Only supported in
	// SpaceType.SPACE.
	Role string `json:"role,omitempty"`

	// State: Output only. State of the membership.
	//
	// Possible values:
	//   "MEMBERSHIP_STATE_UNSPECIFIED" - Default value. Don't use.
	//   "JOINED" - The user is added to the space, and can participate in
	// the space.
	//   "INVITED" - The user is invited to join the space, but hasn't
	// joined it.
	//   "NOT_A_MEMBER" - The user doesn't belong to the space and doesn't
	// have a pending invitation to join the space.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CreateTime") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Membership) MarshalJSON() ([]byte, error) {
	type NoMethod Membership
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Message: A message in a Google Chat space.
type Message struct {
	// ActionResponse: Input only. Parameters that a Chat app can use to
	// configure how its response is posted.
	ActionResponse *ActionResponse `json:"actionResponse,omitempty"`

	// Annotations: Output only. Annotations associated with the `text` in
	// this message.
	Annotations []*Annotation `json:"annotations,omitempty"`

	// ArgumentText: Output only. Plain-text body of the message with all
	// Chat app mentions stripped out.
	ArgumentText string `json:"argumentText,omitempty"`

	// AttachedGifs: Output only. GIF images that are attached to the
	// message.
	AttachedGifs []*AttachedGif `json:"attachedGifs,omitempty"`

	// Attachment: User-uploaded attachment.
	Attachment []*Attachment `json:"attachment,omitempty"`

	// Cards: Deprecated: Use `cards_v2` instead. Rich, formatted, and
	// interactive cards that you can use to display UI elements such as:
	// formatted texts, buttons, and clickable images. Cards are normally
	// displayed below the plain-text body of the message. `cards` and
	// `cards_v2` can have a maximum size of 32 KB.
	Cards []*Card `json:"cards,omitempty"`

	// CardsV2: An array of cards
	// (https://developers.google.com/chat/api/reference/rest/v1/cards).
	// Only Chat apps can create cards. If your Chat app authenticates as a
	// user (https://developers.google.com/chat/api/guides/auth/users), the
	// messages can't contain cards. To learn about cards and how to create
	// them, see Design dynamic, interactive, and consistent UIs with cards
	// (https://developers.google.com/chat/ui).
	CardsV2 []*CardWithId `json:"cardsV2,omitempty"`

	// ClientAssignedMessageId: A custom name for a Chat message assigned at
	// creation. Must start with `client-` and contain only lowercase
	// letters, numbers, and hyphens up to 63 characters in length. Specify
	// this field to get, update, or delete the message with the specified
	// value. Assigning a custom name lets a Chat app recall the message
	// without saving the message `name` from the response body
	// (/chat/api/reference/rest/v1/spaces.messages/get#response-body)
	// returned when creating the message. Assigning a custom name doesn't
	// replace the generated `name` field, the message's resource name.
	// Instead, it sets the custom name as the `clientAssignedMessageId`
	// field, which you can reference while processing later operations,
	// like updating or deleting the message. For example usage, see Name a
	// created message
	// (https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message).
	ClientAssignedMessageId string `json:"clientAssignedMessageId,omitempty"`

	// CreateTime: For spaces created in Chat, the time at which the message
	// was created. This field is output only, except when used in imported
	// spaces. Developer Preview
	// (https://developers.google.com/workspace/preview): For imported
	// spaces, set this field to the historical timestamp at which the
	// message was created in the source in order to preserve the original
	// creation time.
	CreateTime string `json:"createTime,omitempty"`

	// DeleteTime: Output only. The time at which the message was deleted in
	// Google Chat. If the message is never deleted, this field is empty.
	DeleteTime string `json:"deleteTime,omitempty"`

	// DeletionMetadata: Output only. Information about a deleted message. A
	// message is deleted when `delete_time` is set.
	DeletionMetadata *DeletionMetadata `json:"deletionMetadata,omitempty"`

	// EmojiReactionSummaries: Output only. The list of emoji reaction
	// summaries on the message.
	EmojiReactionSummaries []*EmojiReactionSummary `json:"emojiReactionSummaries,omitempty"`

	// FallbackText: A plain-text description of the message's cards, used
	// when the actual cards can't be displayed—for example, mobile
	// notifications.
	FallbackText string `json:"fallbackText,omitempty"`

	// FormattedText: Output only. Contains the message `text` with markups
	// added to communicate formatting. This field might not capture all
	// formatting visible in the UI, but includes the following: * Markup
	// syntax (https://developers.google.com/chat/format-messages) for bold,
	// italic, strikethrough, monospace, and monospace block. * User
	// mentions
	// (https://developers.google.com/chat/format-messages#messages-@mention)
	// using the format ``. * Custom hyperlinks using the format
	// `<{url}|{rendered_text}>` where the first string is the URL and the
	// second is the rendered text—for example, ``. * Custom emoji using
	// the format `:{emoji_name}:`—for example, `:smile:`. This doesn't
	// apply to Unicode emoji, such as `U+1F600` for a grinning face emoji.
	// For more information, see View text formatting sent in a message
	// (https://developers.google.com/chat/format-messages#view_text_formatting_sent_in_a_message)
	FormattedText string `json:"formattedText,omitempty"`

	// LastUpdateTime: Output only. The time at which the message was last
	// edited by a user. If the message has never been edited, this field is
	// empty.
	LastUpdateTime string `json:"lastUpdateTime,omitempty"`

	// MatchedUrl: Output only. A URL in `spaces.messages.text` that matches
	// a link preview pattern. For more information, see Preview links
	// (https://developers.google.com/chat/how-tos/preview-links).
	MatchedUrl *MatchedUrl `json:"matchedUrl,omitempty"`

	// Name: Resource name in the form `spaces/*/messages/*`. Example:
	// `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`
	Name string `json:"name,omitempty"`

	// QuotedMessageMetadata: Output only. Information about a message
	// that's quoted by a Google Chat user in a space. Google Chat users can
	// quote a message to reply to it.
	QuotedMessageMetadata *QuotedMessageMetadata `json:"quotedMessageMetadata,omitempty"`

	// Sender: Output only. The user who created the message. If your Chat
	// app authenticates as a user
	// (https://developers.google.com/chat/api/guides/auth/users), the
	// output populates the user
	// (https://developers.google.com/chat/api/reference/rest/v1/User)
	// `name` and `type`.
	Sender *User `json:"sender,omitempty"`

	// SlashCommand: Output only. Slash command information, if applicable.
	SlashCommand *SlashCommand `json:"slashCommand,omitempty"`

	// Space: If your Chat app authenticates as a user
	// (https://developers.google.com/chat/api/guides/auth/users), the
	// output populates the space
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces)
	// `name`.
	Space *Space `json:"space,omitempty"`

	// Text: Plain-text body of the message. The first link to an image,
	// video, or web page generates a preview chip
	// (https://developers.google.com/chat/how-tos/preview-links). You can
	// also @mention a Google Chat user
	// (https://developers.google.com/chat/format-messages#messages-@mention),
	// or everyone in the space. To learn about creating text messages, see
	// Send a text message
	// (https://developers.google.com/chat/api/guides/v1/messages/create#create-text-messages).
	Text string `json:"text,omitempty"`

	// Thread: The thread the message belongs to. For example usage, see
	// Start or reply to a message thread
	// (https://developers.google.com/chat/api/guides/crudl/messages#start_or_reply_to_a_message_thread).
	Thread *Thread `json:"thread,omitempty"`

	// ThreadReply: Output only. When `true`, the message is a response in a
	// reply thread. When `false`, the message is visible in the space's
	// top-level conversation as either the first message of a thread or a
	// message with no threaded replies. If the space doesn't support reply
	// in threads, this field is always `false`.
	ThreadReply bool `json:"threadReply,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "ActionResponse") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionResponse") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Message) MarshalJSON() ([]byte, error) {
	type NoMethod Message
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// OnClick: An `onclick` action (for example, open a link).
type OnClick struct {
	// Action: A form action is triggered by this `onclick` action if
	// specified.
	Action *FormAction `json:"action,omitempty"`

	// OpenLink: This `onclick` action triggers an open link action if
	// specified.
	OpenLink *OpenLink `json:"openLink,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Action") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *OnClick) MarshalJSON() ([]byte, error) {
	type NoMethod OnClick
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// OpenLink: A link that opens a new window.
type OpenLink struct {
	// Url: The URL to open.
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Url") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Url") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *OpenLink) MarshalJSON() ([]byte, error) {
	type NoMethod OpenLink
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// QuotedMessageMetadata: Information about a quoted message.
type QuotedMessageMetadata struct {
	// LastUpdateTime: Output only. The timestamp when the quoted message
	// was created or when the quoted message was last updated.
	LastUpdateTime string `json:"lastUpdateTime,omitempty"`

	// Name: Output only. Resource name of the quoted message. Format:
	// `spaces/{space}/messages/{message}`
	Name string `json:"name,omitempty"`

	// ForceSendFields is a list of field names (e.g. "LastUpdateTime") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "LastUpdateTime") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *QuotedMessageMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod QuotedMessageMetadata
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Reaction: A reaction to a message.
type Reaction struct {
	// Emoji: The emoji used in the reaction.
	Emoji *Emoji `json:"emoji,omitempty"`

	// Name: The resource name of the reaction. Format:
	// `spaces/{space}/messages/{message}/reactions/{reaction}`
	Name string `json:"name,omitempty"`

	// User: Output only. The user who created the reaction.
	User *User `json:"user,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Emoji") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Emoji") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Reaction) MarshalJSON() ([]byte, error) {
	type NoMethod Reaction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Section: A section contains a collection of widgets that are rendered
// (vertically) in the order that they are specified. Across all
// platforms, cards have a narrow fixed width, so there's currently no
// need for layout properties (for example, float).
type Section struct {
	// Header: The header of the section. Formatted text is supported. For
	// more information about formatting text, see Formatting text in Google
	// Chat apps
	// (https://developers.google.com/chat/format-messages#card-formatting)
	// and Formatting text in Google Workspace Add-ons
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Header string `json:"header,omitempty"`

	// Widgets: A section must contain at least one widget.
	Widgets []*WidgetMarkup `json:"widgets,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Header") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Header") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Section) MarshalJSON() ([]byte, error) {
	type NoMethod Section
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type SetUpSpaceRequest struct {
	// Memberships: Optional. The Google Chat users to invite to join the
	// space. Omit the calling user, as they are added automatically. The
	// set currently allows up to 20 memberships (in addition to the
	// caller). The `Membership.member` field must contain a `user` with
	// `name` populated (format: `users/{user}`) and `type` set to
	// `User.Type.HUMAN`. You can only add human users when setting up a
	// space (adding Chat apps is only supported for direct message setup
	// with the calling app). You can also add members using the user's
	// email as an alias for {user}. For example, the `user.name` can be
	// `users/example@gmail.com`." To invite Gmail users or users from
	// external Google Workspace domains, user's email must be used for
	// `{user}`. Optional when setting `Space.spaceType` to `SPACE`.
	// Required when setting `Space.spaceType` to `GROUP_CHAT`, along with
	// at least two memberships. Required when setting `Space.spaceType` to
	// `DIRECT_MESSAGE` with a human user, along with exactly one
	// membership. Must be empty when creating a 1:1 conversation between a
	// human and the calling Chat app (when setting `Space.spaceType` to
	// `DIRECT_MESSAGE` and `Space.singleUserBotDm` to `true`).
	Memberships []*Membership `json:"memberships,omitempty"`

	// RequestId: Optional. A unique identifier for this request. A random
	// UUID is recommended. Specifying an existing request ID returns the
	// space created with that ID instead of creating a new space.
	// Specifying an existing request ID from the same Chat app with a
	// different authenticated user returns an error.
	RequestId string `json:"requestId,omitempty"`

	// Space: Required. The `Space.spaceType` field is required. To create a
	// space, set `Space.spaceType` to `SPACE` and set `Space.displayName`.
	// If you receive the error message `ALREADY_EXISTS` when setting up a
	// space, try a different `displayName`. An existing space within the
	// Google Workspace organization might already use this display name. To
	// create a group chat, set `Space.spaceType` to `GROUP_CHAT`. Don't set
	// `Space.displayName`. To create a 1:1 conversation between humans, set
	// `Space.spaceType` to `DIRECT_MESSAGE` and set `Space.singleUserBotDm`
	// to `false`. Don't set `Space.displayName` or `Space.spaceDetails`. To
	// create an 1:1 conversation between a human and the calling Chat app,
	// set `Space.spaceType` to `DIRECT_MESSAGE` and `Space.singleUserBotDm`
	// to `true`. Don't set `Space.displayName` or `Space.spaceDetails`. If
	// a `DIRECT_MESSAGE` space already exists, that space is returned
	// instead of creating a new space.
	Space *Space `json:"space,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Memberships") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SetUpSpaceRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SetUpSpaceRequest
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SlashCommand: A slash command
// (https://developers.google.com/chat/how-tos/slash-commands) in Google
// Chat.
type SlashCommand struct {
	// CommandId: The ID of the slash command invoked.
	CommandId int64 `json:"commandId,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "CommandId") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CommandId") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SlashCommand) MarshalJSON() ([]byte, error) {
	type NoMethod SlashCommand
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SlashCommandMetadata: Annotation metadata for slash commands (/).
type SlashCommandMetadata struct {
	// Bot: The Chat app whose command was invoked.
	Bot *User `json:"bot,omitempty"`

	// CommandId: The command ID of the invoked slash command.
	CommandId int64 `json:"commandId,omitempty,string"`

	// CommandName: The name of the invoked slash command.
	CommandName string `json:"commandName,omitempty"`

	// TriggersDialog: Indicates whether the slash command is for a dialog.
	TriggersDialog bool `json:"triggersDialog,omitempty"`

	// Type: The type of slash command.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "ADD" - Add Chat app to space.
	//   "INVOKE" - Invoke slash command in space.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Bot") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Bot") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SlashCommandMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod SlashCommandMetadata
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Space: A space in Google Chat. Spaces are conversations between two
// or more users or 1:1 messages between a user and a Chat app.
type Space struct {
	// AdminInstalled: Output only. Whether the Chat app was installed by a
	// Google Workspace administrator. Administrators can install a Chat app
	// for their domain, organizational unit, or a group of users.
	// Administrators can only install Chat apps for direct messaging
	// between users and the app. To support admin install, your app must
	// feature direct messaging.
	AdminInstalled bool `json:"adminInstalled,omitempty"`

	// DisplayName: The space's display name. Required when creating a space
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces/create).
	// If you receive the error message `ALREADY_EXISTS` when creating a
	// space or updating the `displayName`, try a different `displayName`.
	// An existing space within the Google Workspace organization might
	// already use this display name. For direct messages, this field might
	// be empty. Supports up to 128 characters.
	DisplayName string `json:"displayName,omitempty"`

	// ExternalUserAllowed: Immutable. Whether this space permits any Google
	// Chat user as a member. Input when creating a space in a Google
	// Workspace organization. Omit this field when creating spaces in the
	// following conditions: * The authenticated user uses a Google Account.
	// By default, the space permits any Google Chat user. * The space is
	// used to [import data to Google Chat]
	// (https://developers.google.com/chat/api/guides/import-data-overview).
	// Import mode spaces must only permit members from the same Google
	// Workspace organization. For existing spaces, this field is output
	// only.
	ExternalUserAllowed bool `json:"externalUserAllowed,omitempty"`

	// Name: Resource name of the space. Format: `spaces/{space}`
	Name string `json:"name,omitempty"`

	// SingleUserBotDm: Optional. Whether the space is a DM between a Chat
	// app and a single human.
	SingleUserBotDm bool `json:"singleUserBotDm,omitempty"`

	// SpaceDetails: Details about the space including description and
	// rules.
	SpaceDetails *SpaceDetails `json:"spaceDetails,omitempty"`

	// SpaceHistoryState: The message history state for messages and threads
	// in this space.
	//
	// Possible values:
	//   "HISTORY_STATE_UNSPECIFIED" - Default value. Do not use.
	//   "HISTORY_OFF" - History off. [Messages and threads are kept for 24
	// hours](https://support.google.com/chat/answer/7664687).
	//   "HISTORY_ON" - History on. The organization's [Vault retention
	// rules](https://support.google.com/vault/answer/7657597) specify for
	// how long messages and threads are kept.
	SpaceHistoryState string `json:"spaceHistoryState,omitempty"`

	// SpaceThreadingState: Output only. The threading state in the Chat
	// space.
	//
	// Possible values:
	//   "SPACE_THREADING_STATE_UNSPECIFIED" - Reserved.
	//   "THREADED_MESSAGES" - Named spaces that support message threads.
	// When users respond to a message, they can reply in-thread, which
	// keeps their response in the context of the original message.
	//   "GROUPED_MESSAGES" - Named spaces where the conversation is
	// organized by topic. Topics and their replies are grouped together.
	//   "UNTHREADED_MESSAGES" - Direct messages (DMs) between two people
	// and group conversations between 3 or more people.
	SpaceThreadingState string `json:"spaceThreadingState,omitempty"`

	// SpaceType: The type of space. Required when creating a space or
	// updating the space type of a space. Output only for other usage.
	//
	// Possible values:
	//   "SPACE_TYPE_UNSPECIFIED" - Reserved.
	//   "SPACE" - A place where people send messages, share files, and
	// collaborate. A `SPACE` can include Chat apps.
	//   "GROUP_CHAT" - Group conversations between 3 or more people. A
	// `GROUP_CHAT` can include Chat apps.
	//   "DIRECT_MESSAGE" - 1:1 messages between two humans or a human and a
	// Chat app.
	SpaceType string `json:"spaceType,omitempty"`

	// Threaded: Output only. Deprecated: Use `spaceThreadingState` instead.
	// Whether messages are threaded in this space.
	Threaded bool `json:"threaded,omitempty"`

	// Type: Output only. Deprecated: Use `space_type` instead. The type of
	// a space.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED"
	//   "ROOM" - Conversations between two or more humans.
	//   "DM" - 1:1 Direct Message between a human and a Chat app, where all
	// messages are flat. Note that this doesn't include direct messages
	// between two humans.
	Type string `json:"type,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AdminInstalled") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AdminInstalled") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Space) MarshalJSON() ([]byte, error) {
	type NoMethod Space
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SpaceDataSource: A data source representing a Google Chat space.
// Format: spaces/{space} Developer Preview
// (https://developers.google.com/workspace/preview).
type SpaceDataSource struct {
	// DefaultToCurrentSpace: When `true`, uses the card's Google Chat space
	// as the default selection. The default value is `false`. Developer
	// Preview (https://developers.google.com/workspace/preview).
	DefaultToCurrentSpace bool `json:"defaultToCurrentSpace,omitempty"`

	// ForceSendFields is a list of field names (e.g.
	// "DefaultToCurrentSpace") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. However, any non-pointer, non-interface field appearing in
	// ForceSendFields will be sent to the server regardless of whether the
	// field is empty or not. This may be used to include empty fields in
	// Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DefaultToCurrentSpace") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *SpaceDataSource) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceDataSource
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SpaceDetails: Details about the space including description and
// rules.
type SpaceDetails struct {
	// Description: Optional. A description of the space. For example,
	// describe the space's discussion topic, functional purpose, or
	// participants. Supports up to 150 characters.
	Description string `json:"description,omitempty"`

	// Guidelines: Optional. The space's rules, expectations, and etiquette.
	// Supports up to 5,000 characters.
	Guidelines string `json:"guidelines,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Description") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SpaceDetails) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceDetails
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Status: The `Status` type defines a logical error model that is
// suitable for different programming environments, including REST APIs
// and RPC APIs. It is used by gRPC (https://github.com/grpc). Each
// `Status` message contains three pieces of data: error code, error
// message, and error details. You can find out more about this error
// model and how to work with it in the API Design Guide
// (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of
	// google.rpc.Code.
	Code int64 `json:"code,omitempty"`

	// Details: A list of messages that carry the error details. There is a
	// common set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`

	// Message: A developer-facing error message, which should be in
	// English. Any user-facing error message should be localized and sent
	// in the google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Code") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Code") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// StringInputs: Input parameter for regular widgets. For single-valued
// widgets, it is a single value list. For multi-valued widgets, such as
// checkbox, all the values are presented.
type StringInputs struct {
	// Value: An array of strings entered by the user.
	Value []string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Value") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Value") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *StringInputs) MarshalJSON() ([]byte, error) {
	type NoMethod StringInputs
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TextButton: A button with text and `onclick` action.
type TextButton struct {
	// OnClick: The `onclick` action of the button.
	OnClick *OnClick `json:"onClick,omitempty"`

	// Text: The text of the button.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "OnClick") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "OnClick") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TextButton) MarshalJSON() ([]byte, error) {
	type NoMethod TextButton
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TextParagraph: A paragraph of text. Formatted text supported. For
// more information about formatting text, see Formatting text in Google
// Chat apps
// (https://developers.google.com/chat/format-messages#card-formatting)
// and Formatting text in Google Workspace Add-ons
// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
type TextParagraph struct {
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Text") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Text") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TextParagraph) MarshalJSON() ([]byte, error) {
	type NoMethod TextParagraph
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Thread: A thread in a Google Chat space. For example usage, see Start
// or reply to a message thread
// (https://developers.google.com/chat/api/guides/v1/messages/create#create-message-thread).
// If you specify a thread when creating a message, you can set the
// `messageReplyOption`
// (https://developers.google.com/chat/api/reference/rest/v1/spaces.messages/create#messagereplyoption)
// field to determine what happens if no matching thread is found.
type Thread struct {
	// Name: Output only. Resource name of the thread. Example:
	// `spaces/{space}/threads/{thread}`
	Name string `json:"name,omitempty"`

	// ThreadKey: Optional. Input for creating or updating a thread.
	// Otherwise, output only. ID for the thread. Supports up to 4000
	// characters. This ID is unique to the Chat app that sets it. For
	// example, if multiple Chat apps create a message using the same thread
	// key, the messages are posted in different threads. To reply in a
	// thread created by a person or another Chat app, specify the thread
	// `name` field instead.
	ThreadKey string `json:"threadKey,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Name") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Name") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Thread) MarshalJSON() ([]byte, error) {
	type NoMethod Thread
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TimeInput: Time input values.
type TimeInput struct {
	// Hours: The hour on a 24-hour clock.
	Hours int64 `json:"hours,omitempty"`

	// Minutes: The number of minutes past the hour. Valid values are 0 to
	// 59.
	Minutes int64 `json:"minutes,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Hours") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Hours") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TimeInput) MarshalJSON() ([]byte, error) {
	type NoMethod TimeInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TimeZone: The timezone ID and offset from Coordinated Universal Time
// (UTC). Only supported for the event types `CARD_CLICKED`
// (https://developers.google.com/chat/api/reference/rest/v1/EventType#ENUM_VALUES.CARD_CLICKED)
// and `SUBMIT_DIALOG`
// (https://developers.google.com/chat/api/reference/rest/v1/DialogEventType#ENUM_VALUES.SUBMIT_DIALOG).
type TimeZone struct {
	// Id: The IANA TZ (https://www.iana.org/time-zones) time zone database
	// code, such as "America/Toronto".
	Id string `json:"id,omitempty"`

	// Offset: The user timezone offset, in milliseconds, from Coordinated
	// Universal Time (UTC).
	Offset int64 `json:"offset,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Id") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Id") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TimeZone) MarshalJSON() ([]byte, error) {
	type NoMethod TimeZone
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type UploadAttachmentRequest struct {
	// Filename: Required. The filename of the attachment, including the
	// file extension.
	Filename string `json:"filename,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Filename") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Filename") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *UploadAttachmentRequest) MarshalJSON() ([]byte, error) {
	type NoMethod UploadAttachmentRequest
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type UploadAttachmentResponse struct {
	// AttachmentDataRef: Reference to the uploaded attachment.
	AttachmentDataRef *AttachmentDataRef `json:"attachmentDataRef,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AttachmentDataRef")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AttachmentDataRef") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *UploadAttachmentResponse) MarshalJSON() ([]byte, error) {
	type NoMethod UploadAttachmentResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// User: A user in Google Chat. When returned as an output from a
// request, if your Chat app authenticates as a user
// (https://developers.google.com/chat/api/guides/auth/users), the
// output for a `User` resource only populates the user's `name` and
// `type`.
type User struct {
	// DisplayName: Output only. The user's display name.
	DisplayName string `json:"displayName,omitempty"`

	// DomainId: Unique identifier of the user's Google Workspace domain.
	DomainId string `json:"domainId,omitempty"`

	// IsAnonymous: Output only. When `true`, the user is deleted or their
	// profile is not visible.
	IsAnonymous bool `json:"isAnonymous,omitempty"`

	// Name: Resource name for a Google Chat user. Format: `users/{user}`.
	// `users/app` can be used as an alias for the calling app bot user. For
	// human users, `{user}` is the same user identifier as: - the
	// `{person_id`} for the Person
	// (https://developers.google.com/people/api/rest/v1/people) in the
	// People API, where the Person `resource_name` is `people/{person_id}`.
	// For example, `users/123456789` in Chat API represents the same person
	// as `people/123456789` in People API. - the `id` for a user
	// (https://developers.google.com/admin-sdk/directory/reference/rest/v1/users)
	// in the Admin SDK Directory API. - the user's email address can be
	// used as an alias for `{user}` in API requests. For example, if the
	// People API Person `resourceName` for `user@example.com` is
	// `people/123456789`, you can use `users/user@example.com` as an alias
	// to reference `users/123456789`. Only the canonical resource name (for
	// example `users/123456789`) will be returned from the API.
	Name string `json:"name,omitempty"`

	// Type: User type.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "HUMAN" - Human user.
	//   "BOT" - Chat app user.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DisplayName") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *User) MarshalJSON() ([]byte, error) {
	type NoMethod User
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// UserMentionMetadata: Annotation metadata for user mentions (@).
type UserMentionMetadata struct {
	// Type: The type of user mention.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. Don't use.
	//   "ADD" - Add user to space.
	//   "MENTION" - Mention user in space.
	Type string `json:"type,omitempty"`

	// User: The user mentioned.
	User *User `json:"user,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Type") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Type") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *UserMentionMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod UserMentionMetadata
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// WidgetMarkup: A widget is a UI element that presents text and images.
type WidgetMarkup struct {
	// Buttons: A list of buttons. Buttons is also `oneof data` and only one
	// of these fields should be set.
	Buttons []*Button `json:"buttons,omitempty"`

	// Image: Display an image in this widget.
	Image *Image `json:"image,omitempty"`

	// KeyValue: Display a key value item in this widget.
	KeyValue *KeyValue `json:"keyValue,omitempty"`

	// TextParagraph: Display a text paragraph in this widget.
	TextParagraph *TextParagraph `json:"textParagraph,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Buttons") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Buttons") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *WidgetMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod WidgetMarkup
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// method id "chat.media.download":

type MediaDownloadCall struct {
	s            *Service
	resourceName string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Download: Downloads media. Download is supported on the URI
// `/v1/media/{+name}?alt=media`.
//
//   - resourceName: Name of the media that is being downloaded. See
//     ReadRequest.resource_name.
func (r *MediaService) Download(resourceName string) *MediaDownloadCall {
	c := &MediaDownloadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resourceName = resourceName
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *MediaDownloadCall) Fields(s ...googleapi.Field) *MediaDownloadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *MediaDownloadCall) IfNoneMatch(entityTag string) *MediaDownloadCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download
// methods. Any pending HTTP request will be aborted if the provided
// context is canceled.
func (c *MediaDownloadCall) Context(ctx context.Context) *MediaDownloadCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *MediaDownloadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MediaDownloadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/media/{+resourceName}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resourceName": c.resourceName,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *MediaDownloadCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "chat.media.download" call.
// Exactly one of *Media or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Media.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *MediaDownloadCall) Do(opts ...googleapi.CallOption) (*Media, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Media{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Downloads media. Download is supported on the URI `/v1/media/{+name}?alt=media`.",
	//   "flatPath": "v1/media/{mediaId}",
	//   "httpMethod": "GET",
	//   "id": "chat.media.download",
	//   "parameterOrder": [
	//     "resourceName"
	//   ],
	//   "parameters": {
	//     "resourceName": {
	//       "description": "Name of the media that is being downloaded. See ReadRequest.resource_name.",
	//       "location": "path",
	//       "pattern": "^.*$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/media/{+resourceName}",
	//   "response": {
	//     "$ref": "Media"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.readonly"
	//   ],
	//   "supportsMediaDownload": true
	// }

}

// method id "chat.media.upload":

type MediaUploadCall struct {
	s                       *Service
	parent                  string
	uploadattachmentrequest *UploadAttachmentRequest
	urlParams_              gensupport.URLParams
	mediaInfo_              *gensupport.MediaInfo
	ctx_                    context.Context
	header_                 http.Header
}

// Upload: Uploads an attachment. For an example, see Upload media as a
// file attachment
// (https://developers.google.com/chat/api/guides/v1/media-and-attachments/upload).
// Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users). You can
// upload attachments up to 200 MB. Certain file types aren't supported.
// For details, see File types blocked by Google Chat
// (https://support.google.com/chat/answer/7651457?&co=GENIE.Platform%3DDesktop#File%20types%20blocked%20in%20Google%20Chat).
//
//   - parent: Resource name of the Chat space in which the attachment is
//     uploaded. Format "spaces/{space}".
func (r *MediaService) Upload(parent string, uploadattachmentrequest *UploadAttachmentRequest) *MediaUploadCall {
	c := &MediaUploadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.uploadattachmentrequest = uploadattachmentrequest
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk
// size may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the
// upload request will be determined by sniffing the contents of r,
// unless a MediaOption generated by googleapi.ContentType is
// supplied.
// At most one of Media and ResumableMedia may be set.
func (c *MediaUploadCall) Media(r io.Reader, options ...googleapi.MediaOption) *MediaUploadCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be
// canceled with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType
// identifies the MIME media type of the upload, such as "image/png". If
// mediaType is "", it will be auto-detected. The provided ctx will
// supersede any context previously provided to the Context method.
func (c *MediaUploadCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *MediaUploadCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called
// after every chunk. It should be a low-latency function in order to
// not slow down the upload operation. This should only be called when
// using ResumableMedia (as opposed to Media).
func (c *MediaUploadCall) ProgressUpdater(pu googleapi.ProgressUpdater) *MediaUploadCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *MediaUploadCall) Fields(s ...googleapi.Field) *MediaUploadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *MediaUploadCall) Context(ctx context.Context) *MediaUploadCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *MediaUploadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MediaUploadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.uploadattachmentrequest)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/attachments:upload")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/v1/{+parent}/attachments:upload")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	if body == nil {
		body = new(bytes.Buffer)
		reqHeaders.Set("Content-Type", "application/json")
	}
	body, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.media.upload" call.
// Exactly one of *UploadAttachmentResponse or error will be non-nil.
// Any non-2xx status code is an error. Response headers are in either
// *UploadAttachmentResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *MediaUploadCall) Do(opts ...googleapi.CallOption) (*UploadAttachmentResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &UploadAttachmentResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Uploads an attachment. For an example, see [Upload media as a file attachment](https://developers.google.com/chat/api/guides/v1/media-and-attachments/upload). Requires user [authentication](https://developers.google.com/chat/api/guides/auth/users). You can upload attachments up to 200 MB. Certain file types aren't supported. For details, see [File types blocked by Google Chat](https://support.google.com/chat/answer/7651457?\u0026co=GENIE.Platform%3DDesktop#File%20types%20blocked%20in%20Google%20Chat).",
	//   "flatPath": "v1/spaces/{spacesId}/attachments:upload",
	//   "httpMethod": "POST",
	//   "id": "chat.media.upload",
	//   "mediaUpload": {
	//     "accept": [
	//       "*/*"
	//     ],
	//     "maxSize": "209715200",
	//     "protocols": {
	//       "resumable": {
	//         "multipart": true,
	//         "path": "/resumable/upload/v1/{+parent}/attachments:upload"
	//       },
	//       "simple": {
	//         "multipart": true,
	//         "path": "/upload/v1/{+parent}/attachments:upload"
	//       }
	//     }
	//   },
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "parent": {
	//       "description": "Required. Resource name of the Chat space in which the attachment is uploaded. Format \"spaces/{space}\".",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+parent}/attachments:upload",
	//   "request": {
	//     "$ref": "UploadAttachmentRequest"
	//   },
	//   "response": {
	//     "$ref": "UploadAttachmentResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.create"
	//   ],
	//   "supportsMediaUpload": true
	// }

}

// method id "chat.spaces.create":

type SpacesCreateCall struct {
	s          *Service
	space      *Space
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a named space. Spaces grouped by topics aren't
// supported. For an example, see Create a space
// (https://developers.google.com/chat/api/guides/v1/spaces/create). If
// you receive the error message `ALREADY_EXISTS` when creating a space,
// try a different `displayName`. An existing space within the Google
// Workspace organization might already use this display name. Requires
// user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
func (r *SpacesService) Create(space *Space) *SpacesCreateCall {
	c := &SpacesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.space = space
	return c
}

// RequestId sets the optional parameter "requestId": A unique
// identifier for this request. A random UUID is recommended. Specifying
// an existing request ID returns the space created with that ID instead
// of creating a new space. Specifying an existing request ID from the
// same Chat app with a different authenticated user returns an error.
func (c *SpacesCreateCall) RequestId(requestId string) *SpacesCreateCall {
	c.urlParams_.Set("requestId", requestId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesCreateCall) Fields(s ...googleapi.Field) *SpacesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesCreateCall) Context(ctx context.Context) *SpacesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.space)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.create" call.
// Exactly one of *Space or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesCreateCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Creates a named space. Spaces grouped by topics aren't supported. For an example, see [Create a space](https://developers.google.com/chat/api/guides/v1/spaces/create). If you receive the error message `ALREADY_EXISTS` when creating a space, try a different `displayName`. An existing space within the Google Workspace organization might already use this display name. Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces",
	//   "httpMethod": "POST",
	//   "id": "chat.spaces.create",
	//   "parameterOrder": [],
	//   "parameters": {
	//     "requestId": {
	//       "description": "Optional. A unique identifier for this request. A random UUID is recommended. Specifying an existing request ID returns the space created with that ID instead of creating a new space. Specifying an existing request ID from the same Chat app with a different authenticated user returns an error.",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/spaces",
	//   "request": {
	//     "$ref": "Space"
	//   },
	//   "response": {
	//     "$ref": "Space"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.spaces",
	//     "https://www.googleapis.com/auth/chat.spaces.create"
	//   ]
	// }

}

// method id "chat.spaces.delete":

type SpacesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a named space. Always performs a cascading delete,
// which means that the space's child resources—like messages posted
// in the space and memberships in the space—are also deleted. For an
// example, see Delete a space
// (https://developers.google.com/chat/api/guides/v1/spaces/delete).
// Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users) from a
// user who has permission to delete the space.
//
//   - name: Resource name of the space to delete. Format:
//     `spaces/{space}`.
func (r *SpacesService) Delete(name string) *SpacesDeleteCall {
	c := &SpacesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesDeleteCall) Fields(s ...googleapi.Field) *SpacesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesDeleteCall) Context(ctx context.Context) *SpacesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.delete" call.
// Exactly one of *Empty or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Deletes a named space. Always performs a cascading delete, which means that the space's child resources—like messages posted in the space and memberships in the space—are also deleted. For an example, see [Delete a space](https://developers.google.com/chat/api/guides/v1/spaces/delete). Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users) from a user who has permission to delete the space.",
	//   "flatPath": "v1/spaces/{spacesId}",
	//   "httpMethod": "DELETE",
	//   "id": "chat.spaces.delete",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the space to delete. Format: `spaces/{space}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Empty"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.delete",
	//     "https://www.googleapis.com/auth/chat.import"
	//   ]
	// }

}

// method id "chat.spaces.findDirectMessage":

type SpacesFindDirectMessageCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// FindDirectMessage: Returns the existing direct message with the
// specified user. If no direct message space is found, returns a `404
// NOT_FOUND` error. For an example, see Find a direct message
// (/chat/api/guides/v1/spaces/find-direct-message). With user
// authentication
// (https://developers.google.com/chat/api/guides/auth/users), returns
// the direct message space between the specified user and the
// authenticated user. With service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts),
// returns the direct message space between the specified user and the
// calling Chat app. Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users) or service
// account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
func (r *SpacesService) FindDirectMessage() *SpacesFindDirectMessageCall {
	c := &SpacesFindDirectMessageCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Name sets the optional parameter "name": Required. Resource name of
// the user to find direct message with. Format: `users/{user}`, where
// `{user}` is either the `{person_id}` for the person
// (https://developers.google.com/people/api/rest/v1/people) from the
// People API, or the `id` for the user
// (https://developers.google.com/admin-sdk/directory/reference/rest/v1/users)
// in the Directory API. For example, if the People API
// `Person.resourceName` is `people/123456789`, you can find a direct
// message with that person by using `users/123456789` as the `name`.
// When authenticated as a user
// (https://developers.google.com/chat/api/guides/auth/users), you can
// use the email as an alias for `{user}`. For example,
// `users/example@gmail.com` where `example@gmail.com` is the email of
// the Google Chat user.
func (c *SpacesFindDirectMessageCall) Name(name string) *SpacesFindDirectMessageCall {
	c.urlParams_.Set("name", name)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesFindDirectMessageCall) Fields(s ...googleapi.Field) *SpacesFindDirectMessageCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesFindDirectMessageCall) IfNoneMatch(entityTag string) *SpacesFindDirectMessageCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesFindDirectMessageCall) Context(ctx context.Context) *SpacesFindDirectMessageCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesFindDirectMessageCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesFindDirectMessageCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces:findDirectMessage")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.findDirectMessage" call.
// Exactly one of *Space or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesFindDirectMessageCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Returns the existing direct message with the specified user. If no direct message space is found, returns a `404 NOT_FOUND` error. For an example, see [Find a direct message](/chat/api/guides/v1/spaces/find-direct-message). With [user authentication](https://developers.google.com/chat/api/guides/auth/users), returns the direct message space between the specified user and the authenticated user. With [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts), returns the direct message space between the specified user and the calling Chat app. Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users) or [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts).",
	//   "flatPath": "v1/spaces:findDirectMessage",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.findDirectMessage",
	//   "parameterOrder": [],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the user to find direct message with. Format: `users/{user}`, where `{user}` is either the `{person_id}` for the [person](https://developers.google.com/people/api/rest/v1/people) from the People API, or the `id` for the [user](https://developers.google.com/admin-sdk/directory/reference/rest/v1/users) in the Directory API. For example, if the People API `Person.resourceName` is `people/123456789`, you can find a direct message with that person by using `users/123456789` as the `name`. When [authenticated as a user](https://developers.google.com/chat/api/guides/auth/users), you can use the email as an alias for `{user}`. For example, `users/example@gmail.com` where `example@gmail.com` is the email of the Google Chat user.",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/spaces:findDirectMessage",
	//   "response": {
	//     "$ref": "Space"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.spaces",
	//     "https://www.googleapis.com/auth/chat.spaces.readonly"
	//   ]
	// }

}

// method id "chat.spaces.get":

type SpacesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns details about a space. For an example, see Get a space
// (https://developers.google.com/chat/api/guides/v1/spaces/get).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
//   - name: Resource name of the space, in the form "spaces/*". Format:
//     `spaces/{space}`.
func (r *SpacesService) Get(name string) *SpacesGetCall {
	c := &SpacesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesGetCall) Fields(s ...googleapi.Field) *SpacesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesGetCall) IfNoneMatch(entityTag string) *SpacesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesGetCall) Context(ctx context.Context) *SpacesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.get" call.
// Exactly one of *Space or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesGetCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Returns details about a space. For an example, see [Get a space](https://developers.google.com/chat/api/guides/v1/spaces/get). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the space, in the form \"spaces/*\". Format: `spaces/{space}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Space"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.spaces",
	//     "https://www.googleapis.com/auth/chat.spaces.readonly"
	//   ]
	// }

}

// method id "chat.spaces.list":

type SpacesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists spaces the caller is a member of. Group chats and DMs
// aren't listed until the first message is sent. For an example, see
// List spaces
// (https://developers.google.com/chat/api/guides/v1/spaces/list).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users). Lists
// spaces visible to the caller or authenticated user. Group chats and
// DMs aren't listed until the first message is sent.
func (r *SpacesService) List() *SpacesListCall {
	c := &SpacesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Filter sets the optional parameter "filter": A query filter. Requires
// user authentication
// (https://developers.google.com/chat/api/guides/auth/users). You can
// filter spaces by the space type (`space_type`
// (https://developers.google.com/chat/api/reference/rest/v1/spaces#spacetype)).
// To filter by space type, you must specify valid enum value, such as
// `SPACE` or `GROUP_CHAT` (the `space_type` can't be
// `SPACE_TYPE_UNSPECIFIED`). To query for multiple space types, use the
// `OR` operator. For example, the following queries are valid: ```
// space_type = "SPACE" spaceType = "GROUP_CHAT" OR spaceType =
// "DIRECT_MESSAGE" ``` Invalid queries are rejected by the server with
// an `INVALID_ARGUMENT` error. With service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts),
// this field is ignored and the query always returns all spaces. But
// the Chat API still validates the query syntax with service accounts,
// so invalid queries are still rejected.
func (c *SpacesListCall) Filter(filter string) *SpacesListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number
// of spaces to return. The service might return fewer than this value.
// If unspecified, at most 100 spaces are returned. The maximum value is
// 1,000. If you use a value more than 1,000, it's automatically changed
// to 1,000. Negative values return an `INVALID_ARGUMENT` error.
func (c *SpacesListCall) PageSize(pageSize int64) *SpacesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token,
// received from a previous list spaces call. Provide this parameter to
// retrieve the subsequent page. When paginating, the filter value
// should match the call that provided the page token. Passing a
// different value may lead to unexpected results.
func (c *SpacesListCall) PageToken(pageToken string) *SpacesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesListCall) Fields(s ...googleapi.Field) *SpacesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesListCall) IfNoneMatch(entityTag string) *SpacesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesListCall) Context(ctx context.Context) *SpacesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.list" call.
// Exactly one of *ListSpacesResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListSpacesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *SpacesListCall) Do(opts ...googleapi.CallOption) (*ListSpacesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSpacesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Lists spaces the caller is a member of. Group chats and DMs aren't listed until the first message is sent. For an example, see [List spaces](https://developers.google.com/chat/api/guides/v1/spaces/list). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users). Lists spaces visible to the caller or authenticated user. Group chats and DMs aren't listed until the first message is sent.",
	//   "flatPath": "v1/spaces",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.list",
	//   "parameterOrder": [],
	//   "parameters": {
	//     "filter": {
	//       "description": "Optional. A query filter. Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users). You can filter spaces by the space type ([`space_type`](https://developers.google.com/chat/api/reference/rest/v1/spaces#spacetype)). To filter by space type, you must specify valid enum value, such as `SPACE` or `GROUP_CHAT` (the `space_type` can't be `SPACE_TYPE_UNSPECIFIED`). To query for multiple space types, use the `OR` operator. For example, the following queries are valid: ``` space_type = \"SPACE\" spaceType = \"GROUP_CHAT\" OR spaceType = \"DIRECT_MESSAGE\" ``` Invalid queries are rejected by the server with an `INVALID_ARGUMENT` error. With [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts), this field is ignored and the query always returns all spaces. But the Chat API still validates the query syntax with service accounts, so invalid queries are still rejected.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "Optional. The maximum number of spaces to return. The service might return fewer than this value. If unspecified, at most 100 spaces are returned. The maximum value is 1,000. If you use a value more than 1,000, it's automatically changed to 1,000. Negative values return an `INVALID_ARGUMENT` error.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "Optional. A page token, received from a previous list spaces call. Provide this parameter to retrieve the subsequent page. When paginating, the filter value should match the call that provided the page token. Passing a different value may lead to unexpected results.",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/spaces",
	//   "response": {
	//     "$ref": "ListSpacesResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.spaces",
	//     "https://www.googleapis.com/auth/chat.spaces.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesListCall) Pages(ctx context.Context, f func(*ListSpacesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "chat.spaces.patch":

type SpacesPatchCall struct {
	s          *Service
	name       string
	space      *Space
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a space. For an example, see Update a space
// (https://developers.google.com/chat/api/guides/v1/spaces/update). If
// you're updating the `displayName` field and receive the error message
// `ALREADY_EXISTS`, try a different display name.. An existing space
// within the Google Workspace organization might already use this
// display name. Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
// - name: Resource name of the space. Format: `spaces/{space}`.
func (r *SpacesService) Patch(name string, space *Space) *SpacesPatchCall {
	c := &SpacesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.space = space
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The
// updated field paths, comma separated if there are multiple. Currently
// supported field paths: - `display_name` (Only supports changing the
// display name of a space with the `SPACE` type, or when also including
// the `space_type` mask to change a `GROUP_CHAT` space type to `SPACE`.
// Trying to update the display name of a `GROUP_CHAT` or a
// `DIRECT_MESSAGE` space results in an invalid argument error. If you
// receive the error message `ALREADY_EXISTS` when updating the
// `displayName`, try a different `displayName`. An existing space
// within the Google Workspace organization might already use this
// display name.) - `space_type` (Only supports changing a `GROUP_CHAT`
// space type to `SPACE`. Include `display_name` together with
// `space_type` in the update mask and ensure that the specified space
// has a non-empty display name and the `SPACE` space type. Including
// the `space_type` mask and the `SPACE` type in the specified space
// when updating the display name is optional if the existing space
// already has the `SPACE` type. Trying to update the space type in
// other ways results in an invalid argument error). - `space_details` -
// `space_history_state` (Supports turning history on or off for the
// space (https://support.google.com/chat/answer/7664687) if the
// organization allows users to change their history setting
// (https://support.google.com/a/answer/7664184). Warning: mutually
// exclusive with all other field paths.)
func (c *SpacesPatchCall) UpdateMask(updateMask string) *SpacesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesPatchCall) Fields(s ...googleapi.Field) *SpacesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesPatchCall) Context(ctx context.Context) *SpacesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.space)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.patch" call.
// Exactly one of *Space or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesPatchCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Updates a space. For an example, see [Update a space](https://developers.google.com/chat/api/guides/v1/spaces/update). If you're updating the `displayName` field and receive the error message `ALREADY_EXISTS`, try a different display name.. An existing space within the Google Workspace organization might already use this display name. Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}",
	//   "httpMethod": "PATCH",
	//   "id": "chat.spaces.patch",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Resource name of the space. Format: `spaces/{space}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "updateMask": {
	//       "description": "Required. The updated field paths, comma separated if there are multiple. Currently supported field paths: - `display_name` (Only supports changing the display name of a space with the `SPACE` type, or when also including the `space_type` mask to change a `GROUP_CHAT` space type to `SPACE`. Trying to update the display name of a `GROUP_CHAT` or a `DIRECT_MESSAGE` space results in an invalid argument error. If you receive the error message `ALREADY_EXISTS` when updating the `displayName`, try a different `displayName`. An existing space within the Google Workspace organization might already use this display name.) - `space_type` (Only supports changing a `GROUP_CHAT` space type to `SPACE`. Include `display_name` together with `space_type` in the update mask and ensure that the specified space has a non-empty display name and the `SPACE` space type. Including the `space_type` mask and the `SPACE` type in the specified space when updating the display name is optional if the existing space already has the `SPACE` type. Trying to update the space type in other ways results in an invalid argument error). - `space_details` - `space_history_state` (Supports [turning history on or off for the space](https://support.google.com/chat/answer/7664687) if [the organization allows users to change their history setting](https://support.google.com/a/answer/7664184). Warning: mutually exclusive with all other field paths.)",
	//       "format": "google-fieldmask",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "request": {
	//     "$ref": "Space"
	//   },
	//   "response": {
	//     "$ref": "Space"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.spaces"
	//   ]
	// }

}

// method id "chat.spaces.setup":

type SpacesSetupCall struct {
	s                 *Service
	setupspacerequest *SetUpSpaceRequest
	urlParams_        gensupport.URLParams
	ctx_              context.Context
	header_           http.Header
}

// Setup: Creates a space and adds specified users to it. The calling
// user is automatically added to the space, and shouldn't be specified
// as a membership in the request. For an example, see Set up a space
// (https://developers.google.com/chat/api/guides/v1/spaces/set-up). To
// specify the human members to add, add memberships with the
// appropriate `member.name` in the `SetUpSpaceRequest`. To add a human
// user, use `users/{user}`, where `{user}` can be the email address for
// the user. For users in the same Workspace organization `{user}` can
// also be the `{person_id}` for the person from the People API, or the
// `id` for the user in the Directory API. For example, if the People
// API Person `resourceName` for `user@example.com` is
// `people/123456789`, you can add the user to the space by setting the
// `membership.member.name` to `users/user@example.com` or
// `users/123456789`. For a space or group chat, if the caller blocks or
// is blocked by some members, then those members aren't added to the
// created space. To create a direct message (DM) between the calling
// user and another human user, specify exactly one membership to
// represent the human user. If one user blocks the other, the request
// fails and the DM isn't created. To create a DM between the calling
// user and the calling app, set `Space.singleUserBotDm` to `true` and
// don't specify any memberships. You can only use this method to set up
// a DM with the calling app. To add the calling app as a member of a
// space or an existing DM between two human users, see create a
// membership
// (https://developers.google.com/chat/api/guides/v1/members/create). If
// a DM already exists between two users, even when one user blocks the
// other at the time a request is made, then the existing DM is
// returned. Spaces with threaded replies aren't supported. If you
// receive the error message `ALREADY_EXISTS` when setting up a space,
// try a different `displayName`. An existing space within the Google
// Workspace organization might already use this display name. Requires
// user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
func (r *SpacesService) Setup(setupspacerequest *SetUpSpaceRequest) *SpacesSetupCall {
	c := &SpacesSetupCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.setupspacerequest = setupspacerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesSetupCall) Fields(s ...googleapi.Field) *SpacesSetupCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesSetupCall) Context(ctx context.Context) *SpacesSetupCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesSetupCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesSetupCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.setupspacerequest)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces:setup")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.setup" call.
// Exactly one of *Space or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesSetupCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Creates a space and adds specified users to it. The calling user is automatically added to the space, and shouldn't be specified as a membership in the request. For an example, see [Set up a space](https://developers.google.com/chat/api/guides/v1/spaces/set-up). To specify the human members to add, add memberships with the appropriate `member.name` in the `SetUpSpaceRequest`. To add a human user, use `users/{user}`, where `{user}` can be the email address for the user. For users in the same Workspace organization `{user}` can also be the `{person_id}` for the person from the People API, or the `id` for the user in the Directory API. For example, if the People API Person `resourceName` for `user@example.com` is `people/123456789`, you can add the user to the space by setting the `membership.member.name` to `users/user@example.com` or `users/123456789`. For a space or group chat, if the caller blocks or is blocked by some members, then those members aren't added to the created space. To create a direct message (DM) between the calling user and another human user, specify exactly one membership to represent the human user. If one user blocks the other, the request fails and the DM isn't created. To create a DM between the calling user and the calling app, set `Space.singleUserBotDm` to `true` and don't specify any memberships. You can only use this method to set up a DM with the calling app. To add the calling app as a member of a space or an existing DM between two human users, see [create a membership](https://developers.google.com/chat/api/guides/v1/members/create). If a DM already exists between two users, even when one user blocks the other at the time a request is made, then the existing DM is returned. Spaces with threaded replies aren't supported. If you receive the error message `ALREADY_EXISTS` when setting up a space, try a different `displayName`. An existing space within the Google Workspace organization might already use this display name. Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces:setup",
	//   "httpMethod": "POST",
	//   "id": "chat.spaces.setup",
	//   "parameterOrder": [],
	//   "parameters": {},
	//   "path": "v1/spaces:setup",
	//   "request": {
	//     "$ref": "SetUpSpaceRequest"
	//   },
	//   "response": {
	//     "$ref": "Space"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.spaces",
	//     "https://www.googleapis.com/auth/chat.spaces.create"
	//   ]
	// }

}

// method id "chat.spaces.members.create":

type SpacesMembersCreateCall struct {
	s          *Service
	parent     string
	membership *Membership
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a human membership or app membership for the calling
// app. Creating memberships for other apps isn't supported. For an
// example, see  Create a membership
// (https://developers.google.com/chat/api/guides/v1/members/create).
// When creating a membership, if the specified member has their
// auto-accept policy turned off, then they're invited, and must accept
// the space invitation before joining. Otherwise, creating a membership
// adds the member directly to the specified space. Requires user
// authentication
// (https://developers.google.com/chat/api/guides/auth/users). To
// specify the member to add, set the `membership.member.name` in the
// `CreateMembershipRequest`: - To add the calling app to a space or a
// direct message between two human users, use `users/app`. Unable to
// add other apps to the space. - To add a human user, use
// `users/{user}`, where `{user}` can be the email address for the user.
// For users in the same Workspace organization `{user}` can also be the
// `{person_id}` for the person from the People API, or the `id` for the
// user in the Directory API. For example, if the People API Person
// `resourceName` for `user@example.com` is `people/123456789`, you can
// add the user to the space by setting the `membership.member.name` to
// `users/user@example.com` or `users/123456789`.
//
//   - parent: The resource name of the space for which to create the
//     membership. Format: spaces/{space}.
func (r *SpacesMembersService) Create(parent string, membership *Membership) *SpacesMembersCreateCall {
	c := &SpacesMembersCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.membership = membership
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMembersCreateCall) Fields(s ...googleapi.Field) *SpacesMembersCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMembersCreateCall) Context(ctx context.Context) *SpacesMembersCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMembersCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.membership)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/members")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.create" call.
// Exactly one of *Membership or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMembersCreateCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Creates a human membership or app membership for the calling app. Creating memberships for other apps isn't supported. For an example, see [ Create a membership](https://developers.google.com/chat/api/guides/v1/members/create). When creating a membership, if the specified member has their auto-accept policy turned off, then they're invited, and must accept the space invitation before joining. Otherwise, creating a membership adds the member directly to the specified space. Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users). To specify the member to add, set the `membership.member.name` in the `CreateMembershipRequest`: - To add the calling app to a space or a direct message between two human users, use `users/app`. Unable to add other apps to the space. - To add a human user, use `users/{user}`, where `{user}` can be the email address for the user. For users in the same Workspace organization `{user}` can also be the `{person_id}` for the person from the People API, or the `id` for the user in the Directory API. For example, if the People API Person `resourceName` for `user@example.com` is `people/123456789`, you can add the user to the space by setting the `membership.member.name` to `users/user@example.com` or `users/123456789`.",
	//   "flatPath": "v1/spaces/{spacesId}/members",
	//   "httpMethod": "POST",
	//   "id": "chat.spaces.members.create",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "parent": {
	//       "description": "Required. The resource name of the space for which to create the membership. Format: spaces/{space}",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+parent}/members",
	//   "request": {
	//     "$ref": "Membership"
	//   },
	//   "response": {
	//     "$ref": "Membership"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.memberships",
	//     "https://www.googleapis.com/auth/chat.memberships.app"
	//   ]
	// }

}

// method id "chat.spaces.members.delete":

type SpacesMembersDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a membership. For an example, see Delete a membership
// (https://developers.google.com/chat/api/guides/v1/members/delete).
// Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
//   - name: Resource name of the membership to delete. Chat apps can
//     delete human users' or their own memberships. Chat apps can't
//     delete other apps' memberships. When deleting a human membership,
//     requires the `chat.memberships` scope and
//     `spaces/{space}/members/{member}` format. You can use the email as
//     an alias for `{member}`. For example,
//     `spaces/{space}/members/example@gmail.com` where
//     `example@gmail.com` is the email of the Google Chat user. When
//     deleting an app membership, requires the `chat.memberships.app`
//     scope and `spaces/{space}/members/app` format. Format:
//     `spaces/{space}/members/{member}` or `spaces/{space}/members/app`.
func (r *SpacesMembersService) Delete(name string) *SpacesMembersDeleteCall {
	c := &SpacesMembersDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMembersDeleteCall) Fields(s ...googleapi.Field) *SpacesMembersDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMembersDeleteCall) Context(ctx context.Context) *SpacesMembersDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMembersDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.delete" call.
// Exactly one of *Membership or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMembersDeleteCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Deletes a membership. For an example, see [Delete a membership](https://developers.google.com/chat/api/guides/v1/members/delete). Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}/members/{membersId}",
	//   "httpMethod": "DELETE",
	//   "id": "chat.spaces.members.delete",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the membership to delete. Chat apps can delete human users' or their own memberships. Chat apps can't delete other apps' memberships. When deleting a human membership, requires the `chat.memberships` scope and `spaces/{space}/members/{member}` format. You can use the email as an alias for `{member}`. For example, `spaces/{space}/members/example@gmail.com` where `example@gmail.com` is the email of the Google Chat user. When deleting an app membership, requires the `chat.memberships.app` scope and `spaces/{space}/members/app` format. Format: `spaces/{space}/members/{member}` or `spaces/{space}/members/app`.",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/members/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Membership"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.memberships",
	//     "https://www.googleapis.com/auth/chat.memberships.app"
	//   ]
	// }

}

// method id "chat.spaces.members.get":

type SpacesMembersGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns details about a membership. For an example, see Get a
// membership
// (https://developers.google.com/chat/api/guides/v1/members/get).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users). User
// authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
//   - name: Resource name of the membership to retrieve. To get the app's
//     own membership, you can optionally use
//     `spaces/{space}/members/app`. Format:
//     `spaces/{space}/members/{member}` or `spaces/{space}/members/app`
//     When authenticated as a user
//     (https://developers.google.com/chat/api/guides/auth/users), you can
//     use the user's email as an alias for `{member}`. For example,
//     `spaces/{space}/members/example@gmail.com` where
//     `example@gmail.com` is the email of the Google Chat user.
func (r *SpacesMembersService) Get(name string) *SpacesMembersGetCall {
	c := &SpacesMembersGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMembersGetCall) Fields(s ...googleapi.Field) *SpacesMembersGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMembersGetCall) IfNoneMatch(entityTag string) *SpacesMembersGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMembersGetCall) Context(ctx context.Context) *SpacesMembersGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMembersGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.get" call.
// Exactly one of *Membership or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMembersGetCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Returns details about a membership. For an example, see [Get a membership](https://developers.google.com/chat/api/guides/v1/members/get). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users). [User authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}/members/{membersId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.members.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the membership to retrieve. To get the app's own membership, you can optionally use `spaces/{space}/members/app`. Format: `spaces/{space}/members/{member}` or `spaces/{space}/members/app` When [authenticated as a user](https://developers.google.com/chat/api/guides/auth/users), you can use the user's email as an alias for `{member}`. For example, `spaces/{space}/members/example@gmail.com` where `example@gmail.com` is the email of the Google Chat user.",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/members/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Membership"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.memberships",
	//     "https://www.googleapis.com/auth/chat.memberships.readonly"
	//   ]
	// }

}

// method id "chat.spaces.members.list":

type SpacesMembersListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists memberships in a space. For an example, see List
// memberships
// (https://developers.google.com/chat/api/guides/v1/members/list).
// Listing memberships with app authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// lists memberships in spaces that the Chat app has access to, but
// excludes Chat app memberships, including its own. Listing memberships
// with User authentication
// (https://developers.google.com/chat/api/guides/auth/users) lists
// memberships in spaces that the authenticated user has access to.
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users). User
// authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
//   - parent: The resource name of the space for which to fetch a
//     membership list. Format: spaces/{space}.
func (r *SpacesMembersService) List(parent string) *SpacesMembersListCall {
	c := &SpacesMembersListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A query filter. You can
// filter memberships by a member's role (`role`
// (https://developers.google.com/chat/api/reference/rest/v1/spaces.members#membershiprole))
// and type (`member.type`
// (https://developers.google.com/chat/api/reference/rest/v1/User#type)).
// To filter by role, set `role` to `ROLE_MEMBER` or `ROLE_MANAGER`. To
// filter by type, set `member.type` to `HUMAN` or `BOT`. To filter by
// both role and type, use the `AND` operator. To filter by either role
// or type, use the `OR` operator. For example, the following queries
// are valid: ``` role = "ROLE_MANAGER" OR role = "ROLE_MEMBER"
// member.type = "HUMAN" AND role = "ROLE_MANAGER" ``` The following
// queries are invalid: ``` member.type = "HUMAN" AND member.type =
// "BOT" role = "ROLE_MANAGER" AND role = "ROLE_MEMBER" ``` Invalid
// queries are rejected by the server with an `INVALID_ARGUMENT` error.
func (c *SpacesMembersListCall) Filter(filter string) *SpacesMembersListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number
// of memberships to return. The service might return fewer than this
// value. If unspecified, at most 100 memberships are returned. The
// maximum value is 1,000. If you use a value more than 1,000, it's
// automatically changed to 1,000. Negative values return an
// `INVALID_ARGUMENT` error.
func (c *SpacesMembersListCall) PageSize(pageSize int64) *SpacesMembersListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token,
// received from a previous call to list memberships. Provide this
// parameter to retrieve the subsequent page. When paginating, all other
// parameters provided should match the call that provided the page
// token. Passing different values to the other parameters might lead to
// unexpected results.
func (c *SpacesMembersListCall) PageToken(pageToken string) *SpacesMembersListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// ShowInvited sets the optional parameter "showInvited": When `true`,
// also returns memberships associated with invited members, in addition
// to other types of memberships. If a filter is set, invited
// memberships that don't match the filter criteria aren't returned.
// Currently requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
func (c *SpacesMembersListCall) ShowInvited(showInvited bool) *SpacesMembersListCall {
	c.urlParams_.Set("showInvited", fmt.Sprint(showInvited))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMembersListCall) Fields(s ...googleapi.Field) *SpacesMembersListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMembersListCall) IfNoneMatch(entityTag string) *SpacesMembersListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMembersListCall) Context(ctx context.Context) *SpacesMembersListCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMembersListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/members")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.list" call.
// Exactly one of *ListMembershipsResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListMembershipsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *SpacesMembersListCall) Do(opts ...googleapi.CallOption) (*ListMembershipsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListMembershipsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Lists memberships in a space. For an example, see [List memberships](https://developers.google.com/chat/api/guides/v1/members/list). Listing memberships with [app authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) lists memberships in spaces that the Chat app has access to, but excludes Chat app memberships, including its own. Listing memberships with [User authentication](https://developers.google.com/chat/api/guides/auth/users) lists memberships in spaces that the authenticated user has access to. Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users). [User authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}/members",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.members.list",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "filter": {
	//       "description": "Optional. A query filter. You can filter memberships by a member's role ([`role`](https://developers.google.com/chat/api/reference/rest/v1/spaces.members#membershiprole)) and type ([`member.type`](https://developers.google.com/chat/api/reference/rest/v1/User#type)). To filter by role, set `role` to `ROLE_MEMBER` or `ROLE_MANAGER`. To filter by type, set `member.type` to `HUMAN` or `BOT`. To filter by both role and type, use the `AND` operator. To filter by either role or type, use the `OR` operator. For example, the following queries are valid: ``` role = \"ROLE_MANAGER\" OR role = \"ROLE_MEMBER\" member.type = \"HUMAN\" AND role = \"ROLE_MANAGER\" ``` The following queries are invalid: ``` member.type = \"HUMAN\" AND member.type = \"BOT\" role = \"ROLE_MANAGER\" AND role = \"ROLE_MEMBER\" ``` Invalid queries are rejected by the server with an `INVALID_ARGUMENT` error.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "Optional. The maximum number of memberships to return. The service might return fewer than this value. If unspecified, at most 100 memberships are returned. The maximum value is 1,000. If you use a value more than 1,000, it's automatically changed to 1,000. Negative values return an `INVALID_ARGUMENT` error.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "Optional. A page token, received from a previous call to list memberships. Provide this parameter to retrieve the subsequent page. When paginating, all other parameters provided should match the call that provided the page token. Passing different values to the other parameters might lead to unexpected results.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "parent": {
	//       "description": "Required. The resource name of the space for which to fetch a membership list. Format: spaces/{space}",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "showInvited": {
	//       "description": "Optional. When `true`, also returns memberships associated with invited members, in addition to other types of memberships. If a filter is set, invited memberships that don't match the filter criteria aren't returned. Currently requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//       "location": "query",
	//       "type": "boolean"
	//     }
	//   },
	//   "path": "v1/{+parent}/members",
	//   "response": {
	//     "$ref": "ListMembershipsResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.memberships",
	//     "https://www.googleapis.com/auth/chat.memberships.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesMembersListCall) Pages(ctx context.Context, f func(*ListMembershipsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "chat.spaces.messages.create":

type SpacesMessagesCreateCall struct {
	s          *Service
	parent     string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a message in a Google Chat space. For an example, see
// Create a message
// (https://developers.google.com/chat/api/guides/v1/messages/create).
// Calling this method requires authentication
// (https://developers.google.com/chat/api/guides/auth) and supports the
// following authentication types: - For text messages, user
// authentication or app authentication are supported. - For card
// messages, only app authentication is supported. (Only Chat apps can
// create card messages.)
//
//   - parent: The resource name of the space in which to create a
//     message. Format: `spaces/{space}`.
func (r *SpacesMessagesService) Create(parent string, message *Message) *SpacesMessagesCreateCall {
	c := &SpacesMessagesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.message = message
	return c
}

// MessageId sets the optional parameter "messageId": A custom name for
// a Chat message assigned at creation. Must start with `client-` and
// contain only lowercase letters, numbers, and hyphens up to 63
// characters in length. Specify this field to get, update, or delete
// the message with the specified value. Assigning a custom name lets a
// a Chat app recall the message without saving the message `name` from
// the response body
// (/chat/api/reference/rest/v1/spaces.messages/get#response-body)
// returned when creating the message. Assigning a custom name doesn't
// replace the generated `name` field, the message's resource name.
// Instead, it sets the custom name as the `clientAssignedMessageId`
// field, which you can reference while processing later operations,
// like updating or deleting the message. For example usage, see Name a
// created message
// (https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message).
func (c *SpacesMessagesCreateCall) MessageId(messageId string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("messageId", messageId)
	return c
}

// MessageReplyOption sets the optional parameter "messageReplyOption":
// Specifies whether a message starts a thread or replies to one. Only
// supported in named spaces.
//
// Possible values:
//
//	"MESSAGE_REPLY_OPTION_UNSPECIFIED" - Default. Starts a thread.
//	"REPLY_MESSAGE_FALLBACK_TO_NEW_THREAD" - Creates the message as a
//
// reply to the thread specified by thread ID or `thread_key`. If it
// fails, the message starts a new thread instead.
//
//	"REPLY_MESSAGE_OR_FAIL" - Creates the message as a reply to the
//
// thread specified by thread ID or `thread_key`. If it fails, a
// `NOT_FOUND` error is returned instead.
func (c *SpacesMessagesCreateCall) MessageReplyOption(messageReplyOption string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("messageReplyOption", messageReplyOption)
	return c
}

// RequestId sets the optional parameter "requestId": A unique request
// ID for this message. Specifying an existing request ID returns the
// message created with that ID instead of creating a new message.
func (c *SpacesMessagesCreateCall) RequestId(requestId string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("requestId", requestId)
	return c
}

// ThreadKey sets the optional parameter "threadKey": Deprecated: Use
// thread.thread_key instead. ID for the thread. Supports up to 4000
// characters. To start or add to a thread, create a message and specify
// a `threadKey` or the thread.name. For example usage, see Start or
// reply to a message thread
// (https://developers.google.com/chat/api/guides/crudl/messages#start_or_reply_to_a_message_thread).
func (c *SpacesMessagesCreateCall) ThreadKey(threadKey string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("threadKey", threadKey)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesCreateCall) Fields(s ...googleapi.Field) *SpacesMessagesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesCreateCall) Context(ctx context.Context) *SpacesMessagesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.message)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/messages")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.create" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesCreateCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Creates a message in a Google Chat space. For an example, see [Create a message](https://developers.google.com/chat/api/guides/v1/messages/create). Calling this method requires [authentication](https://developers.google.com/chat/api/guides/auth) and supports the following authentication types: - For text messages, user authentication or app authentication are supported. - For card messages, only app authentication is supported. (Only Chat apps can create card messages.)",
	//   "flatPath": "v1/spaces/{spacesId}/messages",
	//   "httpMethod": "POST",
	//   "id": "chat.spaces.messages.create",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "messageId": {
	//       "description": "Optional. A custom name for a Chat message assigned at creation. Must start with `client-` and contain only lowercase letters, numbers, and hyphens up to 63 characters in length. Specify this field to get, update, or delete the message with the specified value. Assigning a custom name lets a a Chat app recall the message without saving the message `name` from the [response body](/chat/api/reference/rest/v1/spaces.messages/get#response-body) returned when creating the message. Assigning a custom name doesn't replace the generated `name` field, the message's resource name. Instead, it sets the custom name as the `clientAssignedMessageId` field, which you can reference while processing later operations, like updating or deleting the message. For example usage, see [Name a created message](https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message).",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "messageReplyOption": {
	//       "description": "Optional. Specifies whether a message starts a thread or replies to one. Only supported in named spaces.",
	//       "enum": [
	//         "MESSAGE_REPLY_OPTION_UNSPECIFIED",
	//         "REPLY_MESSAGE_FALLBACK_TO_NEW_THREAD",
	//         "REPLY_MESSAGE_OR_FAIL"
	//       ],
	//       "enumDescriptions": [
	//         "Default. Starts a thread.",
	//         "Creates the message as a reply to the thread specified by thread ID or `thread_key`. If it fails, the message starts a new thread instead.",
	//         "Creates the message as a reply to the thread specified by thread ID or `thread_key`. If it fails, a `NOT_FOUND` error is returned instead."
	//       ],
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "parent": {
	//       "description": "Required. The resource name of the space in which to create a message. Format: `spaces/{space}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "requestId": {
	//       "description": "Optional. A unique request ID for this message. Specifying an existing request ID returns the message created with that ID instead of creating a new message.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "threadKey": {
	//       "deprecated": true,
	//       "description": "Optional. Deprecated: Use thread.thread_key instead. ID for the thread. Supports up to 4000 characters. To start or add to a thread, create a message and specify a `threadKey` or the thread.name. For example usage, see [Start or reply to a message thread](https://developers.google.com/chat/api/guides/crudl/messages#start_or_reply_to_a_message_thread).",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+parent}/messages",
	//   "request": {
	//     "$ref": "Message"
	//   },
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.create"
	//   ]
	// }

}

// method id "chat.spaces.messages.delete":

type SpacesMessagesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a message. For an example, see Delete a message
// (https://developers.google.com/chat/api/guides/v1/messages/delete).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users). User
// authentication
// (https://developers.google.com/chat/api/guides/auth/users). Requests
// authenticated with service accounts can only delete messages created
// by the calling Chat app.
//
//   - name: Resource name of the message that you want to delete, in the
//     form `spaces/*/messages/*` Example:
//     `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`.
func (r *SpacesMessagesService) Delete(name string) *SpacesMessagesDeleteCall {
	c := &SpacesMessagesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Force sets the optional parameter "force": When `true`, deleting a
// message also deletes its threaded replies. When `false`, if a message
// has threaded replies, deletion fails. Only applies when
// authenticating as a user
// (https://developers.google.com/chat/api/guides/auth/users). Has no
// effect when [authenticating with a service account]
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
func (c *SpacesMessagesDeleteCall) Force(force bool) *SpacesMessagesDeleteCall {
	c.urlParams_.Set("force", fmt.Sprint(force))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesDeleteCall) Fields(s ...googleapi.Field) *SpacesMessagesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesDeleteCall) Context(ctx context.Context) *SpacesMessagesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.delete" call.
// Exactly one of *Empty or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Deletes a message. For an example, see [Delete a message](https://developers.google.com/chat/api/guides/v1/messages/delete). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users). [User authentication](https://developers.google.com/chat/api/guides/auth/users). Requests authenticated with service accounts can only delete messages created by the calling Chat app.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "DELETE",
	//   "id": "chat.spaces.messages.delete",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "force": {
	//       "description": "When `true`, deleting a message also deletes its threaded replies. When `false`, if a message has threaded replies, deletion fails. Only applies when [authenticating as a user](https://developers.google.com/chat/api/guides/auth/users). Has no effect when [authenticating with a service account] (https://developers.google.com/chat/api/guides/auth/service-accounts).",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "name": {
	//       "description": "Required. Resource name of the message that you want to delete, in the form `spaces/*/messages/*` Example: `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Empty"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages"
	//   ]
	// }

}

// method id "chat.spaces.messages.get":

type SpacesMessagesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns details about a message. For an example, see Read a
// message
// (https://developers.google.com/chat/api/guides/v1/messages/get).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users). User
// authentication
// (https://developers.google.com/chat/api/guides/auth/users). Note:
// Might return a message from a blocked member or space.
//
//   - name: Resource name of the message to retrieve. Format:
//     `spaces/{space}/messages/{message}` If the message begins with
//     `client-`, then it has a custom name assigned by a Chat app that
//     created it with the Chat REST API. That Chat app (but not others)
//     can pass the custom name to get, update, or delete the message. To
//     learn more, see [create and name a message]
//     (https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message).
func (r *SpacesMessagesService) Get(name string) *SpacesMessagesGetCall {
	c := &SpacesMessagesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesGetCall) Fields(s ...googleapi.Field) *SpacesMessagesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMessagesGetCall) IfNoneMatch(entityTag string) *SpacesMessagesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesGetCall) Context(ctx context.Context) *SpacesMessagesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.get" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesGetCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Returns details about a message. For an example, see [Read a message](https://developers.google.com/chat/api/guides/v1/messages/get). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users). [User authentication](https://developers.google.com/chat/api/guides/auth/users). Note: Might return a message from a blocked member or space.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.messages.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the message to retrieve. Format: `spaces/{space}/messages/{message}` If the message begins with `client-`, then it has a custom name assigned by a Chat app that created it with the Chat REST API. That Chat app (but not others) can pass the custom name to get, update, or delete the message. To learn more, see [create and name a message] (https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message).",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.readonly"
	//   ]
	// }

}

// method id "chat.spaces.messages.list":

type SpacesMessagesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists messages in a space that the caller is a member of,
// including messages from blocked members and spaces. For an example,
// see List messages (/chat/api/guides/v1/messages/list). Requires user
// authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
//   - parent: The resource name of the space to list messages from.
//     Format: `spaces/{space}`.
func (r *SpacesMessagesService) List(parent string) *SpacesMessagesListCall {
	c := &SpacesMessagesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A query filter. You can
// filter messages by date (`create_time`) and thread (`thread.name`).
// To filter messages by the date they were created, specify the
// `create_time` with a timestamp in RFC-3339
// (https://www.rfc-editor.org/rfc/rfc3339) format and double quotation
// marks. For example, "2023-04-21T11:30:00-04:00". You can use the
// greater than operator `>` to list messages that were created after a
// timestamp, or the less than operator `<` to list messages that were
// created before a timestamp. To filter messages within a time
// interval, use the `AND` operator between two timestamps. To filter by
// thread, specify the `thread.name`, formatted as
// `spaces/{space}/threads/{thread}`. You can only specify one
// `thread.name` per query. To filter by both thread and date, use the
// `AND` operator in your query. For example, the following queries are
// valid: ``` create_time > "2012-04-21T11:30:00-04:00" create_time >
// "2012-04-21T11:30:00-04:00" AND thread.name =
// spaces/AAAAAAAAAAA/threads/123 create_time >
// "2012-04-21T11:30:00+00:00" AND create_time <
// "2013-01-01T00:00:00+00:00" AND thread.name =
// spaces/AAAAAAAAAAA/threads/123 thread.name =
// spaces/AAAAAAAAAAA/threads/123 ``` Invalid queries are rejected by
// the server with an `INVALID_ARGUMENT` error.
func (c *SpacesMessagesListCall) Filter(filter string) *SpacesMessagesListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// OrderBy sets the optional parameter "orderBy": Optional, if resuming
// from a previous query. How the list of messages is ordered. Specify a
// value to order by an ordering operation. Valid ordering operation
// values are as follows: - `ASC` for ascending. - `DESC` for
// descending. The default ordering is `create_time ASC`.
func (c *SpacesMessagesListCall) OrderBy(orderBy string) *SpacesMessagesListCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number
// of messages returned. The service might return fewer messages than
// this value. If unspecified, at most 25 are returned. The maximum
// value is 1,000. If you use a value more than 1,000, it's
// automatically changed to 1,000. Negative values return an
// `INVALID_ARGUMENT` error.
func (c *SpacesMessagesListCall) PageSize(pageSize int64) *SpacesMessagesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Optional, if
// resuming from a previous query. A page token received from a previous
// list messages call. Provide this parameter to retrieve the subsequent
// page. When paginating, all other parameters provided should match the
// call that provided the page token. Passing different values to the
// other parameters might lead to unexpected results.
func (c *SpacesMessagesListCall) PageToken(pageToken string) *SpacesMessagesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// ShowDeleted sets the optional parameter "showDeleted": Whether to
// include deleted messages. Deleted messages include deleted time and
// metadata about their deletion, but message content is unavailable.
func (c *SpacesMessagesListCall) ShowDeleted(showDeleted bool) *SpacesMessagesListCall {
	c.urlParams_.Set("showDeleted", fmt.Sprint(showDeleted))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesListCall) Fields(s ...googleapi.Field) *SpacesMessagesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMessagesListCall) IfNoneMatch(entityTag string) *SpacesMessagesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesListCall) Context(ctx context.Context) *SpacesMessagesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/messages")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.list" call.
// Exactly one of *ListMessagesResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListMessagesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *SpacesMessagesListCall) Do(opts ...googleapi.CallOption) (*ListMessagesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListMessagesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Lists messages in a space that the caller is a member of, including messages from blocked members and spaces. For an example, see [List messages](/chat/api/guides/v1/messages/list). Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}/messages",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.messages.list",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "filter": {
	//       "description": "A query filter. You can filter messages by date (`create_time`) and thread (`thread.name`). To filter messages by the date they were created, specify the `create_time` with a timestamp in [RFC-3339](https://www.rfc-editor.org/rfc/rfc3339) format and double quotation marks. For example, `\"2023-04-21T11:30:00-04:00\"`. You can use the greater than operator `\u003e` to list messages that were created after a timestamp, or the less than operator `\u003c` to list messages that were created before a timestamp. To filter messages within a time interval, use the `AND` operator between two timestamps. To filter by thread, specify the `thread.name`, formatted as `spaces/{space}/threads/{thread}`. You can only specify one `thread.name` per query. To filter by both thread and date, use the `AND` operator in your query. For example, the following queries are valid: ``` create_time \u003e \"2012-04-21T11:30:00-04:00\" create_time \u003e \"2012-04-21T11:30:00-04:00\" AND thread.name = spaces/AAAAAAAAAAA/threads/123 create_time \u003e \"2012-04-21T11:30:00+00:00\" AND create_time \u003c \"2013-01-01T00:00:00+00:00\" AND thread.name = spaces/AAAAAAAAAAA/threads/123 thread.name = spaces/AAAAAAAAAAA/threads/123 ``` Invalid queries are rejected by the server with an `INVALID_ARGUMENT` error.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "orderBy": {
	//       "description": "Optional, if resuming from a previous query. How the list of messages is ordered. Specify a value to order by an ordering operation. Valid ordering operation values are as follows: - `ASC` for ascending. - `DESC` for descending. The default ordering is `create_time ASC`.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "The maximum number of messages returned. The service might return fewer messages than this value. If unspecified, at most 25 are returned. The maximum value is 1,000. If you use a value more than 1,000, it's automatically changed to 1,000. Negative values return an `INVALID_ARGUMENT` error.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "Optional, if resuming from a previous query. A page token received from a previous list messages call. Provide this parameter to retrieve the subsequent page. When paginating, all other parameters provided should match the call that provided the page token. Passing different values to the other parameters might lead to unexpected results.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "parent": {
	//       "description": "Required. The resource name of the space to list messages from. Format: `spaces/{space}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "showDeleted": {
	//       "description": "Whether to include deleted messages. Deleted messages include deleted time and metadata about their deletion, but message content is unavailable.",
	//       "location": "query",
	//       "type": "boolean"
	//     }
	//   },
	//   "path": "v1/{+parent}/messages",
	//   "response": {
	//     "$ref": "ListMessagesResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesMessagesListCall) Pages(ctx context.Context, f func(*ListMessagesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "chat.spaces.messages.patch":

type SpacesMessagesPatchCall struct {
	s          *Service
	name       string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a message. There's a difference between the `patch`
// and `update` methods. The `patch` method uses a `patch` request while
// the `update` method uses a `put` request. We recommend using the
// `patch` method. For an example, see Update a message
// (https://developers.google.com/chat/api/guides/v1/messages/update).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users). User
// authentication
// (https://developers.google.com/chat/api/guides/auth/users). Requests
// authenticated with service accounts can only update messages created
// by the calling Chat app.
//
//   - name: Resource name in the form `spaces/*/messages/*`. Example:
//     `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`.
func (r *SpacesMessagesService) Patch(name string, message *Message) *SpacesMessagesPatchCall {
	c := &SpacesMessagesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.message = message
	return c
}

// AllowMissing sets the optional parameter "allowMissing": If `true`
// and the message isn't found, a new message is created and
// `updateMask` is ignored. The specified message ID must be
// client-assigned
// (https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message)
// or the request fails.
func (c *SpacesMessagesPatchCall) AllowMissing(allowMissing bool) *SpacesMessagesPatchCall {
	c.urlParams_.Set("allowMissing", fmt.Sprint(allowMissing))
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The
// field paths to update. Separate multiple values with commas.
// Currently supported field paths: - `text` - `attachment` - `cards`
// (Requires service account authentication
// (/chat/api/guides/auth/service-accounts).) - `cards_v2` (Requires
// service account authentication
// (/chat/api/guides/auth/service-accounts).)
func (c *SpacesMessagesPatchCall) UpdateMask(updateMask string) *SpacesMessagesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesPatchCall) Fields(s ...googleapi.Field) *SpacesMessagesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesPatchCall) Context(ctx context.Context) *SpacesMessagesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.message)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.patch" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesPatchCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Updates a message. There's a difference between the `patch` and `update` methods. The `patch` method uses a `patch` request while the `update` method uses a `put` request. We recommend using the `patch` method. For an example, see [Update a message](https://developers.google.com/chat/api/guides/v1/messages/update). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users). [User authentication](https://developers.google.com/chat/api/guides/auth/users). Requests authenticated with service accounts can only update messages created by the calling Chat app.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "PATCH",
	//   "id": "chat.spaces.messages.patch",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "allowMissing": {
	//       "description": "Optional. If `true` and the message isn't found, a new message is created and `updateMask` is ignored. The specified message ID must be [client-assigned](https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message) or the request fails.",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "name": {
	//       "description": "Resource name in the form `spaces/*/messages/*`. Example: `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "updateMask": {
	//       "description": "Required. The field paths to update. Separate multiple values with commas. Currently supported field paths: - `text` - `attachment` - `cards` (Requires [service account authentication](/chat/api/guides/auth/service-accounts).) - `cards_v2` (Requires [service account authentication](/chat/api/guides/auth/service-accounts).)",
	//       "format": "google-fieldmask",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "request": {
	//     "$ref": "Message"
	//   },
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages"
	//   ]
	// }

}

// method id "chat.spaces.messages.update":

type SpacesMessagesUpdateCall struct {
	s          *Service
	name       string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates a message. There's a difference between the `patch`
// and `update` methods. The `patch` method uses a `patch` request while
// the `update` method uses a `put` request. We recommend using the
// `patch` method. For an example, see Update a message
// (https://developers.google.com/chat/api/guides/v1/messages/update).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts)
// and user authentication
// (https://developers.google.com/chat/api/guides/auth/users). User
// authentication
// (https://developers.google.com/chat/api/guides/auth/users). Requests
// authenticated with service accounts can only update messages created
// by the calling Chat app.
//
//   - name: Resource name in the form `spaces/*/messages/*`. Example:
//     `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`.
func (r *SpacesMessagesService) Update(name string, message *Message) *SpacesMessagesUpdateCall {
	c := &SpacesMessagesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.message = message
	return c
}

// AllowMissing sets the optional parameter "allowMissing": If `true`
// and the message isn't found, a new message is created and
// `updateMask` is ignored. The specified message ID must be
// client-assigned
// (https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message)
// or the request fails.
func (c *SpacesMessagesUpdateCall) AllowMissing(allowMissing bool) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("allowMissing", fmt.Sprint(allowMissing))
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The
// field paths to update. Separate multiple values with commas.
// Currently supported field paths: - `text` - `attachment` - `cards`
// (Requires service account authentication
// (/chat/api/guides/auth/service-accounts).) - `cards_v2` (Requires
// service account authentication
// (/chat/api/guides/auth/service-accounts).)
func (c *SpacesMessagesUpdateCall) UpdateMask(updateMask string) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesUpdateCall) Fields(s ...googleapi.Field) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesUpdateCall) Context(ctx context.Context) *SpacesMessagesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.message)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.update" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesUpdateCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Updates a message. There's a difference between the `patch` and `update` methods. The `patch` method uses a `patch` request while the `update` method uses a `put` request. We recommend using the `patch` method. For an example, see [Update a message](https://developers.google.com/chat/api/guides/v1/messages/update). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts) and [user authentication](https://developers.google.com/chat/api/guides/auth/users). [User authentication](https://developers.google.com/chat/api/guides/auth/users). Requests authenticated with service accounts can only update messages created by the calling Chat app.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "PUT",
	//   "id": "chat.spaces.messages.update",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "allowMissing": {
	//       "description": "Optional. If `true` and the message isn't found, a new message is created and `updateMask` is ignored. The specified message ID must be [client-assigned](https://developers.google.com/chat/api/guides/v1/messages/create#name_a_created_message) or the request fails.",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "name": {
	//       "description": "Resource name in the form `spaces/*/messages/*`. Example: `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "updateMask": {
	//       "description": "Required. The field paths to update. Separate multiple values with commas. Currently supported field paths: - `text` - `attachment` - `cards` (Requires [service account authentication](/chat/api/guides/auth/service-accounts).) - `cards_v2` (Requires [service account authentication](/chat/api/guides/auth/service-accounts).)",
	//       "format": "google-fieldmask",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "request": {
	//     "$ref": "Message"
	//   },
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages"
	//   ]
	// }

}

// method id "chat.spaces.messages.attachments.get":

type SpacesMessagesAttachmentsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the metadata of a message attachment. The attachment data
// is fetched using the media API
// (https://developers.google.com/chat/api/reference/rest/v1/media/download).
// For an example, see Get a message attachment
// (https://developers.google.com/chat/api/guides/v1/media-and-attachments/get).
// Requires service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
//
//   - name: Resource name of the attachment, in the form
//     `spaces/*/messages/*/attachments/*`.
func (r *SpacesMessagesAttachmentsService) Get(name string) *SpacesMessagesAttachmentsGetCall {
	c := &SpacesMessagesAttachmentsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesAttachmentsGetCall) Fields(s ...googleapi.Field) *SpacesMessagesAttachmentsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMessagesAttachmentsGetCall) IfNoneMatch(entityTag string) *SpacesMessagesAttachmentsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesAttachmentsGetCall) Context(ctx context.Context) *SpacesMessagesAttachmentsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesAttachmentsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesAttachmentsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.attachments.get" call.
// Exactly one of *Attachment or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *Attachment.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMessagesAttachmentsGetCall) Do(opts ...googleapi.CallOption) (*Attachment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Attachment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Gets the metadata of a message attachment. The attachment data is fetched using the [media API](https://developers.google.com/chat/api/reference/rest/v1/media/download). For an example, see [Get a message attachment](https://developers.google.com/chat/api/guides/v1/media-and-attachments/get). Requires [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts).",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}/attachments/{attachmentsId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.messages.attachments.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the attachment, in the form `spaces/*/messages/*/attachments/*`.",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+/attachments/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Attachment"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot"
	//   ]
	// }

}

// method id "chat.spaces.messages.reactions.create":

type SpacesMessagesReactionsCreateCall struct {
	s          *Service
	parent     string
	reaction   *Reaction
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a reaction and adds it to a message. For an example,
// see Create a reaction
// (https://developers.google.com/chat/api/guides/v1/reactions/create).
// Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users). Only
// unicode emoji are supported.
//
//   - parent: The message where the reaction is created. Format:
//     `spaces/{space}/messages/{message}`.
func (r *SpacesMessagesReactionsService) Create(parent string, reaction *Reaction) *SpacesMessagesReactionsCreateCall {
	c := &SpacesMessagesReactionsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.reaction = reaction
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesReactionsCreateCall) Fields(s ...googleapi.Field) *SpacesMessagesReactionsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesReactionsCreateCall) Context(ctx context.Context) *SpacesMessagesReactionsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesReactionsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesReactionsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.reaction)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/reactions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.reactions.create" call.
// Exactly one of *Reaction or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Reaction.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMessagesReactionsCreateCall) Do(opts ...googleapi.CallOption) (*Reaction, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Reaction{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Creates a reaction and adds it to a message. For an example, see [Create a reaction](https://developers.google.com/chat/api/guides/v1/reactions/create). Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users). Only unicode emoji are supported.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}/reactions",
	//   "httpMethod": "POST",
	//   "id": "chat.spaces.messages.reactions.create",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "parent": {
	//       "description": "Required. The message where the reaction is created. Format: `spaces/{space}/messages/{message}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+parent}/reactions",
	//   "request": {
	//     "$ref": "Reaction"
	//   },
	//   "response": {
	//     "$ref": "Reaction"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.reactions",
	//     "https://www.googleapis.com/auth/chat.messages.reactions.create"
	//   ]
	// }

}

// method id "chat.spaces.messages.reactions.delete":

type SpacesMessagesReactionsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a reaction to a message. For an example, see Delete a
// reaction
// (https://developers.google.com/chat/api/guides/v1/reactions/delete).
// Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
//   - name: Name of the reaction to delete. Format:
//     `spaces/{space}/messages/{message}/reactions/{reaction}`.
func (r *SpacesMessagesReactionsService) Delete(name string) *SpacesMessagesReactionsDeleteCall {
	c := &SpacesMessagesReactionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesReactionsDeleteCall) Fields(s ...googleapi.Field) *SpacesMessagesReactionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesReactionsDeleteCall) Context(ctx context.Context) *SpacesMessagesReactionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesReactionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesReactionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.reactions.delete" call.
// Exactly one of *Empty or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesReactionsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Deletes a reaction to a message. For an example, see [Delete a reaction](https://developers.google.com/chat/api/guides/v1/reactions/delete). Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}/reactions/{reactionsId}",
	//   "httpMethod": "DELETE",
	//   "id": "chat.spaces.messages.reactions.delete",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Name of the reaction to delete. Format: `spaces/{space}/messages/{message}/reactions/{reaction}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+/reactions/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Empty"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.import",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.reactions"
	//   ]
	// }

}

// method id "chat.spaces.messages.reactions.list":

type SpacesMessagesReactionsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists reactions to a message. For an example, see List
// reactions
// (https://developers.google.com/chat/api/guides/v1/reactions/list).
// Requires user authentication
// (https://developers.google.com/chat/api/guides/auth/users).
//
//   - parent: The message users reacted to. Format:
//     `spaces/{space}/messages/{message}`.
func (r *SpacesMessagesReactionsService) List(parent string) *SpacesMessagesReactionsListCall {
	c := &SpacesMessagesReactionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A query filter. You can
// filter reactions by emoji
// (https://developers.google.com/chat/api/reference/rest/v1/Emoji)
// (either `emoji.unicode` or `emoji.custom_emoji.uid`) and user
// (https://developers.google.com/chat/api/reference/rest/v1/User)
// (`user.name`). To filter reactions for multiple emojis or users, join
// similar fields with the `OR` operator, such as `emoji.unicode =
// "🙂" OR emoji.unicode = "👍" and `user.name = "users/AAAAAA" OR
// user.name = "users/BBBBBB". To filter reactions by emoji and user,
// use the `AND` operator, such as `emoji.unicode = "🙂" AND user.name
// = "users/AAAAAA". If your query uses both `AND` and `OR`, group them
// with parentheses. For example, the following queries are valid: ```
// user.name = "users/{user}" emoji.unicode = "🙂"
// emoji.custom_emoji.uid = "{uid}" emoji.unicode = "🙂" OR
// emoji.unicode = "👍" emoji.unicode = "🙂" OR
// emoji.custom_emoji.uid = "{uid}" emoji.unicode = "🙂" AND user.name
// = "users/{user}" (emoji.unicode = "🙂" OR emoji.custom_emoji.uid =
// "{uid}") AND user.name = "users/{user}" ``` The following queries are
// invalid: ``` emoji.unicode = "🙂" AND emoji.unicode = "👍"
// emoji.unicode = "🙂" AND emoji.custom_emoji.uid = "{uid}"
// emoji.unicode = "🙂" OR user.name = "users/{user}" emoji.unicode =
// "🙂" OR emoji.custom_emoji.uid = "{uid}" OR user.name =
// "users/{user}" emoji.unicode = "🙂" OR emoji.custom_emoji.uid =
// "{uid}" AND user.name = "users/{user}" ``` Invalid queries are
// rejected by the server with an `INVALID_ARGUMENT` error.
func (c *SpacesMessagesReactionsListCall) Filter(filter string) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number
// of reactions returned. The service can return fewer reactions than
// this value. If unspecified, the default value is 25. The maximum
// value is 200; values above 200 are changed to 200.
func (c *SpacesMessagesReactionsListCall) PageSize(pageSize int64) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": (If resuming from
// a previous query.) A page token received from a previous list
// reactions call. Provide this to retrieve the subsequent page. When
// paginating, the filter value should match the call that provided the
// page token. Passing a different value might lead to unexpected
// results.
func (c *SpacesMessagesReactionsListCall) PageToken(pageToken string) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesReactionsListCall) Fields(s ...googleapi.Field) *SpacesMessagesReactionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMessagesReactionsListCall) IfNoneMatch(entityTag string) *SpacesMessagesReactionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesReactionsListCall) Context(ctx context.Context) *SpacesMessagesReactionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesReactionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesReactionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/reactions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.reactions.list" call.
// Exactly one of *ListReactionsResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListReactionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *SpacesMessagesReactionsListCall) Do(opts ...googleapi.CallOption) (*ListReactionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListReactionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Lists reactions to a message. For an example, see [List reactions](https://developers.google.com/chat/api/guides/v1/reactions/list). Requires [user authentication](https://developers.google.com/chat/api/guides/auth/users).",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}/reactions",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.messages.reactions.list",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "filter": {
	//       "description": "Optional. A query filter. You can filter reactions by [emoji](https://developers.google.com/chat/api/reference/rest/v1/Emoji) (either `emoji.unicode` or `emoji.custom_emoji.uid`) and [user](https://developers.google.com/chat/api/reference/rest/v1/User) (`user.name`). To filter reactions for multiple emojis or users, join similar fields with the `OR` operator, such as `emoji.unicode = \"🙂\" OR emoji.unicode = \"👍\"` and `user.name = \"users/AAAAAA\" OR user.name = \"users/BBBBBB\"`. To filter reactions by emoji and user, use the `AND` operator, such as `emoji.unicode = \"🙂\" AND user.name = \"users/AAAAAA\"`. If your query uses both `AND` and `OR`, group them with parentheses. For example, the following queries are valid: ``` user.name = \"users/{user}\" emoji.unicode = \"🙂\" emoji.custom_emoji.uid = \"{uid}\" emoji.unicode = \"🙂\" OR emoji.unicode = \"👍\" emoji.unicode = \"🙂\" OR emoji.custom_emoji.uid = \"{uid}\" emoji.unicode = \"🙂\" AND user.name = \"users/{user}\" (emoji.unicode = \"🙂\" OR emoji.custom_emoji.uid = \"{uid}\") AND user.name = \"users/{user}\" ``` The following queries are invalid: ``` emoji.unicode = \"🙂\" AND emoji.unicode = \"👍\" emoji.unicode = \"🙂\" AND emoji.custom_emoji.uid = \"{uid}\" emoji.unicode = \"🙂\" OR user.name = \"users/{user}\" emoji.unicode = \"🙂\" OR emoji.custom_emoji.uid = \"{uid}\" OR user.name = \"users/{user}\" emoji.unicode = \"🙂\" OR emoji.custom_emoji.uid = \"{uid}\" AND user.name = \"users/{user}\" ``` Invalid queries are rejected by the server with an `INVALID_ARGUMENT` error.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "pageSize": {
	//       "description": "Optional. The maximum number of reactions returned. The service can return fewer reactions than this value. If unspecified, the default value is 25. The maximum value is 200; values above 200 are changed to 200.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "Optional. (If resuming from a previous query.) A page token received from a previous list reactions call. Provide this to retrieve the subsequent page. When paginating, the filter value should match the call that provided the page token. Passing a different value might lead to unexpected results.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "parent": {
	//       "description": "Required. The message users reacted to. Format: `spaces/{space}/messages/{message}`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+parent}/reactions",
	//   "response": {
	//     "$ref": "ListReactionsResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.reactions",
	//     "https://www.googleapis.com/auth/chat.messages.reactions.readonly",
	//     "https://www.googleapis.com/auth/chat.messages.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesMessagesReactionsListCall) Pages(ctx context.Context, f func(*ListReactionsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
