/*
 * Copyright (C) 2004 ViASIC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */

/*--------------------------------------------------------------------------------------------------
   This module reads and writes gate-level Verilog designs.
--------------------------------------------------------------------------------------------------*/
#include "vr.h"

FILE *vrFile;
U32 vrFileSize, vrLineNum, vrCharCount;
dbDesign vrCurrentDesign;

/*--------------------------------------------------------------------------------------------------
  Determine if a value is in the range.
--------------------------------------------------------------------------------------------------*/
static bool inRange(
    U32 index,
    U32 left,
    U32 right)
{
    if(left <= right) {
        return left <= index && index <= right;
    }
    return right <= index && index <= left;
}

/*--------------------------------------------------------------------------------------------------
  Build a new netlist in the database.
--------------------------------------------------------------------------------------------------*/
vrNetlist vrNetlistCreate(
    dbDesign design,
    utSym name)
{
    dbNetlist super = dbDesignFindNetlist(design, name);
    vrNetlist netlist;

    if(super != dbNetlistNull) {
        utError("Module %s defined multiple times", utSymGetName(name));
    }
    super = dbNetlistCreate(design, name, DB_SUBCIRCUIT, utSymNull);
    netlist = vrNetlistAlloc(super);
    return netlist;
}

/*--------------------------------------------------------------------------------------------------
  Count the total nets in the parameter.
--------------------------------------------------------------------------------------------------*/
static U32 countParamNets(
    vrParam param)
{
    vrConn conn;
    U32 numNets = 0;

    vrForeachParamConn(param, conn) {
        if(vrConnConst(conn)) {
            numNets += vrConnGetLength(conn);
        } else if(vrConnRange(conn)) {
            numNets += utAbs((S32)vrConnGetLeft(conn) - (S32)vrConnGetRight(conn)) + 1;
        } else if(vrConnGetBus(conn) != dbBusNull) {
            numNets += dbBusGetnumNets(vrConnGetBus(conn));
        } else {
            numNets++;
        }
    } vrEndForeachParamConn;
    return numNets;
}

/*--------------------------------------------------------------------------------------------------
  Check that the connectivity of the mbus matches that of the parameter.
--------------------------------------------------------------------------------------------------*/
static void checkMbusConn(
    dbMbus mbus,
    vrParam param)
{
    vrIdec idec = vrParamGetIdec(param);

    if(countParamNets(param) != dbMbusGetnumMports(mbus)) {
        utError("Signal count mismatch instance %s of module %s, bus %s",
                utSymGetName(vrIdecGetSym(idec)), dbNetlistGetName(dbMbusGetNetlist(mbus)),
                dbMbusGetName(mbus));
    }
}

/*--------------------------------------------------------------------------------------------------
  Check that the connectivity of the mport matches that of the parameter.
--------------------------------------------------------------------------------------------------*/
static void checkMportConn(
    dbMport mport,
    vrParam param)
{
    vrIdec idec = vrParamGetIdec(param);
    dbMbus mbus;

    if(countParamNets(param) != 1) {
        mbus = dbMportGetMbus(mport);
        utError("Signal count mismatch instance %s of module %s, bus %s",
                utSymGetName(vrIdecGetSym(idec)), dbNetlistGetName(dbMbusGetNetlist(mbus)),
                dbMbusGetName(mbus));
    }
}

/*--------------------------------------------------------------------------------------------------
  Build a black-box netlist for the undefined instance type.
--------------------------------------------------------------------------------------------------*/
static dbNetlist buildBlackBoxNetlist(
    dbDesign design,
    vrIdec idec,
    utSym sym)
{
    vrNetlist subNetlist = vrNetlistCreate(design, sym); 
    dbNetlist netlist = vrNetlistSuper(subNetlist);
    vrParam param;
    dbMport mport;
    dbMbus mbus;
    dbBus bus;
    dbInst flag;
    dbNet net;
    dbPort port;
    utSym name;
    U32 numConn;

    vrForeachIdecParam(idec, param) {
        name = vrParamGetSym(param);
        numConn = countParamNets(param);
        if(numConn == 1) {
            mport = dbMportCreate(netlist, name, DB_PAS); 
            dbFlagInstCreate(mport);
            port = dbMportGetFlagPort(mport);
            net = dbNetCreate(netlist, name);
            dbNetInsertPort(net, port);
        } else {
            bus = dbBusCreate(netlist, name, numConn - 1, 0);
            mbus = dbMbusCreate(netlist, name, DB_PAS, numConn - 1, 0);
            flag = dbBusFlagInstCreate(mbus);
            dbBusHookup(bus, dbInstGetfirstPort(flag));
        }
    } vrEndForeachIdecParam;
    return netlist;
}

/*--------------------------------------------------------------------------------------------------
  Verify all modules are declared.  Create netlists for undeclared modules.
--------------------------------------------------------------------------------------------------*/
static void checkForImplicitModules(
    dbDesign design)
{
    dbNetlist netlist, internalNetlist;
    vrIdec idec;
    utSym sym;

    dbForeachDesignNetlist(vrCurrentDesign, netlist) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            vrForeachNetlistIdec(vrNetlistSub(netlist), idec) {
                sym = vrIdecGetInternalNetlistSym(idec);
                internalNetlist = dbDesignFindNetlist(design, sym);
                if(internalNetlist == dbNetlistNull) {
                    utWarning("Module %s not defined.", utSymGetName(sym));
                    buildBlackBoxNetlist(design, idec, sym);
                }
            } vrEndForeachNetlistIdec;
        }
    } dbEndForeachDesignNetlist;
}

/*--------------------------------------------------------------------------------------------------
  Build an instance for the instance declaration.  Attach to nets.
--------------------------------------------------------------------------------------------------*/
static dbInst buildIdecInst(
    vrIdec idec)
{
    dbNetlist netlist = vrNetlistSuper(vrIdecGetNetlist(idec));
    dbNetlist internalNetlist = dbDesignFindNetlist(vrCurrentDesign,
        vrIdecGetInternalNetlistSym(idec));
    utSym name = vrIdecGetSym(idec);
    dbInst inst = dbNetlistFindInst(netlist, vrIdecGetSym(idec));
    dbMport mport;

    if(internalNetlist == dbNetlistNull) {
        utError("Netlist %s of instance %s was not defined",
            utSymGetName(vrIdecGetInternalNetlistSym(idec)), utSymGetName(name));
    }
    if(inst != dbInstNull) {
        utError("Instance %s defined multiple times in netlist %s",
                utSymGetName(name), dbNetlistGetName(netlist));
    }
    inst = dbInstCreate(netlist, name, internalNetlist);
    dbForeachNetlistMport(internalNetlist, mport) {
        dbPortCreate(inst, mport);
    } dbEndForeachNetlistMport;
    return inst;
}

/*--------------------------------------------------------------------------------------------------
  Tie bits of the bus to zero and one as specified by the connection.
--------------------------------------------------------------------------------------------------*/
static U32 connectPortsToConstants(
    dbInst inst,
    dbMbus mbus,
    U32 xMport,
    vrConn conn)
{
    dbNetlist netlist = dbInstGetNetlist(inst);
    dbMport mport;
    dbPort port;
    dbNet net;
    U32 length = vrConnGetLength(conn);
    U32 mask = vrConnGetMask(conn);

    while(length--) {
        if(mask & (1 << length)) {
            net = dbNetlistGetOneNet(netlist);
        } else {
            net = dbNetlistGetZeroNet(netlist);
        }
        mport = dbMbusGetiMport(mbus, xMport);
        port = dbFindPortFromInstMport(inst, mport);
        dbNetAppendPort(net, port);
        xMport++;
    }
    return xMport;
}

/*--------------------------------------------------------------------------------------------------
  Tie bits of the bus to a range of the bus specified by the conn.
--------------------------------------------------------------------------------------------------*/
static U32 connectPortsToRange(
    dbInst inst,
    dbMbus mbus,
    U32 xMport,
    vrConn conn)
{
    dbMport mport;
    dbPort port;
    dbNet net;
    dbBus bus = vrConnGetBus(conn);
    S32 left, right;
    S32 xNet;

    if(vrConnRange(conn)) {
        left = vrConnGetLeft(conn);
        right = vrConnGetRight(conn);
        if(!inRange(left,  dbBusGetLeft(bus), dbBusGetRight(bus)) ||
            !inRange(right, dbBusGetLeft(bus), dbBusGetRight(bus))) {
            utError("Invalid bus index for bus %s on instance %s",
                    dbBusGetName(bus), dbInstGetName(inst));
        }
    } else {
        left = dbBusGetLeft(bus);
        right = dbBusGetRight(bus);
    }
    for(xNet = left; left <= right? xNet <= right: xNet >= right; left <= right? xNet++ : xNet--) {
        mport = dbMbusGetiMport(mbus, xMport);
        port = dbFindPortFromInstMport(inst, mport);
        net = dbBusIndexNet(bus, xNet);
        dbNetAppendPort(net, port);
        xMport++;
    }
    return xMport;
}

/*--------------------------------------------------------------------------------------------------
  Tie a bit of a bus to a bit of the bus specified by the conn.
--------------------------------------------------------------------------------------------------*/
static void connectPortToNet(
    dbInst inst,
    dbMbus mbus,
    U32 xMport,
    vrConn conn)
{
    dbMport mport = dbMbusGetiMport(mbus, xMport);
    dbPort port = dbFindPortFromInstMport(inst, mport);
    dbNet net = vrConnGetNet(conn);

    dbNetAppendPort(net, port);
}

/*--------------------------------------------------------------------------------------------------
  Add a connection from a bus to nets specified by the parameter.
--------------------------------------------------------------------------------------------------*/
static void addBusConnection(
    dbInst inst,
    vrParam param,
    dbMbus mbus)
{
    vrConn conn;
    U32 xMport = 0;

    vrForeachParamConn(param, conn) {
        if(xMport > dbMbusGetusedMports(mbus)) {
            utError("Mismatched signal width on port %s of inst %s",
                utSymGetName(vrParamGetSym(param)), dbInstGetName(inst));
        }
        if(vrConnConst(conn)) {
            xMport = connectPortsToConstants(inst, mbus, xMport, conn);
        } else if(vrConnGetBus(conn) != dbBusNull) {
            xMport = connectPortsToRange(inst, mbus, xMport, conn);
        } else {
            connectPortToNet(inst, mbus, xMport, conn);
            xMport++;
        }
    } vrEndForeachParamConn;
    if(xMport != dbMbusGetusedMports(mbus)) {
        utError("Mismatched signal width on port %s of inst %s",
            utSymGetName(vrParamGetSym(param)), dbInstGetName(inst));
    }
}

/*--------------------------------------------------------------------------------------------------
  Connect a net to the port as specified by the connection.
--------------------------------------------------------------------------------------------------*/
static void addNetConnection(
    dbInst inst,
    vrConn conn,
    dbMport mport)
{
    dbNetlist netlist = dbInstGetNetlist(inst);
    dbPort port = dbFindPortFromInstMport(inst, mport);
    dbNet net;
    dbBus bus;
    utSym sym;

    if(conn == vrConnNull) {
        sym = dbNetlistCreateUniqueNetName(netlist, "N");
        net = dbNetCreate(netlist, sym);
    } else if(vrConnConst(conn)) {
        if(vrConnGetMask(conn)) {
            net = dbNetlistGetOneNet(netlist);
        } else {
            net = dbNetlistGetZeroNet(netlist);
        }
    } else if(vrConnRange(conn)) {
        utAssert(vrConnGetLeft(conn) == vrConnGetRight(conn));
        bus = vrConnGetBus(conn);
        net = dbBusIndexNet(bus, vrConnGetLeft(conn));
    } else {
        if(vrConnGetBus(conn) != dbBusNull) {
            utError("Bus connection to %s on inst %s passed to scalar signal",
                dbBusGetName(vrConnGetBus(conn)), dbInstGetName(inst));
        }
        net = vrConnGetNet(conn);
        utAssert(net != dbNetNull);
    }
    dbNetAppendPort(net, port);
}

/*--------------------------------------------------------------------------------------------------
  Skip over mports until we reach an new mbus, or an mport with no owning mbus.
--------------------------------------------------------------------------------------------------*/
static dbMport skipParam(
    dbMport mport)
{
    dbMbus mbus = dbMportGetMbus(mport);

    do {
        mport = dbMportGetnextNetlistMport(mport);
    } while(mport != dbMportNull && mbus != dbMbusNull && dbMportGetMbus(mport) == mbus);
    return mport;
}

/*--------------------------------------------------------------------------------------------------
  Find the nth parameter on an instance.
--------------------------------------------------------------------------------------------------*/
static utSym findPositionalParamSym(
    dbInst inst,
    U32 paramNum)
{
    dbNetlist internalNetlist = dbInstGetInternalNetlist(inst);
    dbMbus mbus;
    dbMport mport;
    U32 xParam;

    utAssert(internalNetlist != dbNetlistNull);
    mport = dbNetlistGetfirstMport(internalNetlist);
    for(xParam = 1; xParam < paramNum && mport != dbMportNull; xParam++) {
        mport = skipParam(mport);
    }
    if(mport == dbMportNull) {
        utError("Too many parameters passed to inst %s", dbInstGetName(inst));
    }
    mbus = dbMportGetMbus(mport);
    if(mbus != dbMbusNull) {
        return dbMbusGetSym(mbus);
    }
    return dbMportGetSym(mport);
}

/*--------------------------------------------------------------------------------------------------
  Connect nets to ports of the instance as specified by the instance declaration.
--------------------------------------------------------------------------------------------------*/
static void addNetPorts(
    vrIdec idec,
    dbInst inst)
{
    dbNetlist internalNetlist = dbInstGetInternalNetlist(inst);
    vrParam param;
    dbMbus mbus;
    dbMport mport;
    utSym paramSym;
    U32 paramNum = 1;

    vrForeachIdecParam(idec, param) {
        paramSym = vrParamGetSym(param);
        if(paramSym == utSymNull) {
            paramSym = findPositionalParamSym(inst, paramNum);
        }
        mbus = dbNetlistFindMbus(internalNetlist, paramSym);
        if(mbus != dbMbusNull) {
            addBusConnection(inst, param, mbus);
        } else {
            mport = dbNetlistFindMport(internalNetlist, paramSym);
            if(mport == dbMportNull) {
                utError("Pin %s of instance %s does not exist on cell %s",
                        utSymGetName(paramSym), dbInstGetName(inst),
                        dbNetlistGetName(internalNetlist));
            }
            if(vrParamGetfirstConn(param) != vrConnNull) {
                utAssert(vrConnGetnextParamConn(vrParamGetfirstConn(param)) == vrConnNull);
            }
            addNetConnection(inst, vrParamGetfirstConn(param), mport);
        }
        paramNum++;
    } vrEndForeachIdecParam;
}

/*--------------------------------------------------------------------------------------------------
  Convert the instance declaration objects into instances.
--------------------------------------------------------------------------------------------------*/
static void buildInstances(void)
{
    dbNetlist netlist;
    vrIdec idec;
    dbInst inst;

    dbForeachDesignNetlist(vrCurrentDesign, netlist) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            vrForeachNetlistIdec(vrNetlistSub(netlist), idec) {
                inst = buildIdecInst(idec);
                addNetPorts(idec, inst);
            } vrEndForeachNetlistIdec;
        }
    } dbEndForeachDesignNetlist;
}

/*--------------------------------------------------------------------------------------------------
  Find the instance refered to by the path.
--------------------------------------------------------------------------------------------------*/
static dbInst findInstFromPath(
    dbNetlist netlist,
    vrPath path)
{
    vrPath nextPath = vrPathGetnextDefparamPath(path);
    dbInst inst = dbNetlistFindInst(netlist, vrPathGetSym(path));

    if(inst == dbInstNull) {
        /* Invalid path */
        utWarning("defparam statement failed to find instance");
        return dbInstNull;
    }
    if(nextPath != vrPathNull) {
        if(dbInstGetType(inst) != DB_SUBCIRCUIT) {
            utWarning("defparam specifies instance within leaf instance");
            return dbInstNull;
        }
        return findInstFromPath(dbInstGetInternalNetlist(inst), nextPath);
    }
    return inst;
}

/*--------------------------------------------------------------------------------------------------
  Create properties on instances from defparam statements.  This has to be
  done as a post process, since parameters can be assigned before an
  instance is declared.
--------------------------------------------------------------------------------------------------*/
static void evaluateDefparams(void)
{
    dbNetlist netlist;
    vrDefparam defparam;
    dbInst inst;

    dbForeachDesignNetlist(vrCurrentDesign, netlist) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            vrForeachNetlistDefparam(vrNetlistSub(netlist), defparam) {
                netlist = vrNetlistSuper(vrDefparamGetNetlist(defparam));
                inst = findInstFromPath(netlist, vrDefparamGetfirstPath(defparam));
                if(inst != dbInstNull) {
                    dbInstSetValue(inst, vrDefparamGetSym(defparam), vrDefparamGetValue(defparam));
                }
            } vrEndForeachNetlistDefparam;
        }
    } dbEndForeachDesignNetlist;
}

/*--------------------------------------------------------------------------------------------------
  Create vrNetlist extensions for all the dbNetlists.
--------------------------------------------------------------------------------------------------*/
static void extendNetlists(void)
{
    dbNetlist netlist;

    dbForeachDesignNetlist(vrCurrentDesign, netlist) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            vrNetlistAlloc(netlist);
        }
    } dbEndForeachDesignNetlist;
}

/*--------------------------------------------------------------------------------------------------
  Initialize the verilog reader module.
--------------------------------------------------------------------------------------------------*/
void vrInit(void) 
{
    vrDDRStart();
    extendNetlists();
}

/*--------------------------------------------------------------------------------------------------
  Free vrNetlist extensions from all the dbNetlists.
--------------------------------------------------------------------------------------------------*/
static void freeNetlists(void)
{
    dbNetlist netlist;

    dbForeachDesignNetlist(vrCurrentDesign, netlist) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            vrNetlistDestroy(vrNetlistSub(netlist));
        }
    } dbEndForeachDesignNetlist;
}

/*--------------------------------------------------------------------------------------------------
  Free memory used by the verilog reader.
--------------------------------------------------------------------------------------------------*/
void vrClose(void)
{
    freeNetlists();
    vrDDRStop();
}

/*--------------------------------------------------------------------------------------------------
  Warn if nets and insts have the same names.
--------------------------------------------------------------------------------------------------*/
static bool checkNamesInNetlist(
    dbNetlist netlist)
{
    dbNet net;
    dbInst inst;
    dbMport mport;
    utSym sym;

    dbForeachNetlistNet(netlist, net) {
        sym = dbNetGetSym(net);
        inst = dbNetlistFindInst(netlist, sym);
        if(inst != dbInstNull) {
            utWarning("Inst %s and net %s have the same name.",
                dbInstGetName(inst), dbNetGetName(net));
            return false;
        }
    } dbEndForeachNetlistNet;
    dbForeachNetlistMport(netlist, mport) {
        sym = dbMportGetSym(mport);
        inst = dbNetlistFindInst(netlist, sym);
        if(inst != dbInstNull) {
            utWarning("Port %s and inst %s have the same name.",
                dbMportGetName(mport), dbInstGetName(inst));
            return false;
        }
    } dbEndForeachNetlistMport;
    return true;
}

/*--------------------------------------------------------------------------------------------------
  Warn if nets and insts have the same names.
--------------------------------------------------------------------------------------------------*/
void vrCheckNamesInDesign(
    dbDesign design)
{
    dbNetlist netlist;

    dbForeachDesignNetlist(design, netlist) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            if(!checkNamesInNetlist(netlist)) {
                return;
            }
        }
    } dbEndForeachDesignNetlist;
}

/*--------------------------------------------------------------------------------------------------
  Read a Verilog gate-level design into the database.
--------------------------------------------------------------------------------------------------*/
dbDesign vrReadDesign(
    char *designName,
    char *fileName)
{
    utSym name = utSymCreate(designName);

    utLogMessage("Reading Verilog file %s", fileName);
    vrFile = fopen(fileName, "r");
    if(!vrFile) {
        utWarning("Could not open file %s for reading", fileName);
        return dbDesignNull;
    }
    vrFileSize = utFindFileSize(vrFile);
    vrCharCount = 0;
    vrLineNum = 1;
    vrCurrentDesign = dbRootFindDesign(dbTheRoot, name);
    if(vrCurrentDesign == dbDesignNull) {
	vrCurrentDesign = dbDesignCreate(name);
    }
    vrInit();
    if(vrparse()) {
        fclose(vrFile);
        vrClose();
        dbDesignDestroy(vrCurrentDesign);
        return dbDesignNull;
    }
    fclose(vrFile);
    checkForImplicitModules(vrCurrentDesign);
    buildInstances();
    evaluateDefparams();
    vrCheckNamesInDesign(vrCurrentDesign);
    vrClose();
    return vrCurrentDesign;
}

