/*
 * Copyright (C) 2003 ViASIC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */

#include "utnojunk.h"
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include "util.h"
#include "utmem.h"

struct _utRootType  _utRoot;
bool  _utInitialized = false;
bool utAbort = false;
bool utEnableLongjmpOnAbort;
U32 utSetjmpLine[UT_MAX_SETJMP_DEPTH];
char *utSetjmpFile[UT_MAX_SETJMP_DEPTH];
char *utConfigDirectory = NULL;
char *utExeDirectory = NULL;
char *utExeFullPath = NULL;
char *utLogFileName = NULL;
U32 utDebugVal = 0;
U32 utVerboseVal = 0;
bool utLogAccess = false;
U32 utAccessCount = 0;
U32 utAccessHash = 0;
U32 utEnableCoreFile = 0;
#define UT_MAX_BUFFERS 1000
static char *utBuffers[UT_MAX_BUFFERS];
static U16 utNextBuffer, utBufferSizes[UT_MAX_BUFFERS];
static time_t utTimers[UT_MAX_BUFFERS];
static U16 utTimerDepth = 0;
static utErrorProc utUserErrProc = NULL, utUserWarningProc = NULL, utUserStatusProc = NULL;
static bool (*utUserMessageYesNoProc)(char *message) = NULL;
static utExitProc utUserExitProc = NULL;
static utProgressProc utUserProgressProc = NULL;
static char *utVersion = "No version";
U8 utCount1sTable[256];
S16 utSetjmpDepth = 0;
char *utFileBuffer;
U32 utFileBufferPos, utFileBufferSize;
U32 utUserParameter1, utUserParameter2, utUserParameter3;

struct _utSym *utSyms;
U32 utsSym, utmSym;
char *utSymNames;
U32 utsSymName, utmSymName;
utSym *utSymTable;

U64 U64_MIN;
U64 U64_MAX;
S64 S64_MIN;
S64 S64_MAX;
S64 S64_0;
U64 U64_0;

/*--------------------------------------------------------------------------------------------------
  Like ansi sprintf.
--------------------------------------------------------------------------------------------------*/
char *utSprintf(
   char *format,
   ...)
{
   va_list ap;
   char *returnBuffer;

   va_start(ap, format);
   returnBuffer = utVsprintf(format, ap);
   va_end(ap);
   return returnBuffer;
}

/*--------------------------------------------------------------------------------------------------
  Log a message to the debug file.
--------------------------------------------------------------------------------------------------*/
void utLogDebug(
   char *format,
   ...)
{
   va_list ap;
   char *buff;
   FILE *logFile;

   if(utLogFileName == NULL) {
       return;
   }
   logFile = fopen(utLogFileName, "a");
   if(logFile == NULL) {
      return;
   }
   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   fputs("Debug: ", logFile);
   fputs(buff, logFile);
   fputs("\n", logFile);
   if(fclose(logFile) != 0) {
      utExit("utLogDebug: unable to close %s", utLogFileName);
   }
}

/*--------------------------------------------------------------------------------------------------
  Log a message to the debug file.
--------------------------------------------------------------------------------------------------*/
void utLogMessage(
   char *format,
   ...)
{
   va_list ap;
   char *buff;
   FILE *logFile;

   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   if(utUserStatusProc != NULL) {
       utStatus("%s", buff);
   } else {
      ErrorNotify(buff);
   }
   if(utLogFileName == NULL) {
       return;
   }
   logFile = fopen(utLogFileName, "a");
   if(logFile == NULL) {
      return;
   }
   fputs(buff, logFile);
   fputc('\n', logFile);
   if(fclose(logFile) != 0) {
      buff = utSprintf("utLogMessage: unable to close %s", utLogFileName);
      (utUserErrProc)(buff);
      return;
   }
}

/*--------------------------------------------------------------------------------------------------
  Print a message, and start recording the time since this message.
--------------------------------------------------------------------------------------------------*/
void utStartTimer(
   char *format,
   ...)
{
   va_list ap;
   char *buff;

   utTimers[utTimerDepth++] = time(NULL);
   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   utLogMessage("%s", buff);
}

/*--------------------------------------------------------------------------------------------------
  Print a message about the length of the current timer.
--------------------------------------------------------------------------------------------------*/
void utStopTimer(
   char *format,
   ...)
{
   va_list ap;
   char *buff;
   U32 deltaTime = (U32)difftime(time(NULL), utTimers[--utTimerDepth]);
   U32 hours, minutes, seconds;

   hours = deltaTime/3600;
   deltaTime -= hours*3600;
   minutes = deltaTime/60;
   deltaTime -= minutes*60;
   seconds = deltaTime;
   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   utLogMessage("%s %u:%02u:%02u\n", buff, hours, minutes, seconds);
}

/*--------------------------------------------------------------------------------------------------
   Purpose:  Same as utLogDebug but without the newline & prefix.
--------------------------------------------------------------------------------------------------*/
bool utDebug(
    char *format,
    ...)
{
   va_list ap;
   char *buff;
   FILE *logFile;

   if(utLogFileName == NULL) {
       return false;
   }
   logFile = fopen(utLogFileName, "a");
   if(logFile == NULL) {
      return false;
   }
   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   fputs(buff, logFile);
   if(fclose(logFile) != 0) {
      buff = utSprintf("utDebug: unable to close %s", utLogFileName);
      (utUserErrProc)(buff);
   }
   return true; /* So utDebug can be used in ',' expressions */
}

/*--------------------------------------------------------------------------------------------------
  Function to log errors without a message box to log file.
--------------------------------------------------------------------------------------------------*/
void utLogError(
   char *format,
   ...)
{
   va_list ap;
   char *buff1, *buff2;

   va_start(ap, format);
   buff1 = utVsprintf(format, ap);
   va_end(ap);
   buff2 = utSprintf("Error: %s\n", buff1);
   utLogMessage(buff2);
}

/*--------------------------------------------------------------------------------------------------
  Function to Send a message to the log file. Prefix with a time stamp line and a comment character.
--------------------------------------------------------------------------------------------------*/
void utLogTimeStamp(
   char *message)
{
   char *buffer;
   time_t timeInt = time(NULL);
   char *timeStr = ctime(&timeInt);

   buffer = utSprintf("%s: %s\n", message, timeStr);
   utLogMessage(buffer);
}

/*--------------------------------------------------------------------------------------------------
  Function to log the status of a tool to log file
--------------------------------------------------------------------------------------------------*/
void utStatus(
   char *format, ...)
{
   va_list ap;
   char *buff;

   va_start(ap, format);
   buff = utVsprintf(format, ap);
   va_end(ap);
   if(utUserStatusProc != NULL) {
      (utUserStatusProc)(buff);
   }
}

static char *utExitFileName;
static U32 utExitLineNum;
/*--------------------------------------------------------------------------------------------------
  Exit with a fatal error message.
--------------------------------------------------------------------------------------------------*/
void utExit_(
   char *format,
   ...)
{
   va_list ap;
   char *buff;

   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   if(utExitFileName == NULL) {
       utLogMessage("Exit: %s\n", buff);
   } else {    
       utLogMessage("Exit: %s:%u %s\n", utExitFileName, utExitLineNum, buff);
   }    
   if(utUserErrProc != NULL) {
      (utUserErrProc)(buff);
   }
   if(utUserExitProc != NULL) {
      (utUserExitProc)();
   } else {
      utAbnormalProgramTermination();
   }
}

/*--------------------------------------------------------------------------------------------------
  Set the file and line globals so that utExit_ can print them. Return utExit_ so it can be called
  with the user's parameters.
--------------------------------------------------------------------------------------------------*/
utExitProcType utSetFileAndLineAndReturnExitFunc(
    char *fileName,
    U32 lineNum)
{
    utExitFileName = fileName;
    utExitLineNum = lineNum;
    return &utExit_;
}

/*--------------------------------------------------------------------------------------------------
  Post a warning message.
--------------------------------------------------------------------------------------------------*/
void utWarning(
   char *format,
   ...)
{
   va_list ap;
   char *buff;

   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   utLogMessage("Warning: %s\n", buff);
   if(utUserWarningProc != NULL) {
      (utUserWarningProc)(buff);
   }
}

/*--------------------------------------------------------------------------------------------------
  Post a note.
--------------------------------------------------------------------------------------------------*/
void utNote(
   char *format,
   ...)
{
   va_list ap;
   char *buff;

   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   utLogMessage("%s\n", buff);
   if(utUserErrProc != NULL) {
      (utUserErrProc)(buff);
   }
}

/*--------------------------------------------------------------------------------------------------
  Post an error message, and longjump.
  This function just returns a value so that it can be used for inline macros as a return value.
  // could be improved with a macro that provides file & line number info, just like utExit.
--------------------------------------------------------------------------------------------------*/
U32 utError(
   char *format,
   ...)
{
   va_list ap;
   char *buff;

   va_start(ap, format);
   buff = utVsprintf((char *)format, ap);
   va_end(ap);
   utLogMessage("Error: %s\n", buff);
   if(utUserErrProc != NULL) {
      (utUserErrProc)(buff);
   }
   utLongjmp();
   return 0; /* Dummy return */
}

/*--------------------------------------------------------------------------------------------------
  Cry and die.
--------------------------------------------------------------------------------------------------*/
void utAssert_(
   char *fileName,
   U32 line,
   char *text)
{
   utMemCheck();
   utExitFileName = NULL;  /* we don't want "util.c" and the utAssert line number to print */
   utExit_("Assertion failed in file %s on line %u: %s", fileName, line, text);
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the Error callback.
--------------------------------------------------------------------------------------------------*/
void utSetErrorCallback(
   utErrorProc errorProc)
{
   utUserErrProc = errorProc;
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the Warning callback.
--------------------------------------------------------------------------------------------------*/
void utSetWarningCallback(
   utErrorProc warningProc)
{
   utUserWarningProc = warningProc;
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the status update callback.
--------------------------------------------------------------------------------------------------*/
void utSetStatusCallback(
   utErrorProc statusProc)
{
   utUserStatusProc = statusProc;
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the Yes/No messgae box callback.
--------------------------------------------------------------------------------------------------*/
void utSetMessageYesNoCallback(
   utMessageYesNoProc messageProc)
{
   utUserMessageYesNoProc = messageProc;
}

/*--------------------------------------------------------------------------------------------------
  Displays the YES-NO messagebox
--------------------------------------------------------------------------------------------------*/
bool utMessageYesNo(
   char *msg)
{
   if(utUserMessageYesNoProc != NULL) {
      return (*utUserMessageYesNoProc)(msg);
   } else {
      utExit("utMessageYesNo: Callback not set");
   }
   return false; /* Dummy return */
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the Exit callback.
--------------------------------------------------------------------------------------------------*/
void utSetExitCallback(
   utExitProc exitProc)
{
   utUserExitProc = exitProc;
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the logging file and reset it.
--------------------------------------------------------------------------------------------------*/
void utInitLogFile(
   char *fileName)
{
   FILE *file;

   utSetLogFile(fileName);
   file = fopen(fileName, "w");
   if(file != NULL) {
      fclose(file);
   }
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the logging file without resetting it.
--------------------------------------------------------------------------------------------------*/
void utSetLogFile(
   char *fileName)
{
    if(utLogFileName != NULL) {
        utFree(utLogFileName);
    }
    utLogFileName = utMallocType(strlen(fileName) + 1, char);
    utStrcpy(utLogFileName, fileName);
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the configuration file directory.
--------------------------------------------------------------------------------------------------*/
void utSetConfigDirectory(
    char *dirName)
{
    if(utConfigDirectory != NULL) {
        utFree(utConfigDirectory);
    }
    utConfigDirectory = utMallocType(strlen(dirName) + 1, char);
    strcpy(utConfigDirectory, dirName);
}

/*--------------------------------------------------------------------------------------------------
  Set the name of the executable file directory.
--------------------------------------------------------------------------------------------------*/
void utSetExeDirectory(
    char *dirName)
{
    if(utExeDirectory != NULL) {
        utFree(utExeDirectory);
    }
    utExeDirectory = utMallocType(strlen(dirName) + 1, char);
    strcpy(utExeDirectory, dirName);
}

/*--------------------------------------------------------------------------------------------------
  Return the exec directory.
--------------------------------------------------------------------------------------------------*/
char *utGetExeDirectory(void)
{
    return utExeDirectory;
}

/*--------------------------------------------------------------------------------------------------
  Return the config directory.
--------------------------------------------------------------------------------------------------*/
char *utGetConfigDirectory(void)
{
    return utConfigDirectory;
}

/*--------------------------------------------------------------------------------------------------
  Set the complete path and name of the executable file (argv[0])
--------------------------------------------------------------------------------------------------*/
void utSetExeFullPath(
    char *fullName)
{
    if(utExeFullPath != NULL) {
        utFree(utExeFullPath);
    }
    utExeFullPath = utMallocType(strlen(fullName) + 1, char);
    strcpy(utExeFullPath, fullName);
}

/*--------------------------------------------------------------------------------------------------
  Return the full path of the executable file.
--------------------------------------------------------------------------------------------------*/
char *utGetExeFullPath(void)
{
    return utExeFullPath;
}

/*--------------------------------------------------------------------------------------------------
  Just set the version variable.
--------------------------------------------------------------------------------------------------*/
void utSetVersion(
    char *version)
{
    utVersion = (char *)version;
}

/*--------------------------------------------------------------------------------------------------
  Return the version variable.
--------------------------------------------------------------------------------------------------*/
char *utGetVersion(void)
{
    return utVersion;
}

/*--------------------------------------------------------------------------------------------------
  Clear an array in memory.

// optimization: usually the compiler will be able to speed this up with a 32-bit built-in function
// or an optimized library routine.
--------------------------------------------------------------------------------------------------*/
void mtClearMem(
   void *mem,
   U32 sByte)
{
   char *p = (char *)mem;

   while (sByte--) {
      *(p++) = 0;
   }
}

/*--------------------------------------------------------------------------------------------------
  Function to set memory (byte by byte) to a given value.

// optimization: usually the compiler will be able to speed this up with a 32-bit built-in function
// or an optimized library routine.
--------------------------------------------------------------------------------------------------*/
void mtMemset(
   char *mem,
   char c,
   U32 sByte)
{
   while (sByte--) {
      *(mem++) = c;
   }
}

/*--------------------------------------------------------------------------------------------------
  Function to get the first non-zero bit of a word
  THIS FUNCTION BROKEN??  IT LOOKS LIKE IT COULD RETURN 64 for a U16!!!
--------------------------------------------------------------------------------------------------*/
#if 0
U8 utfWordBit(
   U16 word)
{
   U8 mBit = sizeof(word) << 3, bit;

   for (bit = 0, mBit = sizeof(word) << 3; bit < mBit; bit++, word >>= 1) {
       if(word & 0x0001) {
           return(bit);
       }
   }
   return (U8_MAX);
}
#endif

/*--------------------------------------------------------------------------------------------------
  Function to get the next non-zero bit of a word
  THIS FUNCTION BROKEN??  IT LOOKS LIKE IT COULD RETURN 64 for a U16!!!
--------------------------------------------------------------------------------------------------*/
#if 0
U8 utnWordBit(
   U16 word,
   U8 bit)
{
   U8 mBit = sizeof(word) << 3;

   for (word >>= ++bit; bit < mBit; bit++, word >>= 1) {
       if(word & 0x0001) {
           return(bit);
       }
   }
   return (U8_MAX);
}
#endif

/*--------------------------------------------------------------------------------------------------
  Find MS set bit of U32. Returns the bit mask of that bit. Example: 011100 -> 010000
--------------------------------------------------------------------------------------------------*/
U32 utFindMaskMSBit(
   U32 word)
{
    U32 retmask = 0x80000000;

    do {
        if(retmask & word) {
            return retmask;
        }
        retmask = retmask >> 1;
    } while (retmask);
    return 0;  /* not found */
}

/*--------------------------------------------------------------------------------------------------
  Find LS set bit of U32. Returns the bit mask of that bit. Example: 011100 -> 000100
--------------------------------------------------------------------------------------------------*/
U32 utFindMaskLSBit(
   U32 word)
{
    U32 retmask = 0x00000001;

    do {
        if(retmask & word) {
            return retmask;
        }
        retmask = retmask << 1;
    } while (retmask);
    return 0;  /* not found */
}

/*--------------------------------------------------------------------------------------------------
  Find MS set bit of U32. Returns the position of that bit, or U8_MAX if none. Example: 011100 -> 4
  optimization: x86 has BSF (bit scan forward) and BSR (bit scan reverse) instructions, but since
      these are very CISC, they may be not well implemented & slower on modern processors.
--------------------------------------------------------------------------------------------------*/
U8 utFindPosMSBit(
   U32 word)
{
    U32 retmask = 0x80000000;
    U8 pos = 31;

    do {
        if(retmask & word) {
            return pos;
        }
        retmask = retmask >> 1;
        pos--;
    } while (retmask);
    return U8_MAX;  /* not found */
}

/*--------------------------------------------------------------------------------------------------
  Find LS set bit of U32. Returns the position of that bit, or U8_MAX if none. Example: 011100 -> 2
  optimization: x86 has BSF (bit scan forward) and BSR (bit scan reverse) instructions, but since
      these are very CISC, they may be not well implemented & slower on modern processors.
--------------------------------------------------------------------------------------------------*/
U8 utFindPosLSBit(
   U32 word)
{
    U32 retmask = 0x00000001;
    U8 pos = 0;

    do {
        if(retmask & word) {
            return pos;
        }
        retmask = retmask << 1;
        pos++;
    } while (retmask);
    return U8_MAX;  /* not found */
}

/*--------------------------------------------------------------------------------------------------
  Count the number of 1s in the integer. Used to build lookup table.
--------------------------------------------------------------------------------------------------*/
static U8 count1sSlow(
    U32 value)
{
    U8 num1s = 0;

    while(value != 0) {
        if(value & 1) {
            num1s++;
        }
        value >>= 1;
    }
    return num1s;
}

/*--------------------------------------------------------------------------------------------------
  Count the number of 1s in the integer, using a lookup table. Endian doesn't matter.

  8 bits (256 entries) is thought to be a good balance between speed (4 looks per U32) and
  processor cache pollution (with this table)
--------------------------------------------------------------------------------------------------*/
static void initCount1sTable(void)
{
    U32 index;

    for(index = 0; index <= 0xff; index++) {
        utCount1sTable[index] = count1sSlow(index);
    }
}

/*--------------------------------------------------------------------------------------------------
  Count the number of 1s in the integer, using a lookup table.
  Shifting is used because it's probably faster than non-aligned U8 reads from the U32.
--------------------------------------------------------------------------------------------------*/
U8 utCount1s(
    U32 value)
{
    U8 num1s = utCount1sTable[value & 0xff];

    value >>= 8;
    num1s += utCount1sTable[value & 0xff];
    value >>= 8;
    num1s += utCount1sTable[value & 0xff];
    value >>= 8;
    num1s += utCount1sTable[value];
    return num1s;
}

/*--------------------------------------------------------------------------------------------------
  Find the first used heap.
--------------------------------------------------------------------------------------------------*/
utHeapRef utfHeap(void)
{
   utHeapRef Heap;

   for (Heap = 0; Heap < _utfVirgHeap(); Heap++) {
      if(_uttHeapUsed(Heap)) {
         return Heap;
      }
   }
   return ut0Heap;
}

/*--------------------------------------------------------------------------------------------------
  Find the next used heap.
--------------------------------------------------------------------------------------------------*/
utHeapRef utnHeap(
   utHeapRef Heap)
{
   for (Heap++; Heap < _utfVirgHeap(); Heap++) {
      if(_uttHeapUsed(Heap)) {
         return Heap;
      }
   }
   return ut0Heap;
}

/*--------------------------------------------------------------------------------------------------
  Create a heap.
--------------------------------------------------------------------------------------------------*/
utHeapRef utcHeap(void)
{
   utHeapRef Heap;

   if(uttHeapExists(_utfFreeHeap())) {
      Heap = _utfFreeHeap();
      _utfFreeHeap() = _utnHeap(Heap);
   } else {
      if(_utfVirgHeap() == _utmHeap()) {
         _utmHeap() += 1 + _utmHeap()/2;
         _utRoot.Heaps = (struct _utHeap *)mtRealloc((void *)
               _utRoot.Heaps, _utmHeap(), sizeof(struct _utHeap));
         if(!_utRoot.Heaps) {
            utError("UT0108: utcHeap: Out of memory");
         }
      }
      Heap = _utfVirgHeap()++;
   }
   _uttHeapUsed(Heap) = true;
   utfHeapBlock(Heap) = ut0Block;
   return Heap;
}

/*--------------------------------------------------------------------------------------------------
  Delete a heap.
--------------------------------------------------------------------------------------------------*/
void utdHeap(
   utHeapRef Heap)
{
   _uttHeapUsed(Heap) = false;
   _utnHeap(Heap) = _utfFreeHeap();
   _utfFreeHeap() = Heap;
}

/*--------------------------------------------------------------------------------------------------
  Delete a heap, and all blocks it uses.
--------------------------------------------------------------------------------------------------*/
void utFreeHeap(
   utHeapRef Heap)
{
   utBlockRef Block, nBlock;

   for (Block = utfHeapBlock(Heap); uttBlockExists(Block);
         Block = nBlock) {
      nBlock = utnHeapBlock(Heap, Block);
      utdBlock(Block);
   }
   utdHeap(Heap);
}

/*--------------------------------------------------------------------------------------------------
  Add a block to a heap.
--------------------------------------------------------------------------------------------------*/
void utaHeapBlock(
   utHeapRef Heap,
   utBlockRef Block)
{
   utnHeapBlock(Heap, Block) = utfHeapBlock(Heap);
   utfHeapBlock(Heap) = Block;
   utoHeapBlock(Block) = Heap;
   utsHeapBlock(Heap)++;
}

/*--------------------------------------------------------------------------------------------------
  Delete a block from a heap.
--------------------------------------------------------------------------------------------------*/
void utdHeapBlock(
   utHeapRef Heap,
   utBlockRef Block)
{
   utBlockRef pBlock = ut0Block, nBlock;

   for (nBlock = utfHeapBlock(Heap); nBlock != Block;
         nBlock = utnHeapBlock(Heap, nBlock)) {
      pBlock = nBlock;
   }
   if(uttBlockExists(pBlock)) {
      utnHeapBlock(Heap, pBlock) = utnHeapBlock(Heap, Block);
   } else {
      utfHeapBlock(Heap) = utnHeapBlock(Heap, Block);
   }
   utoHeapBlock(Block) = ut0Heap;
   utsHeapBlock(Heap)--;
}

/*--------------------------------------------------------------------------------------------------
  Find the first used block.
--------------------------------------------------------------------------------------------------*/
utBlockRef utfBlock(void)
{
   utBlockRef Block;

   for (Block = 0; Block < _utfVirgBlock(); Block++) {
      if(_uttBlockUsed(Block)) {
         return Block;
      }
   }
   return ut0Block;
}

/*--------------------------------------------------------------------------------------------------
  Find the next used block.
--------------------------------------------------------------------------------------------------*/
utBlockRef utnBlock(
   utBlockRef Block)
{
   for (Block++; Block < _utfVirgBlock(); Block++) {
      if(_uttBlockUsed(Block)) {
         return Block;
      }
   }
   return ut0Block;
}

/*--------------------------------------------------------------------------------------------------
  Create a new block.
--------------------------------------------------------------------------------------------------*/
utBlockRef utcBlock(void)
{
   utBlockRef Block;

   if(uttBlockExists(_utfFreeBlock())) {
      Block = _utfFreeBlock();
      _utfFreeBlock() = _utnBlock(Block);
   } else {
      if(_utfVirgBlock() == _utmBlock()) {
         _utmBlock() += 1 + _utmBlock()/2;
         _utRoot.Blocks = (struct _utBlock *)utRealloc((void *)
               _utRoot.Blocks, _utmBlock(), sizeof(struct _utBlock));
         if(!_utRoot.Blocks) {
            utError("UT0101: utcBlock: Out of memory");
         }
      }
      Block = _utfVirgBlock()++;
   }
   utgBlockMem(Block) = utCalloc(UTHEAPSIZE, sizeof(char));
   if(!utgBlockMem(Block)) {
      utError("UT0102: utcBlockMem: Out of memory");
   }
   utoHeapBlock(Block) = ut0Heap;
   return Block;
}

/*--------------------------------------------------------------------------------------------------
  Delete a block.
--------------------------------------------------------------------------------------------------*/
void utdBlock(
   utBlockRef Block)
{
   _uttBlockUsed(Block) = false;
   _utnBlock(Block) = _utfFreeBlock();
   _utfFreeBlock() = Block;
   if(!utFree(utgBlockMem(Block))) {
      utError("UT0103: utdBlock: Can't free block memory");
   }
}

/*--------------------------------------------------------------------------------------------------
  Free symbol table memory.
--------------------------------------------------------------------------------------------------*/
void closeSyms(void)
{
    mtFree(utSyms);
    utSyms = NULL;
    utmSym = 0;
    utsSym = 0;
    mtFree(utSymNames);
    utSymNames = NULL;
    utmSymName = 0;
    utsSymName = 0;
    mtFree(utSymTable);
    utSymTable = NULL;
}

/*--------------------------------------------------------------------------------------------------
  Free memory used by the utility module.
--------------------------------------------------------------------------------------------------*/
void utStop(void)
{
   if(utInitialized()) {
      if(utSetjmpDepth > 0 && utSetjmpDepth < UT_MAX_SETJMP_DEPTH) {
         utWarning("utClose: utSetjmpDepth != 0 (file %s, line %u)",
               utSetjmpFile[utSetjmpDepth - 1],
               utSetjmpLine[utSetjmpDepth - 1]);
      } else if(utSetjmpDepth != 0) {
         utWarning("utClose: utSetjmpDepth has an invalid value");
         /* prevents crashes */
      }
      if(utTimerDepth != 1) {
          utWarning("utClose: timer started, but never stopped");
      } else {
          utStopTimer("Process completed in");
      }
      utHtblClose();
      mtFree(utFileBuffer);
      utInitialized() = false;
      if(!utFree(_utRoot.Heaps)) {
         utError("UT0104: utClose: Can't free heaps");
      }
      if(!utFree(_utRoot.Blocks)) {
         utError("UT0105: utClose: Can't free blocks");
      }
      if(utLogFileName != NULL) {
          utFree(utLogFileName);
          utLogFileName = NULL;
      }
      if(utConfigDirectory != NULL) {
          utFree(utConfigDirectory);
          utConfigDirectory = NULL;
      }
      if(utExeDirectory != NULL) {
          utFree(utExeDirectory);
          utExeDirectory = NULL;
      }
      if(utExeFullPath != NULL) {
          utFree(utExeFullPath);
          utExeFullPath = NULL;
      }
      utMemClose();
      closeSyms();
   }
}

/*--------------------------------------------------------------------------------------------------
  Allocate initial space for heaps.
--------------------------------------------------------------------------------------------------*/
static void allocHeaps(void)
{
   _utmHeap() = 42;
   _utfVirgHeap() = 0;
   _utfFreeHeap() = ut0Heap;
   _utRoot.Heaps = (struct _utHeap *)utMalloc(_utmHeap(),
         sizeof(struct _utHeap));
   if(!_utRoot.Heaps) {
      utError("UT0106: allocHeaps: Out of memory");
   }
}

/*--------------------------------------------------------------------------------------------------
  Allocate initial space for blocks.
--------------------------------------------------------------------------------------------------*/
static void allocBlocks(void)
{
   _utmBlock() = 42;
   _utfVirgBlock() = 0;
   _utfFreeBlock() = ut0Block;
   _utRoot.Blocks = (struct _utBlock *)utMalloc(_utmBlock(),
         sizeof(struct _utBlock));
   if(!_utRoot.Blocks) {
      utError("UT0107: allocBlocks: Out of memory");
   }
}

/*--------------------------------------------------------------------------------------------------
  Initialize symbol table entries to utSymNull.
--------------------------------------------------------------------------------------------------*/
static void initSymTable(
    utSym *symTable,
    U32 size)
{
    U32 xSym;

    for(xSym = 0; xSym < size; xSym++) {
        symTable[xSym] = utSymNull;
    }
}

/*--------------------------------------------------------------------------------------------------
  Initialize symbol table memory.
--------------------------------------------------------------------------------------------------*/
void initSyms(void)
{
    utmSym = 1024;
    utsSym = 0;
    utSyms = (struct _utSym *)mtCalloc(utmSym, sizeof(struct _utSym));
    utmSymName = 4096;
    utsSymName = 0;
    utSymNames = (char *)mtCalloc(utmSymName, sizeof(char));
    utSymTable = (utSym *)mtCalloc(utmSym, sizeof(utSym));
    initSymTable(utSymTable, utmSym);
}

/*--------------------------------------------------------------------------------------------------
  Initialize generic buffer memory.
--------------------------------------------------------------------------------------------------*/
static void initBuffers(void)
{
    U16 xBuffer;

    utNextBuffer = 0;
    for(xBuffer = 0; xBuffer < UT_MAX_BUFFERS; xBuffer++) {
        utBufferSizes[xBuffer] = 42;
        utBuffers[xBuffer] = (char *)mtCalloc(utBufferSizes[xBuffer],
                                              sizeof(char));
    }
}

/*--------------------------------------------------------------------------------------------------
  Initialize local memory.
--------------------------------------------------------------------------------------------------*/
void utStart(void)
{
   if(!utInitialized()) {
      initSyms();
      utMemInit();
      utUserErrProc = 0;
      utUserExitProc = 0;
      utInitialized() = true;
      allocHeaps();
      allocBlocks();
      utSetjmpDepth = 0;
      utTimerDepth = 0;
      utAbort = false;
      utEnableLongjmpOnAbort = false;
      initBuffers();
      utHtblInit();
      utFileBuffer = (char *)mtCalloc(UTSTRLEN, sizeof(char));
      utUserParameter1 = 0;
      utUserParameter2 = 0;
      utUserParameter3 = 0;
      utStartTimer("");
      utInitSeed(4357);
      initCount1sTable();
      U64_MAX.upper = U32_MAX;
      U64_MAX.lower = U32_MAX;
      U64_MIN.upper = 0;
      U64_MIN.lower = 0;
      S64_MAX.upper = S32_MAX;
      S64_MAX.lower = U32_MAX;
      S64_MIN.upper = S32_MIN;
      S64_MIN.lower = 0;
      S64_0.upper = 0;
      S64_0.lower = 0;
      U64_0.lower = 0;
      U64_0.upper = 0;

/*
      if(!utTestFor64Bit()) {
        utWarning("64 bit math fails!");
      }
*/
  }
}


/*--------------------------------------------------------------------------------------------------
  String compare two strings using huge pointers.
--------------------------------------------------------------------------------------------------*/
S32 utStrcmp(
   char *string1,
   char *string2)
{
   while (*string1 && *string2 && *string1 == *string2) {
      string1++;
      string2++;
   }
   return *string1 - *string2;
}

/*--------------------------------------------------------------------------------------------------
  String compare two strings using huge pointers, case insensitive.
--------------------------------------------------------------------------------------------------*/
S32 utStricmp(
   char *string1,
   char *string2)
{
   while (*string1 && *string2 && tolower(*string1) == tolower(*string2)) {
      string1++;
      string2++;
   }
   return tolower(*string1) - tolower(*string2);
}

/*--------------------------------------------------------------------------------------------------
  Copy a string from string2 to string1.
--------------------------------------------------------------------------------------------------*/
char * utStrcpy(
   char *string1,
   char *string2)
{
   char *ret = string1;

   while (*string2) {
   *string1++ = *string2++;
   }
   *string1 = '\0';
   return ret;
}

/*--------------------------------------------------------------------------------------------------
  Copy a string from string2 to string1.
--------------------------------------------------------------------------------------------------*/
char * utStrncpy(
   char *string1,
   char *string2,
   U32 length)
{
   char *ret = string1;

   while (*string2 && length) {
      *string1++ = *string2++;
      length--;
   }
   while (length) {
      *string1++ = '\0';
      length--;
   }
   return ret;
}

/*--------------------------------------------------------------------------------------------------
  Copy a string from string2 to the end of string1.
--------------------------------------------------------------------------------------------------*/
char * utStrcat(
   char *string1,
   char *string2)
{
   char *ret = string1;

   while (*string1) {
      string1++;
   }
   while (*string2) {
      *string1++ = *string2++;
   }
   *string1 = '\0';
   return ret;
}

/*--------------------------------------------------------------------------------------------------
  Copy a string from string2 to the end of string1.
--------------------------------------------------------------------------------------------------*/
char * utStrncat(
   char *string1,
   char *string2,
   U32 length)
{
   char *ret = string1;

   while (*string1) {
      string1++;
   }
   while (*string2 && length) {
      *string1++ = *string2++;
      length--;
   }
   if(length) {
      *string1 = '\0';
   } else {
      *--string1 = '\0';
   }
   return ret;
}

/*--------------------------------------------------------------------------------------------------
  Return the length of string, without the '\0'.
--------------------------------------------------------------------------------------------------*/
U32 utStrlen(
   char *string)
{
   U32 len = 0;

   while (*string++) {
      len++;
   }
   return len;
}

/*--------------------------------------------------------------------------------------------------
  Convert a string to upper case.
--------------------------------------------------------------------------------------------------*/
void utStrupr(
   char *string)
{
   while (*string) {
      *string = toupper(*string);
      string++;
   }
}

/*--------------------------------------------------------------------------------------------------
  Convert a string to lower case.
--------------------------------------------------------------------------------------------------*/
void utStrlwr(
   char *string)
{
   while (*string) {
      *string = tolower(*string);
      string++;
   }
}

/*--------------------------------------------------------------------------------------------------
  Just like mtRealloc, but pass file name and line number.
--------------------------------------------------------------------------------------------------*/
void *utReallocTrace(
   void *memPtr,
   U32 num,
   U16 size,
   char *fileName,
   U32 line)
{
   utMemRef mem;
   U32 newSize = (num + 1)*size;

   if(memPtr == NULL) {
       return utMallocTrace(num, size, fileName, line);
   }
   mem = utqMemPtr(memPtr);
   utMemCheckTrace(fileName, line);
   if(!uttMemExists(mem)) {
      utExit("utRealloc: Bad memory pointer in %s, line %u", fileName, line);
   }
   utdMem(mem, true, fileName, line);
   memPtr = mtRealloc(memPtr, num + 1, size);
   if(memPtr != NULL) {
      utBuildMem(memPtr, newSize, true, fileName, line);
   } else {
      utLogMessage("utRealloc: unable to allocate memory %lu.  Total used: %lu\n",
         num*(size+1), utUsedMem);
   }
   return memPtr;

}

/*--------------------------------------------------------------------------------------------------
  Just like mtMalloc, but pass file name and line number.
--------------------------------------------------------------------------------------------------*/
void *utMallocTrace(
   U32 sStruct,
   size_t size,
   char *fileName,
   U32 line)
{
   U32 sByte = sStruct*size;
   void *memPtr = mtMalloc(sByte + 1, 1);

   utMemCheckTrace(fileName, line);
   if(memPtr != NULL) {
      utBuildMem(memPtr, sByte + 1, true, fileName, line);
   } else {
      utLogMessage("utRealloc: unable to allocate memory %lu.  Total used: %lu\n",
         sByte, utUsedMem);
   }
   return memPtr;
}

/*--------------------------------------------------------------------------------------------------
  Just like mtCalloc, but pass file name and line number.
--------------------------------------------------------------------------------------------------*/
void *utCallocTrace(
   U32 sStruct,
   size_t size,
   char *fileName,
   U32 line)
{
   U32 sByte = sStruct*size;
   void *memPtr;

   memPtr = utMallocTrace(sStruct, size, fileName, line);
   if(memPtr) {
      mtClearMem((void *)memPtr, sByte);
   }
   return memPtr;
}

/*--------------------------------------------------------------------------------------------------
  Just like mtFree, but pass file name and line number.
--------------------------------------------------------------------------------------------------*/
U16 utFreeTrace(
   void *memPtr,
   char *fileName,
   U32 line)
{
   utMemRef mem = utqMemPtr(memPtr);

   utMemCheckTrace(fileName, line);
   if(!uttMemExists(mem)) {
      utExit("utFree: Bad memory pointer in %s, line %u", fileName, line);
   }
   utdMem(mem, true, fileName, line);
   return mtFree(memPtr);
}



/*--------------------------------------------------------------------------------------------------
  Symbol table support.
--------------------------------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------------------------------
  Just find a hash value for the symbol name.
--------------------------------------------------------------------------------------------------*/
U32 hashSymName(
    char *name)
{
    U32 hashValue = 0xa43ffd97l; /* Nothing special -- just random */

    do {
        hashValue = (hashValue << 5) ^ (*name + hashValue) ^ (hashValue >> 27);
        hashValue = (hashValue ^ *name) * 1103515245 + 12345;
    } while(*name++);
    return hashValue;
}

/*--------------------------------------------------------------------------------------------------
  Find a symbol by the name in the symbol table.
--------------------------------------------------------------------------------------------------*/
static utSym utqSymName(
    char *name)
{
    utSym sym;
    U32 hashValue = hashSymName(name) & (utmSym - 1);

    for(sym = utSymTable[hashValue]; sym != utSymNull;
        sym = utSymGetfirstHashedSym(sym)) {
        if(!utStrcmp(name, utSymGetName(sym))) {
            return sym;
        }
    }
    return utSymNull;
}

/*--------------------------------------------------------------------------------------------------
  Resize the symbol table.  The size must be a power of 2.
--------------------------------------------------------------------------------------------------*/
static void resizeSymTable(
    U32 oldSize)
{
    utSym nextHashedSym, sym, nextSym;
    U32 newSize = oldSize*2;
    U32 xSym, hashValue;
    utSym *newTable = (utSym *)mtCalloc(newSize, sizeof(utSym));

    initSymTable(newTable, newSize);
    for(xSym = 0; xSym < oldSize; xSym++) {
        for(sym = utSymTable[xSym]; sym != utSymNull; sym = nextSym) {
            nextSym = utSymGetfirstHashedSym(sym);
            hashValue = utSymGetHashValue(sym) & (newSize - 1);
            nextHashedSym = newTable[hashValue];
            utSymGetfirstHashedSym(sym) = nextHashedSym;
            newTable[hashValue] = sym;
        }
    }
    mtFree(utSymTable);
    utSymTable = newTable;
}

/*--------------------------------------------------------------------------------------------------
  Add a symbol to the symbol table.
--------------------------------------------------------------------------------------------------*/
static void addSymTableSym(
    utSym sym)
{
    U32 hashValue = utSymGetHashValue(sym) & (utmSym - 1);
    utSym nextHashedSym = utSymTable[hashValue];

    utSymGetfirstHashedSym(sym) = nextHashedSym;
    utSymTable[hashValue] = sym;
}

/*--------------------------------------------------------------------------------------------------
  Build a new symbol, or return an old one with the same name.
--------------------------------------------------------------------------------------------------*/
utSym utSymCreate(
    char *name)
{
    utSym sym = utqSymName(name);
    utSym lowerSym;
    char lowerName[UTSTRLEN];
    U32 length;

    if(sym != utSymNull) {
        return sym;
    }
    if(utsSym == utmSym) {
        resizeSymTable(utmSym);
        utmSym *= 2;
        utSyms = (struct _utSym *)mtRealloc(utSyms, utmSym, sizeof(struct _utSym));
    }
    sym = utsSym++;
    length = utStrlen(name) + 1;
    if(utsSymName + length > utmSymName) {
        utmSymName += utmSymName + 2*length;
        utSymNames = (char *)mtRealloc(utSymNames, utmSymName, sizeof(char));
    }
    utSymGetXName(sym) = utsSymName;
    utStrcpy(utSymNames + utsSymName, name);
    utsSymName += length;
    utSymGetHashValue(sym) = hashSymName(name);
    addSymTableSym(sym);
    utStrcpy(lowerName, name);
    utStrlwr(lowerName);
    lowerSym = utSymCreate(lowerName);
    utSymGetLowerSym(sym) = lowerSym;
    return sym;
}

/*--------------------------------------------------------------------------------------------------
  Build a new symbol, and add it to the symbol table.
--------------------------------------------------------------------------------------------------*/
utSym utSymCreateFormatted(char *format, ...)
{
    char *buff;
    va_list ap;

    va_start(ap, format);
    buff = utVsprintf(format, ap);
    va_end(ap);
    return utSymCreate(buff);
}

/*--------------------------------------------------------------------------------------------------
  Build a new symbol, and add it to the symbol table.
--------------------------------------------------------------------------------------------------*/
utSym utBuildUniqueSym(
    char *name,
    char *suffix)
{
    utSym baseSym, sym;
    char buf[UTSTRLEN];
    char *tail;
    U32 count;

    strcpy(buf, name);
    tail = buf + strlen(buf) - 1;
    while(isdigit(*tail)) {
        tail--;
    }
    tail++;
    strcpy(tail, suffix);
    baseSym = utqSymName(buf);
    if(baseSym == utSymNull) {
        return utSymCreate(buf);
    }
    tail += strlen(suffix);
    count = utSymGetNextDerivedIndex(baseSym);
    do {
        sprintf(tail, "%u", count);
        count++;
        sym = utqSymName(buf);
    } while (sym != utSymNull);
    utSymGetNextDerivedIndex(baseSym) = count;
    return utSymCreate(buf);
}

/*--------------------------------------------------------------------------------------------------
  Allocate a buffer for generic use.  There is only a queue of UT_MAX_BUFFERS which can be in use
  at any time.  This is primarily useful for returning strings from subroutines.
--------------------------------------------------------------------------------------------------*/
void *utMakeBuffer_(
    U32 length)
{
    void *buffer;

    if(length > utBufferSizes[utNextBuffer]) {
        utBufferSizes[utNextBuffer] = length + length/2;
        utBuffers[utNextBuffer] = (char *)mtRealloc(utBuffers[utNextBuffer],
                utBufferSizes[utNextBuffer], sizeof(char));
    }
    buffer = utBuffers[utNextBuffer];
    if(++utNextBuffer == UT_MAX_BUFFERS) {
        utNextBuffer = 0;
    }
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Print a formated string to the file.
--------------------------------------------------------------------------------------------------*/
char *utVsprintf(
   char *format,
   va_list ap)
{
    char buffer[UTSTRLEN];
    char *returnBuffer;

    vsprintf((char *)buffer, (char *)format, ap);
    returnBuffer = utMakeString(strlen(buffer) + 1);
    strcpy(returnBuffer, buffer);
    return returnBuffer;
}

/*--------------------------------------------------------------------------------------------------
  Just replace the file suffix.
--------------------------------------------------------------------------------------------------*/
char *utReplaceSuffix(
    char *originalName,
    char *newSuffix)
{
    U32 length = strlen(originalName);
    char *buffer = utMakeBuffer_(length + strlen(newSuffix) + 1);
    char *endPtr;

    strcpy(buffer, originalName);
    endPtr = buffer + length;
    while(endPtr > buffer && *endPtr != '.') {
        endPtr--;
    }
    if(*endPtr != '.') {
        endPtr = buffer + length;
    }
    strcpy(endPtr, newSuffix);
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Return a temporary copy of a string.
--------------------------------------------------------------------------------------------------*/
char *utCopyString(
    char *string)
{
    char *buffer = utMakeString(strlen(string) + 1);

    strcpy(buffer, string);
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Make a new string by concatenating the two old ones.
--------------------------------------------------------------------------------------------------*/
char *utCatStrings(
    char *string1,
    char *string2)
{
    return utSprintf("%s%s", string1, string2);
}

/*--------------------------------------------------------------------------------------------------
  Convert a string to upper case.
--------------------------------------------------------------------------------------------------*/
char *utStringToUpperCase(
    char *string)
{
    char *buffer = utCopyString(string);
    char *p = buffer;
    
    while(*p != '\0') {
	*p = toupper(*p);
	p++;
    }
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Convert a string to lower case.
--------------------------------------------------------------------------------------------------*/
char *utStringToLowerCase(
    char *string)
{
    char *buffer = utCopyString(string);
    char *p = buffer;
    
    while(*p != '\0') {
	*p = tolower(*p);
	p++;
    }
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Return the compile time for the executable. This is in one place only for consistancy.
--------------------------------------------------------------------------------------------------*/
char *utGetCompileTime(void)
{
    return __DATE__ " " __TIME__;
}

/*--------------------------------------------------------------------------------------------------
  Return the date and time as an ASCII string.
--------------------------------------------------------------------------------------------------*/
char *utGetDateAndTime(void)
{
    time_t timeval = time(NULL);
    struct tm *theTime = localtime(&timeval);

    return utSprintf("%02u/%02u/%02u %02u:%02u:%02u", theTime->tm_mon + 1, theTime->tm_mday,
        theTime->tm_year % 100, theTime->tm_hour, theTime->tm_min, theTime->tm_sec);
}

/*--------------------------------------------------------------------------------------------------
  Return the base name of the path name.
--------------------------------------------------------------------------------------------------*/
char *utBaseName(
    char *name)
{
    char *left = strrchr(name, UTDIRSEP);
    char *right = name + strlen(name);
    char *buffer;

    if(left == NULL) {
        return utCopyString(name);
    }
    left++;
    buffer = utMakeString(right - left + 1);
    strncpy(buffer, left, right - left);
    buffer[right - left] = '\0';
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Return the suffix of the filename.
--------------------------------------------------------------------------------------------------*/
char *utSuffix(
    char *name)
{
    U32 i = 0;
    U32 start = U32_MAX;
    char c = name[i];

    while(c != 0) {
        i++;
        if(c == '.') {
            start = i;
        }
        c = name[i];
    }
    if(start == U32_MAX) {
        return NULL;
    }
    return utCopyString(name + start);
}

/*--------------------------------------------------------------------------------------------------
  Return the directory name of the path name.
--------------------------------------------------------------------------------------------------*/
char *utDirName(
    char *name)
{
    char *end = strrchr(name, UTDIRSEP);
    char *buffer;

    if(end == NULL) {
        return ".";
    }
    buffer = utMakeString(end - name + 1);
    strncpy(buffer, name, end - name);
    buffer[end - name] = '\0';
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Return the file name of the path name.
--------------------------------------------------------------------------------------------------*/
char *utFileName(
    char *name)
{
    char *left = strrchr(name, UTDIRSEP);
    char *right = strrchr(name, 0);
    char *buffer;

    if(left == NULL) {
        return utCopyString(name);
    }
    if(right == NULL) {
        right = name + strlen(name);
    }
    left++;
    buffer = utMakeString(right - left + 1);
    strncpy(buffer, left, right - left);
    buffer[right - left] = '\0';
    return buffer;
}

/*--------------------------------------------------------------------------------------------------
  Determine if the string starts with the prefix.
--------------------------------------------------------------------------------------------------*/
static bool stringStartsWithPrefix(
    char *string,
    char *prefix)
{
    while(*prefix != '\0' && *string == *prefix) {
        string++;
        prefix++;
    }
    return *prefix == '\0';
}

/*--------------------------------------------------------------------------------------------------
  Find the starting index of the substring within the string.
--------------------------------------------------------------------------------------------------*/
S32 utFindSubstring(
    char *string,
    char *substring)
{
    S32 index = 0;

    while(*string != '\0' && !stringStartsWithPrefix(string, substring)) {
        index++;
        string++;
    }
    if(*string == '\0') {
        return -1;
    }
    return index;
}

char *utDoubleSlash(
    char *name)
{
    int nameIndex = 0;
    int retValIndex = 0;
    char *retVal = utMakeString(2*strlen(name));

    while(name[nameIndex] != 0)
    {
        retVal[retValIndex] = name[nameIndex];
        if(name[nameIndex] == '\\')
        {
            retValIndex++;
            retVal[retValIndex] = '\\';
        }
        retValIndex++;
        nameIndex++;
   }
    retVal[retValIndex] = 0;
    return retVal;
}


#undef utSymGetName
/*--------------------------------------------------------------------------------------------------
  For debugging.
--------------------------------------------------------------------------------------------------*/
char *utSymGetName(
    utSym sym)
{
    return (utSymNames + utSyms[sym].xName);
}

/*--------------------------------------------------------------------------------------------------
  Set the callback function that reports progress, and checks for user
  cancelation.
--------------------------------------------------------------------------------------------------*/
void utSetProgressCallback(
    utProgressProc progressProc)
{
    utUserProgressProc = progressProc;
}

/*--------------------------------------------------------------------------------------------------
  Check for user cancelation of a tool run.  Pass the percent done.
--------------------------------------------------------------------------------------------------*/
bool utSetProgress(
    U8 percentDone)
{
    static U8 lastPercentage = U8_MAX;
    bool abort;

    if(percentDone != lastPercentage && utUserProgressProc != NULL) {
        lastPercentage = percentDone;
        abort = utUserProgressProc(percentDone);
        if(abort && utEnableLongjmpOnAbort) {
            utLogMessage("User aborted process");
            utResetProgress();
            utLongjmp();
        }
        return abort;
    }
    return false;
}

/*--------------------------------------------------------------------------------------------------
  Set progress to 0, and clear the utAbort flag.
--------------------------------------------------------------------------------------------------*/
void utResetProgress(void)
{
    utSetProgress(0);
    utAbort = false;
}

/*--------------------------------------------------------------------------------------------------
  Find the size of the file.
--------------------------------------------------------------------------------------------------*/
U32 utFindFileSize(
    FILE *file)
{
    long fileSize;

    fseek(file, 0, SEEK_END);
    fileSize = ftell(file);
    fseek(file, 0, SEEK_SET);
    return fileSize;
}

/*--------------------------------------------------------------------------------------------------
  Just determine if the file exists.
--------------------------------------------------------------------------------------------------*/
bool utFileExists(
    char *fileName)
{
    FILE *file;

    if(fileName == NULL || *fileName == '\0') {
        return false;
    }
    file = fopen(fileName, "r");
    if(file != NULL) {
        fclose(file);
        return true;
    }
    return false;
}

/*--------------------------------------------------------------------------------------------------
  Buffered getc.
--------------------------------------------------------------------------------------------------*/
UTINT utGetc_(
    FILE *file)
{
    utFileBufferSize = fread(utFileBuffer, sizeof(char), UTSTRLEN, file);
    if(utFileBufferSize == 0) {
        return EOF;
    }
    utFileBufferPos = 0;
    return utGetc(file);
}


/*--------------------------------------------------------------------------------------------------
  Open the file, load the buffer.
--------------------------------------------------------------------------------------------------*/
FILE *utFopen(
    char *fileName,
    char *mode)
{
    char *fullPath;
    FILE *file = fopen(fileName, mode);
    if(file==0)
    {
        fullPath = utFullPath(fileName);
        utWarning("%s\nThe current working directory is %s",strerror(errno), fullPath);
    }

    utFileBufferSize = 0;
    utFileBufferPos = 0;
    return file;
}

/*--------------------------------------------------------------------------------------------------
    returns a + 1, unsigned
--------------------------------------------------------------------------------------------------*/
U64 utIncrementU64By1(
    U64 a)
{
    U64 r;

    r.lower = a.lower + 1;
    r.upper = utUnlikely(r.lower == 0) ? a.upper + 1 : a.upper;   /* carry if r.lower rolled over */
    return r;
}

/*--------------------------------------------------------------------------------------------------
    returns a + b, unsigned
--------------------------------------------------------------------------------------------------*/
U64 utAddU64(
    U64 a,
    U64 b)
{
    U64 r;
    r.lower = a.lower + b.lower;
    r.upper = a.upper + b.upper;
    if(r.lower < a.lower && r.lower < b.lower) {
        r.upper++;
    }
    return r;
}

/*--------------------------------------------------------------------------------------------------
    returns a + b, signed
--------------------------------------------------------------------------------------------------*/
S64 utAddS64(
    S64 a,
    S64 b)
{
    S64 r;
    r.lower = a.lower + b.lower;
    r.upper = a.upper + b.upper;
    if(r.lower < a.lower && r.lower < b.lower) {
        r.upper++;
    }
    return r;
}

/*--------------------------------------------------------------------------------------------------
    subtracts b from a and returns that value, unsigned
--------------------------------------------------------------------------------------------------*/
U64 utSubtractU64(
    U64 a,
    U64 b)
{
    U64 r;
    r.upper = a.upper - b.upper;
    r.lower = a.lower - b.lower;
    if(a.lower < b.lower) {
        r.upper--;
    }
    return r;
}

/*--------------------------------------------------------------------------------------------------
    subtracts b from a and returns that value, signed
--------------------------------------------------------------------------------------------------*/
S64 utSubtractS64(
    S64 a,
    S64 b)
{
    S64 r;
    r.upper = a.upper - b.upper;
    r.lower = a.lower - b.lower;
    if(a.lower < b.lower) {
        r.upper--;
    }
    return r;
}

/*--------------------------------------------------------------------------------------------------
    converts a U32 to a U64
--------------------------------------------------------------------------------------------------*/
U64 utConvertU32ToU64(
    U32 a)
{
    U64 r;
    r.upper = 0;
    r.lower = a;
    return r;
}

/*--------------------------------------------------------------------------------------------------
    assumes a < U32_MAX
    converts U64 to a U32
--------------------------------------------------------------------------------------------------*/
U32 utConvertU64ToU32(
    U64 a)
{
    return a.lower;
}

/*--------------------------------------------------------------------------------------------------
    converts a U32 to an S64
--------------------------------------------------------------------------------------------------*/
S64 utConvertU32ToS64(
    U32 a)
{
    S64 r;
    r.upper = 0;
    r.lower = a;
    return r;
}

/*--------------------------------------------------------------------------------------------------
    assumes a>=0 and a <= U32_MAX
    converts an S64 to a U32
--------------------------------------------------------------------------------------------------*/
U32 utConvertS64ToU32(
    S64 a)
{
    return a.lower;
}

/*--------------------------------------------------------------------------------------------------
    converts an S32 to an S64
--------------------------------------------------------------------------------------------------*/
S64 utConvertS32ToS64(
    S32 a)
{
    S64 r;
    U8 bit;
    r.upper = 0;
    r.lower = a;
    bit = utGetBit(a,31);
    if(bit==1)
    {
        r.upper = -1;
    }
    return r;
}

/*--------------------------------------------------------------------------------------------------
    assumes S32_min <= a <= S32_max
    converts an S64 to an S32
--------------------------------------------------------------------------------------------------*/
S32 utConvertS64ToS32(
    S64 a)
{
    return (S32)(a.lower);
}

/*--------------------------------------------------------------------------------------------------
    assumes a is positive
    converts an S64 to a U64
--------------------------------------------------------------------------------------------------*/
U64 utConvertS64ToU64(
    S64 a)
{
    U64 r;
    r.upper = (U32)(a.upper);
    r.lower = a.lower;
    return r;
}

/*--------------------------------------------------------------------------------------------------
    assumes most significant bit of a is 0
    converts a U64 to an S64
--------------------------------------------------------------------------------------------------*/
S64 utConvertU64ToS64(
    U64 a)
{
    S64 r;
    r.upper = (S32)(a.upper);
    r.lower = a.lower;
    return r;
}

/*--------------------------------------------------------------------------------------------------
    assumes a <= S32_MAX
    converts a U64 to an S32
--------------------------------------------------------------------------------------------------*/
S32 utConvertU64ToS32(
    U64 a)
{
    return a.lower;
}

/*--------------------------------------------------------------------------------------------------
    assumes a > 0
    converts an S32 to a U64
--------------------------------------------------------------------------------------------------*/
U64 utConvertS32ToU64(
    S32 a)
{
    U64 r;
    r.upper = 0;
    r.lower = a;
    return r;
}

/*--------------------------------------------------------------------------------------------------
    converts an S64 to a double, with some loss in precision (of course)
--------------------------------------------------------------------------------------------------*/
double utConvertS64ToDouble(
    S64 a)
{   /* avoid compiler-specific notation for a floating-point double constant */
    double r = ((double) 65536.0 * (double) 65536.0) * a.upper + a.lower;
    return r;
}
/*--------------------------------------------------------------------------------------------------
    converts a U32 to a U64
--------------------------------------------------------------------------------------------------*/
U64 utMakeU64(
    U32 a)
{
    return utConvertU32ToU64(a);
}

/*--------------------------------------------------------------------------------------------------
    converts an S32 to an S64
--------------------------------------------------------------------------------------------------*/
S64 utMakeS64(
    S32 a)
{
    return utConvertS32ToS64(a);
}



/*--------------------------------------------------------------------------------------------------
    returns true if a > b
--------------------------------------------------------------------------------------------------*/
bool utgtU64(
    U64 a,
    U64 b)
{
    if(a.upper > b.upper) {
        return true;
    }
    if(a.upper < b.upper) {
        return false;
    }
    if(a.lower > b.lower) {
        return true;
    }
    return false;
}

/*--------------------------------------------------------------------------------------------------
    returns true if a > b
--------------------------------------------------------------------------------------------------*/
bool utgtS64(
    S64 a,
    S64 b)
{
    if(a.upper > b.upper) {
        return true;
    }
    if(a.upper < b.upper) {
        return false;
    }
    if(a.lower > b.lower) {
        return true;
    }
    return false;
}

/*--------------------------------------------------------------------------------------------------
    returns true if a < b
--------------------------------------------------------------------------------------------------*/
bool utltU64(
    U64 a,
    U64 b)
{
    return utgtU64(b,a);
}

/*--------------------------------------------------------------------------------------------------
    return true if a < b
--------------------------------------------------------------------------------------------------*/
bool utltS64(
    S64 a,
    S64 b)
{
    return utgtS64(b,a);
}

/*--------------------------------------------------------------------------------------------------
    returns true if a == b
--------------------------------------------------------------------------------------------------*/
bool uteU64(
    U64 a,
    U64 b)
{
    return (a.lower==b.lower) && (a.upper==b.upper);
}

/*--------------------------------------------------------------------------------------------------
    returns true if a == b
--------------------------------------------------------------------------------------------------*/
bool uteS64(
    S64 a,
    S64 b)
{
    return (a.lower==b.lower) && (a.upper==b.upper);
}

/*--------------------------------------------------------------------------------------------------
    returns true if a >= b
--------------------------------------------------------------------------------------------------*/
bool utgteU64(
    U64 a,
    U64 b)
{
    if(a.upper > b.upper) {
        return true;
    }
    if(a.upper < b.upper) {
        return false;
    }
    if(a.lower >= b.lower) {
        return true;
    }
    return false;
}

/*--------------------------------------------------------------------------------------------------
    return true if a >= b
--------------------------------------------------------------------------------------------------*/
bool utgteS64(
    S64 a,
    S64 b)
{
    if(a.upper > b.upper) {
        return true;
    }
    if(a.upper < b.upper) {
        return false;
    }
    if(a.lower >= b.lower) {
        return true;
    }
    return false;
}

/*--------------------------------------------------------------------------------------------------
    return true if a <= b
--------------------------------------------------------------------------------------------------*/
bool utlteU64(
    U64 a,
    U64 b)
{
    return utgteU64(b, a);
}

/*--------------------------------------------------------------------------------------------------
    returns true if a <= b
--------------------------------------------------------------------------------------------------*/
bool utlteS64(
    S64 a,
    S64 b)
{
    return utgteS64(b, a);
}

/*--------------------------------------------------------------------------------------------------
    returns a / 2
--------------------------------------------------------------------------------------------------*/
S64 utDivideS64By2(
    S64 a)
{
    S64 r;

    r.upper = a.upper >> 1;
    r.lower = a.lower >> 1;
    if(a.upper & 1) {
        a.lower |= 0x80000000;   /* propogate bit down */
    }
    return r;
}

/*--------------------------------------------------------------------------------------------------
    returns a / b
--------------------------------------------------------------------------------------------------*/
S64 utDivideS64ByS16(
    S64 a,
    S16 b,
    S16 *retRemainder)
{
    S64 retval;
    S64 numerator;
    S32 denominator;
    U16 returnArray[4];
    U16 numeratorArray[4];
    U32 remainder;
    bool negative;

    if(b < 0 ) {
        if(a.upper < 0) {
            denominator = -b;
            numerator = utSubtractS64(S64_0, a);
            negative = false;
        } else {
            denominator = -b;
            numerator = a;
            negative = true;
        }
    } else {
        if(a.upper < 0) {
            denominator = b;
            numerator = utSubtractS64(S64_0, a);
            negative = true;
        } else {
            denominator = b;
            numerator = a;
            negative = false;
        }
    }
    numeratorArray[0] = numerator.lower & ((1 << 16) - 1);
    numeratorArray[1] = numerator.lower >> 16;
    numeratorArray[2] = (U16)(numerator.upper & ((1 << 16) - 1));
    numeratorArray[3] = numerator.upper >> 16;
    remainder = numeratorArray[3];
    returnArray[3] = remainder / denominator;
    remainder = remainder - denominator * returnArray[3];
    remainder = (remainder << 16) + numeratorArray[2];
    returnArray[2] = remainder / denominator;
    remainder = remainder - denominator * returnArray[2];
    remainder = (remainder << 16) + numeratorArray[1];
    returnArray[1] = remainder / denominator;
    remainder = remainder - denominator * returnArray[1];
    remainder = (remainder << 16) + numeratorArray[0];
    returnArray[0] = remainder / denominator;
    remainder = remainder - denominator * returnArray[0];
    retval.upper = (returnArray[3] << 16) + returnArray[2];
    retval.lower = (returnArray[1] << 16) + returnArray[0];
    *retRemainder = remainder;
    if(negative) {
        if(remainder > 0) {
            *retRemainder -= b;
            return utSubtractS64(utConvertS32ToS64(-1), retval);
        }
        return utSubtractS64(S64_0, retval);
    }
    return retval;
}

/*--------------------------------------------------------------------------------------------------
    returns a / b
--------------------------------------------------------------------------------------------------*/
S64 utDivideS64ByS32(
    S64 arga,
    S32 argb)
{
    S64 a = arga;
    S32 b = argb;
    U64 numerator;
    U32 denominator;
    U64 retabs;
    S64 retval;
    U64 remainder;
    U32 maxquotient;
    U32 maxremainder;
    U32 temp;
    S64 newQuotient;
    bool negative = false;

    utAssert(argb != 0);
    if(a.upper < 0) {
        negative = !negative;
        a = utSubtractS64(S64_0, a);
    }
    if(b < 0) {
        negative = !negative;
        b = 0 - b;
    }
    numerator = utConvertS64ToU64(a);
    denominator = b;
    maxquotient = U32_MAX / denominator;
    maxremainder = U32_MAX - denominator * maxquotient;
    if(maxremainder + 1 == denominator) {
        maxremainder = 0;
        maxquotient++;
    }
    if(numerator.upper == 0) {
        temp = numerator.lower / denominator;
        if(negative) {
            return utSubtractS64(S64_0, utConvertU32ToS64(temp));
        } else {
            return utConvertU32ToS64(temp);
        }
    }
    retabs = utConvertS64ToU64(utMultiplyS64(utConvertU32ToS64(maxquotient), utConvertU32ToS64(numerator.upper)));
    remainder = utSubtractU64(numerator, utConvertS64ToU64(utMultiplyS64(utConvertU32ToS64(denominator), utConvertU64ToS64(retabs))));
    newQuotient = utDivideS64ByS32(utConvertU64ToS64(remainder), (S32)denominator);
    retabs = utAddU64(retabs, utConvertS64ToU64(newQuotient));
    if(negative) {
        retval = utSubtractS64(S64_0, utConvertU64ToS64(retabs));
    } else {
        retval = utConvertU64ToS64(retabs);
    }
    return retval;
}


/*--------------------------------------------------------------------------------------------------
    returns a * b
    behavior unpredictable incase of overflow
--------------------------------------------------------------------------------------------------*/
S64 utMultiplyS64(
    S64 a,
    S64 b)
{
    bool negative, aIsSmall;
    U32 carry;
    U32 small[2];
    U32 big[4];
    U32 product[4];
    U32 smaller = 0;
    S64 larger, temp, retval;

    if(a.upper < 0) {
        negative = true;
        temp = utSubtractS64(S64_0, a);
    } else {
        negative = false;
        temp = a;
    }
    if(temp.upper > 0) {
        aIsSmall = false;
        larger = temp;
    } else {
        aIsSmall = true;
        smaller = temp.lower;
    }
    if(b.upper < 0) {
        negative = !negative;
        temp = utSubtractS64(S64_0, b);
    } else {
        temp = b;
    }
    if(temp.upper == 0 && !aIsSmall) {
        smaller = temp.lower;
    } else {
        larger = temp;
    }
    big[0] = larger.lower & ((1 << 16) - 1);
    big[1] = larger.lower >> 16;
    big[2] = larger.upper & ((1 << 16) - 1);
    big[3] = larger.upper >> 16;
    small[0] = smaller & ((1 << 16) - 1);
    small[1] = smaller >> 16;
    carry = small[0] * big[0];
    product[0] = carry & ((1 << 16) - 1);
    product[1] = carry >> 16;
    carry = small[0] * big[1] + small[1] * big[0] + product[1];
    product[1] = carry & ((1 << 16) - 1);
    product[2] = carry >> 16;
    carry = product[2] + small[0] * big[2] + small[1] * big[1];
    product[2] = carry & ((1 << 16) - 1);
    product[3] = carry >> 16;
    carry = product[3] + small[0] * big[3] + small[1] * big[2];
    product[3] = carry & ((1 << 16) - 1);
    retval.upper = (1 << 16) * product[3] + product[2];
    retval.lower = (1 << 16) * product[1] + product[0];
    if(negative) {
        retval = utSubtractS64(S64_0, retval);
    }
    return retval;
}

/*--------------------------------------------------------------------------------------------------
  Return a 64-bit integer from the string.
--------------------------------------------------------------------------------------------------*/
S64 utStrToS64(
    char *string,
    U32 length)
{
    S64 value = S64_0;
    S64 ten = utMakeS64(10);
    bool negate = false;

    if(*string == '-') {
        negate = true;
        string++;
        length--;
    }
    while(length--) {
        value = utMultiplyS64(value, ten);
        value = utAddS64(value, utMakeS64(*string - '0'));
        string++;
    }
    if(negate) {
        value = utSubtractS64(S64_0, value);
    }
    return value;
}

/*--------------------------------------------------------------------------------------------------
  Return a string representing the 64-bit integer in base 10.
--------------------------------------------------------------------------------------------------*/
char *utS64ToStr(
    S64 value)
{
    char *buffer = utMakeString(30);
    U32 xChar = 29;
    S16 remainder;
    bool negative = false;
    
    buffer[29] = '\0';
    if(uttzeroS64(value)) {
        buffer[--xChar] = '0';
    } else {
        if(utltS64(value, S64_0)) {
            value = utSubtractS64(S64_0, value);
            negative = true;
        }
        while(!uttzeroS64(value)) {
            value = utDivideS64ByS16(value, 10, &remainder);
            buffer[--xChar] = '0' + remainder;
        }
        if(negative) {
            buffer[--xChar] = '-';
        }
    }
    return buffer + xChar;
}

bool utTestFor64Bit()
{
    S64 bigS, temp, temp2, divisor ;
    S16 smallS = -100, remainder;
    S32 s = smallS;
    U64 bigU;
    U32 u;
    double d;

    temp.upper = 232;
    temp.lower = 3567587328u;
    bigS = temp;
    /* bigS and temp = a trillion */
    bigS = utConvertS32ToS64(s);
    s = utConvertS64ToS32(bigS);
    if(s != -100) {
        /* conversion between S32 and S64 broken */
        return false;
    }
    u = 100;
    bigU = utConvertU32ToU64(u);
    u = utConvertU64ToU32(bigU);
    if(u != 100) {
        /* conversion between U32 and U64 broken */
        return false;
    }
    bigS = utConvertU32ToS64(u);
    u = utConvertS64ToU32(bigS);
    if(u != 100) {
        /* conversion between U32 and S64 broken */
        return false;
    }
    s = 100;
    bigU = utConvertS32ToU64(s);
    s = utConvertU64ToS32(bigU);
    if(s!=100) {
        /* conversion between S32 and U64 broken */
        return false;
    }
    bigS = temp;
    bigU = utConvertS64ToU64(bigS);
    bigS = utConvertU64ToS64(bigU);
    if(!uteS64(bigS, temp)) {
        /* conversion between S64 and U64 broken  */
        return false;
    }
    smallS = -100;
    bigS = utDivideS64ByS16(temp, smallS, &remainder);
    bigS = utMultiplyS64(bigS, utConvertS32ToS64(smallS));
    if(!uteS64(bigS, temp)) {
        /* multiply or divide broken */
        return false;
    }
    bigS = utAddS64(temp, temp);
    bigS = utSubtractS64(bigS, temp);
    if(!uteS64(bigS, temp))
    {
        /* add or subtract broken with S64*/
        return false;
    }
    bigU = utAddU64(utConvertS64ToU64(temp), utConvertS64ToU64(temp));
    bigU = utSubtractU64(bigU, utConvertS64ToU64(temp));
    if(!uteU64(bigU, utConvertS64ToU64(temp))) {
        /* add or subtract broken with U64 */
        return false;
    }
    d = utConvertS64ToDouble(temp);
    if((d<0.9999e12) || (d>1.0001e12)) {
        /* S64 to double broken (positive numbers) */
        return false;
    }
    temp2 = utMultiplyS64(temp, utMakeS64(-1));   /* negative 1 trillion */
    d = utConvertS64ToDouble(temp2);
    if((d<-1.0001e12) || (d>0.9999e12)) {
        /* S64 to double broken (negative numbers) */
        return false;
    }
    for(u=0;u<1000000000;u++) {
        if(u%10000000 == 0) {
            utStatus("%d percent complete with check!!", u/10000000);
        }
        smallS = utRand();
        while(smallS == 0) {
            smallS = utRand();
        }
        bigS.lower = utRand()*utRand();
        bigS.upper = utRand();
        divisor = utConvertS32ToS64(smallS);
        temp = utMultiplyS64(bigS, divisor);
        temp2 = utDivideS64ByS16(temp, (S16)smallS, &remainder);
        if(!uteS64(temp2, bigS)) {
            utAssert(false);
        }
    }
    /* boolean operators not checked */
    return true;
}

U64 utSquareRootU64(
    U64 a)
{
    S64 retval;
    S64 arg = utConvertU64ToS64(a);
    S8 bit;
    retval.upper = 0;
    retval.lower = 0;
    for(bit = 31; bit>=0; bit--) {
          utSetBit(retval.lower, bit, 1);
          if(utgtS64(utMultiplyS64(retval,retval), arg)) {
            utSetBit(retval.lower, bit, 0);
          }
    }
    return utConvertS64ToU64(retval);
}

/*--------------------------------------------------------------------------------------------------
  Find a file name in the directory and return the full path if it exists.  Otherwise return NULL.
--------------------------------------------------------------------------------------------------*/
static char *findFileInDirectory(
    char *fileName,
    char *dirName)
{
    char *name = utSprintf("%s%c%s", dirName, UTDIRSEP, fileName);

    if(utAccess(name, NULL)) {
        return name;
    }
    return NULL;
}

/*--------------------------------------------------------------------------------------------------
  Find a file in the path that has the mode.
--------------------------------------------------------------------------------------------------*/
char *utFindInPath(
    char *name,
    char *path)
{
    char *buf = utCopyString(path);
    char *p = buf;
    char *next, *fileName, *directory;

    while(p != '\0') {
        next = strchr(p, ':');
        if(next != NULL) {
            *next++ = '\0';
        }
        directory = utExpandEnvVariables(p);
        fileName = findFileInDirectory(name, directory);
        if(fileName != NULL) {
            return fileName;
        }
        p = next;
    }
    return NULL;
}

/*--------------------------------------------------------------------------------------------------
  Find the matching '}' in the string.
--------------------------------------------------------------------------------------------------*/
static char *findMatchingBracket(
    char *string)
{
    U32 numBraces = 0;
    char c;

    utDo {
        c = *string;
        if(c == '{') {
            numBraces++;
        } else if(c == '}') {
            numBraces--;
        }
    } utWhile(numBraces != 0) {
        string++;
    } utRepeat;
    return string;
}

/*--------------------------------------------------------------------------------------------------
  Find the first non alpha-numeric character.
--------------------------------------------------------------------------------------------------*/
static char *findFirstNonAlnumChar(
    char *string)
{
    while(isalnum(*string)) {
        string++;
    }
    return string;
}

/*--------------------------------------------------------------------------------------------------
  Convert '\\' characters to '/' and vise-versa, to match UTDIRSEP.
--------------------------------------------------------------------------------------------------*/
char *utConvertDirSeparation(
    char *oldString)
{
    char *newString = utCopyString(oldString);
    char c, *p =  newString;

    utDo {
        c = *p;
    } utWhile(c != '\0') {
        if(c == '/' || c == '\\') {
            *p = UTDIRSEP;
        }
        p++;
    } utRepeat;
    return newString;
}

/*--------------------------------------------------------------------------------------------------
  Expand the string's environment variable.
--------------------------------------------------------------------------------------------------*/
char *expandString(
    char *string,
    char *varStart)
{
    char *ending;
    char *p;

    *varStart++ = '\0';
    if(*varStart == '{') {
        ending = findMatchingBracket(varStart);
        varStart++;
        if(ending == NULL) {
            utWarning("Variable %s does not have a matching '}'" , varStart);
            return string;
        }
        *ending++ = '\0';
    } else {
        ending = findFirstNonAlnumChar(varStart);
        if(ending == NULL) {
            ending = "";
        } else {
            p = ending;
            ending = utCopyString(ending);
            *p = '\0'; /* To terminate varStart */
        }
    }
    return utSprintf("%s%s%s", string, utConvertDirSeparation(getenv(varStart)), ending);
}

/*--------------------------------------------------------------------------------------------------
  Expand the string to replace environment variables with their values.
--------------------------------------------------------------------------------------------------*/
char *utExpandEnvVariables(
    char *string)
{
    char *p;
    bool changed;

    string = utCopyString(string); /* To make a writable copy */
    if(string == NULL) {
        return NULL;
    }
    do {
        changed = false;
        p = strchr(string, '$');
        if(p != NULL) {
            changed = true;
            string = expandString(string, p);
        }
    } while(changed);
    return string;
}

/*--------------------------------------------------------------------------------------------------
  Determine if a name has a range, and what the range is.
--------------------------------------------------------------------------------------------------*/
bool utNameHasRange(
    char *name,
    U32 *left,
    U32 *right)
{
    char *buf = utCopyString(name);
    char *leftPtr = strrchr(buf, '[');
    char *rightPtr, *p, *endLeftPtr, *endRightPtr;

    *left = 0;
    *right = 0;
    if(leftPtr == NULL) {
        return false;
    }
    leftPtr++;
    rightPtr = strchr(leftPtr, ':');
    if(rightPtr == NULL) {
        return false;
    }
    *rightPtr++ = '\0';
    p = strchr(rightPtr, ']');
    if(p == NULL) {
        return false;
    }
    *p++ = '\0';
    if(*p != '\0') {
        return false;
    }
    *left = strtol(leftPtr, &endLeftPtr, 10);
    *right = strtol(rightPtr, &endRightPtr, 10);
    if(*endLeftPtr != '\0' || *endRightPtr != '\0') {
        return false;
    }
    return true;
}
