/*----------------------------------------------------------------------------
  Simple class support routines such as list manipulation.
----------------------------------------------------------------------------*/
#include "cirddr.h"

void (*cirDeviceDestructorHook)(cirDevice);
cirDevice _cirFirstFreeDevice, _cirTempDevice;
utHeapRef _cirDeviceHeap;
void (*cirPinDestructorHook)(cirPin);
cirPin _cirFirstFreePin, _cirTempPin;
utHeapRef _cirPinHeap;
void (*cirRootDestructorHook)(cirRoot);
cirRoot _cirFirstFreeRoot, _cirTempRoot;
utHeapRef _cirRootHeap;
void (*cirAttrDestructorHook)(cirAttr);
cirAttr _cirFirstFreeAttr, _cirTempAttr;
utHeapRef _cirAttrHeap;
struct _utFunctionTable cirNetlistFunctionTable = {utSymNull, cirNetlistDestroy};
utSym cirNetlistClassName;
cirNetlist _cirFirstFreeNetlist, _cirTempNetlist;
utHeapRef _cirNetlistHeap;
void (*cirParamDestructorHook)(cirParam);
cirParam _cirFirstFreeParam, _cirTempParam;
utHeapRef _cirParamHeap;

/*----------------------------------------------------------------------------
  Add the Attr to the head of the list on the Device.
----------------------------------------------------------------------------*/
void cirDeviceInsertAttr(
    cirDevice Device,
    cirAttr _Attr)
{
    cirAttrSetnextDeviceAttr(_Attr, cirDeviceGetfirstAttr(Device));
    cirDeviceSetfirstAttr(Device, _Attr);
    if(cirDeviceGetlastAttr(Device) == cirAttrNull) {
        cirDeviceSetlastAttr(Device, _Attr);
    }
}

/*----------------------------------------------------------------------------
  Remove the Attr from the list on the Device.
----------------------------------------------------------------------------*/
void cirDeviceRemoveAttr(
    cirDevice Device,
    cirAttr _Attr)
{
    cirAttr nextAttr, prevAttr = cirAttrNull;

    for(nextAttr = cirDeviceGetfirstAttr(Device);
        nextAttr != cirAttrNull && nextAttr != _Attr;
        nextAttr = cirAttrGetnextDeviceAttr(nextAttr)) {
        prevAttr = nextAttr;
    }
    if(nextAttr == cirAttrNull) {
        /* Not in list */
        return;
    }
    if(prevAttr != cirAttrNull) {
        cirAttrSetnextDeviceAttr(prevAttr, cirAttrGetnextDeviceAttr(_Attr));
    } else {
        cirDeviceSetfirstAttr(Device, cirAttrGetnextDeviceAttr(_Attr));
    }
    if(cirDeviceGetlastAttr(Device) == _Attr) {
        cirDeviceSetlastAttr(Device, prevAttr);
    }
    cirAttrSetnextDeviceAttr(_Attr, cirAttrNull);
}

/*----------------------------------------------------------------------------
  Add the Attr to the tail of the list on the Device.
----------------------------------------------------------------------------*/
void cirDeviceAppendAttr(
    cirDevice Device,
    cirAttr _Attr)
{
    if(cirDeviceGetlastAttr(Device) == cirAttrNull) {
        cirDeviceSetfirstAttr(Device, _Attr);
    } else {
        cirAttrSetnextDeviceAttr(cirDeviceGetlastAttr(Device), _Attr);
    }
    cirDeviceSetlastAttr(Device, _Attr);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Device.
------------------------------------------------------------------------*/
static utBlockRef buildDeviceBlock(void)
{
    utBlockRef block = utcBlock();
    cirDevice Device;
    U16 x, sDevice;

    utaHeapBlock(_cirDeviceHeap, block);
    sDevice = UTBLOCKSIZE/sizeof(struct _cirDevice);
    _cirFirstFreeDevice = (cirDevice)utgBlockMem(block);
    Device = _cirFirstFreeDevice;
    for(x = 0; x < sDevice; x++) {
        _cirDeviceNextFree(Device) = Device + 1;
        Device++;
    }
    _cirDeviceNextFree(--Device) = cirDeviceNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Device.
----------------------------------------------------------------------------*/
cirDevice cirDeviceAlloc(void)
{
    cirDevice Device;

    if(_cirFirstFreeDevice == cirDeviceNull) {
        buildDeviceBlock();
    }
    Device = _cirFirstFreeDevice;
    _cirFirstFreeDevice = _cirDeviceNextFree(Device);
    memset((void *)Device, 0, sizeof(struct _cirDevice));
    utInitHtbl(&Device->Pins);
    return Device;
}

/*----------------------------------------------------------------------------
  Free a Device.
----------------------------------------------------------------------------*/
void cirDeviceFree(
    cirDevice Device)
{
    utdArray(utfHtblArray(cirDeviceGetPins(Device)));
    _cirDeviceNextFree(Device) = _cirFirstFreeDevice;
    _cirFirstFreeDevice = Device;
}

/*----------------------------------------------------------------------------
  Destroy a Device.
----------------------------------------------------------------------------*/
void cirDeviceDestroy(
    cirDevice Device)
{
    if(cirDeviceDestructorHook != NULL) {
        cirDeviceDestructorHook(Device);
    }
    {
        cirPin _Pin;
        cirSafeForeachDevicePin(Device, _Pin) {
            cirPinDestroy(_Pin);
        } cirEndSafeForeachDevicePin;
    }
    {
        cirAttr _Attr;
        cirSafeForeachDeviceAttr(Device, _Attr) {
            cirAttrDestroy(_Attr);
        } cirEndSafeForeachDeviceAttr;
    }
    {
        cirRoot parentRoot = cirDeviceGetRoot(Device);
        if(parentRoot != cirRootNull) {
            cirRootRemoveDevice(parentRoot, Device);
        }
    }
    cirDeviceFree(Device);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Pin.
------------------------------------------------------------------------*/
static utBlockRef buildPinBlock(void)
{
    utBlockRef block = utcBlock();
    cirPin Pin;
    U16 x, sPin;

    utaHeapBlock(_cirPinHeap, block);
    sPin = UTBLOCKSIZE/sizeof(struct _cirPin);
    _cirFirstFreePin = (cirPin)utgBlockMem(block);
    Pin = _cirFirstFreePin;
    for(x = 0; x < sPin; x++) {
        _cirPinNextFree(Pin) = Pin + 1;
        Pin++;
    }
    _cirPinNextFree(--Pin) = cirPinNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Pin.
----------------------------------------------------------------------------*/
cirPin _cirPinAlloc(void)
{
    cirPin Pin;

    if(_cirFirstFreePin == cirPinNull) {
        buildPinBlock();
    }
    Pin = _cirFirstFreePin;
    _cirFirstFreePin = _cirPinNextFree(Pin);
    memset((void *)Pin, 0, sizeof(struct _cirPin));
    return Pin;
}

/*----------------------------------------------------------------------------
  Free a Pin.
----------------------------------------------------------------------------*/
void _cirPinFree(
    cirPin Pin)
{
    _cirPinNextFree(Pin) = _cirFirstFreePin;
    _cirFirstFreePin = Pin;
}

/*----------------------------------------------------------------------------
  Destroy a Pin.
----------------------------------------------------------------------------*/
void cirPinDestroy(
    cirPin Pin)
{
    if(cirPinDestructorHook != NULL) {
        cirPinDestructorHook(Pin);
    }
    cirPinFree(Pin);
}

/*----------------------------------------------------------------------------
  Add the Device to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void cirRootInsertDevice(
    cirRoot Root,
    cirDevice _Device)
{
    utInsertHtblEntry(cirRootGetDevices(Root), &(_Device)->h);
    cirDeviceSetRoot(_Device, Root);
}

/*----------------------------------------------------------------------------
  Remove the Device from the list on the Root.
----------------------------------------------------------------------------*/
void cirRootRemoveDevice(
    cirRoot Root,
    cirDevice _Device)
{
    utDeleteHtblEntry(&(Root)->Devices, &(_Device)->h);
    cirDeviceSetRoot(_Device, cirRootNull);
}

/*----------------------------------------------------------------------------
  Append the Device to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void cirRootAppendDevice(
    cirRoot Root,
    cirDevice _Device)
{
    utAppendHtblEntry(cirRootGetDevices(Root), &(_Device)->h);
    cirDeviceSetRoot(_Device, Root);
}

/*----------------------------------------------------------------------------
  Add the Netlist to the head of the list on the Root.
----------------------------------------------------------------------------*/
void cirRootInsertNetlist(
    cirRoot Root,
    cirNetlist _Netlist)
{
    cirNetlistSetnextRootNetlist(_Netlist, cirRootGetfirstNetlist(Root));
    cirRootSetfirstNetlist(Root, _Netlist);
    cirNetlistSetRoot(_Netlist, Root);
}

/*----------------------------------------------------------------------------
  Remove the Netlist from the list on the Root.
----------------------------------------------------------------------------*/
void cirRootRemoveNetlist(
    cirRoot Root,
    cirNetlist _Netlist)
{
    cirNetlist nextNetlist, prevNetlist = cirNetlistNull;

    for(nextNetlist = cirRootGetfirstNetlist(Root);
        nextNetlist != cirNetlistNull && nextNetlist != _Netlist;
        nextNetlist = cirNetlistGetnextRootNetlist(nextNetlist)) {
        prevNetlist = nextNetlist;
    }
    if(nextNetlist == cirNetlistNull) {
        /* Not in list */
        return;
    }
    if(prevNetlist != cirNetlistNull) {
        cirNetlistSetnextRootNetlist(prevNetlist, cirNetlistGetnextRootNetlist(_Netlist));
    } else {
        cirRootSetfirstNetlist(Root, cirNetlistGetnextRootNetlist(_Netlist));
    }
    cirNetlistSetnextRootNetlist(_Netlist, cirNetlistNull);
    cirNetlistSetRoot(_Netlist, cirRootNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Root.
------------------------------------------------------------------------*/
static utBlockRef buildRootBlock(void)
{
    utBlockRef block = utcBlock();
    cirRoot Root;
    U16 x, sRoot;

    utaHeapBlock(_cirRootHeap, block);
    sRoot = UTBLOCKSIZE/sizeof(struct _cirRoot);
    _cirFirstFreeRoot = (cirRoot)utgBlockMem(block);
    Root = _cirFirstFreeRoot;
    for(x = 0; x < sRoot; x++) {
        _cirRootNextFree(Root) = Root + 1;
        Root++;
    }
    _cirRootNextFree(--Root) = cirRootNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Root.
----------------------------------------------------------------------------*/
cirRoot cirRootAlloc(void)
{
    cirRoot Root;

    if(_cirFirstFreeRoot == cirRootNull) {
        buildRootBlock();
    }
    Root = _cirFirstFreeRoot;
    _cirFirstFreeRoot = _cirRootNextFree(Root);
    memset((void *)Root, 0, sizeof(struct _cirRoot));
    utInitHtbl(&Root->Devices);
    return Root;
}

/*----------------------------------------------------------------------------
  Free a Root.
----------------------------------------------------------------------------*/
void cirRootFree(
    cirRoot Root)
{
    utdArray(utfHtblArray(cirRootGetDevices(Root)));
    _cirRootNextFree(Root) = _cirFirstFreeRoot;
    _cirFirstFreeRoot = Root;
}

/*----------------------------------------------------------------------------
  Destroy a Root.
----------------------------------------------------------------------------*/
void cirRootDestroy(
    cirRoot Root)
{
    if(cirRootDestructorHook != NULL) {
        cirRootDestructorHook(Root);
    }
    {
        cirDevice _Device;
        cirSafeForeachRootDevice(Root, _Device) {
            cirDeviceDestroy(_Device);
        } cirEndSafeForeachRootDevice;
    }
    {
        cirNetlist _Netlist;
        cirSafeForeachRootNetlist(Root, _Netlist) {
            cirNetlistDestroy(_Netlist);
        } cirEndSafeForeachRootNetlist;
    }
    cirRootFree(Root);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Attr.
------------------------------------------------------------------------*/
static utBlockRef buildAttrBlock(void)
{
    utBlockRef block = utcBlock();
    cirAttr Attr;
    U16 x, sAttr;

    utaHeapBlock(_cirAttrHeap, block);
    sAttr = UTBLOCKSIZE/sizeof(struct _cirAttr);
    _cirFirstFreeAttr = (cirAttr)utgBlockMem(block);
    Attr = _cirFirstFreeAttr;
    for(x = 0; x < sAttr; x++) {
        _cirAttrNextFree(Attr) = Attr + 1;
        Attr++;
    }
    _cirAttrNextFree(--Attr) = cirAttrNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Attr.
----------------------------------------------------------------------------*/
cirAttr _cirAttrAlloc(void)
{
    cirAttr Attr;

    if(_cirFirstFreeAttr == cirAttrNull) {
        buildAttrBlock();
    }
    Attr = _cirFirstFreeAttr;
    _cirFirstFreeAttr = _cirAttrNextFree(Attr);
    memset((void *)Attr, 0, sizeof(struct _cirAttr));
    return Attr;
}

/*----------------------------------------------------------------------------
  Free a Attr.
----------------------------------------------------------------------------*/
void _cirAttrFree(
    cirAttr Attr)
{
    _cirAttrNextFree(Attr) = _cirFirstFreeAttr;
    _cirFirstFreeAttr = Attr;
}

/*----------------------------------------------------------------------------
  Destroy a Attr.
----------------------------------------------------------------------------*/
void cirAttrDestroy(
    cirAttr Attr)
{
    if(cirAttrDestructorHook != NULL) {
        cirAttrDestructorHook(Attr);
    }
    cirAttrFree(Attr);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Netlist.
------------------------------------------------------------------------*/
static utBlockRef buildNetlistBlock(void)
{
    utBlockRef block = utcBlock();
    cirNetlist Netlist;
    U16 x, sNetlist;

    utaHeapBlock(_cirNetlistHeap, block);
    sNetlist = UTBLOCKSIZE/sizeof(struct _cirNetlist);
    _cirFirstFreeNetlist = (cirNetlist)utgBlockMem(block);
    Netlist = _cirFirstFreeNetlist;
    for(x = 0; x < sNetlist; x++) {
        _cirNetlistNextFree(Netlist) = Netlist + 1;
        Netlist++;
    }
    _cirNetlistNextFree(--Netlist) = cirNetlistNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Netlist.
----------------------------------------------------------------------------*/
cirNetlist cirNetlistAlloc(dbNetlist super)
{
    cirNetlist Netlist;

    if(_cirFirstFreeNetlist == cirNetlistNull) {
        buildNetlistBlock();
    }
    Netlist = _cirFirstFreeNetlist;
    _cirFirstFreeNetlist = _cirNetlistNextFree(Netlist);
    memset((void *)Netlist, 0, sizeof(struct _cirNetlist));
    Netlist->_extension.next = dbNetlistGetfirstExtension(super);
    dbNetlistSetfirstExtension(super, &Netlist->_extension);
    Netlist->_extension.functionTable = &cirNetlistFunctionTable;
    Netlist->_super = super;
    utInitHtbl(&Netlist->Params);
    return Netlist;
}

/*----------------------------------------------------------------------------
  Free a Netlist.
----------------------------------------------------------------------------*/
void cirNetlistFree(
    cirNetlist Netlist)
{
    utdArray(utfHtblArray(cirNetlistGetParams(Netlist)));
    dbNetlistSetfirstExtension(cirNetlistSuper(Netlist), Netlist->_extension.next);
    _cirNetlistNextFree(Netlist) = _cirFirstFreeNetlist;
    _cirFirstFreeNetlist = Netlist;
}

/*----------------------------------------------------------------------------
  Destroy a Netlist.
----------------------------------------------------------------------------*/
void cirNetlistDestroy(
    cirNetlist Netlist)
{
    {
        cirParam _Param;
        cirSafeForeachNetlistParam(Netlist, _Param) {
            cirParamDestroy(_Param);
        } cirEndSafeForeachNetlistParam;
    }
    {
        cirRoot parentRoot = cirNetlistGetRoot(Netlist);
        if(parentRoot != cirRootNull) {
            cirRootRemoveNetlist(parentRoot, Netlist);
        }
    }
    cirNetlistFree(Netlist);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Param.
------------------------------------------------------------------------*/
static utBlockRef buildParamBlock(void)
{
    utBlockRef block = utcBlock();
    cirParam Param;
    U16 x, sParam;

    utaHeapBlock(_cirParamHeap, block);
    sParam = UTBLOCKSIZE/sizeof(struct _cirParam);
    _cirFirstFreeParam = (cirParam)utgBlockMem(block);
    Param = _cirFirstFreeParam;
    for(x = 0; x < sParam; x++) {
        _cirParamNextFree(Param) = Param + 1;
        Param++;
    }
    _cirParamNextFree(--Param) = cirParamNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Param.
----------------------------------------------------------------------------*/
cirParam _cirParamAlloc(void)
{
    cirParam Param;

    if(_cirFirstFreeParam == cirParamNull) {
        buildParamBlock();
    }
    Param = _cirFirstFreeParam;
    _cirFirstFreeParam = _cirParamNextFree(Param);
    memset((void *)Param, 0, sizeof(struct _cirParam));
    return Param;
}

/*----------------------------------------------------------------------------
  Free a Param.
----------------------------------------------------------------------------*/
void _cirParamFree(
    cirParam Param)
{
    _cirParamNextFree(Param) = _cirFirstFreeParam;
    _cirFirstFreeParam = Param;
}

/*----------------------------------------------------------------------------
  Destroy a Param.
----------------------------------------------------------------------------*/
void cirParamDestroy(
    cirParam Param)
{
    if(cirParamDestructorHook != NULL) {
        cirParamDestructorHook(Param);
    }
    cirParamFree(Param);
}

/*----------------------------------------------------------------------------
  Initialize memory for the cir&&.
----------------------------------------------------------------------------*/
void cirDDRStart(void)
{
    _cirFirstFreeDevice = cirDeviceNull;
    _cirDeviceHeap = utcHeap();
    buildDeviceBlock();
    _cirFirstFreePin = cirPinNull;
    _cirPinHeap = utcHeap();
    buildPinBlock();
    _cirFirstFreeRoot = cirRootNull;
    _cirRootHeap = utcHeap();
    buildRootBlock();
    _cirFirstFreeAttr = cirAttrNull;
    _cirAttrHeap = utcHeap();
    buildAttrBlock();
    cirNetlistClassName = utSymCreate("cirNetlist");
    cirNetlistFunctionTable.className = cirNetlistClassName;
    _cirFirstFreeNetlist = cirNetlistNull;
    _cirNetlistHeap = utcHeap();
    buildNetlistBlock();
    _cirFirstFreeParam = cirParamNull;
    _cirParamHeap = utcHeap();
    buildParamBlock();
}

/*----------------------------------------------------------------------------
  Free memory for the cir&&.
----------------------------------------------------------------------------*/
void cirDDRStop(void)
{
    utFreeHeap(_cirDeviceHeap);
    utFreeHeap(_cirPinHeap);
    utFreeHeap(_cirRootHeap);
    utFreeHeap(_cirAttrHeap);
    utFreeHeap(_cirNetlistHeap);
    utFreeHeap(_cirParamHeap);
}

