/*----------------------------------------------------------------------------
  Simple class support routines such as list manipulation.
----------------------------------------------------------------------------*/
#include "schddr.h"

void (*schNetDestructorHook)(schNet);
schNet _schFirstFreeNet, _schTempNet;
utHeapRef _schNetHeap;
void (*schCompDestructorHook)(schComp);
schComp _schFirstFreeComp, _schTempComp;
utHeapRef _schCompHeap;
void (*schMpinDestructorHook)(schMpin);
schMpin _schFirstFreeMpin, _schTempMpin;
utHeapRef _schMpinHeap;
void (*schSchemDestructorHook)(schSchem);
schSchem _schFirstFreeSchem, _schTempSchem;
utHeapRef _schSchemHeap;
void (*schRootDestructorHook)(schRoot);
schRoot _schFirstFreeRoot, _schTempRoot;
utHeapRef _schRootHeap;
void (*schSymbolDestructorHook)(schSymbol);
schSymbol _schFirstFreeSymbol, _schTempSymbol;
utHeapRef _schSymbolHeap;
void (*schPinDestructorHook)(schPin);
schPin _schFirstFreePin, _schTempPin;
utHeapRef _schPinHeap;
void (*schWireDestructorHook)(schWire);
schWire _schFirstFreeWire, _schTempWire;
utHeapRef _schWireHeap;
void (*schAttrDestructorHook)(schAttr);
schAttr _schFirstFreeAttr, _schTempAttr;
utHeapRef _schAttrHeap;
void (*schConnDestructorHook)(schConn);
schConn _schFirstFreeConn, _schTempConn;
utHeapRef _schConnHeap;
void (*schGraphicDestructorHook)(schGraphic);
schGraphic _schFirstFreeGraphic, _schTempGraphic;
utHeapRef _schGraphicHeap;
void (*schLineDestructorHook)(schLine);
schLine _schFirstFreeLine, _schTempLine;
utHeapRef _schLineHeap;
void (*schRectDestructorHook)(schRect);
schRect _schFirstFreeRect, _schTempRect;
utHeapRef _schRectHeap;
void (*schCircleDestructorHook)(schCircle);
schCircle _schFirstFreeCircle, _schTempCircle;
utHeapRef _schCircleHeap;
void (*schArcDestructorHook)(schArc);
schArc _schFirstFreeArc, _schTempArc;
utHeapRef _schArcHeap;
void (*schSignalDestructorHook)(schSignal);
schSignal _schFirstFreeSignal, _schTempSignal;
utHeapRef _schSignalHeap;
void (*schBusDestructorHook)(schBus);
schBus _schFirstFreeBus, _schTempBus;
utHeapRef _schBusHeap;
void (*schColDestructorHook)(schCol);
schCol _schFirstFreeCol, _schTempCol;
utHeapRef _schColHeap;

/*----------------------------------------------------------------------------
  Add the Wire to the head of the list on the Net.
----------------------------------------------------------------------------*/
void schNetInsertWire(
    schNet Net,
    schWire _Wire)
{
    schWireSetnextNetWire(_Wire, schNetGetfirstWire(Net));
    schNetSetfirstWire(Net, _Wire);
    schWireSetNet(_Wire, Net);
}

/*----------------------------------------------------------------------------
  Remove the Wire from the list on the Net.
----------------------------------------------------------------------------*/
void schNetRemoveWire(
    schNet Net,
    schWire _Wire)
{
    schWire nextWire, prevWire = schWireNull;

    for(nextWire = schNetGetfirstWire(Net);
        nextWire != schWireNull && nextWire != _Wire;
        nextWire = schWireGetnextNetWire(nextWire)) {
        prevWire = nextWire;
    }
    if(nextWire == schWireNull) {
        /* Not in list */
        return;
    }
    if(prevWire != schWireNull) {
        schWireSetnextNetWire(prevWire, schWireGetnextNetWire(_Wire));
    } else {
        schNetSetfirstWire(Net, schWireGetnextNetWire(_Wire));
    }
    schWireSetnextNetWire(_Wire, schWireNull);
    schWireSetNet(_Wire, schNetNull);
}

/*----------------------------------------------------------------------------
  Add the Pin to the head of the list on the Net.
----------------------------------------------------------------------------*/
void schNetInsertPin(
    schNet Net,
    schPin _Pin)
{
    schPinSetnextNetPin(_Pin, schNetGetfirstPin(Net));
    schNetSetfirstPin(Net, _Pin);
    if(schNetGetlastPin(Net) == schPinNull) {
        schNetSetlastPin(Net, _Pin);
    }
    schPinSetNet(_Pin, Net);
}

/*----------------------------------------------------------------------------
  Remove the Pin from the list on the Net.
----------------------------------------------------------------------------*/
void schNetRemovePin(
    schNet Net,
    schPin _Pin)
{
    schPin nextPin, prevPin = schPinNull;

    for(nextPin = schNetGetfirstPin(Net);
        nextPin != schPinNull && nextPin != _Pin;
        nextPin = schPinGetnextNetPin(nextPin)) {
        prevPin = nextPin;
    }
    if(nextPin == schPinNull) {
        /* Not in list */
        return;
    }
    if(prevPin != schPinNull) {
        schPinSetnextNetPin(prevPin, schPinGetnextNetPin(_Pin));
    } else {
        schNetSetfirstPin(Net, schPinGetnextNetPin(_Pin));
    }
    if(schNetGetlastPin(Net) == _Pin) {
        schNetSetlastPin(Net, prevPin);
    }
    schPinSetnextNetPin(_Pin, schPinNull);
    schPinSetNet(_Pin, schNetNull);
}

/*----------------------------------------------------------------------------
  Add the Pin to the tail of the list on the Net.
----------------------------------------------------------------------------*/
void schNetAppendPin(
    schNet Net,
    schPin _Pin)
{
    if(schNetGetlastPin(Net) == schPinNull) {
        schNetSetfirstPin(Net, _Pin);
    } else {
        schPinSetnextNetPin(schNetGetlastPin(Net), _Pin);
    }
    schNetSetlastPin(Net, _Pin);
    schPinSetNet(_Pin, Net);
}

/*----------------------------------------------------------------------------
  Add the Signal to the head of the list on the Net.
----------------------------------------------------------------------------*/
void schNetInsertSignal(
    schNet Net,
    schSignal _Signal)
{
    schSignalSetnextNetSignal(_Signal, schNetGetfirstSignal(Net));
    schNetSetfirstSignal(Net, _Signal);
    if(schNetGetlastSignal(Net) == schSignalNull) {
        schNetSetlastSignal(Net, _Signal);
    }
    schSignalSetNet(_Signal, Net);
}

/*----------------------------------------------------------------------------
  Remove the Signal from the list on the Net.
----------------------------------------------------------------------------*/
void schNetRemoveSignal(
    schNet Net,
    schSignal _Signal)
{
    schSignal nextSignal, prevSignal = schSignalNull;

    for(nextSignal = schNetGetfirstSignal(Net);
        nextSignal != schSignalNull && nextSignal != _Signal;
        nextSignal = schSignalGetnextNetSignal(nextSignal)) {
        prevSignal = nextSignal;
    }
    if(nextSignal == schSignalNull) {
        /* Not in list */
        return;
    }
    if(prevSignal != schSignalNull) {
        schSignalSetnextNetSignal(prevSignal, schSignalGetnextNetSignal(_Signal));
    } else {
        schNetSetfirstSignal(Net, schSignalGetnextNetSignal(_Signal));
    }
    if(schNetGetlastSignal(Net) == _Signal) {
        schNetSetlastSignal(Net, prevSignal);
    }
    schSignalSetnextNetSignal(_Signal, schSignalNull);
    schSignalSetNet(_Signal, schNetNull);
}

/*----------------------------------------------------------------------------
  Add the Signal to the tail of the list on the Net.
----------------------------------------------------------------------------*/
void schNetAppendSignal(
    schNet Net,
    schSignal _Signal)
{
    if(schNetGetlastSignal(Net) == schSignalNull) {
        schNetSetfirstSignal(Net, _Signal);
    } else {
        schSignalSetnextNetSignal(schNetGetlastSignal(Net), _Signal);
    }
    schNetSetlastSignal(Net, _Signal);
    schSignalSetNet(_Signal, Net);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Net.
------------------------------------------------------------------------*/
static utBlockRef buildNetBlock(void)
{
    utBlockRef block = utcBlock();
    schNet Net;
    U16 x, sNet;

    utaHeapBlock(_schNetHeap, block);
    sNet = UTBLOCKSIZE/sizeof(struct _schNet);
    _schFirstFreeNet = (schNet)utgBlockMem(block);
    Net = _schFirstFreeNet;
    for(x = 0; x < sNet; x++) {
        _schNetNextFree(Net) = Net + 1;
        Net++;
    }
    _schNetNextFree(--Net) = schNetNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Net.
----------------------------------------------------------------------------*/
schNet _schNetAlloc(void)
{
    schNet Net;

    if(_schFirstFreeNet == schNetNull) {
        buildNetBlock();
    }
    Net = _schFirstFreeNet;
    _schFirstFreeNet = _schNetNextFree(Net);
    memset((void *)Net, 0, sizeof(struct _schNet));
    return Net;
}

/*----------------------------------------------------------------------------
  Free a Net.
----------------------------------------------------------------------------*/
void _schNetFree(
    schNet Net)
{
    _schNetNextFree(Net) = _schFirstFreeNet;
    _schFirstFreeNet = Net;
}

/*----------------------------------------------------------------------------
  Destroy a Net.
----------------------------------------------------------------------------*/
void schNetDestroy(
    schNet Net)
{
    if(schNetDestructorHook != NULL) {
        schNetDestructorHook(Net);
    }
    {
        schWire _Wire;
        schSafeForeachNetWire(Net, _Wire) {
            schWireDestroy(_Wire);
        } schEndSafeForeachNetWire;
    }
    {
        schPin _Pin;
        schSafeForeachNetPin(Net, _Pin) {
            schNetRemovePin(Net, _Pin);
        } schEndSafeForeachNetPin;
    }
    {
        schSignal _Signal;
        schSafeForeachNetSignal(Net, _Signal) {
            schSignalDestroy(_Signal);
        } schEndSafeForeachNetSignal;
    }
    if(schNetGetAttr(Net) != schAttrNull) {
        schAttrDestroy(schNetGetAttr(Net));
    }
    {
        schSchem parentSchem = schNetGetSchem(Net);
        if(parentSchem != schSchemNull) {
            schSchemRemoveNet(parentSchem, Net);
        }
    }
    schNetFree(Net);
}

/*----------------------------------------------------------------------------
  Add the Pin to the head of the list on the Comp.
----------------------------------------------------------------------------*/
void schCompInsertPin(
    schComp Comp,
    schPin _Pin)
{
    schPinSetnextCompPin(_Pin, schCompGetfirstPin(Comp));
    schCompSetfirstPin(Comp, _Pin);
    if(schCompGetlastPin(Comp) == schPinNull) {
        schCompSetlastPin(Comp, _Pin);
    }
    schPinSetComp(_Pin, Comp);
}

/*----------------------------------------------------------------------------
  Remove the Pin from the list on the Comp.
----------------------------------------------------------------------------*/
void schCompRemovePin(
    schComp Comp,
    schPin _Pin)
{
    schPin nextPin, prevPin = schPinNull;

    for(nextPin = schCompGetfirstPin(Comp);
        nextPin != schPinNull && nextPin != _Pin;
        nextPin = schPinGetnextCompPin(nextPin)) {
        prevPin = nextPin;
    }
    if(nextPin == schPinNull) {
        /* Not in list */
        return;
    }
    if(prevPin != schPinNull) {
        schPinSetnextCompPin(prevPin, schPinGetnextCompPin(_Pin));
    } else {
        schCompSetfirstPin(Comp, schPinGetnextCompPin(_Pin));
    }
    if(schCompGetlastPin(Comp) == _Pin) {
        schCompSetlastPin(Comp, prevPin);
    }
    schPinSetnextCompPin(_Pin, schPinNull);
    schPinSetComp(_Pin, schCompNull);
}

/*----------------------------------------------------------------------------
  Add the Pin to the tail of the list on the Comp.
----------------------------------------------------------------------------*/
void schCompAppendPin(
    schComp Comp,
    schPin _Pin)
{
    if(schCompGetlastPin(Comp) == schPinNull) {
        schCompSetfirstPin(Comp, _Pin);
    } else {
        schPinSetnextCompPin(schCompGetlastPin(Comp), _Pin);
    }
    schCompSetlastPin(Comp, _Pin);
    schPinSetComp(_Pin, Comp);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Comp.
------------------------------------------------------------------------*/
static utBlockRef buildCompBlock(void)
{
    utBlockRef block = utcBlock();
    schComp Comp;
    U16 x, sComp;

    utaHeapBlock(_schCompHeap, block);
    sComp = UTBLOCKSIZE/sizeof(struct _schComp);
    _schFirstFreeComp = (schComp)utgBlockMem(block);
    Comp = _schFirstFreeComp;
    for(x = 0; x < sComp; x++) {
        _schCompNextFree(Comp) = Comp + 1;
        Comp++;
    }
    _schCompNextFree(--Comp) = schCompNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Comp.
----------------------------------------------------------------------------*/
schComp _schCompAlloc(void)
{
    schComp Comp;

    if(_schFirstFreeComp == schCompNull) {
        buildCompBlock();
    }
    Comp = _schFirstFreeComp;
    _schFirstFreeComp = _schCompNextFree(Comp);
    memset((void *)Comp, 0, sizeof(struct _schComp));
    return Comp;
}

/*----------------------------------------------------------------------------
  Free a Comp.
----------------------------------------------------------------------------*/
void _schCompFree(
    schComp Comp)
{
    _schCompNextFree(Comp) = _schFirstFreeComp;
    _schFirstFreeComp = Comp;
}

/*----------------------------------------------------------------------------
  Destroy a Comp.
----------------------------------------------------------------------------*/
void schCompDestroy(
    schComp Comp)
{
    if(schCompDestructorHook != NULL) {
        schCompDestructorHook(Comp);
    }
    {
        schPin _Pin;
        schSafeForeachCompPin(Comp, _Pin) {
            schPinDestroy(_Pin);
        } schEndSafeForeachCompPin;
    }
    if(schCompGetAttr(Comp) != schAttrNull) {
        schAttrDestroy(schCompGetAttr(Comp));
    }
    {
        schSchem parentSchem = schCompGetSchem(Comp);
        if(parentSchem != schSchemNull) {
            schSchemRemoveComp(parentSchem, Comp);
        }
    }
    {
        schSymbol parentSymbol = schCompGetSymbol(Comp);
        if(parentSymbol != schSymbolNull) {
            schSymbolRemoveComp(parentSymbol, Comp);
        }
    }
    {
        schMpin parentMpin = schCompGetMpin(Comp);
        if(parentMpin != schMpinNull) {
            schMpinSetFlagComp(parentMpin, schCompNull);
        }
    }
    {
        schCol parentCol = schCompGetCol(Comp);
        if(parentCol != schColNull) {
            schColRemoveComp(parentCol, Comp);
        }
    }
    schCompFree(Comp);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Mpin.
------------------------------------------------------------------------*/
static utBlockRef buildMpinBlock(void)
{
    utBlockRef block = utcBlock();
    schMpin Mpin;
    U16 x, sMpin;

    utaHeapBlock(_schMpinHeap, block);
    sMpin = UTBLOCKSIZE/sizeof(struct _schMpin);
    _schFirstFreeMpin = (schMpin)utgBlockMem(block);
    Mpin = _schFirstFreeMpin;
    for(x = 0; x < sMpin; x++) {
        _schMpinNextFree(Mpin) = Mpin + 1;
        Mpin++;
    }
    _schMpinNextFree(--Mpin) = schMpinNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Mpin.
----------------------------------------------------------------------------*/
schMpin _schMpinAlloc(void)
{
    schMpin Mpin;

    if(_schFirstFreeMpin == schMpinNull) {
        buildMpinBlock();
    }
    Mpin = _schFirstFreeMpin;
    _schFirstFreeMpin = _schMpinNextFree(Mpin);
    memset((void *)Mpin, 0, sizeof(struct _schMpin));
    return Mpin;
}

/*----------------------------------------------------------------------------
  Free a Mpin.
----------------------------------------------------------------------------*/
void _schMpinFree(
    schMpin Mpin)
{
    _schMpinNextFree(Mpin) = _schFirstFreeMpin;
    _schFirstFreeMpin = Mpin;
}

/*----------------------------------------------------------------------------
  Destroy a Mpin.
----------------------------------------------------------------------------*/
void schMpinDestroy(
    schMpin Mpin)
{
    if(schMpinDestructorHook != NULL) {
        schMpinDestructorHook(Mpin);
    }
    if(schMpinGetFlagComp(Mpin) != schCompNull) {
        schCompSetMpin(schMpinGetFlagComp(Mpin), schMpinNull);
    }
    if(schMpinGetAttr(Mpin) != schAttrNull) {
        schAttrDestroy(schMpinGetAttr(Mpin));
    }
    {
        schSymbol parentSymbol = schMpinGetSymbol(Mpin);
        if(parentSymbol != schSymbolNull) {
            schSymbolRemoveMpin(parentSymbol, Mpin);
        }
    }
    schMpinFree(Mpin);
}

/*----------------------------------------------------------------------------
  Add the Comp to the tail of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemInsertComp(
    schSchem Schem,
    schComp _Comp)
{
    utInsertHtblEntry(schSchemGetComps(Schem), &(_Comp)->h);
    schCompSetSchem(_Comp, Schem);
}

/*----------------------------------------------------------------------------
  Remove the Comp from the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemRemoveComp(
    schSchem Schem,
    schComp _Comp)
{
    utDeleteHtblEntry(&(Schem)->Comps, &(_Comp)->h);
    schCompSetSchem(_Comp, schSchemNull);
}

/*----------------------------------------------------------------------------
  Append the Comp to the tail of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemAppendComp(
    schSchem Schem,
    schComp _Comp)
{
    utAppendHtblEntry(schSchemGetComps(Schem), &(_Comp)->h);
    schCompSetSchem(_Comp, Schem);
}

/*----------------------------------------------------------------------------
  Add the Net to the tail of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemInsertNet(
    schSchem Schem,
    schNet _Net)
{
    utInsertHtblEntry(schSchemGetNets(Schem), &(_Net)->h);
    schNetSetSchem(_Net, Schem);
}

/*----------------------------------------------------------------------------
  Remove the Net from the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemRemoveNet(
    schSchem Schem,
    schNet _Net)
{
    utDeleteHtblEntry(&(Schem)->Nets, &(_Net)->h);
    schNetSetSchem(_Net, schSchemNull);
}

/*----------------------------------------------------------------------------
  Append the Net to the tail of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemAppendNet(
    schSchem Schem,
    schNet _Net)
{
    utAppendHtblEntry(schSchemGetNets(Schem), &(_Net)->h);
    schNetSetSchem(_Net, Schem);
}

/*----------------------------------------------------------------------------
  Add the Wire to the head of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemInsertWire(
    schSchem Schem,
    schWire _Wire)
{
    schWireSetnextSchemWire(_Wire, schSchemGetfirstWire(Schem));
    schSchemSetfirstWire(Schem, _Wire);
    if(schSchemGetlastWire(Schem) == schWireNull) {
        schSchemSetlastWire(Schem, _Wire);
    } else{
        schWireSetprevSchemWire(schWireGetnextSchemWire(_Wire), _Wire);
    }
    schWireSetSchem(_Wire, Schem);
}

/*----------------------------------------------------------------------------
  Remove the Wire from the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemRemoveWire(
    schSchem Schem,
    schWire _Wire)
{
    schWire nextWire = schWireGetnextSchemWire(_Wire);
    schWire prevWire = schWireGetprevSchemWire(_Wire);

    if(prevWire == schWireNull && schSchemGetfirstWire(Schem) != _Wire) {
        /* Not in list */
        return;
    }
    if(prevWire != schWireNull) {
        schWireSetnextSchemWire(prevWire, nextWire);
    } else {
        schSchemSetfirstWire(Schem, nextWire);
    }
    if(nextWire != schWireNull) {
        schWireSetprevSchemWire(nextWire, prevWire);
    } else {
        schSchemSetlastWire(Schem, prevWire);
    }
    schWireSetprevSchemWire(_Wire, schWireNull);
    schWireSetnextSchemWire(_Wire, schWireNull);
    schWireSetSchem(_Wire, schSchemNull);
}

/*----------------------------------------------------------------------------
  Add the Wire to the tail of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemAppendWire(
    schSchem Schem,
    schWire _Wire)
{
    if(schSchemGetlastWire(Schem) == schWireNull) {
        schSchemSetfirstWire(Schem, _Wire);
    } else {
        schWireSetnextSchemWire(schSchemGetlastWire(Schem), _Wire);
        schWireSetprevSchemWire(_Wire, schSchemGetlastWire(Schem));
    }
    schSchemSetlastWire(Schem, _Wire);
    schWireSetSchem(_Wire, Schem);
}

/*----------------------------------------------------------------------------
  Add the Bus to the tail of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemInsertBus(
    schSchem Schem,
    schBus _Bus)
{
    utInsertHtblEntry(schSchemGetBuss(Schem), &(_Bus)->h);
    schBusSetSchem(_Bus, Schem);
}

/*----------------------------------------------------------------------------
  Remove the Bus from the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemRemoveBus(
    schSchem Schem,
    schBus _Bus)
{
    utDeleteHtblEntry(&(Schem)->Buss, &(_Bus)->h);
    schBusSetSchem(_Bus, schSchemNull);
}

/*----------------------------------------------------------------------------
  Append the Bus to the tail of the list on the Schem.
----------------------------------------------------------------------------*/
void schSchemAppendBus(
    schSchem Schem,
    schBus _Bus)
{
    utAppendHtblEntry(schSchemGetBuss(Schem), &(_Bus)->h);
    schBusSetSchem(_Bus, Schem);
}

/*----------------------------------------------------------------------------
  Add the Col to the end of the array on the Schem.
----------------------------------------------------------------------------*/
void schSchemInsertCol(
    schSchem Schem,
    schCol _Col)
{
    if(schSchemGetusedCols(Schem) == schSchemGetnumCols(Schem)) {
        schSchemReallocCols(Schem, schSchemGetnumCols(Schem) +
            (schSchemGetnumCols(Schem) >> 1) + 1);
    }
    schColSetSchemIndex(_Col, schSchemGetusedCols(Schem));
    schSchemSetiCol(Schem, schSchemGetusedCols(Schem)++, _Col);
    schColSetSchem(_Col, Schem);
}

/*----------------------------------------------------------------------------
  Remove the Col from the array on the Schem.
----------------------------------------------------------------------------*/
void schSchemRemoveCol(
    schSchem Schem,
    schCol _Col)
{
    U32 xCol = schColGetSchemIndex(_Col);

    if(xCol != U32_MAX) {
        schSchemSetiCol(Schem, xCol, schSchemGetiCol(Schem,
            --schSchemGetusedCols(Schem)));
        schColSetSchemIndex(_Col, U32_MAX);
    }
    schColSetSchem(_Col, schSchemNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Schem.
------------------------------------------------------------------------*/
static utBlockRef buildSchemBlock(void)
{
    utBlockRef block = utcBlock();
    schSchem Schem;
    U16 x, sSchem;

    utaHeapBlock(_schSchemHeap, block);
    sSchem = UTBLOCKSIZE/sizeof(struct _schSchem);
    _schFirstFreeSchem = (schSchem)utgBlockMem(block);
    Schem = _schFirstFreeSchem;
    for(x = 0; x < sSchem; x++) {
        _schSchemNextFree(Schem) = Schem + 1;
        Schem++;
    }
    _schSchemNextFree(--Schem) = schSchemNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Schem.
----------------------------------------------------------------------------*/
schSchem schSchemAlloc(void)
{
    schSchem Schem;

    if(_schFirstFreeSchem == schSchemNull) {
        buildSchemBlock();
    }
    Schem = _schFirstFreeSchem;
    _schFirstFreeSchem = _schSchemNextFree(Schem);
    memset((void *)Schem, 0, sizeof(struct _schSchem));
    utInitHtbl(&Schem->Comps);
    utInitHtbl(&Schem->Nets);
    utInitHtbl(&Schem->Buss);
    return Schem;
}

/*----------------------------------------------------------------------------
  Free a Schem.
----------------------------------------------------------------------------*/
void schSchemFree(
    schSchem Schem)
{
    utdArray(utfHtblArray(schSchemGetComps(Schem)));
    utdArray(utfHtblArray(schSchemGetNets(Schem)));
    utdArray(utfHtblArray(schSchemGetBuss(Schem)));
    if(schSchemGetnumCols(Schem) != 0) {
        schSchemFreeCols(Schem);
    }
    _schSchemNextFree(Schem) = _schFirstFreeSchem;
    _schFirstFreeSchem = Schem;
}

/*----------------------------------------------------------------------------
  Destroy a Schem.
----------------------------------------------------------------------------*/
void schSchemDestroy(
    schSchem Schem)
{
    if(schSchemDestructorHook != NULL) {
        schSchemDestructorHook(Schem);
    }
    {
        schComp _Comp;
        schSafeForeachSchemComp(Schem, _Comp) {
            schCompDestroy(_Comp);
        } schEndSafeForeachSchemComp;
    }
    {
        schNet _Net;
        schSafeForeachSchemNet(Schem, _Net) {
            schNetDestroy(_Net);
        } schEndSafeForeachSchemNet;
    }
    {
        schWire _Wire;
        schSafeForeachSchemWire(Schem, _Wire) {
            schWireDestroy(_Wire);
        } schEndSafeForeachSchemWire;
    }
    {
        schBus _Bus;
        schSafeForeachSchemBus(Schem, _Bus) {
            schBusDestroy(_Bus);
        } schEndSafeForeachSchemBus;
    }
    if(schSchemGetAttr(Schem) != schAttrNull) {
        schAttrDestroy(schSchemGetAttr(Schem));
    }
    {
        schCol _Col;
        schSafeForeachSchemCol(Schem, _Col) {
            schSchemRemoveCol(Schem, _Col);
        } schEndSafeForeachSchemCol;
    }
    {
        schRoot parentRoot = schSchemGetRoot(Schem);
        if(parentRoot != schRootNull) {
            schRootRemoveSchem(parentRoot, Schem);
        }
    }
    {
        schSymbol parentSymbol = schSchemGetSymbol(Schem);
        if(parentSymbol != schSymbolNull) {
            schSymbolSetSchem(parentSymbol, schSchemNull);
        }
    }
    schSchemFree(Schem);
}

/*----------------------------------------------------------------------------
  Add the Schem to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void schRootInsertSchem(
    schRoot Root,
    schSchem _Schem)
{
    utInsertHtblEntry(schRootGetSchems(Root), &(_Schem)->h);
    schSchemSetRoot(_Schem, Root);
}

/*----------------------------------------------------------------------------
  Remove the Schem from the list on the Root.
----------------------------------------------------------------------------*/
void schRootRemoveSchem(
    schRoot Root,
    schSchem _Schem)
{
    utDeleteHtblEntry(&(Root)->Schems, &(_Schem)->h);
    schSchemSetRoot(_Schem, schRootNull);
}

/*----------------------------------------------------------------------------
  Append the Schem to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void schRootAppendSchem(
    schRoot Root,
    schSchem _Schem)
{
    utAppendHtblEntry(schRootGetSchems(Root), &(_Schem)->h);
    schSchemSetRoot(_Schem, Root);
}

/*----------------------------------------------------------------------------
  Add the Symbol to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void schRootInsertSymbol(
    schRoot Root,
    schSymbol _Symbol)
{
    utInsertHtblEntry(schRootGetSymbols(Root), &(_Symbol)->h);
    schSymbolSetRoot(_Symbol, Root);
}

/*----------------------------------------------------------------------------
  Remove the Symbol from the list on the Root.
----------------------------------------------------------------------------*/
void schRootRemoveSymbol(
    schRoot Root,
    schSymbol _Symbol)
{
    utDeleteHtblEntry(&(Root)->Symbols, &(_Symbol)->h);
    schSymbolSetRoot(_Symbol, schRootNull);
}

/*----------------------------------------------------------------------------
  Append the Symbol to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void schRootAppendSymbol(
    schRoot Root,
    schSymbol _Symbol)
{
    utAppendHtblEntry(schRootGetSymbols(Root), &(_Symbol)->h);
    schSymbolSetRoot(_Symbol, Root);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Root.
------------------------------------------------------------------------*/
static utBlockRef buildRootBlock(void)
{
    utBlockRef block = utcBlock();
    schRoot Root;
    U16 x, sRoot;

    utaHeapBlock(_schRootHeap, block);
    sRoot = UTBLOCKSIZE/sizeof(struct _schRoot);
    _schFirstFreeRoot = (schRoot)utgBlockMem(block);
    Root = _schFirstFreeRoot;
    for(x = 0; x < sRoot; x++) {
        _schRootNextFree(Root) = Root + 1;
        Root++;
    }
    _schRootNextFree(--Root) = schRootNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Root.
----------------------------------------------------------------------------*/
schRoot schRootAlloc(void)
{
    schRoot Root;

    if(_schFirstFreeRoot == schRootNull) {
        buildRootBlock();
    }
    Root = _schFirstFreeRoot;
    _schFirstFreeRoot = _schRootNextFree(Root);
    memset((void *)Root, 0, sizeof(struct _schRoot));
    utInitHtbl(&Root->Schems);
    utInitHtbl(&Root->Symbols);
    return Root;
}

/*----------------------------------------------------------------------------
  Free a Root.
----------------------------------------------------------------------------*/
void schRootFree(
    schRoot Root)
{
    utdArray(utfHtblArray(schRootGetSchems(Root)));
    utdArray(utfHtblArray(schRootGetSymbols(Root)));
    _schRootNextFree(Root) = _schFirstFreeRoot;
    _schFirstFreeRoot = Root;
}

/*----------------------------------------------------------------------------
  Destroy a Root.
----------------------------------------------------------------------------*/
void schRootDestroy(
    schRoot Root)
{
    if(schRootDestructorHook != NULL) {
        schRootDestructorHook(Root);
    }
    {
        schSchem _Schem;
        schSafeForeachRootSchem(Root, _Schem) {
            schSchemDestroy(_Schem);
        } schEndSafeForeachRootSchem;
    }
    {
        schSymbol _Symbol;
        schSafeForeachRootSymbol(Root, _Symbol) {
            schSymbolDestroy(_Symbol);
        } schEndSafeForeachRootSymbol;
    }
    schRootFree(Root);
}

/*----------------------------------------------------------------------------
  Add the Mpin to the tail of the list on the Symbol.
----------------------------------------------------------------------------*/
void schSymbolInsertMpin(
    schSymbol Symbol,
    schMpin _Mpin)
{
    utInsertHtblEntry(schSymbolGetMpins(Symbol), &(_Mpin)->h);
    schMpinSetSymbol(_Mpin, Symbol);
}

/*----------------------------------------------------------------------------
  Remove the Mpin from the list on the Symbol.
----------------------------------------------------------------------------*/
void schSymbolRemoveMpin(
    schSymbol Symbol,
    schMpin _Mpin)
{
    utDeleteHtblEntry(&(Symbol)->Mpins, &(_Mpin)->h);
    schMpinSetSymbol(_Mpin, schSymbolNull);
}

/*----------------------------------------------------------------------------
  Append the Mpin to the tail of the list on the Symbol.
----------------------------------------------------------------------------*/
void schSymbolAppendMpin(
    schSymbol Symbol,
    schMpin _Mpin)
{
    utAppendHtblEntry(schSymbolGetMpins(Symbol), &(_Mpin)->h);
    schMpinSetSymbol(_Mpin, Symbol);
}

/*----------------------------------------------------------------------------
  Add the Comp to the head of the list on the Symbol.
----------------------------------------------------------------------------*/
void schSymbolInsertComp(
    schSymbol Symbol,
    schComp _Comp)
{
    schCompSetnextSymbolComp(_Comp, schSymbolGetfirstComp(Symbol));
    schSymbolSetfirstComp(Symbol, _Comp);
    if(schSymbolGetlastComp(Symbol) == schCompNull) {
        schSymbolSetlastComp(Symbol, _Comp);
    } else{
        schCompSetprevSymbolComp(schCompGetnextSymbolComp(_Comp), _Comp);
    }
    schCompSetSymbol(_Comp, Symbol);
}

/*----------------------------------------------------------------------------
  Remove the Comp from the list on the Symbol.
----------------------------------------------------------------------------*/
void schSymbolRemoveComp(
    schSymbol Symbol,
    schComp _Comp)
{
    schComp nextComp = schCompGetnextSymbolComp(_Comp);
    schComp prevComp = schCompGetprevSymbolComp(_Comp);

    if(prevComp == schCompNull && schSymbolGetfirstComp(Symbol) != _Comp) {
        /* Not in list */
        return;
    }
    if(prevComp != schCompNull) {
        schCompSetnextSymbolComp(prevComp, nextComp);
    } else {
        schSymbolSetfirstComp(Symbol, nextComp);
    }
    if(nextComp != schCompNull) {
        schCompSetprevSymbolComp(nextComp, prevComp);
    } else {
        schSymbolSetlastComp(Symbol, prevComp);
    }
    schCompSetprevSymbolComp(_Comp, schCompNull);
    schCompSetnextSymbolComp(_Comp, schCompNull);
    schCompSetSymbol(_Comp, schSymbolNull);
}

/*----------------------------------------------------------------------------
  Add the Comp to the tail of the list on the Symbol.
----------------------------------------------------------------------------*/
void schSymbolAppendComp(
    schSymbol Symbol,
    schComp _Comp)
{
    if(schSymbolGetlastComp(Symbol) == schCompNull) {
        schSymbolSetfirstComp(Symbol, _Comp);
    } else {
        schCompSetnextSymbolComp(schSymbolGetlastComp(Symbol), _Comp);
        schCompSetprevSymbolComp(_Comp, schSymbolGetlastComp(Symbol));
    }
    schSymbolSetlastComp(Symbol, _Comp);
    schCompSetSymbol(_Comp, Symbol);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Symbol.
------------------------------------------------------------------------*/
static utBlockRef buildSymbolBlock(void)
{
    utBlockRef block = utcBlock();
    schSymbol Symbol;
    U16 x, sSymbol;

    utaHeapBlock(_schSymbolHeap, block);
    sSymbol = UTBLOCKSIZE/sizeof(struct _schSymbol);
    _schFirstFreeSymbol = (schSymbol)utgBlockMem(block);
    Symbol = _schFirstFreeSymbol;
    for(x = 0; x < sSymbol; x++) {
        _schSymbolNextFree(Symbol) = Symbol + 1;
        Symbol++;
    }
    _schSymbolNextFree(--Symbol) = schSymbolNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Symbol.
----------------------------------------------------------------------------*/
schSymbol schSymbolAlloc(void)
{
    schSymbol Symbol;

    if(_schFirstFreeSymbol == schSymbolNull) {
        buildSymbolBlock();
    }
    Symbol = _schFirstFreeSymbol;
    _schFirstFreeSymbol = _schSymbolNextFree(Symbol);
    memset((void *)Symbol, 0, sizeof(struct _schSymbol));
    utInitHtbl(&Symbol->Mpins);
    return Symbol;
}

/*----------------------------------------------------------------------------
  Free a Symbol.
----------------------------------------------------------------------------*/
void schSymbolFree(
    schSymbol Symbol)
{
    utdArray(utfHtblArray(schSymbolGetMpins(Symbol)));
    _schSymbolNextFree(Symbol) = _schFirstFreeSymbol;
    _schFirstFreeSymbol = Symbol;
}

/*----------------------------------------------------------------------------
  Destroy a Symbol.
----------------------------------------------------------------------------*/
void schSymbolDestroy(
    schSymbol Symbol)
{
    if(schSymbolDestructorHook != NULL) {
        schSymbolDestructorHook(Symbol);
    }
    if(schSymbolGetSchem(Symbol) != schSchemNull) {
        schSchemSetSymbol(schSymbolGetSchem(Symbol), schSymbolNull);
    }
    {
        schMpin _Mpin;
        schSafeForeachSymbolMpin(Symbol, _Mpin) {
            schMpinDestroy(_Mpin);
        } schEndSafeForeachSymbolMpin;
    }
    {
        schComp _Comp;
        schSafeForeachSymbolComp(Symbol, _Comp) {
            schCompDestroy(_Comp);
        } schEndSafeForeachSymbolComp;
    }
    if(schSymbolGetAttr(Symbol) != schAttrNull) {
        schAttrDestroy(schSymbolGetAttr(Symbol));
    }
    if(schSymbolGetGraphic(Symbol) != schGraphicNull) {
        schGraphicDestroy(schSymbolGetGraphic(Symbol));
    }
    {
        schRoot parentRoot = schSymbolGetRoot(Symbol);
        if(parentRoot != schRootNull) {
            schRootRemoveSymbol(parentRoot, Symbol);
        }
    }
    schSymbolFree(Symbol);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Pin.
------------------------------------------------------------------------*/
static utBlockRef buildPinBlock(void)
{
    utBlockRef block = utcBlock();
    schPin Pin;
    U16 x, sPin;

    utaHeapBlock(_schPinHeap, block);
    sPin = UTBLOCKSIZE/sizeof(struct _schPin);
    _schFirstFreePin = (schPin)utgBlockMem(block);
    Pin = _schFirstFreePin;
    for(x = 0; x < sPin; x++) {
        _schPinNextFree(Pin) = Pin + 1;
        Pin++;
    }
    _schPinNextFree(--Pin) = schPinNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Pin.
----------------------------------------------------------------------------*/
schPin _schPinAlloc(void)
{
    schPin Pin;

    if(_schFirstFreePin == schPinNull) {
        buildPinBlock();
    }
    Pin = _schFirstFreePin;
    _schFirstFreePin = _schPinNextFree(Pin);
    memset((void *)Pin, 0, sizeof(struct _schPin));
    return Pin;
}

/*----------------------------------------------------------------------------
  Free a Pin.
----------------------------------------------------------------------------*/
void _schPinFree(
    schPin Pin)
{
    _schPinNextFree(Pin) = _schFirstFreePin;
    _schFirstFreePin = Pin;
}

/*----------------------------------------------------------------------------
  Destroy a Pin.
----------------------------------------------------------------------------*/
void schPinDestroy(
    schPin Pin)
{
    if(schPinDestructorHook != NULL) {
        schPinDestructorHook(Pin);
    }
    {
        schComp parentComp = schPinGetComp(Pin);
        if(parentComp != schCompNull) {
            schCompRemovePin(parentComp, Pin);
        }
    }
    {
        schNet parentNet = schPinGetNet(Pin);
        if(parentNet != schNetNull) {
            schNetRemovePin(parentNet, Pin);
        }
    }
    schPinFree(Pin);
}

/*----------------------------------------------------------------------------
  Add the RConn to the head of the list on the Wire.
----------------------------------------------------------------------------*/
void schWireInsertRConn(
    schWire Wire,
    schConn _Conn)
{
    schConnSetnextWireRConn(_Conn, schWireGetfirstRConn(Wire));
    schWireSetfirstRConn(Wire, _Conn);
    schConnSetRWire(_Conn, Wire);
}

/*----------------------------------------------------------------------------
  Remove the RConn from the list on the Wire.
----------------------------------------------------------------------------*/
void schWireRemoveRConn(
    schWire Wire,
    schConn _Conn)
{
    schConn nextConn, prevConn = schConnNull;

    for(nextConn = schWireGetfirstRConn(Wire);
        nextConn != schConnNull && nextConn != _Conn;
        nextConn = schConnGetnextWireRConn(nextConn)) {
        prevConn = nextConn;
    }
    if(nextConn == schConnNull) {
        /* Not in list */
        return;
    }
    if(prevConn != schConnNull) {
        schConnSetnextWireRConn(prevConn, schConnGetnextWireRConn(_Conn));
    } else {
        schWireSetfirstRConn(Wire, schConnGetnextWireRConn(_Conn));
    }
    schConnSetnextWireRConn(_Conn, schConnNull);
    schConnSetRWire(_Conn, schWireNull);
}

/*----------------------------------------------------------------------------
  Add the LConn to the head of the list on the Wire.
----------------------------------------------------------------------------*/
void schWireInsertLConn(
    schWire Wire,
    schConn _Conn)
{
    schConnSetnextWireLConn(_Conn, schWireGetfirstLConn(Wire));
    schWireSetfirstLConn(Wire, _Conn);
    schConnSetLWire(_Conn, Wire);
}

/*----------------------------------------------------------------------------
  Remove the LConn from the list on the Wire.
----------------------------------------------------------------------------*/
void schWireRemoveLConn(
    schWire Wire,
    schConn _Conn)
{
    schConn nextConn, prevConn = schConnNull;

    for(nextConn = schWireGetfirstLConn(Wire);
        nextConn != schConnNull && nextConn != _Conn;
        nextConn = schConnGetnextWireLConn(nextConn)) {
        prevConn = nextConn;
    }
    if(nextConn == schConnNull) {
        /* Not in list */
        return;
    }
    if(prevConn != schConnNull) {
        schConnSetnextWireLConn(prevConn, schConnGetnextWireLConn(_Conn));
    } else {
        schWireSetfirstLConn(Wire, schConnGetnextWireLConn(_Conn));
    }
    schConnSetnextWireLConn(_Conn, schConnNull);
    schConnSetLWire(_Conn, schWireNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Wire.
------------------------------------------------------------------------*/
static utBlockRef buildWireBlock(void)
{
    utBlockRef block = utcBlock();
    schWire Wire;
    U16 x, sWire;

    utaHeapBlock(_schWireHeap, block);
    sWire = UTBLOCKSIZE/sizeof(struct _schWire);
    _schFirstFreeWire = (schWire)utgBlockMem(block);
    Wire = _schFirstFreeWire;
    for(x = 0; x < sWire; x++) {
        _schWireNextFree(Wire) = Wire + 1;
        Wire++;
    }
    _schWireNextFree(--Wire) = schWireNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Wire.
----------------------------------------------------------------------------*/
schWire _schWireAlloc(void)
{
    schWire Wire;

    if(_schFirstFreeWire == schWireNull) {
        buildWireBlock();
    }
    Wire = _schFirstFreeWire;
    _schFirstFreeWire = _schWireNextFree(Wire);
    memset((void *)Wire, 0, sizeof(struct _schWire));
    return Wire;
}

/*----------------------------------------------------------------------------
  Free a Wire.
----------------------------------------------------------------------------*/
void _schWireFree(
    schWire Wire)
{
    _schWireNextFree(Wire) = _schFirstFreeWire;
    _schFirstFreeWire = Wire;
}

/*----------------------------------------------------------------------------
  Destroy a Wire.
----------------------------------------------------------------------------*/
void schWireDestroy(
    schWire Wire)
{
    if(schWireDestructorHook != NULL) {
        schWireDestructorHook(Wire);
    }
    {
        schConn _Conn;
        schSafeForeachWireRConn(Wire, _Conn) {
            schConnDestroy(_Conn);
        } schEndSafeForeachWireRConn;
    }
    {
        schConn _Conn;
        schSafeForeachWireLConn(Wire, _Conn) {
            schConnDestroy(_Conn);
        } schEndSafeForeachWireLConn;
    }
    if(schWireGetAttr(Wire) != schAttrNull) {
        schAttrDestroy(schWireGetAttr(Wire));
    }
    {
        schNet parentNet = schWireGetNet(Wire);
        if(parentNet != schNetNull) {
            schNetRemoveWire(parentNet, Wire);
        }
    }
    {
        schSchem parentSchem = schWireGetSchem(Wire);
        if(parentSchem != schSchemNull) {
            schSchemRemoveWire(parentSchem, Wire);
        }
    }
    schWireFree(Wire);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Attr.
------------------------------------------------------------------------*/
static utBlockRef buildAttrBlock(void)
{
    utBlockRef block = utcBlock();
    schAttr Attr;
    U16 x, sAttr;

    utaHeapBlock(_schAttrHeap, block);
    sAttr = UTBLOCKSIZE/sizeof(struct _schAttr);
    _schFirstFreeAttr = (schAttr)utgBlockMem(block);
    Attr = _schFirstFreeAttr;
    for(x = 0; x < sAttr; x++) {
        _schAttrNextFree(Attr) = Attr + 1;
        Attr++;
    }
    _schAttrNextFree(--Attr) = schAttrNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Attr.
----------------------------------------------------------------------------*/
schAttr _schAttrAlloc(void)
{
    schAttr Attr;

    if(_schFirstFreeAttr == schAttrNull) {
        buildAttrBlock();
    }
    Attr = _schFirstFreeAttr;
    _schFirstFreeAttr = _schAttrNextFree(Attr);
    memset((void *)Attr, 0, sizeof(struct _schAttr));
    return Attr;
}

/*----------------------------------------------------------------------------
  Free a Attr.
----------------------------------------------------------------------------*/
void _schAttrFree(
    schAttr Attr)
{
    _schAttrNextFree(Attr) = _schFirstFreeAttr;
    _schFirstFreeAttr = Attr;
}

/*----------------------------------------------------------------------------
  Destroy a Attr.
----------------------------------------------------------------------------*/
void schAttrDestroy(
    schAttr Attr)
{
    if(schAttrDestructorHook != NULL) {
        schAttrDestructorHook(Attr);
    }
    if(schAttrGetNextAttr(Attr) != schAttrNull) {
        schAttrDestroy(schAttrGetNextAttr(Attr));
    }
    schAttrFree(Attr);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Conn.
------------------------------------------------------------------------*/
static utBlockRef buildConnBlock(void)
{
    utBlockRef block = utcBlock();
    schConn Conn;
    U16 x, sConn;

    utaHeapBlock(_schConnHeap, block);
    sConn = UTBLOCKSIZE/sizeof(struct _schConn);
    _schFirstFreeConn = (schConn)utgBlockMem(block);
    Conn = _schFirstFreeConn;
    for(x = 0; x < sConn; x++) {
        _schConnNextFree(Conn) = Conn + 1;
        Conn++;
    }
    _schConnNextFree(--Conn) = schConnNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Conn.
----------------------------------------------------------------------------*/
schConn _schConnAlloc(void)
{
    schConn Conn;

    if(_schFirstFreeConn == schConnNull) {
        buildConnBlock();
    }
    Conn = _schFirstFreeConn;
    _schFirstFreeConn = _schConnNextFree(Conn);
    memset((void *)Conn, 0, sizeof(struct _schConn));
    return Conn;
}

/*----------------------------------------------------------------------------
  Free a Conn.
----------------------------------------------------------------------------*/
void _schConnFree(
    schConn Conn)
{
    _schConnNextFree(Conn) = _schFirstFreeConn;
    _schFirstFreeConn = Conn;
}

/*----------------------------------------------------------------------------
  Destroy a Conn.
----------------------------------------------------------------------------*/
void schConnDestroy(
    schConn Conn)
{
    if(schConnDestructorHook != NULL) {
        schConnDestructorHook(Conn);
    }
    {
        schWire parentWire = schConnGetRWire(Conn);
        if(parentWire != schWireNull) {
            schWireRemoveRConn(parentWire, Conn);
        }
    }
    {
        schWire parentWire = schConnGetLWire(Conn);
        if(parentWire != schWireNull) {
            schWireRemoveLConn(parentWire, Conn);
        }
    }
    schConnFree(Conn);
}

/*----------------------------------------------------------------------------
  Add the Line to the head of the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicInsertLine(
    schGraphic Graphic,
    schLine _Line)
{
    schLineSetnextGraphicLine(_Line, schGraphicGetfirstLine(Graphic));
    schGraphicSetfirstLine(Graphic, _Line);
    schLineSetGraphic(_Line, Graphic);
}

/*----------------------------------------------------------------------------
  Remove the Line from the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicRemoveLine(
    schGraphic Graphic,
    schLine _Line)
{
    schLine nextLine, prevLine = schLineNull;

    for(nextLine = schGraphicGetfirstLine(Graphic);
        nextLine != schLineNull && nextLine != _Line;
        nextLine = schLineGetnextGraphicLine(nextLine)) {
        prevLine = nextLine;
    }
    if(nextLine == schLineNull) {
        /* Not in list */
        return;
    }
    if(prevLine != schLineNull) {
        schLineSetnextGraphicLine(prevLine, schLineGetnextGraphicLine(_Line));
    } else {
        schGraphicSetfirstLine(Graphic, schLineGetnextGraphicLine(_Line));
    }
    schLineSetnextGraphicLine(_Line, schLineNull);
    schLineSetGraphic(_Line, schGraphicNull);
}

/*----------------------------------------------------------------------------
  Add the Rect to the head of the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicInsertRect(
    schGraphic Graphic,
    schRect _Rect)
{
    schRectSetnextGraphicRect(_Rect, schGraphicGetfirstRect(Graphic));
    schGraphicSetfirstRect(Graphic, _Rect);
    schRectSetGraphic(_Rect, Graphic);
}

/*----------------------------------------------------------------------------
  Remove the Rect from the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicRemoveRect(
    schGraphic Graphic,
    schRect _Rect)
{
    schRect nextRect, prevRect = schRectNull;

    for(nextRect = schGraphicGetfirstRect(Graphic);
        nextRect != schRectNull && nextRect != _Rect;
        nextRect = schRectGetnextGraphicRect(nextRect)) {
        prevRect = nextRect;
    }
    if(nextRect == schRectNull) {
        /* Not in list */
        return;
    }
    if(prevRect != schRectNull) {
        schRectSetnextGraphicRect(prevRect, schRectGetnextGraphicRect(_Rect));
    } else {
        schGraphicSetfirstRect(Graphic, schRectGetnextGraphicRect(_Rect));
    }
    schRectSetnextGraphicRect(_Rect, schRectNull);
    schRectSetGraphic(_Rect, schGraphicNull);
}

/*----------------------------------------------------------------------------
  Add the Circle to the head of the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicInsertCircle(
    schGraphic Graphic,
    schCircle _Circle)
{
    schCircleSetnextGraphicCircle(_Circle, schGraphicGetfirstCircle(Graphic));
    schGraphicSetfirstCircle(Graphic, _Circle);
    schCircleSetGraphic(_Circle, Graphic);
}

/*----------------------------------------------------------------------------
  Remove the Circle from the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicRemoveCircle(
    schGraphic Graphic,
    schCircle _Circle)
{
    schCircle nextCircle, prevCircle = schCircleNull;

    for(nextCircle = schGraphicGetfirstCircle(Graphic);
        nextCircle != schCircleNull && nextCircle != _Circle;
        nextCircle = schCircleGetnextGraphicCircle(nextCircle)) {
        prevCircle = nextCircle;
    }
    if(nextCircle == schCircleNull) {
        /* Not in list */
        return;
    }
    if(prevCircle != schCircleNull) {
        schCircleSetnextGraphicCircle(prevCircle, schCircleGetnextGraphicCircle(_Circle));
    } else {
        schGraphicSetfirstCircle(Graphic, schCircleGetnextGraphicCircle(_Circle));
    }
    schCircleSetnextGraphicCircle(_Circle, schCircleNull);
    schCircleSetGraphic(_Circle, schGraphicNull);
}

/*----------------------------------------------------------------------------
  Add the Arc to the head of the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicInsertArc(
    schGraphic Graphic,
    schArc _Arc)
{
    schArcSetnextGraphicArc(_Arc, schGraphicGetfirstArc(Graphic));
    schGraphicSetfirstArc(Graphic, _Arc);
    schArcSetGraphic(_Arc, Graphic);
}

/*----------------------------------------------------------------------------
  Remove the Arc from the list on the Graphic.
----------------------------------------------------------------------------*/
void schGraphicRemoveArc(
    schGraphic Graphic,
    schArc _Arc)
{
    schArc nextArc, prevArc = schArcNull;

    for(nextArc = schGraphicGetfirstArc(Graphic);
        nextArc != schArcNull && nextArc != _Arc;
        nextArc = schArcGetnextGraphicArc(nextArc)) {
        prevArc = nextArc;
    }
    if(nextArc == schArcNull) {
        /* Not in list */
        return;
    }
    if(prevArc != schArcNull) {
        schArcSetnextGraphicArc(prevArc, schArcGetnextGraphicArc(_Arc));
    } else {
        schGraphicSetfirstArc(Graphic, schArcGetnextGraphicArc(_Arc));
    }
    schArcSetnextGraphicArc(_Arc, schArcNull);
    schArcSetGraphic(_Arc, schGraphicNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Graphic.
------------------------------------------------------------------------*/
static utBlockRef buildGraphicBlock(void)
{
    utBlockRef block = utcBlock();
    schGraphic Graphic;
    U16 x, sGraphic;

    utaHeapBlock(_schGraphicHeap, block);
    sGraphic = UTBLOCKSIZE/sizeof(struct _schGraphic);
    _schFirstFreeGraphic = (schGraphic)utgBlockMem(block);
    Graphic = _schFirstFreeGraphic;
    for(x = 0; x < sGraphic; x++) {
        _schGraphicNextFree(Graphic) = Graphic + 1;
        Graphic++;
    }
    _schGraphicNextFree(--Graphic) = schGraphicNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Graphic.
----------------------------------------------------------------------------*/
schGraphic _schGraphicAlloc(void)
{
    schGraphic Graphic;

    if(_schFirstFreeGraphic == schGraphicNull) {
        buildGraphicBlock();
    }
    Graphic = _schFirstFreeGraphic;
    _schFirstFreeGraphic = _schGraphicNextFree(Graphic);
    memset((void *)Graphic, 0, sizeof(struct _schGraphic));
    return Graphic;
}

/*----------------------------------------------------------------------------
  Free a Graphic.
----------------------------------------------------------------------------*/
void _schGraphicFree(
    schGraphic Graphic)
{
    _schGraphicNextFree(Graphic) = _schFirstFreeGraphic;
    _schFirstFreeGraphic = Graphic;
}

/*----------------------------------------------------------------------------
  Destroy a Graphic.
----------------------------------------------------------------------------*/
void schGraphicDestroy(
    schGraphic Graphic)
{
    if(schGraphicDestructorHook != NULL) {
        schGraphicDestructorHook(Graphic);
    }
    {
        schLine _Line;
        schSafeForeachGraphicLine(Graphic, _Line) {
            schLineDestroy(_Line);
        } schEndSafeForeachGraphicLine;
    }
    {
        schRect _Rect;
        schSafeForeachGraphicRect(Graphic, _Rect) {
            schRectDestroy(_Rect);
        } schEndSafeForeachGraphicRect;
    }
    {
        schCircle _Circle;
        schSafeForeachGraphicCircle(Graphic, _Circle) {
            schCircleDestroy(_Circle);
        } schEndSafeForeachGraphicCircle;
    }
    {
        schArc _Arc;
        schSafeForeachGraphicArc(Graphic, _Arc) {
            schArcDestroy(_Arc);
        } schEndSafeForeachGraphicArc;
    }
    schGraphicFree(Graphic);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Line.
------------------------------------------------------------------------*/
static utBlockRef buildLineBlock(void)
{
    utBlockRef block = utcBlock();
    schLine Line;
    U16 x, sLine;

    utaHeapBlock(_schLineHeap, block);
    sLine = UTBLOCKSIZE/sizeof(struct _schLine);
    _schFirstFreeLine = (schLine)utgBlockMem(block);
    Line = _schFirstFreeLine;
    for(x = 0; x < sLine; x++) {
        _schLineNextFree(Line) = Line + 1;
        Line++;
    }
    _schLineNextFree(--Line) = schLineNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Line.
----------------------------------------------------------------------------*/
schLine _schLineAlloc(void)
{
    schLine Line;

    if(_schFirstFreeLine == schLineNull) {
        buildLineBlock();
    }
    Line = _schFirstFreeLine;
    _schFirstFreeLine = _schLineNextFree(Line);
    memset((void *)Line, 0, sizeof(struct _schLine));
    return Line;
}

/*----------------------------------------------------------------------------
  Free a Line.
----------------------------------------------------------------------------*/
void _schLineFree(
    schLine Line)
{
    _schLineNextFree(Line) = _schFirstFreeLine;
    _schFirstFreeLine = Line;
}

/*----------------------------------------------------------------------------
  Destroy a Line.
----------------------------------------------------------------------------*/
void schLineDestroy(
    schLine Line)
{
    if(schLineDestructorHook != NULL) {
        schLineDestructorHook(Line);
    }
    {
        schGraphic parentGraphic = schLineGetGraphic(Line);
        if(parentGraphic != schGraphicNull) {
            schGraphicRemoveLine(parentGraphic, Line);
        }
    }
    schLineFree(Line);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Rect.
------------------------------------------------------------------------*/
static utBlockRef buildRectBlock(void)
{
    utBlockRef block = utcBlock();
    schRect Rect;
    U16 x, sRect;

    utaHeapBlock(_schRectHeap, block);
    sRect = UTBLOCKSIZE/sizeof(struct _schRect);
    _schFirstFreeRect = (schRect)utgBlockMem(block);
    Rect = _schFirstFreeRect;
    for(x = 0; x < sRect; x++) {
        _schRectNextFree(Rect) = Rect + 1;
        Rect++;
    }
    _schRectNextFree(--Rect) = schRectNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Rect.
----------------------------------------------------------------------------*/
schRect _schRectAlloc(void)
{
    schRect Rect;

    if(_schFirstFreeRect == schRectNull) {
        buildRectBlock();
    }
    Rect = _schFirstFreeRect;
    _schFirstFreeRect = _schRectNextFree(Rect);
    memset((void *)Rect, 0, sizeof(struct _schRect));
    return Rect;
}

/*----------------------------------------------------------------------------
  Free a Rect.
----------------------------------------------------------------------------*/
void _schRectFree(
    schRect Rect)
{
    _schRectNextFree(Rect) = _schFirstFreeRect;
    _schFirstFreeRect = Rect;
}

/*----------------------------------------------------------------------------
  Destroy a Rect.
----------------------------------------------------------------------------*/
void schRectDestroy(
    schRect Rect)
{
    if(schRectDestructorHook != NULL) {
        schRectDestructorHook(Rect);
    }
    {
        schGraphic parentGraphic = schRectGetGraphic(Rect);
        if(parentGraphic != schGraphicNull) {
            schGraphicRemoveRect(parentGraphic, Rect);
        }
    }
    schRectFree(Rect);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Circle.
------------------------------------------------------------------------*/
static utBlockRef buildCircleBlock(void)
{
    utBlockRef block = utcBlock();
    schCircle Circle;
    U16 x, sCircle;

    utaHeapBlock(_schCircleHeap, block);
    sCircle = UTBLOCKSIZE/sizeof(struct _schCircle);
    _schFirstFreeCircle = (schCircle)utgBlockMem(block);
    Circle = _schFirstFreeCircle;
    for(x = 0; x < sCircle; x++) {
        _schCircleNextFree(Circle) = Circle + 1;
        Circle++;
    }
    _schCircleNextFree(--Circle) = schCircleNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Circle.
----------------------------------------------------------------------------*/
schCircle _schCircleAlloc(void)
{
    schCircle Circle;

    if(_schFirstFreeCircle == schCircleNull) {
        buildCircleBlock();
    }
    Circle = _schFirstFreeCircle;
    _schFirstFreeCircle = _schCircleNextFree(Circle);
    memset((void *)Circle, 0, sizeof(struct _schCircle));
    return Circle;
}

/*----------------------------------------------------------------------------
  Free a Circle.
----------------------------------------------------------------------------*/
void _schCircleFree(
    schCircle Circle)
{
    _schCircleNextFree(Circle) = _schFirstFreeCircle;
    _schFirstFreeCircle = Circle;
}

/*----------------------------------------------------------------------------
  Destroy a Circle.
----------------------------------------------------------------------------*/
void schCircleDestroy(
    schCircle Circle)
{
    if(schCircleDestructorHook != NULL) {
        schCircleDestructorHook(Circle);
    }
    {
        schGraphic parentGraphic = schCircleGetGraphic(Circle);
        if(parentGraphic != schGraphicNull) {
            schGraphicRemoveCircle(parentGraphic, Circle);
        }
    }
    schCircleFree(Circle);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Arc.
------------------------------------------------------------------------*/
static utBlockRef buildArcBlock(void)
{
    utBlockRef block = utcBlock();
    schArc Arc;
    U16 x, sArc;

    utaHeapBlock(_schArcHeap, block);
    sArc = UTBLOCKSIZE/sizeof(struct _schArc);
    _schFirstFreeArc = (schArc)utgBlockMem(block);
    Arc = _schFirstFreeArc;
    for(x = 0; x < sArc; x++) {
        _schArcNextFree(Arc) = Arc + 1;
        Arc++;
    }
    _schArcNextFree(--Arc) = schArcNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Arc.
----------------------------------------------------------------------------*/
schArc _schArcAlloc(void)
{
    schArc Arc;

    if(_schFirstFreeArc == schArcNull) {
        buildArcBlock();
    }
    Arc = _schFirstFreeArc;
    _schFirstFreeArc = _schArcNextFree(Arc);
    memset((void *)Arc, 0, sizeof(struct _schArc));
    return Arc;
}

/*----------------------------------------------------------------------------
  Free a Arc.
----------------------------------------------------------------------------*/
void _schArcFree(
    schArc Arc)
{
    _schArcNextFree(Arc) = _schFirstFreeArc;
    _schFirstFreeArc = Arc;
}

/*----------------------------------------------------------------------------
  Destroy a Arc.
----------------------------------------------------------------------------*/
void schArcDestroy(
    schArc Arc)
{
    if(schArcDestructorHook != NULL) {
        schArcDestructorHook(Arc);
    }
    {
        schGraphic parentGraphic = schArcGetGraphic(Arc);
        if(parentGraphic != schGraphicNull) {
            schGraphicRemoveArc(parentGraphic, Arc);
        }
    }
    schArcFree(Arc);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Signal.
------------------------------------------------------------------------*/
static utBlockRef buildSignalBlock(void)
{
    utBlockRef block = utcBlock();
    schSignal Signal;
    U16 x, sSignal;

    utaHeapBlock(_schSignalHeap, block);
    sSignal = UTBLOCKSIZE/sizeof(struct _schSignal);
    _schFirstFreeSignal = (schSignal)utgBlockMem(block);
    Signal = _schFirstFreeSignal;
    for(x = 0; x < sSignal; x++) {
        _schSignalNextFree(Signal) = Signal + 1;
        Signal++;
    }
    _schSignalNextFree(--Signal) = schSignalNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Signal.
----------------------------------------------------------------------------*/
schSignal _schSignalAlloc(void)
{
    schSignal Signal;

    if(_schFirstFreeSignal == schSignalNull) {
        buildSignalBlock();
    }
    Signal = _schFirstFreeSignal;
    _schFirstFreeSignal = _schSignalNextFree(Signal);
    memset((void *)Signal, 0, sizeof(struct _schSignal));
    return Signal;
}

/*----------------------------------------------------------------------------
  Free a Signal.
----------------------------------------------------------------------------*/
void _schSignalFree(
    schSignal Signal)
{
    _schSignalNextFree(Signal) = _schFirstFreeSignal;
    _schFirstFreeSignal = Signal;
}

/*----------------------------------------------------------------------------
  Destroy a Signal.
----------------------------------------------------------------------------*/
void schSignalDestroy(
    schSignal Signal)
{
    if(schSignalDestructorHook != NULL) {
        schSignalDestructorHook(Signal);
    }
    {
        schNet parentNet = schSignalGetNet(Signal);
        if(parentNet != schNetNull) {
            schNetRemoveSignal(parentNet, Signal);
        }
    }
    schSignalFree(Signal);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Bus.
------------------------------------------------------------------------*/
static utBlockRef buildBusBlock(void)
{
    utBlockRef block = utcBlock();
    schBus Bus;
    U16 x, sBus;

    utaHeapBlock(_schBusHeap, block);
    sBus = UTBLOCKSIZE/sizeof(struct _schBus);
    _schFirstFreeBus = (schBus)utgBlockMem(block);
    Bus = _schFirstFreeBus;
    for(x = 0; x < sBus; x++) {
        _schBusNextFree(Bus) = Bus + 1;
        Bus++;
    }
    _schBusNextFree(--Bus) = schBusNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Bus.
----------------------------------------------------------------------------*/
schBus _schBusAlloc(void)
{
    schBus Bus;

    if(_schFirstFreeBus == schBusNull) {
        buildBusBlock();
    }
    Bus = _schFirstFreeBus;
    _schFirstFreeBus = _schBusNextFree(Bus);
    memset((void *)Bus, 0, sizeof(struct _schBus));
    return Bus;
}

/*----------------------------------------------------------------------------
  Free a Bus.
----------------------------------------------------------------------------*/
void _schBusFree(
    schBus Bus)
{
    _schBusNextFree(Bus) = _schFirstFreeBus;
    _schFirstFreeBus = Bus;
}

/*----------------------------------------------------------------------------
  Destroy a Bus.
----------------------------------------------------------------------------*/
void schBusDestroy(
    schBus Bus)
{
    if(schBusDestructorHook != NULL) {
        schBusDestructorHook(Bus);
    }
    {
        schSchem parentSchem = schBusGetSchem(Bus);
        if(parentSchem != schSchemNull) {
            schSchemRemoveBus(parentSchem, Bus);
        }
    }
    schBusFree(Bus);
}

/*----------------------------------------------------------------------------
  Add the Comp to the end of the array on the Col.
----------------------------------------------------------------------------*/
void schColInsertComp(
    schCol Col,
    schComp _Comp)
{
    if(schColGetusedComps(Col) == schColGetnumComps(Col)) {
        schColReallocComps(Col, schColGetnumComps(Col) +
            (schColGetnumComps(Col) >> 1) + 1);
    }
    schCompSetColIndex(_Comp, schColGetusedComps(Col));
    schColSetiComp(Col, schColGetusedComps(Col)++, _Comp);
    schCompSetCol(_Comp, Col);
}

/*----------------------------------------------------------------------------
  Remove the Comp from the array on the Col.
----------------------------------------------------------------------------*/
void schColRemoveComp(
    schCol Col,
    schComp _Comp)
{
    U32 xComp = schCompGetColIndex(_Comp);

    if(xComp != U32_MAX) {
        schColSetiComp(Col, xComp, schColGetiComp(Col,
            --schColGetusedComps(Col)));
        schCompSetColIndex(_Comp, U32_MAX);
    }
    schCompSetCol(_Comp, schColNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Col.
------------------------------------------------------------------------*/
static utBlockRef buildColBlock(void)
{
    utBlockRef block = utcBlock();
    schCol Col;
    U16 x, sCol;

    utaHeapBlock(_schColHeap, block);
    sCol = UTBLOCKSIZE/sizeof(struct _schCol);
    _schFirstFreeCol = (schCol)utgBlockMem(block);
    Col = _schFirstFreeCol;
    for(x = 0; x < sCol; x++) {
        _schColNextFree(Col) = Col + 1;
        Col++;
    }
    _schColNextFree(--Col) = schColNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Col.
----------------------------------------------------------------------------*/
schCol schColAlloc(void)
{
    schCol Col;

    if(_schFirstFreeCol == schColNull) {
        buildColBlock();
    }
    Col = _schFirstFreeCol;
    _schFirstFreeCol = _schColNextFree(Col);
    memset((void *)Col, 0, sizeof(struct _schCol));
    return Col;
}

/*----------------------------------------------------------------------------
  Free a Col.
----------------------------------------------------------------------------*/
void schColFree(
    schCol Col)
{
    if(schColGetnumComps(Col) != 0) {
        schColFreeComps(Col);
    }
    _schColNextFree(Col) = _schFirstFreeCol;
    _schFirstFreeCol = Col;
}

/*----------------------------------------------------------------------------
  Destroy a Col.
----------------------------------------------------------------------------*/
void schColDestroy(
    schCol Col)
{
    if(schColDestructorHook != NULL) {
        schColDestructorHook(Col);
    }
    {
        schComp _Comp;
        schSafeForeachColComp(Col, _Comp) {
            schColRemoveComp(Col, _Comp);
        } schEndSafeForeachColComp;
    }
    {
        schSchem parentSchem = schColGetSchem(Col);
        if(parentSchem != schSchemNull) {
            schSchemRemoveCol(parentSchem, Col);
        }
    }
    schColFree(Col);
}

/*----------------------------------------------------------------------------
  Initialize memory for the sch&&.
----------------------------------------------------------------------------*/
void schDDRStart(void)
{
    _schFirstFreeNet = schNetNull;
    _schNetHeap = utcHeap();
    buildNetBlock();
    _schFirstFreeComp = schCompNull;
    _schCompHeap = utcHeap();
    buildCompBlock();
    _schFirstFreeMpin = schMpinNull;
    _schMpinHeap = utcHeap();
    buildMpinBlock();
    _schFirstFreeSchem = schSchemNull;
    _schSchemHeap = utcHeap();
    buildSchemBlock();
    _schFirstFreeRoot = schRootNull;
    _schRootHeap = utcHeap();
    buildRootBlock();
    _schFirstFreeSymbol = schSymbolNull;
    _schSymbolHeap = utcHeap();
    buildSymbolBlock();
    _schFirstFreePin = schPinNull;
    _schPinHeap = utcHeap();
    buildPinBlock();
    _schFirstFreeWire = schWireNull;
    _schWireHeap = utcHeap();
    buildWireBlock();
    _schFirstFreeAttr = schAttrNull;
    _schAttrHeap = utcHeap();
    buildAttrBlock();
    _schFirstFreeConn = schConnNull;
    _schConnHeap = utcHeap();
    buildConnBlock();
    _schFirstFreeGraphic = schGraphicNull;
    _schGraphicHeap = utcHeap();
    buildGraphicBlock();
    _schFirstFreeLine = schLineNull;
    _schLineHeap = utcHeap();
    buildLineBlock();
    _schFirstFreeRect = schRectNull;
    _schRectHeap = utcHeap();
    buildRectBlock();
    _schFirstFreeCircle = schCircleNull;
    _schCircleHeap = utcHeap();
    buildCircleBlock();
    _schFirstFreeArc = schArcNull;
    _schArcHeap = utcHeap();
    buildArcBlock();
    _schFirstFreeSignal = schSignalNull;
    _schSignalHeap = utcHeap();
    buildSignalBlock();
    _schFirstFreeBus = schBusNull;
    _schBusHeap = utcHeap();
    buildBusBlock();
    _schFirstFreeCol = schColNull;
    _schColHeap = utcHeap();
    buildColBlock();
}

/*----------------------------------------------------------------------------
  Free memory for the sch&&.
----------------------------------------------------------------------------*/
void schDDRStop(void)
{
    utFreeHeap(_schNetHeap);
    utFreeHeap(_schCompHeap);
    utFreeHeap(_schMpinHeap);
    utFreeHeap(_schSchemHeap);
    utFreeHeap(_schRootHeap);
    utFreeHeap(_schSymbolHeap);
    utFreeHeap(_schPinHeap);
    utFreeHeap(_schWireHeap);
    utFreeHeap(_schAttrHeap);
    utFreeHeap(_schConnHeap);
    utFreeHeap(_schGraphicHeap);
    utFreeHeap(_schLineHeap);
    utFreeHeap(_schRectHeap);
    utFreeHeap(_schCircleHeap);
    utFreeHeap(_schArcHeap);
    utFreeHeap(_schSignalHeap);
    utFreeHeap(_schBusHeap);
    utFreeHeap(_schColHeap);
}

