/*
 * Copyright (C) 2003 ViASIC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */

/*==================================================================================================
   Module : Virtual Operating System
   Purpose: Hide system specific operations from other modules.  This should help in portability.
==================================================================================================*/

#ifndef UTIL_H
#define UTIL_H

#if __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#if defined(_WINDOWS) && !defined(_WIN32)
#else
#include <setjmp.h>
#endif
#if !(defined(_WINDOWS) || defined(_WIN32))
/* #include <unistd.h> */
#else
/* #include <direct.h> */
#endif

#ifdef DMALLOC
#include "dmalloc.h"
#endif

/*--------------------------------------------------------------------------------------------------
   Datadraw defined Data types
--------------------------------------------------------------------------------------------------*/

#ifndef UTTYPES_H
#include "uttypes.h"
#endif

typedef U16 utHeapRef;
typedef U32 utBlockRef;
typedef U32 utSym;
/*--------------------------------------------------------------------------------------------------
   Basic memory management support.
--------------------------------------------------------------------------------------------------*/
#ifndef UTMEM_H
#include "utmem.h"
#endif

/*--------------------------------------------------------------------------------------------------
   Basic property value support.
--------------------------------------------------------------------------------------------------*/
#ifndef UTVALUE_H
#include "utvalue.h"
#endif

/*--------------------------------------------------------------------------------------------------
   Basic symbol based hash table support.
--------------------------------------------------------------------------------------------------*/
#ifndef UTHTBL_H
#include "uthtbl.h"
#endif

/*--------------------------------------------------------------------------------------------------
  Utility Functions needed by utbox.h
--------------------------------------------------------------------------------------------------*/
#define utInOrderOpenOpen(a, b, c) (((a) < (b)) && ((b) < (c)))       /* ( ) */
#define utInOrderOpenClosed(a, b, c) (((a) < (b)) && ((b) <= (c)))    /* ( ] */
#define utInOrderClosedOpen(a, b, c) (((a) <= (b)) && ((b) < (c)))    /* [ ) */
#define utInOrderClosedClosed(a, b, c) (((a) <= (b)) && ((b) <= (c))) /* [ ] */
#define utInOrder(a, b, c) utInOrderClosedClosed(a, b, c)     /* default [ ] */
#define utIncrementU64ByU32(a, b) ((a) = utAddU64((a), utConvertU32ToU64(b)))
#define utDecrementU64ByU32(a, b) ((a) = utSubtractU64((a), utConvertU32ToU64(b)))
/* if using above functions, consider using utIncrementU64By1 */

/*--------------------------------------------------------------------------------------------------
   Support for box and point manipulation.
--------------------------------------------------------------------------------------------------*/
#ifndef UTBOX_H
#include "utbox.h"
#endif

/*--------------------------------------------------------------------------------------------------
  The max size of a string which the utFprintf, utExit, etc functions
  can print without overflowing their buffers.
--------------------------------------------------------------------------------------------------*/
#define UTSTRLEN 4096

/*--------------------------------------------------------------------------------------------------
  Macro to eliminate compiler warnings of unused paramaters.
--------------------------------------------------------------------------------------------------*/
#ifdef NOREF
#undef NOREF
#endif
#define NOREF(x) ((void) x)

#define UTEOF '\0'

extern U32 utDebugVal;
extern U32 utVerboseVal;
extern bool utLogAccess;
extern U32 utAccessCount;
extern U32 utAccessHash;
extern U32 utEnableCoreFile;
extern U8 utCount1sTable[];

extern void utSetConfigDirectory(char *dirName);
extern void utSetExeFullPath(char *fullName);
extern void utSetVersion(char *version);
extern char *utGetVersion(void);
extern char *utGetExeDirectory(void);
extern char *utGetConfigDirectory(void);
extern char *utGetExeFullPath(void);

/*--------------------------------------------------------------------------------------------------
  User parameters are to help programmers vary things from scripts rather than hard coding them.
--------------------------------------------------------------------------------------------------*/
extern U32 utUserParameter1, utUserParameter2, utUserParameter3;

/*--------------------------------------------------------------------------------------------------
  Error Callback function to be provided by the user.
--------------------------------------------------------------------------------------------------*/
typedef void (*utErrorProc)(char *message);
typedef bool (*utMessageYesNoProc)(char *message);
typedef void (*utExitProc)(void);
typedef bool (*utProgressProc)(U8 progress);

/*--------------------------------------------------------------------------------------------------
  File Processing modes
--------------------------------------------------------------------------------------------------*/
typedef enum {
   UT_READ,
   UT_WRITE,
   UT_CALC
} utModeType;

/*--------------------------------------------------------------------------------------------------
  File buffering for systems with slow getc (like Windows).  Only works with one file at a time.
--------------------------------------------------------------------------------------------------*/
extern char *utFileBuffer;
extern U32 utFileBufferPos, utFileBufferSize;
#define utGetc(file) (utFileBufferPos < utFileBufferSize? \
    utFileBuffer[utFileBufferPos++] : utGetc_(file))
extern UTINT utGetc_(FILE *file);
extern FILE *utFopen(char *fileName, char *mode);
extern char *utFullPath(char *relativePath);
extern char *utGlob(char *fileName);
extern char *utGetcwd(void);
#define utFclose(file) fclose(file)

/*--------------------------------------------------------------------------------------------------
  Portable interface to ini-variables
--------------------------------------------------------------------------------------------------*/

extern char szIniFileName[];

extern void utInitProfile (char *programName, char *iniPath, char *defPath);
extern void utCloseProfile(void);
extern bool utGetModuleDirectory(char *moduleName, char *path, S16 max_path);

/* Read desired integer field from profile.  */
extern S32  utgProfileInt(char *section, char *keyName, S32 nDefault);
/* Read desired double field from profile.  */
extern double  utgProfileDouble(char *section, char *keyName, double nDefault);
/* Read desired string field from profile.  */
extern void  utgProfileString(char *section, char *keyName, char *nDefault, char *retStr,
   S16 nSize);
/* Read desired boolean string field from profile.  */
extern bool  utgProfileBoolean(char *section, char *keyName, bool defaultValue);
/* Write desired string field to profile.  */
extern void  utcProfileString(char *section, char *keyName, char *setStr);
/* Write desired S32 field to profile.  */
extern void  utcProfileInt(char *section, char *keyName, S32 setInt);
/* Write desired double field to profile.  */
extern void  utcProfileDouble(char *section, char *keyName, double setDouble);
/* Write desired boolean field to profile.  */
extern void  utcProfileBoolean(char *section, char *keyName, S32 setBoolean);

/* read or write default profile string */
extern bool utGetProfileString(char *szToolName, char *szKeyName, char *szDefault, char *szBuffer,
   S16 nSize);
extern bool utWriteProfileString(char *szSection, char *szKeyName, char *szValue);

/* read or write profile string or integer */
extern S32 utGetPrivateProfileInt(char *szToolName,
   char *szKeyName, S32 nDefault, char *szIniFileName);
extern void utGetPrivateProfileString(char *szToolName,
   char *szKeyName, char *szDefault, char *szBuffer,
   S16 nSize, char *szIniFileName);
extern void utWritePrivateProfileString(char *szToolName,
   char *szKeyName, char *szSetStr, char *szIniFileName);

/*--------------------------------------------------------------------------------------------------
  Portable interface to process & dll launch, and abnormal exit.
--------------------------------------------------------------------------------------------------*/
extern bool utLaunchApp(char* cmdLine, char *wkgDir);
extern bool utLoadLibrary(char* cmdLine);
extern void utAbnormalProgramTermination(void);

/*--------------------------------------------------------------------------------------------------
  Heap object.
--------------------------------------------------------------------------------------------------*/
#define UTMALLOCOVERHEAD 8
#define UTHEAPSIZE (16384 - UTMALLOCOVERHEAD)

/*--------------------------------------------------------------------------------------------------
   Object : utHeap
   Purpose: Represent a garbage collected group of blocks of memory.
--------------------------------------------------------------------------------------------------*/
struct _utHeap {
   utBlockRef fBlock;
   utHeapRef _nHeap;
   bool _used:1;
   U32 sBlock;
};

#define ut0Heap U16_MAX
#define uttHeapExists(Heap) ((Heap) != ut0Heap)
#define utfHeapBlock(Heap) (_utRoot.Heaps[Heap].fBlock)
#define utsHeapBlock(Heap) (_utRoot.Heaps[Heap].sBlock)
#define _utnHeap(Heap) (_utRoot.Heaps[Heap]._nHeap)
#define _uttHeapUsed(Heap) (_utRoot.Heaps[Heap]._used)
extern utHeapRef utfHeap(void);
extern utHeapRef utnHeap(utHeapRef Heap);
extern utHeapRef utcHeap(void);
extern void utdHeap(utHeapRef Heap);
extern void utaHeapBlock(utHeapRef Heap, utBlockRef Block);
extern void utdHeapBlock(utHeapRef Heap, utBlockRef Block);
extern void utFreeHeap(utHeapRef Heap);

/*--------------------------------------------------------------------------------------------------
   Object : utBlock
   Purpose: Represent a block of memory for use in memory allocation.
--------------------------------------------------------------------------------------------------*/
struct _utBlock {
   utHeapRef oHeap;
   utBlockRef nHeapBlock;
   void *mem;
   utBlockRef _nBlock;
   bool _used:1;
};

/* Block size should be multiple of page size, then a bit smaller for malloc overhead */
#define UTBLOCKSIZE ((1 << 14) - 12)
#define ut0Block U32_MAX
#define uttBlockExists(Block) ((Block) != ut0Block)
#define utoHeapBlock(Block) (_utRoot.Blocks[Block].oHeap)
#define utnHeapBlock(Heap, Block) (_utRoot.Blocks[Block].nHeapBlock)
#define utgBlockMem(block) (_utRoot.Blocks[block].mem)
#define _utnBlock(Block) (_utRoot.Blocks[Block]._nBlock)
#define _uttBlockUsed(Block) (_utRoot.Blocks[Block]._used)
extern utBlockRef utfBlock(void);
extern utBlockRef utnBlock(utBlockRef Block);
extern utBlockRef utcBlock(void);
extern void utdBlock(utBlockRef Block);

struct _utRootType {
   struct _utHeap * Heaps;
   U16 mHeap, fVirgHeap, fFreeHeap;
   struct _utBlock * Blocks;
   U32 mBlock, fVirgBlock, fFreeBlock;
};
extern struct _utRootType _utRoot;

#define _utfFreeHeap() (_utRoot.fFreeHeap)
#define _utfVirgHeap() (_utRoot.fVirgHeap)
#define _utmHeap() (_utRoot.mHeap)
#define _utfFreeBlock() (_utRoot.fFreeBlock)
#define _utfVirgBlock() (_utRoot.fVirgBlock)
#define _utmBlock() (_utRoot.mBlock)


/*--------------------------------------------------------------------------------------------------
  Memory allocation.
  There are things called malloc, calloc and free, but they deal in the
  local heap, thus are to be avoided.
--------------------------------------------------------------------------------------------------*/
extern U32 utUsedMem;
#define utMalloc(sStruct, size) utMallocTrace(sStruct, size, __FILE__, __LINE__)
#define utCalloc(sStruct, size) utCallocTrace(sStruct, size, \
      __FILE__, __LINE__)
#define utCallocType(num, type) (type *)utCalloc((num), sizeof(type))
#define utMallocType(num, type) (type *)utMalloc((num), sizeof(type))
#define utRealloc(mem, numBlocks, size) \
      utReallocTrace((void *)mem, numBlocks, size, __FILE__, __LINE__)
#define utResizeArray(array, num) \
        ((array) = utRealloc((void *)(array), (num), sizeof(*(array))))
#define utFree(p) utFreeTrace(p, __FILE__, __LINE__)
#define utDelete(p) utFreeTrace(p, __FILE__, __LINE__)
#define utNew(type) (type *)utCalloc(1, sizeof(type))
#define utNewA(type, num) (type *)utCalloc((num), sizeof(type))

extern void *mtCalloc (U32 numBlocks, U16 size);
extern void *mtMalloc (U32 numBlocks, U16 size);
extern void *mtRealloc (void *mem, U32 numBlocks, U16 size);
extern bool  mtFree (void *p);
#define mtResizeArray(array, num) \
        ((array) = mtRealloc((void *)(array), (num), sizeof(*(array))))
#define mtDelete(p) mtFree(p)
#define mtNew(type) (type *)mtCalloc(1, sizeof(type))
#define mtNewA(type, num) (type *)mtCalloc((num), sizeof(type))

/* These use a queue of buffers */
extern void *utMakeBuffer_(U32 length);
#define utNewBufA(type, num) (type *)utMakeBuffer_((num)*sizeof(type))
#define utNewBuf(type) (type *)utMakeBuffer_(sizeof(type))
#define utMakeString(length) (char *)utMakeBuffer_(length)
extern char *utCopyString(char *string);
extern char *utCatStrings(char *string1, char *string2);
extern char *utStringToUpperCase(char *string);
extern char *utStringToLowerCase(char *string);

/* maxmimum memory usage */
extern U32 utmByte;

/*--------------------------------------------------------------------------------------------------
  utAbort is set to TRUE by the application if the currently executing tool is canceled.
--------------------------------------------------------------------------------------------------*/
extern bool utAbort;
extern bool utEnableLongjmpOnAbort;

extern U32 utSignature;

extern void utInitSeed(U32 seed);
extern U32 utRand(void);
#define utRandN(n) (utRand() % (n))
#define utRandBool() ((bool) (utRand() & 1))
extern U32 utHash(U16 data);
extern U32 utHashSig(U16 data);

/*--------------------------------------------------------------------------------------------------
  Utility Functions
--------------------------------------------------------------------------------------------------*/
#define utInBetweenClosed(a, b, c) (((a)<(c)) ?                         \
                                     (((a) <= (b)) && ((b) <= (c)))     \
                                   : (((c) <= (b)) && ((b) <= (a)))) /* [ ] */
#define utInBetweenOpen(a, b, c) (((a)<(c)) ?                           \
                                     (((a) < (b)) && ((b) < (c)))       \
                                   : (((c) < (b)) && ((b) < (a))))   /* ( ) */
#define utInBetween(a, b, c) utInBetweenClosed(a, b, c)      /* default [ ] */

/* check [low1,high1] is in range [low2,high2]. Example: [0,1] [1,2] overlap */
#define utIntervalsOverlapClosed(low1, high1, low2, high2)        \
    (!(((high1) < (low2)) || ((low1) > (high2))))
/* check [low1,high1] is in range (low2,high2). Example: [0,1] (1,2) don't overlap */
#define utIntervalsOverlapOpen(low1, high1, low2, high2)          \
    (!(((high1) <= (low2)) || ((low1) >= (high2))))
#define utIntervalsOverlap(low1, high1, low2, high2)              \
    utIntervalsOverlapClosed(low1, high1, low2, high2)       /* default [ ] */

extern U32 utFindFileSize(FILE *file);
extern bool utFileExists(char *fileName);
extern char *utSprintf (char *format, ...);
extern void utLogMessage (char *format, ...);
extern char *utGetCompileTime(void);
extern char *utCompileTime(void);
extern char *utGetDateAndTime(void);
extern bool utDebug(char *format,... );
extern void utLogError(char *format, ...);
extern void utLogTimeStamp (char *message);
extern void utStartTimer(char *message, ...);
extern void utStopTimer(char *message, ...);
extern void utStatus (char *format, ...);
extern void utExit_ (char *format, ...);
typedef void (*utExitProcType)(char *format, ...);
extern utExitProcType utSetFileAndLineAndReturnExitFunc(char *fileName, U32 lineNum);
#define utExit utSetFileAndLineAndReturnExitFunc(__FILE__, __LINE__)
extern void utWarning (char *format, ...);
extern void utNote (char *format, ...);
extern U32 utError (char *format, ...);

/* Log a message to the log file and exit if false is passed. */
extern void utAssert_(char *fileName, U32 line, char *text);
#define utAssert(assertion) ((void)(utLikely(assertion) || (utAssert_(__FILE__, __LINE__, #assertion), 0)))
#define utClearMem(mem,sByte) (mtClearMem (mem, sByte))
#define utMemset(bytes, c, sByte) (mtMemset(bytes, c, sByte))
extern void utSetErrorCallback (utErrorProc errorProc);
extern void utSetWarningCallback (utErrorProc warningProc);
extern void utSetStatusCallback (utErrorProc statusProc);
extern void utSetMessageYesNoCallback (utMessageYesNoProc messageProc);
extern bool utMessageYesNo(char *msg);
extern void utSetExitCallback (utExitProc exitProc);
extern void  utInitLogFile (char *fileName);
extern void  utSetLogFile (char *fileName);
extern void utSetProgressCallback(utProgressProc progressProc);
extern bool utSetProgress(U8 percentDone);
extern void utResetProgress(void);
extern void mtClearMem (void *mem, U32 sByte);
extern void mtMemset (char *bytes, char c, U32 sByte);
extern U32 utFindMaskMSBit(U32 word);
extern U32 utFindMaskLSBit(U32 word);
extern U8 utFindPosMSBit(U32 word);
extern U8 utFindPosLSBit(U32 word);
extern U8 utCount1s(U32 value);
#define utCountByte1s(value) (utCount1sTable[value])
extern utHeapRef utfHeap (void);
extern utHeapRef utnHeap (utHeapRef Heap);
extern utHeapRef utcHeap (void);
extern void utdHeap (utHeapRef Heap);
extern void utFreeHeap (utHeapRef Heap);
extern void utaHeapBlock (utHeapRef Heap, utBlockRef Block);
extern void utdHeapBlock (utHeapRef Heap, utBlockRef Block);
extern utBlockRef utfBlock (void);
extern utBlockRef utnBlock (utBlockRef Block);
extern utBlockRef utcBlock (void);
extern void utdBlock (utBlockRef Block);
extern void utStop(void);
extern void utStart(void);
extern S32 utStrcmp (char *string1, char *string2);
extern S32 utStricmp (char *string1, char *string2);
extern char * utStrcpy (char *string1, char *string2);
extern char * utStrncpy (char *string1, char *string2, U32 length);
extern char * utStrcat (char *string1, char *string2);
extern char * utStrncat (char *string1, char *string2, U32 length);
extern U32 utStrlen (char *string);
extern void utStrupr (char *string);
extern void utStrlwr (char *string);
extern char *utReplaceSuffix(char *originalName, char *newSuffix);
extern char *utSuffix(char *name);
extern char *utBaseName(char *name);
extern char *utDirName(char *name);
extern bool utAccess(char *name, char *mode);
extern char *utExecPath(char *name);
extern char *utFindInPath(char *name, char *path);
extern char *utExpandEnvVariables(char *string);
extern bool utNameHasRange(char *name, U32 *left, U32 *right);
extern char *utFileName(char *name);
extern S32 utFindSubstring(char *string, char *substring);
extern char *utDoubleSlash(char *name);
extern void * utReallocTrace(void *memPtr, U32 numBlocks, U16 size, char *fileName,
    U32 line);
extern void * utMallocTrace (U32 sStruct, size_t size, char *fileName, U32 line);
extern void * utCallocTrace (U32 sStruct, size_t size, char *fileName, U32 line);
extern U16 utFreeTrace (void *memPtr, char *fileName, U32 line);
extern char *utVsprintf(char *format, va_list ap);
extern void utSetEnvironmentVariable(char *name, char *value);
extern char *utGetEnvironmentVariable(char *name);
extern char *utConvertDirSeparation(char *oldString);

/* Use this to get rid of warning messages about unreferenced labels */
#define utRefLabel(label) if(utUnlikely(utDebugVal + 1 == utDebugVal)) goto label;
#define utIfDebug(minVal) if(utUnlikely(utDebugVal >= minVal))
#define utIfVerbose(minVal) if(utUnlikely(utVerboseVal >= minVal))

#define UT_MAX_SETJMP_DEPTH 5
#if defined(_WINDOWS) && !defined(_WIN32)
#if !defined(CATCHBUF)
typedef int CATCHBUF[9];
#endif
extern CATCHBUF utCatchBuf[UT_MAX_SETJMP_DEPTH];
#else
extern jmp_buf utJmpBuf[UT_MAX_SETJMP_DEPTH];
#endif

extern S16 utSetjmpDepth;
extern U32 utSetjmpLine[UT_MAX_SETJMP_DEPTH];
extern char *utSetjmpFile[UT_MAX_SETJMP_DEPTH];
#define utUnsetjmp() (utSetjmpDepth--,\
      !utStrcmp(utSetjmpFile[utSetjmpDepth],__FILE__) ||\
      (utExit("Mismatched utUnsetjmp in file %s, line %u", __FILE__, __LINE__), 1))
#define utSetjmp() (++utSetjmpDepth,\
      utSetjmpFile[utSetjmpDepth - 1] = __FILE__,\
      utSetjmpLine[utSetjmpDepth - 1] = __LINE__,\
      setjmp(utJmpBuf[utSetjmpDepth - 1]))
extern void utLongjmp(void);

/*--------------------------------------------------------------------------------------------------
  Symbol table support.
--------------------------------------------------------------------------------------------------*/
struct _utSym {
    utSym lowerSym;
    utSym fHashedSym;
    U32 hashValue;
    U32 xName, nextDerivedIndex;
};
extern struct _utSym *utSyms;
extern U32 utsSym, utmSym;
extern char *utSymNames;
extern U32 utsSymName, utmSymName;
extern utSym *utSymTable;

extern utSym utSymCreate(char *name);
extern utSym utSymCreateFormatted(char *format, ...);
extern utSym utBuildUniqueSym(char *name, char *suffix);

#define utSymNull U32_MAX
#define utSymGetName(sym) (utSymNames + utSyms[sym].xName)
#define utSymGetLowerSym(sym) (utSyms[sym].lowerSym)
#define utSymGetHashValue(sym) (utSyms[sym].hashValue)
#define utSymGetNextDerivedIndex(sym) (utSyms[sym].nextDerivedIndex)
#define utSymGetfirstHashedSym(sym) (utSyms[sym].fHashedSym)
#define utSymGetXName(sym) (utSyms[sym].xName)
#define utxSym(xSym) (xSym)
#define utgxSym(sym) (sym)

#define utSetBit(x,index,bit) (x = ((x & (~(1 << index))) | (bit << index)))
/* ^^ This could be sped up. */
#define utGetBit(x,index) ((x >> index) & 1)
#define utInvertBit(x, index) (utSetBit(x, index, (1 - utGetBit(x, index))))

extern void ErrorNotify(char *string);

extern bool _utInitialized;
#define utInitialized() _utInitialized
extern void utInit(void);
extern void utClose(void);

typedef struct _utExtension *utExtension;
typedef struct _utFunctionTable *utFunctionTable;

struct _utExtension {
    utExtension next;
    utFunctionTable functionTable;
};

#define utExtensionNull NULL

struct _utFunctionTable {
    utSym className;
    void (*destroy)();
};

#define _utOffset(type, field) ((U32) ((char *) &((type *) 0)->field))

#if __cplusplus
}
#endif

#endif

