/*----------------------------------------------------------------------------
  Simple class support routines such as list manipulation.
----------------------------------------------------------------------------*/
#include "htddr.h"

void (*htHtblDestructorHook)(htHtbl);
htHtbl _htFirstFreeHtbl, _htTempHtbl;
utHeapRef _htHtblHeap;
void (*htEntryDestructorHook)(htEntry);
htEntry _htFirstFreeEntry, _htTempEntry;
utHeapRef _htEntryHeap;

/*----------------------------------------------------------------------------
  Add the Entry to the end of the array on the Htbl.
----------------------------------------------------------------------------*/
void htHtblInsertEntry(
    htHtbl Htbl,
    htEntry _Entry)
{
    if(htHtblGetusedEntrys(Htbl) == htHtblGetnumEntrys(Htbl)) {
        htHtblReallocEntrys(Htbl, htHtblGetnumEntrys(Htbl) +
            (htHtblGetnumEntrys(Htbl) >> 1) + 1);
    }
    htEntrySetHtblIndex(_Entry, htHtblGetusedEntrys(Htbl));
    htHtblSetiEntry(Htbl, htHtblGetusedEntrys(Htbl)++, _Entry);
}

/*----------------------------------------------------------------------------
  Remove the Entry from the array on the Htbl.
----------------------------------------------------------------------------*/
void htHtblRemoveEntry(
    htHtbl Htbl,
    htEntry _Entry)
{
    U32 xEntry = htEntryGetHtblIndex(_Entry);

    if(xEntry != U32_MAX) {
        htHtblSetiEntry(Htbl, xEntry, htHtblGetiEntry(Htbl,
            --htHtblGetusedEntrys(Htbl)));
        htEntrySetHtblIndex(_Entry, U32_MAX);
    }
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Htbl.
------------------------------------------------------------------------*/
static utBlockRef buildHtblBlock(void)
{
    utBlockRef block = utcBlock();
    htHtbl Htbl;
    U16 x, sHtbl;

    utaHeapBlock(_htHtblHeap, block);
    sHtbl = UTBLOCKSIZE/sizeof(struct _htHtbl);
    _htFirstFreeHtbl = (htHtbl)utgBlockMem(block);
    Htbl = _htFirstFreeHtbl;
    for(x = 0; x < sHtbl; x++) {
        _htHtblNextFree(Htbl) = Htbl + 1;
        Htbl++;
    }
    _htHtblNextFree(--Htbl) = htHtblNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Htbl.
----------------------------------------------------------------------------*/
htHtbl htHtblAlloc(void)
{
    htHtbl Htbl;

    if(_htFirstFreeHtbl == htHtblNull) {
        buildHtblBlock();
    }
    Htbl = _htFirstFreeHtbl;
    _htFirstFreeHtbl = _htHtblNextFree(Htbl);
    memset((void *)Htbl, 0, sizeof(struct _htHtbl));
    return Htbl;
}

/*----------------------------------------------------------------------------
  Free a Htbl.
----------------------------------------------------------------------------*/
void htHtblFree(
    htHtbl Htbl)
{
    if(htHtblGetnumEntrys(Htbl) != 0) {
        htHtblFreeEntrys(Htbl);
    }
    _htHtblNextFree(Htbl) = _htFirstFreeHtbl;
    _htFirstFreeHtbl = Htbl;
}

/*----------------------------------------------------------------------------
  Destroy a Htbl.
----------------------------------------------------------------------------*/
void htHtblDestroy(
    htHtbl Htbl)
{
    if(htHtblDestructorHook != NULL) {
        htHtblDestructorHook(Htbl);
    }
    {
        htEntry _Entry;
        htSafeForeachHtblEntry(Htbl, _Entry) {
            htEntryDestroy(_Entry);
        } htEndSafeForeachHtblEntry;
    }
    htHtblFree(Htbl);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Entry.
------------------------------------------------------------------------*/
static utBlockRef buildEntryBlock(void)
{
    utBlockRef block = utcBlock();
    htEntry Entry;
    U16 x, sEntry;

    utaHeapBlock(_htEntryHeap, block);
    sEntry = UTBLOCKSIZE/sizeof(struct _htEntry);
    _htFirstFreeEntry = (htEntry)utgBlockMem(block);
    Entry = _htFirstFreeEntry;
    for(x = 0; x < sEntry; x++) {
        _htEntryNextFree(Entry) = Entry + 1;
        Entry++;
    }
    _htEntryNextFree(--Entry) = htEntryNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Entry.
----------------------------------------------------------------------------*/
htEntry _htEntryAlloc(void)
{
    htEntry Entry;

    if(_htFirstFreeEntry == htEntryNull) {
        buildEntryBlock();
    }
    Entry = _htFirstFreeEntry;
    _htFirstFreeEntry = _htEntryNextFree(Entry);
    memset((void *)Entry, 0, sizeof(struct _htEntry));
    return Entry;
}

/*----------------------------------------------------------------------------
  Free a Entry.
----------------------------------------------------------------------------*/
void _htEntryFree(
    htEntry Entry)
{
    _htEntryNextFree(Entry) = _htFirstFreeEntry;
    _htFirstFreeEntry = Entry;
}

/*----------------------------------------------------------------------------
  Destroy a Entry.
----------------------------------------------------------------------------*/
void htEntryDestroy(
    htEntry Entry)
{
    if(htEntryDestructorHook != NULL) {
        htEntryDestructorHook(Entry);
    }
    if(htEntryGetEntry(Entry) != htEntryNull) {
        htEntryDestroy(htEntryGetEntry(Entry));
    }
    htEntryFree(Entry);
}

/*----------------------------------------------------------------------------
  Initialize memory for the ht&&.
----------------------------------------------------------------------------*/
void htDDRStart(void)
{
    _htFirstFreeHtbl = htHtblNull;
    _htHtblHeap = utcHeap();
    buildHtblBlock();
    _htFirstFreeEntry = htEntryNull;
    _htEntryHeap = utcHeap();
    buildEntryBlock();
}

/*----------------------------------------------------------------------------
  Free memory for the ht&&.
----------------------------------------------------------------------------*/
void htDDRStop(void)
{
    utFreeHeap(_htHtblHeap);
    utFreeHeap(_htEntryHeap);
}

