/* gtkcolorbutton.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This file is heavily inspired from the same one from the Gimp:
 * Gimp Color Button
 * Copyright (C) 1999 Sven Neumann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#undef GTK_DISABLE_DEPRECATED
#warning GTK_DISABLE_DEPRECATED

#include "giram.h"

#include "giramintl.h"

#include "gtkcolorbutton.h"

static void  gtk_color_button_destroy(GtkObject *object);
static void  gtk_color_button_clicked(GtkButton *button);
static void  gtk_color_button_paint(GtkColorButton *gcb);
static void  gtk_color_button_state_changed(GtkWidget *widget, GtkStateType previous_state);
static void  gtk_color_button_color_changed(GtkColorButton *gcb);

enum
{
  COLOR_CHANGED,
  LAST_SIGNAL
};

static guint gtk_color_button_signals[LAST_SIGNAL] = { 0 };


static GtkWidgetClass *parent_class = NULL;

/*************************************************************************
* gtk_color_button_destroy
**************************************************************************/
static void gtk_color_button_destroy(GtkObject *object)
{
  GtkColorButton *gcb;

  g_return_if_fail((gcb = GTK_COLOR_BUTTON(object)));

  if (gcb->title)
  {
    g_free(gcb->title);
    gcb->title = NULL;
  }

  if (gcb->gcs)
  {
    gtk_widget_destroy(gcb->gcs);
    gcb->gcs = NULL;
  }

  if (gcb->color)
  {
    g_free(gcb->color);
    gcb->color = NULL;
  }

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (* GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}

/*************************************************************************
*
**************************************************************************/
static void gtk_color_button_class_init(GtkColorButtonClass *class)
{
  GtkObjectClass *object_class;
  GtkButtonClass *button_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass*)class;
  button_class = (GtkButtonClass*)class;
  widget_class = (GtkWidgetClass*)class;

  parent_class = g_type_class_peek_parent(class);

  gtk_color_button_signals[COLOR_CHANGED] =
          g_signal_new("color_changed",
                       G_TYPE_FROM_CLASS(class),
                       G_SIGNAL_RUN_FIRST,
                       G_STRUCT_OFFSET(GtkColorButtonClass, color_changed),
                       NULL, NULL,
                       g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

  class->color_changed = NULL;

  object_class->destroy       = gtk_color_button_destroy;
  button_class->clicked       = gtk_color_button_clicked;
  widget_class->state_changed = gtk_color_button_state_changed;
}

/*************************************************************************
*
**************************************************************************/
static void gtk_color_button_init(GtkColorButton *gcb)
{
  gcb->title    = NULL;

  gcb->color    = NULL;
  gcb->preview  = NULL;
  gcb->gcs      = NULL;
}

/*************************************************************************
*
**************************************************************************/
GType gtk_color_button_get_type(void)
{
  static GType gcb_type = 0;

  if (!gcb_type)
  {
    static const GTypeInfo gcb_info =
    {
      sizeof(GtkColorButtonClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc)gtk_color_button_class_init,
      NULL, /* class_finalize */
      NULL, /* class_data */
      sizeof (GtkColorButton),
      0, /* n_preallocs */
      (GInstanceInitFunc)gtk_color_button_init
    };

    gcb_type = g_type_register_static (GTK_TYPE_BUTTON, "GtkColorButton", &gcb_info, 0);
  }
  return gcb_type;
}
/**
 * gtk_color_button_new:
 * @title: String that wil be used as title for the color_selector.
 * @width: Width of the colorpreview in pixels.
 * @height: Height of the colorpreview in pixels.
 * @color: An array of guchar holding the color (RGB or RGBA)
 * @bpp: May be 3 for RGB or 4 for RGBA.
 *
 * Creates a new GtkColorbutton widget. This returns a button with
 * a preview showing the color. When the button is clicked a
 * GtkColorSelectionDialog is opened. If the user changes the color
 * the new color is written into the array that was used to pass
 * the initial color and the "color_changed" signal is emitted.
 *
 * Returns: Pointer to the new GtkColorButton widget.
 */

GtkWidget *gtk_color_button_new(gchar *title,
                                gint width, gint height,
                                gdouble *color)
{
  GtkColorButton *gcb;
  gint i;

  g_return_val_if_fail(width > 0 && height > 0, NULL);

  gcb = g_object_new(GTK_TYPE_COLOR_BUTTON, NULL);

  gcb->title  = g_strdup(title);
  gcb->width  = width;
  gcb->height = height;

  gcb->color = g_new(gdouble, 5);

  if (color)
  {
    for (i = 0 ; i < 5 ; i++)
      gcb->color[i] = color[i];
  } else
  {
    for (i = 0 ; i < 5 ; i++)
      gcb->color[i] = 0.0;
  }

  gcb->preview = gtk_preview_new(GTK_PREVIEW_COLOR);
  g_signal_connect(G_OBJECT(gcb->preview), "destroy",
                   G_CALLBACK(gtk_widget_destroyed), &gcb->preview);

  gtk_preview_size(GTK_PREVIEW(gcb->preview), width, height);
  gtk_container_add(GTK_CONTAINER(gcb), gcb->preview);
  gtk_widget_show(gcb->preview);
  gtk_color_button_paint(gcb);

  return (GTK_WIDGET(gcb));
}

/*
 * gtk_color_button_update:
 * @gcb: Pointer to a #GimpColorButton.
 *
 * Should be used after the color controlled by a #GimpColorButton
 * was changed. The color is then reread and the change is propagated
 * to the preview and the GtkColorSelectionDialog if one is open.
 */
void gtk_color_button_update(GtkColorButton *gcb)
{
  g_return_if_fail(GTK_IS_COLOR_BUTTON(gcb));

  gtk_color_button_paint(gcb);

  if (gcb->gcs)
    gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->colorsel), gcb->color);
}

static void gtk_color_button_state_changed(GtkWidget *widget,
                                           GtkStateType previous_state)
{
  g_return_if_fail (GTK_IS_COLOR_BUTTON (widget));

  if (!GTK_WIDGET_IS_SENSITIVE(widget) && GTK_COLOR_BUTTON(widget)->gcs)
    gtk_widget_hide(GTK_COLOR_BUTTON(widget)->gcs);

  if (GTK_WIDGET_CLASS(parent_class)->state_changed)
    (* GTK_WIDGET_CLASS(parent_class)->state_changed)(widget, previous_state);
}

static void gtk_color_button_clicked(GtkButton *button)
{
  GtkColorButton *gcb;

  g_return_if_fail (GTK_IS_COLOR_BUTTON(button));
  gcb = GTK_COLOR_BUTTON(button);

  if (!gcb->gcs)
  {
    gcb->gcs = gtk_color_selection_dialog_new(_("Color Selection"));
    g_signal_connect(G_OBJECT(gcb->gcs), "delete_event",
                     G_CALLBACK(gtk_widget_destroy), NULL);
    g_signal_connect(G_OBJECT(gcb->gcs), "destroy",
                     G_CALLBACK(gtk_widget_destroyed), &gcb->gcs);
    g_signal_connect_swapped(G_OBJECT(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->colorsel),
                             "color_changed",
                             G_CALLBACK(gtk_color_button_color_changed), gcb);
    gtk_window_position(GTK_WINDOW(gcb->gcs), GTK_WIN_POS_MOUSE);
    gtk_widget_hide(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->help_button);
    gtk_widget_hide(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->cancel_button);
    g_signal_connect_swapped(G_OBJECT(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->ok_button),
                             "clicked", G_CALLBACK(gtk_widget_destroy),
                             gcb->gcs);
  }
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->colorsel), gcb->color);
  gtk_widget_show(gcb->gcs);
}

static void gtk_color_button_paint(GtkColorButton *gcb)
{
  gint x, y, i;
  guchar *p0, *row;

  g_return_if_fail(GTK_IS_COLOR_BUTTON(gcb));

  p0 = row = g_new(guchar, 3*gcb->width);
  for (x = 0; x < gcb->width; x++)
  {
    for (i = 0; i < 3; i++)
      *p0++ = gcb->color[i] * 255.0;
  }
  for (y = 0; y < gcb->height; y++)
    gtk_preview_draw_row(GTK_PREVIEW(gcb->preview), row, 0, y,
                         gcb->width);
  gtk_widget_queue_draw (gcb->preview);
  g_free(row);
}

static void gtk_color_button_color_changed(GtkColorButton *gcb)
{
  gint               i;
  GtkColorSelection *colorsel;
  gdouble            color[5];

  colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->colorsel);
  gtk_color_selection_get_color(colorsel, color);
  for (i=0 ; i<5 ; i++)
    gcb->color[i] = color[i];
  gtk_color_button_paint(gcb);
  gtk_signal_emit(GTK_OBJECT(gcb),  gtk_color_button_signals[COLOR_CHANGED]);
}

void gtk_color_button_set_color(GtkColorButton *gcb,
                                gdouble        *color)
{
  GtkColorSelection *colorsel;
  gint               i;

  for (i=0 ; i<5 ; i++)
       gcb->color[i] = color[i];
  gtk_color_button_paint(gcb);
  if (gcb->gcs)
  {
    colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(gcb->gcs)->colorsel);
    gtk_color_selection_set_color(colorsel, color);
  }
}

