/* Giram: a simple 3D modeller
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * giramcolorlist.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <math.h>

#undef GTK_DISABLE_DEPRECATED
#warning GTK_DISABLE_DEPRECATED

#include <gtk/gtk.h>
#include "giramcolorlist.h"
#include "giramintl.h"

#include "girampixmap.h"
#include "pixmaps/new.xpm"
#include "pixmaps/duplicate.xpm"
#include "pixmaps/edit.xpm"
#include "pixmaps/delete.xpm"
#include "pixmaps/refresh.xpm"

enum
{
  COLOR_CHANGED,
  LAST_SIGNAL
};

static void giram_color_list_class_init (GiramColorListClass *class);
static void giram_color_list_init       (GiramColorList      *gcl);
static void giram_color_list_destroy    (GtkObject           *object);
static void giram_color_list_map        (GtkWidget           *widget);

static guint giram_color_list_signals[LAST_SIGNAL] = { 0 };

static GtkWindowClass *parent_class = NULL;

GtkType giram_color_list_get_type(void)
{
  static GtkType gcl_type = 0;

  if (!gcl_type)
  {
    GtkTypeInfo gcl_info =
    {
      "GiramColorList",
      sizeof(GiramColorList),
      sizeof(GiramColorListClass),
      (GtkClassInitFunc)giram_color_list_class_init,
      (GtkObjectInitFunc)giram_color_list_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc)NULL
    };

    gcl_type = gtk_type_unique(gtk_window_get_type(), &gcl_info);
  }

  return gcl_type;
}

static void giram_color_list_class_init(GiramColorListClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass *)class;
  widget_class = (GtkWidgetClass *)class;

  parent_class = gtk_type_class(gtk_window_get_type());
#if 0
  giram_color_list_signals[COLOR_CHANGED] =
                    gtk_signal_new("color_changed",
                                   GTK_RUN_FIRST,
                                   object_class->type,
                                   GTK_SIGNAL_OFFSET(GiramColorListClass,
                                                     color_changed),
                                   gtk_marshal_NONE__POINTER,
                                   GTK_TYPE_NONE, 1,
                                   GTK_TYPE_POINTER);
#endif
  //gtk_object_class_add_signals(object_class, giram_color_list_signals, LAST_SIGNAL);

  class->color_changed = NULL;

  object_class->destroy = giram_color_list_destroy;
  widget_class->map     = giram_color_list_map;
}

static gint giram_color_list_add_color(gchar          *name,
                                       gdouble        *color,
                                       GiramColorList *gcl)
{
  gchar     *text[2];
  GdkPixmap *preview;
  gint       row;
  guchar     rgb_image[3*50*15];
  gint       x, y;
  GdkGC     *gc;

  text[0] = "";
  text[1] = name;
  row = gtk_clist_append(GTK_CLIST(gcl->clist), text);
  preview = gdk_pixmap_new(GTK_WIDGET(gcl->clist)->window,
                           50, 15, -1);
  for (y=0 ; y<15 ; y++)
    for (x=0 ; x<50 ; x++)
    {
      rgb_image[3*(y*50+x)+0] = 255*color[0];
      rgb_image[3*(y*50+x)+1] = 255*color[1];
      rgb_image[3*(y*50+x)+2] = 255*color[2];
    }
  gc = gdk_gc_new(GTK_WIDGET(gcl->clist)->window);
  gdk_draw_rgb_image(preview, gc, 0, 0, 50, 15,
                     GDK_RGB_DITHER_NORMAL,
                     rgb_image, 3*50);
  gdk_gc_unref(gc);
  gtk_clist_set_pixmap(GTK_CLIST(gcl->clist), row, 0,
                       preview, NULL);
  gdk_drawable_unref(preview);
  return row;
}

static void giram_color_list_new_color(GiramColorList *gcl)
{
  gdouble *new_color;
  gchar   *name;
  gint     row;

  name = g_strdup(_("New Color"));
  if (g_hash_table_lookup(gcl->hash, name))
  { /* "New Color" already exists! */
    gint i;

    i = 1;
    do
    {
      g_free(name);
      name = g_strdup_printf(_("New Color #%d"), i);
      i++;
    } while (g_hash_table_lookup(gcl->hash, name));
  }
  new_color = g_new(gdouble, 5);
  new_color[0] = new_color[1] = new_color[2] = 1.0;
  new_color[3] = new_color[4] = 0.0;
  g_hash_table_insert(gcl->hash,
                      name,
                      new_color);
  row = giram_color_list_add_color(name, new_color, gcl);
  gtk_clist_select_row(GTK_CLIST(gcl->clist), row, 0);
  gtk_clist_moveto(GTK_CLIST(gcl->clist), row, 0, 0.5, 0.5);
}

static void giram_color_list_duplicate_color(GiramColorList *gcl)
{
  gint     row;
  gchar   *color_name, *new_name;
  gdouble *color, *new_color;

  row = GPOINTER_TO_INT(GTK_CLIST(gcl->clist)->selection->data);
  gtk_clist_get_text(GTK_CLIST(gcl->clist), row, 1, &color_name);
  color = g_hash_table_lookup(gcl->hash, color_name);

  new_name = g_strdup_printf(_("%s Copy"), color_name);
  if (g_hash_table_lookup(gcl->hash, new_name))
  { /* "%s Copy" already exists! */
    gint i;

    i = 1;
    do
    {
      g_free(new_name);
      new_name = g_strdup_printf(_("%s Copy #%d"), color_name, i);
      i++;
    } while (g_hash_table_lookup(gcl->hash, new_name));
  }
  new_color = g_new(gdouble, 5);
  new_color[0] = color[0];
  new_color[1] = color[1];
  new_color[2] = color[2];
  new_color[3] = color[3];
  new_color[4] = color[4];
  g_hash_table_insert(gcl->hash, new_name, new_color);
  row = giram_color_list_add_color(new_name, new_color, gcl);
  gtk_clist_select_row(GTK_CLIST(gcl->clist), row, 0);
  gtk_clist_moveto(GTK_CLIST(gcl->clist), row, 0, 0.5, 0.5);
}

static void giram_color_list_color_sel_ok(GiramColorList *gcl)
{
  gtk_widget_destroy(gcl->color_sel);
}

static void giram_color_list_color_sel_cancel(GiramColorList *gcl)
{
  gtk_widget_destroy(gcl->color_sel);
}

static void giram_color_list_colorsel_color_changed(GiramColorList    *gcl,
                                                    GtkColorSelection *colorsel)
{
  gint       row;
  gchar     *color_name;
  gdouble   *color;
  GdkPixmap *preview;
  guchar     rgb_image[3*50*15];
  gint       x, y;
  GdkGC     *gc;

  row = GPOINTER_TO_INT(GTK_CLIST(gcl->clist)->selection->data);

  gtk_clist_get_text(GTK_CLIST(gcl->clist), row, 1, &color_name);

  color = g_hash_table_lookup(gcl->hash, color_name);

  gtk_color_selection_get_color(colorsel, color);

  preview = gdk_pixmap_new(GTK_WIDGET(gcl->clist)->window,
                           50, 15, -1);
  for (y=0 ; y<15 ; y++)
    for (x=0 ; x<50 ; x++)
    {
      rgb_image[3*(y*50+x)+0] = 255*color[0];
      rgb_image[3*(y*50+x)+1] = 255*color[1];
      rgb_image[3*(y*50+x)+2] = 255*color[2];
    }
  gc = gdk_gc_new(GTK_WIDGET(gcl->clist)->window);
  gdk_draw_rgb_image(preview, gc, 0, 0, 50, 15,
                     GDK_RGB_DITHER_NORMAL,
                     rgb_image, 3*50);
  gdk_gc_unref(gc);
  gtk_clist_set_pixmap(GTK_CLIST(gcl->clist), row, 0,
                       preview, NULL);
  gdk_drawable_unref(preview);
  gtk_signal_emit(GTK_OBJECT(gcl),
                  giram_color_list_signals[COLOR_CHANGED],
                  color);
}

static void giram_color_list_edit_color(GiramColorList *gcl)
{
  GtkColorSelection *colorsel;
  gint               row;
  gchar             *color_name;
  gdouble           *color;

  if (!gcl->color_sel)
  {
    gcl->color_sel = gtk_color_selection_dialog_new(_("Edit color"));
    colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(gcl->color_sel)->colorsel);
    //gtk_color_selection_set_opacity(colorsel, FALSE);
    gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(gcl->color_sel)->help_button);
    gtk_signal_connect(GTK_OBJECT(gcl->color_sel), "destroy",
                       (GtkSignalFunc)gtk_widget_destroyed, &gcl->color_sel);
    gtk_signal_connect_object(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(gcl->color_sel)->ok_button),
                              "clicked",
                              (GtkSignalFunc)giram_color_list_color_sel_ok,
                              GTK_OBJECT(gcl));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(gcl->color_sel)->cancel_button),
                              "clicked",
                              (GtkSignalFunc)giram_color_list_color_sel_cancel,
                              GTK_OBJECT(gcl));
    gtk_window_position(GTK_WINDOW(gcl->color_sel), GTK_WIN_POS_MOUSE);
    gtk_signal_connect_object(GTK_OBJECT(colorsel), "color_changed",
                              (GtkSignalFunc)giram_color_list_colorsel_color_changed,
                              GTK_OBJECT(gcl));
  }
  gtk_widget_show_now(gcl->color_sel);
  colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(gcl->color_sel)->colorsel);

  row = GPOINTER_TO_INT(GTK_CLIST(gcl->clist)->selection->data);

  gtk_clist_get_text(GTK_CLIST(gcl->clist), row, 1, &color_name);

  color = g_hash_table_lookup(gcl->hash, color_name);

  gtk_color_selection_set_color(colorsel, color);
}

static void giram_color_list_delete_color(GiramColorList *gcl)
{
  gint   row;
  gchar *color_name;

  row = GPOINTER_TO_INT(GTK_CLIST(gcl->clist)->selection->data);
  gtk_clist_get_text(GTK_CLIST(gcl->clist), row, 1, &color_name);

  g_hash_table_remove(gcl->hash, color_name);

  gtk_clist_remove(GTK_CLIST(gcl->clist), row);
}

static void giram_color_list_refresh(GiramColorList *gcl)
{
  gtk_clist_clear(GTK_CLIST(gcl->clist));
  g_hash_table_foreach(gcl->hash,
                       (GHFunc)giram_color_list_add_color,
                       gcl);
}

static void giram_color_list_select_color(GiramColorList *gcl,
                                          gint            row)
{
  gchar   *color_name;
  gdouble *color;

  gtk_clist_get_text(GTK_CLIST(gcl->clist), row, 1, &color_name);
  color = g_hash_table_lookup(gcl->hash, color_name);
  if (gcl->color_sel)
  {
    GtkWidget *colorsel;

    colorsel = GTK_COLOR_SELECTION_DIALOG(gcl->color_sel)->colorsel;
    gtk_signal_handler_block_by_func(GTK_OBJECT(colorsel),
                                     (GtkSignalFunc)giram_color_list_colorsel_color_changed,
                                     gcl);
    gtk_color_selection_set_color(GTK_COLOR_SELECTION(colorsel), color);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(colorsel),
                                       (GtkSignalFunc)giram_color_list_colorsel_color_changed,
                                       gcl);
  }
  gtk_signal_emit(GTK_OBJECT(gcl),
                  giram_color_list_signals[COLOR_CHANGED],
                  color);
  gtk_widget_set_sensitive(gcl->duplicate_button, TRUE);
  gtk_widget_set_sensitive(gcl->edit_button, TRUE);
  gtk_widget_set_sensitive(gcl->delete_button, TRUE);
}

static void giram_color_list_init(GiramColorList *gcl)
{
  GtkWidget *vbox, *scr, *hbox, *button, *pixmap;
  GtkWidget *separator;
  gchar     *titles[2] = {"Color", "Name" };

  gtk_widget_set_usize(GTK_WIDGET(gcl), 0, 200);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gcl), vbox);
  gtk_widget_show(vbox);

  scr = gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), scr);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scr),
                                 GTK_POLICY_NEVER,
                                 GTK_POLICY_ALWAYS);
  gtk_widget_show(scr);

  gcl->clist = gtk_clist_new_with_titles(2, titles);
  gtk_clist_set_selection_mode(GTK_CLIST(gcl->clist), GTK_SELECTION_BROWSE);
  gtk_container_add(GTK_CONTAINER(scr), gcl->clist);
  gtk_clist_set_sort_column(GTK_CLIST(gcl->clist), 1);
  gtk_clist_set_auto_sort(GTK_CLIST(gcl->clist), TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gcl->clist), 0, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gcl->clist), 1, TRUE);
  gtk_clist_set_row_height(GTK_CLIST(gcl->clist), 20);
  gtk_signal_connect_object(GTK_OBJECT(gcl->clist), "select_row",
                            (GtkSignalFunc)giram_color_list_select_color,
                            GTK_OBJECT(gcl));
  gtk_widget_show(gcl->clist);

  hbox = gtk_hbox_new(TRUE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  /* 'new' button */
  gcl->new_button = gtk_button_new();
  gtk_box_pack_start_defaults(GTK_BOX(hbox), gcl->new_button);
  pixmap = giram_pixmap_new(new_xpm);
  gtk_container_add(GTK_CONTAINER(gcl->new_button), pixmap);
  gtk_widget_show(pixmap);
  gtk_signal_connect_object(GTK_OBJECT(gcl->new_button), "clicked",
                            (GtkSignalFunc)giram_color_list_new_color,
                            GTK_OBJECT(gcl));
  gtk_widget_show(gcl->new_button);

  /* button 'duplicate' */
  gcl->duplicate_button = gtk_button_new();
  gtk_box_pack_start_defaults(GTK_BOX(hbox), gcl->duplicate_button);
  pixmap = giram_pixmap_new(duplicate_xpm);
  gtk_widget_show(pixmap);
  gtk_container_add(GTK_CONTAINER(gcl->duplicate_button), pixmap);
  gtk_signal_connect_object(GTK_OBJECT(gcl->duplicate_button), "clicked",
                            (GtkSignalFunc)giram_color_list_duplicate_color,
                            GTK_OBJECT(gcl));
  gtk_widget_show(gcl->duplicate_button);
  gtk_widget_set_sensitive(gcl->duplicate_button, FALSE);

  /* button 'edit' */
  gcl->edit_button = gtk_button_new();
  gtk_box_pack_start_defaults(GTK_BOX(hbox), gcl->edit_button);
  pixmap = giram_pixmap_new(edit_xpm);
  gtk_widget_show(pixmap);
  gtk_container_add(GTK_CONTAINER(gcl->edit_button), pixmap);
  gtk_signal_connect_object(GTK_OBJECT(gcl->edit_button), "clicked",
                            (GtkSignalFunc)giram_color_list_edit_color,
                            GTK_OBJECT(gcl));
  gtk_widget_show(gcl->edit_button);
  gtk_widget_set_sensitive(gcl->edit_button, FALSE);

  /* button 'delete' */
  gcl->delete_button = gtk_button_new();
  gtk_box_pack_start_defaults(GTK_BOX(hbox), gcl->delete_button);
  pixmap = giram_pixmap_new(delete_xpm);
  gtk_widget_show(pixmap);
  gtk_container_add(GTK_CONTAINER(gcl->delete_button), pixmap);
  gtk_signal_connect_object(GTK_OBJECT(gcl->delete_button), "clicked",
                            (GtkSignalFunc)giram_color_list_delete_color,
                            GTK_OBJECT(gcl));
  gtk_widget_show(gcl->delete_button);
  gtk_widget_set_sensitive(gcl->delete_button, FALSE);

  /* button 'refresh' */
  gcl->refresh_button = gtk_button_new();
  gtk_box_pack_start_defaults(GTK_BOX(hbox), gcl->refresh_button);
  pixmap = giram_pixmap_new(refresh_xpm);
  gtk_widget_show(pixmap);
  gtk_container_add(GTK_CONTAINER(gcl->refresh_button), pixmap);
  gtk_signal_connect_object(GTK_OBJECT(gcl->refresh_button), "clicked",
                            (GtkSignalFunc)giram_color_list_refresh,
                            GTK_OBJECT(gcl));
  gtk_widget_show(gcl->refresh_button);

  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);
  gtk_widget_show(separator);

  /* 'cancel' button */
  button = gtk_button_new_with_label(_("Close"));
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                            (GtkSignalFunc)gtk_widget_destroy,
                            GTK_OBJECT(gcl));
  gtk_widget_show(button);
}

GtkWidget *giram_color_list_new(GHashTable *hash)
{
  GiramColorList *gcl;

  gcl = gtk_type_new(giram_color_list_get_type());

  gcl->hash = hash;
  gcl->color_sel = NULL;

  return GTK_WIDGET(gcl);
}

static void giram_color_list_destroy(GtkObject *object)
{
  GiramColorList *gcl = GIRAM_COLOR_LIST(object);

  if (gcl->color_sel)
  {
    gtk_widget_destroy(gcl->color_sel);
    gcl->color_sel = NULL;
  }
}

static void giram_color_list_map(GtkWidget *widget)
{
  GiramColorList *gcl = GIRAM_COLOR_LIST(widget);

  if (GTK_WIDGET_CLASS(parent_class)->map)
    (*GTK_WIDGET_CLASS(parent_class)->map)(widget);
  g_hash_table_foreach(gcl->hash,
                       (GHFunc)giram_color_list_add_color,
                       gcl);
}

typedef struct _GHashNode GHashNode;

struct _GHashNode
{
  gpointer   key;
  gpointer   value;
  GHashNode *next;
};

struct _GHashTable
{
          gint size;
            gint nnodes;
              guint frozen;
                GHashNode **nodes;
                  GHashFunc hash_func;
                    GCompareFunc key_compare_func;
};


void giram_color_list_set_color(GiramColorList *gcl,
                                gdouble *color)
{
  GHashNode *node; /* FIXME: shouldn't use this here: use foreach instead */
  gint       i;
  gdouble   *node_color;
  gboolean   done;

  done = FALSE;
  for (i = 0; i < gcl->hash->size; i++)
    for (node = gcl->hash->nodes[i]; node; node = node->next)
    {
      node_color = node->value;
      if ( (fabs(node_color[0]-color[0]) < 1e-6) &&
           (fabs(node_color[1]-color[1]) < 1e-6) &&
           (fabs(node_color[2]-color[2]) < 1e-6) )
      {
        gchar *color_name;
        gint row;

        done = TRUE;
        color_name = node->key;
        for (row = 0; row < GTK_CLIST(gcl->clist)->rows ; row++)
        {
          gchar *text;
          gtk_clist_get_text(GTK_CLIST(gcl->clist), row, 1, &text);
          if (!strcmp(color_name, text))
          {
            gtk_clist_select_row(GTK_CLIST(gcl->clist), row, 1);
            gtk_clist_moveto(GTK_CLIST(gcl->clist), row, 0, 0.5, 0.5);
            return;
          }
        }
      }
    }
  if (!done)
  {
    gdouble *new_color;
    gchar *name;
    gint row;

    name = g_strdup(_("New Color"));
    if (g_hash_table_lookup(gcl->hash, name))
    { /* "New Color" already exists! */
      gint i;

      i = 1;
      do
      {
        g_free(name);
        name = g_strdup_printf(_("New Color #%d"), i);
        i++;
      } while (g_hash_table_lookup(gcl->hash, name));
    }
    new_color = g_new(gdouble, 5);
    new_color[0] = color[0];
    new_color[1] = color[1];
    new_color[2] = color[2];
    new_color[3] = color[3];
    new_color[4] = color[4];
    g_hash_table_insert(gcl->hash,
                        name,
                        new_color);
    if (GTK_WIDGET_MAPPED(gcl))
    {
      row = giram_color_list_add_color(name, new_color, gcl);
      gtk_clist_select_row(GTK_CLIST(gcl->clist), row, 0);
      gtk_clist_moveto(GTK_CLIST(gcl->clist), row, 0, 0.5, 0.5);
    }
  }
}

