/* viewZY.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "giram.h"
#include "object.h"
#include "camera.h"
#include "utils.h"
#include "tools/tools.h"

#include "viewZY.h"

/****************************************************************************
*  draw_snap_grid_ZY
*****************************************************************************/
static void draw_snap_grid_ZY(GtkWidget *canvas, GdkDrawable *drawable, gint width, gint height)
{
  double       x, y;
  double       Zoom, Xoff, Yoff;
  double       StartX, StartY;
  double       EndX, EndY;
  double       Xinc, Yinc;
  ViewStruct  *view_data;
  FrameStruct *Frame;

  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  Frame = view_data->frame;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  if (!Frame->ShowGrid)
    return;

  Xinc = Frame->SnapGrid[2];
  Yinc = Frame->SnapGrid[1];
  StartX = Xoff - width / 2.0 / Zoom;
  StartY = Yoff - height / 2.0 / Zoom;

  StartX = Xinc*((int)(StartX / Xinc));
  StartY = Yinc*((int)(StartY / Yinc));
  EndX = width / 2.0 / Zoom;
  EndY = height / 2.0 / Zoom;

  for (x=StartX; x<EndX ; x+=Xinc)
  {
    for (y=StartY ; y<EndY ; y+=Yinc)
    {
      gdk_draw_point(drawable, giram_white_gc,
                     floor(width  / 2.0 + (x-Xoff) * Zoom),
                     floor(height / 2.0 - (y-Yoff) * Zoom) );
    }
  }
}

/*****************************************************************************
*  DrawCenterZY
******************************************************************************/
void DrawCenterZY(GtkWidget *canvas, double Xoff, double Yoff, double Zoom,
                  Vector Center)
{
  gdk_draw_line(canvas->window, giram_green_gc,
                canvas->allocation.width/2+(Center[2]-Xoff)*Zoom-10,
                canvas->allocation.height/2-(Center[1]-Yoff)*Zoom-10,
                canvas->allocation.width/2+(Center[2]-Xoff)*Zoom+10,
                canvas->allocation.height/2-(Center[1]-Yoff)*Zoom+10);
  gdk_draw_line(canvas->window, giram_green_gc,
                canvas->allocation.width/2+(Center[2]-Xoff)*Zoom-10,
                canvas->allocation.height/2-(Center[1]-Yoff)*Zoom+10,
                canvas->allocation.width/2+(Center[2]-Xoff)*Zoom+10,
                canvas->allocation.height/2-(Center[1]-Yoff)*Zoom-10);
}

/******************************************************************************
*  draw_camera_zy
*******************************************************************************/
static void draw_camera_zy(GtkWidget *canvas)
{
  gint          width, height;
  ViewStruct   *view_data;
  CameraStruct *camera;
  gdouble       Zoom, Xoff, Yoff;

  width  = canvas->allocation.width;
  height = canvas->allocation.height;
  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  camera = view_data->frame->all_cameras->data;

  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->Location[2]-Xoff)*Zoom,
                height/2-(camera->Location[1]-Yoff)*Zoom,
                width /2+(camera->P00[2]-Xoff)*Zoom,
                height/2-(camera->P00[1]-Yoff)*Zoom);
  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->Location[2]-Xoff)*Zoom,
                height/2-(camera->Location[1]-Yoff)*Zoom,
                width /2+(camera->P01[2]-Xoff)*Zoom,
                height/2-(camera->P01[1]-Yoff)*Zoom);
  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->Location[2]-Xoff)*Zoom,
                height/2-(camera->Location[1]-Yoff)*Zoom,
                width /2+(camera->P10[2]-Xoff)*Zoom,
                height/2-(camera->P10[1]-Yoff)*Zoom);
  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->Location[2]-Xoff)*Zoom,
                height/2-(camera->Location[1]-Yoff)*Zoom,
                width /2+(camera->P11[2]-Xoff)*Zoom,
                height/2-(camera->P11[1]-Yoff)*Zoom);
  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->P00[2]-Xoff)*Zoom,
                height/2-(camera->P00[1]-Yoff)*Zoom,
                width /2+(camera->P01[2]-Xoff)*Zoom,
                height/2-(camera->P01[1]-Yoff)*Zoom);
  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->P01[2]-Xoff)*Zoom,
                height/2-(camera->P01[1]-Yoff)*Zoom,
                width /2+(camera->P11[2]-Xoff)*Zoom,
                height/2-(camera->P11[1]-Yoff)*Zoom);
  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->P11[2]-Xoff)*Zoom,
                height/2-(camera->P11[1]-Yoff)*Zoom,
                width /2+(camera->P10[2]-Xoff)*Zoom,
                height/2-(camera->P10[1]-Yoff)*Zoom);
  gdk_draw_line(canvas->window, giram_yellow_gc,
                width /2+(camera->P10[2]-Xoff)*Zoom,
                height/2-(camera->P10[1]-Yoff)*Zoom,
                width /2+(camera->P00[2]-Xoff)*Zoom,
                height/2-(camera->P00[1]-Yoff)*Zoom);
}

/*****************************************************************************
*  ExposeWireFrameViewZY
******************************************************************************/
static void ExposeWireFrameViewZY(GtkWidget *canvas, GdkEventExpose *ev)
{
  ObjectStruct       *Tmp;
  LightSourceStruct  *TmpLight;
  TriangleListStruct *TmpTri;
  double              Zoom, Xoff, Yoff;
  ViewStruct         *view_data;
  GdkGC              *gc;
  GSList             *tmp_list;
  gint                width, height;

  gdk_window_set_background(canvas->window, &(canvas->style->black));

  gdk_window_clear(canvas->window);
  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  width  = canvas->allocation.width;
  height = canvas->allocation.height;

  for (tmp_list = view_data->frame->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    Tmp = tmp_list->data;
    if (Tmp->visible)
    {
      if (Tmp->selected)
        gc = giram_red_gc;
      else
        gc = giram_white_gc;
      for (TmpTri = Tmp->FirstTriangle ; TmpTri ; TmpTri = TmpTri->Next)
      {
        gdk_draw_line(canvas->window, gc,
                      width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P1[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P2[1]-Yoff)*Zoom);
        gdk_draw_line(canvas->window, gc,
                      width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P2[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P3[1]-Yoff)*Zoom);
        gdk_draw_line(canvas->window, gc,
                      width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P3[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P1[1]-Yoff)*Zoom);
      }
    } /* if (Tmp->visible) */
  }
  /* Draw the Light Sources */
  for (tmp_list = view_data->frame->all_light_sources ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list) )
  {
    TmpLight = tmp_list->data;
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10);
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10);
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10);
  }
  /* Draw the Rotation/Scale Center if any... */
  if (current_tool->ID == MT_ROTATE)
  {
    DrawCenterZY(canvas, Xoff, Yoff, Zoom, view_data->frame->RotationCenter);
  }
  if (current_tool->ID == MT_SCALE)
  {
    DrawCenterZY(canvas, Xoff, Yoff, Zoom, view_data->frame->ScaleCenter);
  }
  /* Draw the camera */
  draw_camera_zy(canvas);
  /* Draw the grid */
  draw_snap_grid_ZY(canvas, canvas->window, width, height);
}

/*****************************************************************************
*  ExposeHiddenFacesViewZY
******************************************************************************/
static void ExposeHiddenFacesViewZY(GtkWidget *canvas, GdkEventExpose *ev)
{
  ObjectStruct       *Tmp;
  LightSourceStruct  *TmpLight;
  TriangleListStruct *TmpTri;
  double              Zoom, Xoff, Yoff;
  ViewStruct         *view_data;
  GdkGC              *gc;
  Triangle2D         *TabTri;
  int                 nbTri;
  register int        i;
  GdkPoint            points[3];
  int                 max_triangles;
  GSList             *tmp_list;
  gint                width, height;

  gdk_window_set_background(canvas->window, &(canvas->style->black));

  gdk_window_clear(canvas->window);
  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  width  = canvas->allocation.width;
  height = canvas->allocation.height;
  nbTri = 0;
  max_triangles = 256;
  TabTri = g_new(Triangle2D, max_triangles);
  for (tmp_list = view_data->frame->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    Tmp = tmp_list->data;
    if (Tmp->visible)
    {
      for (TmpTri = Tmp->FirstTriangle ; TmpTri ; TmpTri = TmpTri->Next)
      {
        if (nbTri >= max_triangles)
        {
          max_triangles *= 2;
          TabTri = g_renew(Triangle2D, TabTri, max_triangles);
        }
        TabTri[nbTri].P1[0] = width/2+(TmpTri->P1[2]-Xoff)*Zoom;
        TabTri[nbTri].P1[1] = height/2-(TmpTri->P1[1]-Yoff)*Zoom;
        TabTri[nbTri].P2[0] = width/2+(TmpTri->P2[2]-Xoff)*Zoom;
        TabTri[nbTri].P2[1] = height/2-(TmpTri->P2[1]-Yoff)*Zoom;
        TabTri[nbTri].P3[0] = width/2+(TmpTri->P3[2]-Xoff)*Zoom;
        TabTri[nbTri].P3[1] = height/2-(TmpTri->P3[1]-Yoff)*Zoom;
        TabTri[nbTri].selected = Tmp->selected;
        TabTri[nbTri].distance = -(TmpTri->P1[0]+TmpTri->P2[0]+TmpTri->P3[0]);
        nbTri++;
      }
    } /* if (Tmp->Visible) */
  }
  /* Sorting ... */
  qsort(TabTri, nbTri, sizeof(Triangle2D), comparTri);
  /* Displaying */
  for (i=0 ; i<nbTri ; i++)
  {
    points[0].x=TabTri[i].P1[0]; points[0].y=TabTri[i].P1[1];
    points[1].x=TabTri[i].P2[0]; points[1].y=TabTri[i].P2[1];
    points[2].x=TabTri[i].P3[0]; points[2].y=TabTri[i].P3[1];
    gdk_draw_polygon(canvas->window, giram_black_gc, TRUE, points, 3);
    if (TabTri[i].selected)
      gc = giram_red_gc;
    else
      gc = giram_white_gc;
    gdk_draw_line(canvas->window, gc, points[0].x, points[0].y,
                                      points[1].x, points[1].y);
    gdk_draw_line(canvas->window, gc, points[1].x, points[1].y,
                                      points[2].x, points[2].y);
    gdk_draw_line(canvas->window, gc, points[2].x, points[2].y,
                                      points[0].x, points[0].y);
  }
  /* Freeing ... */
  g_free(TabTri);
  /* Draw the Light Sources */
  for (tmp_list = view_data->frame->all_light_sources ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list) )
  {
    TmpLight = tmp_list->data;
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10);
    gdk_draw_line(canvas->window, giram_white_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10);
    gdk_draw_line(canvas->window, giram_white_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10);
  }
  /* Draw the Rotation/Scale Center if any... */
  if (current_tool->ID == MT_ROTATE)
  {
    DrawCenterZY(canvas, Xoff, Yoff, Zoom, view_data->frame->RotationCenter);
  }
  if (current_tool->ID == MT_SCALE)
  {
    DrawCenterZY(canvas, Xoff, Yoff, Zoom, view_data->frame->ScaleCenter);
  }
  /* Draw the camera */
  draw_camera_zy(canvas);
  /* Draw the grid */
  draw_snap_grid_ZY(canvas, canvas->window, width, height);
}

#define AMBIENT 0.1
#define DIFFUSE 0.6
/*****************************************************************************
*  ExposeGouraudViewZY
******************************************************************************/
static void ExposeGouraudViewZY(GtkWidget *canvas, GdkEventExpose *ev)
{
  ObjectStruct       *Tmp;
  LightSourceStruct  *TmpLight;
  TriangleListStruct *TmpTri;
  double              Zoom, Xoff, Yoff;
  ViewStruct         *view_data;
  Triangle2DNormCol  *TabTri;
  int                 nbTri;
  register int        i, j;
  int                 max_triangles;
  double              Color[3], ColorResult[3];
  Vector              NormalLight;
  LightSourceStruct  *Light;
  double              alpha;
  double             *Zbuffer;
  double              Norme;
  guchar             *rgb_buf;
  GSList             *tmp_list, *tmp_ls;
  gint                width, height;

  gdk_window_set_background(canvas->window, &(canvas->style->black));

  gdk_window_clear(canvas->window);
  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  width  = canvas->allocation.width;
  height = canvas->allocation.height;
  nbTri = 0;
  max_triangles = 256;
  TabTri = g_new(Triangle2DNormCol, max_triangles);
  for (tmp_list = view_data->frame->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    Tmp = tmp_list->data;
    if (Tmp->visible)
    {
      if (Tmp->selected)
      { /* Selected Objects are red (XXX this should be settable) */
        Color[0] = 1.0;
        Color[1] = 0.0;
        Color[2] = 0.0;
      }
      else
      { /* UnSelected Objects are white (XXX this should be settable) */
        Color[0] = 1.0;
        Color[1] = 1.0;
        Color[2] = 1.0;
      }
      for (TmpTri=Tmp->FirstTriangle ; TmpTri ; TmpTri=TmpTri->Next)
      {
        if (nbTri >= max_triangles)
        {
          max_triangles *= 2;
          TabTri = g_renew(Triangle2DNormCol, TabTri, max_triangles);
        }
        TabTri[nbTri].P[0][0] = width/2+(TmpTri->P1[2]-Xoff)*Zoom;
        TabTri[nbTri].P[0][1] = height/2-(TmpTri->P1[1]-Yoff)*Zoom;
        TabTri[nbTri].P[1][0] = width/2+(TmpTri->P2[2]-Xoff)*Zoom;
        TabTri[nbTri].P[1][1] = height/2-(TmpTri->P2[1]-Yoff)*Zoom;
        TabTri[nbTri].P[2][0] = width/2+(TmpTri->P3[2]-Xoff)*Zoom;
        TabTri[nbTri].P[2][1] = height/2-(TmpTri->P3[1]-Yoff)*Zoom;
        /* Calcul des distances des trois sommets a la camera */
        TabTri[nbTri].Distance[0] = -TmpTri->P1[0];
        TabTri[nbTri].Distance[1] = -TmpTri->P2[0];
        TabTri[nbTri].Distance[2] = -TmpTri->P3[0];
        /* Calcul des couleurs des trois sommets */
        /* 1er sommet */
        ColorResult[0] = Color[0] * AMBIENT;
        ColorResult[1] = Color[1] * AMBIENT;
        ColorResult[2] = Color[2] * AMBIENT;
        for (tmp_ls = view_data->frame->all_light_sources ;
             tmp_ls ;
             tmp_ls = g_slist_next(tmp_ls) )
        {
          Light = tmp_ls->data;
          NormalLight[0] = Light->Location[0]-TmpTri->P1[0];
          NormalLight[1] = Light->Location[1]-TmpTri->P1[1];
          NormalLight[2] = Light->Location[2]-TmpTri->P1[2];
          Norme = V3DLength(NormalLight);
          NormalLight[0] /= Norme;
          NormalLight[1] /= Norme;
          NormalLight[2] /= Norme;
          alpha = V3DDot(TmpTri->N1, NormalLight) * DIFFUSE;
          if (Tmp->Type == PLANE_OBJECT)
            alpha = fabs(alpha);
          if (alpha > 0.0)
          {
            ColorResult[0] += alpha*Color[0]*Light->Color[0];
            ColorResult[1] += alpha*Color[1]*Light->Color[1];
            ColorResult[2] += alpha*Color[2]*Light->Color[2];
          }
        }
        TabTri[nbTri].Color[0][0] = CLAMP(ColorResult[0], 0.0, 1.0);
        TabTri[nbTri].Color[0][1] = CLAMP(ColorResult[1], 0.0, 1.0);
        TabTri[nbTri].Color[0][2] = CLAMP(ColorResult[2], 0.0, 1.0);
        /* 2eme sommet */
        ColorResult[0] = Color[0] * AMBIENT;
        ColorResult[1] = Color[1] * AMBIENT;
        ColorResult[2] = Color[2] * AMBIENT;
        for (tmp_ls = view_data->frame->all_light_sources ;
             tmp_ls ;
             tmp_ls = g_slist_next(tmp_ls) )
        {
          Light = tmp_ls->data;
          NormalLight[0] = Light->Location[0]-TmpTri->P2[0];
          NormalLight[1] = Light->Location[1]-TmpTri->P2[1];
          NormalLight[2] = Light->Location[2]-TmpTri->P2[2];
          Norme = V3DLength(NormalLight);
          NormalLight[0] /= Norme;
          NormalLight[1] /= Norme;
          NormalLight[2] /= Norme;
          alpha = V3DDot(TmpTri->N2, NormalLight) * DIFFUSE;
          if (Tmp->Type == PLANE_OBJECT)
            alpha = fabs(alpha);
          if (alpha > 0.0)
          {
            ColorResult[0] += alpha*Color[0]*Light->Color[0];
            ColorResult[1] += alpha*Color[1]*Light->Color[1];
            ColorResult[2] += alpha*Color[2]*Light->Color[2];
          }
        }
        TabTri[nbTri].Color[1][0] = CLAMP(ColorResult[0], 0.0, 1.0);
        TabTri[nbTri].Color[1][1] = CLAMP(ColorResult[1], 0.0, 1.0);
        TabTri[nbTri].Color[1][2] = CLAMP(ColorResult[2], 0.0, 1.0);
        /* 3eme sommet */
        ColorResult[0] = Color[0] * AMBIENT;
        ColorResult[1] = Color[1] * AMBIENT;
        ColorResult[2] = Color[2] * AMBIENT;
        for (tmp_ls = view_data->frame->all_light_sources ;
             tmp_ls ;
             tmp_ls = g_slist_next(tmp_ls) )
        {
          Light = tmp_ls->data;
          NormalLight[0] = Light->Location[0]-TmpTri->P3[0];
          NormalLight[1] = Light->Location[1]-TmpTri->P3[1];
          NormalLight[2] = Light->Location[2]-TmpTri->P3[2];
          Norme = V3DLength(NormalLight);
          NormalLight[0] /= Norme;
          NormalLight[1] /= Norme;
          NormalLight[2] /= Norme;
          alpha = V3DDot(TmpTri->N3, NormalLight) * DIFFUSE;
          if (Tmp->Type == PLANE_OBJECT)
            alpha = fabs(alpha);
          if (alpha > 0.0)
          {
            ColorResult[0] += alpha*Color[0]*Light->Color[0];
            ColorResult[1] += alpha*Color[1]*Light->Color[1];
            ColorResult[2] += alpha*Color[2]*Light->Color[2];
          }
        }
        TabTri[nbTri].Color[2][0] = CLAMP(ColorResult[0], 0.0, 1.0);
        TabTri[nbTri].Color[2][1] = CLAMP(ColorResult[1], 0.0, 1.0);
        TabTri[nbTri].Color[2][2] = CLAMP(ColorResult[2], 0.0, 1.0);
        nbTri++;
      } /* for */
    } /* if (Tmp->Visible) */
  }
  /* Displaying */
  Zbuffer = g_new(double, width * height);
  for (i=0 ; i<width ; i++)
    for (j=0 ; j<height ; j++)
      Zbuffer[j*width+i] = 1E20;
  rgb_buf = g_malloc0(3*width * height);
  for (i=0 ; i<nbTri ; i++)
    DrawGouraudTriangleZBuffer(canvas, Zbuffer, &(TabTri[i]), rgb_buf);
  gdk_draw_rgb_image(canvas->window, giram_white_gc, 0,0,
                     width, height,
                     GDK_RGB_DITHER_MAX,
                     rgb_buf, 3*width);
  /* Freeing ... */
  g_free(rgb_buf);
  g_free(TabTri);
  g_free(Zbuffer);
  /* Draw the Light Sources */
  for (tmp_list = view_data->frame->all_light_sources ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list) )
  {
    TmpLight = tmp_list->data;
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10);
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10);
    gdk_draw_line(canvas->window, giram_yellow_gc,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom-10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom+10,
                  width/2+(TmpLight->Location[2]-Xoff)*Zoom+10,
                  height/2-(TmpLight->Location[1]-Yoff)*Zoom-10);
  }
  /* Draw the Rotation/Scale Center if any... */
  if (current_tool->ID == MT_ROTATE)
  {
    DrawCenterZY(canvas, Xoff, Yoff, Zoom, view_data->frame->RotationCenter);
  }
  if (current_tool->ID == MT_SCALE)
  {
    DrawCenterZY(canvas, Xoff, Yoff, Zoom, view_data->frame->ScaleCenter);
  }
  /* Draw the camera */
  draw_camera_zy(canvas);
  /* Draw the grid */
  draw_snap_grid_ZY(canvas, canvas->window, width, height);
}

/****************************************************************************
*  view_zy_expose
*****************************************************************************/
void view_zy_expose(GtkWidget *canvas, GdkEventExpose *event)
{
  ViewStruct         *view_data;

  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  switch (view_data->render_style)
  {
    case GIRAM_EDITING_WIREFRAME:
      ExposeWireFrameViewZY(canvas, event);
      break;

    case GIRAM_EDITING_HIDDEN:
      ExposeHiddenFacesViewZY(canvas, event);
      break;

    case GIRAM_EDITING_GOURAUD:
      ExposeGouraudViewZY(canvas, event);
  }
}

/*****************************************************************************
*  draw_unselected_ZY
******************************************************************************/
void draw_unselected_ZY(GtkWidget *canvas, GdkDrawable *drawable, gint width, gint height)
{
  ObjectStruct       *Tmp;
  TriangleListStruct *TmpTri;
  double              Zoom, Xoff, Yoff;
  ViewStruct         *view_data;
  GSList             *tmp_list;

  gdk_draw_rectangle(drawable, giram_black_gc, TRUE, 0, 0, width, height);

  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  draw_snap_grid_ZY(canvas, drawable, width, height);

  for (tmp_list = view_data->frame->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    Tmp = tmp_list->data;
    if (Tmp->visible && !Tmp->selected)
    {
      for (TmpTri = Tmp->FirstTriangle ; TmpTri ; TmpTri = TmpTri->Next)
      {
        gdk_draw_line(drawable, giram_white_gc,
                      width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P1[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P2[1]-Yoff)*Zoom);
        gdk_draw_line(drawable, giram_white_gc,
                      width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P2[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P3[1]-Yoff)*Zoom);
        gdk_draw_line(drawable, giram_white_gc,
                      width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P3[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P1[1]-Yoff)*Zoom);
      }
    }
  }
  /* Draw the Rotation/Scale Center if any... */
/*  if (ModeTool == MT_ROTATE)
  {
    DrawCenterZY(DrawingArea, Xoff, Yoff, Zoom, UserData->FrameView->RotationCenter);
  }
  if (ModeTool == MT_SCALE)
  {
    DrawCenterZY(DrawingArea, Xoff, Yoff, Zoom, UserData->FrameView->ScaleCenter);
  }*/
}

/*****************************************************************************
*  draw_all_ZY
******************************************************************************/
void draw_all_ZY(GtkWidget *canvas, GdkDrawable *drawable, gint width, gint height)
{
  ObjectStruct       *Tmp;
  TriangleListStruct *TmpTri;
  double              Zoom, Xoff, Yoff;
  ViewStruct         *view_data;
  GSList             *tmp_list;

  gdk_draw_rectangle(drawable, giram_black_gc, TRUE, 0, 0, width, height);

  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  draw_snap_grid_ZY(canvas, drawable, width, height);

  for (tmp_list = view_data->frame->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    Tmp = tmp_list->data;
    if (Tmp->visible)
    {
      for (TmpTri = Tmp->FirstTriangle ; TmpTri ; TmpTri = TmpTri->Next)
      {
        gdk_draw_line(drawable, giram_white_gc,
                      width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P1[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P2[1]-Yoff)*Zoom);
        gdk_draw_line(drawable, giram_white_gc,
                      width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P2[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P3[1]-Yoff)*Zoom);
        gdk_draw_line(drawable, giram_white_gc,
                      width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P3[1]-Yoff)*Zoom,
                      width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                      height/2-(TmpTri->P1[1]-Yoff)*Zoom);
      }
    }
  }
}

/*****************************************************************************
*  DrawSelectionZY
******************************************************************************/
void DrawSelectionZY(GtkWidget *DrawingArea, TriangleListStruct *Tri,
                     GdkGC *gc, double Xoff, double Yoff, double Zoom)
{
  TriangleListStruct *TmpTri;
  ViewStruct         *view_data;

  /*DrawSnapGridZY(DrawingArea);*/
  for (TmpTri = Tri ; TmpTri ; TmpTri = TmpTri->Next)
  {
    gdk_draw_line(DrawingArea->window, gc,
                  DrawingArea->allocation.width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                  DrawingArea->allocation.height/2-(TmpTri->P1[1]-Yoff)*Zoom,
                  DrawingArea->allocation.width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                  DrawingArea->allocation.height/2-(TmpTri->P2[1]-Yoff)*Zoom);
    gdk_draw_line(DrawingArea->window, gc,
                  DrawingArea->allocation.width/2+(TmpTri->P2[2]-Xoff)*Zoom,
                  DrawingArea->allocation.height/2-(TmpTri->P2[1]-Yoff)*Zoom,
                  DrawingArea->allocation.width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                  DrawingArea->allocation.height/2-(TmpTri->P3[1]-Yoff)*Zoom);
    gdk_draw_line(DrawingArea->window, gc,
                  DrawingArea->allocation.width/2+(TmpTri->P3[2]-Xoff)*Zoom,
                  DrawingArea->allocation.height/2-(TmpTri->P3[1]-Yoff)*Zoom,
                  DrawingArea->allocation.width/2+(TmpTri->P1[2]-Xoff)*Zoom,
                  DrawingArea->allocation.height/2-(TmpTri->P1[1]-Yoff)*Zoom);
  }
  /* Draw the Rotation/Scale Center if any... */
  if (current_tool->ID == MT_ROTATE)
  {
    view_data = g_object_get_data(G_OBJECT(DrawingArea), "ViewData");
    DrawCenterZY(DrawingArea, Xoff, Yoff, Zoom, view_data->frame->RotationCenter);
  }
  if (current_tool->ID == MT_SCALE)
  {
    view_data = g_object_get_data(G_OBJECT(DrawingArea), "ViewData");
    DrawCenterZY(DrawingArea, Xoff, Yoff, Zoom, view_data->frame->ScaleCenter);
  }
}

