/* view.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include "giram.h"
#include <gdk/gdkkeysyms.h>

#include "tools/tools.h"
#include "tools/tool_select.h"
#include "tools/tool_options.h"
#include "object.h"
#include "csgtree.h"
#include "viewXY.h"
#include "viewXZ.h"
#include "viewZY.h"
#include "utils.h"
#include "camera.h"
#include "filetypes/povray/SavePov.h"
#include "plugins.h"
#include "edit.h"
#include "snap.h"
#include "filetypes/dxf/dxf.h"
#ifdef WITH_S3D
#include "filetypes/s3d/s3d.h"
#endif
#include "filetypes/gts/gts.h"

//#include "giramwidgets.h"
#include "giramcursor.h"
#include "giramrc.h"
#include "giramintl.h"

#include "widgets/giramviewshell.h"

#include "pixmaps/navbutton.xpm"

static GtkItemFactory *ViewFactory;

/*************************************************************************
*  giram_view_set_title
**************************************************************************/
void giram_view_set_title(ViewStruct *view)
{
  gchar     *title, *ptr;
  GtkWidget *canvas, *window;
  gchar     *short_name, *long_name, *style = "";
  gint       length, i;

  canvas = view->canvas;
  window = gtk_widget_get_toplevel(canvas);

  if (view->frame->file_name)
  {
    long_name = view->frame->file_name;
    short_name = (gchar *)g_path_get_basename(long_name);
  } else
  {
    long_name = short_name = _("Untitled");
  }

  switch (view->camera_style)
  {
    case ORTHO_XY_CAMERA:
      style = "X-Y";
      break;

    case ORTHO_XZ_CAMERA:
      style = "X-Z";
      break;

    case ORTHO_ZY_CAMERA:
      style = "Z-Y";
      break;
  }

  if (view_title_format == NULL)
    gtk_window_set_title(GTK_WINDOW(window), "Giram");

  length = 0;
  for (i = 0 ; view_title_format[i] ; i++)
  {
    if (view_title_format[i] == '%')
    {
      switch (view_title_format[i+1])
      {
        case 't': /* camera style */
          i++; length += strlen(style);
          break;
        case 's': /* short file name */
          i++; length += strlen(short_name);
          break;
        case 'f': /* full file name */
          i++; length += strlen(long_name);
          break;
        case '%': /* the percent char */
          i++; length++;
          break;
        case 0:
          g_message(_("The view title format string ended within %%-sequence"));
          break;
        default:
          i++;
          g_message(_("The view title format string contains unknown format sequence '%%%c'"),
                      view_title_format[i]);
      }
    } else
      length++;
  }

  ptr = title = g_malloc(length+1);
  for (i = 0 ; view_title_format[i] ; i++)
  {
    if (view_title_format[i] == '%')
    {
      switch (view_title_format[i+1])
      {
        case 't': /* camera style */
          i++; ptr = g_stpcpy(ptr, style);
          break;
        case 's': /* short file name */
          i++; ptr = g_stpcpy(ptr, short_name);
          break;
        case 'f': /* long file name */
          i++; ptr = g_stpcpy(ptr, long_name);
          break;
        case '%':
          i++; *ptr++ = '%';
          break;
        case 0:
          break;
        default:
          *ptr++ = '%';
          i++;
          *ptr++ = view_title_format[i];
      }
    } else
      *ptr++ = view_title_format[i];
  }
  *ptr = 0;

  gtk_window_set_title(GTK_WINDOW(window), title);
}

/*****************************************************************************
*  SetWireFrameExpose
******************************************************************************/
static void SetWireFrameExpose(void)
{
  ViewStruct *view_data;
  GtkWidget  *canvas;

  view_data = get_current_view_data();
  view_data->render_style = GIRAM_EDITING_WIREFRAME;

  canvas = view_data->canvas;
  gtk_widget_queue_draw(canvas);
}

/*****************************************************************************
*  SetHiddenFacesExpose
******************************************************************************/
static void SetHiddenFacesExpose(void)
{
  ViewStruct *view_data;
  GtkWidget  *canvas;

  view_data = get_current_view_data();
  view_data->render_style = GIRAM_EDITING_HIDDEN;

  canvas = view_data->canvas;
  gtk_widget_queue_draw(canvas);
}

/*****************************************************************************
*  SetGouraudExpose
******************************************************************************/
static void SetGouraudExpose(void)
{
  ViewStruct *view_data;
  GtkWidget  *canvas;

  view_data = get_current_view_data();
  view_data->render_style = GIRAM_EDITING_GOURAUD;

  canvas = view_data->canvas;
  gtk_widget_queue_draw(canvas);
}

/*****************************************************************************
*  CallPovFront
******************************************************************************/
static void CallPovFront(void)
{
  ViewStruct  *view_data;
  FrameStruct *LocalFrame;
  gchar       *TmpFileName;
  gchar       *CommandLine;

  TmpFileName = g_strdup("/tmp/GiramXXXXXX");
  view_data = get_current_view_data();
  LocalFrame = view_data->frame;
  close(mkstemp(TmpFileName));
  SavePovByName(LocalFrame, TmpFileName);
  CommandLine = g_strdup_printf("povfront -p -s %s &", TmpFileName);
  system(CommandLine);
  g_free(CommandLine);
  g_free(TmpFileName);
}

/*****************************************************************************
*  ZoomIn
******************************************************************************/
void ZoomIn(void)
{
  ViewStruct *view_data;

  view_data = get_current_view_data();

  view_data->zoom *= 1.1;
  gtk_widget_queue_draw(GTK_WIDGET(view_data->canvas));
  /* FIXME: Rulers and scrollers */
}

/*****************************************************************************
*  ZoomOut
******************************************************************************/
void ZoomOut(void)
{
  ViewStruct *view_data;

  view_data = get_current_view_data();

  view_data->zoom /= 1.1;
  gtk_widget_queue_draw(GTK_WIDGET(view_data->canvas));
  /* FIXME: Rulers and scrollers */
}

/*****************************************************************************
*  FitToSelectionFunc
******************************************************************************/
void FitToSelectionFunc(void)
{
  ViewStruct         *view_data;
  GList              *tmp_select;
  ObjectStruct       *object;
  TriangleListStruct *TmpTri;
  Vector              MaxExt, MinExt;
  double              ZoomX=1.0, ZoomY=1.0;
  GtkWidget          *canvas;

  view_data = get_current_view_data();
  canvas = view_data->canvas;

  MinExt[0] = MinExt[1] = MinExt[2] = 1e100;
  MaxExt[0] = MaxExt[1] = MaxExt[2] = -1e100;
  if (view_data->frame->selection == NULL)
    return;
  for (tmp_select = view_data->frame->selection;
       tmp_select ;
       tmp_select = g_list_next(tmp_select) )
  {
    object = tmp_select->data;
    /* Find the extents */
    for (TmpTri = object->FirstTriangle ;
         TmpTri ;
         TmpTri = TmpTri->Next)
    {
      if (MinExt[0] > TmpTri->P1[0]) MinExt[0] = TmpTri->P1[0];
      if (MinExt[0] > TmpTri->P2[0]) MinExt[0] = TmpTri->P2[0];
      if (MinExt[0] > TmpTri->P3[0]) MinExt[0] = TmpTri->P3[0];
      if (MinExt[1] > TmpTri->P1[1]) MinExt[1] = TmpTri->P1[1];
      if (MinExt[1] > TmpTri->P2[1]) MinExt[1] = TmpTri->P2[1];
      if (MinExt[1] > TmpTri->P3[1]) MinExt[1] = TmpTri->P3[1];
      if (MinExt[2] > TmpTri->P1[2]) MinExt[2] = TmpTri->P1[2];
      if (MinExt[2] > TmpTri->P2[2]) MinExt[2] = TmpTri->P2[2];
      if (MinExt[2] > TmpTri->P3[2]) MinExt[2] = TmpTri->P3[2];

      if (MaxExt[0] < TmpTri->P1[0]) MaxExt[0] = TmpTri->P1[0];
      if (MaxExt[0] < TmpTri->P2[0]) MaxExt[0] = TmpTri->P2[0];
      if (MaxExt[0] < TmpTri->P3[0]) MaxExt[0] = TmpTri->P3[0];
      if (MaxExt[1] < TmpTri->P1[1]) MaxExt[1] = TmpTri->P1[1];
      if (MaxExt[1] < TmpTri->P2[1]) MaxExt[1] = TmpTri->P2[1];
      if (MaxExt[1] < TmpTri->P3[1]) MaxExt[1] = TmpTri->P3[1];
      if (MaxExt[2] < TmpTri->P1[2]) MaxExt[2] = TmpTri->P1[2];
      if (MaxExt[2] < TmpTri->P2[2]) MaxExt[2] = TmpTri->P2[2];
      if (MaxExt[2] < TmpTri->P3[2]) MaxExt[2] = TmpTri->P3[2];
    }
  }
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      ZoomX = canvas->allocation.width / (MaxExt[0]-MinExt[0]);
      ZoomY = canvas->allocation.height / (MaxExt[1]-MinExt[1]);
      view_data->x_off = (MaxExt[0]+MinExt[0]) / 2.0;
      view_data->y_off = (MaxExt[1]+MinExt[1]) / 2.0;
      break;

    case ORTHO_XZ_CAMERA:
      ZoomX = canvas->allocation.width / (MaxExt[0]-MinExt[0]);
      ZoomY = canvas->allocation.height / (MaxExt[2]-MinExt[2]);
      view_data->x_off = (MaxExt[0]+MinExt[0]) / 2.0;
      view_data->y_off = (MaxExt[2]+MinExt[2]) / 2.0;
      break;

    case ORTHO_ZY_CAMERA:
      ZoomX = canvas->allocation.width / (MaxExt[2]-MinExt[2]);
      ZoomY = canvas->allocation.height / (MaxExt[1]-MinExt[1]);
      view_data->x_off = (MaxExt[2]+MinExt[2]) / 2.0;
      view_data->y_off = (MaxExt[1]+MinExt[1]) / 2.0;
      break;
  }
  view_data->zoom = MIN(ZoomX, ZoomY) * 0.95;
  gtk_widget_queue_draw(canvas);
}

/*****************************************************************************
*  FitToSceneFunc
******************************************************************************/
void FitToSceneFunc(void)
{
  ViewStruct         *view_data;
  GSList             *tmp_list;
  ObjectStruct       *TmpObject;
  TriangleListStruct *TmpTri;
  Vector              MaxExt, MinExt;
  double              ZoomX=1.0, ZoomY=1.0;
  GtkWidget          *canvas;

  view_data = get_current_view_data();
  if (!view_data)
    return;
  canvas = view_data->canvas;

  MinExt[0] = MinExt[1] = MinExt[2] = 1e100;
  MaxExt[0] = MaxExt[1] = MaxExt[2] = -1e100;
  if (!(view_data->frame->all_objects))
    return;
  for (tmp_list = view_data->frame->all_objects;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    TmpObject = tmp_list->data;
    /* Find the extents */
    for (TmpTri = TmpObject->FirstTriangle ;
         TmpTri ;
         TmpTri = TmpTri->Next)
    {
      if (MinExt[0] > TmpTri->P1[0]) MinExt[0] = TmpTri->P1[0];
      if (MinExt[0] > TmpTri->P2[0]) MinExt[0] = TmpTri->P2[0];
      if (MinExt[0] > TmpTri->P3[0]) MinExt[0] = TmpTri->P3[0];
      if (MinExt[1] > TmpTri->P1[1]) MinExt[1] = TmpTri->P1[1];
      if (MinExt[1] > TmpTri->P2[1]) MinExt[1] = TmpTri->P2[1];
      if (MinExt[1] > TmpTri->P3[1]) MinExt[1] = TmpTri->P3[1];
      if (MinExt[2] > TmpTri->P1[2]) MinExt[2] = TmpTri->P1[2];
      if (MinExt[2] > TmpTri->P2[2]) MinExt[2] = TmpTri->P2[2];
      if (MinExt[2] > TmpTri->P3[2]) MinExt[2] = TmpTri->P3[2];

      if (MaxExt[0] < TmpTri->P1[0]) MaxExt[0] = TmpTri->P1[0];
      if (MaxExt[0] < TmpTri->P2[0]) MaxExt[0] = TmpTri->P2[0];
      if (MaxExt[0] < TmpTri->P3[0]) MaxExt[0] = TmpTri->P3[0];
      if (MaxExt[1] < TmpTri->P1[1]) MaxExt[1] = TmpTri->P1[1];
      if (MaxExt[1] < TmpTri->P2[1]) MaxExt[1] = TmpTri->P2[1];
      if (MaxExt[1] < TmpTri->P3[1]) MaxExt[1] = TmpTri->P3[1];
      if (MaxExt[2] < TmpTri->P1[2]) MaxExt[2] = TmpTri->P1[2];
      if (MaxExt[2] < TmpTri->P2[2]) MaxExt[2] = TmpTri->P2[2];
      if (MaxExt[2] < TmpTri->P3[2]) MaxExt[2] = TmpTri->P3[2];
    }
  }
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      ZoomX = canvas->allocation.width / (MaxExt[0]-MinExt[0]);
      ZoomY = canvas->allocation.height / (MaxExt[1]-MinExt[1]);
      view_data->x_off = (MaxExt[0]+MinExt[0]) / 2.0;
      view_data->y_off = (MaxExt[1]+MinExt[1]) / 2.0;
      break;

    case ORTHO_XZ_CAMERA:
      ZoomX = canvas->allocation.width / (MaxExt[0]-MinExt[0]);
      ZoomY = canvas->allocation.height / (MaxExt[2]-MinExt[2]);
      view_data->x_off = (MaxExt[0]+MinExt[0]) / 2.0;
      view_data->y_off = (MaxExt[2]+MinExt[2]) / 2.0;
      break;

    case ORTHO_ZY_CAMERA:
      ZoomX = canvas->allocation.width / (MaxExt[2]-MinExt[2]);
      ZoomY = canvas->allocation.height / (MaxExt[1]-MinExt[1]);
      view_data->x_off = (MaxExt[2]+MinExt[2]) / 2.0;
      view_data->y_off = (MaxExt[1]+MinExt[1]) / 2.0;
      break;
  }
  view_data->zoom = MIN(ZoomX, ZoomY) * 0.95;
  gtk_widget_queue_draw(canvas);
}

/*****************************************************************************
*  SelectAll
******************************************************************************/
static void SelectAll(void)
{
  ViewStruct   *view_data;
  GSList       *tmp_list;
  ObjectStruct *tmp_object;
  ViewStruct   *TmpView;

  view_data = get_current_view_data();
  DestroySelection(&(view_data->frame->selection));

  for (tmp_list = view_data->frame->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    tmp_object = tmp_list->data;
    view_data->frame->selection = g_list_append(view_data->frame->selection,
                                                tmp_object);
    tmp_object->selected = TRUE;
    CSGTreeSelect(tmp_object);
  }

  /* We redraw all the Views */
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*****************************************************************************
*  SelectNone
******************************************************************************/
static void SelectNone(void)
{
  ViewStruct *view_data;
  GSList     *tmp_list;
  ViewStruct *TmpView;

  view_data = get_current_view_data();
  DestroySelection(&(view_data->frame->selection));

  /* We redraw all the Views */
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*****************************************************************************
*  InvertSelection
******************************************************************************/
static void InvertSelection(void)
{
  ViewStruct      *view_data;
  ViewStruct      *TmpView;
  GList           *tmp_selection;
  GList           *Prev;
  GSList          *tmp_list;
  ObjectStruct    *tmp_object;

  view_data = get_current_view_data();
  tmp_selection = NULL;
  for (tmp_list = view_data->frame->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    tmp_object = tmp_list->data;

    tmp_selection = g_list_append(tmp_selection, tmp_object);
    tmp_object->selected = TRUE;
    CSGTreeSelect(tmp_object);
  }

  if (view_data->frame->selection)
  {
    for (Prev = view_data->frame->selection ;
         Prev ;
         Prev = g_list_next(Prev) )
    {
      tmp_object = Prev->data;
      tmp_selection = g_list_remove(tmp_selection, tmp_selection);
      tmp_object->selected = FALSE;
      CSGTreeUnselect(tmp_object);
    }
  }
  DestroySelection(&(view_data->frame->selection));
  view_data->frame->selection = tmp_selection;
  /* Update the CSG Tree selection. A little drastic, perhaps, but it'll
   * do until someone comes up with a better way. */
  giram_create_tree_model(view_data->frame);
  /* We redraw all the Views */
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*****************************************************************************
*  translate_selection
******************************************************************************/
static void translate_selection(GtkWidget *Dialog)
{
  Vector         Vect;
  GtkWidget    **Data;
  ViewStruct    *view_data;
  GList         *selection;
  ObjectStruct  *object;
  GSList        *tmp_list;
  ViewStruct    *TmpView;

  Data = g_object_get_data(G_OBJECT(Dialog), "WidgetData");
  view_data = get_current_view_data();

  Vect[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[0])));
  Vect[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[1])));
  Vect[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[2])));
  if (view_data->frame->selection)
  {
    for (selection = view_data->frame->selection ;
         selection ;
         selection = g_list_next(selection) )
    {
      object = selection->data;
      giram_object_translate(object, Vect);
      giram_object_build_triangle_mesh(object);
    }
    for (tmp_list = view_data->frame->all_views ;
         tmp_list ;
         tmp_list = tmp_list->next)
    {
      TmpView = tmp_list->data;
      gtk_widget_queue_draw(TmpView->canvas);
    }
  }
  gtk_widget_destroy(Dialog);
}

/*****************************************************************************
*  rotate_selection
******************************************************************************/
static void rotate_selection(GtkWidget *Dialog)
{
  Vector          Vect;
  GtkWidget     **Data;
  ViewStruct     *view_data;
  GList          *selection;
  ObjectStruct   *object;
  GSList         *tmp_list;
  ViewStruct     *TmpView;

  Data = g_object_get_data(G_OBJECT(Dialog), "WidgetData");
  view_data = get_current_view_data();

  Vect[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[0])));
  Vect[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[1])));
  Vect[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[2])));
  if (view_data->frame->selection)
  {
    for (selection = view_data->frame->selection ;
         selection ;
         selection = g_list_next(selection) )
    {
      object = selection->data;
      giram_object_rotate(object, Vect);
      giram_object_build_triangle_mesh(object);
    }
    for (tmp_list = view_data->frame->all_views ;
         tmp_list ;
         tmp_list = tmp_list->next)
    {
      TmpView = tmp_list->data;
      gtk_widget_queue_draw(TmpView->canvas);
    }
  }
  gtk_widget_destroy(Dialog);
}

/*****************************************************************************
*  scale_selection
******************************************************************************/
static void scale_selection(GtkWidget *Dialog)
{
  Vector         Vect;
  GtkWidget    **Data;
  ViewStruct    *view_data;
  GList         *selection;
  ObjectStruct  *object;
  GSList        *tmp_list;
  ViewStruct    *TmpView;

  Data = g_object_get_data(G_OBJECT(Dialog), "WidgetData");
  view_data = get_current_view_data();

  Vect[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[0])));
  Vect[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[1])));
  Vect[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[2])));
  if (fabs(Vect[0]) < 10e-8)
  {
    Vect[0] = 1.0;
    g_message(_("You can't scale by 0, changed to 1"));
  }
  if (fabs(Vect[1]) < 10e-8)
  {
    Vect[1] = 1.0;
    g_message(_("You can't scale by 0, changed to 1"));
  }
  if (fabs(Vect[2]) < 10e-8)
  {
    Vect[2] = 1.0;
    g_message(_("You can't scale by 0, changed to 1"));
  }
  if (view_data->frame->selection)
  {
    for (selection = view_data->frame->selection ;
         selection ;
         selection = g_list_next(selection) )
    {
      object = selection->data;
      giram_object_scale(object, Vect);
      giram_object_build_triangle_mesh(object);
    }
    for (tmp_list = view_data->frame->all_views ;
         tmp_list ;
         tmp_list = tmp_list->next)
    {
      TmpView = tmp_list->data;
      gtk_widget_queue_draw(TmpView->canvas);
    }
  }
  gtk_widget_destroy(Dialog);
}

/*************************************************************************
*  TRSSelectionDialog_response
**************************************************************************/
static void TRSSelectionDialog_response(GtkWidget *dialog, gint response, gpointer callback)
{
  switch (response)
  {
    case GTK_RESPONSE_ACCEPT:
      ((GtkCallback)(callback))(dialog, NULL);
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(dialog);
      break;
  }
}

/*************************************************************************
*  TRSSelectionDialog
**************************************************************************/
static void TRSSelectionDialog(gchar *title, GtkSignalFunc CallBack)
{
  GtkWidget  *dialog, *Table, *Label, *Entry;
  GtkWidget **Data;

  dialog = gtk_dialog_new_with_buttons(title, NULL, 0,
                                       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                       NULL);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
  g_signal_connect(G_OBJECT(dialog), "response",
                   G_CALLBACK(TRSSelectionDialog_response), CallBack);

  Data = g_new(GtkWidget*, 3);
  g_object_set_data(G_OBJECT(dialog), "WidgetData", Data);

  Table = gtk_table_new(2, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 5);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), Table);
  /* X */
  Label = gtk_label_new(" x: ");
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 0,1,
                   GTK_FILL | GTK_EXPAND,
                   GTK_FILL | GTK_EXPAND, 0, 0);

  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 0,1,
                   GTK_FILL | GTK_EXPAND,
                   GTK_EXPAND, 0, 0);
  Data[0] = Entry;

  /* Y */
  Label = gtk_label_new(" y: ");
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 1,2,
                   GTK_FILL | GTK_EXPAND,
                   GTK_FILL | GTK_EXPAND, 0, 0);

  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 1,2,
                   GTK_FILL | GTK_EXPAND,
                   GTK_EXPAND, 0, 0);
  Data[1] = Entry;

  /* Z */
  Label = gtk_label_new(" z: ");
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 2,3,
                   GTK_FILL | GTK_EXPAND,
                   GTK_FILL | GTK_EXPAND, 0, 0);

  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 2,3,
                   GTK_FILL | GTK_EXPAND,
                   GTK_EXPAND, 0, 0);
  Data[2] = Entry;

  gtk_widget_show_all(dialog);
}

/*****************************************************************************
*  TranslateSelectionDialog
******************************************************************************/
static void TranslateSelectionDialog(void)
{
  TRSSelectionDialog(_("Translate Selection"), (GtkSignalFunc)translate_selection);
}

/*****************************************************************************
*  RotateSelectionDialog
******************************************************************************/
static void RotateSelectionDialog(void)
{
  TRSSelectionDialog(_("Rotate Selection"), (GtkSignalFunc)rotate_selection);
}

/*****************************************************************************
*  ScaleSelectionDialog
******************************************************************************/
static void ScaleSelectionDialog(void)
{
  TRSSelectionDialog(_("Scale Selection"), (GtkSignalFunc)scale_selection);
}

/*****************************************************************************
*  DrawSelection
******************************************************************************/
void DrawSelection(GtkWidget *DrawingArea, TriangleListStruct *Tri,
                   GdkGC *gc, double Xoff, double Yoff, double Zoom)
{ /* FIXME: OO me */
  ViewStruct *view_data;

  view_data = g_object_get_data(G_OBJECT(DrawingArea), "ViewData");
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      DrawSelectionXY(DrawingArea, Tri, gc, Xoff, Yoff, Zoom);
      break;

    case ORTHO_XZ_CAMERA:
      DrawSelectionXZ(DrawingArea, Tri, gc, Xoff, Yoff, Zoom);
      break;

    case ORTHO_ZY_CAMERA:
      DrawSelectionZY(DrawingArea, Tri, gc, Xoff, Yoff, Zoom);
      break;
  }
}

/*****************************************************************************
*  draw_unselected
******************************************************************************/
void draw_unselected(GtkWidget *canvas, GdkDrawable *drawable, gint style, gint width, gint height)
{ /* FIXME: OO me */
  switch (style)
  {
    case ORTHO_XY_CAMERA:
      draw_unselected_XY(canvas, drawable, width, height);
      break;

    case ORTHO_XZ_CAMERA:
      draw_unselected_XZ(canvas, drawable, width, height);
      break;

    case ORTHO_ZY_CAMERA:
      draw_unselected_ZY(canvas, drawable, width, height);
      break;
  }
}

/*****************************************************************************
*  draw_all
******************************************************************************/
void draw_all(GtkWidget  *canvas, GdkDrawable *drawable, gint style, gint width, gint height)
{ /* FIXME: OO me */
  switch (style)
  {
    case ORTHO_XY_CAMERA:
      draw_all_XY(canvas, drawable, width, height);
      break;

    case ORTHO_XZ_CAMERA:
      draw_all_XZ(canvas, drawable, width, height);
      break;

    case ORTHO_ZY_CAMERA:
      draw_all_ZY(canvas, drawable, width, height);
      break;
  }
}

/****************************************************************************
*  key_to_state
*****************************************************************************/
static int key_to_state(gint key)
{
  switch (key)
  {
    case GDK_Alt_L:
    case GDK_Alt_R:
      return GDK_MOD1_MASK;
    case GDK_Shift_L:
    case GDK_Shift_R:
      return GDK_SHIFT_MASK;
    case GDK_Control_L:
    case GDK_Control_R:
      return GDK_CONTROL_MASK;
    default:
      return 0;
  }
}

/*************************************************************************
*  view_expose
**************************************************************************/
static void giram_view_expose(GtkWidget *canvas, GdkEvent *event)
{
  GdkEventExpose *eevent = (GdkEventExpose *)event;
  ViewStruct     *view_data;
  
  view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      view_xy_expose(canvas, eevent);
      break;
    case ORTHO_XZ_CAMERA:
      view_xz_expose(canvas, eevent);
      break;
    case ORTHO_ZY_CAMERA:
      view_zy_expose(canvas, eevent);
      break;
  }
}

/*************************************************************************
*  giram_view_update_current_view_data
**************************************************************************/
static void giram_view_update_current_view_data(GtkWidget *canvas)
{
  ViewStruct  *new_current_view_data;

  new_current_view_data = g_object_get_data(G_OBJECT(canvas), "ViewData");
  if (current_view_data != new_current_view_data)
  {
    GtkWidget *widget;
    /*  FIXME: set menu sensitivity */
    current_view_data = new_current_view_data;
    widget  = gtk_item_factory_get_widget(ViewFactory, "/View/Show Rulers");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget),
                                   current_view_data->show_rulers);
  }
}

/*************************************************************************
*  event_dispatcher
**************************************************************************/
static gboolean giram_view_event_dispatcher(GtkWidget *canvas, GdkEvent *event, gpointer Menu)
{
  GtkWidget   *status_bar;
  guint        id;
  GdkEventKey *kevent;

  guint state = 0;
  switch (event->type)
  {
    case GDK_EXPOSE:
      giram_view_expose(canvas, event);
      return TRUE;
      break;

    case GDK_LEAVE_NOTIFY:
      status_bar = GIRAM_VIEW_SHELL(current_view_data->shell)->statusbar;
      id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "mouse coord");
      gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);
      gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, "");  
      break;

    case GDK_BUTTON_PRESS:
    {
      GdkEventButton *bevent = (GdkEventButton *)event;

      giram_view_update_current_view_data(canvas);
      
      state = bevent->state;

      switch (bevent->button)
      { 
        case  3: /* Right Button ==> Popup Menu */
          state |= GDK_BUTTON3_MASK;
          gtk_menu_popup(GTK_MENU(Menu), NULL, NULL, NULL, NULL,
                         bevent->button, bevent->time);
          break;
        case 1: /* Left Button : this is the big game */
          state |= GDK_BUTTON1_MASK;
          if (current_tool->button_press)
            current_tool->button_press(canvas, bevent);
          break;
        case 2: /* middle button : pan */
          state |= GDK_BUTTON2_MASK;
          pan_tool->button_press(canvas, bevent);
          break;
      }
      break;
    }
    case GDK_2BUTTON_PRESS:
    case GDK_3BUTTON_PRESS:
      {
        GdkEventButton *bevent = (GdkEventButton *)event;

        giram_view_update_current_view_data(canvas);
        state = bevent->state;
        break;
      }

    case GDK_MOTION_NOTIFY:
    {
      GdkEventMotion *mevent = (GdkEventMotion *)event;
      gchar     *coord_txt = NULL;
      gdouble    xs, ys, xoff, yoff, zoom;

      giram_view_update_current_view_data(canvas);
      state = mevent->state;
      xoff = current_view_data->x_off;
      yoff = current_view_data->y_off;
      zoom = current_view_data->zoom;
      xs = xoff+(mevent->x-canvas->allocation.width /2)/zoom;
      ys = yoff-(mevent->y-canvas->allocation.height/2)/zoom;

      status_bar = GIRAM_VIEW_SHELL(current_view_data->shell)->statusbar;
      id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "mouse coord");
      gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);
      switch (current_view_data->camera_style)
      {
        case ORTHO_XY_CAMERA:
          coord_txt = g_strdup_printf("<%g, %g, 0>", xs, ys);
          break;
        case ORTHO_XZ_CAMERA:
          coord_txt = g_strdup_printf("<%g, 0, %g>", xs, ys);
          break;
        case ORTHO_ZY_CAMERA:
          coord_txt = g_strdup_printf("<0, %g, %g>", ys, xs);
          break;
      }
      gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, coord_txt);  
      g_free(coord_txt);

      if (state & GDK_BUTTON2_MASK)
      {
        pan_tool->motion(canvas, mevent);
      } else
      {
        if (current_tool->motion)
          current_tool->motion(canvas, mevent);
      }
      return FALSE;
    }
    case GDK_BUTTON_RELEASE:
    {
      GdkEventButton *bevent = (GdkEventButton*)event;
      state = bevent->state;

      giram_view_update_current_view_data(canvas);
      if (bevent->button == 2)
        pan_tool->button_release(canvas, bevent);
      else if (current_tool->button_release)
        current_tool->button_release(canvas, bevent);
      return TRUE;
    }
    case GDK_KEY_PRESS:
      kevent = (GdkEventKey *)event;
      state = kevent->state;

      giram_view_update_current_view_data(canvas);
      switch (kevent->keyval)
      {
        case GDK_Left:
          current_view_data->x_off += 30/current_view_data->zoom;
          gtk_widget_queue_draw(canvas);
          break;
        
        case GDK_Right:
          current_view_data->x_off -= 30/current_view_data->zoom;
          gtk_widget_queue_draw(canvas);
          break;

        case GDK_Up:
          current_view_data->y_off -= 30/current_view_data->zoom;
          gtk_widget_queue_draw(canvas);
          break;
        
        case GDK_Down:
          current_view_data->y_off += 30/current_view_data->zoom;
          gtk_widget_queue_draw(canvas);
          break;
        case GDK_Alt_L: case GDK_Alt_R:
        case GDK_Shift_L: case GDK_Shift_R:
        case GDK_Control_L: case GDK_Control_R:
          state |= key_to_state(kevent->keyval);
          break;
      }
      break;
    case GDK_KEY_RELEASE:
      kevent = (GdkEventKey *)event;
      state = kevent->state;

      switch (kevent->keyval)
      {
        case GDK_Alt_L: case GDK_Alt_R:
        case GDK_Shift_L: case GDK_Shift_R:
        case GDK_Control_L: case GDK_Control_R:
          state &= ~key_to_state(kevent->keyval);
          break;
      }
      break;
    default:
      break;
  }
  
  if (!(state & (GDK_BUTTON1_MASK | GDK_BUTTON2_MASK | GDK_BUTTON3_MASK)))
  {
    if (current_tool->cursor_update)
      current_tool->cursor_update(canvas, state);
  }

  return FALSE;
}

/*************************************************************************
*  giram_view_configure_callback
**************************************************************************/
static gboolean giram_view_configure_callback(GtkWidget *canvas,
                                              GdkEventConfigure *ev,
                                              ViewStruct *view_data)
{
  /* setup the rulers */
  gtk_ruler_set_range(GTK_RULER(view_data->hruler),
                      view_data->x_off-canvas->allocation.width / 2.0 / view_data->zoom,
                      view_data->x_off+canvas->allocation.width / 2.0 / view_data->zoom,
                      0.0, 50.0);
  gtk_ruler_set_range(GTK_RULER(view_data->vruler),
                      view_data->y_off-canvas->allocation.height / 2.0 / view_data->zoom,
                      view_data->y_off+canvas->allocation.height / 2.0 / view_data->zoom,
                      0.0, 50.0);
  return FALSE; 
}

/*****************************************************************************
*  giram_view_toggle_rulers
******************************************************************************/
static void giram_view_toggle_rulers(GtkWidget *widget, gpointer data)
{
  ViewStruct *view_data = current_view_data;

  /* FIXME ugly, but the "widget" parameter is always NULL  ??? */
  widget  = gtk_item_factory_get_widget(ViewFactory, "/View/Show Rulers");
  if (!widget) return;
  if (!GTK_CHECK_MENU_ITEM(widget)->active)
  {
    if (GTK_WIDGET_VISIBLE(view_data->origin))
    {
      gtk_widget_hide(view_data->origin);
      gtk_widget_hide(view_data->hruler);
      gtk_widget_hide(view_data->vruler);
      view_data->show_rulers = FALSE;

      gtk_widget_queue_resize(GTK_WIDGET(view_data->origin->parent));
    }
  } else
  {
    if (!GTK_WIDGET_VISIBLE(view_data->origin))
    {
      gtk_widget_show(view_data->origin);
      gtk_widget_show(view_data->hruler);
      gtk_widget_show(view_data->vruler);
      view_data->show_rulers = TRUE;

      gtk_widget_queue_resize(GTK_WIDGET(view_data->origin->parent));
    }
  }
}

/*****************************************************************************
*  gview_delete
******************************************************************************/
static gint gview_delete(GtkWidget *window, GdkEvent *ev, ViewStruct *FrameView)
{
  g_print("The deletion of an XYZ view has been catched!\n");
  FrameView->frame->all_views = g_slist_remove(FrameView->frame->all_views, FrameView);
  g_free(FrameView);
  return FALSE;
}

static GtkItemFactoryEntry ViewMenu[] =
{
  {"/tearoff1",                      NULL, NULL,                     0, "<Tearoff>"},
  {"/_File",                         NULL, NULL,                     0, "<Branch>"},
  {"/File/---",                      NULL, NULL,                     0, "<Tearoff>" },
  {"/File/Load _Pov",                NULL, LoadPovFunc,              0, "<StockItem>", GTK_STOCK_OPEN},
  {"/File/Load AutoCAD _DXF",        NULL, LoadDXFFunc,              0, "<StockItem>", GTK_STOCK_OPEN},
#ifdef USE_LIB3DS
  {"/File/Load 3DStudio _3DS",       NULL, Load3DSFunc,              0, "<StockItem>", GTK_STOCK_OPEN},
#endif
  {"/File/Load GTS",                 NULL, LoadGTSFunc,              0, "<StockItem>", GTK_STOCK_OPEN},
  {"/File/Dum1",                     NULL, NULL,                     0, "<Separator>"},
  {"/File/Save P_ov",                NULL, SavePovFunc,              0, "<StockItem>", GTK_STOCK_SAVE },
  {"/File/Save AutoCAD D_XF",        NULL, SaveDXFFunc,              0, "<StockItem>", GTK_STOCK_SAVE },
#ifdef WITH_S3D
  {"/File/Save _Bishop S3D",         NULL, SaveS3DFunc,              0, "<StockItem>", GTK_STOCK_SAVE },
#endif
  {"/File/Dum2",                     NULL, NULL,                     0, "<Separator>"},
  {"/File/_Quit",                    NULL, ExitFunc,                 0, "<StockItem>", GTK_STOCK_QUIT },
  {"/_Edit",                         NULL, NULL,                     0, "<Branch>"},
  {"/Edit/---",                      NULL, NULL,                     0, "<Tearoff>"},
  {"/Edit/_Cut",                     NULL, CutSelection,             0, "<StockItem>", GTK_STOCK_CUT },
  {"/Edit/C_opy",                    NULL, CopySelection,            0, "<StockItem>", GTK_STOCK_COPY },
  {"/Edit/_Paste",                   NULL, PasteSelection,           0, "<StockItem>", GTK_STOCK_PASTE },
  {"/Edit/_Snap...",                 NULL, SnapDialogShow,           0, "<Item>"},
  {"/_View",                         NULL, NULL,                     0, "<Branch>"},
  {"/View/---",                      NULL, NULL,                     0, "<Tearoff>"},
  {"/View/Zoom _in",                 NULL, ZoomIn,                   0, "<StockItem>", GTK_STOCK_ZOOM_IN },
  {"/View/Zoom _out",                NULL, ZoomOut,                  0, "<StockItem>", GTK_STOCK_ZOOM_OUT },
  {"/View/Fit To Se_lection",        NULL, FitToSelectionFunc,       0, "<StockItem>", GTK_STOCK_ZOOM_FIT },
  {"/View/Fit To _Scene",            NULL, FitToSceneFunc,           0, "<StockItem>", GTK_STOCK_ZOOM_FIT },
  {"/View/---",                      NULL, NULL,                     0, "<Separator>" },
  {"/View/Show _Rulers",             NULL, giram_view_toggle_rulers, 0, "<ToggleItem>" },
  {"/View/Show S_crollbars",         NULL, NULL,                     0, "<ToggleItem>" },
  {"/_Selection",                    NULL, NULL,                     0, "<Branch>"},
  {"/Selection/---",                 NULL, NULL,                     0, "<Tearoff>"},
  {"/Selection/Select _All",         NULL, SelectAll,                0, "<Item>"},
  {"/Selection/Select _None",        NULL, SelectNone,               0, "<Item>"},
  {"/Selection/_Invert Selection",   NULL, InvertSelection,          0, "<Item>"},
  {"/Selection/Export to _DXF",      NULL, SaveSelectedDXFFunc,      0, "<Item>"},
#ifdef WITH_S3D
  {"/Selection/Export to _S3D",      NULL, SaveSelectedS3DFunc,      0, "<Item>"},
#endif
  {"/Selection/Import DXF _Objects", NULL, InsertDXFFunc,            0, "<Item>"},
#ifdef USE_LIB3DS
  {"/Selection/Import 3DS Objects",  NULL, Insert3DSFunc,            0, "<Item>"},
#endif
  {"/Selection/Import GTS Objects",  NULL, InsertGTSFunc,            0, "<Item>"},
  {"/_Operations",                   NULL, NULL,                     0, "<Branch>"},
  {"/Operations/---",                NULL, NULL,                     0, "<Tearoff>"},
  {"/Operations/_Translate",         NULL, TranslateSelectionDialog, 0, "<Item>"},
  {"/Operations/_Rotate",            NULL, RotateSelectionDialog,    0, "<Item>"},
  {"/Operations/_Scale",             NULL, ScaleSelectionDialog,     0, "<Item>"},
  {"/Operations/_Edit Camera",       NULL, EditCamera,               0, "<Item>"},
  {"/_Render",                       NULL, NULL,                     0, "<Branch>"},
  {"/Render/---",                    NULL, NULL,                     0, "<Tearoff>"},
  {"/Render/_Wire Frame",            NULL, SetWireFrameExpose,       0, "<Item>"},
  {"/Render/_Hidden Faces",          NULL, SetHiddenFacesExpose,     0, "<Item>"},
  {"/Render/_Gouraud",               NULL, SetGouraudExpose,         0, "<Item>"},
  {"/Render/_PovRay Rendering",      NULL, CallPovFront,             0, "<Item>"},
  {"/_Dialogs",                      NULL, NULL,                     0, "<Branch>"},
  {"/Dialogs/---",                   NULL, NULL,                     0, "<Tearoff>"},
  {"/Dialogs/_CSG Tree",             NULL, ShowCSGTree,              0, "<Item>"},
  {"/Dialogs/_Tool Options",         NULL, ShowToolOptionsDialog,    0, "<Item>"},
/*  {"/_Plug-Ins",                     NULL, NULL,                     0, "<Branch>"}*/
};
static guint NbViewEntries = G_N_ELEMENTS(ViewMenu);

static GtkAccelGroup *ViewAccelGroup;

/*****************************************************************************
*  CreateViewMenu
******************************************************************************/
void CreateViewMenu(void)
{
  static gboolean initialized = FALSE;

  if (!initialized)
  {
    /* New Keyboard accel group */
    ViewAccelGroup = gtk_accel_group_new();

    /* Creation of the item factory */
    ViewFactory = gtk_item_factory_new(GTK_TYPE_MENU, "<View>", ViewAccelGroup);
    gtk_item_factory_create_items(ViewFactory, NbViewEntries, ViewMenu, 0);
    initialized = TRUE;
  }
}

gboolean gdisplay_origin_button_press(GtkWidget      *widget,
                                      GdkEventButton *event,
                                      gpointer        data)
{
#if 0
  GDisplay *gdisp;

  gdisp = (GDisplay *) data;

  if (! gimp_busy && event->button == 1)
  {
    gint x, y;

    gdisplay_origin_menu_position(GTK_MENU(gdisp->ifactory->widget),
                                  &x, &y, widget);
    gtk_item_factory_popup_with_data(gdisp->ifactory,
                                     gdisp->gimage, NULL,
                                     x, y,
                                     1, event->time);
  }

  /*  Stop the signal emission so the button doesn't grab the
   *  pointer away from us
   */
  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "button_press_event");
#endif
  return FALSE;
}



#define CANVAS_EVENT_MASK (GDK_EXPOSURE_MASK            | \
                           GDK_POINTER_MOTION_MASK      | \
                           /*GDK_POINTER_MOTION_HINT_MASK |*/ \
                           GDK_BUTTON_PRESS_MASK        | \
                           GDK_BUTTON_RELEASE_MASK      | \
                           GDK_STRUCTURE_MASK           | \
                           GDK_ENTER_NOTIFY_MASK        | \
                           GDK_LEAVE_NOTIFY_MASK        | \
                           GDK_KEY_PRESS_MASK           | \
                           GDK_KEY_RELEASE_MASK         | \
                           GDK_PROXIMITY_OUT_MASK)

/*****************************************************************************
*  giram_view_new
******************************************************************************/
ViewStruct *giram_view_new(FrameStruct *LocalFrame, int CameraStyle)
{
  GtkWidget     *upper_hbox, *inner_table;
  GtkWidget     *lower_hbox, *hsb, *vsb;
  GtkWidget     *arrow;
  GtkAdjustment *hsbdata, *vsbdata;
  GtkWidget     *nav_ebox;

  GdkPixbuf *nav_pixbuf;
  GtkWidget *navigation_image;

  ViewStruct *view_data;
  GtkWidget  *menu;

  CreateViewMenu();
  view_data = g_new(ViewStruct, 1);
  current_view_data = view_data;
  view_data->camera_style = CameraStyle;

  view_data->render_style = GIRAM_EDITING_WIREFRAME;
  view_data->zoom = 50.0;
  view_data->x_off = 0.0;
  view_data->y_off = 0.0;

  LocalFrame->all_views = g_slist_append(LocalFrame->all_views, view_data);

  view_data->frame = LocalFrame;
  
/*  gtk_widget_set_events(window, GDK_POINTER_MOTION_MASK
                              | GDK_POINTER_MOTION_HINT_MASK
                              | GDK_BUTTON_PRESS_MASK
                              | GDK_KEY_PRESS_MASK
                              | GDK_KEY_RELEASE_MASK);*/
/*  gtk_container_set_border_width(GTK_CONTAINER(window), 0);
  g_signal_connect(G_OBJECT(window), "delete_event",
                   G_CALLBACK(gview_delete), view_data);*/

  /*  shell
   *     |
   *     +-- disp_vbox
   *            |
   *            +-- upper_hbox
   *            |      |
   *            |      +-- inner_table
   *            |      |      |
   *            |      |      +-- origin
   *            |      |      +-- hruler
   *            |      |      +-- vruler
   *            |      |      +-- canvas
   *            |      |
   *            |      +-- vscrollbar
   *            |
   *            +-- lower_hbox
   *                   |
   *                   +-- hscrollbar
   *                   +-- navbutton
   */

  /*  first, set up the container hierarchy  *********************************/

  /*  a vbox for everything in the view */
  view_data->disp_vbox = gtk_vbox_new(FALSE, 1);
  gtk_widget_show(view_data->disp_vbox);

  /*  a hbox for the inner_table and the vertical scrollbar  */
  upper_hbox = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(view_data->disp_vbox), upper_hbox, TRUE, TRUE, 0);
  gtk_widget_show(upper_hbox);

  /*  the table containing origin, rulers and the canvas  */
  inner_table = gtk_table_new(2, 2, FALSE);
  gtk_table_set_col_spacing(GTK_TABLE(inner_table), 0, 1);
  gtk_table_set_row_spacing(GTK_TABLE(inner_table), 0, 1);
  gtk_box_pack_start(GTK_BOX(upper_hbox), inner_table, TRUE, TRUE, 0);
  gtk_widget_show(inner_table);

  /*  the hbox containing horizontal scrollbar and nav button  */
  lower_hbox = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(view_data->disp_vbox), lower_hbox, FALSE, FALSE, 0);
  gtk_widget_show(lower_hbox);

  /*  create the scrollbars  *************************************************/

  /*  the horizontal scrollbar  */
  /*gdisp->*/hsbdata = GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 512/*width*/, 1, 1, 512/*width*/));
  /*gdisp->*/hsb = gtk_hscrollbar_new(/*gdisp->*/hsbdata);
  GTK_WIDGET_UNSET_FLAGS(/*gdisp->*/hsb, GTK_CAN_FOCUS);

  /*  the vertical scrollbar  */
  /*gdisp->*/vsbdata = GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 384/*height*/, 1, 1, 384/*height*/));
  /*gdisp->*/vsb = gtk_vscrollbar_new(/*gdisp->*/vsbdata);
  GTK_WIDGET_UNSET_FLAGS(/*gdisp->*/vsb, GTK_CAN_FOCUS);

  /*  create the contents of the inner_table  ********************************/

  /*  the menu popup button  */
  view_data->origin = gtk_button_new();
  GTK_WIDGET_UNSET_FLAGS(view_data->origin, GTK_CAN_FOCUS);
  gtk_widget_set_events(GTK_WIDGET(view_data->origin),
                        GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
#if 0
  gtk_signal_connect(GTK_OBJECT(/*gdisp->*/origin), "button_press_event",
                     (GtkSignalFunc)gview_origin_button_press, /*gdisp*/NULL);
#endif

  arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
  gtk_container_set_border_width(GTK_CONTAINER(view_data->origin), 0);
  gtk_container_add(GTK_CONTAINER(view_data->origin), arrow);
  gtk_widget_show(arrow);

  /*  the horizontal ruler  */
  view_data->hruler = gtk_hruler_new();
  gtk_ruler_set_range(GTK_RULER(view_data->hruler),
                      -512.0 / 2.0 / 50.0,
                      +512.0 / 2.0 / 50.0,
                      0.0, 50.0);
  /*  the vertical ruler  */
  view_data->vruler = gtk_vruler_new();
  gtk_ruler_set_range(GTK_RULER(view_data->vruler),
                      -384.0 / 2.0 / 50.0,
                      +384.0 / 2.0 / 50.0,
                      0.0, 50.0);

  /*  the canvas  */
  view_data->canvas = gtk_drawing_area_new();
  g_object_set_data(G_OBJECT(view_data->canvas), "ViewData", view_data);
  gtk_widget_set_size_request(view_data->canvas, 50, 30);
  gtk_widget_set_events(view_data->canvas, CANVAS_EVENT_MASK);
  gtk_widget_set_extension_events(view_data->canvas, GDK_EXTENSION_EVENTS_ALL);
  GTK_WIDGET_SET_FLAGS(view_data->canvas, GTK_CAN_FOCUS);
  
  g_signal_connect_swapped(G_OBJECT(view_data->canvas), "motion_notify_event",
                           G_CALLBACK(GTK_WIDGET_GET_CLASS(view_data->hruler)->motion_notify_event),
                           view_data->hruler);
  g_signal_connect_swapped(G_OBJECT(view_data->canvas), "motion_notify_event",
                           G_CALLBACK(GTK_WIDGET_GET_CLASS(view_data->vruler)->motion_notify_event),
                           view_data->vruler);
  g_signal_connect(G_OBJECT(view_data->canvas), "configure_event",
                   G_CALLBACK(giram_view_configure_callback), view_data);

  /*  set the active display before doing any other canvas event processing  */
#if 0
  gtk_signal_connect(GTK_OBJECT(/*gdisp->*/canvas), "event",
                     (GtkSignalFunc)gdisplay_shell_events, gdisp);
  gtk_signal_connect(GTK_OBJECT(/*gdisp->*/canvas), "event",
                     GTK_SIGNAL_FUNC (gdisplay_canvas_events), gdisp);
#endif

  /*  the navigation window button  */
  nav_ebox = gtk_event_box_new();
#if 0
  gtk_signal_connect(GTK_OBJECT(nav_ebox), "button_press_event",
                     (GtkSignalFunc)nav_popup_click_handler, /*gdisp*/NULL);
#endif

  /*  create the image  ****************************************************/
  nav_pixbuf = gdk_pixbuf_new_from_xpm_data(navbutton_xpm);
  navigation_image = gtk_image_new_from_pixbuf(nav_pixbuf);
  gdk_pixbuf_unref(nav_pixbuf);
  gtk_container_add(GTK_CONTAINER(nav_ebox), navigation_image);
  gtk_widget_show(navigation_image);

  /*  pack all the widgets  **************************************************/

  /*  fill the upper_hbox  */
  gtk_box_pack_start(GTK_BOX(upper_hbox), /*gdisp->*/vsb, FALSE, FALSE, 0);

  /*  fill the inner_table  */
  gtk_table_attach(GTK_TABLE(inner_table), view_data->origin, 0, 1, 0, 1,
                   GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(inner_table), view_data->hruler, 1, 2, 0, 1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(inner_table), view_data->vruler, 0, 1, 1, 2,
                   GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(inner_table), view_data->canvas, 1, 2, 1, 2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  /*  fill the lower_hbox  */
  gtk_box_pack_start(GTK_BOX(lower_hbox), /*gdisp->*/hsb, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(lower_hbox), nav_ebox, FALSE, FALSE, 0);

  /*  show everything  *******************************************************/

/*  if (giramrc.show_rulers)
  {*/
    gtk_widget_show(view_data->origin);
    gtk_widget_show(view_data->hruler);
    gtk_widget_show(view_data->vruler);
    view_data->show_rulers = TRUE;
    {
      GtkWidget *widget;
      /*  FIXME: set menu sensitivity */
      widget  = gtk_item_factory_get_widget(ViewFactory, "/View/Show Rulers");
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget),
                                     current_view_data->show_rulers);
    }
/*  }*/
  gtk_widget_show(view_data->canvas);

  gtk_widget_show(/*gdisp->*/vsb);
  gtk_widget_show(/*gdisp->*/hsb);

  gtk_widget_show(nav_ebox);

  /*  set the focus to the canvas area  */
  gtk_widget_grab_focus(view_data->canvas);

//  giram_view_set_title(view_data);

  //gtk_accel_group_attach(ViewAccelGroup, GTK_OBJECT(window));

  /* We get the menu from the factory */
  menu = gtk_item_factory_get_widget(ViewFactory, "<View>");

  /* The "Plugins" SubMenu */
/*  gtk_menu_append(GTK_MENU(menu), CreatePlugInsMenu(DrawingArea));*/
  g_signal_connect(G_OBJECT(view_data->canvas), "event",
                   G_CALLBACK(giram_view_event_dispatcher), menu);

/*  {
    GdkCursor *cursor;

    cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                              GIRAM_TOOL_CURSOR_NONE,
                              GIRAM_CURSOR_MODIFIER_NONE);
    gdk_window_set_cursor(view_data->canvas->window, cursor);
    gdk_cursor_unref(cursor);
  }*/
  return view_data;
}

