/* tool_superellipsoid.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "csgtree.h"
#include "snap.h"
#include "giramcursor.h"
#include "utils.h"
#include "primitives/superellipsoid.h"

#include "pixmaps/superellipsoid.xpm"

#include "tool_superellipsoid.h"

#include "giramintl.h"

/****************************************************************************
*  giram_superellipsoid_dialog_area_expose
*****************************************************************************/
static gboolean giram_superellipsoid_dialog_area_expose(GtkWidget *area,
                                                        GdkEventExpose *ev,
                                                        GtkWidget *dialog)
{
  GtkAdjustment   *adj;
  GtkWidget       *spin;
  gdouble          ns, ew;
  Vector           P1, P2, P3, P4, TP1, TP2, TP3, TP4;
  int              i,j;
  double           t1, s1, c1, t2, s2, c2;
  Vector           rotate;
  TransformStruct  transform;
  GdkPixmap        *pixmap;
  gint              detail_level;

  pixmap = gdk_pixmap_new(area->window,
                          area->allocation.width,
                          area->allocation.height,
                          -1);
  gdk_draw_rectangle(pixmap,
                     area->style->bg_gc[GTK_STATE_NORMAL],
                     TRUE, 0, 0,
                     area->allocation.width,
                     area->allocation.height);

  adj  = g_object_get_data(G_OBJECT(dialog), "east-west");
  ew   = adj->value;
  adj  = g_object_get_data(G_OBJECT(dialog), "north-south");
  ns   = adj->value;

  adj       = g_object_get_data(G_OBJECT(dialog), "rotate-x");
  rotate[0] = adj->value;
  adj       = g_object_get_data(G_OBJECT(dialog), "rotate-y");
  rotate[1] = adj->value;
  rotate[2] = 0.0;
  MIdentity(transform.Direct);
  MIdentity(transform.Inverse);
  ComputeRotateTrans(&transform, rotate);

  spin = g_object_get_data(G_OBJECT(dialog), "detail");
  detail_level = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));

  /* First the North Pole */
  V3Deq(P1, 0.0, 0.0, 1.0);
  for (i=0; i<detail_level; i++)
  { /* The triangles which are adjacents to the pole */
    t1 = M_PI / detail_level;
    if (sin(t1) >= 0.0) s1 =  pow( sin(t1), ns);
    else                s1 = -pow(-sin(t1), ns);
    if (cos(t1) >= 0.0) c1 =  pow( cos(t1), ns);
    else                c1 = -pow(-cos(t1), ns);
    t2 = i * 2.0 * M_PI / detail_level;
    if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
    else                s2 = -pow(-sin(t2), ew);
    if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
    else                c2 = -pow(-cos(t2), ew);
    V3Deq(P2, s1*c2, s1*s2, c1);
    t2 = (i+1) * 2.0 * M_PI / detail_level;
    if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
    else                s2 = -pow(-sin(t2), ew);
    if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
    else                c2 = -pow(-cos(t2), ew);
    V3Deq(P3, s1*c2, s1*s2, c1);
    MEvaluatePoint(TP1, &transform, P1);
    MEvaluatePoint(TP2, &transform, P2);
    MEvaluatePoint(TP3, &transform, P3);
    gdk_draw_line(pixmap, area->style->black_gc,
                  area->allocation.width /2+TP1[0]*100,
                  area->allocation.height/2-TP1[1]*100,
                  area->allocation.width /2+TP2[0]*100,
                  area->allocation.height/2-TP2[1]*100);
    gdk_draw_line(pixmap, area->style->black_gc,
                  area->allocation.width /2+TP1[0]*100,
                  area->allocation.height/2-TP1[1]*100,
                  area->allocation.width /2+TP3[0]*100,
                  area->allocation.height/2-TP3[1]*100);
    gdk_draw_line(pixmap, area->style->black_gc,
                  area->allocation.width /2+TP3[0]*100,
                  area->allocation.height/2-TP3[1]*100,
                  area->allocation.width /2+TP2[0]*100,
                  area->allocation.height/2-TP2[1]*100);
  }
  /* Then the Mid Zone */
  for (i=1 ; i<detail_level-1 ; i++)
  {
    for (j=0 ; j<detail_level ; j++)
    {
      t1 = i * M_PI / detail_level;
      if (sin(t1) >= 0.0) s1 =  pow( sin(t1), ns);
      else                s1 = -pow(-sin(t1), ns);
      if (cos(t1) >= 0.0) c1 =  pow( cos(t1), ns);
      else                c1 = -pow(-cos(t1), ns);
      t2 = j * 2.0 * M_PI / detail_level;
      if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
      else                s2 = -pow(-sin(t2), ew);
      if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
      else                c2 = -pow(-cos(t2), ew);
      V3Deq(P1, s1*c2, s1*s2, c1);
      t1 = (i+1) * M_PI / detail_level;
      if (sin(t1) >= 0.0) s1 =  pow( sin(t1), ns);
      else                s1 = -pow(-sin(t1), ns);
      if (cos(t1) >= 0.0) c1 =  pow( cos(t1), ns);
      else                c1 = -pow(-cos(t1), ns);
      t2 = j * 2.0 * M_PI / detail_level;
      if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
      else                s2 = -pow(-sin(t2), ew);
      if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
      else                c2 = -pow(-cos(t2), ew);
      V3Deq(P2, s1*c2, s1*s2, c1);
      t1 = i * M_PI / detail_level;
      if (sin(t1) >= 0.0) s1 =  pow( sin(t1), ns);
      else                s1 = -pow(-sin(t1), ns);
      if (cos(t1) >= 0.0) c1 =  pow( cos(t1), ns);
      else                c1 = -pow(-cos(t1), ns);
      t2 = (j+1) * 2.0 * M_PI / detail_level;
      if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
      else                s2 = -pow(-sin(t2), ew);
      if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
      else                c2 = -pow(-cos(t2), ew);
      V3Deq(P3, s1*c2, s1*s2, c1);
      t1 = (i+1) * M_PI / detail_level;
      if (sin(t1) >= 0.0) s1 =  pow( sin(t1), ns);
      else                s1 = -pow(-sin(t1), ns);
      if (cos(t1) >= 0.0) c1 =  pow( cos(t1), ns);
      else                c1 = -pow(-cos(t1), ns);
      t2 = (j+1) * 2.0 * M_PI / detail_level;
      if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
      else                s2 = -pow(-sin(t2), ew);
      if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
      else                c2 = -pow(-cos(t2), ew);
      V3Deq(P4, s1*c2, s1*s2, c1);
      MEvaluatePoint(TP1, &transform, P1);
      MEvaluatePoint(TP2, &transform, P2);
      MEvaluatePoint(TP3, &transform, P3);
      MEvaluatePoint(TP4, &transform, P4);
      gdk_draw_line(pixmap, area->style->black_gc,
                    area->allocation.width /2+TP1[0]*100,
                    area->allocation.height/2-TP1[1]*100,
                    area->allocation.width /2+TP2[0]*100,
                    area->allocation.height/2-TP2[1]*100);
      gdk_draw_line(pixmap, area->style->black_gc,
                    area->allocation.width /2+TP2[0]*100,
                    area->allocation.height/2-TP2[1]*100,
                    area->allocation.width /2+TP4[0]*100,
                    area->allocation.height/2-TP4[1]*100);
      gdk_draw_line(pixmap, area->style->black_gc,
                    area->allocation.width /2+TP4[0]*100,
                    area->allocation.height/2-TP4[1]*100,
                    area->allocation.width /2+TP3[0]*100,
                    area->allocation.height/2-TP3[1]*100);
      gdk_draw_line(pixmap, area->style->black_gc,
                    area->allocation.width /2+TP3[0]*100,
                    area->allocation.height/2-TP3[1]*100,
                    area->allocation.width /2+TP1[0]*100,
                    area->allocation.height/2-TP1[1]*100);
    }
  }
  /* And the South Pole */
  V3Deq(P1, 0.0, 0.0, -1.0);
  for (i=0 ; i<detail_level ; i++)
  { /* The triangles which are adjacents to the south pole */
    t1 = M_PI - M_PI / detail_level;
    if (sin(t1) >= 0.0) s1 =  pow( sin(t1), ns);
    else                s1 = -pow(-sin(t1), ns);
    if (cos(t1) >= 0.0) c1 =  pow( cos(t1), ns);
    else                c1 = -pow(-cos(t1), ns);
    t2 = i * 2.0 * M_PI / detail_level;
    if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
    else                s2 = -pow(-sin(t2), ew);
    if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
    else                c2 = -pow(-cos(t2), ew);
    V3Deq(P2, s1*c2, s1*s2, c1);
    t2 = (i+1) * 2.0 * M_PI / detail_level;
    if (sin(t2) >= 0.0) s2 =  pow( sin(t2), ew);
    else                s2 = -pow(-sin(t2), ew);
    if (cos(t2) >= 0.0) c2 =  pow( cos(t2), ew);
    else                c2 = -pow(-cos(t2), ew);
    V3Deq(P3, s1*c2, s1*s2, c1);
    MEvaluatePoint(TP1, &transform, P1);
    MEvaluatePoint(TP2, &transform, P2);
    MEvaluatePoint(TP3, &transform, P3);
    gdk_draw_line(pixmap, area->style->black_gc,
                  area->allocation.width /2+TP1[0]*100,
                  area->allocation.height/2-TP1[1]*100,
                  area->allocation.width /2+TP2[0]*100,
                  area->allocation.height/2-TP2[1]*100);
    gdk_draw_line(pixmap, area->style->black_gc,
                  area->allocation.width /2+TP1[0]*100,
                  area->allocation.height/2-TP1[1]*100,
                  area->allocation.width /2+TP3[0]*100,
                  area->allocation.height/2-TP3[1]*100);
    gdk_draw_line(pixmap, area->style->black_gc,
                  area->allocation.width /2+TP3[0]*100,
                  area->allocation.height/2-TP3[1]*100,
                  area->allocation.width /2+TP2[0]*100,
                  area->allocation.height/2-TP2[1]*100);
  }
  gdk_window_set_back_pixmap(area->window, pixmap, FALSE);
  gdk_window_clear(area->window);
  g_object_unref(pixmap);
  return TRUE;
}

/*****************************************************************************
*  NewSuperEllipsoidOK
******************************************************************************/
static void NewSuperEllipsoidOK(GtkWidget *dialog)
{
  GtkWidget     *area, *spin;
  ViewStruct    *view_data;
  ObjectStruct  *SuperEllipsoid;
  GSList        *tmp_list;
  ViewStruct    *TmpView;
  GtkAdjustment *adj;
  gdouble        ns, ew;
  Vector         rotate, translate;
  gdouble       *get_trans;

  area = g_object_get_data(G_OBJECT(dialog), "area");
  view_data = g_object_get_data(G_OBJECT(area), "ViewData");
  adj = g_object_get_data(G_OBJECT(dialog), "east-west");
  ew = adj->value;
  adj = g_object_get_data(G_OBJECT(dialog), "north-south");
  ns = adj->value;
  adj       = g_object_get_data(G_OBJECT(dialog), "rotate-x");
  rotate[0] = adj->value;
  adj       = g_object_get_data(G_OBJECT(dialog), "rotate-y");
  rotate[1] = adj->value;
  rotate[2] = 0.0;
  SuperEllipsoid = giram_superellipsoid_new(ns, ew);
  SuperEllipsoid->name = create_uniq_object_name(view_data->frame, _("superellipsoid"));
  spin = g_object_get_data(G_OBJECT(dialog), "detail");
  SuperEllipsoid->detail_level = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
  view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                 SuperEllipsoid);
  SuperEllipsoid->frame = view_data->frame;
  giram_object_rotate(SuperEllipsoid, rotate);

  get_trans = g_object_get_data(G_OBJECT(dialog), "translate");
  V3Dcopy(translate, get_trans);
  giram_object_translate(SuperEllipsoid, translate);
  giram_object_build_triangle_mesh(SuperEllipsoid);
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gtk_widget_destroy(dialog);
}

/*****************************************************************************
*  tool_superellipsoid_dialog_response
******************************************************************************/
static void tool_superellipsoid_dialog_response(GtkWidget *dialog, gint response)
{
  switch (response)
  {
    case GTK_RESPONSE_ACCEPT:
      NewSuperEllipsoidOK(dialog);
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(dialog);
      break;
  }
}

/*****************************************************************************
*  tool_superellipsoid_button_press
******************************************************************************/
static void tool_superellipsoid_button_press(GtkWidget *DrawingArea, GdkEventButton *bevent)
{
  GtkWidget     *dialog, *table, *label;
  GtkWidget     *canvas;
  GtkAdjustment *adj;
  GtkWidget     *scale;
  GtkWidget     *hbox, *spin;
  gdouble       *translate;
  gdouble        Xorg, Yorg;
  ViewStruct    *view_data;

  /* remember the translation vector */
  translate = g_new(gdouble, 3);
  MouseToReal(bevent->x, bevent->y, DrawingArea, &Xorg, &Yorg);
  view_data = g_object_get_data(G_OBJECT(DrawingArea), "ViewData");
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(translate, Xorg, Yorg, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(translate, Xorg, 0.0, Yorg);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(translate, 0.0, Yorg, Xorg);
      break;
  }

  dialog = gtk_dialog_new_with_buttons(_("New SuperEllipsoid"), NULL, 0,
                                       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                       NULL);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
  g_signal_connect(G_OBJECT(dialog), "response",
                   G_CALLBACK(tool_superellipsoid_dialog_response), NULL);
  g_object_set_data(G_OBJECT(dialog), "translate", translate);
  g_object_set_data(G_OBJECT(dialog), "area", DrawingArea);

  /* The canvas, for user feedback, sourrounded by to slider to rotate the SE. */
  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  canvas = gtk_drawing_area_new();
  gtk_widget_set_size_request(canvas, 200, 200);
  gtk_table_attach_defaults(GTK_TABLE(table), canvas, 0,1, 0,1);
  g_signal_connect(G_OBJECT(canvas), "expose_event",
                   G_CALLBACK(giram_superellipsoid_dialog_area_expose),
                   dialog);
  /* The vertical slider for a rotation around the X axis */
  adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -180.0, 180.0, 1.0, 5.0, 5.0);
  g_signal_connect_swapped(G_OBJECT(adj), "value_changed",
                           G_CALLBACK(gtk_widget_queue_draw),
                           canvas);
  scale = gtk_vscale_new(adj);
  gtk_table_attach(GTK_TABLE(table), scale, 1,2, 0,1,
                   GTK_FILL, GTK_FILL, 0,0);
  g_object_set_data(G_OBJECT(dialog), "rotate-x", adj);

  /* The horizontal slider for a rotation around the Y axis */
  adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -180.0, 180.0, 1.0, 5.0, 5.0);
  g_signal_connect_swapped(G_OBJECT(adj), "value_changed",
                           G_CALLBACK(gtk_widget_queue_draw),
                           canvas);
  scale = gtk_hscale_new(adj);
  gtk_table_attach(GTK_TABLE(table), scale, 0,1, 1,2,
                   GTK_FILL, GTK_FILL, 0,0);
  g_object_set_data(G_OBJECT(dialog), "rotate-y", adj);

  /* The East-West Parameter */
  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  label = gtk_label_new(_("East-West Parameter: "));
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 5.0, 0.1, 0.1, 1.0);
  g_signal_connect_swapped(G_OBJECT(adj), "value_changed",
                           G_CALLBACK(gtk_widget_queue_draw),
                           canvas);
  scale = gtk_hscale_new(adj);
  gtk_widget_set_size_request(scale, 150, 30);
  gtk_range_set_update_policy(GTK_RANGE(scale), GTK_UPDATE_CONTINUOUS);
  gtk_scale_set_draw_value(GTK_SCALE(scale), TRUE);
  gtk_table_attach(GTK_TABLE(table), scale, 1,2, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  g_object_set_data(G_OBJECT(dialog), "east-west", adj);

  /* The North-South Parameter */
  label = gtk_label_new(_("North-South Parameter: "));
  gtk_table_attach(GTK_TABLE(table), label, 0,1, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  adj = (GtkAdjustment*)gtk_adjustment_new(1.0, 0.1, 5.0, 0.1, 0.1, 1.0);
  g_signal_connect_swapped(G_OBJECT(adj), "value_changed",
                           G_CALLBACK(gtk_widget_queue_draw),
                           canvas);
  scale = gtk_hscale_new(adj);
  gtk_widget_set_size_request(scale, 150, 30);
  gtk_range_set_update_policy(GTK_RANGE(scale), GTK_UPDATE_CONTINUOUS);
  gtk_scale_set_draw_value(GTK_SCALE(scale), TRUE);
  gtk_table_attach(GTK_TABLE(table), scale, 1,2, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  g_object_set_data(G_OBJECT(dialog), "north-south", adj);

  /* The detail level spin button */
  hbox = gtk_hbox_new(FALSE, 4);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Detail Level:"));
  gtk_box_pack_start_defaults(GTK_BOX(hbox), label);
  adj = (GtkAdjustment*)gtk_adjustment_new(16, 4, 128, 1, 4, 4);
  g_signal_connect_swapped(G_OBJECT(adj), "value_changed",
                           G_CALLBACK(gtk_widget_queue_draw),
                           canvas);
  spin = gtk_spin_button_new(adj, 1.0, 0);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), spin);
  g_object_set_data(G_OBJECT(dialog), "detail", spin);

  gtk_widget_show_all(dialog);
}

/****************************************************************************
*  tool_superellipsoid_cursor_update
*****************************************************************************/
static void tool_superellipsoid_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;
          
  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_superellipsoid_register
*****************************************************************************/
GiramTool *giram_tool_superellipsoid_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New SuperEllipsoid");
  tool->Icon           = ellipsoid_icon;
  tool->Path           = "<ToolBar/Shape3D>";
  tool->ID             = MT_NEW_SUPERELLIPSOID;
  tool->OptionsFunc    = NULL;
  tool->button_press   = tool_superellipsoid_button_press;
  tool->motion         = NULL;
  tool->button_release = NULL;
  tool->cursor_update  = tool_superellipsoid_cursor_update;

  return tool;
}

