/* tool_rotate.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <math.h>
#include "giram.h"
#include "object.h"
#include "trimesh.h"
#include "view.h"
#include "utils.h"
#include "giramcursor.h"
#include "csgtree.h"
#include "giramobjecteditor.h"
#include "widgets/giramviewshell.h"
#include "tool_rotate.h"

#include "giramintl.h"

#include "pixmaps/rotate.xpm"

TriangleListStruct *AllSelectionTri, *InitialSelectionTri;
double Xorg, Yorg, OrgNorme;
guint id;

static GdkPixmap *backpixmap = NULL;

#define NONE   1
#define MOVING 2
int ROTATE_STATE = NONE;

int SnapToAngleFlag;
double SnapAngle;

/*****************************************************************************
*  ToggleSnapToAngle
******************************************************************************/
static void ToggleSnapToAngle(GtkWidget *SpinButton)
{
  SnapToAngleFlag = !SnapToAngleFlag;
  gtk_widget_set_sensitive(SpinButton, SnapToAngleFlag);
}

/*****************************************************************************
*  SnapAngleChange
******************************************************************************/
static void SnapAngleChange(GtkWidget *SpinButton)
{
  SnapAngle = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SpinButton));
}

/*****************************************************************************
*  BuildRotateOptions
******************************************************************************/
static GtkWidget *BuildRotateOptions(GtkWidget *VBox)
{
  GtkWidget *vbox;
  GtkWidget *SnapToAngle;
  GtkWidget *Label;
  GtkWidget *HBox;
  GtkWidget *SpinButton;
  GtkAdjustment *adj;

  /*  the main vbox  */
  vbox = gtk_vbox_new(FALSE, 1);
  /*  the main label  */
  Label = gtk_label_new(_("Rotate"));
  gtk_box_pack_start(GTK_BOX(vbox), Label, FALSE, FALSE, 0);
  gtk_widget_show(Label);

  /* The Toggle button for enabling/disabling snaping */
  SnapToAngle = gtk_check_button_new_with_label(_("Snap to angle"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SnapToAngle), SnapToAngleFlag);
  gtk_box_pack_start(GTK_BOX(vbox), SnapToAngle, FALSE, FALSE, 0);
  gtk_widget_show(SnapToAngle);

  /* The horizontal box for the angle label/spin button pair */
  HBox = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(vbox), HBox, TRUE, TRUE, 0);
  gtk_widget_show(HBox);
  /* The Label for the angle */
  Label = gtk_label_new(_("Angle:"));
  gtk_box_pack_start(GTK_BOX(HBox), Label, FALSE, FALSE, 0);
  gtk_widget_show(Label);
  /* The spin button for the angle */
  adj = (GtkAdjustment *) gtk_adjustment_new(SnapAngle, 1.0, 90.0, 1.0, 5.0, 0.0);
  SpinButton = gtk_spin_button_new(adj, 0.2, 2);
  gtk_widget_set_sensitive(SpinButton, SnapToAngleFlag);
  gtk_box_pack_start(GTK_BOX(HBox), SpinButton, TRUE, TRUE, 0);
  gtk_widget_show(SpinButton);

  g_signal_connect_swapped(G_OBJECT(SnapToAngle), "toggled",
                           G_CALLBACK(ToggleSnapToAngle),
                           SpinButton);

  g_signal_connect_swapped(G_OBJECT(adj), "value_changed",
                           G_CALLBACK(SnapAngleChange),
                           SpinButton);

  gtk_box_pack_start_defaults(GTK_BOX(VBox), vbox);
  gtk_widget_show(vbox);
  return vbox;
}

/*****************************************************************************
*  ToolInitRotateSelection
******************************************************************************/
static void ToolInitRotateSelection(GtkWidget *DrawingArea, GdkEventButton *bevent)
{
  ViewStruct *view_data;
  double      Zoom, Xoff, Yoff;
  GSList     *tmp_list;
  ViewStruct *TmpView;
  GtkWidget  *StatusBar;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(StatusBar), "rotate");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  Xorg = Xoff+(bevent->x-DrawingArea->allocation.width / 2)/Zoom;
  Yorg = Yoff-(bevent->y-DrawingArea->allocation.height/ 2)/Zoom;

  if (bevent->state & 4) /* Ctrl */
  { /* The user only want to move the Rotation Center */
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        view_data->frame->RotationCenter[0] = Xorg;
        view_data->frame->RotationCenter[1] = Yorg;
        break;

      case ORTHO_XZ_CAMERA:
        view_data->frame->RotationCenter[0] = Xorg;
        view_data->frame->RotationCenter[2] = Yorg;
        break;

      case ORTHO_ZY_CAMERA:
        view_data->frame->RotationCenter[2] = Xorg;
        view_data->frame->RotationCenter[1] = Yorg;
        break;
    }
    for (tmp_list = view_data->frame->all_views ;
         tmp_list ;
         tmp_list = tmp_list->next)
    {
      TmpView = tmp_list->data;
      gtk_widget_queue_draw(TmpView->canvas);
    }
    return;
  }
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      Xorg -= view_data->frame->RotationCenter[0];
      Yorg -= view_data->frame->RotationCenter[1];
      break;

    case ORTHO_XZ_CAMERA:
      Xorg -= view_data->frame->RotationCenter[0];
      Yorg -= view_data->frame->RotationCenter[2];
      break;

    case ORTHO_ZY_CAMERA:
      Xorg -= view_data->frame->RotationCenter[2];
      Yorg -= view_data->frame->RotationCenter[1];
      break;
  }
  OrgNorme = sqrt(Xorg*Xorg+Yorg*Yorg);
  if (OrgNorme == 0.0)
    return;
  if (view_data->frame->selection == NULL)
    return;

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(DrawingArea->window,
                              DrawingArea->allocation.width,
                              DrawingArea->allocation.height,
                              -1);

  draw_unselected(DrawingArea, backpixmap, view_data->camera_style,
                  DrawingArea->allocation.width,
                  DrawingArea->allocation.height);
  gdk_window_set_back_pixmap(DrawingArea->window, backpixmap, FALSE);
  gdk_window_clear(DrawingArea->window);

  AllSelectionTri = ComputeSelectionTriangleMesh(view_data->frame->selection);
  InitialSelectionTri = ComputeSelectionTriangleMesh(view_data->frame->selection);

  DrawSelection(DrawingArea, AllSelectionTri, giram_purple_gc, Xoff, Yoff, Zoom);

  ROTATE_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(StatusBar), id, _("Angle: "));
  gdk_pointer_grab(DrawingArea->window, FALSE,
                   GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  ToolMotionRotateSelection
******************************************************************************/
static void ToolMotionRotateSelection(GtkWidget *DrawingArea, GdkEventMotion *Mev)
{
  ViewStruct         *view_data;
  GtkWidget          *StatusBar;
  double              Zoom, Xoff, Yoff;
  double              XTrans, YTrans;
  double              Norme, CosAngle, Angle;
  Matrix              MatTrans;
  TriangleListStruct *TmpTri, *TmpOrigTri;
  gchar              *Message;

  if (ROTATE_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);
  gdk_window_clear(DrawingArea->window);
  XTrans = Xoff+(Mev->x-DrawingArea->allocation.width /2)/Zoom;
  YTrans = Yoff-(Mev->y-DrawingArea->allocation.height/2)/Zoom;
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      XTrans -= view_data->frame->RotationCenter[0];
      YTrans -= view_data->frame->RotationCenter[1];
      break;

    case ORTHO_XZ_CAMERA:
      XTrans -= view_data->frame->RotationCenter[0];
      YTrans -= view_data->frame->RotationCenter[2];
      break;

    case ORTHO_ZY_CAMERA:
      XTrans -= view_data->frame->RotationCenter[2];
      YTrans -= view_data->frame->RotationCenter[1];
      break;
  }
  Norme = sqrt(XTrans*XTrans + YTrans*YTrans);
  if (Norme == 0.0)
    Angle = 0.0;
  else
  {
    CosAngle = (Xorg*XTrans + Yorg*YTrans) / (OrgNorme * Norme);
    Angle = acos(CosAngle);
    if (XTrans*Yorg-Xorg*YTrans > 0.0)
    {
      Angle = -Angle;
    }
  }
  if (SnapToAngleFlag)
  {
    Angle = SnapAngle*(floor(Angle*180.0/M_PI/SnapAngle))/180.0*M_PI;
  }
  Message = g_strdup_printf(_("Angle: %g"), Angle*180.0/M_PI);
  gtk_statusbar_push(GTK_STATUSBAR(StatusBar), id, Message);
  g_free(Message);
  MIdentity(MatTrans);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      MatTrans[0][0] = cos(Angle);
      MatTrans[1][1] = cos(Angle);
      MatTrans[0][1] = sin(Angle);
      MatTrans[1][0] = -sin(Angle);
      MatTrans[3][0] = -view_data->frame->RotationCenter[0]*cos(Angle)
                       +view_data->frame->RotationCenter[1]*sin(Angle)
                       +view_data->frame->RotationCenter[0];
      MatTrans[3][1] = -view_data->frame->RotationCenter[0]*sin(Angle)
                       -view_data->frame->RotationCenter[1]*cos(Angle)
                       +view_data->frame->RotationCenter[1];
      break;

    case ORTHO_XZ_CAMERA:
      MatTrans[0][0] = cos(Angle);
      MatTrans[2][2] = cos(Angle);
      MatTrans[0][2] = sin(Angle);
      MatTrans[2][0] = -sin(Angle);
      MatTrans[3][0] = -view_data->frame->RotationCenter[0]*cos(Angle)
                       +view_data->frame->RotationCenter[2]*sin(Angle)
                       +view_data->frame->RotationCenter[0];
      MatTrans[3][2] = -view_data->frame->RotationCenter[0]*sin(Angle)
                       -view_data->frame->RotationCenter[2]*cos(Angle)
                       +view_data->frame->RotationCenter[2];
      break;

    case ORTHO_ZY_CAMERA:
      MatTrans[2][2] = cos(Angle);
      MatTrans[1][1] = cos(Angle);
      MatTrans[2][1] = sin(Angle);
      MatTrans[1][2] = -sin(Angle);
      MatTrans[3][1] = -view_data->frame->RotationCenter[1]*cos(Angle)
                       -view_data->frame->RotationCenter[2]*sin(Angle)
                       +view_data->frame->RotationCenter[1];
      MatTrans[3][2] =  view_data->frame->RotationCenter[1]*sin(Angle)
                       -view_data->frame->RotationCenter[2]*cos(Angle)
                       +view_data->frame->RotationCenter[2];
      break;
  }
  for (TmpTri = AllSelectionTri, TmpOrigTri = InitialSelectionTri ;
       TmpTri ;
       TmpTri = TmpTri->Next, TmpOrigTri = TmpOrigTri->Next)
  {
    TmpTri->P1[0] = TmpOrigTri->P1[0]*MatTrans[0][0] + TmpOrigTri->P1[1]*MatTrans[1][0] + TmpOrigTri->P1[2]*MatTrans[2][0] + MatTrans[3][0];
    TmpTri->P1[1] = TmpOrigTri->P1[0]*MatTrans[0][1] + TmpOrigTri->P1[1]*MatTrans[1][1] + TmpOrigTri->P1[2]*MatTrans[2][1] + MatTrans[3][1];
    TmpTri->P1[2] = TmpOrigTri->P1[0]*MatTrans[0][2] + TmpOrigTri->P1[1]*MatTrans[1][2] + TmpOrigTri->P1[2]*MatTrans[2][2] + MatTrans[3][2];
    TmpTri->P2[0] = TmpOrigTri->P2[0]*MatTrans[0][0] + TmpOrigTri->P2[1]*MatTrans[1][0] + TmpOrigTri->P2[2]*MatTrans[2][0] + MatTrans[3][0];
    TmpTri->P2[1] = TmpOrigTri->P2[0]*MatTrans[0][1] + TmpOrigTri->P2[1]*MatTrans[1][1] + TmpOrigTri->P2[2]*MatTrans[2][1] + MatTrans[3][1];
    TmpTri->P2[2] = TmpOrigTri->P2[0]*MatTrans[0][2] + TmpOrigTri->P2[1]*MatTrans[1][2] + TmpOrigTri->P2[2]*MatTrans[2][2] + MatTrans[3][2];
    TmpTri->P3[0] = TmpOrigTri->P3[0]*MatTrans[0][0] + TmpOrigTri->P3[1]*MatTrans[1][0] + TmpOrigTri->P3[2]*MatTrans[2][0] + MatTrans[3][0];
    TmpTri->P3[1] = TmpOrigTri->P3[0]*MatTrans[0][1] + TmpOrigTri->P3[1]*MatTrans[1][1] + TmpOrigTri->P3[2]*MatTrans[2][1] + MatTrans[3][1];
    TmpTri->P3[2] = TmpOrigTri->P3[0]*MatTrans[0][2] + TmpOrigTri->P3[1]*MatTrans[1][2] + TmpOrigTri->P3[2]*MatTrans[2][2] + MatTrans[3][2];
  }
  DrawSelection(DrawingArea, AllSelectionTri, giram_purple_gc, Xoff, Yoff, Zoom);
}

/*****************************************************************************
*  ToolReleaseRotateSelection
******************************************************************************/
static void ToolReleaseRotateSelection(GtkWidget *DrawingArea, GdkEventButton *Bev)
{
  ViewStruct        *view_data;
  GtkWidget         *StatusBar;
  double             Zoom, Xoff, Yoff;
  double             XTrans, YTrans;
  double             Norme, CosAngle, Angle;
  Vector             RotateVect, TransVect;
  GList             *selection;
  ObjectStruct      *tmp_object;
  GSList            *tmp_list;
  ViewStruct        *TmpView;
/*  GiramObjectEditor *goe;*/

  if (ROTATE_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);
  XTrans = Xoff+(Bev->x-DrawingArea->allocation.width /2)/Zoom;
  YTrans = Yoff-(Bev->y-DrawingArea->allocation.height/2)/Zoom;

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      XTrans -= view_data->frame->RotationCenter[0];
      YTrans -= view_data->frame->RotationCenter[1];
      break;

    case ORTHO_XZ_CAMERA:
      XTrans -= view_data->frame->RotationCenter[0];
      YTrans -= view_data->frame->RotationCenter[2];
      break;

    case ORTHO_ZY_CAMERA:
      XTrans -= view_data->frame->RotationCenter[2];
      YTrans -= view_data->frame->RotationCenter[1];
      break;
  }
  Norme = sqrt(XTrans*XTrans+YTrans*YTrans);
  if (Norme == 0.0)
    Angle = 0.0;
  else
  {
    CosAngle = (Xorg*XTrans+Yorg*YTrans) / (OrgNorme * Norme);
    Angle = acos(CosAngle);
    if (XTrans*Yorg-Xorg*YTrans > 0.0)
    {
      Angle = -Angle;
    }
  }
  if (SnapToAngleFlag)
  {
    Angle = SnapAngle*(floor(Angle*180.0/M_PI/SnapAngle))/180.0*M_PI;
  }
  Angle = Angle*180/M_PI;
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(RotateVect, 0.0, 0.0, Angle);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(RotateVect, 0.0, -Angle, 0.0);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(RotateVect, -Angle, 0.0, 0.0);
      break;
  }
  TransVect[0] = -view_data->frame->RotationCenter[0];
  TransVect[1] = -view_data->frame->RotationCenter[1];
  TransVect[2] = -view_data->frame->RotationCenter[2];
  for (selection = view_data->frame->selection ;
       selection ;
       selection = g_list_next(selection) )
  {
    tmp_object = selection->data;
    giram_object_translate(tmp_object, TransVect);
    giram_object_rotate(tmp_object, RotateVect);
    giram_object_translate(tmp_object, view_data->frame->RotationCenter);
    giram_object_build_triangle_mesh(tmp_object);
  }

  ROTATE_STATE = NONE;
  { /* Freeing the two temporary mesh */
    TriangleListStruct *Tmp1, *Tmp2;
    Tmp1 = AllSelectionTri;
    AllSelectionTri = NULL;
    while(Tmp1)
    {
      Tmp2=Tmp1->Next;
      g_free(Tmp1);
      Tmp1=Tmp2;
    }
    Tmp1 = InitialSelectionTri;
    InitialSelectionTri = NULL;
    while(Tmp1)
    {
      Tmp2=Tmp1->Next;
      g_free(Tmp1);
      Tmp1=Tmp2;
    }
  }

  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_object_editor_update(csgtree_object_editor);
  giram_create_tree_model(view_data->frame);

  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/****************************************************************************
*  tool_rotate_cursor_update
*****************************************************************************/
static void tool_rotate_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_rotate_register
*****************************************************************************/
GiramTool *giram_tool_rotate_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("Rotate");
  tool->Icon           = rotate_icon;
  tool->Path           = "<ToolBar>";
  tool->ID             = MT_ROTATE;
  tool->OptionsFunc    = BuildRotateOptions;
  tool->button_press   = ToolInitRotateSelection;
  tool->motion         = ToolMotionRotateSelection;
  tool->button_release = ToolReleaseRotateSelection;
  tool->cursor_update  = tool_rotate_cursor_update;

  return tool;
}

