/* tool_move.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <math.h>
#include "giram.h"
#include "object.h"
#include "trimesh.h"
#include "view.h"
#include "object.h"
#include "snap.h"
#include "utils.h"
#include "csgtree.h"
#include "giramobjecteditor.h"
#include "pixmaps/move.xpm"
#include "giramcursor.h"
#include "widgets/giramviewshell.h"

#include "tool_move.h"

#include "giramintl.h"

TriangleListStruct *AllSelectionTri, *InitialSelectionTri;
double Xorg, Yorg;
guint id;

static GdkPixmap *backpixmap = NULL;

#define NONE   1
#define MOVING 2
int TRANSLATE_STATE = NONE;

/*****************************************************************************
*  ToolInitMoveSelection
******************************************************************************/
static void ToolInitMoveSelection(GtkWidget *DrawingArea, GdkEventButton *bevent)
{
  ViewStruct *view_data;
  double      Zoom, Xoff, Yoff;
  GtkWidget  *StatusBar;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(StatusBar), "translate");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  MouseToReal(bevent->x, bevent->y, DrawingArea, &Xorg, &Yorg);

  if (view_data->frame->selection == NULL)
    return;

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(DrawingArea->window,
                              DrawingArea->allocation.width,
                              DrawingArea->allocation.height,
                              -1);

  draw_unselected(DrawingArea, backpixmap, view_data->camera_style,
                  DrawingArea->allocation.width,
                  DrawingArea->allocation.height);
  gdk_window_set_back_pixmap(DrawingArea->window, backpixmap, FALSE);
  gdk_window_clear(DrawingArea->window);

  AllSelectionTri = ComputeSelectionTriangleMesh(view_data->frame->selection);
  InitialSelectionTri = ComputeSelectionTriangleMesh(view_data->frame->selection);

  DrawSelection(DrawingArea, AllSelectionTri, giram_purple_gc, Xoff, Yoff, Zoom);

  TRANSLATE_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(StatusBar), id, _("Translation vector: "));
  gdk_pointer_grab(DrawingArea->window, FALSE,
                   GDK_POINTER_MOTION_MASK |
                   GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  ToolMotionMoveSelection
******************************************************************************/
static void ToolMotionMoveSelection(GtkWidget *DrawingArea, GdkEventMotion *Mev)
{
  ViewStruct         *view_data;
  GtkWidget          *StatusBar;
  double              Zoom, Xoff, Yoff, Xold, Yold;
  double              XTrans, YTrans;
  Vector              Trans;
  TriangleListStruct *TmpTri, *TmpOrigTri;
  gchar              *Message;

  if (TRANSLATE_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);

  MouseToReal(Mev->x, Mev->y, DrawingArea, &Xold, &Yold);

  XTrans = Xold-Xorg;
  YTrans = Yold-Yorg;
  if (Mev->state & GDK_SHIFT_MASK)
  {
    if (fabs(XTrans) > fabs(YTrans))
      YTrans = 0.0;
    else
      XTrans = 0.0;
  }
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(Trans, XTrans, YTrans, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(Trans, XTrans, 0.0, YTrans);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(Trans, 0.0, YTrans, XTrans);
      break;
  }
  Message = g_strdup_printf(_("Translation vector: <%g, %g, %g>"),
                            Trans[0], Trans[1], Trans[2]);
  gtk_statusbar_push(GTK_STATUSBAR(StatusBar), id, Message);
  g_free(Message);
  for (TmpTri = AllSelectionTri, TmpOrigTri = InitialSelectionTri ;
       TmpTri ;
       TmpTri = TmpTri->Next, TmpOrigTri = TmpOrigTri->Next)
  {
    TmpTri->P1[0] = TmpOrigTri->P1[0] + Trans[0];
    TmpTri->P1[1] = TmpOrigTri->P1[1] + Trans[1];
    TmpTri->P1[2] = TmpOrigTri->P1[2] + Trans[2];
    TmpTri->P2[0] = TmpOrigTri->P2[0] + Trans[0];
    TmpTri->P2[1] = TmpOrigTri->P2[1] + Trans[1];
    TmpTri->P2[2] = TmpOrigTri->P2[2] + Trans[2];
    TmpTri->P3[0] = TmpOrigTri->P3[0] + Trans[0];
    TmpTri->P3[1] = TmpOrigTri->P3[1] + Trans[1];
    TmpTri->P3[2] = TmpOrigTri->P3[2] + Trans[2];
  }
  gdk_window_clear(DrawingArea->window);
  DrawSelection(DrawingArea, AllSelectionTri, giram_purple_gc, Xoff, Yoff, Zoom);
}

/*****************************************************************************
*  ToolReleaseMoveSelection
******************************************************************************/
static void ToolReleaseMoveSelection(GtkWidget *DrawingArea, GdkEventButton *Bev)
{
  ViewStruct        *view_data;
  GtkWidget         *StatusBar;
  double             Xold, Yold;
  double             Zoom;
  double             XTrans, YTrans;
  Vector             Trans;
  GList             *selection;
  ObjectStruct      *tmp_object;
  GSList            *tmp_list;
  ViewStruct        *TmpView;

  if (TRANSLATE_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;

  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);
  MouseToReal(Bev->x, Bev->y, DrawingArea, &Xold, &Yold);

  XTrans = Xold-Xorg;
  YTrans = Yold-Yorg;
  if (Bev->state & GDK_SHIFT_MASK)
  {
    if (fabs(XTrans) > fabs(YTrans))
      YTrans = 0.0;
    else
      XTrans = 0.0;
  }
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(Trans, XTrans, YTrans, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(Trans, XTrans, 0.0, YTrans);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(Trans, 0.0, YTrans, XTrans);
      break;
  }
  for (selection = view_data->frame->selection ;
       selection ;
       selection = g_list_next(selection) )
  {
    tmp_object = selection->data;
    giram_object_translate(tmp_object, Trans);
    giram_object_build_triangle_mesh(tmp_object);
  }

  TRANSLATE_STATE = NONE;
  { /* Freeing the two temporary mesh */
    TriangleListStruct *Tmp1, *Tmp2;

    Tmp1 = AllSelectionTri;
    AllSelectionTri = NULL;
    while(Tmp1)
    {
      Tmp2=Tmp1->Next;
      g_free(Tmp1);
      Tmp1=Tmp2;
    }
    Tmp1 = InitialSelectionTri;
    InitialSelectionTri = NULL;
    while(Tmp1)
    {
      Tmp2=Tmp1->Next;
      g_free(Tmp1);
      Tmp1=Tmp2;
    }
  }
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_object_editor_update(csgtree_object_editor);
  giram_create_tree_model(view_data->frame);

  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/****************************************************************************
*  tool_move_cursor_update
*****************************************************************************/
static void tool_move_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GDK_FLEUR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_move_register
*****************************************************************************/
GiramTool *giram_tool_move_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("Move");
  tool->Icon           = move_icon;
  tool->Path           = "<ToolBar>";
  tool->ID             = MT_MOVE;
  tool->OptionsFunc    = NULL;
  tool->button_press   = ToolInitMoveSelection;
  tool->motion         = ToolMotionMoveSelection;
  tool->button_release = ToolReleaseMoveSelection;
  tool->cursor_update  = tool_move_cursor_update;

  return tool;
}

