/* light_source.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "giram.h"
#include "csgtree.h"
#include "snap.h"
#include "utils.h"
#include "giramcursor.h"

#include "pixmaps/light_source.xpm"

#include "tool_light_source.h"

#include "giramintl.h"

/*****************************************************************************
*  NewLightSourceOK
******************************************************************************/
static void NewLightSourceOK(GtkWidget *Dialog)
{
  GObject           **Data;
  Vector              Position;
  Vector              Color;
  LightSourceStruct  *TmpLightSource;
  GSList             *tmp_list;
  ViewStruct         *TmpView;
  ViewStruct         *view_data;

  Data = g_object_get_data(G_OBJECT(Dialog), "WidgetData");
  view_data = g_object_get_data(Data[0], "ViewData");
  Position[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[1])));
  Position[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[2])));
  Position[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[3])));
  gtk_color_selection_get_color(GTK_COLOR_SELECTION(Data[4]), Color);

  TmpLightSource = g_new(LightSourceStruct, 1);
  V3Dcopy(TmpLightSource->Location, Position);
  V5Deq(TmpLightSource->Color, Color[0], Color[1], Color[2], 0.0, 0.0);
  TmpLightSource->Type = POINT_LIGHTSOURCE;
  V3Deq(TmpLightSource->PointAt, 0.0, 0.0, 1.0);
  TmpLightSource->Radius = 0.35;
  TmpLightSource->FallOff = 0.35;
  TmpLightSource->Tightness = 10.0;
  TmpLightSource->AreaLight = FALSE;
  V3Deq(TmpLightSource->Area1, 0.0, 0.0, 1.0);
  V3Deq(TmpLightSource->Area2, 0.0, 1.0, 0.0);
  TmpLightSource->AreaSize1 = 0.0;
  TmpLightSource->AreaSize2 = 0.0;
  TmpLightSource->Adaptive = 100.0;
  TmpLightSource->Jitter = FALSE;
  //TmpLightSource->LooksLike = NULL;
  TmpLightSource->FadeDistance = 0.0;
  TmpLightSource->FadePower = 0.0;
  TmpLightSource->AtmosphericAttenuation = FALSE;
  TmpLightSource->Atmosphere = FALSE;
  TmpLightSource->name = NULL;
  TmpLightSource->name = create_uniq_light_source_name(view_data->frame, _("light source"));
  TmpLightSource->frame = view_data->frame;
  view_data->frame->all_light_sources = g_slist_append(view_data->frame->all_light_sources,
                                                       TmpLightSource);
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gtk_widget_destroy(Dialog);
}

/*************************************************************************
*  tool_light_source_dialog_response
**************************************************************************/
static void tool_light_source_dialog_response(GtkWidget *dialog, gint response)
{
  switch (response)
  {
    case GTK_RESPONSE_ACCEPT:
      NewLightSourceOK(dialog);
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(dialog);
      break;
  }
}

/*************************************************************************
*  ToolNewLightSource
**************************************************************************/
static void ToolNewLightSource(GtkWidget *DrawingArea, GdkEventButton *bevent)
{
  GtkWidget   *Dialog, *ColorSel, *Table, *Label, *Entry;
  GtkObject  **Data;
  ViewStruct  *view_data;
  char         CoordText[20];
  double       x, y;

  view_data = get_current_view_data();

  Dialog = gtk_dialog_new_with_buttons(_("New Light Source"), NULL, 0,
                                       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                       NULL);
  gtk_window_set_position(GTK_WINDOW(Dialog), GTK_WIN_POS_MOUSE);
  g_signal_connect(G_OBJECT(Dialog), "response",
                   G_CALLBACK(tool_light_source_dialog_response), NULL);

  Data = g_new(GtkObject*, 5);

  Data[0] = GTK_OBJECT(DrawingArea);

  g_object_set_data(G_OBJECT(Dialog), "WidgetData", Data);

  /* The Table for the position of the light */
  Table = gtk_table_new(3, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(Dialog)->vbox), Table);
  MouseToReal(bevent->x, bevent->y, DrawingArea, &x, &y);
  /* The X position */
  Label = gtk_label_new("Position X");
  gtk_table_attach_defaults(GTK_TABLE(Table), Label, 0,1, 0,1);
  Entry = gtk_entry_new();
  Data[1] = GTK_OBJECT(Entry);
  gtk_table_attach_defaults(GTK_TABLE(Table), Entry, 1,2, 0,1);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      sprintf(CoordText, "%g", x);
      break;

    case ORTHO_XZ_CAMERA:
      sprintf(CoordText, "%g", x);
      break;

    case ORTHO_ZY_CAMERA:
      strcpy(CoordText, "0.0");
      break;
  }
  gtk_entry_set_text(GTK_ENTRY(Entry), CoordText);
  /* The Y position */
  Label = gtk_label_new("Position Y");
  gtk_table_attach_defaults(GTK_TABLE(Table), Label, 0,1, 1,2);
  Entry = gtk_entry_new();
  Data[2] = GTK_OBJECT(Entry);
  gtk_table_attach_defaults(GTK_TABLE(Table), Entry, 1,2, 1,2);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      sprintf(CoordText, "%g", y);
      break;

    case ORTHO_XZ_CAMERA:
      strcpy(CoordText, "0.0");
      break;

    case ORTHO_ZY_CAMERA:
      sprintf(CoordText, "%g", y);
      break;
  }
  gtk_entry_set_text(GTK_ENTRY(Entry), CoordText);
  /* The Z position */
  Label = gtk_label_new("Position Z");
  gtk_table_attach_defaults(GTK_TABLE(Table), Label, 0,1, 2,3);
  Entry = gtk_entry_new();
  Data[3] = GTK_OBJECT(Entry);
  gtk_table_attach_defaults(GTK_TABLE(Table), Entry, 1,2, 2,3);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      strcpy(CoordText, "0.0");
      break;

    case ORTHO_XZ_CAMERA:
      sprintf(CoordText, "%g", y);
      break;

    case ORTHO_ZY_CAMERA:
      sprintf(CoordText, "%g", x);
      break;
  }
  gtk_entry_set_text(GTK_ENTRY(Entry), CoordText);

  /* The Color Selector */
  ColorSel = gtk_color_selection_new();
  Data[4] = GTK_OBJECT(ColorSel);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(Dialog)->vbox), ColorSel);

  gtk_widget_show_all(Dialog);
}

/****************************************************************************
*  tool_light_source_cursor_update
*****************************************************************************/
static void tool_light_source_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_light_source_register
*****************************************************************************/
GiramTool *giram_tool_light_source_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Light Source");
  tool->Icon           = lightsource_icon;
  tool->Path           = "<ToolBar>";
  tool->ID             = MT_NEW_LIGHTSOURCE;
  tool->OptionsFunc    = NULL;
  tool->button_press   = ToolNewLightSource;
  tool->motion         = NULL;
  tool->button_release = NULL;
  tool->cursor_update  = tool_light_source_cursor_update;

  return tool;
}

