/* disc.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "primitives/disc.h"
#include "csgtree.h"
#include "snap.h"
#include "utils.h"
#include "view.h"
#include "giramcursor.h"
#include "widgets/giramviewshell.h"

#include "pixmaps/disc.xpm"

#include "tool_disc.h"

#include "giramintl.h"

static int DiscWithHoleFlag;

static double Xorg, Yorg, Radius, HoleRadius;
static guint id;
static Vector Center;

static GdkPixmap *backpixmap;

#define NONE           1
#define MOVING         2
#define BETWEEN_RADIUS 3
#define HOLE_RADIUS    4

static int DISC_STATE = NONE;

/*****************************************************************************
*  tool_disc_button_press
******************************************************************************/
static void tool_disc_button_press(GtkWidget *canvas, GdkEventButton *bevent)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;
  double      Zoom, Xoff, Yoff;

  if (DISC_STATE == BETWEEN_RADIUS)
  {
    DISC_STATE = HOLE_RADIUS;
    return;
  }
  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "new disc");
  MouseToReal(bevent->x, bevent->y, canvas, &Xorg, &Yorg);

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(canvas->window,
                              canvas->allocation.width,
                              canvas->allocation.height,
                              -1);

  draw_all(canvas, backpixmap, view_data->camera_style,
           canvas->allocation.width,
           canvas->allocation.height);
  gdk_window_set_back_pixmap(canvas->window, backpixmap, FALSE);
  gdk_window_clear(canvas->window);

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(Center, Xorg, Yorg, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(Center, Xorg, 0.0, Yorg);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(Center, 0.0, Yorg, Xorg);
      break;
  }
  Radius = 0.0;

  DISC_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, _("Disc: "));
  gdk_pointer_grab(canvas->window, FALSE,
                   GDK_BUTTON_PRESS_MASK |
                   GDK_POINTER_MOTION_MASK |
                   GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  tool_disc_motion
******************************************************************************/
static void tool_disc_motion(GtkWidget *canvas, GdkEventMotion *Mev)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;
  double      Zoom, Xoff, Yoff, Xold, Yold;
  gchar      *Message;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  if (DISC_STATE == MOVING)
  {
    gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

    gdk_window_clear(canvas->window);

    MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
    Radius = sqrt((Xold-Xorg)*(Xold-Xorg)+(Yold-Yorg)*(Yold-Yorg));
    gdk_draw_arc(canvas->window, giram_purple_gc, FALSE,
                 canvas->allocation.width/2+(Xorg-Radius-Xoff)*Zoom,
                 canvas->allocation.height/2-(Yorg+Radius-Yoff)*Zoom,
                 Radius*2.0*Zoom, Radius*2.0*Zoom,  0, 64*360);
    Message = g_strdup_printf(_("Disc: Center <%g, %g, %g>, Radius: %g"),
                              Center[0], Center[1], Center[2], Radius);
    gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
    g_free(Message);
  } else if ((DISC_STATE == HOLE_RADIUS) || (DISC_STATE == BETWEEN_RADIUS))
  {
    gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

    gdk_window_clear(canvas->window);

    MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
    HoleRadius = sqrt((Xold-Xorg)*(Xold-Xorg)+(Yold-Yorg)*(Yold-Yorg));
    gdk_draw_arc(canvas->window, giram_purple_gc, FALSE,
                 canvas->allocation.width/2+(Xorg-Radius-Xoff)*Zoom,
                 canvas->allocation.height/2-(Yorg+Radius-Yoff)*Zoom,
                 Radius*2.0*Zoom, Radius*2.0*Zoom,  0, 64*360);
    gdk_draw_arc(canvas->window, giram_purple_gc, FALSE,
                 canvas->allocation.width/2+(Xorg-HoleRadius-Xoff)*Zoom,
                 canvas->allocation.height/2-(Yorg+HoleRadius-Yoff)*Zoom,
                 HoleRadius*2.0*Zoom, HoleRadius*2.0*Zoom, 0,64*360);
    Message = g_strdup_printf(_("Disc: Center <%g, %g, %g>, Radius: %g, Hole Radius: %g"),
            Center[0], Center[1], Center[2],
            MAX(Radius, HoleRadius),
            MIN(Radius, HoleRadius));
    gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
    g_free(Message);
  }
}

/*****************************************************************************
*  tool_disc_button_release
******************************************************************************/
static void tool_disc_button_release(GtkWidget *canvas, GdkEventButton *Bev)
{
  ViewStruct   *view_data;
  GtkWidget    *status_bar;
  double        Zoom, Xoff, Yoff;
  GSList       *tmp_list;
  ViewStruct   *TmpView;
  ObjectStruct *Disc;
  Vector        Normal;

  if (DISC_STATE == MOVING)
  {
    DISC_STATE = BETWEEN_RADIUS;
    HoleRadius = 0.0;
    return;
  }
  if (DISC_STATE != HOLE_RADIUS)
    return;
  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(Normal, 0.0, 0.0, 1.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(Normal, 0.0, 1.0, 0.0);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(Normal, 1.0, 0.0, 0.0);
      break;
  }
  Disc = giram_disc_new(Center, Normal, Radius, HoleRadius);
  Disc->name = create_uniq_object_name(view_data->frame, _("disc"));
  view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                 Disc);
  Disc->frame = view_data->frame;
  giram_object_build_triangle_mesh(Disc);
  DISC_STATE = NONE;
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/*****************************************************************************
*  ToggleDiscWithHole
******************************************************************************/
static void ToggleDiscWithHole(GtkWidget *Check)
{
  DiscWithHoleFlag = !DiscWithHoleFlag;
}

/*****************************************************************************
*  BuildDiscOptions
******************************************************************************/
static GtkWidget *BuildDiscOptions(GtkWidget *VBox)
{
  GtkWidget *vbox;
  GtkWidget *Check;
  GtkWidget *label;

  /*  the main vbox  */
  vbox = gtk_vbox_new (FALSE, 1);
  /*  the main label  */
  label = gtk_label_new(_("Disc"));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

  Check = gtk_check_button_new_with_label(_("Disc with Hole"));
  DiscWithHoleFlag = TRUE;
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Check), DiscWithHoleFlag);
  g_signal_connect(G_OBJECT(Check), "toggled",
                   G_CALLBACK(ToggleDiscWithHole), NULL);
  gtk_box_pack_start(GTK_BOX(vbox), Check, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(VBox), vbox, TRUE, TRUE, 0);
  gtk_widget_show_all(vbox);
  return vbox;
}

/****************************************************************************
*  tool_disc_cursor_update
*****************************************************************************/
static void tool_disc_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_disc_register
*****************************************************************************/
GiramTool *giram_tool_disc_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Disc");
  tool->Icon           = disc_icon;
  tool->Path           = "<ToolBar/Shape2D>";
  tool->ID             = MT_NEW_DISC;
  tool->OptionsFunc    = BuildDiscOptions;
  tool->button_press   = tool_disc_button_press;
  tool->motion         = tool_disc_motion;
  tool->button_release = tool_disc_button_release;
  tool->cursor_update  = tool_disc_cursor_update;

  return tool;
}

