/* tool_cone.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "primitives/cone.h"
#include "csgtree.h"
#include "snap.h"
#include "utils.h"
#include "view.h"
#include "giramcursor.h"
#include "widgets/giramviewshell.h"

#include "pixmaps/cone.xpm"

#include "tool_cone.h"

#include "giramintl.h"

int ConeFunnelFlag = 0;
int ConeOpenFlag   = 0;

double Xorg, Yorg, Xold, Yold;
double Radius1, Radius2, ConeLength;

guint id;

#define NONE           1
#define MOVING         2
#define BETWEEN_RADIUS 3
#define SECOND_RADIUS  4

int CONE_STATE = NONE;

static GdkPixmap *backpixmap = NULL;

/*****************************************************************************
*  tool_cone_button_press
******************************************************************************/
static void tool_cone_button_press(GtkWidget *canvas, GdkEventButton *bevent)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;

  if (CONE_STATE == BETWEEN_RADIUS)
  {
    CONE_STATE = SECOND_RADIUS;
    return;
  }

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "new cone");
  MouseToReal(bevent->x, bevent->y, canvas, &Xorg, &Yorg);

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(canvas->window,
                              canvas->allocation.width,
                              canvas->allocation.height,
                              -1);

  draw_all(canvas, backpixmap, view_data->camera_style,
           canvas->allocation.width,
           canvas->allocation.height);
  gdk_window_set_back_pixmap(canvas->window, backpixmap, FALSE);
  gdk_window_clear(canvas->window);

  Xold = Xorg; Yold = Yorg;

  CONE_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, _("Cone: "));
  gdk_pointer_grab(canvas->window, FALSE,
                   GDK_BUTTON_PRESS_MASK |
                   GDK_POINTER_MOTION_MASK |
                   GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  tool_cone_motion
******************************************************************************/
static void tool_cone_motion(GtkWidget *canvas, GdkEventMotion *Mev)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;
  double      Zoom, Xoff, Yoff;
  char        Message[100];
  double      x1, x2;
  double      gx1, gy1;
  Vector      From;
  double      Length, Radius;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  if (CONE_STATE == MOVING)
  {
    gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

    gdk_window_clear(canvas->window);

    MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
    if (Xorg < Xold)
    {
      x1 = Xorg-(Xold-Xorg); x2 = Xold;
    } else
    {
      x1 = Xold; x2 = Xorg+Xorg-Xold;
    }
    gdk_draw_line(canvas->window, giram_purple_gc,
                  canvas->allocation.width/2+(x1-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom,
                  canvas->allocation.width/2+(x2-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_purple_gc,
                  canvas->allocation.width/2+(Xorg-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yold-Yoff)*Zoom,
                  canvas->allocation.width/2+(x2-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_purple_gc,
                  canvas->allocation.width/2+(x1-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom,
                  canvas->allocation.width/2+(Xorg-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yold-Yoff)*Zoom);

    gx1 = Xorg;
    gy1 = Yorg;
    Radius = fabs(Xorg-Xold);
    Length = fabs(Yorg-Yold);
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        V3Deq(From, gx1, gy1, 0.0);
        break;

      case ORTHO_XZ_CAMERA:
        V3Deq(From, gx1, 0.0, gy1);
        break;

      case ORTHO_ZY_CAMERA:
        V3Deq(From, 0.0, gy1, gx1);
        break;
    }
    sprintf(Message, _("Cone: From <%g, %g, %g>, Length: %g, Radius: %g"),
                       From[0], From[1], From[2], Length, Radius);
    gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
  } else if (!ConeFunnelFlag &&
             ((CONE_STATE == SECOND_RADIUS) || (CONE_STATE == BETWEEN_RADIUS)))
  {
    gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

    gdk_window_clear(canvas->window);

    MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
    Radius2 = Xold-Xorg;
    gdk_draw_line(canvas->window, giram_purple_gc,
                  canvas->allocation.width/2+(Xorg-Radius1-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom,
                  canvas->allocation.width/2+(Xorg+Radius1-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_purple_gc,
                  canvas->allocation.width/2+(Xorg-Radius1-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom,
                  canvas->allocation.width/2+(Xorg-Radius2-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-ConeLength-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_purple_gc,
                  canvas->allocation.width/2+(Xorg+Radius1-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-Yoff)*Zoom,
                  canvas->allocation.width/2+(Xorg+Radius2-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-ConeLength-Yoff)*Zoom);
    gdk_draw_line(canvas->window, giram_purple_gc,
                  canvas->allocation.width/2+(Xorg-Radius2-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-ConeLength-Yoff)*Zoom,
                  canvas->allocation.width/2+(Xorg+Radius2-Xoff)*Zoom,
                  canvas->allocation.height/2-(Yorg-ConeLength-Yoff)*Zoom);
    sprintf(Message, _("Cone: Length: %g, Radius 1: %g, Radius 2: %g"),
            ConeLength, Radius1, Radius2);
    gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
  }
}

/*****************************************************************************
*  tool_cone_button_release
******************************************************************************/
static void tool_cone_button_release(GtkWidget *canvas, GdkEventButton *Bev)
{
  ViewStruct    *view_data;
  GtkWidget     *status_bar;
  double         Zoom, Xoff, Yoff;
  GSList        *tmp_list;
  ViewStruct    *TmpView;
  ObjectStruct  *Cone;
  double         gx1, gy1, gx2, gy2;
  double         Radius;
  Vector         From, To;

  if (CONE_STATE == MOVING)
  {
    view_data = get_current_view_data();
    status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
    Zoom = view_data->zoom;
    Xoff = view_data->x_off;
    Yoff = view_data->y_off;

    if (ConeFunnelFlag)
    {
      gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);
      gx1 = Xorg;
      gy1 = Yorg;
      MouseToReal(Bev->x, Bev->y, canvas, &Xold, &Yold);
      gx2 = Xold;
      gy2 = Yold;
      Radius = fabs(Xorg-Xold);
      switch (view_data->camera_style)
      {
        case ORTHO_XY_CAMERA:
          V3Deq(From, gx1, gy2, 0.0);
          V3Deq(To,   gx1, gy1, 0.0);
          break;

        case ORTHO_XZ_CAMERA:
          V3Deq(From, gx1, 0.0, gy2);
          V3Deq(To,   gx1, 0.0, gy1);
          break;

        case ORTHO_ZY_CAMERA:
          V3Deq(From, 0.0, gy2, gx1);
          V3Deq(To,   0.0, gy1, gx1);
          break;
      }
      Cone = giram_cone_new(From, 0.0, To, Radius);
      Cone->name = create_uniq_object_name(view_data->frame, _("cone"));
      view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                     Cone);
      Cone->frame = view_data->frame;
      giram_object_build_triangle_mesh(Cone);
      CONE_STATE = NONE;

      for (tmp_list = view_data->frame->all_views ;
           tmp_list ;
           tmp_list = tmp_list->next)
      {
        TmpView = tmp_list->data;
        gtk_widget_queue_draw(TmpView->canvas);
      }
      giram_create_tree_model(view_data->frame);
      gdk_pointer_ungrab(Bev->time);
      g_object_unref(backpixmap);
      backpixmap = NULL;
    } else
    {
      CONE_STATE = BETWEEN_RADIUS;
      MouseToReal(Bev->x, Bev->y, canvas, &Xold, &Yold);
      Radius1 = fabs(Xorg-Xold);
      Radius2 = 0.0;
      ConeLength = Yorg-Yold;
    }
  } else if ((!ConeFunnelFlag) && (CONE_STATE == SECOND_RADIUS))
  {
    view_data = get_current_view_data();
    status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
    Zoom = view_data->zoom;
    Xoff = view_data->x_off;
    Yoff = view_data->y_off;

    gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

    gx1 = Xorg;
    gy1 = Yorg;
    MouseToReal(Bev->x, Bev->y, canvas, &Xold, &Yold);
    gx2 = Xold;
    gy2 = Yorg-ConeLength;
    Radius = fabs(Xorg-Xold);
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        V3Deq(From, gx1, gy1, 0.0);
        V3Deq(To,   gx1, gy2, 0.0);
        break;

      case ORTHO_XZ_CAMERA:
        V3Deq(From, gx1, 0.0, gy1);
        V3Deq(To,   gx1, 0.0, gy2);
        break;

      case ORTHO_ZY_CAMERA:
        V3Deq(From, 0.0, gy1, gx1);
        V3Deq(To,   0.0, gy2, gx1);
        break;
    }
    Cone = giram_cone_new(From, Radius1, To, Radius2);
    Cone->name = create_uniq_object_name(view_data->frame, _("cone"));
    view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                   Cone);
    giram_object_build_triangle_mesh(Cone);

    CONE_STATE = NONE;
    for (tmp_list = view_data->frame->all_views ;
         tmp_list ;
         tmp_list = tmp_list->next)
    {
      TmpView = tmp_list->data;
      gtk_widget_queue_draw(TmpView->canvas);
    }
    giram_create_tree_model(view_data->frame);
    gdk_pointer_ungrab(Bev->time);
    g_object_unref(backpixmap);
    backpixmap = NULL;
  }
}

/*****************************************************************************
*  ToggleConeFunnel
******************************************************************************/
static void ToggleConeFunnel(GtkWidget *Check)
{
  ConeFunnelFlag = !ConeFunnelFlag;
}

/*****************************************************************************
*  ToggleConeOpen
******************************************************************************/
static void ToggleConeOpen(GtkWidget *Check)
{
  ConeOpenFlag = !ConeOpenFlag;
}

/*****************************************************************************
*  BuildConeOptions
******************************************************************************/
static GtkWidget *BuildConeOptions(GtkWidget *VBox)
{
  GtkWidget *vbox, *check, *label;

  /*  the main vbox  */
  vbox = gtk_vbox_new (FALSE, 1);
  /*  the main label  */
  label = gtk_label_new(_("Cone"));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  /* The Funnel Check Button */
  check = gtk_check_button_new_with_label(_("Funnel Only"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), ConeFunnelFlag);
  g_signal_connect(G_OBJECT(check), "toggled",
                   G_CALLBACK(ToggleConeFunnel), NULL);
  gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, FALSE, 0);
  /* The Open Check Button */
  check = gtk_check_button_new_with_label(_("Open"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), ConeOpenFlag);
  g_signal_connect(G_OBJECT(check), "toggled",
                   G_CALLBACK(ToggleConeOpen), NULL);
  gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(VBox), vbox, TRUE, TRUE, 0);
  gtk_widget_show_all(vbox);
  return vbox;
}

/****************************************************************************
*  tool_cone_cursor_update
*****************************************************************************/
static void tool_cone_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_cone_register
*****************************************************************************/
GiramTool *giram_tool_cone_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Cone");
  tool->Icon           = cone_icon;
  tool->Path           = "<ToolBar/Shape3D>";
  tool->ID             = MT_NEW_CONE;
  tool->OptionsFunc    = BuildConeOptions;
  tool->button_press   = tool_cone_button_press;
  tool->motion         = tool_cone_motion;
  tool->button_release = tool_cone_button_release;
  tool->cursor_update  = tool_cone_cursor_update;

  return tool;
}

