/* texture.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#undef GTK_DISABLE_DEPRECATED
#warning GTK_DISABLE_DEPRECATED
#include "giram.h"
#include "utils.h"
#include "widgets/gtkcolorbutton.h"
#include "widgets/gtkcolormapbutton.h"
#include "texture.h"

#include "widgets/giramwidgets.h"


#include "giramintl.h"

#define UPDATE_SENSITIVE(a, b, c, d)  do { \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "First Color")), (a));        \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "First Color Label")), (a));  \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "Second Color")), (b));       \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "Second Color Label")), (b)); \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "Third Color")), (c));        \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "Third Color Label")), (c));  \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "Color Map")), (d));        \
 gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(editor, "Color Map Label")), (d));  \
 } while (0)

static struct {
  GiramPigmentType type;
  gchar *label;
} pigment_type_list[] =
{
  { PAT_SOLID_COLOR, "Solid"},
  { PAT_BRICK,       "Brick"},
  { PAT_CHECKER,     "Checker"},
  { PAT_HEXAGON,     "Hexagon"},
  { PAT_ONION,       "Onion"},
  { PAT_LEOPARD,     "Leopard"},
  { PAT_LAST,        NULL}
};

/*************************************************************************
*  texture_editor_first_color_changed
**************************************************************************/
static void texture_editor_first_color_changed(GtkColorButton *gcb,
                                               GtkWidget      *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->Pigment->Color[0] = gcb->color[0];
  texture->Pigment->Color[1] = gcb->color[1];
  texture->Pigment->Color[2] = gcb->color[2];
}

/*************************************************************************
*  texture_editor_second_color_changed
**************************************************************************/
static void texture_editor_second_color_changed(GtkColorButton *gcb,
                                                GtkWidget      *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->Pigment->Color2[0] = gcb->color[0];
  texture->Pigment->Color2[1] = gcb->color[1];
  texture->Pigment->Color2[2] = gcb->color[2];
}

/*************************************************************************
*  texture_editor_third_color_changed
**************************************************************************/
static void texture_editor_third_color_changed(GtkColorButton *gcb,
                                               GtkWidget      *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->Pigment->Color3[0] = gcb->color[0];
  texture->Pigment->Color3[1] = gcb->color[1];
  texture->Pigment->Color3[2] = gcb->color[2];
}
/*************************************************************************
*  texture_editor_color_map_changed
**************************************************************************/
static void texture_editor_color_map_changed(GtkColorMapButton *gcmb,
                                             GtkWidget         *editor)
{
 /* FIXME:!*/
}

/*************************************************************************
*  texture_editor_ambient_changed
**************************************************************************/
static void texture_editor_ambient_changed(GtkColorButton *gcb,
                                           GtkWidget      *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->ambient[0] = gcb->color[0];
  texture->finish->ambient[1] = gcb->color[1];
  texture->finish->ambient[2] = gcb->color[2];
}

/*************************************************************************
*  texture_editor_diffuse_changed
**************************************************************************/
static void texture_editor_diffuse_changed(GtkAdjustment *adj,
                                           GtkWidget     *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->diffuse = adj->value;
}

/*************************************************************************
*  texture_editor_reflection_changed
**************************************************************************/
static void texture_editor_reflection_changed(GtkColorButton *gcb,
                                              GtkWidget      *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->reflection[0] = gcb->color[0];
  texture->finish->reflection[1] = gcb->color[1];
  texture->finish->reflection[2] = gcb->color[2];
}

/*************************************************************************
*  texture_editor_brilliance_changed
**************************************************************************/
static void texture_editor_brilliance_changed(GtkAdjustment *adj,
                                              GtkWidget     *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->brilliance = adj->value;
}

/*************************************************************************
*  texture_editor_crand_changed
**************************************************************************/
static void texture_editor_crand_changed(GtkAdjustment *adj,
                                         GtkWidget     *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->crand = adj->value;
}

/*************************************************************************
*  texture_editor_phong_changed
**************************************************************************/
static void texture_editor_phong_changed(GtkAdjustment *adj,
                                         GtkWidget     *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->phong = adj->value;
}

/*************************************************************************
*  texture_editor_phong_size_changed
**************************************************************************/
static void texture_editor_phong_size_changed(GtkAdjustment *adj,
                                              GtkWidget     *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->phong_size = adj->value;
}

/*************************************************************************
*  texture_editor_specular_changed
**************************************************************************/
static void texture_editor_specular_changed(GtkAdjustment *adj,
                                            GtkWidget     *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->specular = adj->value;
}

/*************************************************************************
*  texture_editor_roughness_changed
**************************************************************************/
static void texture_editor_roughness_changed(GtkAdjustment *adj,
                                             GtkWidget     *editor)
{
  TextureStruct *texture;

  texture = g_object_get_data(G_OBJECT(editor), "Texture");
  texture->finish->roughness = adj->value;
}

/*************************************************************************
* texture_editor_pigment_type_changed
**************************************************************************/
static void texture_editor_pigment_type_changed(GtkWidget *entry,
                                                GObject *editor)
{
  guint          i;
  TextureStruct *texture;

  texture = g_object_get_data(editor, "Texture");
  for (i=0 ; pigment_type_list[i].label ; i++)
    if (!strcmp(pigment_type_list[i].label, gtk_entry_get_text(GTK_ENTRY(entry))))
    {
      texture->Pigment->Type = pigment_type_list[i].type;
      break;
    }

  switch (texture->Pigment->Type)
  {
    case PAT_SOLID_COLOR:
      UPDATE_SENSITIVE(TRUE, FALSE, FALSE, FALSE);
      break;
    case PAT_BRICK:
      UPDATE_SENSITIVE(TRUE, TRUE, FALSE, FALSE);
      break;
    case PAT_CHECKER:
      UPDATE_SENSITIVE(TRUE, TRUE, FALSE, FALSE);
      break;
    case PAT_HEXAGON:
      UPDATE_SENSITIVE(TRUE, TRUE, TRUE, FALSE);
      break;
    case PAT_ONION: case PAT_LEOPARD:
      UPDATE_SENSITIVE(FALSE, FALSE, FALSE, TRUE);
      break;
    case PAT_LAST:
      g_message("Error: unknown pattern type\n");
      break;
  }
}

/*************************************************************************
*  texture_editor_set_texture
**************************************************************************/
void texture_editor_set_texture(GObject *editor, TextureStruct *texture)
{
  GtkColorButton *gcb;
  guint           i;
  GtkWidget      *combo;
  GtkAdjustment  *adj;

  g_object_set_data(editor, "Texture", texture);
  /* pigment part */
  combo = g_object_get_data(editor, "Combo");
  for (i=0 ; pigment_type_list[i].label ; i++)
    if (pigment_type_list[i].type == texture->Pigment->Type)
    {
      gtk_list_select_item(GTK_LIST(GTK_COMBO(combo)->list), i);
      break;
    }

  gcb = g_object_get_data(editor, "First Color");
  gtk_color_button_set_color(gcb, texture->Pigment->Color);
  gcb = g_object_get_data(editor, "Second Color");
  gtk_color_button_set_color(gcb, texture->Pigment->Color2);
  gcb = g_object_get_data(editor, "Third Color");
  gtk_color_button_set_color(gcb, texture->Pigment->Color3);
  
  /* finish part */
  gcb = g_object_get_data(editor, "Ambient");
  gtk_color_button_set_color(gcb, texture->finish->ambient);
  adj = g_object_get_data(editor, "Diffuse");
  gtk_adjustment_set_value(adj, texture->finish->diffuse);
  gcb = g_object_get_data(editor, "Reflection");
  gtk_color_button_set_color(gcb, texture->finish->reflection);
  adj = g_object_get_data(editor, "Brilliance");
  gtk_adjustment_set_value(adj, texture->finish->brilliance);
  adj = g_object_get_data(editor, "Crand");
  gtk_adjustment_set_value(adj, texture->finish->crand);
  adj = g_object_get_data(editor, "Phong");
  gtk_adjustment_set_value(adj, texture->finish->phong);
  adj = g_object_get_data(editor, "PhongSize");
  gtk_adjustment_set_value(adj, texture->finish->phong_size);
  adj = g_object_get_data(editor, "Specular");
  gtk_adjustment_set_value(adj, texture->finish->specular);
  adj = g_object_get_data(editor, "Roughness");
  gtk_adjustment_set_value(adj, texture->finish->roughness);
}

/*****************************************************************************
*  texture_editor
******************************************************************************/
GtkWidget *texture_editor(TextureStruct *texture)
{
  GtkWidget *Dialog, *button, *label, *Combo, *table, *HBox;
  GtkWidget *notebook, *spin;
  GtkObject *adj;
  GList     *TypeList = NULL;
  guint      i;

  Dialog = gtk_dialog_new_with_buttons(_("Texture Editor"), NULL, 0,
                                       GTK_STOCK_CLOSE, GTK_RESPONSE_REJECT,
                                       NULL);
  gtk_window_set_position(GTK_WINDOW(Dialog), GTK_WIN_POS_MOUSE);
  g_signal_connect(G_OBJECT(Dialog), "response",
                   G_CALLBACK(gtk_widget_destroy), NULL);

  notebook = gtk_notebook_new();
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(Dialog)->vbox), notebook);
  /* Pigment page */
  HBox = gtk_hbox_new(FALSE, 0);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), HBox,
                           gtk_label_new(_("Pigment")));
  label = gtk_label_new(_("Pigment type:"));
  gtk_box_pack_start_defaults(GTK_BOX(HBox), label);
  /* The Pigment type combo box */
  Combo = gtk_combo_new();
  g_object_set_data(G_OBJECT(Dialog), "Combo", Combo);
  gtk_box_pack_start_defaults(GTK_BOX(HBox), Combo);
  for (i=0 ; pigment_type_list[i].label ; i++)
  {
    TypeList = g_list_append(TypeList, pigment_type_list[i].label);
  }
  gtk_combo_set_popdown_strings(GTK_COMBO(Combo), TypeList);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(Combo)->entry), FALSE);
  g_object_set_data(G_OBJECT(Dialog), "Entry", GTK_COMBO(Combo)->entry);
  g_signal_connect(G_OBJECT(GTK_COMBO(Combo)->entry), "changed",
                   G_CALLBACK(texture_editor_pigment_type_changed), Dialog);

  table = gtk_table_new(4, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(HBox), table);

  label = gtk_label_new(_("Solid color"));
  g_object_set_data(G_OBJECT(Dialog), "First Color Label", label);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 0,1);
  button = gtk_color_button_new("Pigment Color", 50, 20, NULL);
  gtk_table_attach_defaults(GTK_TABLE(table), button, 1,2, 0,1);
  g_object_set_data(G_OBJECT(Dialog), "First Color", button);
  g_signal_connect(G_OBJECT(button), "color_changed",
                   G_CALLBACK(texture_editor_first_color_changed), Dialog);

  label = gtk_label_new(_("Second color"));
  g_object_set_data(G_OBJECT(Dialog), "Second Color Label", label);
  gtk_widget_set_sensitive(label, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  button = gtk_color_button_new("Pigment Color", 50, 20, NULL);
  g_object_set_data(G_OBJECT(Dialog), "Second Color", button);
  gtk_widget_set_sensitive(button, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), button, 1,2, 1,2);
  g_signal_connect(G_OBJECT(button), "color_changed",
                   G_CALLBACK(texture_editor_second_color_changed), Dialog);

  label = gtk_label_new(_("Third color"));
  g_object_set_data(G_OBJECT(Dialog), "Third Color Label", label);
  gtk_widget_set_sensitive(label, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  button = gtk_color_button_new("Pigment Color", 50, 20, NULL);
  g_object_set_data(G_OBJECT(Dialog), "Third Color", button);
  gtk_widget_set_sensitive(button, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), button, 1,2, 2,3);
  g_signal_connect(G_OBJECT(button), "color_changed",
                   G_CALLBACK(texture_editor_third_color_changed), Dialog);

  label = gtk_label_new(_("Color map"));
  g_object_set_data(G_OBJECT(Dialog), "Color Map Label", label);
  gtk_widget_set_sensitive(label, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  button = gtk_color_map_button_new(50, 20, NULL);
  g_object_set_data(G_OBJECT(Dialog), "Color Map", button);
  gtk_widget_set_sensitive(button, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), button, 1,2, 3,4);
  g_signal_connect(G_OBJECT(button), "color_map_changed",
                   G_CALLBACK(texture_editor_color_map_changed), Dialog);

  /* Finish page */
  table = gtk_table_new(9, 2, FALSE);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                           gtk_label_new(_("Finish")));

  /* Ambient (color) */
  label = gtk_label_new(_("Ambient:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 0,1);
  button = gtk_color_button_new("Ambient", 30, 20, NULL);
  gtk_table_attach_defaults(GTK_TABLE(table), button, 1,2, 0,1);
  g_object_set_data(G_OBJECT(Dialog), "Ambient", button);
  g_signal_connect(G_OBJECT(button), "color_changed",
                   G_CALLBACK(texture_editor_ambient_changed), Dialog);

  /* Diffuse (double) */
  label = gtk_label_new(_("Diffuse"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  spin = giram_spin_button_new(&adj, 0.6, 0.0, 1.0, 0.001, 0.01, 0.01, 1.0, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1,2, 1,2);
  g_object_set_data(G_OBJECT(Dialog), "Diffuse", adj);
  g_signal_connect(adj, "value_changed",
                   G_CALLBACK(texture_editor_diffuse_changed), Dialog);

  /* Reflection (color) */
  label = gtk_label_new(_("Reflection:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  button = gtk_color_button_new("Reflection", 30, 20, NULL);
  gtk_table_attach_defaults(GTK_TABLE(table), button, 1,2, 2,3);
  g_object_set_data(G_OBJECT(Dialog), "Reflection", button);
  g_signal_connect(G_OBJECT(button), "color_changed",
                   G_CALLBACK(texture_editor_reflection_changed), Dialog);

  /* Brilliance (double) */
  label = gtk_label_new(_("Brilliance"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  spin = giram_spin_button_new(&adj, 0.6, 1.0, 10.0, 0.001, 0.01, 0.01, 1.0, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1,2, 3,4);
  g_object_set_data(G_OBJECT(Dialog), "Brilliance", adj);
  g_signal_connect(adj, "value_changed",
                   G_CALLBACK(texture_editor_brilliance_changed), Dialog);

  /* Crand (double) */
  label = gtk_label_new(_("Crand"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 4,5);
  spin = giram_spin_button_new(&adj, 0.0, 0.0, 1.0, 0.0001, 0.001, 0.001, 0.001, 5);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1,2, 4,5);
  g_object_set_data(G_OBJECT(Dialog), "Crand", adj);
  g_signal_connect(adj, "value_changed",
                   G_CALLBACK(texture_editor_crand_changed), Dialog);

  /* Phong (double) */
  label = gtk_label_new(_("Phong"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 5,6);
  spin = giram_spin_button_new(&adj, 0.0, 0.0, 1.0, 0.01, 0.01, 0.01, 0.01, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1,2, 5,6);
  g_object_set_data(G_OBJECT(Dialog), "Phong", adj);
  g_signal_connect(adj, "value_changed",
                   G_CALLBACK(texture_editor_phong_changed), Dialog);

  /* PhongSize (double) */
  label = gtk_label_new(_("PhongSize"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 6,7);
  spin = giram_spin_button_new(&adj, 40.0, 1.0, 250.0, 1, 1, 1, 1, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1,2, 6,7);
  g_object_set_data(G_OBJECT(Dialog), "PhongSize", adj);
  g_signal_connect(adj, "value_changed",
                   G_CALLBACK(texture_editor_phong_size_changed), Dialog);

  /* Specular (double) */
  label = gtk_label_new(_("Specular"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 7,8);
  spin = giram_spin_button_new(&adj, 0.0, 0.0, 1.0, 0.01, 0.01, 0.01, 0.01, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1,2, 7,8);
  g_object_set_data(G_OBJECT(Dialog), "Specular", adj);
  g_signal_connect(adj, "value_changed",
                   G_CALLBACK(texture_editor_specular_changed), Dialog);

  /* PhongSize (double) */
  label = gtk_label_new(_("Roughness"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 8,9);
  spin = giram_spin_button_new(&adj, 0.05, 0.0005, 1.0, 0.0001, 0.001, 0.001, 0.001, 6);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1,2, 8,9);
  g_object_set_data(G_OBJECT(Dialog), "Roughness", adj);
  g_signal_connect(adj, "value_changed",
                   G_CALLBACK(texture_editor_roughness_changed), Dialog);

  texture_editor_set_texture(G_OBJECT(Dialog), texture);
  gtk_widget_show_all(Dialog);

  return Dialog;
}

