/* triangle.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "giram.h"
#include "triangle.h"
#include "trimesh.h"
#include "csgtree.h"

#include "giramintl.h"

static void giram_triangle_build_triangle_mesh(ObjectStruct *triangle);
static gboolean giram_triangle_inside(ObjectStruct *triangle, double x, double y, double z);
static gboolean giram_triangle_is_intersection(ObjectStruct *triangle, Vector org, Vector dir);
static gboolean giram_triangle_find_intersection_segment(ObjectStruct *triangle,
                                                         Vector in_point, Vector out_point,
                                                         Vector inter_point, Vector inter_norm);
/*****************************************************************************
*  ComputeTriangleTriangleMesh
******************************************************************************/
static void giram_triangle_build_triangle_mesh(ObjectStruct *triangle)
{ /* XXX: clipped_by */
  Vector          P1, P2, P3, N1, N2, N3;
  TriangleStruct *ttriangle = (TriangleStruct *)triangle;

  if (triangle->FirstTriangle)
    DestroyObjectTriangleMesh(triangle);
  V3Dcopy(P1, ttriangle->P1); V3Dcopy(N1, ttriangle->N1);
  V3Dcopy(P2, ttriangle->P2); V3Dcopy(N2, ttriangle->N2);
  V3Dcopy(P3, ttriangle->P3); V3Dcopy(N3, ttriangle->N3);
  if (triangle->Trans)
  {
    MEvaluatePoint(P1, triangle->Trans, P1);
    MEvaluatePoint(P2, triangle->Trans, P2);
    MEvaluatePoint(P3, triangle->Trans, P3);
    MEvaluateVector(N1, triangle->Trans, N1);
    MEvaluateVector(N2, triangle->Trans, N2);
    MEvaluateVector(N3, triangle->Trans, N3);
  }
  AddTriangleToObjectMesh(triangle, P1,P2,P3, N1,N2,N3);
}

/*****************************************************************************
*  giram_triangle_inside
******************************************************************************/
static gboolean giram_triangle_inside(ObjectStruct *triangle, double x, double y, double z)
{
  return FALSE; /* We cannot be "inside" a triangle... */
}

/*****************************************************************************
*  giram_triangle_is_intersection
******************************************************************************/
static gboolean giram_triangle_is_intersection(ObjectStruct *triangle, Vector origin, Vector direction)
{ /* XXX: clipped_by */
  Vector          org, dir;
  int             i;
  Matrix          Mat, InvMat;
  double          D;
  double          u, v;
  TriangleStruct *ttriangle = (TriangleStruct *)triangle;

  if (triangle->Trans)
  {
    MInverseEvaluatePoint(org, triangle->Trans, origin);
    MInverseEvaluateVector(dir, triangle->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      org[i] = origin[i];
      dir[i] = direction[i];
    }
  }
  Mat[0][0] = ttriangle->P2[0] - ttriangle->P1[0];
  Mat[0][1] = ttriangle->P2[1] - ttriangle->P1[1];
  Mat[0][2] = ttriangle->P2[2] - ttriangle->P1[2];
  Mat[1][0] = ttriangle->P3[0] - ttriangle->P1[0];
  Mat[1][1] = ttriangle->P3[1] - ttriangle->P1[1];
  Mat[1][2] = ttriangle->P3[2] - ttriangle->P1[2];
  Mat[2][0] = -dir[0];
  Mat[2][1] = -dir[1];
  Mat[2][2] = -dir[2];
  D = Mat[0][0]*Mat[1][1]*Mat[2][2]+Mat[0][1]*Mat[1][2]*Mat[2][0]+
      Mat[0][2]*Mat[1][0]*Mat[2][1]-Mat[2][0]*Mat[1][1]*Mat[0][2]-
      Mat[2][1]*Mat[1][2]*Mat[0][0]-Mat[2][2]*Mat[1][0]*Mat[0][1];
  if (D == 0.0)
    return FALSE;
  InvMat[0][0] =  (Mat[1][1]*Mat[2][2]-Mat[2][1]*Mat[1][2])/D;
  InvMat[0][1] = -(Mat[0][1]*Mat[2][2]-Mat[2][1]*Mat[0][2])/D;
  InvMat[0][2] =  (Mat[0][1]*Mat[1][2]-Mat[1][1]*Mat[0][2])/D;
  InvMat[1][0] = -(Mat[1][0]*Mat[2][2]-Mat[2][0]*Mat[1][2])/D;
  InvMat[1][1] =  (Mat[0][0]*Mat[2][2]-Mat[2][0]*Mat[0][2])/D;
  InvMat[1][2] = -(Mat[0][0]*Mat[1][2]-Mat[1][0]*Mat[0][2])/D;
  InvMat[2][0] =  (Mat[1][0]*Mat[2][1]-Mat[2][0]*Mat[1][1])/D;
  InvMat[2][1] = -(Mat[0][0]*Mat[2][1]-Mat[2][0]*Mat[0][1])/D;
  InvMat[2][2] =  (Mat[0][0]*Mat[1][1]-Mat[1][0]*Mat[0][1])/D;
  u = InvMat[0][0]*(org[0]-ttriangle->P1[0])+
      InvMat[1][0]*(org[1]-ttriangle->P1[1])+
      InvMat[2][0]*(org[2]-ttriangle->P1[2]);
  v = InvMat[0][1]*(org[0]-ttriangle->P1[0])+
      InvMat[1][1]*(org[1]-ttriangle->P1[1])+
      InvMat[2][1]*(org[2]-ttriangle->P1[2]);
  if ((u<0.0) || (u>1.0) || (v<0.0) || (v>1.0))
    return FALSE;
  if (u+v>1.0)
    return FALSE;
  return TRUE;
}

/*****************************************************************************
*  giram_triangle_find_intersection_segment
******************************************************************************/
static gboolean giram_triangle_find_intersection_segment(ObjectStruct *triangle,
                                                         Vector in_point, Vector out_point,
                                                         Vector inter_point, Vector inter_norm)
{
  return FALSE; /* Triangles shouldn't be used in CSG... */
}

/*****************************************************************************
*  giram_triangle_new
******************************************************************************/
ObjectStruct *giram_triangle_new(Vector P1, Vector P2, Vector P3)
{
  ObjectStruct   *triangle;
  TriangleStruct *ttriangle;
  static GiramObjectClass *triangle_klass = NULL;
  Vector          V1, V2;

  if (triangle_klass == NULL)
  {
    triangle_klass = giram_object_class_new();

    triangle_klass->name                      = _("Triangle");
    triangle_klass->build_triangle_mesh       = giram_triangle_build_triangle_mesh;
    triangle_klass->inside                    = giram_triangle_inside;
    triangle_klass->is_intersection           = giram_triangle_is_intersection;
    triangle_klass->find_intersection_segment = giram_triangle_find_intersection_segment;
  }
  ttriangle = g_new(TriangleStruct, 1);
  triangle = (ObjectStruct *)ttriangle;
  InitObject(triangle);
  triangle->Type = TRIANGLE_OBJECT;
  triangle->klass = triangle_klass;

  V3Dcopy(ttriangle->P1, P1);
  V3Dcopy(ttriangle->P2, P2);
  V3Dcopy(ttriangle->P3, P3);

  V1[0] = ttriangle->P2[0] - ttriangle->P1[0];
  V1[1] = ttriangle->P2[1] - ttriangle->P1[1];
  V1[2] = ttriangle->P2[2] - ttriangle->P1[2];
  V2[0] = ttriangle->P3[0] - ttriangle->P1[0];
  V2[1] = ttriangle->P3[1] - ttriangle->P1[1];
  V2[2] = ttriangle->P3[2] - ttriangle->P1[2];
  VCross(ttriangle->N1, V1, V2);
  VCross(ttriangle->N2, V1, V2);
  VCross(ttriangle->N3, V1, V2);

  return triangle;
}

/*****************************************************************************
*  PDBCreateTriangle
******************************************************************************/
/*int PDBCreateTriangle(int *Id, Vector *V1, Vector *V2, Vector *V3)
{
  ObjectStruct *triangle;
  FrameStruct  *LocalFrame;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  triangle = giram_triangle_new(*V1, *V2, *V3);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message(_("PDBCreateTriangle() called with an unknown Id (%d)\n"), *Id);
    return 0;
  }
  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, triangle);
  giram_object_build_triangle_mesh(triangle);
  for (tmp_list = LocalFrame->all_views ; tmp_list ; tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return triangle->Id;
}*/

