/* disc.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "disc.h"
#include "trimesh.h"
#include "csgtree.h"


#include "giramintl.h"

static void giram_disc_build_triangle_mesh(ObjectStruct *disc);
static gboolean giram_disc_inside(ObjectStruct *disc, double x, double y, double z);
static gboolean giram_disc_is_intersection(ObjectStruct *disc, Vector org, Vector dir);
static gboolean giram_disc_find_intersection_segment(ObjectStruct *disc,
                                                    Vector in_point, Vector out_point,
                                                    Vector inter_point, Vector inter_norm);

/*****************************************************************************
*  giram_disc_build_triangle_mesh
******************************************************************************/
static void giram_disc_build_triangle_mesh(ObjectStruct *disc)
{ /* XXX: clipped_by */
  DiscStruct *ddisc = (DiscStruct *)disc;
  Vector      P1, P2, P3, P4;
  Vector      N;
  int         i;

  disc->detail_level = 16; /* FIXME */

  if (disc->FirstTriangle)
    DestroyObjectTriangleMesh(disc);
  V3Deq(N, 0.0, 1.0, 0.0);
  if (disc->Trans)
  {
    MEvaluateVector(N, disc->Trans, N);
  }
  for (i=0 ; i<disc->detail_level ; i++)
  {
    P1[0] = ddisc->HoleRadius*cos(i*2.0*M_PI/disc->detail_level);
    P1[1] = 0.0;
    P1[2] = ddisc->HoleRadius*sin(i*2.0*M_PI/disc->detail_level);
    P2[0] = ddisc->Radius*cos((i*2.0+1.0)*M_PI/disc->detail_level);
    P2[1] = 0.0;
    P2[2] = ddisc->Radius*sin((i*2.0+1.0)*M_PI/disc->detail_level);
    P3[0] = ddisc->HoleRadius*cos((i+1)*2.0*M_PI/disc->detail_level);
    P3[1] = 0.0;
    P3[2] = ddisc->HoleRadius*sin((i+1)*2.0*M_PI/disc->detail_level);
    P4[0] = ddisc->Radius*cos(((i+1)*2.0+1.0)*M_PI/disc->detail_level);
    P4[1] = 0.0;
    P4[2] = ddisc->Radius*sin(((i+1)*2.0+1.0)*M_PI/disc->detail_level);
    if (disc->Trans)
    {
      MEvaluatePoint(P1, disc->Trans, P1);
      MEvaluatePoint(P2, disc->Trans, P2);
      MEvaluatePoint(P3, disc->Trans, P3);
      MEvaluatePoint(P4, disc->Trans, P4);
    }
    if (ddisc->HoleRadius > 10e-6)
      AddTriangleToObjectMesh(disc, P1,P2,P3, N,N,N);
    AddTriangleToObjectMesh(disc, P2,P3,P4, N,N,N);
  }
}

/*****************************************************************************
*  giram_disc_inside
******************************************************************************/
static gboolean giram_disc_inside(ObjectStruct *disc, double x, double y, double z)
{
  return FALSE; /* We cannot be "Inside" a flat disc... */
}

/*****************************************************************************
*  giram_disc_find_intersection_segment
******************************************************************************/
static gboolean giram_disc_find_intersection_segment(ObjectStruct *disc,
                                                    Vector in_point, Vector out_point,
                                                    Vector inter_point, Vector inter_norm)
{
  return FALSE; /* This fonction cannot be called anyway... */
}

/*****************************************************************************
*  giram_disc_is_intersection FIXME
******************************************************************************/
static gboolean giram_disc_is_intersection(ObjectStruct *disc, Vector origin, Vector direction)
{
  Vector Org, Dir;
  int    i;

  if (disc->Trans)
  {
    MInverseEvaluatePoint(Org, disc->Trans, origin);
    MInverseEvaluateVector(Dir, disc->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      Org[i] = origin[i];
      Dir[i] = direction[i];
    }
  }
  return FALSE;
}

/*****************************************************************************
*  giram_disc_new
******************************************************************************/
ObjectStruct *giram_disc_new(Vector center, Vector Norm, double Radius, double HoleRadius)
{
  ObjectStruct *disc;
  DiscStruct   *ddisc;
  static GiramObjectClass *disc_klass = NULL;
  Vector Normal, U1, U2;
  double Length, ULength;

  if (disc_klass == NULL)
  {
    disc_klass = giram_object_class_new();

    disc_klass->name                      = _("Disc");
    disc_klass->build_triangle_mesh       = giram_disc_build_triangle_mesh;
    disc_klass->inside                    = giram_disc_inside;
    disc_klass->is_intersection           = giram_disc_is_intersection;
    disc_klass->find_intersection_segment = giram_disc_find_intersection_segment;
  }
  ddisc = g_new(DiscStruct, 1);
  disc = (ObjectStruct *)ddisc;
  InitObject(disc);
  disc->Type = DISC_OBJECT;
  disc->klass = disc_klass;
  ddisc->Radius = MAX(Radius, HoleRadius);
  ddisc->HoleRadius = MIN(Radius, HoleRadius);

  disc->Trans = g_new(TransformStruct, 1);
  Normal[0] = Norm[0]; Normal[1] = Norm[1]; Normal[2] = Norm[2];
  Length = V3DLength(Normal);
  Normal[0] /= Length;
  Normal[1] /= Length;
  Normal[2] /= Length;
  if ( (fabs(Normal[0])>fabs(Normal[1])) && (fabs(Normal[0])>fabs(Normal[2])) )
  { /* The normal is mainly upon the X axis */
    V3Deq(U1, -Normal[1]/Normal[0], 1, 0);
  } else if ( (fabs(Normal[1])>fabs(Normal[0])) && (fabs(Normal[1])>fabs(Normal[2])) )
  { /* The normal is mainly upon the Y axis */
    V3Deq(U1, 1, -Normal[0]/Normal[1], 0);
  } else
  { /* The normal axis is mainly upon the Z axis */
    V3Deq(U1, 0, 1, -Normal[1]/Normal[2]);
  }
  ULength = V3DLength(U1);
  U1[0]/=ULength; U1[1]/=ULength; U1[2]/=ULength;
  VCross(U2, Normal, U1);
  ULength = V3DLength(U2);
  U2[0]/=ULength; U2[1]/=ULength; U2[2]/=ULength;

  disc->Trans->Direct[0][0] = U1[0];
  disc->Trans->Direct[0][1] = U1[1];
  disc->Trans->Direct[0][2] = U1[2];
  disc->Trans->Direct[0][3] = 0.0;
  disc->Trans->Direct[1][0] = Normal[0];
  disc->Trans->Direct[1][1] = Normal[1];
  disc->Trans->Direct[1][2] = Normal[2];
  disc->Trans->Direct[1][3] = 0.0;
  disc->Trans->Direct[2][0] = U2[0];
  disc->Trans->Direct[2][1] = U2[1];
  disc->Trans->Direct[2][2] = U2[2];
  disc->Trans->Direct[2][3] = 0.0;
  disc->Trans->Direct[3][0] = 0.0;
  disc->Trans->Direct[3][1] = 0.0;
  disc->Trans->Direct[3][2] = 0.0;
  disc->Trans->Direct[3][3] = 1.0;
  MInverse(disc->Trans->Inverse, disc->Trans->Direct);
  giram_object_translate(disc, center);

  return disc;
}

/*****************************************************************************
*  PDBCreateDisc
******************************************************************************/
/*int PDBCreateDisc(int *Id, Vector *Center, Vector *pNormal,
                  double *Radius, double *HoleRadius)
{
  ObjectStruct *disc;
  FrameStruct  *LocalFrame;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  disc = giram_disc_new(*Center, *pNormal, *Radius, *HoleRadius);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message("PDBCreateBox() called with an unknown Id (%d)\n", *Id);
    return 0;
  }

  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, disc);
  giram_object_build_triangle_mesh(disc);
  for (tmp_list = LocalFrame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return disc->Id;
}*/

