/* box.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "box.h"
#include "trimesh.h"
#include "csgtree.h"

#include "giramintl.h"

static void giram_box_build_triangle_mesh(ObjectStruct *box);
static gboolean giram_box_inside(ObjectStruct *box, double x, double y, double z);
static gboolean giram_box_is_intersection(ObjectStruct *box, Vector org, Vector dir);
static gboolean giram_box_click_in_XY(ObjectStruct *box, double x, double y);
static gboolean giram_box_click_in_XZ(ObjectStruct *box, double x, double z);
static gboolean giram_box_click_in_ZY(ObjectStruct *box, double z, double y);
static gboolean giram_box_find_intersection_segment(ObjectStruct *box,
                                                    Vector in_point, Vector out_point,
                                                    Vector inter_point, Vector inter_norm);

/*****************************************************************************
*  giram_box_build_triangle_mesh
******************************************************************************/
static void giram_box_build_triangle_mesh(ObjectStruct *box)
{ /* XXX: clipped_by */ /* XXX: should be scalable: use detaillevel in a way */
  Vector     P[8];
  Vector     N[6];
  BoxStruct *bbox = (BoxStruct *)box;

  if (box->FirstTriangle)
    DestroyObjectTriangleMesh(box);
  P[0][0] = bbox->MinCorner[0];
  P[0][1] = bbox->MinCorner[1];
  P[0][2] = bbox->MinCorner[2];
  P[1][0] = bbox->MinCorner[0];
  P[1][1] = bbox->MinCorner[1];
  P[1][2] = bbox->MaxCorner[2];
  P[2][0] = bbox->MinCorner[0];
  P[2][1] = bbox->MaxCorner[1];
  P[2][2] = bbox->MinCorner[2];
  P[3][0] = bbox->MinCorner[0];
  P[3][1] = bbox->MaxCorner[1];
  P[3][2] = bbox->MaxCorner[2];
  P[4][0] = bbox->MaxCorner[0];
  P[4][1] = bbox->MinCorner[1];
  P[4][2] = bbox->MinCorner[2];
  P[5][0] = bbox->MaxCorner[0];
  P[5][1] = bbox->MinCorner[1];
  P[5][2] = bbox->MaxCorner[2];
  P[6][0] = bbox->MaxCorner[0];
  P[6][1] = bbox->MaxCorner[1];
  P[6][2] = bbox->MinCorner[2];
  P[7][0] = bbox->MaxCorner[0];
  P[7][1] = bbox->MaxCorner[1];
  P[7][2] = bbox->MaxCorner[2];
  V3Deq(N[0],  0.0, -1.0,  0.0);
  V3Deq(N[1],  0.0,  1.0,  0.0);
  V3Deq(N[2],  0.0,  0.0, -1.0);
  V3Deq(N[3],  0.0,  0.0,  1.0);
  V3Deq(N[4], -1.0,  0.0,  0.0);
  V3Deq(N[5],  1.0,  0.0,  0.0);

  if (box->Inverse)
  {
    N[0][1] = 1.0; N[1][1] = -1.0;
    N[2][2] = 1.0; N[3][2] = -1.0;
    N[4][0] = 1.0; N[5][0] = -1.0;
  }

  if (box->Trans)
  {
    MEvaluatePoint(P[0], box->Trans, P[0]);
    MEvaluatePoint(P[1], box->Trans, P[1]);
    MEvaluatePoint(P[2], box->Trans, P[2]);
    MEvaluatePoint(P[3], box->Trans, P[3]);
    MEvaluatePoint(P[4], box->Trans, P[4]);
    MEvaluatePoint(P[5], box->Trans, P[5]);
    MEvaluatePoint(P[6], box->Trans, P[6]);
    MEvaluatePoint(P[7], box->Trans, P[7]);
    MEvaluateVector(N[0], box->Trans, N[0]);
    MEvaluateVector(N[1], box->Trans, N[1]);
    MEvaluateVector(N[2], box->Trans, N[2]);
    MEvaluateVector(N[3], box->Trans, N[3]);
    MEvaluateVector(N[4], box->Trans, N[4]);
    MEvaluateVector(N[5], box->Trans, N[5]);
  }

  AddTriangleToObjectMesh(box, P[0],P[1],P[4], N[0], N[0], N[0]);
  AddTriangleToObjectMesh(box, P[5],P[1],P[4], N[0], N[0], N[0]);
  AddTriangleToObjectMesh(box, P[2],P[3],P[6], N[1], N[1], N[1]);
  AddTriangleToObjectMesh(box, P[7],P[3],P[6], N[1], N[1], N[1]);
  AddTriangleToObjectMesh(box, P[0],P[2],P[4], N[2], N[2], N[2]);
  AddTriangleToObjectMesh(box, P[2],P[6],P[4], N[2], N[2], N[2]);
  AddTriangleToObjectMesh(box, P[1],P[3],P[5], N[3], N[3], N[3]);
  AddTriangleToObjectMesh(box, P[7],P[3],P[5], N[3], N[3], N[3]);
  AddTriangleToObjectMesh(box, P[0],P[1],P[2], N[4], N[4], N[4]);
  AddTriangleToObjectMesh(box, P[1],P[2],P[3], N[4], N[4], N[4]);
  AddTriangleToObjectMesh(box, P[4],P[5],P[6], N[5], N[5], N[5]);
  AddTriangleToObjectMesh(box, P[5],P[6],P[7], N[5], N[5], N[5]);
}

/*****************************************************************************
*  giram_box_inside
******************************************************************************/
static gboolean giram_box_inside(ObjectStruct *box, double x, double y, double z)
{
  Vector     vect;
  BoxStruct *bbox = (BoxStruct *)box;

  V3Deq(vect, x, y, z);
  if (box->Trans)
  {
    MInverseEvaluatePoint(vect, box->Trans, vect);
  }
  if ((vect[0] < bbox->MinCorner[0]) || (vect[0] > bbox->MaxCorner[0]) ||
      (vect[1] < bbox->MinCorner[1]) || (vect[1] > bbox->MaxCorner[1]) ||
      (vect[2] < bbox->MinCorner[2]) || (vect[2] > bbox->MaxCorner[2]))
    return box->Inverse;
  else
    return !(box->Inverse);
}

/*****************************************************************************
*  giram_box_is_intersection
******************************************************************************/
static gboolean giram_box_is_intersection(ObjectStruct *box, Vector origin, Vector direction)
{
  Vector     Org, Dir;
  int        i;
  double     tnear, tfar;
  double     t1, t2, tmp;
  BoxStruct *bbox = (BoxStruct *)box;

  if (box->Trans)
  {
    MInverseEvaluatePoint(Org, box->Trans, origin);
    MInverseEvaluateVector(Dir, box->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      Org[i] = origin[i];
      Dir[i] = direction[i];
    }
  }

  tnear = -G_MAXDOUBLE; tfar = G_MAXDOUBLE;

  if (fabs(Dir[0]) > 10e-6)
  {
    t1 = (bbox->MinCorner[0]-Org[0]) / Dir[0];
    t2 = (bbox->MaxCorner[0]-Org[0]) / Dir[0];
    if ( t1 > t2 )
    {
      tmp = t1; t1 = t2 ; t2 = tmp;
    }
    if (t1 > tnear)
      tnear = t1;
    if (t2 < tfar)
      tfar = t2;
    if (tnear > tfar)
      return 0;
  } else
  {
    if (Org[0] < bbox->MinCorner[0] ||
        Org[0] > bbox->MaxCorner[0])
      return 0;
  }
  if (fabs(Dir[1]) > 10e-6)
  {
    t1 = (bbox->MinCorner[1]-Org[1]) / Dir[1];
    t2 = (bbox->MaxCorner[1]-Org[1]) / Dir[1];
    if ( t1 > t2 )
    {
      tmp = t1; t1 = t2 ; t2 = tmp;
    }
    if (t1 > tnear)
      tnear = t1;
    if (t2 < tfar)
      tfar = t2;
    if (tnear > tfar)
      return 0;
  } else
  {
    if (Org[1] < bbox->MinCorner[1] ||
        Org[1] > bbox->MaxCorner[1])
      return 0;
  }
  if (fabs(Dir[2]) > 10e-6)
  {
    t1 = (bbox->MinCorner[2]-Org[2]) / Dir[2];
    t2 = (bbox->MaxCorner[2]-Org[2]) / Dir[2];
    if ( t1 > t2 )
    {
      tmp = t1; t1 = t2 ; t2 = tmp;
    }
    if (t1 > tnear)
      tnear = t1;
    if (t2 < tfar)
      tfar = t2;
    if (tnear > tfar)
      return 0;
  } else
  {
    if (Org[2] < bbox->MinCorner[2] ||
        Org[2] > bbox->MaxCorner[2])
      return 0;
  }
  return 1;
}

/*****************************************************************************
*  giram_box_click_in_XY
******************************************************************************/
static gboolean giram_box_click_in_XY(ObjectStruct *box, double x, double y)
{
  Vector     origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector     direction = {0.0, 0.0, 1.0, 0.0, 0.0};
  BoxStruct *bbox = (BoxStruct *)box;

  if (box->Trans)
  {
    origin[0] = x;
    origin[1] = y;
    return giram_object_is_intersection(box, origin, direction);
  } else if ( (bbox->MinCorner[0] < x) && (bbox->MaxCorner[0] > x) &&
              (bbox->MinCorner[1] < y) && (bbox->MaxCorner[1] > y))
    return !(box->Inverse);
  else
    return box->Inverse;
}

/****************************************************************************
*  giram_box_click_in_XZ
*****************************************************************************/
static gboolean giram_box_click_in_XZ(ObjectStruct *box, double x, double z)
{
  Vector     origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector     direction = {0.0, 1.0, 0.0, 0.0, 0.0};
  BoxStruct *bbox = (BoxStruct *)box;

  if (box->Trans)
  {
    origin[0] = x;
    origin[2] = z;
    return giram_object_is_intersection(box, origin, direction);
  } else if ( (bbox->MinCorner[0] < x) && (bbox->MaxCorner[0] > x) &&
              (bbox->MinCorner[2] < z) && (bbox->MaxCorner[2] > z))
    return !(box->Inverse);
  else
    return box->Inverse;
}

/****************************************************************************
*  giram_box_click_in_ZY
*****************************************************************************/
static gboolean giram_box_click_in_ZY(ObjectStruct *box, double z, double y)
{
  Vector     origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector     direction = {1.0, 0.0, 0.0, 0.0, 0.0};
  BoxStruct *bbox = (BoxStruct *)box;

  if (box->Trans)
  {
    origin[2] = z;
    origin[1] = y;
    return giram_object_is_intersection(box, origin, direction);
  } else if ( (bbox->MinCorner[2] < z) && (bbox->MaxCorner[2] > z) &&
              (bbox->MinCorner[1] < y) && (bbox->MaxCorner[1] > y))
    return !(box->Inverse);
  else
    return box->Inverse;
}

/*****************************************************************************
*  giram_box_find_intersection_segment
******************************************************************************/
static gboolean giram_box_find_intersection_segment(ObjectStruct *box,
                                                    Vector in_point, Vector out_point,
                                                    Vector inter_point, Vector inter_norm)
{
  inter_point[0] = (in_point[0]+out_point[0])/2.0;
  inter_point[1] = (in_point[1]+out_point[1])/2.0;
  inter_point[2] = (in_point[2]+out_point[2])/2.0;
  /* FIXME */ return 0;
}

/*****************************************************************************
*  giram_box_new
******************************************************************************/
ObjectStruct *giram_box_new(Vector MinCoord, Vector MaxCoord)
{
  ObjectStruct *Box;
  BoxStruct    *BBox;

  static GiramObjectClass *box_klass = NULL;

  if (box_klass == NULL)
  {
    box_klass = giram_object_class_new();

    box_klass->name                      = _("Box");
    box_klass->build_triangle_mesh       = giram_box_build_triangle_mesh;
    box_klass->inside                    = giram_box_inside;
    box_klass->is_intersection           = giram_box_is_intersection;
    box_klass->click_in_XY               = giram_box_click_in_XY;
    box_klass->click_in_XZ               = giram_box_click_in_XZ;
    box_klass->click_in_ZY               = giram_box_click_in_ZY;
    box_klass->find_intersection_segment = giram_box_find_intersection_segment;
  }
  BBox = g_new(BoxStruct, 1);
  Box = (ObjectStruct *)BBox;
  InitObject(Box);
  Box->Type = BOX_OBJECT;
  Box->klass = box_klass;
  BBox->MinCorner[0] = MIN(MinCoord[0], MaxCoord[0]);
  BBox->MinCorner[1] = MIN(MinCoord[1], MaxCoord[1]);
  BBox->MinCorner[2] = MIN(MinCoord[2], MaxCoord[2]);
  BBox->MaxCorner[0] = MAX(MaxCoord[0], MinCoord[0]);
  BBox->MaxCorner[1] = MAX(MaxCoord[1], MinCoord[1]);
  BBox->MaxCorner[2] = MAX(MaxCoord[2], MinCoord[2]);
  return Box;
}

/*****************************************************************************
*  PDBCreateBox
******************************************************************************/
/*int PDBCreateBox(int *Id, Vector *MinCoord, Vector *MaxCoord)
{
  ObjectStruct    *Box;
  FrameStruct     *LocalFrame;
  GSList          *tmp_list;
  ViewStruct      *TmpView;

  Box = giram_box_new(*MinCoord, *MaxCoord);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message("PDBCreateBox() called with an unknown Id (%d)\n", *Id);
    return 0;
  }
  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, Box);
  giram_object_build_triangle_mesh(Box);
  for (tmp_list = LocalFrame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return Box->Id;
}*/

