/* dxf_read.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2001 DindinX <David@dindinx.org>
 * Copyright (C) 1999-2001 Noah Davis of VIEWS Net, Inc. <noah@viewsnet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * This file is from Noah Davis.
 */

/*
 * Funcs for dealing with file access of a DXF ASCII file.
 * Noah Davis
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>

#ifndef DXF_STANDALONE
#include "giram.h"
#endif

#include "dxf.h"


/* Open a DFX file stream.
 * This and dfx_close() are the only two functions shared by both ASCII
 * and binary DFX reads.
 */

FILE *dxf_open(char *fname)
{
  FILE *fp;

  fp = fopen(fname, "r");

  return fp;
}

/* Get a line from a DXF file. Return NULL on error.*/
int dxf_getline(FILE *fp, char *line, DXF_Data *dat)
{
  if (dat->type == DXF_ASCII)
    return (dxf_getaline(fp, line));
  else
  {
    fprintf(stderr, "Binary parsing not implemented yet\n");
    return FALSE;
  }
}

/* Get a line from a DXF ASCII file.
 * returns positive on success, FALSE otherwise.
 */
int dxf_getaline(FILE *fp, char *line)
{
  char *err;
  err = fgets(line, DXF_LINE_MAX, fp);

  if (err==NULL)
    return FALSE;
  else
    return TRUE;
}

/* Close a DFX file stream.
 * This and dfx_open() are the only two functions shared between ASCII
 * and binary DFX reads.
 */
int dxf_close(FILE *fp)
{
  return (fclose(fp));
}

/* Determine the type of error for a DXF ASCII file read.
 * Since fgets() doesn't discern between EOF and other stream errors,
 * we figgure it out here.
 * returns either EOF (for end of file), or the value of the global errno.
 */
int dxf_error(FILE *fp)
{
  if (feof(fp))
    return EOF;
  else
    return errno;
}

/* Get a group code and value pair (wrapper)
 */
int dxf_getpair(FILE *fp, int *code, char *value, DXF_Data *dat)
{
  int good;

  if (dat->type == DXF_ASCII)
  {
    good = dxf_getapair(fp, code, value);
    if (!good) return (good);
    if ((code==NULL)||(value==NULL))
       return(good);
    while ((*code) == 999)
    {
	    good = dxf_getapair(fp, code, value);
      if (!good) return (good);
    }
    return (good);
  }
  else
    return (dxf_getbpair(fp, code, value));
}

/* Get a group code and value pair. Binary style.
 * Returns FALSE on fail, else positive.
 * Silly but nescessary use of fseek(). May break char streams on some
 * platforms... should maybe use fgetpos() and fsetpos(). If 'code' and
 * 'value' are NULL when this func is called, the rewind is enabled, and no
 * read is done. (return value is pointless)
 * This routine makes some BOLD assumptions about datatype sizes and
 * endianness. This NEEDS to be corrected.
 */
int dxf_getbpair(FILE *fp,int *code, char *value)
{
  char buf[512];
  unsigned char gc;
  short int tgc;
  double dvalue;

  /* First, get the group code */
  if (!(fread(&gc, 1, 1, fp)))
    return FALSE;

  /* Check to be sure this isn't an extended code */
  if (gc == 255)
  {           /* It is */
    if (!fread (&tgc, 2, 1, fp))
      return (FALSE);
  }
  else
    tgc = (short int)gc;

  if ( ((tgc>=10)&&(tgc<=59)) || ((tgc>=140)&&(tgc<=147)) ||
       ((tgc==210) || (tgc==220) || (tgc==230)))
  {
    /* It's a double, in 8 bytes */
    if (!fread(&dvalue, 8, 1,fp))
      return FALSE;
    sprintf(buf, "%e", dvalue);
    strcpy(value, buf);
    *code = tgc;
    return TRUE;
  }
  return TRUE;
}

/* Get a group code and value pair. ASCII style.
 * Returns FALSE on fail, else positive.
 * Silly but nescessary use of fseek(). May break char streams on some
 * platforms... should maybe use fgetpos() and fsetpos(). If 'code' and
 * 'value' are NULL when this func is called, the rewind is enabled, and no
 * read is done. (return value is pointless)
 */
int dxf_getapair(FILE *fp, int *code, char *value)
{
  char buf[DXF_LINE_MAX];
  static long fpos=0;

  /* Kludgy, but the spec sucks, what can I say?
   * Some entities have an end marker, some don't.
   * for the ones that don't, a rewind func, built in to the read func!
   * (DOH!)
   * If some func just read a pair, realised that they weren't more
   * entity data but rather the begining of a new entity, and thus
   * the end of this entity, they can use this to put the pair back on
   * the "stack" for the calling func to parse.
   */
  if ( (code==NULL) && (value==NULL) )
  {
    fseek(fp, fpos, SEEK_SET);
    return(TRUE);
  }
  /* Okay, no rewind... on with the show */

  /* Set our current pos. before nuking it with a read */
  fpos = ftell(fp);

  if (!dxf_getaline(fp, buf))
    return FALSE;

  *code = atoi(buf);

  if (!dxf_getaline(fp, buf))
    return FALSE;

  strncpy(value, buf, DXF_LINE_MAX);

  return TRUE;
}

/* Parse a section
 */
int dxf_parse_section(FILE *fp,DXF_Data *dat)
{
  int code;
  char value[DXF_LINE_MAX];
  int err;
  if (!dxf_getpair(fp, &code, value, dat))
    return FALSE;
  if (code != DXF_CODE_NAME)
    return FALSE;

  if (strstr(value, "ENTITIES"))
  {
    err = dxf_parse_entity(fp, dat);
    if (!err)
      return FALSE;
    else
      return TRUE;
  }

#ifdef DXF_DEBUG
  cr_strip(value);
  fprintf(stderr, "%s section not yet implemented\n", value);
#endif
  /* For now, seek EOS */
  while ((err = dxf_getpair(fp, &code, value, dat)))
  {
    dat->lineno++;
    if (code != DXF_CODE_ENTITY)
      continue;
    if (!strstr(value, "ENDSEC"))
      continue;
    break;
  }
  if (err)
    return TRUE;
  else
    return FALSE;
}

/* This tries to determine if a DXF file is ASCII or binary.
 * We don't deal w/ binary right now, but we will.
 * this func. uses rewind()... if this breaks anything for anyone, please
 * let me know.
 */
void dxf_determine_type(FILE *fp, DXF_Data *dat)
{
  char head[22];
  int err;

  err = (int)fgets(head, 22, fp);

  if (err == FALSE)
  {
    dat->type = DXF_ERROR;
    return;
  }

  if (!strncmp("AutoCAD Binary DXF\n\r", head, 20))
    dat->type = DXF_BINARY;
  else
    dat->type = DXF_ASCII;

  rewind(fp);
}

/* This reads a NULL terminated string from fp */
int dxf_read_bstring(FILE *fp, char *string)
{
  char c;

  c = fgetc(fp);

  while ((c != EOF)&&(c != 0))
    *string++ = c;

  if (c == EOF)
    return FALSE;

  *string=0;

  return TRUE;
}

