/* dxf.h
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2001 DindinX <David@dindinx.org>
 * Copyright (C) 1999-2001 Noah Davis of VIEWS Net, Inc. <noah@viewsnet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * This file is from Noah Davis.
 */

#ifndef __DXF_H__
#define __DXF_H__

/* Some defines
 */
#define DXF_LINE_MAX 256
#define DXF_MAX_POINTS 128

#define DXF_ASCII 1
#define DXF_BINARY 2
#define DXF_ERROR -1

#ifndef NULL
#define NULL 0
#endif

#ifndef EOF
#define EOF 0
#endif

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

/* A couple of kludge datatypes for standalone (which doesn't compile
 * exporter)
 */

#ifdef DXF_STANDALONE
typedef int FrameStruct;
typedef int ObjectStruct;
#else
#include "object.h"
#endif


/* A simple three piece vector type */
typedef double DXF_VEC[3];

/*
 * These are the group codes as defined in the DXF standard.
 */

#define DXF_CODE_ENTITY 0
#define DXF_CODE_EVALUE 1
#define DXF_CODE_NAME 2
#define DXF_CODE_LAYER 8


/* Structs for individual entities.
 */

typedef struct DXFVERTEX
{
  char *layer;
  int flags;
  double x, y, z; /* Position */
  int vert_index1, vert_index2, vert_index3, vert_index4; /* Mesh vertex index */
  struct DXFVERTEX *next;
} DXF_Vertex;


typedef struct DXFLINE
{
  char *layer;
  double x1, y1, z1;       /* Start point of line */
  double x2, y2, z2;       /* End point of line */
  double thickness;        /* Thickness ignored right now */
  double extx, exty, extz; /* Extrusion */
  struct DXFLINE *next;
} DXF_Line;

typedef struct DXFPOLYLINE
{
  char *layer;
  double elevation;
  int mcount, ncount;       /* Mesh N and M vertex counts */
  int mdensity, ndensity;   /* Smooth surface M and N density */
  double extx, exty, extz;  /* Extrusion vector */
  double thickness;
  int flags;
  DXF_Vertex *verts;
   
  struct DXFPOLYLINE *next; /* Linked List pointer */
} DXF_Polyline;

/* Struct for a 3DFACE */
typedef struct DXF3DFACE
{
  char *layer;
  double x1,y1,z1; /* First point */
  double x2,y2,z2; /* Second point */
  double x3,y3,z3; /* Third point */
  double x4,y4,z4; /* Fourth point, if same as third, object is triangle */
  struct DXF3DFACE *next;
} DXF_3DFace;

/* Struct for a SOLID */
typedef struct DXFSOLID
{
  char *layer;
  double x1,y1,z1;       /* First point */
  double x2,y2,z2;       /* Second point */
  double x3,y3,z3;       /* Third point */
  double x4,y4,z4;       /* Fourth point, if same as third, object is triangle */
  double extx,exty,extz; /* Extrusion vector */
  double thickness;
  struct DXFSOLID *next;
} DXF_Solid;

/* Struct for an ARC */
typedef struct DXFARC
{
  char *layer;
  double x,y,z;           /* Center */
  double radius;
  double start_angle;
  double end_angle;
  double extx,exty,extz; /* Extrusion vector */
  struct DXFARC *next;
} DXF_Arc;

/* Struct for a CIRCLE */
typedef struct DXFCIRCLE
{
  char *layer;
  double x,y,z; /* Center */
  double radius;
  double extx,exty,extz;
  struct DXFCIRCLE *next;
} DXF_Circle;

/* Struct for a POINT */
typedef struct DXFPOINT
{
  char *layer;
  double x,y,z;
  double thickness;
  double extx, exty, extz; /* Extrusions */
  struct DXFPOINT *next;
} DXF_Point;


/* A simple linked list of strings for layer names
 */
typedef struct DXFLAYER
{
  char name[DXF_LINE_MAX];
  struct DXFLAYER *next;
} DXF_Layer;

/* A structure containing vital data that is passed to most funcs.
 * Can be expanded in the future to include transform data etc...
 */

typedef struct DXFDATA
{
  int lineno;
  int type;
  DXF_Layer *layers;
  int layer_count;
  int entities;
  int pline_count;
  int vert_count;
  int line_count;
  int threedface_count;
  int arc_count;
  int circle_count;
  int solid_count;
  int point_count;
  double bbox_xmax, bbox_ymax, bbox_zmax; /* Bounding-box max */
  double bbox_xmin, bbox_ymin, bbox_zmin; /* Bounding-box min */
  DXF_Polyline *plines; /* A linked list of polylines */
  DXF_Vertex *verts; /* A linked list of solitary vertices */
  DXF_Line *lines; /* A linked list of solitary lines */
  DXF_3DFace *threedfaces; /* A linked list of 3d-faces */
  DXF_Arc *arcs;
  DXF_Circle *circles;
  DXF_Solid *solids;
  DXF_Point *points;
} DXF_Data;

/* Protos.....
 */

/*       dxf.c                 */
DXF_Data *dxf_parse(char *fname);
void dxf_err(char *string, DXF_Data *dat);
void cr_strip(char *string);
void dxf_dobox(double x, double y, double z, DXF_Data *dat);
int dxf_add_polyline(DXF_Polyline *pline, DXF_Data *dat);
int dxf_add_vertex(DXF_Vertex *vert, DXF_Data *dat);
int dxf_add_line(DXF_Line *line, DXF_Data *dat);
int dxf_add_3dface(DXF_3DFace *face, DXF_Data *dat);
int dxf_add_arc(DXF_Arc *arc, DXF_Data *dat);
int dxf_add_circle(DXF_Circle *circle, DXF_Data *dat);
int dxf_add_solid(DXF_Solid *solid, DXF_Data *dat);
int dxf_add_point(DXF_Point *point, DXF_Data *dat);
int dxf_add_vertex_to_polyline(DXF_Vertex *vertex, DXF_Polyline *pline);
char *dxf_add_layer(char *name, DXF_Data *dat);
void dxf_free_all(DXF_Data *dat);

#ifndef DXF_STANDALONE
void LoadDXF(GtkWidget *w, GtkFileSelection *FS);
void InsertDXF(GtkWidget *w, GtkFileSelection *FS);
#endif

/*       dxf_read.c            */
FILE *dxf_open(char *fname);
int dxf_getline(FILE *fp,char *line, DXF_Data *dat);
int dxf_getaline(FILE *fp, char *line);
int dxf_close(FILE *fp);
int dxf_error(FILE *fp);
int dxf_getapair(FILE *fp, int *code, char *value);
int dxf_getpair(FILE *fp, int *code, char *value, DXF_Data *dat);
int dxf_parse_section(FILE *fp, DXF_Data *dat);
void dxf_determine_type(FILE *fp, DXF_Data *dat);
int dxf_read_bstring(FILE *fp, char *string);
int dxf_getbpair(FILE *fp, int *code, char *value);

/*       dxf_entity.c          */
int dxf_parse_entity(FILE *fp, DXF_Data *dat);
DXF_Polyline *dxf_parse_polyline(FILE *fp, DXF_Data *dat);
DXF_Line *dxf_parse_line(FILE *fp, DXF_Data *dat);
DXF_Vertex *dxf_parse_vertex(FILE *fp, DXF_Data *dat);
DXF_3DFace *dxf_parse_3dface(FILE *fp, DXF_Data *dat);
DXF_Arc *dxf_parse_arc(FILE *fp, DXF_Data *dat);   
DXF_Circle *dxf_parse_circle(FILE *fp, DXF_Data *dat);
DXF_Solid *dxf_parse_solid(FILE *fp, DXF_Data *dat);
DXF_Point *dxf_parse_point(FILE *fp, DXF_Data *dat);

/*      dxf_write.c           */
int dxf_save_file(FrameStruct *frame, char *fname, int binary, int SelectedOnly, char *Comment);
int dxf_write_header(FILE *fp, int bin, char *Comments);
int dxf_write_object(FILE *fp, ObjectStruct *obj);
int dxf_write_end(FILE *fp, int bin);

#ifndef DXF_STANDALONE
void SaveDXF(GtkWidget *w, GtkFileSelection *FS);
void SaveSelectedDXF(GtkWidget *w, GtkFileSelection *FS);
#endif

#endif /* __DXF_H__ */
