/* protocol.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999 David Odin <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include "../src/giram.h"
#include "protocol.h"
#include "giramlib.h"

int GiramToPlugIn;
int PlugInToGiram;
int FrameId;

/*****************************************************************************
*  SendString
******************************************************************************/
static void SendRawString(int pipe, const char *Text)
{
  guint size;

  if (Text)
  {
    size = strlen(Text);
    write(pipe, &size, sizeof(int));
    if (size > 0)
      write(pipe, Text, size);
  }
}

/*****************************************************************************
*  SendRawInt
******************************************************************************/
static void SendRawInt(int pipe, int Int)
{
  write(pipe, &Int, sizeof(int));
}

/*****************************************************************************
*  SendRawDouble
******************************************************************************/
static void SendRawDouble(int pipe, double Double)
{
  write(pipe, &Double, sizeof(double));
}

/*****************************************************************************
*  SendString
******************************************************************************/
static void SendString(int pipe, const char *String)
{
  int type;

  type = STRING_TYPE;
  write(pipe, &type, sizeof(int));
  SendRawString(pipe, String);
}

/*****************************************************************************
*  SendFunc
******************************************************************************/
static void SendFunc(int pipe, char *Func)
{
  int type;

  type = FUNC_TYPE;
  write(pipe, &type, sizeof(int));
  SendRawString(pipe, Func);
}

/*****************************************************************************
*  SendFrameId
******************************************************************************/
static void SendFrameId(int pipe, int Id)
{
  int type;

  type = FRAMEID_TYPE;
  write(pipe, &type, sizeof(int));
  SendRawInt(pipe, Id);
}

/*****************************************************************************
*  SendObjectId
******************************************************************************/
static void SendObjectId(int pipe, int Id)
{
  int type;

  type = OBJECTID_TYPE;
  write(pipe, &type, sizeof(int));
  SendRawInt(pipe, Id);
}

/*****************************************************************************
*  SendDouble
******************************************************************************/
static void SendDouble(int pipe, double Double)
{
  int type;

  type = DOUBLE_TYPE;
  write(pipe, &type, sizeof(int));
  SendRawDouble(pipe, Double);
}

/*****************************************************************************
*  SendVector3
******************************************************************************/
static void SendVector3(int pipe, Vector Vect)
{
  int type;

  type = VECTOR3_TYPE;
  write(pipe, &type, sizeof(int));
  SendRawDouble(pipe, Vect[0]);
  SendRawDouble(pipe, Vect[1]);
  SendRawDouble(pipe, Vect[2]);
}

/*****************************************************************************
*  SendTransform
******************************************************************************/
static void SendTransform(int pipe, TransformStruct *Trans)
{
  int type, i, j;

  type = TRANSFORM_TYPE;
  write(pipe, &type, sizeof(int));
  for (i=0 ; i<4 ; i++)
    for (j=0 ; j<4 ; j++)
      SendRawDouble(pipe, Trans->Direct[i][j]);
  for (i=0 ; i<4 ; i++)
    for (j=0 ; j<4 ; j++)
      SendRawDouble(pipe, Trans->Inverse[i][j]);
}

/*****************************************************************************
*  SendEndPlugins
******************************************************************************/
static void SendEndPlugins(int pipe)
{
  int type;

  type = ENDPLUGINS_TYPE;
  write(pipe, &type, sizeof(int));
}

/*****************************************************************************
*  DeadPipe
*  what to do if the pipe breaks.
******************************************************************************/
static void DeadPipe(int Dum)
{
  exit(0);
}

/*****************************************************************************
*  giram_init
******************************************************************************/
void giram_init(int argc, char *argv[], const char *Title)
{
  int done;
  int Type;

  if (argc < 6)
  {
/*    printf("%s must be called by Giram-%s\n", argv[0], VERSION);*/
    exit(1);
  }
  if (strcmp(argv[1], "-giram"))
  {
/*    printf("%s must be called by Giram-%s\n", argv[0], VERSION);*/
    exit(1);
  }
  /* A dead Pipe means Giram has died or want to kill us */
  signal(SIGPIPE, DeadPipe);

  PlugInToGiram = atoi(argv[3]);
  GiramToPlugIn = atoi(argv[4]);
  FrameId = atoi(argv[5]);
  SendRawString(PlugInToGiram, "ACK!");
  
  if (!strcmp(argv[2], "-init"))
  {
    SendString(PlugInToGiram, Title);
    SendEndPlugins(PlugInToGiram);
    done = FALSE;
    while (!done)
    {
      read(GiramToPlugIn, &Type, sizeof(int));
      if (Type == ENDPLUGINS_TYPE)
        done = TRUE;
    }
    exit(0);
  }
}

/*****************************************************************************
*  giram_exit
******************************************************************************/
void giram_exit(void)
{
  int done, Type;

  SendEndPlugins(PlugInToGiram);
  /* We now wait to be killed */
  done = FALSE;
  while (!done)
  {
    read(GiramToPlugIn, &Type, sizeof(int));
    if (Type == ENDPLUGINS_TYPE)
      done = TRUE;
  }
  close(PlugInToGiram);
  close(GiramToPlugIn);
  exit(0);
}

/*****************************************************************************
*  giram_send_trans
******************************************************************************/
int giram_send_trans(int ObjectID, TransformStruct *Trans)
{
  int a;

  SendFunc(PlugInToGiram, "TransformObject");
  SendFrameId(PlugInToGiram, FrameId);
  SendObjectId(PlugInToGiram, ObjectID);
  SendTransform(PlugInToGiram, Trans);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_box
******************************************************************************/
int giram_send_box(Vector Vect1, Vector Vect2)
{
  int a;

  SendFunc(PlugInToGiram, "CreateBox");
  SendFrameId(PlugInToGiram, FrameId);
  SendVector3(PlugInToGiram, Vect1);
  SendVector3(PlugInToGiram, Vect2);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_cone
******************************************************************************/
int giram_send_cone(Vector Base, double BaseRadius,
                    Vector Apex, double ApexRadius)
{
  int a;

  SendFunc(PlugInToGiram, "CreateCone");
  SendFrameId(PlugInToGiram, FrameId);
  SendVector3(PlugInToGiram, Base);
  SendDouble(PlugInToGiram, BaseRadius);
  SendVector3(PlugInToGiram, Apex);
  SendDouble(PlugInToGiram, ApexRadius);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_cylinder
******************************************************************************/
int giram_send_cylinder(Vector Base, Vector Apex, double Radius)
{
  int a;

  SendFunc(PlugInToGiram, "CreateCylinder");
  SendFrameId(PlugInToGiram, FrameId);
  SendVector3(PlugInToGiram, Base);
  SendVector3(PlugInToGiram, Apex);
  SendDouble(PlugInToGiram, Radius);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_disc
******************************************************************************/
int giram_send_disc(Vector Center, Vector Normal, double Radius, double HoleRadius)
{
  int a;

  SendFunc(PlugInToGiram, "CreateDisc");
  SendFrameId(PlugInToGiram, FrameId);
  SendVector3(PlugInToGiram, Center);
  SendVector3(PlugInToGiram, Normal);
  SendDouble(PlugInToGiram, Radius);
  SendDouble(PlugInToGiram, HoleRadius);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_plane
******************************************************************************/
int giram_send_plane(Vector Normal, double Distance)
{
  int a;

  SendFunc(PlugInToGiram, "CreatePlane");
  SendFrameId(PlugInToGiram, FrameId);
  SendVector3(PlugInToGiram, Normal);
  SendDouble(PlugInToGiram, Distance);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_sphere
******************************************************************************/
int giram_send_sphere(Vector Center, double Radius)
{
  int a;

  SendFunc(PlugInToGiram, "CreateSphere");
  SendFrameId(PlugInToGiram, FrameId);
  SendVector3(PlugInToGiram, Center);
  SendDouble(PlugInToGiram, Radius);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_superellipsoid
******************************************************************************/
int giram_send_superellipsoid(double E, double N)
{
  int a;

  SendFunc(PlugInToGiram, "CreateSuperEllipsoid");
  SendFrameId(PlugInToGiram, FrameId);
  SendDouble(PlugInToGiram, E);
  SendDouble(PlugInToGiram, N);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_torus
******************************************************************************/
int giram_send_torus(double MajorRadius, double MinorRadius)
{
  int a;

  SendFunc(PlugInToGiram, "CreateTorus");
  SendFrameId(PlugInToGiram, FrameId);
  SendDouble(PlugInToGiram, MajorRadius);
  SendDouble(PlugInToGiram, MinorRadius);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

/*****************************************************************************
*  giram_send_triangle
******************************************************************************/
int giram_send_triangle(Vector V1, Vector V2, Vector V3)
{
  int a;

  SendFunc(PlugInToGiram, "CreateTriangle");
  SendFrameId(PlugInToGiram, FrameId);
  SendVector3(PlugInToGiram, V1);
  SendVector3(PlugInToGiram, V2);
  SendVector3(PlugInToGiram, V3);
  read(GiramToPlugIn, &a, sizeof(int));
  return a;
}

