/* giramobject.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999 David Odin <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "vectors.h"
#include "giramlibprivate.h"
#include "protocol.h"
#include "giramlib.h"

/*****************************************************************************
*  giramlib_box_create
******************************************************************************/
int giramlib_box_create(Vector Vect1, Vector Vect2)
{
  GiramObject *Box;
  GiramBox    *BBox;

  BBox = g_new(GiramBox, 1);
  Box = (GiramObject *)BBox;
  Box->Next = GiramFirstObject;
  GiramFirstObject = Box;
  Box->Type = BoxType;
  Box->Realised = FALSE;
  Box->Id = NextObjectId++;
  Box->Trans = NULL;

  V3Dcopy(BBox->MinCorner, Vect1);
  V3Dcopy(BBox->MaxCorner, Vect2);
  return Box->Id;
}

/*****************************************************************************
*  giram_cone_create
******************************************************************************/
int giramlib_cone_create(Vector Base, double BaseRadius, 
                         Vector Apex, double ApexRadius)
{
  GiramObject *Cone;
  GiramCone   *CCone;

  CCone = g_new(GiramCone, 1);
  Cone = (GiramObject *)CCone;
  Cone->Next = GiramFirstObject;
  GiramFirstObject = Cone;
  Cone->Type = ConeType;
  Cone->Realised = FALSE;
  Cone->Id = NextObjectId++;
  Cone->Trans = NULL;

  V3Dcopy(CCone->Base, Base);
  CCone->BaseRadius = BaseRadius;
  V3Dcopy(CCone->Apex, Apex);
  CCone->ApexRadius = ApexRadius;
  return Cone->Id;
}

/*****************************************************************************
*  giram_cylinder_create
******************************************************************************/
int giramlib_cylinder_create(Vector Base, Vector Apex, double Radius)
{
  GiramObject   *Cylinder;
  GiramCylinder *CCylinder;

  CCylinder = g_new(GiramCylinder, 1);
  Cylinder = (GiramObject *)CCylinder;
  Cylinder->Next = GiramFirstObject;
  GiramFirstObject = Cylinder;
  Cylinder->Type = ConeType;
  Cylinder->Realised = TRUE;
  Cylinder->Id = NextObjectId++;
  Cylinder->Trans = NULL;

  V3Dcopy(CCylinder->Base, Base);
  V3Dcopy(CCylinder->Apex, Apex);
  CCylinder->Radius = Radius;
  return Cylinder->Id;
}

/*****************************************************************************
*  giram_sphere_create
******************************************************************************/
int giramlib_sphere_create(Vector Center, double Radius)
{
  GiramObject *Sphere;
  GiramSphere *SSphere;

  SSphere = g_new(GiramSphere, 1);
  Sphere = (GiramObject *)SSphere;
  Sphere->Next = GiramFirstObject;
  GiramFirstObject = Sphere;
  Sphere->Type = SphereType;
  Sphere->Realised = FALSE;
  Sphere->Id = NextObjectId++;
  Sphere->Trans = NULL;

  V3Dcopy(SSphere->Center, Center);
  SSphere->Radius = Radius;
  return Sphere->Id;
}

/*****************************************************************************
*  giram_triangle_create
******************************************************************************/
int giramlib_triangle_create(Vector V1, Vector V2, Vector V3)
{
  GiramObject   *Triangle;
  GiramTriangle *TTriangle;

  TTriangle = g_new(GiramTriangle, 1);
  Triangle = (GiramObject *)TTriangle;
  Triangle->Next = GiramFirstObject;
  GiramFirstObject = Triangle;
  Triangle->Type = TriangleType;
  Triangle->Realised = FALSE;
  Triangle->Id = NextObjectId++;
  Triangle->Trans = NULL;

  V3Dcopy(TTriangle->P1, V1);
  V3Dcopy(TTriangle->P2, V2);
  V3Dcopy(TTriangle->P3, V3);
  return Triangle->Id;
}

/*****************************************************************************
*  giram_object_realize
******************************************************************************/
void giramlib_object_realize(int ObjectId)
{
  GiramObject *TmpObject;
  int ObjectID = -1;

  for (TmpObject = GiramFirstObject;
       TmpObject && TmpObject->Id != ObjectId;
       TmpObject = TmpObject->Next)
    ;
  if (TmpObject == NULL || TmpObject->Realised)
    return;
  switch (TmpObject->Type)
  {
    case BicubicPatchType:

      break;
    case BlobType:

      break;
    case BoxType:
      {
        GiramBox *BBox = (GiramBox *)TmpObject;

        ObjectID = giram_send_box(BBox->MinCorner, BBox->MaxCorner);
      }
      break;
    case ConeType:
      {
        GiramCone *CCone = (GiramCone *)TmpObject;

        ObjectID = giram_send_cone(CCone->Base, CCone->BaseRadius,
                                   CCone->Apex, CCone->ApexRadius);
      }
      break;
    case CSGType:

      break;
    case CubicType:

      break;
    case CylinderType:
      {
        GiramCylinder *CCylinder = (GiramCylinder *)TmpObject;

        ObjectID = giram_send_cylinder(CCylinder->Base, CCylinder->Apex,
                                       CCylinder->Radius);
      }
      break;
    case DiscType:
      {
        GiramDisc *DDisc = (GiramDisc *)TmpObject;

        ObjectID = giram_send_disc(DDisc->Center, DDisc->Normal,
                                   DDisc->Radius, DDisc->HoleRadius);
      }
      break;
    case HeightFieldType:

      break;
    case JuliaFractalType:

      break;
    case LatheType:

      break;
    case MeshType:

      break;
    case PlaneType:
      {
        GiramPlane *PPlane = (GiramPlane *)TmpObject;

        ObjectID = giram_send_plane(PPlane->Normal, PPlane->Distance);
      }
      break;
    case PolyType:

      break;
    case PolygonType:

      break;
    case PrismType:

      break;
    case QuadricType:

      break;
    case QuarticType:

      break;
    case SorType:

      break;
    case SphereType:
      {
        GiramSphere *SSphere = (GiramSphere *)TmpObject;

        ObjectID = giram_send_sphere(SSphere->Center, SSphere->Radius);
      }
      break;
    case SuperEllipsoidType:
      {
        GiramSuperEllipsoid *SSuperEllipsoid = (GiramSuperEllipsoid *)TmpObject;

        ObjectID = giram_send_superellipsoid(SSuperEllipsoid->E, SSuperEllipsoid->N);
      }
      break;
    case TextType:

      break;
    case TorusType:
      {
        GiramTorus *TTorus = (GiramTorus *)TmpObject;

        ObjectID = giram_send_torus(TTorus->Major, TTorus->Minor);
      }
      break;
    case TriangleType:
      {
        GiramTriangle *TTriangle = (GiramTriangle *)TmpObject;

        ObjectID = giram_send_triangle(TTriangle->P1, TTriangle->P2, TTriangle->P3);
      }
      break;
  }
  if (TmpObject->Trans)
  {
    giram_send_trans(ObjectID, TmpObject->Trans);
  }
  TmpObject->Realised = TRUE;
}

/*****************************************************************************
*  giram_object_realize_all
******************************************************************************/
void giramlib_object_realize_all(void)
{
  GiramObject *TmpObject;
  int ObjectID = -1;

  for (TmpObject = GiramFirstObject; TmpObject ; TmpObject = TmpObject->Next)
  {
    if (!(TmpObject->Realised))
    {
      switch (TmpObject->Type)
      {
        case BicubicPatchType:

          break;
        case BlobType:

          break;
        case BoxType:
          {
            GiramBox *BBox = (GiramBox *)TmpObject;

            ObjectID = giram_send_box(BBox->MinCorner, BBox->MaxCorner);
          }
          break;
        case ConeType:
          {
            GiramCone *CCone = (GiramCone *)TmpObject;

            ObjectID = giram_send_cone(CCone->Base, CCone->BaseRadius,
                                       CCone->Apex, CCone->ApexRadius);
          }
          break;
        case CSGType:

          break;
        case CubicType:

          break;
        case CylinderType:
          {
            GiramCylinder *CCylinder = (GiramCylinder *)TmpObject;

            ObjectID = giram_send_cylinder(CCylinder->Base, CCylinder->Apex,
                                           CCylinder->Radius);
          }
          break;
        case DiscType:
          {
            GiramDisc *DDisc = (GiramDisc *)TmpObject;

            ObjectID = giram_send_disc(DDisc->Center, DDisc->Normal,
                                       DDisc->Radius, DDisc->HoleRadius);
          }
          break;
        case HeightFieldType:

          break;
        case JuliaFractalType:

          break;
        case LatheType:

          break;
        case MeshType:

          break;
        case PlaneType:
          {
            GiramPlane *PPlane = (GiramPlane *)TmpObject;

            ObjectID = giram_send_plane(PPlane->Normal, PPlane->Distance);
          }
          break;
        case PolyType:

          break;
        case PolygonType:

          break;
        case PrismType:

          break;
        case QuadricType:

          break;
        case QuarticType:

          break;
        case SorType:

          break;
        case SphereType:
          {
            GiramSphere *SSphere = (GiramSphere *)TmpObject;

            ObjectID = giram_send_sphere(SSphere->Center, SSphere->Radius);
          }
          break;
        case SuperEllipsoidType:
          {
            GiramSuperEllipsoid *SSuperEllipsoid = (GiramSuperEllipsoid *)TmpObject;

            ObjectID = giram_send_superellipsoid(SSuperEllipsoid->E,
                                                 SSuperEllipsoid->N);
          }
          break;
        case TextType:

          break;
        case TorusType:
          {
            GiramTorus *TTorus = (GiramTorus *)TmpObject;

            ObjectID = giram_send_torus(TTorus->Major, TTorus->Minor);
          }
          break;
        case TriangleType:
          {
            GiramTriangle *TTriangle = (GiramTriangle *)TmpObject;

            ObjectID = giram_send_triangle(TTriangle->P1, TTriangle->P2,
                                           TTriangle->P3);
          }
          break;
      }
      if (TmpObject->Trans)
      {
        giram_send_trans(ObjectID, TmpObject->Trans);
      }
      TmpObject->Realised = TRUE;
    }
  }
}

/*****************************************************************************
*  giram_object_translate
******************************************************************************/
int giramlib_object_translate(int objectID, Vector TransVect)
{
  GiramObject *TmpObject;
  TransformStruct Trans;

  for (TmpObject = GiramFirstObject;
       TmpObject && TmpObject->Id != objectID ;
       TmpObject = TmpObject->Next)
    ;
  if (TmpObject)
  {
    if (TmpObject->Trans)
    {
      ComputeTranslateTrans(&Trans, TransVect);
      ComposeTrans(TmpObject->Trans, &Trans);
    } else
    {
      TmpObject->Trans = g_new(TransformStruct, 1);
      ComputeTranslateTrans(TmpObject->Trans, TransVect);
    }
    return objectID;
  }
  return -1;
}

/*****************************************************************************
*  giram_object_rotate
******************************************************************************/
int giramlib_object_rotate(int objectID, Vector RotateVect)
{
  GiramObject *TmpObject;
  TransformStruct Trans;

  for (TmpObject = GiramFirstObject;
       TmpObject && TmpObject->Id != objectID ;
       TmpObject = TmpObject->Next)
    ;
  if (TmpObject)
  {
    if (TmpObject->Trans)
    {
      ComputeRotateTrans(&Trans, RotateVect);
      ComposeTrans(TmpObject->Trans, &Trans);
    } else
    {
      TmpObject->Trans = g_new(TransformStruct, 1);
      ComputeRotateTrans(TmpObject->Trans, RotateVect);
    }
    return objectID;
  }
  return -1;
}

/*****************************************************************************
*  giram_object_scale
******************************************************************************/
int giramlib_object_scale(int objectID, Vector ScaleVect)
{
  GiramObject *TmpObject;
  TransformStruct Trans;

  for (TmpObject = GiramFirstObject;
       TmpObject && TmpObject->Id != objectID ;
       TmpObject = TmpObject->Next)
    ;
  if (TmpObject)
  {
    if (TmpObject->Trans)
    {
      ComputeScaleTrans(&Trans, ScaleVect);
      ComposeTrans(TmpObject->Trans, &Trans);
    } else
    {
      TmpObject->Trans = g_new(TransformStruct, 1);
      ComputeScaleTrans(TmpObject->Trans, ScaleVect);
    }
    return objectID;
  }
  return -1;
}

