#!/usr/bin/env python

"""
synopsis:
  "Post" process an XML file (possibly generated by a gDS module).

usage:
  post_process.py [-h] [-f] [-v] oldfile newfile

positional arguments:
  oldfile        The input file
  newfile        The output file

optional arguments:
  -h, --help     show this help message and exit
  -f, --force    Force overwrite of outfile if it exists
  -v, --verbose  Print messages during actions.

examples:
  python post_process.py oldfilename.xml newfilename.xml
  python post_process.py --force oldfilename.xml newfilename.xml

notes:

  you can provide custom behavior by modifying the `process`
  function.  Or, if there is an import-able module named
  `postprocesscallback.py` containing a function named `callback`,
  it will be called instead of `process`.

"""


import sys
import os
import argparse
from lxml import etree

try:
    from postprocesscallback import callback
except ModuleNotFoundError:
    callback = None


def dbg_msg(opts, msg):
    """Print a message if verbose is on."""
    if opts.verbose:
        print(msg)


#
# An example post-processing function.
# Replace this function with your own.
def process(root):
    for elem in root.iter():
        elem.tag = etree.QName(elem).localname
    etree.cleanup_namespaces(root)


def makefile(outfilename, opts):
    """Create output file.  Error if exists and not opts.force."""
    if (not opts.force) and os.path.exists(outfilename):
        sys.exit(f'outfile "{outfilename}" exists.  Use -f to overwrite.')
    outfile = open(outfilename, 'w')
    return outfile


def run(opts):
    doc = etree.parse(opts.oldfile)
    root = doc.getroot()
    if callback is not None:
        callback(root)
    else:
        process(root)
    outfile = makefile(opts.newfile, opts)
    doc.write(outfile.buffer)
    outfile.close()


def main():
    description = """\
synopsis:
  "Post" process an XML file (possibly generated by a gDS module).
"""
    epilog = """\

examples:
  python post_process.py oldfile newfile
  python post_process.py --force oldfile newfile
"""
    parser = argparse.ArgumentParser(
        description=description,
        epilog=epilog,
        formatter_class=argparse.RawDescriptionHelpFormatter,
    )
    parser.add_argument(
        "oldfile",
        help="The input file"
    )
    parser.add_argument(
        "newfile",
        help="The output file"
    )
    parser.add_argument(
        "-f", "--force",
        action="store_true",
        help="Force overwrite of outfile if it exists",
    )
    parser.add_argument(
        "-v", "--verbose",
        action="store_true",
        help="Print messages during actions.",
    )
    options = parser.parse_args()
    run(options)


if __name__ == '__main__':
    #import pdb; pdb.set_trace()
    #import ipdb; ipdb.set_trace()
    main()
