/*
** Copyright (C) 17 Jun 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#ifdef __linux__
#include <linux/version.h>
#define KERNEL_VERS(major,minor,sublvl) (((major)<<16)+((minor)<<8)+(sublvl))
/* I don't know exactly when cdrom got the ioctls but sometime in 2.1.x */
#if LINUX_VERSION_CODE >= KERNEL_VERS(2,1,0)
#define IS_RECENT_LINUX 1
#endif
#endif

#ifdef IS_RECENT_LINUX
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#endif

#include "multisession.h"
#include "common_gtk.h"
#include "preferences.h"
#include "mainwindow.h"
#include "globals.h"

GtkWidget *enable_multisession, *force_data_multisession, *old_session, *enable_cdplus;

static void check_multisession_button_oldsession(GtkWidget *widget, gpointer *data) {
	if (0 != strlen(gtk_entry_get_text(GTK_ENTRY((old_session)))))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_multisession), TRUE);
}
static void check_multisession_button_data(GtkWidget *widget, gpointer *data) {
	gtk_widget_set_sensitive(force_data_multisession,
			GTK_TOGGLE_BUTTON(enable_multisession)->active);
	gtk_widget_set_sensitive(enable_cdplus,
			GTK_TOGGLE_BUTTON(enable_multisession)->active);
}

void multisession_options(GtkWidget *vbox) {
	GtkWidget *frame, *hbox, *label;

	frame = make_frame(_("Multi Session"), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);
	gtk_container_add(GTK_CONTAINER(frame), hbox);

	enable_multisession = gtk_check_button_new_with_label(_("multi session"));
	gtk_widget_show(enable_multisession);
	gtk_box_pack_start(GTK_BOX(hbox), enable_multisession, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, enable_multisession, _("Make multi session image "
				"(appends new image to old image) - first additional session "
				"will waste about 22 MB, subsequent sessions about 13 MB in overhead. "
				"NOTE: currently directories in the old sessions gets completely "
				"overwritten when a new dir with the same name in a new session is "
				"added (ie the files inside it not in the new session \"disappears\") "
				"(some OSes allow you to mount individual sessions).\n"
				" * * * NOTE * * *\n"
				"It seems like multisession doesn't work in gcombust at the moment!!!\n"),
			NULL);

	force_data_multisession = gtk_check_button_new_with_label(_("Use data"));
	gtk_widget_show(force_data_multisession);
	gtk_box_pack_start(GTK_BOX(hbox), force_data_multisession, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, force_data_multisession, _("Some drives have "
				"no hardware support for multi session type tracks, and this "
				"option needs to be enabled on those (this includes some "
				"Sony drives). Currently you also have to enable this option "
				"for making multi session disks in DAO/SAO mode."),
			NULL);
	gtk_widget_set_sensitive(force_data_multisession, FALSE);

	enable_cdplus = gtk_check_button_new_with_label(_("CDplus"));
	gtk_widget_show(enable_cdplus);
	gtk_box_pack_start(GTK_BOX(hbox), enable_cdplus, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, enable_cdplus,  _("Make an image for CDplus. "
				"The first session of CDplus consist of audio tracks and "
				"the second session is a data track. When making a "
				"multi session CD, mkisofs usually reads the directory "
				"structure from the previous session. This option "
				"disables that (as audio tracks have no directory "
				"structure)."), NULL); 
	gtk_widget_set_sensitive(enable_cdplus, FALSE);

	gtk_signal_connect(GTK_OBJECT(enable_multisession),
			"clicked", (GtkSignalFunc) check_multisession_button_data, NULL);

	old_session = gtk_entry_new();
	gtk_widget_show(old_session);
	gtk_box_pack_end(GTK_BOX(hbox), old_session, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(old_session),
			"changed", (GtkSignalFunc) check_multisession_button_oldsession, NULL);

#ifdef __FreeBSD__
	gtk_tooltips_set_tip(tooltips, old_session, _("Path to old session image, usually "
				"the drive where the multi session CD is (/dev/rcd0a), but can also "
				"be an existing image on disk. Leave this empty for first session."),
			NULL);
#else
	gtk_tooltips_set_tip(tooltips, old_session, _("Path to old session image, usually "
				"the drive where the multi session CD is (/dev/hdb), but can also "
				"be an existing image on disk. Leave this empty for first session."),
			NULL);
#endif

	label = gtk_label_new(_("Path to old session: "));
	gtk_widget_show(label);
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);
}

#ifdef IS_RECENT_LINUX
/* Platform dependant code used to determine whether the inserted cd is blank or not.
 * If the disk is blank and multisession selected, old_session must be set to "" */

int check_blank_disk(void) {
	if (misc_prefs.do_ms_check && GTK_TOGGLE_BUTTON(enable_multisession)->active) {
		char *old_session_path;

		old_session_path = gtk_entry_get_text(GTK_ENTRY(old_session));

		if ((old_session_path) && (strlen(old_session_path) > 0)) {
			int fd, res;

			if (0 != strncmp("/dev/", old_session_path, 5)) {
				/* old session not on a cdrom */
				return 1;
			}

			if (-1 == (fd = open(old_session_path, O_RDONLY | O_NONBLOCK))) {
				alert_user_of_error(" Error while trying to open the old session. \n"
						" Please check the old session field. \n\n"
						" (This check can be turned off in the preferences) ");
				return 0;
			}

			res = ioctl(fd, CDROM_DISC_STATUS);
			close(fd);

			if (-1 == res) {
				alert_user_of_error(" Ioctl error while trying to recognize the old session, aborting. \n\n"
						" (This check can be turned off in the preferences) ");
				return 0;
			}
			switch (res) {
				case CDS_AUDIO:
					alert_user_of_error(" The CDROM used as old session is an audio cd. \n"
							" You MUST NOT perform multi session operations on audio cds. \n\n"
							" (This check can be turned off in the preferences) ");
					return 0;
					break;
				case CDS_NO_DISC:
					alert_user_of_error(" No CDROM present in the drive for the old session field. \n\n"
							" (This check can be turned off in the preferences) ");
					return 0;
					break;
				case CDS_DATA_1:
				case CDS_DATA_2:
				case CDS_XA_2_1:
				case CDS_XA_2_2:
				case CDS_MIXED:
					/* The inserted CD is a correct data CD */
					break;
				case CDS_NO_INFO:
					alert_user_of_error(" Could not check validity of CDROM used as old session, \n\n"
							" (This check can be turned off in the preferences) ");
					/* fall trough, I get CDS_NO_INFO for empty disks on my drive */
				default:
					alert_user_of_error(" The CDROM to be used as old session is probably blank. \n"
							" If this is the case, you have to leave the old session field empty. \n"
							" (This check can be turned off in the preferences) ");
					return 0;
			}
		}
	}
	return 1;
}

/* This has nothing to do with multisession support, but the ioctl is in this
 * file already... */
int read_will_block_cd(int fd) {
	int res;

	res = ioctl(fd, CDROM_DISC_STATUS);

	switch (res) {
		case -1:
		case CDS_DATA_1:
		case CDS_DATA_2:
		case CDS_XA_2_1:
		case CDS_XA_2_2:
		case CDS_MIXED:
			return 0;
		case CDS_AUDIO:
		case CDS_NO_DISC:
		case CDS_NO_INFO:
		default:
			return 1;
	}
}

#else

int check_blank_disk(void) {
	return 1;
}

int read_will_block_cd(int fd) {
	return 0;
}
#endif

