/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include "multisession.h"
#include "vector_commands.h"
#include "common_gtk.h"
#include "modify_file_set.h"
#include "filepicker.h"
#include "command.h"
#include "linebuffer.h"
#include "optimize_usage.h"
#include "contractions.h"
#include "locks.h"
#include "mainwindow.h"
#include "globals.h"

/* static gint sel_row_path = -1; */
static long int lastsize = 0;
static GList *file_path_list, *dir_path_list;

static void update_totalsize_label(void) {
	char *sizeinfolabel;
	char tmp_buf[SIZE_BUF_SIZE];

	if (sectorsize < 0)
		sectorsize = 150;
	if (totalsize < 0)
		totalsize = 0;

	sizeinfolabel = string_append(_("Sectors: "), NULL);
	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%6.0f", (double)sectorsize) == -1)
		g_warning("%s::%i: sector size too large? Sector-result is probably wrong",
				__FILE__, __LINE__);
	sizeinfolabel = string_append(sizeinfolabel, tmp_buf);
	sizeinfolabel = string_append(sizeinfolabel, _("\nSize: "));
	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%5.1f", (double)totalsize/(1024*1024)) == -1)
		g_warning("%s::%i: file size too large? Total size-result is probably wrong",
				__FILE__, __LINE__);
	sizeinfolabel = string_append(sizeinfolabel, tmp_buf);
	sizeinfolabel = string_append(sizeinfolabel, " MB");

	if (opt_cdsize >= sectorsize) {
		GtkRcStyle *rc_style;
		rc_style = gtk_rc_style_new();
		gtk_widget_modify_style(GTK_WIDGET(total_size), rc_style);
		gtk_rc_style_unref(rc_style);
	} else {
		GtkRcStyle *rc_style;
		GdkColor color;

		gdk_color_parse("red", &color);

		rc_style = gtk_rc_style_new();

		rc_style->fg[GTK_STATE_NORMAL] = color;
		rc_style->fg[GTK_STATE_INSENSITIVE] = color;
		rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_FG;
		rc_style->color_flags[GTK_STATE_INSENSITIVE] |= GTK_RC_FG;

		gtk_widget_modify_style(GTK_WIDGET(total_size), rc_style);

		gtk_rc_style_unref(rc_style);
	}

	gtk_label_set(GTK_LABEL(total_size), sizeinfolabel);
	free(sizeinfolabel);
}

static void error_when_executing(cmd_v *command, const char *error_details, char *output) {
	char *cmd_string, *error_msg;

	cmd_string = cmd_to_string(command);
	error_msg = g_strdup_printf(_("%s\n\n The following command was executed when the error occurred: \n%s"), error_details, cmd_string);

	if (NULL == output)
		alert_user_of_error(error_msg);
	else
		alert_user_of_error_msg(error_msg, output);

	free(cmd_string);
	free(error_msg);
}

static struct {
	const char *mkisofs_error;
	const char *msg_to_user;
} mkisofs_size_errors[] = {
	{"Usage: mkisofs", N_(" mkisofs received arguments it did "
			"not understand, \n maybe you are using "
			"too old a version of mkisofs? ")},
	{"Invalid node -", N_(" mkisofs detected an invalid filename "
			"in the list ")},
	{"volume size must be >= 800K", N_(" HFS volume size must be >= 800K "
			"- file/dir size too small (the size \n"
			" estimate might be a little bit wrong), "
			"add more files before burning ")},
	{"Unable to sort directory ", N_(" mkisofs failed to sort directory, "
			"(maybe a problem with multi session, \n try "
			"enabling the No RR option if multi session "
			"is used). It it also possible \n that there "
			"is a duplicate filename in the list, if so, "
			"please rename it. \n It is also quite "
			"possible that there is a bug in mkisofs. ")},
	{"Joliet tree sort failed.", N_(" Joliet tree sort failed - maybe "
			"the image contains \n a filename-component "
			"longer than 64 characters? ")}

};

static const char *search_for_mkisofs_error(const char *msgbuf) {
	unsigned int i;

	for (i = 0; i < sizeof(mkisofs_size_errors)/sizeof(mkisofs_size_errors[0]); i++) {
		if (strstr(msgbuf, mkisofs_size_errors[i].mkisofs_error))
			return mkisofs_size_errors[i].msg_to_user;
	}
	return NULL;

}

void execute_recalc_size(cmd_v *command) {
	int done = 0;
	FILE *output;
	char in_buf[BUF_S];
	char *saved_output;
	const char *error_msg = NULL;

	lastsize = 0;

	if (NULL == (output = popen_r_stderr(command))) {
		error_when_executing(command, _(" The command could not be executed. "), NULL);
		return;
	}

	saved_output = g_strdup("");
	while (!(done)) {
		if (fgets(in_buf, BUF_S, output) != NULL) {
			char *s, *start;
			s = g_strconcat(saved_output, in_buf, NULL);
			g_free(saved_output);
			saved_output = s;

			if (feof(output)) {
				error_msg = _(" EOF reached reading "
						"mkisofs size output ");
				done = 1;
			} else if (ferror(output)) {
				error_msg = _(" error reading mkisofs "
						"size output ");
				done = 1;
			} else if ((start = strstr(in_buf, "Total extents scheduled to be written"))) {
				int add = 40;
				if (strstr(start, "(inc HFS)"))
					add = 50;
				g_assert((start-in_buf)+add < BUF_S);
				lastsize = atol(start + add);
				sectorsize += lastsize;
				totalsize = 2048 * sectorsize;
				done = 1;
			} else if (NULL != (error_msg = search_for_mkisofs_error(in_buf))) {
				done = 1;
			}
		} else {
			error_msg = _(" There were and unrecognized error "
					"with mkisofs when calculating the "
					"size; \n perhaps your mkisofs "
					"is too old for gcombust? ");
			done = 1;
		}
	}
	if (NULL != error_msg)
		error_when_executing(command, error_msg, saved_output);
	g_free(saved_output);
	pclose(output);
}

long int get_old_session_size(config_cdr_data *cdr_info) {
	long int size = 0;
	if ((GTK_TOGGLE_BUTTON(enable_multisession)->active) &&
			(0 != strlen(gtk_entry_get_text(GTK_ENTRY((old_session)))))) {
		char *lastsize = get_last_session_info(cdr_info);
		if (NULL != lastsize) {
			char *comma = strchr(lastsize, ',');
			if (NULL != comma) {
				comma++;
				size = atoi(comma);
			}
			free(lastsize);
		}
	}
	return size;
}

/* TODO: the size for each entry should be recalculated to reflect changes
 * 	 on follow symlink flags etc. (at least if auto bin packing is
 *	 implemented someday)
 */
void recalc_size(GtkWidget *widget, gpointer data) {
	cmd_v *mkisofs_size_cmd;
	config_cdr_data *cdr_info = (config_cdr_data *) data;

	if ((mkisofs_size_cmd = make_mkisofs_command(TRUE, NULL, cdr_info)) == NULL)
		return;

	sectorsize = 150; /* for 74m CDs, see comment elsewhere */

	sectorsize += get_old_session_size(cdr_info);

	if (!(is_running())) {
		execute_recalc_size(mkisofs_size_cmd);
		not_running();
	}

	destroy_cmd(mkisofs_size_cmd);

	update_totalsize_label();
	make_estimate_inaccurate();
	gtk_widget_set_sensitive(total_size, 1);
}

/* Checks if a path is in the file-list
 * returns TRUE if it's already there, FALSE if it's not */
static int path_is_in_iso(char *path) {
	int i;
	file_data *info;

	for (i = 0; i < GTK_CLIST(clist)->rows; i ++) {
		info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
		g_assert(NULL != info);
		if (!strcmp(path, info->realpath))
			return TRUE;
	}
	return FALSE;
}

static char *my_basename_n_r(char *path, int level) {
	int i;

	if (0 == level)
		return &path[1];

	for (i = strlen(path)-1; i > 0; i--) {
		if ('/' == path[i])
			level--;
		if (0 == level)
			return &path[i+1];
	}
	return &path[1];
}

static char *my_basename_n(char *path, int level) {
	unsigned int i;

	if (0 == level)
		return &path[1];

	for (i = 1; i < strlen(path); i++) {
		if ('/' == path[i])
			level--;
		if (0 == level)
			return &path[i+1];
	}
	return my_basename(path);
}


/* path: a normal path, eg /usr/bin/mkisofs (note: it will be modified)
 * level: how many path components to include in the path
 *   if direction == 1:
 *        1: the "normal" (include one component, eg
 *           /usr/bin/mkisofs => mkisofs, /usr/bin/ => bin/
 *        2: /usr/bin/mkisofs => bin/mkisofs, /usr/bin/ => usr/bin/
 *        3: /usr/bin/mkisofs => usr/bin/mkisofs, /usr/bin/ => usr/bin/
 *        ...etc
 *        0: include all (ie treat as infinity, eg same as 3 above)
 *   if direction == 0:
 *        0: include all
 *        1: ignore top level, eg
 *           /usr/bin/mkisofs => bin/mkisofs, /usr/bin/ => usr/bin/
 *        2: ignore two topmost levels, eg
 *           /usr/bin/mkisofs => mkisofs, /usr/bin/ => bin/
 *        3: /usr/bin/mkisofs => mkisofs, /usr/bin/ => bin/
 *        (eg prevent empty filenames)
 */
static char *get_user(char *path, int level, int direction) {
	char *ret;
	int trailing_slash;

	trailing_slash = path[strlen(path)-1] == '/';

	my_strip_trailing_slashes(path);

	if (1 == direction)
		ret = g_strdup(my_basename_n_r(path, level));
	else
		ret = g_strdup(my_basename_n(path, level));

	/* there may be a mkisofs bug / windows bug that causes strange
	 * behaviour when directories end with slashes, but for now at least
	 * they are still preserved (hey, it looks better :) */
	if (trailing_slash) {
		char *tmp;
		tmp = ret;
		ret = g_strdup_printf("%s/", ret);
		g_free(tmp);
	}

	return ret;
}

static void add_path_to_iso_files(gpointer data) {
	cmd_v *exec_string;
	char *text[2];
	int newrow;
	file_data *info;
	char *path;

	path = strdup(data);
	if (path_is_in_iso(path)) {
		alert_user_of_error(_("You tried to add a duplicate file/dir to the file list!"));
		free(path);
		return;
	}

	if ((exec_string = make_mkisofs_command(TRUE, path, mainptr->cdr_option_info)) == NULL) {
		free(path);
		return;
	}
	if (!(is_running())) {
		if (0 == GTK_CLIST(clist)->rows) {
			/* as reading the multisession info takes quite some time,
			 * we only read it for the first file added (and also on
			 * recalc/optimize actions) */
			sectorsize = 150;
			sectorsize += get_old_session_size(mainptr->cdr_option_info);
		}
		execute_recalc_size(exec_string);
		not_running();
	}
	destroy_cmd(exec_string);

	info = malloc(sizeof(file_data));
	info->size = lastsize;

	info->realpath = g_strdup(path);
	info->userpath = get_user(path,
			gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(path_default_level_sb)),
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(path_default_from_end)));

	text[0] = info->userpath;
	text[1] = g_strdup_printf("%.1f", (double)lastsize*2048/(1024*1024));

	newrow = gtk_clist_append(GTK_CLIST(clist), text);
	gtk_clist_set_row_data(GTK_CLIST(clist), newrow, info); 

	g_free(text[1]);
	free(path);
}

static void path_added_update_gui(void) {
	update_totalsize_label();
	make_estimate_inaccurate();

	mark_size_inaccurate();
}

/* Add file to the (iso) file-list */
void add_path_to_files(char *file) {
	add_path_to_iso_files(file);
	path_added_update_gui();
}

static void add_path_from_list_to_iso_files(gpointer basename, gpointer dir_base) {
	char *path;

	path = g_strdup_printf("%s%s", (char *)dir_base, (char *)basename);
	add_path_to_iso_files(path);
	g_free(path);
	g_free(basename);
}

static void add_path_to_file_list(gpointer row, gpointer clist) {
	char *tmp;

	gtk_clist_get_text(GTK_CLIST(clist), GPOINTER_TO_INT(row), 0, &tmp);
	file_path_list = g_list_append(file_path_list, g_strdup(tmp));
}

static void add_path_to_dir_list(gpointer row, gpointer clist) {
	char *tmp;

	gtk_clist_get_text(GTK_CLIST(clist), GPOINTER_TO_INT(row), 0, &tmp);
	dir_path_list = g_list_append(dir_path_list, g_strdup(tmp));
}

/* Add file to the (iso) file-list
 * add the filepath contained in the file_selection data to the isofile clist
 *
 * The GTK+ file selector API does not provide a logical way to get the
 * directory we are in, but the following should do it.  Also note that
 * since about gtk+ 1.2.10, some Linux distributions changed the
 * semantics of gtk_file_selection_set_filename, without changing the
 * version number of gtk+. I knew the - so called - stable versions of
 * gtk+ were a moving target to support, but now even the SAME version
 * of gtk+ is a moving target. Keep up the good work!
 *
 * This function contains workarounds to the GTK+ problems, and workarounds to
 * the bugs caused by the original workarounds. I have not found a workaround
 * for saving/restoring the completions yet, so they are lost.
 *
 */
void selected_file(GtkWidget *widget, gpointer sel) {
	char *dir_base;
	GtkAdjustment *adj;
	gfloat file_adj_val, dir_adj_val;

	cursor_wait();

	gtk_clist_freeze(GTK_CLIST(GTK_FILE_SELECTION(sel)->file_list));
	gtk_clist_freeze(GTK_CLIST(GTK_FILE_SELECTION(sel)->dir_list));

	/* save the position the user has scrolled to (some of the following
	 * calls reset it) */
	adj = gtk_clist_get_vadjustment(GTK_CLIST(GTK_FILE_SELECTION(sel)->file_list));
	file_adj_val = adj->value;
	adj = gtk_clist_get_vadjustment(GTK_CLIST(GTK_FILE_SELECTION(sel)->dir_list));
	dir_adj_val = adj->value;

	/* copy the path names of the selected files (since they are cleared in
	 * the next step) */
	file_path_list = NULL;
	dir_path_list = NULL;
	if (NULL != (GTK_CLIST(GTK_FILE_SELECTION(sel)->file_list))->selection)
		g_list_foreach((GTK_CLIST(GTK_FILE_SELECTION(sel)->file_list))->selection, add_path_to_file_list,
				GTK_CLIST(GTK_FILE_SELECTION(sel)->file_list));
	if (NULL != (GTK_CLIST(GTK_FILE_SELECTION(sel)->dir_list))->selection)
		g_list_foreach((GTK_CLIST(GTK_FILE_SELECTION(sel)->dir_list))->selection, add_path_to_dir_list,
				GTK_CLIST(GTK_FILE_SELECTION(sel)->dir_list));

	/* then retrive the current directory (this also clears the completions
	 * incase the user pressed tab => it must be done after we have made a
	 * copy of the selected files above) */
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(sel), "");
	dir_base = gtk_file_selection_get_filename(GTK_FILE_SELECTION(sel));

	/* add the paths to the clist and deallocate the individual paths */
	if (NULL != file_path_list)
		g_list_foreach(file_path_list, add_path_from_list_to_iso_files, dir_base);
	if (NULL != dir_path_list)
		g_list_foreach(dir_path_list, add_path_from_list_to_iso_files, dir_base);

	g_list_free(file_path_list);
	g_list_free(dir_path_list);

	/* reset to where the user was when he pressed OK */
	adj = gtk_clist_get_vadjustment(GTK_CLIST(GTK_FILE_SELECTION(sel)->file_list));
	gtk_adjustment_set_value(adj, file_adj_val);
	adj = gtk_clist_get_vadjustment(GTK_CLIST(GTK_FILE_SELECTION(sel)->dir_list));
	gtk_adjustment_set_value(adj, dir_adj_val);

	gtk_clist_thaw(GTK_CLIST(GTK_FILE_SELECTION(sel)->file_list));
	gtk_clist_thaw(GTK_CLIST(GTK_FILE_SELECTION(sel)->dir_list));

	path_added_update_gui();

	cursor_reset();
}

void del_selected_rows(GtkWidget *widget, gpointer data) {
	file_data *info;
	int i;

	for (i = MAX_ROWS-1; i >= 0; i--) {
		if (1 == selected_files[i]) {
			info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
			if (info == NULL) {
				g_warning("modify_file_set.c::del_selected_rows: "
						"trying to access non-existant clist row!");
				break;
			}
			totalsize -= info->size*2048;
			sectorsize -= info->size;
			update_totalsize_label();
			make_estimate_inaccurate();
			mark_size_inaccurate();
			gtk_clist_remove(GTK_CLIST(clist), i);
			g_free(info->realpath);
			g_free(info->userpath);
			free(info);
		}
	}
}

/*
void selected_dir(GtkWidget *widget, gpointer data) {
	gchar *text[1];

	text[0] = g_strdup(gtk_file_selection_get_filename(data));

	gtk_clist_append(GTK_CLIST(exclude_dir), text);
	totalrows_path++;
	mark_size_inaccurate();
}

void remove_dir(GtkWidget *widget, gpointer data) {
	if (sel_row_path != -1) {
		gtk_clist_remove(GTK_CLIST(exclude_dir), sel_row_path);
		totalrows_path--;
		mark_size_inaccurate();
	}
}
*/
