/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "vector_commands.h"
#include "multisession.h"
#include "common_gtk.h"
#include "modify_file_set.h"
#include "command.h"
#include "linebuffer.h"
#include "filepicker.h"
#include "cdrecord_options.h"
#include "audio.h"
#include "contractions.h"
#include "locks.h"
#include "rip_audio.h"
#include "extra_options.h"
#include "globals.h"

char *cdrecord_path = NULL;
char *mkisofs_path = NULL;
char *diff_path = NULL;
char *isocdrom_device_path = NULL;
int no_of_tracks;

/* TODO: file is getting too large and broad!!! */

static int make_audio_tracks_command(cmd_v *cmd) {
	int i, l;
	file_data *info;

	if (0 == GTK_CLIST(audio_clist)->rows) {
		alert_user_of_error(_("Nothing to burn - no audio tracks selected."));
		return FALSE;
	}
	
	if (already_data_track) {
		l = 1;
		no_of_tracks++;
		info = gtk_clist_get_row_data(GTK_CLIST(audio_clist), 0);
		g_assert(NULL != info);
		add_option_to_cmd(cmd, info->realpath);
	} else
		l = 0;

	add_option_to_cmd(cmd, "-audio");

	if (GTK_TOGGLE_BUTTON(swab_audio)->active)
		add_option_to_cmd(cmd, "-swab");

	for (i = l; i < GTK_CLIST(audio_clist)->rows; i++) {
		no_of_tracks++;
		info = gtk_clist_get_row_data(GTK_CLIST(audio_clist), i);
		g_assert(NULL != info);
		add_option_to_cmd(cmd, info->realpath);
	}
	return TRUE;
}

/*
char make_audio_track(char *track_path) {
	char cdr_cmd;
	cdr_cmd = make_cdrecord_common();
}
*/

/* add the device section of cdrecord to the command */

static void make_cdrecord_device(cmd_v *cdr_cmd, config_cdr_data *cdr_info) {
	char *s;

	if (0 != strlen(gtk_entry_get_text(GTK_ENTRY(cdr_info->device)))) {
		s = string_append("dev=", NULL);
		s = string_append(s, gtk_entry_get_text(GTK_ENTRY(cdr_info->device)));
		add_option_to_cmd(cdr_cmd, s);
		free(s);
	} else {
		/* room for xx,xx,xx\0 */
		char device[9];
		if (-1 == g_snprintf(device, 9, "%i,%i,%i",
					gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->scsibus)),
					gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->target)),
					gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->lun)))
		   )
			g_warning("%s::%i: devicename buffer too small", __FILE__, __LINE__);
		s = string_append("dev=", NULL);
		s = string_append(s, device);
		add_option_to_cmd(cdr_cmd, s);
		free(s);
	}

	if (0 != strlen(gtk_entry_get_text(GTK_ENTRY(cdr_info->driver)))) {
		s = string_append("driver=", NULL);
		s = string_append(s, gtk_entry_get_text(GTK_ENTRY(cdr_info->driver)));
		add_option_to_cmd(cdr_cmd, s);
		free(s);
	}
}

static cmd_v *make_cdrecord_common(config_cdr_data *cdr_info, int can_eject) {
	cmd_v *cdr_cmd;
	char *tmp_text;

	no_of_tracks = 0;

	cdr_cmd = get_new_cmd();
	add_option_to_cmd(cdr_cmd, cdrecord_path);

	add_option_to_cmd(cdr_cmd, "-v");

	if (GTK_TOGGLE_BUTTON(dummy)->active) {
		alert_user_of_error(_("Please note that you have the dummy option enabled: you are starting a test run."));
		add_option_to_cmd(cdr_cmd, "-dummy");
	}
	if (can_eject && GTK_TOGGLE_BUTTON(eject)->active)
		add_option_to_cmd(cdr_cmd, "-eject");
	if (GTK_TOGGLE_BUTTON(pad)->active)
		add_option_to_cmd(cdr_cmd, "-pad");

	tmp_text = g_strdup_printf("speed=%i", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(speed_spin)));
	add_option_to_cmd(cdr_cmd, tmp_text);
	free(tmp_text);

	if (9 != gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(grace_time_spin))) {
		tmp_text = g_strdup_printf("gracetime=%i", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(grace_time_spin)));
		add_option_to_cmd(cdr_cmd, tmp_text);
		free(tmp_text);
	}

	make_cdrecord_device(cdr_cmd, cdr_info);

	if (GTK_TOGGLE_BUTTON(dao)->active)
		add_option_to_cmd(cdr_cmd, "-dao");
	if (GTK_TOGGLE_BUTTON(overburn)->active)
		add_option_to_cmd(cdr_cmd, "-overburn");
	if ((strcmp((tmp_text = gtk_entry_get_text(GTK_ENTRY(fsentry))), ""))) {
		char *fs;
		fs = string_append("fs=", NULL);
		fs = string_append(fs, tmp_text);
		add_option_to_cmd(cdr_cmd, fs);
		free(fs);
	}

	add_extra_burn_options(cdr_cmd);

	return cdr_cmd;
}

/* stores the size of the image/track/"pipe" in sectorsize */
static void get_track_size(config_cdr_data *cdr_info) {
	cmd_v *mkisofs_size_cmd;

	if ((mkisofs_size_cmd = make_mkisofs_command(TRUE, NULL, cdr_info)) == NULL)
		return;

	sectorsize = 0;

	if (!(is_running())) {
		execute_recalc_size(mkisofs_size_cmd);
		not_running();
	}

	destroy_cmd(mkisofs_size_cmd);
}

/* makes cdrecord command */
cmd_v *make_cdrecord_command(config_cdr_data *cdr_info) {
	char *tmp;
	cmd_v *cdr_cmd;

	cdr_cmd = make_cdrecord_common(cdr_info, TRUE);

	if (GTK_TOGGLE_BUTTON(enable_burnproof)->active)
		add_option_to_cmd(cdr_cmd, "driveropts=burnproof");

	if (GTK_TOGGLE_BUTTON(disable_fixate)->active)
		add_option_to_cmd(cdr_cmd, "-nofix");

	/* TODO: check that CD really is multisession CD */
	if (GTK_TOGGLE_BUTTON(enable_multisession)->active) {
		add_option_to_cmd(cdr_cmd, "-multi");
		if (!(GTK_TOGGLE_BUTTON(use_image)->active))
			add_option_to_cmd(cdr_cmd, "-waiti");
		if (GTK_TOGGLE_BUTTON(force_data_multisession)->active)
			add_option_to_cmd(cdr_cmd, "-data");
	}

	if (GTK_TOGGLE_BUTTON(use_image)->active) {
		if (!(strcmp((tmp = gtk_entry_get_text(GTK_ENTRY(image_path))), ""))) {
			alert_user_of_error(_("No image is selected, but the use image"
						" button is checked.\n Please select an image."));
			destroy_cmd(cdr_cmd);
			return NULL;
		}	
		add_option_to_cmd(cdr_cmd, tmp);
		no_of_tracks++;
	} else if (GTK_TOGGLE_BUTTON(audio_tracks)->active) {
		if (FALSE == make_audio_tracks_command(cdr_cmd)) {
			destroy_cmd(cdr_cmd);
			return NULL;
		}
	} else if (GTK_TOGGLE_BUTTON(no_image)->active) {
		if (GTK_TOGGLE_BUTTON(need_track_size)->active
				|| GTK_TOGGLE_BUTTON(dao)->active) {
			char tsize[10];
			get_track_size(cdr_info);
			if (0 == sectorsize) {
				alert_user_of_error(_("Could not retrieve image size, aborting burn."));
				destroy_cmd(cdr_cmd);
				return NULL;
			}
			tmp = string_append("tsize=", NULL);
			if (-1 == g_snprintf(tsize, 10, "%lli", sectorsize)) {
				g_warning("%s::%i: size buffer too small!", __FILE__, __LINE__);
				destroy_cmd(cdr_cmd);
				return NULL;
			}
			tmp = string_append(tmp, tsize);
			tmp = string_append(tmp, "s");
			add_option_to_cmd(cdr_cmd, tmp);
			free(tmp);
		}
		add_option_to_cmd(cdr_cmd, "-");
	} else if (GTK_TOGGLE_BUTTON(copy_iso_cd)->active) {
		if (!(strcmp((tmp = gtk_entry_get_text(GTK_ENTRY(iso_device_path))), ""))) {
			alert_user_of_error(_("No in-device entered, but the copy existing ISO 9660"
						"CD option is enabled.\n Please enter a valid device."));
			destroy_cmd(cdr_cmd);
			return NULL;
		}
		add_option_to_cmd(cdr_cmd, "-isosize");
		add_option_to_cmd(cdr_cmd, tmp);
	}
	else
		g_warning("comand.c::make_cdrecord_command: no mode selected (impossible error)!");

	return cdr_cmd;
}

/* make command for blanking a cd-rw */

cmd_v *make_blank_command(config_cdr_data *cdr_info) {
	cmd_v *cdr_cmd;

	cdr_cmd = make_cdrecord_common(cdr_info, TRUE);
	add_option_to_cmd(cdr_cmd, "blank=");

	if (GTK_TOGGLE_BUTTON(blank_all)->active)
		add_option_to_cmd(cdr_cmd, "all");
	else if (GTK_TOGGLE_BUTTON(blank_fast)->active)
		add_option_to_cmd(cdr_cmd, "fast");
	else {
		g_warning("comand.c::make_blank_command: no blank-mode selected (impossible error)!");
		destroy_cmd(cdr_cmd);
		return NULL;
	}

	return cdr_cmd;
}

/* make command for reading ATIP info from disc */

cmd_v *make_cdrecord_read_atip_command(config_cdr_data *cdr_info) {
	cmd_v *cdr_cmd;

	cdr_cmd = get_new_cmd();
	add_option_to_cmd(cdr_cmd, cdrecord_path);
	make_cdrecord_device(cdr_cmd, cdr_info);
	add_option_to_cmd(cdr_cmd, "-atip");

	return cdr_cmd;
}

/* make command for ejecting/closing the tray */

cmd_v *make_cdrecord_trayaction_command(config_cdr_data *cdr_info, guint action) {
	cmd_v *cdr_cmd;

	cdr_cmd = get_new_cmd();
	add_option_to_cmd(cdr_cmd, cdrecord_path);
	make_cdrecord_device(cdr_cmd, cdr_info);

	switch (action) {
		case TRAY_CLOSE:
			add_option_to_cmd(cdr_cmd, "-load");
			break;
		case TRAY_EJECT:
			add_option_to_cmd(cdr_cmd, "-eject");
			break;
		default:
			g_assert_not_reached();
	}

	return cdr_cmd;
}

/* takes a path and returns a path suitable for mkisofs */
static char *make_mkisofs_path(char *realpath, char *userpath) {
	struct stat buf;
	char *newpath, *erealpath, *euserpath;
	int rval;

	g_assert(NULL != realpath);
	g_assert(NULL != userpath);

	if (GTK_TOGGLE_BUTTON(mkisofs_follow_symlinks)->active)
		rval = stat(realpath, &buf);
	else
		rval = lstat(realpath, &buf);

	if (0 != rval) {
		g_warning("%s::%i: Error stat():ing %s", __FILE__, __LINE__, realpath);
		/* FIXME: what action should be taken? */
	}

	erealpath = escape_graft_chars(realpath);
	euserpath = escape_graft_chars(userpath);

	newpath = string_append(euserpath, NULL);
	my_strip_trailing_slashes(newpath);
	if (((buf.st_mode & S_IFMT) == S_IFDIR) &&
			(GTK_TOGGLE_BUTTON(mkisofs_include_dirname)->active)) {
		/* dir1/=dir2 */
		newpath = string_append(newpath, "/=");
	} else {
		/* file1=file2 */
		/* dir1=dir2 */
		newpath = string_append(newpath, "=");
	}
	newpath = string_append(newpath, erealpath);
	free(euserpath);
	free(erealpath);

	return newpath;
}

/* returns a cmd_v with all options except for the path of the files/dirs
 * if printsize == 1 we just want to generate a command for getting the
 * size of the selection */

static cmd_v *make_mkisofs_c_options(int printsize) {
	cmd_v *cmd;
	char *tmp, *s;
	unsigned int i, l;
	GList *tmp_list;

	cmd = get_new_cmd();

	add_option_to_cmd(cmd, mkisofs_path);

	add_option_to_cmd(cmd, "-graft-points");

	if ((!printsize) && GTK_TOGGLE_BUTTON(mkisofs_special_options[4].widget)->active) {
		for (i = 0; i < (torito_length/sizeof(options)); i++) {
			tmp = gtk_entry_get_text(GTK_ENTRY(torito_options[i].widget));
			if (strlen(tmp) != 0) {
				s = string_append(torito_options[i].cmd_line, NULL);
				s = string_append(s, tmp);
				add_option_to_cmd(cmd, s);
				free(s);
			}
		}
	}

	if (printsize) {
		add_option_to_cmd(cmd, "-print-size");
	}

	if (!GTK_TOGGLE_BUTTON(mkisofs_include_all)->active)
		add_option_to_cmd(cmd, "-nobak");
	if (GTK_TOGGLE_BUTTON(mkisofs_follow_symlinks)->active)
		add_option_to_cmd(cmd, "-f");

	for (i = 0; i < (m_o_length/sizeof(options)); i++) {
		if (GTK_TOGGLE_BUTTON(mkisofs_options[i].widget)->active)
			add_option_to_cmd(cmd, mkisofs_options[i].cmd_line);
	}

	for (i = 0; i < (m_o_s_length/sizeof(options)); i++) { 
		/* NULL check is for commands that doesn't result in an option,
		 * like bootable CD support */
		if ((mkisofs_special_options[i].cmd_line != NULL) &&
				GTK_TOGGLE_BUTTON(mkisofs_special_options[i].widget)->active)
			add_option_to_cmd(cmd, mkisofs_special_options[i].cmd_line);
	}

	for (i = 0; i < (m_o_t_length/sizeof(options)); i++) {
		tmp = gtk_entry_get_text(GTK_ENTRY(mkisofs_text_options[i].widget));
		if (strlen(tmp) != 0) {
			s = string_append(mkisofs_text_options[i].cmd_line, NULL);
			s = string_append(s, tmp);
			add_option_to_cmd(cmd, s);
			free(s);
		}
	}

	if (GTK_TOGGLE_BUTTON(use_image)->active && !printsize) {
		tmp = gtk_entry_get_text(GTK_ENTRY(image_path));
		s = string_append("-o", NULL);
		s = string_append(s, tmp);
		add_option_to_cmd(cmd, s);
		free(s);
	}

	l = g_list_length(exclude_pattern_list);
	tmp_list = exclude_pattern_list;
	for (i = 0; i < l; i++) {
		s = string_append("-m", NULL);
		s = string_append(s, tmp_list->data);
		add_option_to_cmd(cmd, s);
		free(s);
		tmp_list = tmp_list->next;
	}
/*
	for (i = 0; i < totalrows_path; i++) {
		gtk_clist_get_text(GTK_CLIST(exclude_dir), i, 0, &tmp);
		cmd = string_append(cmd, " -x \"");
		cmd = string_append(cmd, tmp);
		cmd = string_append(cmd, "\"");
	}
*/

	for (i = 0; i < (h_o_length/sizeof(options)); i++) {
		if (GTK_TOGGLE_BUTTON(mkhybrid_options[i].widget)->active)
			add_option_to_cmd(cmd, mkhybrid_options[i].cmd_line);
	}
	if (GTK_TOGGLE_BUTTON(mkhybrid_options[0].widget)->active || GTK_TOGGLE_BUTTON(mkhybrid_options[1].widget)->active) {
		tmp = gtk_entry_get_text(GTK_ENTRY(mkhybrid_text_options[0].widget));
		if (strlen(tmp) != 0) {
			add_option_to_cmd(cmd, mkhybrid_text_options[0].cmd_line);
			add_option_to_cmd(cmd, tmp);
		}
	}

	add_extra_mkisofs_options(cmd);

	return cmd;
}

/* returns a string with xx,yy suitable for mkisofs multisession
 * returns NULL if error */
char *get_last_session_info(config_cdr_data *cdr_info) {
	cmd_v *cdr_cmd;
	FILE *f;
	char in_buf[BUF_S];
	char *s;

	cdr_cmd = get_new_cmd();
	add_option_to_cmd(cdr_cmd, cdrecord_path);
	add_option_to_cmd(cdr_cmd, "-msinfo");
	make_cdrecord_device(cdr_cmd, cdr_info);

	if (NULL == (f = popen_r_stdout(cdr_cmd))) {
		g_warning("%s::%i: popen_r_stdout() for msinfo failed", __FILE__, __LINE__);
		destroy_cmd(cdr_cmd);
		return NULL;
	}

	destroy_cmd(cdr_cmd);

	if (NULL == fgets(in_buf, BUF_S, f)) {
		g_warning("%s::%i: reading from msinfo command failed", __FILE__, __LINE__);
		pclose(f);
		return NULL;
	}
	pclose(f);
	in_buf[strlen(in_buf)-1] = '\0'; /* get rid of '\n' */

	if (NULL == (s = strchr(in_buf, ','))) {
		g_warning("%s::%i: session info is bogous - probably not reading a multisession image?",
				__FILE__, __LINE__);
		return NULL;
	}

	if (*(++s) == '-') {
		alert_user_of_error(" This doesn't seem to be a multisession CD, aborting operation. "
				" (Make sure the CD isn't mounted)");
		return NULL;
	}

	return g_strdup(in_buf);
}

/* adds the dir/filenames from the Data CD clist to the mkisofs cmd in cmd */

static void add_dirs_to_mkisofs(cmd_v *cmd) {
	char *tmp2;
	file_data *info;
	int i;

	add_option_to_cmd(cmd, "--");

	/* TODO: probably shouldn't make bootable multisession cd
	 *       later: well, why not? maybe even enhance the support */
	for (i = 0; i < GTK_CLIST(clist)->rows; i++) {
		info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
		g_assert(NULL != info);
		tmp2 = make_mkisofs_path(info->realpath, info->userpath);
		add_option_to_cmd(cmd, tmp2);
		free(tmp2);
	}
}

/* make mksiofs command for making an image
 * if pathname == NULL this is used for checking the size of one dir/file
 * in the fast version of the space-optimize routine */

cmd_v *make_mkisofs_command(int printsize, char *pathname, config_cdr_data *cdr_info) {
	cmd_v *cmd;
	char *tmp2;

	if ((0 == GTK_CLIST(clist)->rows) && (pathname == NULL)) {
		alert_user_of_error(_("You haven't selected any files to make an image from!"));
		return NULL;
	}

	if (GTK_TOGGLE_BUTTON(use_image)->active && !printsize) {
		char *tmp;
		tmp = gtk_entry_get_text(GTK_ENTRY(image_path));
		if (0 == strlen(tmp)) {
			alert_user_of_error(_(" You selected to make an image but you \n"
						" haven't selected the filename of the image \n"
						" you wish to make! "));
			return NULL;
		}
	}

	cmd = make_mkisofs_c_options(printsize);

	if (pathname != NULL) { /* just want to check the size of one entry */
		add_option_to_cmd(cmd, "--");
		tmp2 = escape_graft_chars(pathname);
		add_option_to_cmd(cmd, tmp2); /* bootable cd */
		free(tmp2);
	} else { /* go through all added files */
		if (GTK_TOGGLE_BUTTON(enable_multisession)->active) {
			char *msinfo = NULL;
			if (0 != strlen(gtk_entry_get_text(GTK_ENTRY(old_session)))) {
				/* not first session */

				msinfo = get_last_session_info(cdr_info);
				if (NULL == msinfo) {
					destroy_cmd(cmd);
					return NULL;
				}

				tmp2 = string_append("-C", NULL);
				tmp2 = string_append(tmp2, msinfo);
				add_option_to_cmd(cmd, tmp2);
				free(tmp2);

				if (!(GTK_TOGGLE_BUTTON(enable_cdplus)->active)) {
					tmp2 = string_append("-M", NULL);
					tmp2 = string_append(tmp2,
							gtk_entry_get_text(GTK_ENTRY(old_session)));
					add_option_to_cmd(cmd, tmp2);
					free(tmp2);
				}
			}
			add_dirs_to_mkisofs(cmd);
			if (NULL != msinfo)
				free(msinfo);
		} else {
			add_dirs_to_mkisofs(cmd);
		}
	}

	return cmd;
}

/* called to make command for more thorough space-optimizing binpacking
 * routine, it checks all file/dirs
 * (only for iso images at the moment) */

cmd_v *make_mkisofs_c_optimize(file_data *check_these[], int no_of_selections) {
	cmd_v *cmd;
	char *tmp2;
	int j;

	if (0 == no_of_selections)
		return NULL;

	cmd = make_mkisofs_c_options(TRUE);

	add_option_to_cmd(cmd, "--");

	for (j = 0; j < no_of_selections; j++) {
		tmp2 = make_mkisofs_path(check_these[j]->realpath, check_these[j]->userpath);
		add_option_to_cmd(cmd, tmp2);
		free(tmp2);
	}
	return cmd;
}

static cmd_v *get_cdda2wav_common(void) {
	cmd_v *cmd;
	char *s, *tmp;

	if (!(strcmp((tmp = gtk_entry_get_text(GTK_ENTRY(audio_device_path))), ""))) {
		alert_user_of_error(_("You need to supply a device name to read audio tracks from."));
		return NULL;
	}

	cmd = get_new_cmd();
	add_option_to_cmd(cmd, cdda2wav_path);

	s = string_append("-D", NULL);
	s = string_append(s, tmp);
	add_option_to_cmd(cmd, s);
	free(s);

	add_option_to_cmd(cmd, "-g");
	add_option_to_cmd(cmd, "-H");

	return cmd;
}

static int get_no_of_audio_tracks(void) {
	cmd_v *cmd;
	char *s;
	FILE *output;
	int n_tracks = 0;

	if (NULL == (cmd = get_cdda2wav_common()))
		return 0;

	add_option_to_cmd(cmd, "-J");

	if (NULL == (output = popen_r_stderr(cmd))) {
		g_warning("%s::%i: popen_r_stderr() failed for getting no of audio tracks",
				__FILE__, __LINE__);
		destroy_cmd(cmd);
		return 0;
	}

	destroy_cmd(cmd);

	while (1) {
		char line[BUF_S];
		s = fgets(line, BUF_S, output);
		if (NULL == s)
			break;
		if (!(strncmp("Tracks:", s, strlen("Tracks:")))) {
			n_tracks = atoi(s + strlen("Tracks:"));
			break;
		}
	}
	pclose(output);
	return (LONG_MAX == n_tracks) ? 0 : n_tracks;
}

static cmd_v *make_cdda2wav_command_for_track(int i) {
	cmd_v *cmd;
	char *s;
	char tmp[10];

	if (-1 == g_snprintf(tmp, 10, "%i,%i", i, i)) {
		g_warning("%s::%i: Could not fit track info in buffer", __FILE__, __LINE__);
		return NULL;
	}

	if (NULL == (cmd = get_cdda2wav_common()))
		return NULL;

	add_option_to_cmd(cmd, "-Ocdr");

	s = string_append("-t", NULL);
	s = string_append(s, tmp);
	add_option_to_cmd(cmd, s);
	add_option_to_cmd(cmd, "-");
	free(s);
	return cmd;
}

cmd_v *make_cdrecord_cmd_for_audio_track(config_cdr_data *cdr_info) {
	cmd_v *cmd;

	cmd = make_cdrecord_common(cdr_info, FALSE);

	if (GTK_TOGGLE_BUTTON(enable_burnproof)->active)
		add_option_to_cmd(cmd, "driveropts=burnproof");

	if (GTK_TOGGLE_BUTTON(enable_multisession)->active) {
		add_option_to_cmd(cmd, "-multi");
	}

	if (GTK_TOGGLE_BUTTON(cdr_info->swab_audio)->active)
		add_option_to_cmd(cmd, "-swab");

	add_option_to_cmd(cmd, "-audio");
	add_option_to_cmd(cmd, "-nofix");
	add_option_to_cmd(cmd, "-");

	return cmd;
}
cmd_v *make_cdrecord_cmd_for_audio_track_fixate(config_cdr_data *cdr_info) {
	cmd_v *cmd;

	cmd = make_cdrecord_common(cdr_info, TRUE);

	if (GTK_TOGGLE_BUTTON(enable_burnproof)->active)
		add_option_to_cmd(cmd, "driveropts=burnproof");

	if (GTK_TOGGLE_BUTTON(enable_multisession)->active) {
		add_option_to_cmd(cmd, "-multi");
	}
	add_option_to_cmd(cmd, "-audio");
	add_option_to_cmd(cmd, "-fix");

	return cmd;
}
static void free_command_list(gpointer node, gpointer data) {
	destroy_cmd(node);
}

GSList *make_cdda2wav_list_cmd(void) {
	int i, n_tracks;
	cmd_v *cdda2wav_cmd;
	GSList *cdda2wav_burn_commands = NULL;

	if (0 >= (n_tracks = get_no_of_audio_tracks())) {
		alert_user_of_error(_("Could not read number of tracks from audio CD"));
		return NULL;
	}

	for (i = 1; i <= n_tracks; i++) {
		if (NULL == (cdda2wav_cmd = make_cdda2wav_command_for_track(i))) {
			g_warning("%s::%i: Could not build command for cdda2wav", __FILE__, __LINE__);
			if (NULL != cdda2wav_burn_commands) {
				g_slist_foreach(cdda2wav_burn_commands, free_command_list, NULL);
				cdda2wav_burn_commands = NULL;
			}
			return NULL;
		}
		cdda2wav_burn_commands = g_slist_append(cdda2wav_burn_commands, cdda2wav_cmd);
	}

	return cdda2wav_burn_commands;
}

/* this could probably be made to detect more errors */
int check_if_exe_ok(char *path) {
	struct stat buf;

	if (-1 == (stat(path, &buf))) {
		char *error_msg;
		switch (errno) {
			case ENOENT:
				error_msg = g_strdup_printf("Trying to execute %s, "
						"which doesn't exist.", path);
				break;
			case EACCES:
				error_msg = g_strdup_printf("Access denied when trying "
						"to execute %s.", path);
				break;
			default:
				error_msg = g_strdup_printf("Access to %s failed, "
						"unknown error.", path);
		}
		alert_user_of_error(error_msg);
		return 1;
	}

	return 0;
}

