/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "audio.h"
#include "vector_commands.h"
#include "clist_common.h"
#include "linebuffer.h"
#include "common_gtk.h"
#include "modify_file_set.h"
#include "rip_audio.h"
#include "mainwindow.h"
#include "globals.h"

GtkWidget *audio_clist;
int already_data_track = 0;

static GtkWidget *audio_size_label, *add_data_track_tool;
static int track_s = -1;
long int total_audio_size = 150;
static char *dir_base = NULL;
enum {AUDIO_LABEL_LENGTH = 100};

#define pregapsecs ((GTK_CLIST(audio_clist)->rows > 1) ? (2*(GTK_CLIST(audio_clist)->rows-1)) : 0)

/* The following defines are borrowed from cdrecord.h in cdrtools,
 * copyrighted by J. Schilling */
#define     msample           (44100 * 2)       /* one 16bit audio sample */
#define     ssample           (msample * 2)           /* one stereo sample    */
#define samples(v)      ((v) / ssample)         /* # of stereo samples  */
#define     minutes(v)  ((int)((samples(2352*v) + pregapsecs) / 60))
#define     seconds(v)  ((int)((samples(2352*v) + pregapsecs) % 60))

static void add_path_to_audio_tracks(gpointer data, gpointer user_data) {
	char *text[2];
	char size[AUDIO_LABEL_LENGTH];
	struct stat buf;
	file_data *info;
	gint newrow;
	long int size_sec;
	char *path;

	if (NULL == user_data)
		path = g_strdup(data);
	else {
		char *tmp;
		gtk_clist_get_text(user_data, GPOINTER_TO_INT(data), 0, &tmp);
		path = string_append(dir_base, NULL);
		path = string_append(path, tmp);
	}

	info = malloc(sizeof(file_data));
	if (-1 == (stat(path, &buf))) {
		g_warning("%s::%i: could not stat %s", __FILE__, __LINE__, path);
		return;
	}
	if (S_ISDIR(buf.st_mode)) {
		char *s = string_append(_(" Cannot add "), NULL);
		s = string_append(s, path);
		s = string_append(s, _(" as audio track, it's a directory "));
		alert_user_of_error(s);
		free(s);
		return;
	}

	g_snprintf(size, 10, "%.1f", (float)(buf.st_size)/(1024*1024));

	text[0] = path;
	text[1] = g_strdup(size);
	newrow = gtk_clist_append(GTK_CLIST(audio_clist), text);

	size_sec = (long int)(buf.st_size) / 2352;
	if (size_sec * 2352 < (long int)buf.st_size)
		size_sec++;

	info->size = size_sec;
	info->realpath = g_strdup(path);
	info->userpath = g_strdup(path); /* not actually used, but easier than to check all frees... */
	gtk_clist_set_row_data(GTK_CLIST(audio_clist), newrow, info);

	total_audio_size += size_sec;

	g_snprintf(size, AUDIO_LABEL_LENGTH,
			_("Total size: %lu MB (/783) Sectors: %lu (/333000) Time: (%02d:%02d)"),
			2352*total_audio_size/(1024*1024), total_audio_size,
			minutes(total_audio_size), seconds(total_audio_size));
	gtk_label_set(GTK_LABEL(audio_size_label), size);

	free(path);
}


/* if widget != NULL add the path in data to audio_clist
 * if widget == NULL add data to audio_clist
 */
void add_track(GtkWidget *widget, gpointer data) {
	if (NULL != widget) {
		char *tmp;
		GList *list;

		dir_base = g_strdup(gtk_file_selection_get_filename(data));
		if (NULL == (tmp = strrchr(dir_base, '/'))) {
			g_warning("%s::%i: no / in path, this is a bug", __FILE__, __LINE__);
			return;
		}
		*(tmp+1) = '\0';

		list = (GTK_CLIST(GTK_FILE_SELECTION(data)->file_list))->selection;
		if (NULL != list) {
			g_list_foreach(list, add_path_to_audio_tracks,
					(GTK_CLIST(GTK_FILE_SELECTION(data)->file_list)));
		}
		free(dir_base);
		dir_base = NULL;
		gtk_clist_unselect_all(GTK_CLIST(GTK_FILE_SELECTION(data)->file_list));
	} else {
		add_path_to_audio_tracks(data, NULL);
	}
}

static void add_data_track(GtkWidget *widget, gpointer data) {
	char *text[2];
	char size[AUDIO_LABEL_LENGTH];
	char *path;
	struct stat buf;
	GdkColor color;
	file_data *info;
	long int size_sec;

	color.red = 32000;
	color.green = 56000;
	color.blue = 56000;

	info = malloc(sizeof(file_data));
	path = gtk_file_selection_get_filename(data);

	if (-1 == (stat(path, &buf))) {
		g_warning("%s::%i: could not stat %s", __FILE__, __LINE__, path);
		return;
	}
	if (S_ISDIR(buf.st_mode)) {
		char *s = string_append(_(" Cannot add "), NULL);
		s = string_append(s, path);
		s = string_append(s, _(" as data track, it's a directory "));
		alert_user_of_error(s);
		free(s);
		return;
	}

	g_snprintf(size, 10, "%.1f", (float)(buf.st_size)/(1024*1024));

	text[0] = path;
	text[1] = g_strdup(size);
	gtk_clist_insert(GTK_CLIST(audio_clist), 0, text);

	size_sec = (long int)(buf.st_size) / 2048;
	if (size_sec * 2048 < (long int)buf.st_size)
		size_sec++;

	info->size = size_sec;
	info->realpath = g_strdup(path);
	info->userpath = g_strdup(path);
	gtk_clist_set_row_data(GTK_CLIST(audio_clist), 0, info);

	gtk_clist_set_background(GTK_CLIST(audio_clist), 0, &color);

	total_audio_size += size_sec;

	g_snprintf(size, AUDIO_LABEL_LENGTH,
			_("Total size: %lu MB (/783) Sectors: %lu (/333000) Time: (%02d:%02d)"),
			2352*total_audio_size/(1024*1024), total_audio_size,
			minutes(total_audio_size), seconds(total_audio_size));
	gtk_label_set(GTK_LABEL(audio_size_label), size);

	if (already_data_track)
		gtk_clist_remove(GTK_CLIST(audio_clist), 1);

	already_data_track = 1;

	gtk_widget_set_sensitive(add_data_track_tool, 0);

	gtk_widget_destroy(data);
}

static void del_track_no(int track) {
	file_data *info;

	info = gtk_clist_get_row_data(GTK_CLIST(audio_clist), track);
	total_audio_size -= info->size;
	if (total_audio_size < 150)
		total_audio_size = 150;
	g_free(info->realpath);
	g_free(info->userpath);
	free(info);
	gtk_clist_remove(GTK_CLIST(audio_clist), track);
}

static void del_all_tracks(GtkWidget *widget, gpointer data) {
	char size[AUDIO_LABEL_LENGTH];

	gtk_widget_set_sensitive(add_data_track_tool, 1);
	already_data_track = 0;

	while (GTK_CLIST(audio_clist)->rows > 0)
		del_track_no(0);

	g_snprintf(size, AUDIO_LABEL_LENGTH,
			_("Total size: %lu MB (/783) Sectors: %lu (/333000) Time: (%02d:%02d)"),
			2352*total_audio_size/(1024*1024), total_audio_size,
			minutes(total_audio_size), seconds(total_audio_size));
	gtk_label_set(GTK_LABEL(audio_size_label), size);
}

static void del_selected_track(GtkWidget *widget, gpointer data) {
	char size[AUDIO_LABEL_LENGTH];

	if (-1 == track_s)
		return;
        if (0 == track_s) {
		gtk_widget_set_sensitive(add_data_track_tool, 1);
		already_data_track = 0;
	}

	del_track_no(track_s);

	g_snprintf(size, AUDIO_LABEL_LENGTH,
			_("Total size: %lu MB (/783) Sectors: %lu (/333000) Time: (%02d:%02d)"),
			2352*total_audio_size/(1024*1024), total_audio_size,
			minutes(total_audio_size), seconds(total_audio_size));
	gtk_label_set(GTK_LABEL(audio_size_label), size);
}

static void move_track(GtkWidget *widget, gpointer track) {
	char *to_be_moved[2];
	char size[5];
	int row = track_s;
	file_data *info;

	if ((1 == already_data_track) && (1 == row + (int)track))
		return;

	if ((row != -1) && (row + (int)track != 0)
			&& (row + (int)track != GTK_CLIST(audio_clist)->rows)) {
		info = gtk_clist_get_row_data(GTK_CLIST(audio_clist), row);
		g_assert(NULL != info);
		to_be_moved[0] = g_strdup(info->realpath);
		g_snprintf(size, 5, "%.1f", (float)(info->size)*2352/(1024*1024));
		to_be_moved[1] = g_strdup(size);
		gtk_clist_remove(GTK_CLIST(audio_clist), row);
		gtk_clist_insert(GTK_CLIST(audio_clist), row - 1 + 2*(int)track, to_be_moved);
		gtk_clist_select_row(GTK_CLIST(audio_clist), row - 1 + 2*(int)track, 0);
        	gtk_clist_set_row_data(GTK_CLIST(audio_clist), row - 1 + 2*(int)track, info);
	}
}

static void selected_track(GtkWidget *widget, gint row, gint column,
		GdkEventButton *event, gpointer data) {
	track_s = row;
}

static void unselected_track(GtkWidget *widget, gint row, gint column,
		GdkEventButton *event, gpointer data) {
	track_s = -1;
}

static void rip_cd_pressed(GtkWidget *widget, gpointer track) {
	open_rip_audio_req();
}

static GtkWidget *create_toolbar(void) {
	GtkWidget *toolbar, *add_track_tool, *rip_cd_tool, *del_all_tool;
	GtkWidget *del_selection_tool, *move_up_tool, *move_down_tool;

	toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_TEXT);
	add_track_tool = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Add audio"),
			_("Opens file selector to let you add an audio track"), "Private",
			NULL, GTK_SIGNAL_FUNC(choose_file), (gpointer) add_track);

	add_data_track_tool = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Add data"),
			_("Opens file selector to let you add a data track if you want a mixed mode "
			"CD (only one data track is possible, it'll be the first one)"), "Private",
			NULL, GTK_SIGNAL_FUNC(choose_file), (gpointer) add_data_track);

	del_selection_tool = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Del selected"),
			_("Deletes the selected files"), "Private",
			NULL, GTK_SIGNAL_FUNC(del_selected_track), NULL);

	del_all_tool = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Del all"),
			_("Deletes all files"), "Private",
			NULL, GTK_SIGNAL_FUNC(del_all_tracks), NULL);

	move_up_tool = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Up"),
			_("Moves selected track up one step"), "Private",
			NULL, GTK_SIGNAL_FUNC(move_track), (gpointer)0);

	move_down_tool = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Down"),
			_("Moves selected track down one step"), "Private",
			NULL, GTK_SIGNAL_FUNC(move_track), (gpointer)1);

	rip_cd_tool = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Dump CD"),
			_("Dumps content of an audio CD into sound wav files (using cdda2wav)."),
			"Private", NULL, GTK_SIGNAL_FUNC(rip_cd_pressed), NULL);

	gtk_widget_show(add_track_tool);
	gtk_widget_show(del_selection_tool);
	gtk_widget_show(move_up_tool);
	gtk_widget_show(move_down_tool);
	gtk_widget_show(toolbar);

	return toolbar;
}

static void create_track_clist(void) {

	audio_clist = gtk_clist_new(2);
	gtk_widget_show(audio_clist);

	gtk_clist_set_column_title(GTK_CLIST(audio_clist), 0, _("Track"));
	gtk_clist_set_column_title(GTK_CLIST(audio_clist), 1, _("Size (MB)"));
/*	gtk_clist_set_column_width(GTK_CLIST(audio_clist), 0, 500);
	gtk_clist_set_column_width(GTK_CLIST(audio_clist), 1, 20); */
	gtk_clist_column_titles_passive(GTK_CLIST(audio_clist));
	gtk_clist_column_titles_show(GTK_CLIST(audio_clist));
	gtk_clist_set_column_justification(GTK_CLIST(audio_clist), 0, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_justification(GTK_CLIST(audio_clist), 1, GTK_JUSTIFY_RIGHT); 

	gtk_clist_set_column_auto_resize(GTK_CLIST(audio_clist), 0, TRUE);

	gtk_signal_connect(GTK_OBJECT(audio_clist), "select_row",
			GTK_SIGNAL_FUNC(selected_track), NULL);
	gtk_signal_connect(GTK_OBJECT(audio_clist), "unselect_row",
			GTK_SIGNAL_FUNC(unselected_track), NULL);
}

static GtkWidget *create_track_size() {
	GtkWidget *hbox;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);

	audio_size_label = gtk_label_new(_("Total size: 0 MB (/783) Sectors: 150 (/333000) Time: (00:00)"));
	gtk_widget_show(audio_size_label);
	gtk_box_pack_start(GTK_BOX(hbox), audio_size_label, FALSE, TRUE, 4);

	return hbox;
}

static void dropped_audio(GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		GtkSelectionData *selection_data, guint info, guint time, gpointer data) {
	if (0 == info) {
		/* see ranting comment in filepicker.c:dropped_data */
		GList *list, *names;

		switch (context->action) {
			case GDK_ACTION_COPY:
				list = names = gnome_uri_list_extract_filenames((gchar *)selection_data->data, 0);
				break;
			case GDK_ACTION_MOVE:
				list = names = gnome_uri_list_extract_filenames((gchar *)selection_data->data, 1);
				break;
			default:
				g_warning("%s::%i: Unrecognized GdkDragAction (please report to author): %i %i %i",
						__FILE__, __LINE__, context->actions, context->suggested_action,
						context->action);
				return;
		}

		cursor_wait();
		mainwindow_disable_tabs();
		while (names) {
			add_track(0, (gchar *) names->data);
			names = names->next;
			while (gtk_events_pending()) gtk_main_iteration_do(0);
		}
		mainwindow_enable_tabs();
		gnome_uri_list_free_strings(list);
		cursor_reset();
	} else {
		g_message("%s %i: unknown type dropped, should not happen!",
				__FILE__, __LINE__);
	}
}

void create_audio(GtkWidget *tab) {
	GtkWidget *vbox, *toolbar, *hbox, *swin;
      GtkTargetEntry drop_types[] = {
            { "text/uri-list", 0, 0}
      };

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vbox);
	gtk_container_add(GTK_CONTAINER(tab), vbox);

	toolbar = create_toolbar();
	gtk_box_pack_start(GTK_BOX(vbox), toolbar, FALSE, TRUE, 0);

	create_track_clist();
	swin = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(swin);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), audio_clist);
	gtk_box_pack_start(GTK_BOX(vbox), swin, TRUE, TRUE, 0);

	gtk_drag_dest_set(GTK_WIDGET(swin),
			GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
			drop_types, 1, GDK_ACTION_COPY | GDK_ACTION_MOVE);
	gtk_signal_connect(GTK_OBJECT(swin), "drag_data_received",
			GTK_SIGNAL_FUNC(dropped_audio), swin);

	hbox = create_track_size();
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 4);
}
