
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


#include <string.h>
#include <errno.h>
#include <stdlib.h>


#include "safe-ctype.h"
#include "runtime.h"


/* read.c -- Deal with formatted reads */


/* extract_integer()-- Extract an integer from a pointer. */

int extract_integer(void *src, int length) {
int value;

    switch(length) {
    case 1:  value = *((G95_INT1 *) src);  break;
    case 2:  value = *((G95_INT2 *) src);  break;
    case 4:  value = *((G95_INT4 *) src);  break;

#ifdef G95_INT8
    case 8:  value = *((G95_INT8 *) src);  break;
#endif

#ifdef G95_INT16
    case 16: value = *((G95_INT16 *) src); break;
#endif

    default:
	value = 0;
	internal_error("extract_integer(): Bad integer kind");
    }

    return value;
}


/* set_integer()-- Store an integer value into an integer variable. */

void set_integer(int value, void *dest, int length) {

    switch(length) {
    case 1: *((G95_INT1 *)   dest) = value; break;
    case 2: *((G95_INT2 *)   dest) = value; break;
    case 4: *((G95_INT4 *)   dest) = value; break;

#ifdef G95_INT8
    case 8: *((G95_INT8 *)   dest) = value; break;
#endif

#ifdef G95_INT16
    case 16: *((G95_INT16 *) dest) = value; break;
#endif

    default:
	internal_error("set_integer(): Bad integer kind");
    }
}



/* set_read()-- Set a real variable from an integer value */

void set_real(int value, void *dest, int length) {

    switch(length) {
    case 4:  *(float *)  dest = value; break;
    case 8:  *(double *) dest = value; break;

#if HAVE_REAL_10
    case 10:
	asm("fild %0\n"
	    "mov %1, %" EAX "\n"
	    "fstpt (%" EAX ")\n" : : "m" (value), "m" (dest) : EAX);
	break;
#endif

    default:
	internal_error("set_real(): Bad real kind");
    }
}



/* extract_mint()-- Extract a G95_MINT from a pointer/kind. */

G95_MINT extract_mint(void *src, int length) {
G95_MINT value;

    switch(length) {
    case 1:  value = *((G95_INT1 *) src);  break;
    case 2:  value = *((G95_INT2 *) src);  break;
    case 4:  value = *((G95_INT4 *) src);  break;

#ifdef G95_INT8
    case 8:  value = *((G95_INT8 *) src);  break;
#endif

#ifdef G95_INT16
    case 16: value = *((G95_INT16 *) src); break;
#endif

    default:
	value = 0;
	internal_error("extract_mint(): Bad integer kind");
    }

    return value;
}



/* set_mint()-- Set a G95_MINT to a pointer/kind. */

void set_mint(G95_MINT value, void *dest, int length) {

    switch(length) {
    case 1: *((G95_INT1 *) dest) = value;  break;
    case 2: *((G95_INT2 *) dest) = value;  break;
    case 4: *((G95_INT4 *) dest) = value;  break;

#ifdef G95_INT8
    case 8: *((G95_INT8 *) dest) = value;  break;
#endif

#ifdef G95_INT16
    case 16: *((G95_INT16 *) dest) = value;  break;
#endif

    default:
	internal_error("set_mint(): Bad integer kind");
    }
}



/* extract_dint()-- Extract a G95_DINT from a pointer/kind. */

G95_DINT extract_dint(void *src, int length) {
G95_DINT value;

    switch(length) {
    case 1:  value = *((G95_INT1 *) src);  break;
    case 2:  value = *((G95_INT2 *) src);  break;
    case 4:  value = *((G95_INT4 *) src);  break;

#ifdef G95_INT8
    case 8:  value = *((G95_INT8 *) src);  break;
#endif

#ifdef G95_INT16
    case 16: value = *((G95_INT16 *) src); break;
#endif

    default:
	value = 0;
	internal_error("extract_dint(): Bad integer kind");
    }

    return value;
}



/* read_l()-- Read a logical value */

void read_l(fnode *f, char *dest, int length) {
unsigned w;
char *p;

    w = f->u.w;
    p = read_block(&w);
    if (p == NULL)
	return;

    while(*p == ' ') {
	if (--w == 0)
	    goto bad;

	p++;
    }

    if (*p == '.') {
	if (--w == 0)
	    goto bad;

	p++;
    }

    switch(*p) {
    case 't': case 'T':   set_integer(1, dest, length); break;
    case 'f': case 'F':   set_integer(0, dest, length); break;
    default: bad:
	generate_error(ERROR_READ_VALUE, "Bad value on logical read");
	break;
    }
}



/* read_a()-- Read a character record.  This one is pretty easy. */

void read_a(fnode *f, char *p, int length) {
char *source;
unsigned w;

    w = f->u.w;
    if (w == -1)
	w = length;

    source = read_block(&w);
    if (source == NULL)
	return;

    if (w >= length)
	memcpy(p, source+w-length, length);

    else {
	memcpy(p, source, w);
	memset(p+w, ' ', length-w);
    }
}



/* eat_leading_spaces()-- Given a character pointer and a width,
 * ignore the leading spaces. */

char *eat_leading_spaces(unsigned *width, char *p) {

    for(;;) {
	if (*width == 0 || *p != ' ')
	    break;
  
	(*width)--;
	p++;
    }

    return p;
}



/* read_f()-- Read a floating point number with F-style editing, which
 * is what all of the other floating point descriptors behave as.  The
 * tricky part is that optional spaces are allowed after an E or D,
 * and the implicit decimal point if a decimal point is not present in
 * the input. */

void read_f(fnode *f, char *dest, int length) {
int seen_dp, seen_digit, exponent, exponent_sign;
char *p, *buffer, *n;
unsigned w;

    w = f->u.w;

    p = read_block(&w);
    if (p == NULL)
	return;

    p = eat_leading_spaces(&w, p);
    if (w == 0) {
	convert_real(dest, "0", length);
	return;
    }

    if (w+2 < SCRATCH_SIZE)
	buffer = scratch;

    else
	buffer = temp_alloc(w+2);

    n = buffer;

    /* Optional sign */

    if (*p == '-' || *p == '+') {
	*n++ = *p++;
	if (--w == 0)
	    goto bad_float;
    }

    seen_digit = 0;
    seen_dp = 0;
    exponent_sign = 1;
    exponent = 0;

    while(w > 0) {
	switch(*p) {
	case '0':  case '1':  case '2':  case '3':  case '4':
	case '5':  case '6':  case '7':  case '8':  case '9':
	    seen_digit = 1;
	    *n++ = *p++;
	    w--;
	    break;

	case ',':
	    if (ioparm->current_decimal != DECIMAL_COMMA)
		goto bad_float;

	    goto dp;

	case '.':
	    if (ioparm->current_decimal != DECIMAL_POINT)
		goto bad_float;

	dp:
	    if (seen_dp)
		goto bad_float;

	    seen_dp = 1;

	    *n++ = '.';
	    p++; w--;
	    break;

	case ' ':
	    if (ioparm->blank_status == BLANK_ZERO)
		*n++ = '0';

	    p++; w--;
	    break;

	case '-':
	    exponent_sign = -1;
	    /* Fall through */

	case '+':
	    p++; w--;
	    goto exp2;

	case 'd': case 'e': case 'D': case 'E':
	    p++; w--;
	    goto exp1;

	default:
	    goto bad_float;
	}
    }

/* No exponent has been seen, so we use the current scale factor */

    exponent = -ioparm->scale_factor;
    goto done;

bad_float:
    generate_error(ERROR_READ_VALUE, "Bad value during floating point read"); 
    goto cleanup;

/* At this point the start of an exponent has been found */

exp1:
    while(w > 0 && *p == ' ') {
	w--; p++;
    }

    switch(*p) {
    case '-':
	exponent_sign = -1;
	/* Fall through */

    case '+':
	w--; p++;
	break;
    }

    if (w == 0)
	goto done;

/* At this point a digit string is required.  We calculate the value
 * of the exponent in order to take account of the scale factor and
 * the d parameter before explicit conversion takes place. */

exp2:
    while(w > 0) {
	if (isdigit(*p))
	    exponent = 10*exponent + *p - '0';

	else if (*p == ' ') {
	    if (ioparm->blank_status == BLANK_ZERO)
		exponent = exponent * 10;

	} else
	    break;

	p++;
	w--;

	if (exponent > 999999)
	    goto bad_float;
    }

    /* Only allow trailing blanks */

    while(w > 0) {
	if (*p != ' ')
	    goto bad_float;

	p++; w--;
    }

    exponent = exponent * exponent_sign;

done:
    if (!seen_digit)
	goto bad_float;

    if (!seen_dp)
	exponent -= f->u.real.d;

    *n++ = 'E';
    if (exponent >= 0)
	*n++ = '+';

    strcpy(n, int_to_a(exponent));

    /* The number is syntactically correct and ready for conversion.
     * The only thing that can go wrong at this point is overflow or
     * underflow. */

    convert_real(dest, buffer, length);

cleanup:
    if (buffer != scratch)
	temp_free((void *) &buffer);
}
