/*
 * videolib.h
 *
 * API for routines to convert images from one format to another.
 *
 * (C) 1999 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef __VIDEOLIB_H
#define __VIDEOLIB_H

typedef unsigned long  VL_UINT32;
typedef          long  VL_INT32;
typedef unsigned short VL_UINT16;
typedef unsigned char  VL_UINT8;
typedef          char  VL_INT8;
typedef int            VL_BOOL;

#define VL_FALSE 0
#define VL_TRUE  ( !VL_FALSE )


typedef struct {
  VL_INT32  x,y,w,h;
  VL_UINT32 bytes_per_line;
} VL_GEOM;

typedef enum {
  VL_PIXELTYPE_RGB,
  VL_PIXELTYPE_YUV
} VL_PIXELTYPE;

typedef enum {
 VL_FRAME_PLANAR,                            /*  E.g. YYYYY..UUUUU...VVVVV */
 VL_FRAME_PACKED                             /*  E.g. YVUVYVUV...          */
} VL_YUV_FRAME_PACKING;

/*  FIXME  */
typedef enum { COLORMAP_BEST, 
               COLORMAP_PREDEF_CUBE, 
               COLORMAP_PREDEF_ARBITRARY } VL_COLORMAP_TYPE;

typedef struct {
  VL_UINT32 pixel;
  VL_UINT8  r;
  VL_UINT8  g;
  VL_UINT8  b;
} VL_COLOR;

typedef struct {
  VL_UINT32         id;                      /*  Unique ID <internal use>   */
  VL_COLORMAP_TYPE  type;                    /*  How the map is set up      */
  VL_COLOR         *color;                   /*  Color lookup table         */
  VL_UINT32         len;                     /*  How many colors            */
  VL_UINT32         dim[3];                  /*  CUBE: colors in R,G,B      */
  VL_UINT32         corners[2][3];           /*  CUBE: small,largest corner */
} VL_COLORMAP;
  
typedef struct {
  VL_PIXELTYPE           type;               /* RGB, YUV                   */

  struct {                                   /* RGB-only attributes        */
    VL_BOOL              direct_color;       /*  T = packed RGB, F = cmap  */
    VL_UINT32            Bpp;                /*  Bytes per pixel           */
    VL_UINT32            mask[3];            /*  DC:  Pixel masks          */
    VL_COLORMAP         *colormap;           /*  NDC: Colormap             */
    unsigned             swap_bytes  :1;     /*  Bytes in shorts swapped   */
    unsigned             swap_shorts :1;     /*  Shorts in longs swapped   */
  } rgb;

  /* YUV-only attributes  */
  struct {                                   /* YUV-only attributes        */
    VL_UINT32            samp_size[3];       /*  [YUV] Bits per sample     */
    VL_UINT32            samp_int_h[3];      /*  [YUV] Horiz samp interval */
    VL_UINT32            samp_int_v[3];      /*  [YUV] Vert  samp interval */
    VL_YUV_FRAME_PACKING frame_packing;      /*  Order YUV data is stored  */
    char                 comp_order[30];     /*  "YUYV", "YVU", etc.       */
    unsigned             order_t_to_b :1;    /*  Scanline order TtoB;BtoT  */
    unsigned             order_l_to_r :1;    /*  Column   order RtoL;LtoR  */
    unsigned             y_trans      :1;    /*  LSb Y = transparency?     */
  } yuv;

} VL_PIXEL_GEOM;

typedef struct {
  /*  FIXME:  Need a palette.  What about palette modes?              */
  /*  FIXME:  Support optimimum palette per frame or passed colormap  */
  VL_INT8       *buf;                        /*  Pixel data                */
  VL_GEOM        geom;                       /*  Dimensional geometry      */
  VL_PIXEL_GEOM  pix_geom;                   /*  Pixel geometry            */
} VL_IMAGE;



#ifdef __cplusplus
extern "C" {
#endif


void VIDEOLIBConvertImage( VL_IMAGE *src, VL_IMAGE *dst );

VL_COLORMAP *VIDEOLIBNewColormap( VL_UINT32 num_colors );

#ifdef __cplusplus
}  /* Close 'extern "C"' */
#endif

#endif
