#include <math.h>
#include "common.h"

/* internal function prototypes */
gpointer read_thread(gpointer data);
void reset_buffer(struct Camera *camera, struct RawBuf *buf);
gint open_cam(struct Camera *camera, GString *device_name);
gint close_cam(struct Camera *camera);

void mmap_setup(struct Camera *camera);
void mmap_teardown(struct Camera *camera);

void set_status(struct Camera *camera);
void get_status(struct Camera *camera);
void set_size(struct Camera *camera, FwksVidWinSize size);

int read_into(struct Camera *camera, struct RawBuf *buf);

void read_averaged(struct Camera *camera, gint num, struct Interface *interface);
void average(struct Camera *camera, gint num, struct Interface *interface);

#ifdef undef
gint radius_avg_pix(const struct RawBuf *buf,
                    const gint x, const gint y,
                    const gint r, const gint channel);

gint radius_avg_pix(const struct RawBuf *buf,
                    const gint x, const gint y,
                    const gint r, const gint channel)
{
    /* find the average value of a group of pixels centered at x,y
     * with radius r in one channel.  The radius is used to select
     * pixels around the center, not necessarily in a circle (more
     * like a square, 'cause we don't want to waste too much cpu) */

    guchar *p;  /* pointer to pixel */
    glong sum = 0;  /* sum of values of pixels */
    gint i, j;  /* counters */

    for (i=x-r; i<x+r+1; i++) {  /* for each x value in the square of "radius" r */
        for (j=y-r; j<y+r+1; j++) {  /* for each y */
            if (i>0 && i<buf->width
                && j>0 && j<buf->height)
            {
                p = buf->data + i*buf->bpp + j*(buf->bpp*buf->width) + channel;
                sum += *p;
            }
        }
    }
    return sum / ((2*r+1) * (2*r+1));
}
#endif

int frameworks_camera_initialize(struct Camera *camera,
                                 GString *device_name,
                                 gboolean force_read)
{
    camera->mmap = NULL;
    camera->force_read = force_read;

    camera->buf_a = malloc(sizeof(struct RawBuf));
    camera->buf_b = malloc(sizeof(struct RawBuf));
    camera->buf_avg = malloc(sizeof(struct RawBuf));
    camera->buf_a->data = NULL;
    camera->buf_b->data = NULL;

    camera->buf_avg->data = NULL;
    camera->data_for_avg = g_ptr_array_new();  /* hold frames to avg */

    camera->device = 0;  /* video device */
    camera->bgr = TRUE;  /* Use RGB24 or BGR24? */
    camera->open = TRUE;  /* no camera is open, but we start out by
                           * opening one (it displays "no camera
                           * device" if doesn't have an open cam) */

    camera->read_mutex = g_mutex_new();  /* lock when reading from device */
    camera->buf_b_mutex = g_mutex_new();  /* lock when touching buffer b */
    camera->buf_b_cond = g_cond_new();  /* threads can wait for buffer b */
    camera->buf_b_ready = FALSE;  /* condition for waiting on buffer b */

    g_mutex_lock(camera->read_mutex);
    if (open_cam(camera, device_name) != 0)
        g_mutex_unlock(camera->read_mutex);

    /* read thread reads data from the camera continually; readies
     * buf_b for the interface threads. */
    g_thread_create(read_thread, camera, TRUE, NULL);

    return 1;
}

gpointer read_thread(gpointer data)
{
    struct Camera *camera;
    camera = (struct Camera *) data;
    gpointer tmp;

    while (fwks_quit == FALSE) {
        g_mutex_lock(camera->read_mutex);
        reset_buffer(camera, camera->buf_a);
        read_into(camera, camera->buf_a);
        g_mutex_unlock(camera->read_mutex);

        g_mutex_lock(camera->buf_b_mutex);
        /* swap buffers */
        tmp = camera->buf_a;
        camera->buf_a = camera->buf_b;
        camera->buf_b = tmp;
        /* ok, interface can render now */
        camera->buf_b_ready = TRUE;
        g_cond_signal(camera->buf_b_cond);
        g_mutex_unlock(camera->buf_b_mutex);
    }
    /* Allow display_thread to continue and quit. */
    g_mutex_lock(camera->buf_b_mutex);
    camera->buf_b = NULL;
    camera->buf_b_ready = TRUE;
    g_cond_signal(camera->buf_b_cond);
    g_mutex_unlock(camera->buf_b_mutex);
    return NULL;
}

void reset_buffer(struct Camera *camera, struct RawBuf *buf)
{
    if (camera->vid_win.height == buf->height
        && camera->vid_win.width == buf->width
        && camera->bytes_per_pixel == buf->bpp) {
        if (buf->format == FWKS_PALETTE_RGB24
            && camera->bgr == TRUE) {
            buf->format = FWKS_PALETTE_BGR24;
        }
        if (buf->format == FWKS_PALETTE_BGR24
            && camera->bgr == FALSE) {
            buf->format = FWKS_PALETTE_RGB24;
        }
        return;
    }

    /* otherwise, we need to reset */
    buf->height = camera->vid_win.height;
    buf->width = camera->vid_win.width;

    buf->format = colorspace_palette(V4L, camera->vid_pic.palette);
    if (buf->format == FWKS_PALETTE_RGB24
        && camera->bgr == TRUE)
        buf->format = FWKS_PALETTE_BGR24;

    buf->bpp = camera->bytes_per_pixel;

    buf->data = realloc(buf->data, ceilf(buf->bpp) * buf->height * buf->width);

    mmap_teardown(camera);
    if (!camera->force_read)
        mmap_setup(camera);
}

gint open_cam(struct Camera *camera, GString *device_name)
{
    if (camera->device <= 0) {
        camera->device_name = device_name;
	camera->device = open(device_name->str, O_RDWR);
	if (camera->device < 0) {
            /* frameworks_error(device_string) */
            camera->open = FALSE;
	    perror(device_name->str);
            return 0;
            /* exit(1); */
	}
        camera->open = TRUE;
        get_status(camera);
        set_size(camera, MAXIMUM);
        set_status(camera);
        get_status(camera);
        if (!camera->force_read)
            mmap_setup(camera);
        return 1;
    }
    return 0;
}

gint close_cam(struct Camera *camera)
{
    if (camera->open == TRUE) {
        mmap_teardown(camera);
	close(camera->device);
        g_string_free(camera->device_name, TRUE);
        camera->device_name = NULL;
	camera->device = 0;
        camera->open = FALSE;
        return 1;
    }
    return 0;
}

/* Set up an mmap, if available*/
void mmap_setup(struct Camera *camera)
{
    gint ret;

    if (camera->mmap != NULL)
        mmap_teardown(camera);
    if (camera->vid_caps.type & VID_TYPE_CAPTURE) {
        /* mmap method is available */
        ret = ioctl(camera->device, VIDIOCGMBUF, &camera->vmbuf);
        if (ret < 0) {
            perror("ioctl(VIDIOCGMBUF)");
            camera->mmap = NULL;
        } else {
            camera->mmap = mmap(0,
                                camera->vmbuf.size,
                                PROT_READ | PROT_WRITE,
                                MAP_SHARED, camera->device,
                                0);
            if (camera->mmap == (guchar *) -1) {
                perror("mmap");
                camera->mmap = NULL;
            }
        }
    } else {
        camera->mmap = NULL;
    }
}

void mmap_teardown(struct Camera *camera)
{
    if (camera->mmap != NULL) {
        munmap(camera->mmap, camera->vmbuf.size);
        camera->mmap = NULL;
    }
}

void frameworks_camera_change_device(struct Camera *camera, GString *device_name)
{
    if (camera->open == TRUE) {
        g_mutex_lock(camera->read_mutex);
        if (close_cam(camera) == 1
            && open_cam(camera, device_name) == 1)
        {
            g_mutex_unlock(camera->read_mutex);
        }
    } else {
        if (open_cam(camera, device_name) == 1)
            g_mutex_unlock(camera->read_mutex);
    }
}

/* frameworks_camera_update_status - requests a change of state with
 * set_status, then runs get_status to see what actually happened.
 */
void frameworks_camera_update_status(struct Camera *camera)
{
    if (camera->open == TRUE) {
        g_mutex_lock(camera->read_mutex);
        set_status(camera);
        get_status(camera);
        g_mutex_unlock(camera->read_mutex);
    }
}

void set_status(struct Camera *camera)
{
    if (ioctl(camera->device, VIDIOCSPICT, &camera->vid_pic) == -1) {
	/* error */
        perror("ioctl(VIDIOCSPICT)");
	/* printf("error setting video_picture\n"); */
    }
    if (ioctl(camera->device, VIDIOCSWIN, &camera->vid_win) == -1) {
        perror("ioctl(VIDIOCSWIN)");
        /* printf("error setting video_window\n"); */
    }
}

void get_status(struct Camera *camera)
{
    ioctl(camera->device, VIDIOCGCAP, &camera->vid_caps);
    ioctl(camera->device, VIDIOCGWIN, &camera->vid_win);
    ioctl(camera->device, VIDIOCGPICT, &camera->vid_pic);

    /* apparently MONOCHORME means grey scale? this was in gqcam...*/
    if (camera->vid_caps.type & VID_TYPE_MONOCHROME) {
	camera->bytes_per_pixel = 1;
    }

    /* set bytes_per_pixel based on the palette */
    camera->bytes_per_pixel =
        colorspace_bytes_per_pixel(colorspace_palette(V4L, camera->vid_pic.palette));
}

/* read_into - reads one image into the RawBuf buf. Makes an effort to
 * ensure that some data was read...  this should not block
 * indefinitely.  returns 1 on success, -1 on failure to read any
 * data.  buf had better be the right size.
 */
int read_into(struct Camera *camera, struct RawBuf *buf)
{
    gint bytes_read = 0;
    gint bytes_requested = 0;
    gboolean success = FALSE;
    gint i = 0;

    gint n;
    struct video_mmap vmmap;

    bytes_requested = buf->width * buf->height * buf->bpp;

#undef __TESTING__
#ifdef __TESTING__
    /* BEGIN testing stuff */
    int fd;
    fd = open("/home/pat/misc/pixelformats/yuv410p_320x240.raw", O_RDONLY);
    buf->width = 320;
    buf->height = 240;
    buf->format = FWKS_PALETTE_YUV410P;
    buf->bpp = 1.125;
    buf->data = realloc(buf->data, ceilf(buf->bpp) * buf->height * buf->width);
    read(fd, buf->data, buf->height*buf->width*ceilf(buf->bpp));
    close(fd);
    return;
    /* END testing stuff */
#endif

    /* use mmap if available, but in a completely idiotic manner due
     * to the fact that frameworks is heavily biased toward the read()
     * method. (i.e. we mmap only to memcpy right away) */
    /* code below taken from http://www.jsk.t.u-tokyo.ac.jp/~takeshi/Video4Linux/v4ltest.c.html */
    if (camera->mmap != NULL) {
        /* mmap method is available and working */
        vmmap.frame = 0;
        vmmap.width = buf->width;
        vmmap.height = buf->height;
        vmmap.format = camera->vid_pic.palette;
        if(ioctl(camera->device, VIDIOCMCAPTURE, &vmmap) < 0) {
            perror("ioctl(VIDIOCMCAPTURE)");
            mmap_teardown(camera);  /* switch to read() */
            return -1;
        }
        n = 0;
        for (i=0; i<3; i++) {
            if (ioctl(camera->device, VIDIOCSYNC, &n) == -1)  /* failure */
            {
                if (errno == EINTR) {
                    ;   /* try again on EINTR */
                } else {
                    success = FALSE;
                    perror("ioctl(VIDIOCSYNC): ");
                    mmap_teardown(camera);  /* switch to read() */
                    break;  /* don't try again */
                }
            } else {  /* success */
                /* now there's a frame at map */
                success = TRUE;
                memcpy(buf->data, camera->mmap + camera->vmbuf.offsets[0], bytes_requested);
                break;
            }
        }
    } else {  /* use read method */
        for (i=0; i<3; i++) {  /* give it 3 tries to get some data */
            bytes_read = read(camera->device, buf->data, bytes_requested);
            if (bytes_read > 0) {

                /* more than zero isn't really success (should be "if
                 * (bytes_read == bytes_requested)" ), but on my ibm pc
                 * camera, it always reads one line of pixels short,
                 * -pat */

                success = TRUE;
                break;
            }
        }
    }

    if (success) {
	return 1;
    }
    else
	return -1;
}

/* reads an averaged frame into camera->buf_avg */
void frameworks_camera_read_averaged(struct Camera *camera, gint num, struct Interface *interface)
{
    g_mutex_lock(camera->read_mutex);
    read_averaged(camera, num, interface);
    average(camera, num, interface);
    g_mutex_unlock(camera->read_mutex);
}

/* read an averaged frame into data_averaged; should not be called externally */
void read_averaged(struct Camera *camera, gint num, struct Interface *interface)
{

    gint i;  /* counter */
    struct RawBuf *buf;

    /* make sure data_for_avg and num are the same size */
    while (camera->data_for_avg->len != num) {
	if (camera->data_for_avg->len > num) {
            buf = g_ptr_array_remove_index(camera->data_for_avg, 0);
            g_free(buf->data);
            g_free(buf);
            buf = NULL;
	} else {  /* (camera->data_for_avg->len < num) */
            buf = g_malloc(sizeof(struct RawBuf));
            buf->data = NULL;
	    g_ptr_array_add(camera->data_for_avg, buf);
	}
    }

    /* fill data_for_avg up with image data */
    for (i = 0; i < num; i++) {
        buf = g_ptr_array_index(camera->data_for_avg, i);
        reset_buffer(camera, buf);
	read_into(camera, buf);
        if (interface != NULL)
            frameworks_interface_update_progressbar(interface, "Grabbing...",
                                                    ((gdouble) i + 1)/ ((gdouble) num) * 0.75 );
    }
}

/* average (arithmetic mean) the frames in camera->data_for_avg, placing the
 * averaged frame into camera->data_averaged.  Make sure stuff is in
 * data_for_avg before calling this (TODO: should probably check for NULL
 * ptrs itself).  Note: this function only handles byte sized image formats
 */
void average(struct Camera *camera, gint num, struct Interface *interface)
{
    gint i;			/* counter */
    gint j;			/* counter */
    gint total = 0;
    gint64 bytes;
    struct RawBuf * buf;

    reset_buffer(camera, camera->buf_avg);

    bytes = frameworks_camera_get_capture_width(camera) *
        frameworks_camera_get_capture_height(camera) *
        camera->bytes_per_pixel;

    /* for each byte */
    for (i = 0; i < bytes; i++) {
	/* for each frame in data_for_avg */
	for (j = 0; j < num; j++) {
            buf = g_ptr_array_index(camera->data_for_avg, j);
	    total += (gint) *(buf->data + i);
	}
	*(camera->buf_avg->data + i) = total / num;
	total = 0;
        if (interface != NULL && i % 5000 == 0)
            frameworks_interface_update_progressbar(interface, "Grabbing...",
                                                    0.75 + ((gdouble)i/(gdouble)bytes) * 0.25);
    }
}

int frameworks_camera_get_max_width(struct Camera *camera)
{
    return camera->vid_caps.maxwidth;
}

int frameworks_camera_get_max_height(struct Camera *camera)
{
    return camera->vid_caps.maxheight;
}

int frameworks_camera_get_capture_width(struct Camera *camera)
{
    return camera->vid_win.width;
}

int frameworks_camera_get_capture_height(struct Camera *camera)
{
    return camera->vid_win.height;
}

/* the following functions set various controls on the camera.  If
 * passed "-1", they just return the current value of the respective
 * control */

int frameworks_camera_set_brightness(struct Camera *camera, int brightness)
{
    if (brightness != -1) {
        camera->vid_pic.brightness = brightness;
        frameworks_camera_update_status(camera);
    }
    return camera->vid_pic.brightness;
}

int frameworks_camera_set_whitebalance(struct Camera *camera, int whitebalance)
{
    if (whitebalance != -1 && whitebalance != camera->vid_pic.whiteness) {
        camera->vid_pic.whiteness = whitebalance;
        frameworks_camera_update_status(camera);
    }
    return camera->vid_pic.whiteness;
}

int frameworks_camera_set_contrast(struct Camera *camera, int contrast)
{
    if (contrast != -1 && contrast != camera->vid_pic.contrast) {
        camera->vid_pic.contrast = contrast;
        frameworks_camera_update_status(camera);
    }
    return camera->vid_pic.contrast;
}

int frameworks_camera_set_hue(struct Camera *camera, int hue)
{
    if (hue != -1 && hue != camera->vid_pic.hue) {
        camera->vid_pic.hue = hue;
        frameworks_camera_update_status(camera);
    }
    return camera->vid_pic.hue;
}

int frameworks_camera_set_color(struct Camera *camera, int color)
{
    if (color != -1 && color != camera->vid_pic.colour) {
        camera->vid_pic.colour = color;
        frameworks_camera_update_status(camera);
    }
    return camera->vid_pic.colour;
}

void frameworks_camera_set_size(struct Camera *camera, FwksVidWinSize size)
{
    g_mutex_lock(camera->read_mutex);

    set_size(camera, size);
    set_status(camera);
    get_status(camera);

    g_mutex_unlock(camera->read_mutex);
}

void set_size(struct Camera *camera, FwksVidWinSize size)
{
    if (size == MAXIMUM) {
        camera->vid_win.height = camera->vid_caps.maxheight;
        camera->vid_win.width = camera->vid_caps.maxwidth;
    }
    else if (size == HALF) {
        camera->vid_win.height = camera->vid_caps.maxheight / 2;
        camera->vid_win.width = camera->vid_caps.maxwidth / 2;
    }
    else if (size == QUARTER) {
        camera->vid_win.height = camera->vid_caps.maxheight / 4;
        camera->vid_win.width = camera->vid_caps.maxwidth / 4;
    }
    else if (size == MINIMUM) {
        camera->vid_win.height = camera->vid_caps.minheight;
        camera->vid_win.width = camera->vid_caps.minwidth;
    }
}
